package OpenInteract2::Brick::Comments;

use strict;
use base qw( OpenInteract2::Brick );
use OpenInteract2::Exception;

my %INLINED_SUBS = (
    'comments-1.23.zip' => 'COMMENTS123ZIP',
);

sub get_name {
    return 'comments';
}

sub get_resources {
    return (
        'comments-1.23.zip' => [ 'pkg comments-1.23.zip', 'no' ],
    );
}

sub load {
    my ( $self, $resource_name ) = @_;
    my $inline_sub_name = $INLINED_SUBS{ $resource_name };
    unless ( $inline_sub_name ) {
        OpenInteract2::Exception->throw(
            "Resource name '$resource_name' not found ",
            "in ", ref( $self ), "; cannot load content." );
    }
    return $self->$inline_sub_name();
}

OpenInteract2::Brick->register_factory_type( get_name() => __PACKAGE__ );

=pod

=head1 NAME

OpenInteract2::Brick::Comments - Base-64 encoded OI2 package 'comments-1.23.zip' shipped with distribution

=head1 SYNOPSIS

  oi2_manage create_website --website_dir=/path/to/site

=head1 DESCRIPTION

Are you sure you even need to be reading this? If you are just looking
to install a package just follow the instructions from the SYNOPSIS.

Still here? This class holds the Base64-encoded versions of package
file "comments-1.23.zip" shipped with OpenInteract2. Once you decode them you
should store them as a ZIP file and then read them in with
Archive::Zip or some other utility.

A typical means to do this is:

 my $brick = OpenInteract2::Brick->new( 'comments' );

 # there's only one resource in this brick...
 my ( $pkg_name ) = $brick->list_resources;
 my $pkg_info = $brick->load_resource( $pkg_name );
 my $pkg_file = OpenInteract2::Util->decode_base64_and_store(
     \$pkg_info->{content}
 );

 # $pkg_file now references a file on the filesystem that you can
 # manipulate as normal

These resources are associated with OpenInteract2 version 1.99_06.


=head2 Resources

You can grab resources individually using the names below and
C<load_resource()> and C<copy_resources_to()>, or you can copy all the
resources at once using C<copy_all_resources_to()> -- see
L<OpenInteract2::Brick> for details.

=over 4


=item B<comments-1.23.zip>


=back

=head1 COPYRIGHT

Copyright (c) 2005 Chris Winters. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS


Chris Winters E<lt>chris@cwinters.comE<gt>


=cut


sub COMMENTS123ZIP {
    return <<'SOMELONGSTRING';
UEsDBBQAAAAIAMVSZDIN7BTzpxcAAB9pAAAuAAAAY29tbWVudHMtMS4yMy9PcGVuSW50ZXJhY3Qy
L0FjdGlvbi9Db21tZW50cy5wbdU9a3PbOJLf/StwimdEjWXJzu7cVcmxElfimXFV4kw5zuxdxVkW
LUIS1xSpkJQdb6z97deNN0BQjziZu1XN2BYJNBqNfqOBzKPRTTSh5O2cZmdZRYtoVD0dDE5GVZJn
g8HLfDajWVUe7ew8Ibtn8YDIJ735rHtLDnuHP5OnBwc/9w/+0j/4Kzn8efD06eDwgKTRaJrfk9PP
c7K7s7MoKSmrIhlVR+zv6wh+fLoL/MOSDm/12+Wb14PBaVYlVUJL/ux1PhkM4Mdf57RIifFBaBNa
hWk+mdBCgngTJelg8I5m8Qz+4s+cMV/mWVlFMCMGYgD9Zed6w4p+rsRYLy//u7FdfpNQ76v3FSKx
s7uG2oPBH6cX787enpNjUs6LJKvGQeuHuPfDwdO41SWfdi/obVJiF74Au+T4X6QfXMV7nase+9UH
1HZm9yQgu2JC+G331ekvJ+9fX4anL9++Og1fn53Dj9PzXy9/g4H+6+BItHl99ubs8vSCuJ9j0h4M
Bm3GC6+Scp5G96TMZ7C0i9ksKu5Jko1zEszTRUkikibZTYeM84JEGcmv/0FH1c5Oubgm5TS/C2WX
LzsImWNa0nRMOjDMi/CIPWa4PzwcGwsbkNdvfw1Pfv+dzUn3hQHCURqVZZf/ncQMEoO5PwwRAHsd
RlkMLzn8RZbSsrS6A7KxAYGjJ3HZH95FRRaQ1ssoy/KKxJIIfDIDkuVipgSm3eru1EgIn5ZowYfb
I2evyCS5pVkLZyQbFbRaFJlCf1ZOAtIeSdGjRZEXPTFqGEdVFM6SskyySVsCWSri/DCPimhWAjEC
STI+l6QMY3q9mJAff3RmyR7DNH+h1WgKUNUC42q2NbEGklJthTxyEG8NIgtj0tsoNahYExU2o3cc
/P5wjAOGkyJfzIMvFvXuprSg5HgIVGB0OybPycn5K0FuXCx40rYpDkMvWJ8PXv74qFsvO5xofA7J
GFnihb38nIpfxBqEbAmWMGhr90VLkhx/0hTEXvcTnC3oZ6wwG6QcRWlUkBdfTKItyZAcWINba8aE
7Mcf5QP4JleKxmqh8huLnRgA+e7YGGx/+OHgo263VH850/BhUONuC6PzXCET3YLuja5T2iWjgkYV
clRVRFmZACVJntFVuK7hmIzeOZyCn2wxC6W0IAMc1EWRs5H4QBODQ2ptNZMZbZPYard0poCCICRd
SoExhmD1wFA1S7u/jwltQjMWxLU/OXv9/uIUTDPXSgwyE1XFDRyPBoWkPi2vaMNfL2oLtLS+eZhF
koDxyrEkxf5QkDKm5ahI5mj2jurTK+XqfhHNwQNIKUobgwfP+YNNui6KdEmI0RUfNE1mWZN30Z1o
qXFgH9W6iGZLo4upIGzVPqEZsHZFgV2BybMq8C7RlVDiXWCiLJoJRai8BamUxGhtUCGwXmACwFC/
TkqwNVIWolGRA9tz/Etuj1Ns8U3s8G5BPy1oiRwP7tH+UHzVlmEOrmZYJv+kSBvxdn/I5gb2Tb0F
K+Ylw8ODJJvsEtNxtEircJOuT38+Ik/AMYUh4wRFhI5BSKper2fjB8qjGT14CUMguEMx7yfkckqB
/zMK+rfKyTUlB/vo4MYIWYFO8ztahNf5IkNTFZi0+El9xaH3ySEMYJjTxXze3FP362BPhROMfb2o
yB0lc9RzhxwjMk4miwJsDOBZTSmp6AwcmIqSAHwyMl3Moox2fJ6Doqk5DaYKzQd75FCrGBNt1tJ8
YLVUU2ByNtRzcpqwKZtN8IFu4yr9+gqaLWAVeVeD0rr3Zn7LKoclTWZJJb8ANi2TUl2TGI5azosY
4hfdsT3Py4rGIcRFr07fvXR8nFGeLmYZx4C1ljpgK+eGC1UUx9yvCW/ofd3fHDHzEvIpo9JAf7PL
HCDTPNjGybJJSmOZJsXrNjnuVsmU6UgFo7KZ60SpBdQ+lEWtJ+R0EsXA/tOkJPhfBjogGaEfItWA
/EiTHaKOVqqVmdXAHesrjLcIJH4Bz4irDYtCOqJYaXv1XzqSEYQzWX1Zdya9rZqcrf3hCPi0Cuv0
X/45dg2Zjan4umWL0lRbNxZsgq4oqmS0QJ7gdq5Hgkux4CVIJb7g+g+AtFlkqHTCjFbTPAa+IGr2
3HFFZSqsJokKmrUr6PwTapikoPFPCDDJIEYBmU+qXkdb1vD6PhT+1+NtrGn9wnGRz0Kh4wIRF4E8
tpW32jZ02/YhsooWt4+ShfC3X2LbvvaeUROwWJcxfAIw+Lv2tqEvIyysdijdSX8IrDFaH+v+LQFO
kgG9xVEfPF7xRyvgVX6+4504s7MIiR0syn1tsNCka5wgwVE0m7v6m6yHZRrEmuDySCthwjR1RoOT
IUiBHxFtwVdt0KQuFq9tmisfuGN0qIBfoxQiTV8H9fJ7uwS+HEakhOCxOQxGOOY72E7DCfgM2/gC
Ftuc4m/OMhi1ew0467W9B6H0ouND+P2BVbkHbyZELVx+Q4JWV3gJddMNg3fsqax3Pf4Ug2fZOplu
jQiquJTK6XE7I9XWnxPESSQZvzrCpN81BWIsiKt55la/I49u6Jgca2DQsXNt8pVWoyvF1IW1tVrl
8gGcUCT01pAQ0K/GlLx6dXuR8TjbSwdfOXyTwyfemzy9VfZPzvs8VzMds4AODSU37Wev7Ml/5cTZ
rBF0o/FwVcSGYE0eE1ny7yjJMa3A8lqyfKmCD+DKLEr38ywFT7+gEJnAa4K0nIUlrdB0grtw9UOH
PZYE55+AK0/+al7QkIKGm1NMUQa7wv8M2VYLjRPmxH4j9WCj0aQBmsVYZijOxpiduEOTOC/y2yQG
3CM5yS6pohvKchRiFZilBC03ouBgCxBAwRz63EZFwibYBjrcJvQuxH26Nusxy285mLZBo7boL0Ez
OHwsNrpI2zSLlBQnczxTpvaHXwRsMz9Yf1knn4mlw6UmySD+ADdoPk8Tm2qTIrpmM0GW7L+/eN2n
uPHJduYEEAwbxGRxnxJnCUEL/QwmB/z6EsCDIwy/54s0ZZl6iJMEuRx3XkFiyFPgZvl2xyugWjdL
Z3IMTjeY9FCm8SyVJKZdU13y46ow2ZCjoNuZ47kOqgBgLrNlCWd0dk2LkHlTRX21nAZqxSQqzntY
b00NFzjYxUAhOo3KkK9P0AHOe07a9xTC0wFptw38wB8CMcdM+jX1GGPrdQ036y1LkWvcbMDExs16
aQmIhWiWS1S/o3plqtLODDDF17TQNR3YNfHX2lBEdaF01E3ZxUdLJ7Bm0ZzpoujOWwd5eiglassN
fZPtvXCvC15HF+wwHX9bM8xzDTqId+J2yTT8pVpZbtFm8+qerf3jbJq7hGZ0JXTcugyJAFH3VA0x
A/cczVPcnGF7xVvwPIVoHbhcU8fOilY5U/hGrYd3mwRSshgF011EAtMJEYWQgc/HVVuMrWAGIco1
ZSk7mUpLMrERaw7S2So3zBT9WpMSeCI8rWQNQ2yrXlef2RLogdPkeDva9rhGpibt6k62GafV7nsD
Hl4Y3z2+RdF1NLZtcjZX06gwlWSMpMgGdY3dbdKsR+ZsD4ws3chMdnLQxlYms9qh3N82VAQ+4slZ
yUaNIxhAACnmqi3NQZKKMqXi1xrneZWMG6tmNqqYYU+4i2jnnsy8k03fLklHDZh/5NxjzzeAP8Zy
KkhKY7+GP7T4QAvTat2+3Ro4GFk9UaIOQZoODPfBr0TWY6Rd3Ka8AxasfBUbWVreQr/uYsOSiCYo
hEumH+0VY8+9sYlop+sm7I5OzUOtn2CFWj/+3PLshenVdm1Tg1uTbYbLsYJVa2WI+zEfx2IIHUdI
BnBHXr/yG+4B+OfseI71vQHliDJZkj6R8V7UJulCGzlFvwnVAN3wne0iWflQA7hvanY+xRaz+jwx
BaIpzBib8ehqAtumSKcaVHdek2GapIZCkZV9dBoSeLZtc4MrjpvoAj07h6f1m2ZdYCZfcQeWmRIb
qq1ovWbF5jvsBYsBMgw/udrvsBrjNKmMyt+uBcIaRJt1AQxg8eoOhCksyVACX2oallVe0CbadRtR
8xBU0OcfeZIFNtI+GLy7t0QbqIrliNQwmmxyxNxXqi+YayR5awM53YB+nidYboNWeO8/3xj29bfT
k1dYYo1v0JvrKtMpCVZQzF1txG0Ns+OjB80zMQUSArbvvH8gski/4EarmWwzq6Q2tp+yxwuZ3QSb
TdMYNRpW53Ptpl0dmCEvuubOa2OQwj+G7SSmWSPaNkrGAk+WRqMpQ55hgNUhLk7uJoVwrur7IAKE
qX1F23+Rsv/3q3Kv3/e/g1e71rvGbW/9Vm59n5RlMslYrk+l9QcSl2MxjL1Hpk2rbLdULe2c5bsq
Gd1gnFckk2lFsvyOBHdJNSVVMqP/zDPaUQzgOhO4fbmUbMRFFXf6qVr/J+Q0Kxfg5b6/eA2+ZUVS
GgHjlVVUVIQN8tvl5e8KvuHZe70dIOTs4e/TqpoHZef5oN9/WONc8V5mckRNm6k7xuaLTGbmb2mB
hzZYDI1vCrqPnIKED8D2i2Ryp17qEVgJfq5taqlkTRTM3JJ8gSnze/KMjvKYDp/1+W9SUnbYhJc2
YO1foaWPpdeneco2kUkbD4m8fHt+eXp+aWQdXyCcUIHRGQ8uvwsmvQd1R02lvIFdtdUNBH5B76fn
HYkk3znUjeaLcuqMCwr+qRnRG5j3yN4eR4Q3eJgkJZ19VuT5lVbAjDHJx4xAeMqIVBHutFzjURa2
bHyHJhpjYhbHVWTr1FjVntizD38fftwbPjxM6l6oUEK+lpKd2RIztArUOEwoEcGknJUrRnaOSg0G
NEOsgyZGaURsE0Cqtea4k0WVcwFluOPRn1Xo4vQDlDOQsQ+g1a46Vx8/7nUenkVkCg7ucWv3sDXc
PXzWj4YPEzXIVXaVkf0heTYfrobdD64KaAz/P2CfTh9AzYf9iSEglbBAxsqS64gpKgmaM/JhXc2L
DqjobVFw1fw4+Qw6TDbXLmBVJLMQJBqzIFGFeg4IRsOUZpNqGugB/ArXJqPJ+Rzph2cAmhvA45bo
9hLwBIpaCD3rQzugbzmzhgEIe3uOMjvPs30RlrCk610SgxNQ+hRVbctkqCov6rv+9e0TXflsg3U3
JFaCdfc9OFAxl9GUwjKLwj2WltL1zrKczypGst0B9NfOMXgwE53SuyAcs/a7hciqG23kXqNsI9Or
bbfk6IXEQha0GSqWV5/e0PvSQNZkO64pXQhdTiR8hBWtYKt9ppC/sfcc7VVQ3pm2QIqyVilubQrb
VxtbJX1rDq6IWACLH7vewZvLC60dcjc7sRsp0a0xGX/VtmN30Z5+aiqXQ/HpCUPPOfPL9piBUxXd
CislfYkMT6YCy1OaaX/KXjx14MlxIHgGxSt2Rq5w2JBP0ZL1jrLNZLYhzIX/jh2rBTT7U/iGoREq
PKSQlHzKam0jKZssOOmZoQRv5l2Bps1XvhSqKywGz5d7eNCKSn1+IUuTreM+v2O4fS87/8varDz0
95JZ3FhToi0ouGb/RMzdCjADW+7l9qTCsUSO27hsUFebYlipQ4pHFA1Cq40KBTVMCAOqRekC5U97
2CC/aSv25QzIT/LgXtSiFJtdyM6ijJWdQsEKjEmOU0ORU7JWlx276PTQM3HZ0C5nVSQ3jx04aGJt
zAjRBB9WHqnslzSLEcsMtwWSERO0soYdEjKMwF8LrXZ2BYTbZ17kI7AX23RhdjYEMhZWr3on32aT
XTivcyIrPSdfVokd2HfySC8yeueELbPocz29CQ9DFo+wQcSpr+Yj9HVHEbuhUWTZtf5V8fwq62uc
NM8yeRJz4J2GHCV3M5Fbd0C/iy5HWaH3g+275KAre/RIe6/ZXjrdeJ8NtxCN0Vn3lbuG3CK3rrJW
lxOcp5qekL9NKSh8go+qKda8sUArxuqv+3wBXJ0xnTGn+TwFYZvmmG8SPA1tMKoHKBRi6XsGZAZ8
iWIJZqSaRpUA2iMXHA+eIklKbiQXzCLkY4CAIrBv8bPAolyMkNfHEETf9wTf+WVm013KjTJnrLpL
BJ5Y3YX4jpNCH97kM6Rigl3UQ3GOLcXBQwElzjEEq7Bc21PFZmkV3AayDgAxNeUrFlifRGqf5w04
RwrjLDeofu859NHgf1kZ8RTM1WIeGm+cokNNTRjPXq8SRpkjHow8QLM4dxJDBlikDkO0oYBig8Ta
ee5jM4YFUkajzYf13ADh+H3aPjEvyy8jXQYbPZcH5lG8GOd5D4aSVY9ceep8jtgaYakcvr/kpwI6
UicXFyf/01ja/Zxt6vp7L/1dBiuorvWzkEmdWBLH8BTqTuittzrcTY7+VfnT1QP8QE0canI7hzvw
s3rT3X/xwGone9XNAk0Odr2P3KMgehem1oZPW7TB/Xr2wGq27HCvLtCL+YhTRaizTTZnK7TqwoHW
B4b6QKD2seYdri5nERwhUxU2epveUXHCFD6K6Ha4e9B3b9zwFtJwnGUd+Hl+B7pAxUzoKrHASRTd
Kh16ly/SGHxQrIdmx8uVMcwzADNeFNVUdC39do/7vPjnPY4orVqDi/ZNzdomB0oaylibC011V4Y8
NArzeWOkXi+DNeyiPTar9HFh+gPILczAe1xZNQovguGhMaznDaVzi/tWWAEhDAzz/9gY9yx/JOrA
RXhyU41VsLGmUQnIgJai/GDnSmF3GXds+m4ld23QSVhvAs+sDEOcxKwUfk5HoJXxpimu+UTgxr8A
y88wvSvqJZ9IgcTW7PyOwENzrKyIaorRjxy3QVoZD5HX1ln+zra6NEOYdMJTx2KKqyorxfkJPX3Q
xgWg9qhom+uEe/jFJ73OX3OpgVeS7dr3rw0GArGw+Nz/TuQyzpLAiq8km0WrAWlzxNvrqWaBXE9B
wEOQ0C1LaORydoqRs/BnkXxwYhUspJPGAwTUmosAgoFVilKFu0mTBPmXxprJbYAbnm9dXXOIn8fV
HeLHqj3crIqzq52cLe/1WnXc1aYQeAWDRn5q1U8I+DE9Emg+OhsmuYs3jhdzyg36CjmVKy95ygln
dVGyiAPs14b/75Rx2u2W601sAwK1HTQ+RVTEVkPn1jKZrMwEb/LE5vberLirRHQ3R3z8AU6RbPWE
vmv83M3SmRZ4mdi0R5CZXeE/NK3BI2Jdy4lkyZSBUkDifJnf0H8t6yNmwPfuQTmHjv+2od2muxG1
YK/Ba/mO4R/bHvi2EmOs8xwLZ0rfKaVvIjzGSA2SI/PP3jz4N4+XAPrmd2d+I4P8eBP8GLvr3k5S
M7xEUB70VY2zVPmrUYWFP5VDKm0ZXWPDKL/L6Wu80nM31QhTESivyya02jUKD/wUbvumfVBLmhr3
fPqyvkZWjRa3qq7BqTc23qkOxiE36zC6oau80zQ2uHhhLf/ub2viNLRxNDdsRNZfJQuajyetOZJk
LpsMdJZW6tEp7A9FJSJrG4rXK1KVrBII68aFc+4kshkUQHqcTHRxjxVmNnKhxYHvxO6fuurE5GpQ
nLZ1VpF30yXF7Gy3xLhtTUCzobnZhSOJm9F8ooRNqpyfzML38giKP5X8nHz6FLSshi3yTH7nxBl6
ew6I3WqbDC/ezo2cn4lwzeMEwBTYB1mzyuu0Y2QT702a1VsqvhoqHqs3Umw+VCzvGPFvYrxhxvi7
DVOq31XyJ+Rh36Em8fGnwmrb9CrPb3oEdf2xDbwoM6xJvbpCkvsJ4n39CIn5aZ3TO6UoeUUzgbm9
O7344/QiPD95c9r6SmXsIIk1jwbUvgkMyx5NKpk9dfW/iHlWEsf13oAaF6dv3v5xytJ/yWyesno6
3Mil+sDTRhv7rNZJBHurcdjsRPfqMwjN1+FpABse51pxZMu4PaDh0BYflGxzbMu8ksB/cKuRHz2X
LnnOepk4NZ/22mKQpkNi29wqKHiI3TrE7iCRd+Yz3+pummCuzOs79lqk52LaCn5lfY0r8NrqCryO
dZl7w/X58vZMtHPL/z+Ouu9+uK92ydelwpozYd77Er/FjXEiebJBFdiaf4WgeZE2+JcIvnaltlot
Y8WMVWN0a7xl3L7O/7uf4ReW0L6qDpW5bFDQ0UZnpnk7ee22/z45Vpe+UvE0HffcsP/au8HrN3x8
7xujDcL8O94QzbHfqGbz//DGxev8c2gzrGbmw6Od/wVQSwMEFAAAAAgAr0zPMMhvTB69AAAAsQEA
ACsAAABjb21tZW50cy0xLjIzL2NvbmYvc3BvcHNfY29tbWVudF9ub3RpZnkuaW5pfZDBCoMwEETv
+xX5hh4FD+JJCnqoNykhxhW26MaatODfV6NWUqR7nH2zs0ylTd8jO8nGUTvdQXfKWnFMLIoBOWOH
o9LuEkXpasg9D2SVEAEOLWHXnEiyIavNG8dFmtACNTJgYxE+I6kBYj2i11Z0dVp8vpA1SlY9eqch
+WOeGWAjiS2O7k9CrSxKp+oOzxkw9QO126MWYKtA+A5IK0eG4bv/kwVVqIm5kqFTc+1JWmZFfpig
TG5Xf2VD9m8+UEsDBBQAAAAIABa1SjA5SQcGNgAAADwAAAAbAAAAY29tbWVudHMtMS4yMy9NQU5J
RkVTVC5TS0lQi0lyDguOSeKqU+GKy880is9NzEtMT9XLyU9X4YrRy89JAVFJidkQKjm7tACosCS3
gCsungsAUEsDBBQAAAAIAHYFZDJbsipf4wAAAAUCAAAnAAAAY29tbWVudHMtMS4yMy9kYXRhL2lu
c3RhbGxfc2VjdXJpdHkuZGF0nY7BSgMxEIbvfYo5CLkstIgHWalQPHny4HEpISazJZJkYjJrKeK7
m2S7Xgqi5jAkmfm++a8y6ilZPsEWBlhBOR9gfaTEkk8RYXsPgl5eUbPoWhsgR4pZaqdybu2niOEx
MCal+brvn8/G7/nRojOSksFU5wd4O65hxmeztAayprKt1fY8S6TDd3Swhv1i46RCHil5aXBUkys5
qRLNLBaBKAB8zsxwEXGn2VLo+wfyHgNn0TX1pgNxFLVQcqZcbpa1f1IcqiJbRqmMt0EeEk2x/N3+
2rarXCEuAm3+pfgx0P5u9QVQSwMEFAAAAAgAzn0+L/AmM/cmAAAAJAAAADIAAABjb21tZW50cy0x
LjIzL3N0cnVjdC9jb21tZW50X3N1bW1hcnlfZ2VuZXJhdG9yLnNxbHMOcnUMcVVwd/VzDXIM8Q9S
yM+MT87PzU3NK4kvLs3NjS9OLQQAUEsDBBQAAAAIABJIsTAjsYxu2gAAAOMBAAAqAAAAY29tbWVu
dHMtMS4yMy90ZW1wbGF0ZS9jb21tZW50X2RldGFpbC50bXBsnZHBC4IwFMbv+ysegWhQRtdQL2UR
VEYeJYbZMsO52Oah/741N6KwSzttH7/ve2/vZQ7sD8k8TlMgnDOOKREiLwk4R4SC6zTKHNimKw/c
glFKGin8M5F5VfuykjVxYajIYKJAhBS6XoLhdED2Trf22ewTGINoKc35A0LL+CWR2KpjhQV3KOpc
iHBgiOR0I4UcRNrP9APrflTIz4K4Az1oeQ2gSFPCNwFKRvDPsZW/8jp5qH/w6vNrjFVzYQZ1R9Dj
CSb3bqjxJo21ogbRs4+GYXO327C+3eJlewJQSwMEFAAAAAgAxVJkMkprQ3m3AgAAEQgAAC0AAABj
b21tZW50cy0xLjIzL09wZW5JbnRlcmFjdDIvQ29tbWVudERpc2FibGUucG3FVVFvmzAQfudXnDoq
QAsJydYXUGiidg+RqnZKp2nTOiEKl4aVGIpN2iplv33GQIhL2k3aQ3mI7fPd931n3zmpH9z6NwgX
KZIZYZj5ARvZ9kmyWiFhpxH1r2N0FOUdqLPQBtneT1e9NQz7QxhZ1tHA+jCwPsLwyB6N7KEFsR8s
k0f49JCCqig5RaAsiwLmiHmHkM8fGNzd63Dy5RsYnHPyqijbnl1OofzGIupV58+Y0YiyElb9C+rX
T/PL2cU5R6VpFhG20A8Ow/6hNQoPenCnznEd0SghtshbhfFvGOhX4Xvjqi+GQam8TDBICGU+YTA9
O/NmpzB2QeNTjW/T/Boi6oUVZQgbpUxj9Qg6qEHsU9rj4xKDW09eRSEYXNfEc5oANWK44qYqzHS9
G2RbXK/c1F+GqlAyZHlGKiRHKSp1NcTbKstJjJSW1ALLqNW0iljEYuSBeRYLeh00MvC1Hmhag1F+
0QJ0SWiL04hNrn9hwDgErv0YNpIy010gC5YyROFIGK3USQd/eyBkkZQHUnGZbjV6IdIgi1LGa8rp
xFUpllFltOluxLrY41ieAew48vUzt0LpztqUnp7GdaW2Uc/vkOC9LucmNpoFr3DpRiXPOlvOJHlG
4V63Om23ueN9PnXGrshd9tjWWUKEB39RTDfI0GfohfxHN1r/wjBd6q+5rUq86HQFbNsCydt3RVXN
nY4QFtPNcJXs5lIL7/L8RxI84H6JGTaKuGnC2yZHKnpwF0UWvbcFa6wxaFXEGI5hen7K/duSKW0X
82cGQ2srNc3pshGxo7wnlXR1sxhT/Ad6ietFpi58c8l055bF++HdZEme7vRPTezWEnogQEvDVYNf
4coPtc4ni4YD7/ifynw+/a6BT0KggR/7GUw2zX4BLlhgyM/Fcb1ruj+sn2DzxyvEhajxoaP8AVBL
AwQUAAAACACWWVAw10mvLm0AAACDAAAAKgAAAGNvbW1lbnRzLTEuMjMvdGVtcGxhdGUvY29tbWVu
dF9vYmplY3QudG1wbItW1VXwdFMoLcpRSMxLUSjJLMlJVdBVjeWySVTIKEpNs1WKVgXLqsYq2QGZ
EAWqsTb6iXZc0UDNrj7BrmANII5vsLuGgnpyfm5ual5JsV5mXlq+Xmledl5+eV58flJWanKJuoIm
XLWrnwuIDQBQSwMEFAAAAAgA1FlQMGwdY393AAAA4gAAACsAAABjb21tZW50cy0xLjIzL3RlbXBs
YXRlL2NvbW1lbnRfc3VtbWFyeS50bXBsi1bVVfB0U0jOz81NzSuJTy0qyi9S0FWN5VJQULApLkjM
U0jOSSwutlWCqgguzc1NLKpUsotWVfANdtdQUIdKFOsVQ6T0wGaoK2gqqMba6IOMsOOKBtri6hPs
CjOZTKPVdRRKi3J0FGA25ZXmxsPUYNrn5wK2DgBQSwMEFAAAAAgANkOxMM792CL+AAAARQIAAC0A
AABjb21tZW50cy0xLjIzL3RlbXBsYXRlL2NvbW1lbnRfZm9ybV9wYWdlLnRtcGyVkEFvgkAQhe/8
iokJARLBpEcrXlprPFibcjRmY9ex3QbYzbLa/vwOzEqwiUm7F2B4O+97bxumsHoC/faJ0t0H4A99
C1UfNeT+V8YPccBGWmWc0jWrF8+PkIa7YPZxN9+GsC6WMURSVxXWrsmO2lbC7N8xc8qVGEEC4W42
IW0QkJqd2albYkCW+6bJR37DpnMdzUmcwgCBuF5eNw+LooCL13Tq3wTrYjjZkgDz3iKjwbiP+J/T
wQ838SDhCn5l7gTMEI2voBMOOTFtnr45X4SxeFb4JRx+O/hrocLfGhZLl252E4M+o7XqgELq2tGE
Ul05t3sudB7udtMtDal+AFBLAwQUAAAACACdvmMy6QZjn+UAAAAhAgAALAAAAGNvbW1lbnRzLTEu
MjMvY29uZi9zcG9wc19jb21tZW50X2Rpc2FibGUuaW5phZFNTsQwDIX3OUUO0BXLkUZCgs2sYI+Q
5SYeYUidNk6QuD39SadTGEQ2kV78+T07Ly52HUkGz4ptoFfjAqraq3O0Tz3JSTIldPnucHhYkMeF
eKakrNlgYFS4ov/hzHhZuzcyZ6bgd9oiTfFc/KQ0l32RGvawLz7aOgKwNywu0TxXLaqQgpIriTZI
4qJicm8TfVGVhkLiCAS7qkaGH+sawcFIBBallG9G0Q/uoYin8zbmfTMypfeY6TfR2HmJjY3tO7k8
9WhRCfIa748sRocAow2WkHV1qj3WGSatfsPqaLa3W/G/AVBLAwQUAAAACADOfT4v+zsysqkAAABQ
AQAAKAAAAGNvbW1lbnRzLTEuMjMvc3RydWN0L2NvbW1lbnRfc3VtbWFyeS5zcWyNkD8PgjAQxXc+
RZcmkDioCXFG7EAiDKR7U6GJ1f4h7dWEb28lMoAM3njvdy/3XtmSghJEi/OVoM5qLQwwH7TmbkRp
sl4x2SOMq6ZsSU0aivEuIop7jxbz4q67c5ee8gwZC8gEpSJpbw/Rwcfjhzzmm2hw6k9TkKDEijzk
+wVqgmbfQPPD0kAUYgJgg/UgembNJGB8icXQqiYYz/eDk1MvTzGmW81kSfYGUEsDBBQAAAAIAI1Q
UzCffMsnuQAAAJkBAAAgAAAAY29tbWVudHMtMS4yMy9zdHJ1Y3QvY29tbWVudC5zcWyNkDsPgjAU
hXd+xV2aQOJgNMQZsQOJMJDuTYUmon2Ythj59wKivBbPdIcv555z4hxHBAOJjmcMuqKFlpIrB74H
w0mrEr5CKMniHKc4IwhtWkQwa2GmJzPFlRn/EAagtANVC9GS+nLjxczrR+7COfrQ1vGSagWTv6c2
JUlSjNCIwsAaqpjkS9v9du1qKJesEgsy7MgZVRuxitkX+lC27tv8UbvQynVzTuT4awxlKslMA3fe
+NO9Ay94A1BLAwQUAAAACADrQrEwB0AJfq8BAAC1AwAALwAAAGNvbW1lbnRzLTEuMjMvdGVtcGxh
dGUvY29tbWVudF9ub3RpZnlfbGlzdC50bXBslVPvb9owEP2ev+IUDUG1ErZ9bEOkiabbJAZTUT9F
leUkB3hzbGQbJvrX9+I4lLVSp/pDfth37917dy4G8OtuOctXK0BjtGENWss3CIOHKEq3X7JiAD9X
30YwrHTToHI2UdqJ9ZFJYV3ihJM4hAsKTycUTTm7LCoGY/hxC3bfNNwcryPwS5e/sXKsRlvB9MTa
w15dhS/WxY1gbyRlTXuYJOTT9mVAfO/y1b5G7LYv+jrfkBsSmdsa5PXwEuLU7riCSnJrp3HIWXrY
OPtwpjidtIFZEvc8+XyVv4NR6Z70BLC4gTF1KZ2Q42R56zhlok2seETI4JM/jtJaHIBLsVFUIOGj
iTMfvpjN729ycLyUyEptajRYsxI3QvnMtouV3itHjgWss7Qt1YKGGf13BJKXKNkfPFoKLc6UCLXW
ieINklUvdrHhQg7hgWanI4Pb5V3+dfYdOvmE5NV0Gp2B4nlSiZRVWmpDYweHoM3pHQkDSF3dDm2H
4sn9cNLuq0Nfw+mUnib7R3X3/gif+xJ7z9/wD1Ud2kK+d41pWx16sfvPfaI2+19RcSe0sv3VCi0O
/E9QSwMEFAAAAAgAN7lxMgXEx3TnAgAAbwgAADIAAABjb21tZW50cy0xLjIzL09wZW5JbnRlcmFj
dDIvU1FMSW5zdGFsbC9Db21tZW50cy5wbaVVXU/bMBR9z6+4gnRpRdcUxtCWCoSGmFQJwUalvUCJ
3MQJ3pI4tR1YRbvfPtv5aJK2U6X5IU2v7z3n3A87KfJ+oRDDXYqTcSIwQ544cZzJ95txwgWKIse5
onGME8FHhnEI5th3oLQM0rj/AseDMzgZDj/aww/28ScYnjrDz87pGUTIe6YLuP6dgmkYGcfABSOe
GOn3GZKP+Wt3NzP0JKO5hzDH+XF9Pxnf3cI58JSRRATdg44/6AxP/IM+zM17/EI4oYmjtZpw/gfs
7qN/1Hsc6B9bMcULuBRoFmEuYaQwA+TycooBn0dQrsLmJlSQYKG26q4uz+IYMW2vbD7hCln7Si5F
xfE8w4m3jc0t9/5B2/DZxt8EaQlpRueKOl/HN9cTqUZq8XGAskjA+QU8VFWZ9g2gsg0RhnzDKlFz
q8Ky+qVcq62n7mRAY1mt3Bp41pYibiDUyqlkpmFuruvvt53IrOZUsRA1a2o42/QhTuQQCsr2SWFd
i3Zue8LUpkjjbJSziaMSUkPMsxmEWLgxCeWmHHmZT0BZrN/hTbPJXnfB5DgKoCfbfemOSnNnXYSA
5pNQLF3CSqWsWKlHijuEBMUYaFB4BYzGgBGLCGbwgpk6ebxC4ilNuetFiPMCqXXCr0rgNTlDCVdZ
uCo93a7Hd26cJSF26ewn9kSBN81jeuuEiqLrfBoJ7SvjVgNYG8BlGzYqtS/wJAfYM03+TF/djEX9
vTJnWGQsUfGNjsrgekHU30Ya05GxykeoxdoYnDzUpJHvMvoq3wqMxjCVxvcXb4VOCbNSFy9f2grW
XtryPKcRWtjLsM3bSO1/uDVCQfvUfXhypke9ZaMfy4pbHRv5gco8WTt1QYqCuShmNfA7AgIiR799
wJRCLORTLFLcEFmgmvrSfcsvvBWQQEVqZzwHKzdbrbYWMWm40mPTivkW7vAns63+uhJf5H23I6z4
FKyaaWMvY0Qs6lmXdSL557nyGfioKNrxyPgLUEsDBBQAAAAIAM59Pi8ih/zXIQAAAB8AAAAqAAAA
Y29tbWVudHMtMS4yMy9zdHJ1Y3QvY29tbWVudF9nZW5lcmF0b3Iuc3Fscw5ydQxxVXB39XMNcgzx
D1LIz4xPzs/NTc0riS9OLQQAUEsDBBQAAAAIAM59Pi8UMKQAJwAAACUAAAAwAAAAY29tbWVudHMt
MS4yMy9zdHJ1Y3QvY29tbWVudF9ub3RpZnlfc2VxdWVuY2Uuc3Fscw5ydQxxVQh2DQx19XN2VcjP
jE/Oz81NzSuJz8svyUyrjC9OLQQAUEsDBBQAAAAIADe5cTKT+lFCEgIAANAFAAAqAAAAY29tbWVu
dHMtMS4yMy9PcGVuSW50ZXJhY3QyL0NvbW1lbnRhYmxlLnBtzVTva+JAEP2ev2JIU4ycP6K1x92K
XouVIpRaajkOrIQ1mcTcJdl0d6OV1vvbL9nEVqge/VC4mw9JmJ15897sIwl1flEfYZxgPIolcurI
NiEDFkUYSzoPsatpR2CMXAI7yUYS1ZbQapxA27JOm9ZJs/UFrA6xvpLOZwips2BrGD4mYGhaKhCE
5IEju+r7ivmEZI9OgjyEnXhYmeCjtEPm+8ihWpS/YRYLSWMpVDnJag8XSnyUJe7g7kdepxmHhRLy
fXg7GY2voQci4UEsPVM/dhvHVtvVa/Bg3OIyEAGLiRJuQO83NM1791P1vqFezXxAtAYTjJKWJtK5
kuQUY2yRRhHla3jScsFFrcDQg2o29MzuqrTqfn7u7SzDhKvxpX1+c6NQy16jQAtQZM24pGEJm8d+
mZNifL3voXQWts9Zmpga7IknWC2QI/T6UHFCKvIR3+D8+gLY/Cc60g7cPFOp7e0GyNikqnsKHL1C
ZA2EQ0PKS831foZRhRlsqgpkU6gPvHwpZ9nJq5p8I/U+cs6yZv2Or4PYB8lAyYByubBdrsd4yRL0
A/z0aXajO8T0GeSZF14qM0//AuDQ1F9IUJxIxlfPb2Z7ylGmPIY0dtErshttJ/96cfX+1Jp1tc0b
p4gPssgL3Dsd8n9Z4xAE4272g8gJJExIdG0Ww8VwMqh8lJfEu0y0CuQC/q2TtnyVh1pd7Q9QSwME
FAAAAAgA8T1kMtlUdl9sAgAA5QYAACgAAABjb21tZW50cy0xLjIzL3RlbXBsYXRlL2Rpc2FibGVf
bGlzdC50bXBsnVRtb9owEP7Or7AqRaRSSat13xqQEE07NFoqQj9MCFkOcSGrHSPbbO1+/c4vgUAD
nZYPkXx3z91zr7Og00JIF5pR1EUP6X2I2gvBOS21ivJCkYxRzAqlI2vTRuc3YD8eRmuypNjKQg8H
TSeYt1oz6/JpMh4kaYqolEJiTpUCwE1NoTTRG7XTWGy8+tKbBd5hMI8v4W08ouEd8mxyR0cVfyjq
oSuwsvrHwej5NkHaEs6EzKkE04wuixJ19m1WlIAWS/E7RIxklOFX+q4g/RkCfh/zj3KiafuiQeFq
ctGMWjCiVCOsyI14yURGWGQZRGWm1uBojs4rtnfjSdIffEOFphy47WXvk6alzXYjGRhATzh5ta8Q
9QfT4fgRma+7JYBJzosSQk/76Xcjd3jH//j3PBnhp/6k/5ACxrAB/kATKMRaIqBRdRQKiheCCRmi
hdiUGsyZEOvIPQyiB5FinZsWA11TVvwiJCc6dI63SYoSChT86AS8E+Rti40vAejxljDMmRkLCxTZ
T7rQthKmfJ55TNBK0pfuGcQ7NAvmZ70D8W7qyC5CMkqTus8jmJ394603rxOuUHYmDpI58AjF3dfX
s6h13CfQvLKRb22tcvCXPTM1hqHb02NbQ8sc+VV2+Zter49HcxtZClz1r4q73gsYr65PMDaDgKsj
43b/Gnb/VFyHURvOiXyvxaynZk3sGdhuxYmN8I5BwqleidzI7pOp832qZLtDUzdy18VlRt/0wcmp
EflwNU6vJHwl4dQ68FfG3kN4f71qIPsfPMyN+mcS29H9jIhvWMYLT+IXYRt6lETVjk/Lbya2KZST
/wVQSwMEFAAAAAgAT0JkMsOHK4L4AQAAKwUAAB0AAABjb21tZW50cy0xLjIzL2NvbmYvYWN0aW9u
LmluaaVUXWvbMBR9168Q9H2UjjIwBJKt3ihsHaTZ9hCKkOXrRK2ka3TlpR758ZOUeE6abmWbn4zu
Pefc76VCa8GFO6aMJOLH34R/bsFduwBeqnBRFDMVNLqieLdDEdMkCFTn4QjVA7Eg6UHU0MjOhNFi
NAVm5aMAhTUIox0MpjfnbO8uWrkCQfoHZMPF5S+DQnzQIJy0EA372IV2DTKHQTe9oK66BxUG0hvY
DG68RQpQc3T8tpx/LefiZvapZOyML5DLuuayC2hl0IpnLq1kSpb4fUchBx6hwNFzix4iLEWxBSu1
mTaIr6IMJwjE1+ChYGe7gFIQb7Ux/IMMQNsq/q6mViuPhE3QkGBHvr5TwG9br90qhgtuW6WXaVhD
hUTZnS2HlHLtdejv2FX5fvbl42Ks9LycXTEPFr/DQWe+za8XJaM1bsSgOT6zpYONGKraoLeHY/Hy
MKSWD0JZAmwb+kx0MCiTWN4olR0GraoXmPv274KpQSPPC3rUWSt9/5/p7VlOtQYZWVvtTlfrJalZ
gp2s1vNrtV+pYRCedrrzJkZR68QvTXqvNcnKgNA1C7E/Jo6lIOy8OtgoKorBLbOPGXlQv70Wf1fD
Y9wxPws6GHjGbZ7N/I/nh++6MJ6MHbPCLiIn/DLXxKGDp+SpvEqqNQh4bPUh54S/Prc/AVBLAwQU
AAAACABjuWMy2I/LJygAAAAmAAAAMQAAAGNvbW1lbnRzLTEuMjMvc3RydWN0L2NvbW1lbnRfZGlz
YWJsZV9zZXF1ZW5jZS5zcWxzDnJ1DHFVCHYNDHX1c3ZVyM+MT87PzU3NK4lPySxOTMpJjS9OLQQA
UEsDBBQAAAAIAMVSZDI7+I2XpwIAADwJAAAyAAAAY29tbWVudHMtMS4yMy9PcGVuSW50ZXJhY3Qy
L0FjdGlvbi9Db21tZW50QWRtaW4ucG2dVE1P4zAQvfdXjLpBSbSlX7tcUhUVAQckBBJIe0Ky3NiF
bPOF7XRBpfvb14mTxnHaUjaHtHLG8968eTMp9pf4mcJ9SuObWFCGfTH2vAtfBEnseZdJFNFYXJAo
iCedzjewbogH+mk/jXorGPVHMB4OzwbDH4PhTxideeOxNxpCiP2X5B2u31KwOp2MU+CCBb6YFP/n
WL5e/zi70cGViNYRxDzv1/XD4839HcAUeMqCWCyc7gnpnwzHpNuDV+uBrgKe3yuIWjD9CwPniXx3
n/rFzyCH4tkcwoALWHdAPtE7OGBxGi7AlXlnaFIdWyTgeB5SIo8NeiWtKxVwer6gwn9BzyzJUkel
zZ+EEcpgeg52lQnJcq+uHy/tXhG0cRUYoyJjsWJxev5MY4kjKPITiRgLZw3b+wVxmbHmlufYqKrK
s8OFlSApZjhCC5ZEiNHXjHLhgO2HmHO7B3Yy/019gQJig1vr4aMiQCbTk2zvVaHBIgeugl2oBcmT
BKR9vw2XP3SFQ1h/Jn1ZdI3YKzBc2NSZFKNZg4umBSYEUcYShpb0PS9HQfB+cdjPP5coUpuuNevq
LDc135BTE0Ay+fiYgn1xe2tP9mFzgUXGTXB1qqOjZCkJGHWaTNTb4HJcoRVMnKBmRzc7XErfqJ8J
Ks0pMF8WLs/NaeuGpPHnftQGDe3yRv1N92Kj3b78KH+KIpom1BLrvVfOarRDo3HkuBvpje5qs0Kq
a+viaNOKlNcBmpHbiTCjtyGMRsmKOnr/jzC80khCzb7im68uC1OBQ9uiprtrY9QKHbk06vZ+1kRl
zpaRmpvjkJh7Ba3l3DJqrwZN3r1jWc6lajXaO55tzKql2iiU0/HlpdAENxbg1jT7O6hE3bEEj12A
RvX5DmwJaepqtvT/19ho0vkHUEsDBBQAAAAIAIAETzAm+kn32wAAAH8BAAAjAAAAY29tbWVudHMt
MS4yMy90ZW1wbGF0ZS9jb21tZW50LnRtcGxlTk9rwyAUv/spHoFHOkjTe7Ae2malsK5lnsYoIY12
pDQq6vr5ZxKzQyaCz9/fRw2jztQKmkft3DppdNdJ5fnP9S4bn7AvhAjlbsQAL3TVWxi9WkaoaJ8z
81YrH56EkeA+vIJ+SmtbIatmJEJCIP6hS7z0hvKNl6NiKp4L3neBJ3QVmhkh1Mzqz9p5aYf2JZjh
A2s4f5y2JedTqCuKOFVREtOPfL+AdFLlPdmq7zSLSRmBeE6HXNReVjdtu9ov/rYddKLSKoMUN4Ai
A/wEfBR4BDQpvEC4w/6GkV9QSwMEFAAAAAgABgRkMnSDYMHgBQAAgRQAACoAAABjb21tZW50cy0x
LjIzL21zZy9jb21tZW50cy1tZXNzYWdlcy1lbi5tc2etWN1v2zYQf89fceiApgE6B9vemqbFkHRY
gDYb2u5h2AqBls42Z4nUSMqu99fvjh/6VpNi84sk8r7vd3ekc11VqJxd1do6qbZwDb/SGxawPsEf
2XefQCt6fv/pLE+UBTohy5WTrkQivwnrcOuXOzKls/hORPca0ofTUEhbl+K0miO2A2o7IO/oN9pU
rQVsMIjEMiIiyTtXlUT2jMT+/PHd2ys4f1q6K8x1gU+37oo/Ltuvc6gNMqdw9gp2ztUvLi9BNE7n
BoVDC6VUe7u6GOkxWGG1RkOK3qdXqTb69cQeJzcnonpTUbg4ugqP4HYkvABOgp3nyE5oM9usAYj3
d7TPYY9YA63Y3MjaSa2W+RrFnBzY5+DfmWeN8wwUsaAnKWrpORnB0kXOoCmqKjTQziP4KeQHSWGg
XIa38T5nOCR6uJPVYotjKNpFmmyqqKPlbK2UqFjSPT1GO+jzRXn79t0A536zMbz12/u3o3Vy/S/M
2fYP4W203wP91HhPwa57VPmyNBP+xtfXN6P1gpBKy7f0GO1EpF3Dx1Ei/K5O5iZrCKAvfB8YO6z2
Sh9V1jIIBXENwlq/XEtp3UOp6mj0AU3M0luuONd1g43RVWhLBCVuS6AbB3pD7z98Ws0J8xnXjc24
bFPe6XuOVuFnl+ju6X2WZrFTiQPhQqxL7HseK4P5Jx3znvdkLrh27TyLbapKmFPWyxrCRpelPnKz
rlHXJHInDghrRAXC7ql1U2yo2LwcSZ9/nnExUylQ8WFqNRzE1ZeVkqv/k16RtN5BLhT3BBoiaCqp
cMEG0q364fGxXuviRErtrLp+1KMHKzRGc+288c+wKv9hD/IJ/iIPUb8UQNZurp9wjJ68SlB9Dc8Y
cRcvL8WrsdEEnWNv3P3om3r6Zot9lOowWf1EpYrqjVRpGTiFx92xBymDkPZWU/o+rMKPSj6JSigL
dW4XmCnOfeU6Fi/VvYmvkJfCWpp7XbY7k8ZC0xPaX2vRlBbVkJQTpeZJfT8DmBE76nGJwRs9ZLjh
pSmlLABGon8Jjt/dTqn7wY7UH3lpSulHz2xuqBzGzXfAlJAYmD7UmPOJQaTG6ivquJP5jsrJ7nRT
FikjLXJWsfjuNnDSDZTIuePqP299O4d1KajVibLskr7xRNar5BIOYTxKotlFEdVc+n2htcjiHhnj
T0cI0l811MRrow+yYO/D5mPcGGsQRZGNIHYNP1HfDe0gbo0nlnXCNbbPnOn9qFzamqPqDPb5MXOU
juy79bNmYg2d+WhWZRO3v87pWAaLPg+19JHXOR5o2vixD1HIQixGpodwQDi80kaXCjb4UbE5m423
3vcGnm3yHK3dNGVJeC6KAYR6THEQeF6KpIH+KIiDTNo0U7yggeyp1Oju1wk2MRRD0RPQCyeySlob
bk/UjoTv2wb/bqQh9jR4u/FJpVUWk8wEcWE+Zht0hJE0ufxXb2w9zOq7e7CoQ0kQuoSMGSHrU3fA
WxLj/QtU8xK9ODqnFsN7YFhi5q7d3N1O0E9FJZnzJp4bwl2wlcIY5CNge/2bkzHwqTviBonBm7hK
3oXqTNIIDcI5ke+wWPIuFD25aP6LbKVjaXXRkG0VL3lE0G8xQu9fh5D2JjrO7ODg9QhBBvNw7OkE
bdFBXP4y3NJZc1RF7UGEzyE+LpfUaLbygKrXQluLBaQT31i+P7h3qRnruaPr6oZ8lWwp75Ggui7l
LNDXp/Zq8zDAvNLlIoze+RbaquSbFf/xQHFfxayWerudmbRd/rPu7p1uqDd+rDD+GB+DFkdu5+iP
BIUMJBxTqPyagSCBuAz1uxV80GYmpFPda1E8Xjm3woEeb86WTxfPGCEXYBErG0/3Uh1EKYvH2+LX
s6KpMaPCyvczscfPMS3DGwbjivnA8z0Hq/m6MnCHgzQd0FMjpPUW+PNFxy2G4fB/hnTTJkRHKzor
lLxwCnbadn4E9A9TtGDKYNzNHRRODw2/L5Z9z9MajZX+z5npPCVG4+e8DBq8ERxXzmRM9tm/UEsD
BBQAAAAIAI1QUzCIS1wsuQAAAJkBAAAqAAAAY29tbWVudHMtMS4yMy9zdHJ1Y3QvY29tbWVudF9p
bnRlcmJhc2Uuc3FsjZA/D4IwFMR3PsVbmkDiYDTEGbEDiTAQ9qaUJqL9Y1ow4dsLiEhh8aY3/HL3
7uIcRwWGIjpfMeiaMC0lVw34HkwnqSv4CqEki3Oc4qxAaNcjgloLjl7UsBs1/ikMQOkGVCtET+ry
zpnjNZOH0EWf2ja8IlrBIvfSf1kkKUboh8LEGqKo5Gvb437ragiXtBYrMhxIh2qN2Lw5FvpQth3b
/FGbadUMcy5UCl3OcaaW1HTw4J2/3DvwgjdQSwMEFAAAAAgAwUzPMKnE14iaAQAAsAMAACQAAABj
b21tZW50cy0xLjIzL2NvbmYvc3BvcHNfY29tbWVudC5pbmmFU8lOwzAQvc9X+ANy4lipElXhUCEB
opU4oMpynSkYvKQeG+jfM0mcpGURuUQzmbfMkicdnEOftqCtIhInz1zcNehXPmFUOl3MZsu+9B4j
GUqgrFEkT2B/ASBmiyT3Mbh/qNfZORWP66PXYEgJce4H9gZtfZbrU7I2pMM7xq7siASmlufFc1E6
laYG43XELihFBUSSUOeIE8iHPquiflE7i2JUuKz22dqEnwPJXFDevaJOlWgCcWvSK4cV63KfPAV+
s8Mka5VwhHSVtQweCA8ZvcYO1csEIwfT/BV8kMYTU/zaE72ZRmZf436a12XFmNy0ij8R7KzdXSVC
57rl2ClCmYY+v3kAOljJ9CrbRINCwQ6e29yw9Sk3kpUBATwVTrZjs/PyOYbcbNkv9RfQGv1mrppm
VY1Ep4CzoZcAnTJ2jHK0p9IRVTLB9zs36biFzkbr0/CKVO1McTZ7fFhtruEjxG5pD9eLK8jUnhuP
YGJMfM1v/Ce1xFjOhO+MD7MEZRV9MOH4ehurWH+x3KzubqctwWaxvukmV0rgC1BLAwQUAAAACABZ
r3wxuQ13XNACAADEBgAAJgAAAGNvbW1lbnRzLTEuMjMvT3BlbkludGVyYWN0Mi9Db21tZW50LnBt
hVRrb5swFP3Or7jKUkG2vNVKG1FYqyydInVN1WbTpLZCBhzCCpjapmnUZr99tnkE+sj4QIg595z7
OJcEuXfIxzBPcDyLOabI5UPTnJAowjEfadoHaM48E/KDbhK1H2DQPYRhv3/YGwx6wy/QH5pHn82j
AYTIXZENTB8TaGpayjAwTgNX0MjnM+KbprgdJpiGULnu1wb4mNsh8X1MoZXBX2UUM45izhTcFNj3
gRw/8px3svgtcdrx2wWa5uzqRCUxVvC3UReYsoBxSdR8j+fX9PJqNj8XPCyhQcyXRuPA6x70h16j
DffNS/wQsIDEpmpeE8Z/oWfceJ9aN13105NJRhswoJmXprHUUW1haRQhuoEnTeaZYRgOl9ASYsf2
SB2rqOfncaWRBpzNv9snFxeKTYLSOMSMFfEd64lHie1mFRQyW0GbKeVqTfnCdkMkIseynx0rJOQu
TWzi/MEuN0B/QaFLwZcMiAZYEuAHFFYEVO47hY61xNxd2T4laWLUYbVrvcIUw9gS4nlmX+Hk/Btk
OdmBJ0/09vsEIo1UEVyX3VBM23b5v+Tawi1sWyXXdlddsJTdPK71rBhHx8KUEjGGxoJugtgHTkBV
B3m/oJjsktDyrLEn58Y64CvQ6/nqu4Ndwvp+Hifl4KLUX3FQOZqihkZ1atvyCYdixV7OS8mxnTeN
yow71nX/9jVXdqeYpzTe67+Rts28z/Z4v10IbupbsM/X4zKmlHBJKhA5ktVUMltVyZWT78PKhoN+
NT2bThYwmf88XxgfW3B6Of8BB0xv5wQdy0EM2xw5IS56InkoWb/ehSLEc2xRh1ytet+luLiEZ2Ue
9fl6DhTvchY/JA4KbQ9xxEhKXWyvUOyFuB6XD0QuEhMeDXFlZ3LL53Z/y+r/++pUcR1rjWgstuEU
BSH25DaIKPHDRRPiNHLEp58siz1gNUvWzSOap0ymxjgYaf8AUEsDBBQAAAAIAMYBfDGSN/7PEAEA
ACACAAAoAAAAY29tbWVudHMtMS4yMy90ZW1wbGF0ZS9jb21tZW50X2xpc3QudG1wbJ2RUUvDMBSF
3/srLoOwDrb6vtnB2FodqJVNn8oIMbuTSpOUJEX996ZJW+aDL/bp5p6bL+eelgSeD8U2Ox4BtVaa
CjSGveMqAhgVY5ltzSABOUVRSWCfd4I8s1pJBFeAaYVg+tvfhV2Wb14fXkC9fdBKXhSkUOwTd0Ju
6RkN11VjKyVjCD2YwcKRbxvgNTMmnXAlBEpbeHWy9tCSLOAK4ZiDx37aLJd9RcNcDK2u3c10NJK4
xtzT/vPZytZ4TQuN2WokPh7vYpgOfhI/FLxM57/M9wvfNOsuzuxp588+27w4ZJvt/bCVe2/g+ZmS
/L03BH3EudrlzniXNcUv5K1F50/iJx2SuigtRm9ht/BDyOkHUEsDBBQAAAAIAFkDZDIyxYr/TwEA
ADEFAAAWAAAAY29tbWVudHMtMS4yMy9NQU5JRkVTVIWU226DMAyG73mXgrQ3qHaQ0M5i91EILsuW
U2MjrW+/0CatOiXZHfj7Y//YDref3MyAzfP2pX+4Hz7OD+3w2L81jotvPkMrjWyENbuOC5LWXN7R
WYdMWK3BUCHMJol8VFDCxpLcHUoUF625P+GJE++kQeJKMQSxeEmHNkQbjXMXT+BGA2JwjRswbQDN
qwPTGwIf3N90t9Gs03lwF92W+HAyNByMKGryCbbH5iUN/ifYTlqajMi5Sorh/ak/NehKhOQXQalF
Le7VnxCbwYQUZH0OyjX/yBFy0IbKKksQ9gsYkWVpKcqoejwuTZHUvydqys6joOYgLWYZ1T0kUaV9
UXHlgkA7xQm60f6cL4kHsY6VAroI0rCzUTYBcakKcGe9riDm1p9CniuJpZIrqh2NTa9ksOMXiBJ0
FsOWFmCa1jVNm5YpeTQjBV9vJAOdevULUEsDBBQAAAAIAI5QUzA1prUwswAAAJ4BAAAnAAAAY29t
bWVudHMtMS4yMy9zdHJ1Y3QvY29tbWVudF9vcmFjbGUuc3FsjdBBC4IwFAfwu5/iXQYKHcKQzlY7
BHUJ72POQattLzYN/PZNs9S89GAw2I+393/7C80LCkW+O1FAxQQaI20NcQTDlakKPqXCi8VwGq1X
QWjuPczqyZ24cpfG2yyZUixvUsx6jTTN5vaBvpYVQztaQg5hzOJ4poSMFAbrmOVGLvpu1su2jknD
lf6lWUdnrHF6OWkf6s180yf6J7tAW3c7nZTQWH4/dMpw18JdtvF06UmUvABQSwMEFAAAAAgAzn0+
Lyl1rRqZAAAALQEAACcAAABjb21tZW50cy0xLjIzL3N0cnVjdC9jb21tZW50X25vdGlmeS5zcWx9
zz0LwjAQBuC9v+KWQAIZRCjOtWQQaofSvcR6YjQfmqZC/r3BCtJSvPF47uXeshFFK6At9pWA3hmD
NnTWBXWJQLPFplNnAEIOddmIo6hbQngiWg4DzOYlfX+Vnu5yBukS7Kh1ku50wz58QpZym88pGqn0
emi+mUkrDcIfOamHV0b6CHeMdKUUS2S06jkinfrw37P8+wzL2BtQSwMEFAAAAAgAzn0+Ly3haTol
AAAAIwAAADEAAABjb21tZW50cy0xLjIzL3N0cnVjdC9jb21tZW50X3N1bW1hcnlfc2VxdWVuY2Uu
c3Fscw5ydQxxVQh2DQx19XN2VcjPjE/Oz81NzSuJLy7NzY0vTi0EAFBLAwQUAAAACADOfT4v+1S3
syAAAAAeAAAAKQAAAGNvbW1lbnRzLTEuMjMvc3RydWN0L2NvbW1lbnRfc2VxdWVuY2Uuc3Fscw5y
dQxxVQh2DQx19XN2VcjPjE/Oz81NzSuJL04tBABQSwMEFAAAAAgAzkzPMOPC4za7AAAAsQEAACwA
AABjb21tZW50cy0xLjIzL2NvbmYvc3BvcHNfY29tbWVudF9zdW1tYXJ5LmluaX2QywrCMBBF9/MV
+QaXQhelKxHsot0VGdJkhEiT1Ewq+PfWPqwtapY35+SGWylvLbmI3Fkrw+MMqpHMYjmJyFtyBxcp
SBV3+302GsUogGEpxIqHi6FGf4lQG1b+TuEVPYjBaFyxidh8B40G41SgIRzZUWW6deQUoZOWBtUb
/LSxJ8B5NI4pxH8FtWTCKOuGfkDg6yupOFe9iGkEMa/wvvpXBNUmFP0gbSP71dOsPOSnRYMyLY7D
OxMyK09QSwMEFAAAAAgAzn0+Lw8RMjEoAAAAJgAAADEAAABjb21tZW50cy0xLjIzL3N0cnVjdC9j
b21tZW50X25vdGlmeV9nZW5lcmF0b3Iuc3Fscw5ydQxxVXB39XMNcgzxD1LIz4xPzs/NTc0ric/L
L8lMq4wvTi0EAFBLAwQUAAAACAD5Bnwx4QHe4ZkAAABRAQAALgAAAGNvbW1lbnRzLTEuMjMvdGVt
cGxhdGUvbm90aWZpY2F0aW9uX2VtYWlsLnRtcGx1jsEKgzAQRO9+xV6CLZR49wd66qlHkWDDSlNM
Islq8e+b2EjFktOys/tmpmFwu19PUEqrNRry3FhS/cINvkXSygt4dDM6YTqNcAbWFkXDwE9ad27h
9vFCSYIUDbjenkRjXVXxZcex9p+Z3LC5HWoo01seuXINrCG8pCMfraefaw4O3nk2BWfQbfvyxQ73
01r8oEprKMyofgBQSwMEFAAAAAgAIAJkMuigNdWoAAAAUwEAACgAAABjb21tZW50cy0xLjIzL3N0
cnVjdC9jb21tZW50X2Rpc2FibGUuc3FsjdC9CsIwEADgvU9xS6CBDioU51ozFGyHkj2kacBommia
CH17C2pNwcGbDu7j/sqWFJQALQ4nAlYxYYdBGs96NfJOS0gTeKdM9fAJhKqmbElNGopQtpCeWfMl
x7kxrWqCEBjrwQStZyo0H0dYxYM7ceYu3ec4lra7SOHjsYvc5T9pcPrPpl75+bi13OabFb05NXA3
wVVOafwEPNeCUfcg09c1WbQqTvATUEsDBBQAAAAIAIAGZDKDI4NAYgoAAAQZAAAVAAAAY29tbWVu
dHMtMS4yMy9DaGFuZ2VzdVhdb9s4Fn3vr+A+KQEaR5JlO1EWXRRNiumgabpNin0ZIKAlOuZGEjWi
lNSD/fF77iVlS1amaFFbpu73Pfdc/lAv2mpTia22rWl2YmMacVer6kvVqkZmrahl9iyflMhMWaqq
tbN376JZPBfic6PFrWyESEQYpvEqjSNxc/8g4jBcvHsn+M/dl29nUXSZCpnnolKvAhJJW+ClPcq8
1FUgrBE704lMViLXVq4L5QX0WgXMwv+6gSGFtFbhyebwq6xyoaus6HIlAm0fvZA88GI2XeUUQ9Gr
Yj3ZVmXP4nWr2q1q6KHdmq7ISX9dyJ2QbK/X0IsxTen8j4X4j8qd/7GI52mySufLv/E/TqFNVggi
dIlAlVIXAsaQgnbbKJn/KxCtgcq8MXVuXit2qDS53uy8JHqzgIyOckFJcg/WqrBiLS1sgcDeG8Qy
QKRkQbJ7CbZb26zRa5VfiVxlOtfVE2ntqv0vYtOYkiU7s8SrLgrRqNK89AmB6EZUptUbnUkK6RUE
cCxZHPn3Q5WqXMOMUsEvFlEZpP6XGviSGfOslQtmJMQtrP+s1iK+QPzS+TxdrN4M5gqxJGtEbjLr
zMWncyQfntx9idP0Y127EmHRoRD3nRe9EtEyDf++TuOLVHxEnZJ5cq0L3e44K3hUN4pyL9sWgW7V
r5Z+6KvPy3jRUvxTZSZXH0Qrn6w4KfSzEp1Vs1o1xelI13y1rwnfYLPMVBtx9mH/XVeaXIguhXjo
lLhWmRBwYZUm8zRe9C4k41ILU/FZ/xLr7gmBb7fCtrLtLDJhLWSyvAuERLZOHlo3SuH3Yj6R99Mq
//a5ahoUHPqjabOu5cBGKxfYb+aFcxali0UaTq0iYzgtn1ywzj5kpkPf72NHspYjWbAoSdLwYiLr
h6pkifqCIa8npwHFKvDdSl915QuAO32v0alYjFWEaZSkcTJRcSZ+1rlsUf+qhNxWnfdAVQAfZy0e
UuYLFG9Xc6GY9X8VUFJXG0NtOajwXFFT1WTM7CD/Bt3WqEELCG2pxBQVmaFC2VEbV3wCtdMIaZ8t
KV337dP4/lL5bCD4lrpCotfopbMXrV4JAZHCBo2JYiyNbSGbc2lPuXX2lfNG1Ehjoyy6PQMs6EJN
fTigL30BXlBzcNVlpmkoKq0u1V+mUpQbCutABhXGEEaEQ8U+7oJSfQV/ANjKqfOR/vnjqxeCwKFn
qBs7Wei/oJxQ06EVh6+Gy6oJrJedo4vgVd3oF5nt9gOmylRTubEWJa7XfkepoGRQ1WGcAutvrsdl
8umA5qbJFUV4j8mUQ/YIIdyiz6hwDqGiU9JLqWXT6qwrMEO8b2dnwmKMseQdI7i1JtOHwMrBS30U
ELLB+LKK3y4p5PsY5M5M5+N85COGU5wAUiY+XpsqwPTvWojMuobAEFnyjniDLVn8SmBMZ7fSxb0f
ljLbY2jHXXV+yE2JXO1e5W42c0bFbgSQUQA59Odixf15ZNR9SRXu0Qxl+Ytf9vPjFlMb74BuJHj/
YvLyrX/tWe3oVY7QUZtb7vPHmvCXmv2Kclhx//UWyAIefNOFRY/2Oa9lm22dI2FPjGBLJIDHi0se
ONdTXLz/91fYAIwtCrQ599vGfTBIZMvtSxpK/UQVhRrraVNPa3wWfJ84vuBmME/AwEMKpW7jIcel
D40yJliPtitLCf7XJ3ZoTjAkhHHw/m128rYkdF/gPj12TRG4kR0g7ecE4+fBIbL80OM5PycDCoYR
SAdI+YE5ExTo8PKQdAyj8CLFSI2mFUOBzhBh9uiRdD6qXHNwT07pmVXEcRvADfyzVBSeozqre8yV
TKyoGnDaWl/DhH1PHVfOOAlA9IP4Q08Erh0f0S8oQYU0kwhxjRJ8AFaO1PAccvA9W7PLF8cuM+98
02XmKIDNBtOWid4+BsH3H3efbu7vRYNQZKYwTeACion+sO2YKoFzoHLDBDGdjMge+7g8eSC7lPal
yTqMftx/PzmuWkBU6ScafMeQUc0LARwwinvyzP5JbAkmLQ/EECQhilIQuGRqEo8+9tjxV9/U1rHC
A1oUjm+GngwQA0LFIIjwNNqLne/BRkMO1XW10U+++Pj9xJnFZCIhRga+Oo8n71Me9tgJWaBPot3V
3oj5WMgyXeDv20IGQEHMHdgvPauAHL+JkBykDc6gDxbLiZwhCWM6DtLfb1KACAvYUzR52Gc65Kby
k6ponKkeLtY7cf/97vu90x05KnmHHgWtjKB7yYG8HusmUu1BDCKJKhGVps89aYeHX5yDY94Whm5Y
3atazEOK0hxwGk409KzNjHbXGFaGs9gT6I/gxPFSxDFT1WQiw82WflEzGGdVTsCMEZcboh1nG+16
aa2wBzTi5Nvdww39TM/3Ux0A0BUsAG+1WKv/ccpWXLii+1hjY3RWJLzkHFnx2RDhOvugc25ZV8IW
H8ESPNtxXvl25Q10ybFHx07r59iruy/nvznHzvtAz+pSnDhvxR/NHxX9G0wIPKhOr9wGb1pajQxI
QVcjn/0K6XZRdFwrfnu4/ZqmN1jVW60Gc5P2NWf40hUNGx5xSsH2p91HReN2dOZMIIi/UKOkcEQb
8Vf22+qaljQFQuJF0LpW1m2/xLmNHnTU2bE4toPQdP5mA6JrDUET0LjL2v1OwIbsZoCrQbjePPNo
UI+AHxy96keELfTTlii5G55ILfGqzWCiOtpq+TlCTjcGA9cxcdRhmXiRRceXIv2kGW/hNLGDb2yL
f9AQN6y7gsklRPcwiVWXlgcXo+Q4RmBmydu5CsYeBz2nJc3DjZqyWStTux55Uv2Eda7RHMD2o15g
OGWL2K31xjKR5WsSl+8ZGgA7AObGc7+DrxEmbYmMbroCoYVNUISi3Slm16ObrUqp3A5t60Nvhuv/
n50CePjo9GOLc+Lukhz3zRXdGVExjnkVm+1iOXdAzVcRS5460WDLHU8NXfSr2pCl2n1VudgyRWXZ
8Vg2Q1wSvim70KVuCVtoieAWOtHE0sIzvkZ679xy10cg6FbYZ13X6LvRRRT3OixpdqecYYDBmn9y
7I9af39J5dq2X3r8kjsTH0GX0j58/tKNuStXoOns/yqAXUAYWj3tRFcVyIGXgrhT9pANXiBCvkMa
RADciC42FpMI+EHRX2ryLZKEiW2Lelxrnyo/dVjYgoUBXqM3wlnx4ilotz5k6T2hDQGVbXeweauI
Asonew48ajRKVDkkjC4H914JZ+0ynQ9tDvm25uhMOGBl7szq+Ew4vD9zZ5bjMxGN6sXF+MxicmbO
w3Z4JpmcORo7IW+Y4zPxYkxtwv3CNzgTg92Nz0THZ4gBHtkTjs6EdJWFMI7OhJfHZxJsC2Pfw4vJ
mVUaju0JV5MzmBhHupbHZ+h27UjX4vhMdMG3DMMzyeRMAlHjM/PjM2E8rg1Xpg/9nYpjAda3TDhO
Qbiia7f5dAAyuPPi5De7YMiOTrZq914wgthaZQ7C/d2DW6Qc/2Gu2N/EzimZdGN+OdH23XemfVYA
brpx8BdLYJ7YKkpZ0W4uPvmnugLPkIV4IQprqn4v3gy70Zv6njYq6eczxiZ+UW2GUPwfUEsDBBQA
AAAIAClIsTA9iSI//AAAAJ4BAAAuAAAAY29tbWVudHMtMS4yMy90ZW1wbGF0ZS9ib3hfY29tbWVu
dF9yZWNlbnQudG1wbGVQXWvCMBR991cchFAHa93zbAsy2ynohOkehpTQ2ut0NE1J0oft1y9NVyfs
Pp1wPnLuPTAfiySdv633MGcShAjbVeAgb5RsSJkL6Rl8lo1GB4ZViqMUgmqjA335tvoIDz0LhE1s
JZvd8wTeVVVLPmAPd2BZOG1il5Wsd8lgtc90+5rMn5ZDvm0y+GZWYEcWn3Q0vFUV/sjggwzXrRC5
+gpuFF0uXK5bJijaqiJjv0eY46zoFI0td2Ng2bhrP8Tq1lGubx5D1uhL/BvrsRcrc0P8JJXIzeSa
0UhtqOSyvofH3n0mfFaCLR/Zpr8EwkJhGvfrJy8L/B65g13/H1BLAwQUAAAACADOfT4vt/vutaEA
AABTAQAALwAAAGNvbW1lbnRzLTEuMjMvc3RydWN0L2NvbW1lbnRfc3VtbWFyeV9vcmFjbGUuc3Fs
cw5ydQxxVQhxdPJxVUjOz81NzSuJLy7NzU0sqlTQ4EIXis9MUcjMK1HIywfi0pwcHaCKnMTiYgUU
UJZYlJyRWGSkYW6qiaw0PykrNbkEZAamUiNTrGpLi3KINbYksyQnFV2poakBitq80tx4qJ9gjgb6
BygB9EVJfEF+cUlqSnx+HlhCVdUFGDYhnr6uqqow/QVFmeCgyU6t1MAWOJpcmgBQSwMEFAAAAAgA
UgZkMkGvj6gGAQAApQEAABkAAABjb21tZW50cy0xLjIzL3BhY2thZ2UuaW5pTY/NTsMwEITvfoq9
IE5NoVxQRCukniKBEBSJAyC0TVaNFf/h3TTK22MTUuqT1zOe/eY9YN3hgT6VQ0twOmuovbXkhNWR
ImvvTsJ1sbpR2Evr479720bN8KadJDfc1Xm8r4dpLlLWRvXRwPmCViSUy+UwDMW5can423xpx4LG
UEzGp0CuyjrWsirL3fNDNalluZ0hG+I66iAT6Bp22gZDcwn4awmLBTgP0kbCRrsDpAodRov5ElHS
E8OHSglpRjdKm01Gd5T+oBRQySWDJUyR4oFsb1CyRnB7dQHsTf9LwH0IRlMzZe3HmSOn8chClmHQ
0s5cPO149EfcJ+rXMVAmeEmQvgD1A1BLAwQUAAAACAAdo2kys/hQhTAKAAAlGQAAKwAAAGNvbW1l
bnRzLTEuMjMvT3BlbkludGVyYWN0Mi9BcHAvQ29tbWVudHMucG2tWG1z2zYS/o5fgWmckTRny4nv
OnOjOLrYjtr6LrE9ltNeps1oIBIS2ZAEQ4BW1Xv57ffsAqQoyXb64TqJ3RDEvj/P7rJU0We11PK6
1MVl4XSlIncyGp2V5Wh0YfJcF86+EuKZPLiMR7J5Mizzw3v5cngiT168+Pb4xZ+PX76QL16OTv4y
+vavMlNRYtZy8lspD4SorZbWVWnkXvH/zxV+fFn16dxU0PiAbjnw7+6cvFcFbIU5B0+ZOxr9OLmd
Xl5fydfSllVauEX/m+fx8PmLk/ibQ/nl4FbfpzY1xYg9OJCv/yuP+7/Efxr8MuRfx9D+5mkNk3/e
XN/eSWggT9LCOpVlZLXI1/Lg6uz9BEe9KLzf4wheGSeVzLVLTHwof60t/llI7aMQS1vPBf62wv4l
JP6DuL48WOm5TZ2exWklB5D8ZvaqOT3IOSh4+GCwjsaFhoW9IHVW+nz3yFaSEK4fjUtVqbwvu5rg
3nhH98OXgtBZlClr6dJsdnN28Y+z7yezWXOn0q6uis1V/ZuOaodk/kew2zAzuBze3E9A8MWHd9De
XGo3m6O+Prf3v9RptVc85/TKqyc08Atf0zErVK63DeV3+c2HTK70MrV4NFvgoanWM7cudaNgP1Ti
Jf7OZpOrt7OZEK9LE+NnolX8UtKNh3Rs6lIeyWmal5mWofLSYtkkpxUz/Xh1fTO9nAohn8k7jUpD
FU5v8Ez6/K1NLVeqcNIZqQs134izcmEqmTrZN6UDgFQmj45IDF2JIMa6lOTF8eYGhODAzH/VkZPp
gl+NTdFz+CldktrBCMEU8mfE3X4SwYa9ar7CqZCpVVSVu4fBfzJVyOFwKCQZ9b2GC0lrPACW56pa
sw+ty8EwVcQyTm2ZKZxXJpeKJDiNYCoHoafl+OfnVKN26FKHiDz/dDqvxjA7PC0NkhzPTLF3Qj9m
AFBOJ8flmIzD2fXlEMYjiLOABIDUJmY1C+bOgrm9w8bG1ywP+H/+iSP2TH6nXZR0fbRfc+7/7slp
cjJuyu/0GP/4o+7N1zNvXsfBcddDcnCKGzKHQcBahFvZmopLb8rrUMZ6oerMyaLO57p6VHej1svp
7epA+3pKzR8Ve9jcgJt1QSmDYFbVoO/t5AbYnlxdfJRX13fA852RZWVKXUEptTyCRINZBguiVmex
TNS95lRTQcLSe11RD5NmId6d/nD3/h2aEgDvUm3/vf3PcdNSdDyUl10EQlxBIRcbiMzz1JJcK1cE
5QI9a66lqp05IrJfVqpMNNrXKklRejB1qQvAMMvWAhat0d3jYcfZ6cXt5c0dejEc7TrGLOUrVkm7
xVlW2jUqMB/Ku0SDxnGrMMIlFUSCzw5lpRz//qyqXEmu97U08KWSC61AyZpIxxtIvubA9FwLFUVo
uSZHWlepS6C3ojdybS1ZNDeqioNqmaWftZyCiRKS/+H8HIEDe1mR64YY85oS0QUfkW24H9wMmt6b
e2bROzA/5Al1r9KMn8ArUy8TeQvXDOI2weTU8hUcR/riwKBdRHNopGNxi60TERttKbWIFnIC3x1Y
kq33A0dEEbdMydVSFenvKhB5aVL2TCxhUtkmY5PM6/O/Ty7u0DbOT8PhOCSVDE14sEuj1kJ2JQC7
0mWlbZvtYrnJNrOtUFT63szIgMhRsChacFnO5kk1N7Unc2anSvapDR8iC4jkofxw+25wKOg4ppwg
Jp20HJLOesOE3ZZAyvCbFavMGtFq57e4FdGd4Mfl20b2Fr+21Za6nhWIOFzXBIM2Ug2ZdyKmKEXx
kSmQpUZOojr+N+1qLw5im+xRbJVLI5RjtWXWSDJ5Ejg2zZgRpSqgAV6tCP6eUlAaptBo+9ZHOG5D
sjFIARbFZ/IWd0TT0eFyMDX9nRDQhNg3Fw4XvGqqot+6L2JdmFxloQIP5RwZLkyRANWLOqMyiQIb
4g/XFlc0KjfN89o3+yB2IH+CKxoEBCM78GESP65Lqor4uNK5uQeeML9gHVkuQaAiSlSx1DylpIwM
a6KU3pY7mWs86CYV5JguKKcfULyWByBlOUCgTD4k+K4CyTZMYyn+IcqM7cBtQ/kxTFEcdhqiuqn3
YZe+A1FSFumyrvwZ3hKUPpRDbqpA196AiN9AigxzId5gpoQR2Uqtib6dR5FlQ4FEMrWJIU7bgmiJ
4OyCCP1BIiiirI7JQ7SOYG0NbolMrH2xRt6auoiRK0quCFQ5xOTqE44KRUl5XMAnIrLGa8QAQyNz
KpXgPrxktyo5jMEk9E/IhQ9YwrQvyZqTtkoMO7jBSNto/YxLSttHnt4FKknTtEHPsBV10gTYoNci
TFT5c80tAVM/VV1BpXt9Ka3mBis8cjaHmzZoPqfaUzwvGy2d0XMq6zb5BIG+DWNuiBGvZJo4sudv
8MbSG3iAzdftwMRk1Q4w5EMP8dmqwUaRJgO7vBl0gVvEfq2x6dYbf8y1RffmaRFKkQcLGip4xAp0
0pUAnkFCFqGzUz16hHAVB5IMyWpgRNkVLDhAm1dr1t2VPAyFGuxPqBNKR6DtbYO65/WEqdk2CsU2
qPb419sp+7udY5bGAwoLk5lQcZ4Wnvl5NEfL5kGLy4z5OsOmyQMCM24Kdgz4LTSHTHDvD0lpM25l
v8eaeyy31+ruDehyZXW2YAJ7aM8Ahs8oZHMdx83OVwKAhdsJxaMrlVym92CQAEE8oaf0Nj9/qJvu
G9NuBTDnbaOR2GSn53mRX1UltlVtT+tQcbHtI9Wd3XTEYq/OEHHQGy7TJMrDhjBVTJMLo5VXkIBQ
VxmauAlxW0naVBqvCD2xh9u+2mAUJV/qyDeSFoMd6h80tHwip26daZto0MEF+a8tFTvxbpaZlWei
9o3Iv8GtCMrSgkHEw2gzaTXLrx11QnfhBybKzuWPUvsZHmVQ4WoRN8XB7+yMYd0EXDcpvnlEBAW5
Sj3/BjEhkz79xHlYfNohVfbPOmTfjCrDQUfnDQ+O0Dm9Obt6RO3+yNm0B4RHPeTJtG5ceUJsM33C
k0eENAh8Qkiv2bL/1iPn9KYf3354N5lOaDIPldD9HILl8PGPJeMRBqc042GFBtIa5aExMPPU1R3a
hQ94+HhpPVqIqbbmYOobvMIaLGKO5+HuSCV2WYO4YJPCZjI8PyVrwke8eMwZp/1YhWoQ3Q3ZrwzQ
y9tNrrC2YbzTmBapO1NFOI2d1fEqDTmrhjSs4DGis+bQCClL5Bx3/KzVS63qcda2PpA13yus36ux
+PHAyIuM705MP9yN0gKdKfWA2gAxhJGBRV8Wd6i4P0Ap3PLHRZ/7xwedzsAaxiLEhm1tZmPa+Auz
M96sNa31xJToV87PxGxv+KR5cUoT2mI83LHQbplGY0JVqXWlFxSlzTjYBuir5lFjqzrHuvsZRi7S
yrpNoU8nE3n2bnotRHejHsl3p4lz5ej4eLVaDXN/RA1yiBX3eLyZWz/c/XB9C5hcJBXM+IlqAz1z
cpq5cUSP3kQr/2wIRyanS4L16wgL1/8AUEsDBBQAAAAIAM59Pi82WiENfwAAACYBAAAqAAAAY29t
bWVudHMtMS4yMy90ZW1wbGF0ZS9jb21tZW50X3Bvc3Rlci50bXBssykuSMxTSM5JLC62VUrOz81N
zSsJyC8uSS1SsuOKVtVV8HRTgArrFYDF40uLchR0VWO5bBIVMopS02yVolWxKVGNVbLDlMlLzE0F
StnoJ0KMd/UJxrQhNTcxE80OkEhJvhWmgRC1eCzTRbXNFWwuiINDMUShnwvEfn1Q+NgBAFBLAwQU
AAAACADMQWQyj5+/f5IDAABFEAAAKAAAAGNvbW1lbnRzLTEuMjMvdGVtcGxhdGUvY29tbWVudF9m
b3JtLnRtcGzFV21v2jAQ/s6vsCqhUK2FdlO1SbxIVUunal2pBv2EqigvB3hN4sw20G7af9/ZeSGF
EJIKtHxoiePnubvnzj57XD8l1/2by8e7EZEz8IF0yeC2qX+aIWchcElBtGsEn4cfg6v+cEiAc8ZN
H4SwpkBO60+12rhObm8IFaZLhWV74BI12gmJ5dFp0D1yIJDAj3odu4dTvw+/NojhMN/HYdFMMAY5
RlinZfc6rbCnSft3w35ENfuYg5ww7jcllR4kWJyGk126WLes6ZIIFM60YUqDBrm8Gt0O7jHuhNYg
o8vhNzVgua5xokPPeQJLq2UEsBTGCfFBzpirBh4Gw1HkTqTL/dXd43WfSBWjaTPuAgc3sq7mjDEF
DpsHErFnbbLQfiseyUIjUXc1Jfr/gZzrTxl+z7LBM3VkEl6kydmyEQ8+wys63M0IR4MJa6oItseX
PphVDr/mFL1GjvPdAK2MftBkyATqb5a0hdHPIYbGzjYzDLsJBP2d2v50tnu+b714EEzlTM8nx22y
L8E35Abfol4JDfLkK4st0E9TVBPwoqKAF4cVcM7fK185ZIF4SHBY6T5fHFQ6Mbd/giMPvtrL28kT
O0b/f6U7kpNMu0B1TYd5jCMs6SseTCR2FUI60kW0J0ILRz/qIUIUdYIWM4Tr3BSmKHnDXrKWA+wl
yik01ZJur4Z/uWpyu31Me1+Ol5liSsvI4mA1MgmNewaSBMUpXSUzm84YWQREr8XK1peiqcr71dTi
rC+5FWZCWFAu55ZnVBYSlVsTLkdapJtgpLoou0en50fbDioBM2fS95KjigIpA1lf3lWSefnd4gLH
c51vA4/PJ5vKrZeFMwPn2WYvaVmgnAmJGW2OJRZ79olrBXleQVTEhtR51mtizYUkmt1aRifZgEk6
eTU9KtTHMfkTe5U4Fa1TfM1PowZjNnG/UkL+zQ8iQxow5CxJOg8SWvJUS7Z9fbbGceFwaisFYk/b
ieV9RIQTd4T0zqgUcTYs7XX//nqPhR1ZKl3WAjxsNBt73RuRK9WmFj4lymSjColW1pxQ8PRFQr9W
8yI6FiQE+rUSQbzA4jDe1lzeGtvfhrXRkNCyT9+kSN3IHElZUBBSJOF6g9UlEnJYUFjGNXJ4c7gz
GVs2pm2XQgjc1X06692Mui4EjfTS6XiWUKs69SHbdtW3/CDXaZg+b5kUa32DJv0WX2VVw+K+6pct
vF7H13Ncw/Wnf1BLAwQUAAAACAA3uXEyOFfrR3QDAABvCAAAMQAAAGNvbW1lbnRzLTEuMjMvT3Bl
bkludGVyYWN0Mi9Db21tZW50U3VtbWFyeVN5bmMucG2NVVlT40YQfvev6PKarJz4giWpRC47uByz
oUIwZdgkD1RNDVLL1q40EjMjg4t1fnt6RpItcWTRg46ePr4+vlbKvS98iTBPUZwJjZJ7+sh1p0kc
o9BXWRxzubnaCG/YaLyD1pnvwvOzXhp31nDY+wBHg8GP/cGH/uHPMDh2B7+4xz9BxL1VsoHZQwqt
RiNTCErL0NND+36eLF2Xbscpyggq1929A0vULEqWS5TQztWf4RRKc6GVVXdJ93VFjQ+68Du9/sfo
NVrfTNt1/5otrs7mFzAClcpQ6MBpHvi9g8GR3+zAXWuB61CFiXBt/i0Y/Qt958b/oX3Ts4++iRNv
wIFWga6hsluQWYSKsgsocCI38NgwOed6XsSV6kBLKmhT2BM2tIdpplZw8mjk3fFjmijNFF8jIw8U
fwvbDtx8pwgz8/I0mMrzyM0l6kwKYOxyMv1j8nHGyO02B/OSUQ2RwiioYbG5fP06qnTIgfP5Rza5
vLQ5Frat0tsod9IdG4MdLquWCSqFsmEK5XYR3Ebae/jfZnXHAu+d9nBnaKInt5+ZraaNXyo+WtF2
j6kQDJ8FpSNygZ5moV/V3wufxSM5KeKaR5UkrM8dmO44QO2tnFfCtHdmFe9hYAp0UitN2YjuGKVM
qAHN08nZ+afFjDjKhUi06bkMcY1QAJMYoEThoQ/NTs1P7WoGMomhGIh+AdKl8E2oFPjlQmUy2tLB
iNrqY/AtbR3qCE1l34s+f7/X3u7eMCIy13M2lQ5FkJiG5H6648Kfj8qTYWoI8Wag1hfJjeDNeEuj
XPAU+Daf7Io99V0zw1n0mSHrfpb2woIP78AsHshS0Cssu0DPjO73oV7RcCHxwdiB5KQjSZGLwjYU
nkRjEYolhLoDnzPSCwV4XGGv16tQ8y4iGHe72l7NzmfTa5jOP11cO9/vp/B0Mf8TDlT5/ffvs8UM
Slb9CpOL32A3vUaSV2BYXR8EuhicJ8SwR/vdECW3PGI+11wlmfSQUV5+hN1xKjHlEp3XZ9Ze5Ya2
yXVKt5rfRsgEj5Ho0x7WghN3HtDLNDpPlkHnBbKXxkVyL3GyzseChpbu4GVS7juZBGVrlWEW1KhV
7OrBsDpNtX0hsrjc2LTJ2raGJhsbSib3jEvJyw2bF71ibv4bb8/ilIcRbQydgLV7bR88BV2RHdpf
Dd3/A1BLAwQUAAAACADOfT4vsIZUio8AAAAwAQAALgAAAGNvbW1lbnRzLTEuMjMvc3RydWN0L2Nv
bW1lbnRfbm90aWZ5X29yYWNsZS5zcWx9zz0LwjAQBuC9v+LGBDJIIThXyeYk3UuMJ0bzUdNEyL83
WEHj1213PLzcu96KrhfQd6uNAOWtRRcH56M+ZCDN22XQewDtIpQVXDKGFWHkNEE1VxnUUYaWLDl9
pX53QhXvIR+05bVFK7X5EcsXFXXSIvyjMxuDtjJkOGMmX3rRQpLTl4Rk7sSe/7LHO7ShN1BLAwQU
AAAACAB6sHwxGqHw96ICAADxBwAALQAAAGNvbW1lbnRzLTEuMjMvdGVtcGxhdGUvY29tbWVudF9s
aXN0X3BhZ2UudG1wbK1VUW/aMBB+76+4ISGo2rL1tU0iIUq3ah1UpXtCKDLNFbImcWQ7dJ3243d2
4sQB1jJpeYDYd9/d+b7vnHkXpjeDnK0wVLFKsA/fZp/70HvkaYqZkoMklirU9oGx9+AYji+PAODu
fjoaz2YgFVOFDFOUkrxaJhSCC2uB7uLoyFufB/PuATm6C+8j+RIifxPANyg2Mb70TnVi+yT8BUW4
5EUWnUKR580iK9LQhqnS5JSFUtxcQx1fxr8QAvgEZ7pobZyMbr9fjUGxZYIUS0QoMAqXuIoz4zTv
nhG8yBT4bvadAGtkBA0Ff+lDwpaYhM/4Kgk0B32A5ohx9sQHJmLvdHtbFssf+KjKM2/ZIqZwF5Fz
qVDsBai1oJrItKB+2HKvp/fj4eiL7QjVVzetOa3ZCAuRkJlElLJn1Ks+DEcPN9MJ7dpUVNDDcPZV
70SxzBP22uZr32Pjx1GTfUCLUn3ATQtMdse+QhXKIk2ZeB04HrpmTwmgk1ltEgMkhIQL0gBsWBKv
Mr+jeN6B6v2R4qHoBDqbpyKtwpJgrRla232PAXXwye8Yh6Yn3UUnaLYsZwbNglaEuZlBTVz4xEXK
VL9GGd6ikGeVVtsoexbLzcWFLaDk+61Sne5sVbqnh2YsW6XTrwhaqi//T+C8UgiMJ1fvzQ+6E3IN
5hqiCYUPPoWZ3rujC157dg3Myy1bIl6tlSFrK5TvVxXdCdzEvJCly/h2Nq6lnFemEtBgz+C8FFvt
8I9i1/fU+0p3im2VciDQXFa+8358CaXiHbpbJ6gI/+ulWjsncfZsb2NWNdfSCr/rZrs0Bf4uT2/l
yvCncvLsYcd4bDFzYpkxxv/PisOITX8AYJeJUqUOD3W9moMJLXYba75IW/NTN+S972HG69b32t+3
KtwfUEsBAhQDFAAAAAgAxVJkMg3sFPOnFwAAH2kAAC4AAAAAAAAAAQAAACSBAAAAAGNvbW1lbnRz
LTEuMjMvT3BlbkludGVyYWN0Mi9BY3Rpb24vQ29tbWVudHMucG1QSwECFAMUAAAACACvTM8wyG9M
Hr0AAACxAQAAKwAAAAAAAAABAAAAJIHzFwAAY29tbWVudHMtMS4yMy9jb25mL3Nwb3BzX2NvbW1l
bnRfbm90aWZ5LmluaVBLAQIUAxQAAAAIABa1SjA5SQcGNgAAADwAAAAbAAAAAAAAAAEAAAAkgfkY
AABjb21tZW50cy0xLjIzL01BTklGRVNULlNLSVBQSwECFAMUAAAACAB2BWQyW7IqX+MAAAAFAgAA
JwAAAAAAAAABAAAAJIFoGQAAY29tbWVudHMtMS4yMy9kYXRhL2luc3RhbGxfc2VjdXJpdHkuZGF0
UEsBAhQDFAAAAAgAzn0+L/AmM/cmAAAAJAAAADIAAAAAAAAAAQAAACSBkBoAAGNvbW1lbnRzLTEu
MjMvc3RydWN0L2NvbW1lbnRfc3VtbWFyeV9nZW5lcmF0b3Iuc3FsUEsBAhQDFAAAAAgAEkixMCOx
jG7aAAAA4wEAACoAAAAAAAAAAQAAACSBBhsAAGNvbW1lbnRzLTEuMjMvdGVtcGxhdGUvY29tbWVu
dF9kZXRhaWwudG1wbFBLAQIUAxQAAAAIAMVSZDJKa0N5twIAABEIAAAtAAAAAAAAAAEAAAAkgSgc
AABjb21tZW50cy0xLjIzL09wZW5JbnRlcmFjdDIvQ29tbWVudERpc2FibGUucG1QSwECFAMUAAAA
CACWWVAw10mvLm0AAACDAAAAKgAAAAAAAAABAAAAJIEqHwAAY29tbWVudHMtMS4yMy90ZW1wbGF0
ZS9jb21tZW50X29iamVjdC50bXBsUEsBAhQDFAAAAAgA1FlQMGwdY393AAAA4gAAACsAAAAAAAAA
AQAAACSB3x8AAGNvbW1lbnRzLTEuMjMvdGVtcGxhdGUvY29tbWVudF9zdW1tYXJ5LnRtcGxQSwEC
FAMUAAAACAA2Q7Ewzv3YIv4AAABFAgAALQAAAAAAAAABAAAAJIGfIAAAY29tbWVudHMtMS4yMy90
ZW1wbGF0ZS9jb21tZW50X2Zvcm1fcGFnZS50bXBsUEsBAhQDFAAAAAgAnb5jMukGY5/lAAAAIQIA
ACwAAAAAAAAAAQAAACSB6CEAAGNvbW1lbnRzLTEuMjMvY29uZi9zcG9wc19jb21tZW50X2Rpc2Fi
bGUuaW5pUEsBAhQDFAAAAAgAzn0+L/s7MrKpAAAAUAEAACgAAAAAAAAAAQAAACSBFyMAAGNvbW1l
bnRzLTEuMjMvc3RydWN0L2NvbW1lbnRfc3VtbWFyeS5zcWxQSwECFAMUAAAACACNUFMwn3zLJ7kA
AACZAQAAIAAAAAAAAAABAAAAJIEGJAAAY29tbWVudHMtMS4yMy9zdHJ1Y3QvY29tbWVudC5zcWxQ
SwECFAMUAAAACADrQrEwB0AJfq8BAAC1AwAALwAAAAAAAAABAAAAJIH9JAAAY29tbWVudHMtMS4y
My90ZW1wbGF0ZS9jb21tZW50X25vdGlmeV9saXN0LnRtcGxQSwECFAMUAAAACAA3uXEyBcTHdOcC
AABvCAAAMgAAAAAAAAABAAAAJIH5JgAAY29tbWVudHMtMS4yMy9PcGVuSW50ZXJhY3QyL1NRTElu
c3RhbGwvQ29tbWVudHMucG1QSwECFAMUAAAACADOfT4vIof81yEAAAAfAAAAKgAAAAAAAAABAAAA
JIEwKgAAY29tbWVudHMtMS4yMy9zdHJ1Y3QvY29tbWVudF9nZW5lcmF0b3Iuc3FsUEsBAhQDFAAA
AAgAzn0+LxQwpAAnAAAAJQAAADAAAAAAAAAAAQAAACSBmSoAAGNvbW1lbnRzLTEuMjMvc3RydWN0
L2NvbW1lbnRfbm90aWZ5X3NlcXVlbmNlLnNxbFBLAQIUAxQAAAAIADe5cTKT+lFCEgIAANAFAAAq
AAAAAAAAAAEAAAAkgQ4rAABjb21tZW50cy0xLjIzL09wZW5JbnRlcmFjdDIvQ29tbWVudGFibGUu
cG1QSwECFAMUAAAACADxPWQy2VR2X2wCAADlBgAAKAAAAAAAAAABAAAAJIFoLQAAY29tbWVudHMt
MS4yMy90ZW1wbGF0ZS9kaXNhYmxlX2xpc3QudG1wbFBLAQIUAxQAAAAIAE9CZDLDhyuC+AEAACsF
AAAdAAAAAAAAAAEAAAAkgRowAABjb21tZW50cy0xLjIzL2NvbmYvYWN0aW9uLmluaVBLAQIUAxQA
AAAIAGO5YzLYj8snKAAAACYAAAAxAAAAAAAAAAEAAAAkgU0yAABjb21tZW50cy0xLjIzL3N0cnVj
dC9jb21tZW50X2Rpc2FibGVfc2VxdWVuY2Uuc3FsUEsBAhQDFAAAAAgAxVJkMjv4jZenAgAAPAkA
ADIAAAAAAAAAAQAAACSBxDIAAGNvbW1lbnRzLTEuMjMvT3BlbkludGVyYWN0Mi9BY3Rpb24vQ29t
bWVudEFkbWluLnBtUEsBAhQDFAAAAAgAgARPMCb6SffbAAAAfwEAACMAAAAAAAAAAQAAACSBuzUA
AGNvbW1lbnRzLTEuMjMvdGVtcGxhdGUvY29tbWVudC50bXBsUEsBAhQDFAAAAAgABgRkMnSDYMHg
BQAAgRQAACoAAAAAAAAAAQAAACSB1zYAAGNvbW1lbnRzLTEuMjMvbXNnL2NvbW1lbnRzLW1lc3Nh
Z2VzLWVuLm1zZ1BLAQIUAxQAAAAIAI1QUzCIS1wsuQAAAJkBAAAqAAAAAAAAAAEAAAAkgf88AABj
b21tZW50cy0xLjIzL3N0cnVjdC9jb21tZW50X2ludGVyYmFzZS5zcWxQSwECFAMUAAAACADBTM8w
qcTXiJoBAACwAwAAJAAAAAAAAAABAAAAJIEAPgAAY29tbWVudHMtMS4yMy9jb25mL3Nwb3BzX2Nv
bW1lbnQuaW5pUEsBAhQDFAAAAAgAWa98MbkNd1zQAgAAxAYAACYAAAAAAAAAAQAAACSB3D8AAGNv
bW1lbnRzLTEuMjMvT3BlbkludGVyYWN0Mi9Db21tZW50LnBtUEsBAhQDFAAAAAgAxgF8MZI3/s8Q
AQAAIAIAACgAAAAAAAAAAQAAACSB8EIAAGNvbW1lbnRzLTEuMjMvdGVtcGxhdGUvY29tbWVudF9s
aXN0LnRtcGxQSwECFAMUAAAACABZA2QyMsWK/08BAAAxBQAAFgAAAAAAAAABAAAAJIFGRAAAY29t
bWVudHMtMS4yMy9NQU5JRkVTVFBLAQIUAxQAAAAIAI5QUzA1prUwswAAAJ4BAAAnAAAAAAAAAAEA
AAAkgclFAABjb21tZW50cy0xLjIzL3N0cnVjdC9jb21tZW50X29yYWNsZS5zcWxQSwECFAMUAAAA
CADOfT4vKXWtGpkAAAAtAQAAJwAAAAAAAAABAAAAJIHBRgAAY29tbWVudHMtMS4yMy9zdHJ1Y3Qv
Y29tbWVudF9ub3RpZnkuc3FsUEsBAhQDFAAAAAgAzn0+Ly3haTolAAAAIwAAADEAAAAAAAAAAQAA
ACSBn0cAAGNvbW1lbnRzLTEuMjMvc3RydWN0L2NvbW1lbnRfc3VtbWFyeV9zZXF1ZW5jZS5zcWxQ
SwECFAMUAAAACADOfT4v+1S3syAAAAAeAAAAKQAAAAAAAAABAAAAJIETSAAAY29tbWVudHMtMS4y
My9zdHJ1Y3QvY29tbWVudF9zZXF1ZW5jZS5zcWxQSwECFAMUAAAACADOTM8w48LjNrsAAACxAQAA
LAAAAAAAAAABAAAAJIF6SAAAY29tbWVudHMtMS4yMy9jb25mL3Nwb3BzX2NvbW1lbnRfc3VtbWFy
eS5pbmlQSwECFAMUAAAACADOfT4vDxEyMSgAAAAmAAAAMQAAAAAAAAABAAAAJIF/SQAAY29tbWVu
dHMtMS4yMy9zdHJ1Y3QvY29tbWVudF9ub3RpZnlfZ2VuZXJhdG9yLnNxbFBLAQIUAxQAAAAIAPkG
fDHhAd7hmQAAAFEBAAAuAAAAAAAAAAEAAAAkgfZJAABjb21tZW50cy0xLjIzL3RlbXBsYXRlL25v
dGlmaWNhdGlvbl9lbWFpbC50bXBsUEsBAhQDFAAAAAgAIAJkMuigNdWoAAAAUwEAACgAAAAAAAAA
AQAAACSB20oAAGNvbW1lbnRzLTEuMjMvc3RydWN0L2NvbW1lbnRfZGlzYWJsZS5zcWxQSwECFAMU
AAAACACABmQygyODQGIKAAAEGQAAFQAAAAAAAAABAAAAJIHJSwAAY29tbWVudHMtMS4yMy9DaGFu
Z2VzUEsBAhQDFAAAAAgAKUixMD2JIj/8AAAAngEAAC4AAAAAAAAAAQAAACSBXlYAAGNvbW1lbnRz
LTEuMjMvdGVtcGxhdGUvYm94X2NvbW1lbnRfcmVjZW50LnRtcGxQSwECFAMUAAAACADOfT4vt/vu
taEAAABTAQAALwAAAAAAAAABAAAAJIGmVwAAY29tbWVudHMtMS4yMy9zdHJ1Y3QvY29tbWVudF9z
dW1tYXJ5X29yYWNsZS5zcWxQSwECFAMUAAAACABSBmQyQa+PqAYBAAClAQAAGQAAAAAAAAABAAAA
JIGUWAAAY29tbWVudHMtMS4yMy9wYWNrYWdlLmluaVBLAQIUAxQAAAAIAB2jaTKz+FCFMAoAACUZ
AAArAAAAAAAAAAEAAAAkgdFZAABjb21tZW50cy0xLjIzL09wZW5JbnRlcmFjdDIvQXBwL0NvbW1l
bnRzLnBtUEsBAhQDFAAAAAgAzn0+LzZaIQ1/AAAAJgEAACoAAAAAAAAAAQAAACSBSmQAAGNvbW1l
bnRzLTEuMjMvdGVtcGxhdGUvY29tbWVudF9wb3N0ZXIudG1wbFBLAQIUAxQAAAAIAMxBZDKPn79/
kgMAAEUQAAAoAAAAAAAAAAEAAAAkgRFlAABjb21tZW50cy0xLjIzL3RlbXBsYXRlL2NvbW1lbnRf
Zm9ybS50bXBsUEsBAhQDFAAAAAgAN7lxMjhX60d0AwAAbwgAADEAAAAAAAAAAQAAACSB6WgAAGNv
bW1lbnRzLTEuMjMvT3BlbkludGVyYWN0Mi9Db21tZW50U3VtbWFyeVN5bmMucG1QSwECFAMUAAAA
CADOfT4vsIZUio8AAAAwAQAALgAAAAAAAAABAAAAJIGsbAAAY29tbWVudHMtMS4yMy9zdHJ1Y3Qv
Y29tbWVudF9ub3RpZnlfb3JhY2xlLnNxbFBLAQIUAxQAAAAIAHqwfDEaofD3ogIAAPEHAAAtAAAA
AAAAAAEAAAAkgYdtAABjb21tZW50cy0xLjIzL3RlbXBsYXRlL2NvbW1lbnRfbGlzdF9wYWdlLnRt
cGxQSwUGAAAAAC8ALwAHEAAAdHAAAAAA

SOMELONGSTRING
}

