package Egg::View::Mail::Mailer::CMD;
#
# Masatoshi Mizuno E<lt>lusheE<64>cpan.orgE<gt>
#
# $Id: CMD.pm 285 2008-02-28 04:20:55Z lushe $
#
use strict;
use warnings;
use Carp qw/ croak /;

our $VERSION = '0.01';

sub _setup {
	my($class, $e)= @_;
	my $c= $class->config;
	$c->{cmd_path} ||= do {
		  -e '/usr/sbin/sendmail'       ? '/usr/sbin/sendmail'
		: -e '/usr/local/sbin/sendmail' ? '/usr/local/sbin/sendmail'
		: -e '/usr/bin/sendmail'        ? '/usr/bin/sendmail'
		: die q{'sendmail' command is not found.};
	  };
	$c->{cmd_option} ||= '-t -i';
	unless ($class->can('valid_to_address')) {
		no strict 'refs';  ## no critic.
		no warnings 'redefine';
		my $regexp= $c->{email_regexp} || qr{^[\w\d\-_]+@[\w\d\.\-_]+$};
		*{"${class}::valid_to_address"}=
		   sub { $_[1]=~m{$regexp} || croak qq"'$_[1]' is bad address." };
	}
	$class->next::method($e);
}
sub mail_send {
	my $self= shift;
	my $data= $_[0] ? ($_[1] ? {@_}: $_[0]) : croak q{I want mail data.};
	my $c= $self->config;
	$self->valid_to_address($data->{to}) || return 0;
	my $cmd_line= "$c->{cmd_path} $c->{cmd_option} $data->{to}";
	if ($data->{debug}) {
		$self->e->debug_out("# + mailsend : $cmd_line\n${$data->{body}}");
	} else {
		open  MSEND, "| $cmd_line";  ## no critic
		print MSEND ${$data->{body}};
		close MSEND;
	}
	1;
}

1;

__END__

=head1 NAME

Egg::View::Mail::Mailer::CMD - Mail is transmitted by using the sendmail command. 

=head1 SYNOPSIS

  package MyApp::View::Mail::MyComp;
  use base qw/ Egg::View::Mail::Base /;
  
  ...........
  .....
  
  __PACKAGE__->setup_mailer('CMD');

=head1 DESCRIPTION

It is Mailer system component to transmit mail by using the sendmail command. 

Use is enabled specifying 'CMD' for the first argument of 'setup_mailer' method.

=head1 CONFIGURATION

=head3 cmd_path

PATH of sendmail command.

It is set if the whereabouts of '/usr/sbin/sendmail', '/usr/local/sbin/sendmail',
 '/usr/bin/sendmail' is confirmed and it is found at the unspecification.
The exception is generated when not found anywhere.

=head3 cmd_option

Start option to pass to command line.

Default is '-t -i'.

=head3 email_regexp

Regular expression for easy format check in mail address.

This stays in an easy check so that there is no obstacle to pass the mail address
 to the command line. Please check the check on the format of a detailed mail 
 address beforehand by the module such as L<Email::Valid>.

Default is '^[\w\d\-_]+@[\w\d\.\-_]+$'.

Moreover, it is possible to check it there making 'Valid_to_address' method for 
the MAIL controller. Please look at the source code in detail.

=head3 debug

It operates by debug mode.

When an effective value is given, the content of mail comes to be sent to 'debug_out'
of the project without doing actual Mail Sending.

It tried to send the content of what mail by outputting 'debug_out' can be checked.

=head1 METHODS

=head2 mail_send ([MAIL_DATA_HASH])

This method is what 'send' method of L<Egg::View::Mail::Base> calls it internally.

The obstacle is generated by operating the component built in when calling 
directly.

=head1 SEE ALSO

L<Egg::Release>,
L<Egg::View::Mail>,
L<Egg::View::Mail::Base>,
L<Egg::View::Mail::Mailer::SMTP>,

=head1 AUTHOR

Masatoshi Mizuno E<lt>lusheE<64>cpan.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2008 Bee Flag, Corp. E<lt>L<http://egg.bomcity.com/>E<gt>, All Rights Reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.8.6 or,
at your option, any later version of Perl 5 you may have available.

=cut

