/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ɵRuntimeError as RuntimeError } from '@angular/core';
const LINE_START = '\n - ';
export function invalidTimingValue(exp) {
    return new RuntimeError(3000 /* RuntimeErrorCode.INVALID_TIMING_VALUE */, ngDevMode && `The provided timing value "${exp}" is invalid.`);
}
export function negativeStepValue() {
    return new RuntimeError(3100 /* RuntimeErrorCode.NEGATIVE_STEP_VALUE */, ngDevMode && 'Duration values below 0 are not allowed for this animation step.');
}
export function negativeDelayValue() {
    return new RuntimeError(3101 /* RuntimeErrorCode.NEGATIVE_DELAY_VALUE */, ngDevMode && 'Delay values below 0 are not allowed for this animation step.');
}
export function invalidStyleParams(varName) {
    return new RuntimeError(3001 /* RuntimeErrorCode.INVALID_STYLE_PARAMS */, ngDevMode &&
        `Unable to resolve the local animation param ${varName} in the given list of values`);
}
export function invalidParamValue(varName) {
    return new RuntimeError(3003 /* RuntimeErrorCode.INVALID_PARAM_VALUE */, ngDevMode && `Please provide a value for the animation param ${varName}`);
}
export function invalidNodeType(nodeType) {
    return new RuntimeError(3004 /* RuntimeErrorCode.INVALID_NODE_TYPE */, ngDevMode && `Unable to resolve animation metadata node #${nodeType}`);
}
export function invalidCssUnitValue(userProvidedProperty, value) {
    return new RuntimeError(3005 /* RuntimeErrorCode.INVALID_CSS_UNIT_VALUE */, ngDevMode && `Please provide a CSS unit value for ${userProvidedProperty}:${value}`);
}
export function invalidTrigger() {
    return new RuntimeError(3006 /* RuntimeErrorCode.INVALID_TRIGGER */, ngDevMode &&
        'animation triggers cannot be prefixed with an `@` sign (e.g. trigger(\'@foo\', [...]))');
}
export function invalidDefinition() {
    return new RuntimeError(3007 /* RuntimeErrorCode.INVALID_DEFINITION */, ngDevMode && 'only state() and transition() definitions can sit inside of a trigger()');
}
export function invalidState(metadataName, missingSubs) {
    return new RuntimeError(3008 /* RuntimeErrorCode.INVALID_STATE */, ngDevMode &&
        `state("${metadataName}", ...) must define default values for all the following style substitutions: ${missingSubs.join(', ')}`);
}
export function invalidStyleValue(value) {
    return new RuntimeError(3002 /* RuntimeErrorCode.INVALID_STYLE_VALUE */, ngDevMode && `The provided style string value ${value} is not allowed.`);
}
export function invalidProperty(prop) {
    return new RuntimeError(3009 /* RuntimeErrorCode.INVALID_PROPERTY */, ngDevMode &&
        `The provided animation property "${prop}" is not a supported CSS property for animations`);
}
export function invalidParallelAnimation(prop, firstStart, firstEnd, secondStart, secondEnd) {
    return new RuntimeError(3010 /* RuntimeErrorCode.INVALID_PARALLEL_ANIMATION */, ngDevMode &&
        `The CSS property "${prop}" that exists between the times of "${firstStart}ms" and "${firstEnd}ms" is also being animated in a parallel animation between the times of "${secondStart}ms" and "${secondEnd}ms"`);
}
export function invalidKeyframes() {
    return new RuntimeError(3011 /* RuntimeErrorCode.INVALID_KEYFRAMES */, ngDevMode && `keyframes() must be placed inside of a call to animate()`);
}
export function invalidOffset() {
    return new RuntimeError(3012 /* RuntimeErrorCode.INVALID_OFFSET */, ngDevMode && `Please ensure that all keyframe offsets are between 0 and 1`);
}
export function keyframeOffsetsOutOfOrder() {
    return new RuntimeError(3200 /* RuntimeErrorCode.KEYFRAME_OFFSETS_OUT_OF_ORDER */, ngDevMode && `Please ensure that all keyframe offsets are in order`);
}
export function keyframesMissingOffsets() {
    return new RuntimeError(3202 /* RuntimeErrorCode.KEYFRAMES_MISSING_OFFSETS */, ngDevMode && `Not all style() steps within the declared keyframes() contain offsets`);
}
export function invalidStagger() {
    return new RuntimeError(3013 /* RuntimeErrorCode.INVALID_STAGGER */, ngDevMode && `stagger() can only be used inside of query()`);
}
export function invalidQuery(selector) {
    return new RuntimeError(3014 /* RuntimeErrorCode.INVALID_QUERY */, ngDevMode &&
        `\`query("${selector}")\` returned zero elements. (Use \`query("${selector}", { optional: true })\` if you wish to allow this.)`);
}
export function invalidExpression(expr) {
    return new RuntimeError(3015 /* RuntimeErrorCode.INVALID_EXPRESSION */, ngDevMode && `The provided transition expression "${expr}" is not supported`);
}
export function invalidTransitionAlias(alias) {
    return new RuntimeError(3016 /* RuntimeErrorCode.INVALID_TRANSITION_ALIAS */, ngDevMode && `The transition alias value "${alias}" is not supported`);
}
export function validationFailed(errors) {
    return new RuntimeError(3500 /* RuntimeErrorCode.VALIDATION_FAILED */, ngDevMode && `animation validation failed:\n${errors.map(err => err.message).join('\n')}`);
}
export function buildingFailed(errors) {
    return new RuntimeError(3501 /* RuntimeErrorCode.BUILDING_FAILED */, ngDevMode && `animation building failed:\n${errors.map(err => err.message).join('\n')}`);
}
export function triggerBuildFailed(name, errors) {
    return new RuntimeError(3404 /* RuntimeErrorCode.TRIGGER_BUILD_FAILED */, ngDevMode &&
        `The animation trigger "${name}" has failed to build due to the following errors:\n - ${errors.map(err => err.message).join('\n - ')}`);
}
export function animationFailed(errors) {
    return new RuntimeError(3502 /* RuntimeErrorCode.ANIMATION_FAILED */, ngDevMode &&
        `Unable to animate due to the following errors:${LINE_START}${errors.map(err => err.message).join(LINE_START)}`);
}
export function registerFailed(errors) {
    return new RuntimeError(3503 /* RuntimeErrorCode.REGISTRATION_FAILED */, ngDevMode &&
        `Unable to build the animation due to the following errors: ${errors.map(err => err.message).join('\n')}`);
}
export function missingOrDestroyedAnimation() {
    return new RuntimeError(3300 /* RuntimeErrorCode.MISSING_OR_DESTROYED_ANIMATION */, ngDevMode && 'The requested animation doesn\'t exist or has already been destroyed');
}
export function createAnimationFailed(errors) {
    return new RuntimeError(3504 /* RuntimeErrorCode.CREATE_ANIMATION_FAILED */, ngDevMode &&
        `Unable to create the animation due to the following errors:${errors.map(err => err.message).join('\n')}`);
}
export function missingPlayer(id) {
    return new RuntimeError(3301 /* RuntimeErrorCode.MISSING_PLAYER */, ngDevMode && `Unable to find the timeline player referenced by ${id}`);
}
export function missingTrigger(phase, name) {
    return new RuntimeError(3302 /* RuntimeErrorCode.MISSING_TRIGGER */, ngDevMode &&
        `Unable to listen on the animation trigger event "${phase}" because the animation trigger "${name}" doesn\'t exist!`);
}
export function missingEvent(name) {
    return new RuntimeError(3303 /* RuntimeErrorCode.MISSING_EVENT */, ngDevMode &&
        `Unable to listen on the animation trigger "${name}" because the provided event is undefined!`);
}
export function unsupportedTriggerEvent(phase, name) {
    return new RuntimeError(3400 /* RuntimeErrorCode.UNSUPPORTED_TRIGGER_EVENT */, ngDevMode &&
        `The provided animation trigger event "${phase}" for the animation trigger "${name}" is not supported!`);
}
export function unregisteredTrigger(name) {
    return new RuntimeError(3401 /* RuntimeErrorCode.UNREGISTERED_TRIGGER */, ngDevMode && `The provided animation trigger "${name}" has not been registered!`);
}
export function triggerTransitionsFailed(errors) {
    return new RuntimeError(3402 /* RuntimeErrorCode.TRIGGER_TRANSITIONS_FAILED */, ngDevMode &&
        `Unable to process animations due to the following failed trigger transitions\n ${errors.map(err => err.message).join('\n')}`);
}
export function triggerParsingFailed(name, errors) {
    return new RuntimeError(3403 /* RuntimeErrorCode.TRIGGER_PARSING_FAILED */, ngDevMode &&
        `Animation parsing for the ${name} trigger have failed:${LINE_START}${errors.map(err => err.message).join(LINE_START)}`);
}
export function transitionFailed(name, errors) {
    return new RuntimeError(3505 /* RuntimeErrorCode.TRANSITION_FAILED */, ngDevMode && `@${name} has failed due to:\n ${errors.map(err => err.message).join('\n- ')}`);
}
//# sourceMappingURL=data:application/json;base64,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