"use strict";
// Copyright 2021-2024 Buf Technologies, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
Object.defineProperty(exports, "__esModule", { value: true });
exports.getFieldIntrinsicDefaultValue = exports.getFieldExplicitDefaultValue = exports.getFieldTyping = exports.literalString = void 0;
const protobuf_1 = require("@bufbuild/protobuf");
const { localName, getUnwrappedFieldType, scalarZeroValue } = protobuf_1.codegenInfo;
/**
 * @deprecated Please use GeneratedFile.string() instead
 */
function literalString(value) {
    return ('"' +
        value
            .split("\\")
            .join("\\\\")
            .split('"')
            .join('\\"')
            .split("\r")
            .join("\\r")
            .split("\n")
            .join("\\n") +
        '"');
}
exports.literalString = literalString;
/**
 * @deprecated
 */
function getFieldTyping(field, file) {
    const typing = [];
    let optional = false;
    switch (field.fieldKind) {
        case "scalar":
            typing.push(scalarTypeScriptType(field.scalar, field.longType));
            optional = field.optional;
            break;
        case "message": {
            const baseType = getUnwrappedFieldType(field);
            if (baseType !== undefined) {
                typing.push(scalarTypeScriptType(baseType, protobuf_1.LongType.BIGINT));
            }
            else {
                typing.push(file.import(field.message).toTypeOnly());
            }
            optional = true;
            break;
        }
        case "enum":
            typing.push(file.import(field.enum).toTypeOnly());
            optional = field.optional;
            break;
        case "map": {
            let keyType;
            switch (field.mapKey) {
                case protobuf_1.ScalarType.INT32:
                case protobuf_1.ScalarType.FIXED32:
                case protobuf_1.ScalarType.UINT32:
                case protobuf_1.ScalarType.SFIXED32:
                case protobuf_1.ScalarType.SINT32:
                    keyType = "number";
                    break;
                default:
                    keyType = "string";
                    break;
            }
            let valueType;
            switch (field.mapValue.kind) {
                case "scalar":
                    valueType = scalarTypeScriptType(field.mapValue.scalar, protobuf_1.LongType.BIGINT);
                    break;
                case "message":
                    valueType = file.import(field.mapValue.message).toTypeOnly();
                    break;
                case "enum":
                    valueType = file.import(field.mapValue.enum).toTypeOnly();
                    break;
            }
            typing.push("{ [key: ", keyType, "]: ", valueType, " }");
            optional = false;
            break;
        }
    }
    if (field.repeated) {
        typing.push("[]");
        optional = false;
    }
    return { typing, optional };
}
exports.getFieldTyping = getFieldTyping;
/**
 * @deprecated
 */
function getFieldExplicitDefaultValue(field, protoInt64Symbol) {
    switch (field.fieldKind) {
        case "enum": {
            const value = field.enum.values.find((v) => v.number === field.getDefaultValue());
            if (value !== undefined) {
                return [value.parent, ".", localName(value)];
            }
            break;
        }
        case "scalar": {
            const defaultValue = field.getDefaultValue();
            if (defaultValue === undefined) {
                break;
            }
            switch (field.scalar) {
                case protobuf_1.ScalarType.FLOAT:
                case protobuf_1.ScalarType.DOUBLE: {
                    return defaultValue;
                }
                case protobuf_1.ScalarType.INT64:
                case protobuf_1.ScalarType.SINT64:
                case protobuf_1.ScalarType.SFIXED64:
                    return [protoInt64Symbol, `.parse("${defaultValue.toString()}")`];
                case protobuf_1.ScalarType.UINT64:
                case protobuf_1.ScalarType.FIXED64:
                    return [protoInt64Symbol, `.uParse("${defaultValue.toString()}")`];
                case protobuf_1.ScalarType.INT32:
                case protobuf_1.ScalarType.FIXED32:
                case protobuf_1.ScalarType.UINT32:
                case protobuf_1.ScalarType.SFIXED32:
                case protobuf_1.ScalarType.SINT32:
                    return defaultValue;
                case protobuf_1.ScalarType.BOOL: {
                    return defaultValue;
                }
                case protobuf_1.ScalarType.STRING: {
                    if (typeof defaultValue == "string") {
                        return literalString(defaultValue);
                    }
                    break;
                }
                case protobuf_1.ScalarType.BYTES: {
                    if (defaultValue instanceof Uint8Array) {
                        return defaultValue;
                    }
                    break;
                }
            }
            break;
        }
        default:
            break;
    }
    return undefined;
}
exports.getFieldExplicitDefaultValue = getFieldExplicitDefaultValue;
/**
 * @deprecated
 */
function getFieldIntrinsicDefaultValue(field) {
    if (field.repeated) {
        return {
            defaultValue: "[]",
            typingInferrable: false,
        };
    }
    if (field.fieldKind == "map") {
        return {
            defaultValue: "{}",
            typingInferrable: false,
        };
    }
    let defaultValue = undefined;
    let typingInferrable = false;
    if (field.parent.file.syntax == "proto3") {
        switch (field.fieldKind) {
            case "enum": {
                if (!field.optional) {
                    const zeroValue = field.enum.values.find((v) => v.number === 0);
                    if (zeroValue === undefined) {
                        throw new Error("invalid proto3 enum: missing 0 value");
                    }
                    defaultValue = [field.enum, ".", localName(zeroValue)];
                    typingInferrable = true;
                }
                break;
            }
            case "scalar":
                if (!field.optional) {
                    typingInferrable = true;
                    if (field.scalar === protobuf_1.ScalarType.STRING) {
                        defaultValue = literalString("");
                    }
                    else {
                        defaultValue = scalarZeroValue(field.scalar, field.longType);
                        if (typeof defaultValue === "string") {
                            defaultValue = literalString(defaultValue);
                        }
                    }
                }
                break;
            default:
                break;
        }
    }
    return {
        defaultValue,
        typingInferrable,
    };
}
exports.getFieldIntrinsicDefaultValue = getFieldIntrinsicDefaultValue;
/**
 * @deprecated
 */
function scalarTypeScriptType(type, longType) {
    switch (type) {
        case protobuf_1.ScalarType.STRING:
            return "string";
        case protobuf_1.ScalarType.BOOL:
            return "boolean";
        case protobuf_1.ScalarType.UINT64:
        case protobuf_1.ScalarType.SFIXED64:
        case protobuf_1.ScalarType.FIXED64:
        case protobuf_1.ScalarType.SINT64:
        case protobuf_1.ScalarType.INT64:
            if (longType === protobuf_1.LongType.STRING) {
                return "string";
            }
            return "bigint";
        case protobuf_1.ScalarType.BYTES:
            return "Uint8Array";
        default:
            return "number";
    }
}
