﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/athena/AthenaRequest.h>
#include <aws/athena/Athena_EXPORTS.h>
#include <aws/athena/model/SessionState.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Athena {
namespace Model {

/**
 */
class ListSessionsRequest : public AthenaRequest {
 public:
  AWS_ATHENA_API ListSessionsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListSessions"; }

  AWS_ATHENA_API Aws::String SerializePayload() const override;

  AWS_ATHENA_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The workgroup to which the session belongs.</p>
   */
  inline const Aws::String& GetWorkGroup() const { return m_workGroup; }
  inline bool WorkGroupHasBeenSet() const { return m_workGroupHasBeenSet; }
  template <typename WorkGroupT = Aws::String>
  void SetWorkGroup(WorkGroupT&& value) {
    m_workGroupHasBeenSet = true;
    m_workGroup = std::forward<WorkGroupT>(value);
  }
  template <typename WorkGroupT = Aws::String>
  ListSessionsRequest& WithWorkGroup(WorkGroupT&& value) {
    SetWorkGroup(std::forward<WorkGroupT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A filter for a specific session state. A description of each state
   * follows.</p> <p> <code>CREATING</code> - The session is being started, including
   * acquiring resources.</p> <p> <code>CREATED</code> - The session has been
   * started.</p> <p> <code>IDLE</code> - The session is able to accept a
   * calculation.</p> <p> <code>BUSY</code> - The session is processing another task
   * and is unable to accept a calculation.</p> <p> <code>TERMINATING</code> - The
   * session is in the process of shutting down.</p> <p> <code>TERMINATED</code> -
   * The session and its resources are no longer running.</p> <p>
   * <code>DEGRADED</code> - The session has no healthy coordinators.</p> <p>
   * <code>FAILED</code> - Due to a failure, the session and its resources are no
   * longer running.</p>
   */
  inline SessionState GetStateFilter() const { return m_stateFilter; }
  inline bool StateFilterHasBeenSet() const { return m_stateFilterHasBeenSet; }
  inline void SetStateFilter(SessionState value) {
    m_stateFilterHasBeenSet = true;
    m_stateFilter = value;
  }
  inline ListSessionsRequest& WithStateFilter(SessionState value) {
    SetStateFilter(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of sessions to return.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListSessionsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A token generated by the Athena service that specifies where to continue
   * pagination if a previous request was truncated. To obtain the next set of pages,
   * pass in the <code>NextToken</code> from the response object of the previous page
   * call.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListSessionsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_workGroup;
  bool m_workGroupHasBeenSet = false;

  SessionState m_stateFilter{SessionState::NOT_SET};
  bool m_stateFilterHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace Athena
}  // namespace Aws
