// Copyright © Advanced Micro Devices, Inc., or its affiliates.
// SPDX-License-Identifier:  MIT

#pragma once

#include <cstdint>
#include <ostream>
#include <vector>

#include <hipdnn_sdk/utilities/StringUtil.hpp>

namespace test_conv_common
{

struct ConvTestCase
{
    std::vector<int64_t> _xDims;
    std::vector<int64_t> _wDims;
    std::vector<int64_t> _yDims;
    std::vector<int64_t> _convPrePadding;
    std::vector<int64_t> _convPostPadding;
    std::vector<int64_t> _convStride;
    std::vector<int64_t> _convDilation;
    unsigned _seed;

    ConvTestCase(std::vector<int64_t>&& xDims,
                 std::vector<int64_t>&& wDims,
                 std::vector<int64_t>&& convPrePadding,
                 std::vector<int64_t>&& convPostPadding,
                 std::vector<int64_t>&& convStride,
                 std::vector<int64_t>&& convDilation,
                 unsigned seed)
        : _xDims(std::move(xDims))
        , _wDims(std::move(wDims))
        , _convPrePadding(std::move(convPrePadding))
        , _convPostPadding(std::move(convPostPadding))
        , _convStride(std::move(convStride))
        , _convDilation(std::move(convDilation))
        , _seed(seed)
    {
        // Indices for dimensions
        // N - Batch size, always at index 0
        // C - Channels, always at index 1
        // D - Depth (for 5D tensors), always at index 2 if present
        // H - Height, always at index 2 for 4D tensors and index 3 for 5D tensors
        // W - Width, always at index 3 for 4D tensors and index 4 for 5D tensors
        constexpr int N = 0; // Batch size index

        if(_xDims.size() != _wDims.size())
        {
            throw std::invalid_argument("xDims and wDims must have the same number of dimensions.");
        }

        // Ensure xDims has at least 3 dimensions (N, C, and at least 1 spatial dimension)
        if(_xDims.size() < 3)
        {
            throw std::invalid_argument(
                "xDims must have at least 3 dimensions (N, C, and at least 1 spatial dimension).");
        }

        // Determine the number of spatial dimensions
        auto spatialDims = _xDims.size() - 2; // Exclude N and C

        // Validate that the convolution parameter vectors match the number of spatial dimensions
        if(_convPrePadding.size() != spatialDims || _convPostPadding.size() != spatialDims
           || _convDilation.size() != spatialDims || _convStride.size() != spatialDims)
        {
            throw std::invalid_argument(
                "Convolution parameter vectors must match the number of spatial dimensions.");
        }

        // Calculate output dimensions based on input dimensions and convolution parameters
        auto n = _xDims[N];
        auto cOut = _wDims[N];
        std::vector<int64_t> outputDims = {n, cOut};

        for(size_t i = 0; i < spatialDims; ++i)
        {
            auto paddedInputSize = _xDims[2 + i] + _convPrePadding[i] + _convPostPadding[i];
            auto effectiveKernelSize = (_convDilation[i] * (_wDims[2 + i] - 1)) + 1;
            auto dimOut = ((paddedInputSize - effectiveKernelSize) / _convStride[i]) + 1;
            outputDims.push_back(dimOut);
        }

        _yDims = outputDims;
    }

    friend std::ostream& operator<<(std::ostream& ss, const ConvTestCase& tc)
    {
        using namespace hipdnn_sdk::test_utilities;

        ss << "(x:";
        vecToStream(ss, tc._xDims);
        ss << " w:";
        vecToStream(ss, tc._wDims);
        ss << " y:";
        vecToStream(ss, tc._yDims);
        ss << " prePad:";
        vecToStream(ss, tc._convPrePadding);
        ss << " postPad:";
        vecToStream(ss, tc._convPostPadding);
        ss << " stride:";
        vecToStream(ss, tc._convStride);
        ss << " dilation:";
        vecToStream(ss, tc._convDilation);
        ss << " seed:" << tc._seed;
        ss << ")";

        return ss;
    }
};

inline std::vector<ConvTestCase> getConvTestCases4D()
{
    unsigned seed = std::random_device{}();

    return {
        // Filter 1x1
        {{1, 16, 16, 16}, {1, 16, 1, 1}, {0, 0}, {0, 0}, {1, 1}, {1, 1}, seed},
        // Filter 3x3
        // No Padding
        {{1, 16, 16, 16}, {1, 16, 3, 3}, {0, 0}, {0, 0}, {1, 1}, {1, 1}, seed},
        // Padding = 1
        {{1, 16, 16, 16}, {1, 16, 3, 3}, {1, 1}, {1, 1}, {1, 1}, {1, 1}, seed},
        // Stride = 2
        {{1, 16, 16, 16}, {1, 16, 3, 3}, {1, 1}, {1, 1}, {2, 2}, {1, 1}, seed},
        // Dilation = 2
        {{1, 16, 16, 16}, {1, 16, 3, 3}, {2, 2}, {2, 2}, {1, 1}, {2, 2}, seed},
        // Batched convolution
        {{8, 16, 16, 16}, {1, 16, 1, 1}, {0, 0}, {0, 0}, {1, 1}, {1, 1}, seed},
        // Non-square
        {{1, 16, 16, 8}, {1, 16, 3, 3}, {1, 1}, {1, 1}, {1, 1}, {1, 1}, seed},
    };
}

inline std::vector<ConvTestCase> getConvTestCases5D()
{
    unsigned seed = std::random_device{}();

    return {
        // Filter 1x1
        {{1, 16, 16, 16, 16}, {1, 16, 1, 1, 1}, {0, 0, 0}, {0, 0, 0}, {1, 1, 1}, {1, 1, 1}, seed},
        // Filter 3x3
        // No Padding
        {{1, 16, 16, 16, 16}, {1, 16, 3, 3, 3}, {0, 0, 0}, {0, 0, 0}, {1, 1, 1}, {1, 1, 1}, seed},
        // Padding = 1
        {{1, 16, 16, 16, 16}, {1, 16, 3, 3, 3}, {1, 1, 1}, {1, 1, 1}, {1, 1, 1}, {1, 1, 1}, seed},
        // Stride = 2
        {{1, 16, 16, 16, 16}, {1, 16, 3, 3, 3}, {1, 1, 1}, {1, 1, 1}, {2, 2, 2}, {1, 1, 1}, seed},
        // Dilation = 2
        {{1, 16, 16, 16, 16}, {1, 16, 3, 3, 3}, {2, 2, 2}, {2, 2, 2}, {1, 1, 1}, {2, 2, 2}, seed},
        // Batched convolution
        {{8, 16, 16, 16, 16}, {1, 16, 1, 1, 1}, {0, 0, 0}, {0, 0, 0}, {1, 1, 1}, {1, 1, 1}, seed},
        // Non-square
        {{1, 16, 16, 8, 4}, {1, 16, 3, 3, 3}, {1, 1, 1}, {1, 1, 1}, {1, 1, 1}, {1, 1, 1}, seed},
    };
}

} // namespace test_conv_common
