// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

package org.apache.doris.nereids.trees.expressions.functions.scalar;

import org.apache.doris.catalog.FunctionSignature;
import org.apache.doris.nereids.trees.expressions.Expression;
import org.apache.doris.nereids.trees.expressions.functions.CustomSignature;
import org.apache.doris.nereids.trees.expressions.functions.PropagateNullable;
import org.apache.doris.nereids.trees.expressions.visitor.ExpressionVisitor;
import org.apache.doris.nereids.types.DataType;
import org.apache.doris.nereids.types.JsonType;
import org.apache.doris.nereids.types.VarcharType;
import org.apache.doris.nereids.util.ExpressionUtils;

import com.google.common.base.Preconditions;

import java.util.ArrayList;
import java.util.List;

/**
 * ScalarFunction 'json_insert'. This class is generated by GenerateFunction.
 */
public class JsonInsert extends ScalarFunction implements CustomSignature, PropagateNullable {

    /**
     * constructor with 3 or more arguments.
     */
    public JsonInsert(Expression arg0, Expression arg1, Expression arg2, Expression... varArgs) {
        super("json_insert", ExpressionUtils.mergeArguments(arg0, arg1, arg2, varArgs));
    }

    @Override
    public FunctionSignature customSignature() {
        List<DataType> arguments = new ArrayList<>();
        arguments.add(VarcharType.SYSTEM_DEFAULT); // json_str
        for (int i = 1; i < arity(); i++) {
            if ((i & 1) == 0 && (getArgumentType(i).isComplexType() || getArgumentType(i).isJsonType())) {
                // keep origin type for BE Serialization
                arguments.add(JsonType.INSTANCE);
            } else {
                arguments.add(VarcharType.SYSTEM_DEFAULT);
            }
        }
        return FunctionSignature.of(VarcharType.SYSTEM_DEFAULT, arguments);
    }

    /**
     * withChildren.
     */
    @Override
    public JsonInsert withChildren(List<Expression> children) {
        Preconditions.checkArgument(children.size() >= 3);
        return new JsonInsert(children.get(0), children.get(1), children.get(2),
            children.subList(3, children.size()).toArray(new Expression[0]));
    }

    @Override
    public <R, C> R accept(ExpressionVisitor<R, C> visitor, C context) {
        return visitor.visitJsonInsert(this, context);
    }
}
