/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.fs.s3.common;

import com.amazonaws.services.securitytoken.model.Credentials;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Random;
import java.util.UUID;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicReference;
import javax.annotation.Nullable;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.configuration.MemorySize;
import org.apache.flink.core.fs.EntropyInjectingFileSystem;
import org.apache.flink.core.fs.ICloseableRegistry;
import org.apache.flink.core.fs.Path;
import org.apache.flink.core.fs.PathsCopyingFileSystem;
import org.apache.flink.core.fs.RecoverableWriter;
import org.apache.flink.core.fs.RefCountedFileWithStream;
import org.apache.flink.core.fs.RefCountedTmpFileCreator;
import org.apache.flink.fs.s3.common.AbstractS3FileSystemFactory;
import org.apache.flink.fs.s3.common.token.AbstractS3DelegationTokenReceiver;
import org.apache.flink.fs.s3.common.writer.S3AccessHelper;
import org.apache.flink.fs.s3.common.writer.S3RecoverableWriter;
import org.apache.flink.fs.s3hadoop.common.HadoopFileSystem;
import org.apache.flink.util.FileUtils;
import org.apache.flink.util.IOUtils;
import org.apache.flink.util.Preconditions;
import org.apache.flink.util.StringUtils;
import org.apache.flink.util.function.FunctionWithException;
import org.apache.hadoop.fs.FileSystem;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FlinkS3FileSystem
extends HadoopFileSystem
implements EntropyInjectingFileSystem,
PathsCopyingFileSystem {
    private static final Logger LOG = LoggerFactory.getLogger(FlinkS3FileSystem.class);
    private static final long PROCESS_KILL_SLEEP_TIME_MS = 50L;
    @Nullable
    private final String entropyInjectionKey;
    private final int entropyLength;
    public static final long S3_MULTIPART_MIN_PART_SIZE = 0x500000L;
    private final String localTmpDir;
    private final FunctionWithException<File, RefCountedFileWithStream, IOException> tmpFileCreator;
    @Nullable
    private final S3AccessHelper s3AccessHelper;
    private final Executor uploadThreadPool;
    private final long s3uploadPartSize;
    private final int maxConcurrentUploadsPerStream;
    @Nullable
    private final S5CmdConfiguration s5CmdConfiguration;

    public FlinkS3FileSystem(FileSystem hadoopS3FileSystem, @Nullable S5CmdConfiguration s5CmdConfiguration, String localTmpDirectory, @Nullable String entropyInjectionKey, int entropyLength, @Nullable S3AccessHelper s3UploadHelper, long s3uploadPartSize, int maxConcurrentUploadsPerStream) {
        super(hadoopS3FileSystem);
        this.s5CmdConfiguration = s5CmdConfiguration;
        if (entropyInjectionKey != null && entropyLength <= 0) {
            throw new IllegalArgumentException("Entropy length must be >= 0 when entropy injection key is set");
        }
        this.entropyInjectionKey = entropyInjectionKey;
        this.entropyLength = entropyLength;
        this.localTmpDir = (String)Preconditions.checkNotNull((Object)localTmpDirectory);
        this.tmpFileCreator = RefCountedTmpFileCreator.inDirectories((File[])new File[]{new File(localTmpDirectory)});
        this.s3AccessHelper = s3UploadHelper;
        this.uploadThreadPool = Executors.newCachedThreadPool();
        Preconditions.checkArgument((s3uploadPartSize >= 0x500000L ? 1 : 0) != 0);
        this.s3uploadPartSize = s3uploadPartSize;
        this.maxConcurrentUploadsPerStream = maxConcurrentUploadsPerStream;
        LOG.info("Created Flink S3 FS, s5Cmd configuration: {}", (Object)s5CmdConfiguration);
    }

    public boolean canCopyPaths(Path source, Path destination) {
        return this.canCopyPaths();
    }

    private boolean canCopyPaths() {
        return this.s5CmdConfiguration != null;
    }

    public void copyFiles(List<PathsCopyingFileSystem.CopyRequest> requests, ICloseableRegistry closeableRegistry) throws IOException {
        Preconditions.checkState((boolean)this.canCopyPaths(), (Object)"#downloadFiles has been called illegally");
        ArrayList<String> artefacts = new ArrayList<String>();
        artefacts.add(this.s5CmdConfiguration.path);
        artefacts.addAll(this.s5CmdConfiguration.args);
        artefacts.add("run");
        ArrayList<PathsCopyingFileSystem.CopyRequest> batch = new ArrayList<PathsCopyingFileSystem.CopyRequest>();
        long runningSizeBytes = 0L;
        long runningSizeFiles = 0L;
        for (int i = 0; i < requests.size(); ++i) {
            PathsCopyingFileSystem.CopyRequest request = requests.get(i);
            batch.add(request);
            if ((runningSizeBytes += request.getSize()) < this.s5CmdConfiguration.maxBatchSizeBytes && ++runningSizeFiles < this.s5CmdConfiguration.maxBatchSizeFiles && i != requests.size() - 1) continue;
            LOG.info("Copy {} files using s5cmd, total size: {}, args: {}", new Object[]{requests.size(), runningSizeBytes, artefacts});
            this.castSpell(this.convertToSpells(batch), closeableRegistry, artefacts.toArray(new String[0]));
            runningSizeFiles = 0L;
            runningSizeBytes = 0L;
            batch.clear();
        }
    }

    private List<String> convertToSpells(List<PathsCopyingFileSystem.CopyRequest> requests) throws IOException {
        ArrayList<String> spells = new ArrayList<String>();
        for (PathsCopyingFileSystem.CopyRequest request : requests) {
            Files.createDirectories(Paths.get(request.getDestination().toUri()).getParent(), new FileAttribute[0]);
            spells.add(String.format("cp %s %s", request.getSource().toUri().toString(), request.getDestination().getPath()));
        }
        return spells;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void castSpell(List<String> spells, ICloseableRegistry closeableRegistry, String ... artefacts) throws IOException {
        int exitCode = 0;
        AtomicReference maybeCloseableRegistryException = new AtomicReference();
        File tmpWorkingDir = new File(this.localTmpDir, "s5cmd_" + UUID.randomUUID());
        java.nio.file.Path tmpWorkingPath = Files.createDirectories(tmpWorkingDir.toPath(), new FileAttribute[0]);
        try {
            ProcessBuilder hogwart = new ProcessBuilder(artefacts).directory(tmpWorkingDir);
            this.s5CmdConfiguration.configureEnvironment(hogwart.environment());
            File inScrolls = new File(tmpWorkingDir, "s5cmd_input");
            Preconditions.checkState((boolean)inScrolls.createNewFile());
            File outScrolls = new File(tmpWorkingDir, "s5cmd_output");
            Preconditions.checkState((boolean)outScrolls.createNewFile());
            FileUtils.writeFileUtf8((File)inScrolls, (String)(String.join((CharSequence)System.lineSeparator(), spells) + System.lineSeparator()));
            Process wizard = hogwart.redirectErrorStream(true).redirectInput(inScrolls).redirectOutput(outScrolls).start();
            try (Closeable ignore = closeableRegistry.registerCloseableTemporarily(() -> {
                maybeCloseableRegistryException.set(new IOException("Copy process destroyed by CloseableRegistry."));
                FlinkS3FileSystem.destroyProcess(wizard);
            });){
                exitCode = wizard.waitFor();
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                FlinkS3FileSystem.destroyProcess(wizard);
                throw new IOException(this.createSpellErrorMessage(exitCode, outScrolls, artefacts), e);
            }
            catch (IOException e) {
                FlinkS3FileSystem.destroyProcess(wizard);
                throw new IOException(this.createSpellErrorMessage(exitCode, outScrolls, artefacts), e);
            }
            if (exitCode != 0) {
                throw new IOException(this.createSpellErrorMessage(exitCode, outScrolls, artefacts), (Throwable)maybeCloseableRegistryException.get());
            }
        }
        finally {
            IOUtils.deleteFileQuietly((java.nio.file.Path)tmpWorkingPath);
        }
    }

    private static void destroyProcess(Process processToDestroy) {
        LOG.info("Destroying s5cmd copy process.");
        processToDestroy.destroy();
        IOUtils.closeAllQuietly((AutoCloseable[])new AutoCloseable[]{processToDestroy.getInputStream(), processToDestroy.getOutputStream(), processToDestroy.getErrorStream()});
        FlinkS3FileSystem.sleepForProcessTermination(processToDestroy);
        if (!processToDestroy.isAlive()) {
            return;
        }
        LOG.info("Forcibly destroying s5cmd copy process.");
        processToDestroy.destroyForcibly();
        FlinkS3FileSystem.sleepForProcessTermination(processToDestroy);
        if (processToDestroy.isAlive()) {
            LOG.warn("Could not destroy s5cmd copy process.");
        }
    }

    private static void sleepForProcessTermination(Process processToDestroy) {
        if (processToDestroy.isAlive()) {
            try {
                Thread.sleep(50L);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
    }

    private String createSpellErrorMessage(int exitCode, File outFile, String ... artefacts) {
        String output = "Unknown: cannot read copy process output.";
        try {
            output = FileUtils.readFileUtf8((File)outFile);
        }
        catch (IOException e) {
            LOG.info("Error while reading s5cmd output from file {}.", (Object)outFile, (Object)e);
        }
        return "Failed to cast s5cmd spell [" + String.join((CharSequence)" ", artefacts) + "]" + String.format(" [exit code = %d]", exitCode) + " [cfg: " + this.s5CmdConfiguration + "]" + " maybe due to:\n" + output;
    }

    @Nullable
    public String getEntropyInjectionKey() {
        return this.entropyInjectionKey;
    }

    public String generateEntropy() {
        return StringUtils.generateRandomAlphanumericString((Random)ThreadLocalRandom.current(), (int)this.entropyLength);
    }

    public String getLocalTmpDir() {
        return this.localTmpDir;
    }

    @Override
    public RecoverableWriter createRecoverableWriter() throws IOException {
        if (this.s3AccessHelper == null) {
            throw new UnsupportedOperationException("This s3 file system implementation does not support recoverable writers.");
        }
        return S3RecoverableWriter.writer(this.getHadoopFileSystem(), this.tmpFileCreator, this.s3AccessHelper, this.uploadThreadPool, this.s3uploadPartSize, this.maxConcurrentUploadsPerStream);
    }

    public static class S5CmdConfiguration {
        private final String path;
        private final List<String> args;
        @Nullable
        private final String accessArtifact;
        @Nullable
        private final String secretArtifact;
        @Nullable
        private final String endpoint;
        private long maxBatchSizeFiles;
        private long maxBatchSizeBytes;

        public S5CmdConfiguration(String path, String args, @Nullable String accessArtifact, @Nullable String secretArtifact, @Nullable String endpoint, int maxBatchSizeFiles, long maxBatchSizeBytes) {
            if (!path.isEmpty()) {
                File s5CmdFile = new File(path);
                Preconditions.checkArgument((boolean)s5CmdFile.isFile(), (String)"Unable to find s5cmd binary under [%s]", (Object[])new Object[]{path});
                Preconditions.checkArgument((boolean)s5CmdFile.canExecute(), (String)"s5cmd binary under [%s] is not executable", (Object[])new Object[]{path});
            }
            this.path = path;
            this.args = Arrays.asList(args.split("\\s+"));
            this.accessArtifact = accessArtifact;
            this.secretArtifact = secretArtifact;
            this.endpoint = endpoint;
            this.maxBatchSizeFiles = maxBatchSizeFiles;
            this.maxBatchSizeBytes = maxBatchSizeBytes;
        }

        public static Optional<S5CmdConfiguration> of(Configuration flinkConfig) {
            return flinkConfig.getOptional(AbstractS3FileSystemFactory.S5CMD_PATH).map(s2 -> new S5CmdConfiguration((String)s2, (String)flinkConfig.get(AbstractS3FileSystemFactory.S5CMD_EXTRA_ARGS), (String)flinkConfig.get(AbstractS3FileSystemFactory.ACCESS_KEY), (String)flinkConfig.get(AbstractS3FileSystemFactory.SECRET_KEY), (String)flinkConfig.get(AbstractS3FileSystemFactory.ENDPOINT), (Integer)flinkConfig.get(AbstractS3FileSystemFactory.S5CMD_BATCH_MAX_FILES), ((MemorySize)flinkConfig.get(AbstractS3FileSystemFactory.S5CMD_BATCH_MAX_SIZE)).getBytes()));
        }

        private void configureEnvironment(Map<String, String> environment) {
            Credentials credentials = AbstractS3DelegationTokenReceiver.getCredentials();
            if (credentials != null) {
                S5CmdConfiguration.maybeSetEnvironmentVariable(environment, "AWS_ACCESS_KEY_ID", credentials.getAccessKeyId());
                S5CmdConfiguration.maybeSetEnvironmentVariable(environment, "AWS_SECRET_ACCESS_KEY", credentials.getSecretAccessKey());
                S5CmdConfiguration.maybeSetEnvironmentVariable(environment, "AWS_SESSION_TOKEN", credentials.getSessionToken());
            } else {
                S5CmdConfiguration.maybeSetEnvironmentVariable(environment, "AWS_ACCESS_KEY_ID", this.accessArtifact);
                S5CmdConfiguration.maybeSetEnvironmentVariable(environment, "AWS_SECRET_ACCESS_KEY", this.secretArtifact);
                S5CmdConfiguration.maybeSetEnvironmentVariable(environment, "S3_ENDPOINT_URL", this.endpoint);
            }
        }

        private static void maybeSetEnvironmentVariable(Map<String, String> environment, String key, @Nullable String value) {
            if (value == null) {
                return;
            }
            String oldValue = environment.put(key, value);
            if (oldValue != null) {
                LOG.warn("FlinkS3FileSystem configuration overwrote environment variable's [{}] old value [{}] with [{}]", new Object[]{key, oldValue, value});
            }
        }

        public String toString() {
            return "S5CmdConfiguration{path='" + this.path + "', args=" + this.args + ", accessArtifact='" + (this.accessArtifact == null ? null : "****") + "', secretArtifact='" + (this.secretArtifact == null ? null : "****") + "', endpoint='" + this.endpoint + "'}";
        }
    }
}

