/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.connector.kinesis.table.util;

import org.apache.flink.configuration.Configuration;
import org.apache.flink.connector.aws.config.AWSConfigConstants;
import org.apache.flink.connector.kinesis.table.util.KinesisStreamsConnectorSinkOptionsUtils.KinesisProducerOptionsMapper;

import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Test;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/** Unit tests for {@link KinesisProducerOptionsMapper}. */
class KinesisProducerOptionsMapperTest {

    @Test
    void testProducerVerifyCertificateOptionsMapping() {
        Map<String, String> deprecatedOptions = new HashMap<>();
        deprecatedOptions.put("sink.producer.verify-certificate", "false");
        Map<String, String> expectedOptions = new HashMap<>();
        expectedOptions.put(AWSConfigConstants.TRUST_ALL_CERTIFICATES, "true");

        KinesisProducerOptionsMapper producerOptionsMapper =
                new KinesisStreamsConnectorSinkOptionsUtils.KinesisProducerOptionsMapper(
                        deprecatedOptions);
        Map<String, String> actualMappedProperties =
                producerOptionsMapper.mapDeprecatedClientOptions();

        Assertions.assertThat(actualMappedProperties).isEqualTo(expectedOptions);
    }

    @Test
    void testProducerEndpointExtraction() {
        Map<String, String> deprecatedOptions = new HashMap<>();
        Map<String, String> expectedOptions = new HashMap<>();
        deprecatedOptions.put("sink.producer.kinesis-endpoint", "some-end-point.kinesis");
        expectedOptions.put(AWSConfigConstants.AWS_ENDPOINT, "https://some-end-point.kinesis");

        KinesisProducerOptionsMapper producerOptionsMapper =
                new KinesisProducerOptionsMapper(deprecatedOptions);
        Map<String, String> actualMappedProperties =
                producerOptionsMapper.mapDeprecatedClientOptions();

        Assertions.assertThat(actualMappedProperties).isEqualTo(expectedOptions);
    }

    @Test
    void testProducerEndpointAndPortExtraction() {
        Map<String, String> deprecatedOptions = new HashMap<>();
        Map<String, String> expectedOptions = new HashMap<>();
        deprecatedOptions.put("sink.producer.kinesis-endpoint", "some-end-point.kinesis");
        deprecatedOptions.put("sink.producer.kinesis-port", "1234");
        expectedOptions.put(AWSConfigConstants.AWS_ENDPOINT, "https://some-end-point.kinesis:1234");

        KinesisProducerOptionsMapper producerOptionsMapper =
                new KinesisProducerOptionsMapper(deprecatedOptions);
        Map<String, String> actualMappedProperties =
                producerOptionsMapper.mapDeprecatedClientOptions();

        Assertions.assertThat(actualMappedProperties).isEqualTo(expectedOptions);
    }

    @Test
    void testProducerOptionsMapperDoesNotModifyOptionsInstance() {
        Map<String, String> deprecatedOptions = new HashMap<>();
        deprecatedOptions.put("sink.producer.kinesis-endpoint", "some-end-point.kinesis");
        deprecatedOptions.put("sink.producer.kinesis-port", "1234");

        Map<String, String> deprecatedOptionsImmutable =
                Collections.unmodifiableMap(deprecatedOptions);
        Assertions.assertThatNoException()
                .isThrownBy(
                        () ->
                                new KinesisProducerOptionsMapper(deprecatedOptionsImmutable)
                                        .mapDeprecatedClientOptions());
        Assertions.assertThatNoException()
                .isThrownBy(
                        () ->
                                new KinesisProducerOptionsMapper(
                                                new Configuration(), deprecatedOptionsImmutable)
                                        .mapDeprecatedClientOptions());
    }
}
