/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hive.ql.exec.tez;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.AtomicDouble;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hive.common.metrics.common.Metrics;
import org.apache.hadoop.hive.common.metrics.common.MetricsFactory;
import org.apache.hadoop.hive.conf.HiveConf;
import org.apache.hadoop.hive.ql.exec.tez.TezSessionPoolManager;
import org.apache.hadoop.hive.ql.exec.tez.TezSessionState;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TezSessionPoolManagerMetrics {
    private static final Logger LOG = LoggerFactory.getLogger(TezSessionPoolManagerMetrics.class);
    public static final String TEZ_SESSION_METRIC_RUNNING_TASKS = "tez_session_running_tasks";
    public static final String TEZ_SESSION_METRIC_PENDING_TASKS = "tez_session_pending_tasks";
    public static final String TEZ_SESSION_METRIC_TASK_BACKLOG_RATIO = "tez_session_task_backlog_ratio";
    public static final String TEZ_SESSION_METRIC_TASK_BACKLOG_RATIO_PRESENT_SINCE = "tez_session_task_backlog_ratio_%d_present_since";
    private static final Integer[] TRACKED_TASK_BACKLOG_RATIOS = new Integer[]{1, 2, 5, 10, 20, 50, 100};
    private final TezSessionPoolManager poolManager;
    private final Metrics metrics;
    private ScheduledExecutorService mainExecutor;
    @VisibleForTesting
    final TezSessionMetric<Integer> runningTasksCount = new TezSessionMetric<Integer>("tez_session_running_tasks", 0);
    @VisibleForTesting
    final TezSessionMetric<Integer> pendingTasksCount = new TezSessionMetric<Integer>("tez_session_pending_tasks", 0);
    @VisibleForTesting
    final TezSessionMetric<Double> taskBacklogRatio = new TezSessionMetric<Double>("tez_session_task_backlog_ratio", 0.0);
    private final TaskBacklogRatioMetric[] taskBacklogRatioDurations = (TaskBacklogRatioMetric[])Arrays.stream(TRACKED_TASK_BACKLOG_RATIOS).map(ratio -> new TaskBacklogRatioMetric(String.format(TEZ_SESSION_METRIC_TASK_BACKLOG_RATIO_PRESENT_SINCE, ratio), (Integer)ratio)).toArray(TaskBacklogRatioMetric[]::new);

    public TezSessionPoolManagerMetrics(TezSessionPoolManager poolManager) {
        this.poolManager = poolManager;
        this.metrics = MetricsFactory.getInstance();
    }

    private static void waitForTasksToFinish(List<Future<?>> collectTasks, ExecutorService collectorExecutor) {
        try {
            CompletableFuture.allOf(collectTasks.toArray(new CompletableFuture[0])).get();
        }
        catch (InterruptedException e) {
            LOG.info("Interrupted while collecting session metrics", (Throwable)e);
            Thread.currentThread().interrupt();
        }
        catch (ExecutionException e) {
            LOG.error("Exception while collecting session metrics", e.getCause());
            throw new RuntimeException(e.getCause());
        }
        finally {
            collectorExecutor.shutdown();
        }
    }

    public TezSessionPoolManagerMetrics start(HiveConf conf) {
        if (this.metrics == null) {
            LOG.warn("Metrics are not enabled, cannot start TezSessionPoolManagerMetrics (check {})", (Object)HiveConf.ConfVars.HIVE_SERVER2_METRICS_ENABLED.varname);
            return this;
        }
        this.mainExecutor = Executors.newSingleThreadScheduledExecutor(new ThreadFactoryBuilder().setNameFormat("TezSessionPoolManagerMetrics worker thread").setDaemon(true).build());
        long collectIntervalSeconds = HiveConf.getTimeVar((Configuration)conf, (HiveConf.ConfVars)HiveConf.ConfVars.HIVE_SERVER2_TEZ_SESSIONS_METRICS_COLLECTION_INTERVAL, (TimeUnit)TimeUnit.SECONDS);
        this.mainExecutor.scheduleAtFixedRate(this::collectMetrics, 0L, collectIntervalSeconds, TimeUnit.SECONDS);
        LOG.info("Starting TezSessionPoolManagerMetrics, collection interval: {}s", (Object)collectIntervalSeconds);
        this.runningTasksCount.createGauge(this.metrics);
        this.pendingTasksCount.createGauge(this.metrics);
        this.taskBacklogRatio.createGauge(this.metrics);
        for (TaskBacklogRatioMetric m : this.taskBacklogRatioDurations) {
            m.createGauge(this.metrics);
        }
        this.clearMetrics();
        return this;
    }

    public void stop() {
        if (this.mainExecutor != null) {
            this.mainExecutor.shutdown();
            this.mainExecutor = null;
        }
    }

    void collectMetrics() {
        List<TezSessionState> sessions = this.poolManager.getSessions();
        LOG.debug("Updating metrics, session count: {}", (Object)sessions.size());
        if (sessions.isEmpty()) {
            this.clearMetrics();
            return;
        }
        AtomicDouble currentRunningTasksCount = new AtomicDouble();
        AtomicDouble currentPendingTasksCount = new AtomicDouble();
        ArrayList collectTasks = new ArrayList();
        ExecutorService collectorExecutor = Executors.newFixedThreadPool(Math.min(sessions.size(), 10), new ThreadFactoryBuilder().setNameFormat("TezSessionPoolManagerMetrics collector thread - #%d").build());
        long start = Time.monotonicNow();
        for (TezSessionState session : sessions) {
            collectTasks.add(CompletableFuture.runAsync(() -> {
                Map<String, Double> metrics = session.getMetrics();
                LOG.debug("Achieved metrics from Tez session ({}): {}", (Object)session.getSessionId(), metrics);
                currentRunningTasksCount.addAndGet(metrics.getOrDefault(this.runningTasksCount.name, 0.0).doubleValue());
                currentPendingTasksCount.addAndGet(metrics.getOrDefault(this.pendingTasksCount.name, 0.0).doubleValue());
            }, collectorExecutor));
        }
        TezSessionPoolManagerMetrics.waitForTasksToFinish(collectTasks, collectorExecutor);
        long elapsed = Time.monotonicNow() - start;
        this.refreshMetrics(currentRunningTasksCount, currentPendingTasksCount);
        LOG.debug("Collected metrics from {} sessions (in {}ms): {}", new Object[]{sessions.size(), elapsed, this.toMetricsString()});
    }

    @VisibleForTesting
    void refreshMetrics(AtomicDouble currentRunningTasksCount, AtomicDouble currentPendingTasksCount) {
        this.runningTasksCount.setValue((int)currentRunningTasksCount.get());
        this.pendingTasksCount.setValue((int)currentPendingTasksCount.get());
        double currentTaskBacklogRatio = (double)((Integer)this.pendingTasksCount.value).intValue() / (double)((Integer)this.runningTasksCount.value + 1);
        this.taskBacklogRatio.setValue(currentTaskBacklogRatio);
        for (TaskBacklogRatioMetric m : this.taskBacklogRatioDurations) {
            m.refresh(currentTaskBacklogRatio);
        }
    }

    private void clearMetrics() {
        this.runningTasksCount.setValue(0);
        this.pendingTasksCount.setValue(0);
        this.taskBacklogRatio.setValue(0.0);
        Arrays.stream(this.taskBacklogRatioDurations).forEach(m -> {
            m.timeSet = 0L;
        });
    }

    private String toMetricsString() {
        return String.format("[running tasks: %d, pending tasks: %d, task backlog ratio: %.2f, task backlog ratios seen: {%s}]", this.runningTasksCount.value, this.pendingTasksCount.value, this.taskBacklogRatio.value, this.getBackLogRatioDurationsString());
    }

    private String getBackLogRatioDurationsString() {
        long now = Time.monotonicNow();
        return Arrays.stream(this.taskBacklogRatioDurations).filter(m -> m.timeSet > 0L).map(m -> m.name + ": " + (now - m.timeSet) / 1000L + "s").collect(Collectors.joining(", "));
    }

    @VisibleForTesting
    static class TezSessionMetric<T> {
        String name;
        T value;
        long timeSet = 0L;

        TezSessionMetric(String name, T value) {
            this.name = name;
            this.value = value;
        }

        public void createGauge(Metrics metrics) {
            metrics.addGauge(this.name, () -> this.value);
        }

        public void setValue(T value) {
            this.value = value;
            this.timeSet = Time.monotonicNow();
        }
    }

    private static class TaskBacklogRatioMetric
    extends TezSessionMetric<Long> {
        private final Integer ratio;

        public TaskBacklogRatioMetric(String name, Integer ratio) {
            super(name, 0L);
            this.ratio = ratio;
        }

        @Override
        public void createGauge(Metrics metrics) {
            metrics.addGauge(this.name, () -> this.timeSet == 0L ? 0L : (Time.monotonicNow() - this.timeSet) / 1000L);
        }

        public void refresh(double currentTaskBacklogRatio) {
            if (currentTaskBacklogRatio >= (double)this.ratio.intValue()) {
                if (this.timeSet == 0L) {
                    this.timeSet = Time.monotonicNow();
                }
            } else {
                this.timeSet = 0L;
            }
        }
    }
}

