/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.s3a;

import java.io.Closeable;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InterruptedIOException;
import java.net.SocketTimeoutException;
import java.nio.ByteBuffer;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.IntFunction;
import javax.annotation.Nullable;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.classification.VisibleForTesting;
import org.apache.hadoop.fs.CanSetReadahead;
import org.apache.hadoop.fs.CanUnbuffer;
import org.apache.hadoop.fs.FSInputStream;
import org.apache.hadoop.fs.FileRange;
import org.apache.hadoop.fs.StreamCapabilities;
import org.apache.hadoop.fs.VectoredReadUtils;
import org.apache.hadoop.fs.impl.CombinedFileRange;
import org.apache.hadoop.fs.s3a.HttpChannelEOFException;
import org.apache.hadoop.fs.s3a.Invoker;
import org.apache.hadoop.fs.s3a.RangeNotSatisfiableEOFException;
import org.apache.hadoop.fs.s3a.S3AInputPolicy;
import org.apache.hadoop.fs.s3a.S3AReadOpContext;
import org.apache.hadoop.fs.s3a.S3AUtils;
import org.apache.hadoop.fs.s3a.S3ObjectAttributes;
import org.apache.hadoop.fs.s3a.VectoredIOContext;
import org.apache.hadoop.fs.s3a.impl.ChangeTracker;
import org.apache.hadoop.fs.s3a.impl.SDKStreamDrainer;
import org.apache.hadoop.fs.s3a.statistics.S3AInputStreamStatistics;
import org.apache.hadoop.fs.statistics.DurationTracker;
import org.apache.hadoop.fs.statistics.IOStatistics;
import org.apache.hadoop.fs.statistics.IOStatisticsAggregator;
import org.apache.hadoop.fs.statistics.IOStatisticsSource;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.util.Preconditions;
import org.apache.hadoop.util.StringUtils;
import org.apache.hadoop.util.functional.CallableRaisingIOE;
import org.apache.hadoop.util.functional.FutureIO;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.core.ResponseInputStream;
import software.amazon.awssdk.services.s3.model.GetObjectRequest;
import software.amazon.awssdk.services.s3.model.GetObjectResponse;

@InterfaceAudience.Private
@InterfaceStability.Evolving
public class S3AInputStream
extends FSInputStream
implements CanSetReadahead,
CanUnbuffer,
StreamCapabilities,
IOStatisticsSource {
    public static final String E_NEGATIVE_READAHEAD_VALUE = "Negative readahead value";
    public static final String OPERATION_OPEN = "open";
    public static final String OPERATION_REOPEN = "re-open";
    private static final boolean CLOSE_WRAPPED_STREAM_ON_NEGATIVE_READ = true;
    private static final int TMP_BUFFER_MAX_SIZE = 65536;
    private final AtomicBoolean stopVectoredIOOperations = new AtomicBoolean(false);
    private long pos;
    private volatile boolean closed;
    private ResponseInputStream<GetObjectResponse> wrappedStream;
    private final S3AReadOpContext context;
    private final InputStreamCallbacks client;
    private final ExecutorService boundedThreadPool;
    private final String bucket;
    private final String key;
    private final String pathStr;
    private final long contentLength;
    private final String uri;
    private static final Logger LOG = LoggerFactory.getLogger(S3AInputStream.class);
    private final S3AInputStreamStatistics streamStatistics;
    private S3AInputPolicy inputPolicy;
    private long readahead = 65536L;
    private final VectoredIOContext vectoredIOContext;
    private long nextReadPos;
    private long contentRangeFinish;
    private long contentRangeStart;
    private final ChangeTracker changeTracker;
    private final IOStatistics ioStatistics;
    private long asyncDrainThreshold;
    private final IOStatisticsAggregator threadIOStatistics;

    public S3AInputStream(S3AReadOpContext ctx, S3ObjectAttributes s3Attributes, InputStreamCallbacks client, S3AInputStreamStatistics streamStatistics, ExecutorService boundedThreadPool) {
        Preconditions.checkArgument((boolean)org.apache.commons.lang3.StringUtils.isNotEmpty((CharSequence)s3Attributes.getBucket()), (Object)"No Bucket");
        Preconditions.checkArgument((boolean)org.apache.commons.lang3.StringUtils.isNotEmpty((CharSequence)s3Attributes.getKey()), (Object)"No Key");
        long l = s3Attributes.getLen();
        Preconditions.checkArgument((l >= 0L ? 1 : 0) != 0, (Object)"Negative content length");
        this.context = ctx;
        this.bucket = s3Attributes.getBucket();
        this.key = s3Attributes.getKey();
        this.pathStr = s3Attributes.getPath().toString();
        this.contentLength = l;
        this.client = client;
        this.uri = "s3a://" + this.bucket + "/" + this.key;
        this.streamStatistics = streamStatistics;
        this.ioStatistics = streamStatistics.getIOStatistics();
        this.changeTracker = new ChangeTracker(this.uri, ctx.getChangeDetectionPolicy(), streamStatistics.getChangeTrackerStatistics(), s3Attributes);
        this.setInputPolicy(ctx.getInputPolicy());
        this.setReadahead(ctx.getReadahead());
        this.asyncDrainThreshold = ctx.getAsyncDrainThreshold();
        this.boundedThreadPool = boundedThreadPool;
        this.vectoredIOContext = this.context.getVectoredIOContext();
        this.threadIOStatistics = Objects.requireNonNull(ctx.getIOStatisticsAggregator());
    }

    private void setInputPolicy(S3AInputPolicy inputPolicy) {
        this.inputPolicy = inputPolicy;
        this.streamStatistics.inputPolicySet(inputPolicy.ordinal());
    }

    @VisibleForTesting
    public S3AInputPolicy getInputPolicy() {
        return this.inputPolicy;
    }

    private synchronized void reopen(String reason, long targetPos, long length, boolean forceAbort) throws IOException {
        if (this.isObjectStreamOpen()) {
            this.closeStream("reopen(" + reason + ")", forceAbort, false);
        }
        this.contentRangeFinish = S3AInputStream.calculateRequestLimit(this.inputPolicy, targetPos, length, this.contentLength, this.readahead);
        LOG.debug("reopen({}) for {} range[{}-{}], length={}, streamPosition={}, nextReadPosition={}, policy={}", new Object[]{this.uri, reason, targetPos, this.contentRangeFinish, length, this.pos, this.nextReadPos, this.inputPolicy});
        GetObjectRequest request = (GetObjectRequest)((GetObjectRequest.Builder)this.client.newGetRequestBuilder(this.key).range(S3AUtils.formatRange(targetPos, this.contentRangeFinish - 1L)).applyMutation(this.changeTracker::maybeApplyConstraint)).build();
        long opencount = this.streamStatistics.streamOpened();
        String operation = opencount == 0L ? OPERATION_OPEN : OPERATION_REOPEN;
        String text = String.format("%s %s at %d", operation, this.uri, targetPos);
        this.wrappedStream = (ResponseInputStream)Invoker.onceTrackingDuration(text, this.uri, this.streamStatistics.initiateGetRequest(), () -> this.client.getObject(request));
        this.changeTracker.processResponse((GetObjectResponse)this.wrappedStream.response(), operation, targetPos);
        this.contentRangeStart = targetPos;
        this.pos = targetPos;
    }

    public synchronized long getPos() throws IOException {
        return this.nextReadPos < 0L ? 0L : this.nextReadPos;
    }

    public synchronized void seek(long targetPos) throws IOException {
        this.checkNotClosed();
        if (targetPos < 0L) {
            throw new EOFException("Cannot seek to a negative offset " + targetPos);
        }
        if (this.contentLength <= 0L) {
            return;
        }
        this.nextReadPos = targetPos;
    }

    private void seekQuietly(long positiveTargetPos) {
        try {
            this.seek(positiveTargetPos);
        }
        catch (IOException ioe) {
            LOG.debug("Ignoring IOE on seek of {} to {}", new Object[]{this.uri, positiveTargetPos, ioe});
        }
    }

    private void seekInStream(long targetPos, long length) throws IOException {
        this.checkNotClosed();
        if (!this.isObjectStreamOpen()) {
            return;
        }
        long diff = targetPos - this.pos;
        if (diff > 0L) {
            boolean skipForward;
            int available = this.wrappedStream.available();
            long forwardSeekRange = Math.max(this.readahead, (long)available);
            long remainingInCurrentRequest = this.remainingInCurrentRequest();
            long forwardSeekLimit = Math.min(remainingInCurrentRequest, forwardSeekRange);
            boolean bl = skipForward = remainingInCurrentRequest > 0L && diff < forwardSeekLimit;
            if (skipForward) {
                LOG.debug("Forward seek on {}, of {} bytes", (Object)this.uri, (Object)diff);
                long skipped = this.wrappedStream.skip(diff);
                if (skipped > 0L) {
                    this.pos += skipped;
                }
                this.streamStatistics.seekForwards(diff, skipped);
                if (this.pos == targetPos) {
                    LOG.debug("Now at {}: bytes remaining in current request: {}", (Object)this.pos, (Object)this.remainingInCurrentRequest());
                    return;
                }
                LOG.warn("Failed to seek on {} to {}. Current position {}", new Object[]{this.uri, targetPos, this.pos});
            } else {
                this.streamStatistics.seekForwards(diff, 0L);
            }
        } else if (diff < 0L) {
            this.streamStatistics.seekBackwards(diff);
            if (this.inputPolicy.isAdaptive()) {
                LOG.info("Switching to Random IO seek policy");
                this.setInputPolicy(S3AInputPolicy.Random);
            }
        } else if (this.remainingInCurrentRequest() > 0L) {
            return;
        }
        this.closeStream("seekInStream()", false, false);
        this.pos = targetPos;
    }

    public boolean seekToNewSource(long targetPos) throws IOException {
        return false;
    }

    private void lazySeek(long targetPos, long len) throws IOException {
        Invoker invoker = this.context.getReadInvoker();
        invoker.retry("lazySeek to " + targetPos, this.pathStr, true, () -> {
            this.seekInStream(targetPos, len);
            if (!this.isObjectStreamOpen()) {
                this.reopen("read from new offset", targetPos, len, false);
            }
        });
    }

    private void incrementBytesRead(long bytesRead) {
        this.streamStatistics.bytesRead(bytesRead);
        if (this.context.stats != null && bytesRead > 0L) {
            this.context.stats.incrementBytesRead(bytesRead);
        }
    }

    public synchronized int read() throws IOException {
        this.checkNotClosed();
        if (this.contentLength == 0L || this.nextReadPos >= this.contentLength) {
            return -1;
        }
        try {
            this.lazySeek(this.nextReadPos, 1L);
        }
        catch (RangeNotSatisfiableEOFException e) {
            LOG.debug("Downgrading 416 response attempt to read at {} to -1 response", (Object)this.nextReadPos);
            return -1;
        }
        Invoker invoker = this.context.getReadInvoker();
        int byteRead = (Integer)invoker.retry("read", this.pathStr, true, () -> {
            int b;
            if (!this.isObjectStreamOpen()) {
                this.reopen("failure recovery", this.getPos(), 1L, false);
            }
            try {
                b = this.wrappedStream.read();
            }
            catch (SocketTimeoutException | HttpChannelEOFException e) {
                this.onReadFailure(e, true);
                throw e;
            }
            catch (IOException e) {
                this.onReadFailure(e, false);
                throw e;
            }
            return b;
        });
        if (byteRead >= 0) {
            ++this.pos;
            ++this.nextReadPos;
            this.incrementBytesRead(1L);
        } else {
            this.streamReadResultNegative();
        }
        return byteRead;
    }

    private void onReadFailure(IOException ioe, boolean forceAbort) {
        GetObjectResponse objectResponse;
        GetObjectResponse getObjectResponse = objectResponse = this.wrappedStream == null ? null : (GetObjectResponse)this.wrappedStream.response();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Got exception while trying to read from stream {}, client: {} object: {}, trying to recover: ", new Object[]{this.uri, this.client, objectResponse, ioe});
        } else {
            LOG.info("Got exception while trying to read from stream {}, client: {} object: {}, trying to recover: " + ioe, new Object[]{this.uri, this.client, objectResponse});
        }
        this.streamStatistics.readException();
        this.closeStream("failure recovery", forceAbort, false);
    }

    private void streamReadResultNegative() {
        this.closeStream("wrappedStream.read() returned -1", false, false);
    }

    public synchronized int read(byte[] buf, int off, int len) throws IOException {
        this.checkNotClosed();
        this.validatePositionedReadArgs(this.nextReadPos, buf, off, len);
        if (len == 0) {
            return 0;
        }
        if (this.contentLength == 0L || this.nextReadPos >= this.contentLength) {
            return -1;
        }
        try {
            this.lazySeek(this.nextReadPos, len);
        }
        catch (RangeNotSatisfiableEOFException e) {
            return -1;
        }
        Invoker invoker = this.context.getReadInvoker();
        this.streamStatistics.readOperationStarted(this.nextReadPos, len);
        int bytesRead = (Integer)invoker.retry("read", this.pathStr, true, () -> {
            int bytes;
            if (!this.isObjectStreamOpen()) {
                this.reopen("failure recovery", this.getPos(), 1L, false);
            }
            try {
                bytes = this.wrappedStream.read(buf, off, len);
            }
            catch (SocketTimeoutException | HttpChannelEOFException e) {
                this.onReadFailure(e, true);
                throw e;
            }
            catch (EOFException e) {
                LOG.debug("EOFException raised by http stream read(); downgrading to a -1 response", (Throwable)e);
                return -1;
            }
            catch (IOException e) {
                this.onReadFailure(e, false);
                throw e;
            }
            return bytes;
        });
        if (bytesRead > 0) {
            this.pos += (long)bytesRead;
            this.nextReadPos += (long)bytesRead;
            this.incrementBytesRead(bytesRead);
        } else {
            this.streamReadResultNegative();
        }
        this.streamStatistics.readOperationCompleted(len, bytesRead);
        return bytesRead;
    }

    private void checkNotClosed() throws IOException {
        if (this.closed) {
            throw new IOException(this.uri + ": " + "Stream is closed!");
        }
    }

    public synchronized void close() throws IOException {
        if (!this.closed) {
            this.closed = true;
            try {
                this.stopVectoredIOOperations.set(true);
                this.closeStream("close() operation", false, true);
                this.client.close();
                super.close();
            }
            finally {
                this.streamStatistics.close();
                this.mergeThreadIOStatistics(this.streamStatistics.getIOStatistics());
            }
        }
    }

    private void mergeThreadIOStatistics(IOStatistics streamIOStats) {
        this.threadIOStatistics.aggregate(streamIOStats);
    }

    private CompletableFuture<Boolean> closeStream(String reason, boolean forceAbort, boolean blocking) {
        CompletableFuture<Boolean> operation;
        if (!this.isObjectStreamOpen()) {
            return CompletableFuture.completedFuture(false);
        }
        long remaining = this.remainingInCurrentRequest();
        LOG.debug("Closing stream {}: {}", (Object)reason, (Object)(forceAbort ? "abort" : "soft"));
        boolean shouldAbort = forceAbort || remaining > this.readahead;
        SDKStreamDrainer<ResponseInputStream<GetObjectResponse>> drainer = new SDKStreamDrainer<ResponseInputStream<GetObjectResponse>>(this.uri, this.wrappedStream, shouldAbort, (int)remaining, this.streamStatistics, reason);
        if (blocking || shouldAbort || remaining <= this.asyncDrainThreshold) {
            operation = CompletableFuture.completedFuture(drainer.apply());
        } else {
            LOG.debug("initiating asynchronous drain of {} bytes", (Object)remaining);
            operation = this.client.submit(drainer);
        }
        this.wrappedStream = null;
        return operation;
    }

    @InterfaceStability.Unstable
    public synchronized boolean resetConnection() throws IOException {
        this.checkNotClosed();
        LOG.info("Forcing reset of connection to {}", (Object)this.uri);
        return (Boolean)FutureIO.awaitFuture(this.closeStream("reset()", true, true));
    }

    public synchronized int available() throws IOException {
        this.checkNotClosed();
        long remaining = this.remainingInFile();
        if (remaining > Integer.MAX_VALUE) {
            return Integer.MAX_VALUE;
        }
        return (int)remaining;
    }

    @InterfaceAudience.Private
    @InterfaceStability.Unstable
    public synchronized long remainingInFile() {
        return this.contentLength - this.pos;
    }

    @InterfaceAudience.Private
    @InterfaceStability.Unstable
    public synchronized long remainingInCurrentRequest() {
        return this.contentRangeFinish - this.pos;
    }

    @InterfaceAudience.Private
    @InterfaceStability.Unstable
    public synchronized long getContentRangeFinish() {
        return this.contentRangeFinish;
    }

    @InterfaceAudience.Private
    @InterfaceStability.Unstable
    public synchronized long getContentRangeStart() {
        return this.contentRangeStart;
    }

    public boolean markSupported() {
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @InterfaceStability.Unstable
    public String toString() {
        String s = this.streamStatistics.toString();
        S3AInputStream s3AInputStream = this;
        synchronized (s3AInputStream) {
            StringBuilder sb = new StringBuilder("S3AInputStream{");
            sb.append(this.uri);
            sb.append(" wrappedStream=").append(this.isObjectStreamOpen() ? OPERATION_OPEN : "closed");
            sb.append(" read policy=").append((Object)this.inputPolicy);
            sb.append(" pos=").append(this.pos);
            sb.append(" nextReadPos=").append(this.nextReadPos);
            sb.append(" contentLength=").append(this.contentLength);
            sb.append(" contentRangeStart=").append(this.contentRangeStart);
            sb.append(" contentRangeFinish=").append(this.contentRangeFinish);
            sb.append(" remainingInCurrentRequest=").append(this.remainingInCurrentRequest());
            sb.append(" ").append(this.changeTracker);
            sb.append(" ").append(this.vectoredIOContext);
            sb.append('\n').append(s);
            sb.append('}');
            return sb.toString();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void readFully(long position, byte[] buffer, int offset, int length) throws IOException {
        this.checkNotClosed();
        this.validatePositionedReadArgs(position, buffer, offset, length);
        this.streamStatistics.readFullyOperationStarted(position, length);
        if (length == 0) {
            return;
        }
        S3AInputStream s3AInputStream = this;
        synchronized (s3AInputStream) {
            long oldPos = this.getPos();
            try {
                int nbytes;
                this.seek(position);
                for (int nread = 0; nread < length; nread += nbytes) {
                    nbytes = this.read(buffer, offset + nread, length - nread);
                    if (nbytes >= 0) continue;
                    throw new EOFException("End of file reached before reading fully.");
                }
            }
            finally {
                this.seekQuietly(oldPos);
            }
        }
    }

    public int minSeekForVectorReads() {
        return this.vectoredIOContext.getMinSeekForVectorReads();
    }

    public int maxReadSizeForVectorReads() {
        return this.vectoredIOContext.getMaxReadSizeForVectorReads();
    }

    public void readVectored(List<? extends FileRange> ranges, IntFunction<ByteBuffer> allocate) throws IOException {
        LOG.debug("Starting vectored read on path {} for ranges {} ", (Object)this.pathStr, ranges);
        this.checkNotClosed();
        if (this.stopVectoredIOOperations.getAndSet(false)) {
            LOG.debug("Reinstating vectored read operation for path {} ", (Object)this.pathStr);
        }
        List sortedRanges = VectoredReadUtils.validateNonOverlappingAndReturnSortedRanges(ranges);
        for (FileRange fileRange : ranges) {
            this.validateRangeRequest(fileRange);
            CompletableFuture result = new CompletableFuture();
            fileRange.setData(result);
        }
        if (VectoredReadUtils.isOrderedDisjoint((List)sortedRanges, (int)1, (int)this.minSeekForVectorReads())) {
            LOG.debug("Not merging the ranges as they are disjoint");
            this.streamStatistics.readVectoredOperationStarted(sortedRanges.size(), sortedRanges.size());
            for (FileRange fileRange : sortedRanges) {
                ByteBuffer buffer = allocate.apply(fileRange.getLength());
                this.boundedThreadPool.submit(() -> this.readSingleRange(range, buffer));
            }
        } else {
            LOG.debug("Trying to merge the ranges as they are not disjoint");
            List combinedFileRanges = VectoredReadUtils.mergeSortedRanges((List)sortedRanges, (int)1, (int)this.minSeekForVectorReads(), (int)this.maxReadSizeForVectorReads());
            this.streamStatistics.readVectoredOperationStarted(sortedRanges.size(), combinedFileRanges.size());
            LOG.debug("Number of original ranges size {} , Number of combined ranges {} ", (Object)ranges.size(), (Object)combinedFileRanges.size());
            for (CombinedFileRange combinedFileRange : combinedFileRanges) {
                this.boundedThreadPool.submit(() -> this.readCombinedRangeAndUpdateChildren(combinedFileRange, allocate));
            }
        }
        LOG.debug("Finished submitting vectored read to threadpool on path {} for ranges {} ", (Object)this.pathStr, ranges);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void readCombinedRangeAndUpdateChildren(CombinedFileRange combinedFileRange, IntFunction<ByteBuffer> allocate) {
        LOG.debug("Start reading combined range {} from path {} ", (Object)combinedFileRange, (Object)this.pathStr);
        ResponseInputStream<GetObjectResponse> rangeContent = null;
        try {
            rangeContent = this.getS3ObjectInputStream("readCombinedFileRange", combinedFileRange.getOffset(), combinedFileRange.getLength());
            this.populateChildBuffers(combinedFileRange, (InputStream)rangeContent, allocate);
        }
        catch (Exception ex) {
            try {
                LOG.debug("Exception while reading a range {} from path {} ", new Object[]{combinedFileRange, this.pathStr, ex});
                for (FileRange child : combinedFileRange.getUnderlying()) {
                    child.getData().completeExceptionally(ex);
                }
            }
            catch (Throwable throwable) {
                IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{rangeContent});
                throw throwable;
            }
            IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{rangeContent});
        }
        IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{rangeContent});
        LOG.debug("Finished reading range {} from path {} ", (Object)combinedFileRange, (Object)this.pathStr);
    }

    private void populateChildBuffers(CombinedFileRange combinedFileRange, InputStream objectContent, IntFunction<ByteBuffer> allocate) throws IOException {
        if (combinedFileRange.getUnderlying().size() == 1) {
            FileRange child = (FileRange)combinedFileRange.getUnderlying().get(0);
            ByteBuffer buffer = allocate.apply(child.getLength());
            this.populateBuffer(child.getLength(), buffer, objectContent);
            child.getData().complete(buffer);
        } else {
            FileRange prev = null;
            for (FileRange child : combinedFileRange.getUnderlying()) {
                if (prev != null && prev.getOffset() + (long)prev.getLength() < child.getOffset()) {
                    long drainQuantity = child.getOffset() - prev.getOffset() - (long)prev.getLength();
                    this.drainUnnecessaryData(objectContent, drainQuantity);
                }
                ByteBuffer buffer = allocate.apply(child.getLength());
                this.populateBuffer(child.getLength(), buffer, objectContent);
                child.getData().complete(buffer);
                prev = child;
            }
        }
    }

    private void drainUnnecessaryData(InputStream objectContent, long drainQuantity) throws IOException {
        int drainBytes = 0;
        while ((long)drainBytes < drainQuantity) {
            int readCount;
            byte[] drainBuffer;
            if ((long)(drainBytes + 16384) <= drainQuantity) {
                drainBuffer = new byte[16384];
                readCount = objectContent.read(drainBuffer);
            } else {
                drainBuffer = new byte[(int)(drainQuantity - (long)drainBytes)];
                readCount = objectContent.read(drainBuffer);
            }
            drainBytes += readCount;
        }
        this.streamStatistics.readVectoredBytesDiscarded(drainBytes);
        LOG.debug("{} bytes drained from stream ", (Object)drainBytes);
    }

    private void validateRangeRequest(FileRange range) throws EOFException {
        VectoredReadUtils.validateRangeRequest((FileRange)range);
        if (range.getOffset() + (long)range.getLength() > this.contentLength) {
            String errMsg = String.format("Requested range [%d, %d) is beyond EOF for path %s", range.getOffset(), range.getLength(), this.pathStr);
            LOG.warn(errMsg);
            throw new RangeNotSatisfiableEOFException(errMsg, null);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void readSingleRange(FileRange range, ByteBuffer buffer) {
        LOG.debug("Start reading range {} from path {} ", (Object)range, (Object)this.pathStr);
        ResponseInputStream<GetObjectResponse> objectRange = null;
        try {
            long position = range.getOffset();
            int length = range.getLength();
            objectRange = this.getS3ObjectInputStream("readSingleRange", position, length);
            this.populateBuffer(length, buffer, (InputStream)objectRange);
            range.getData().complete(buffer);
        }
        catch (Exception ex) {
            try {
                LOG.warn("Exception while reading a range {} from path {} ", new Object[]{range, this.pathStr, ex});
                range.getData().completeExceptionally(ex);
            }
            catch (Throwable throwable) {
                IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{objectRange});
                throw throwable;
            }
            IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{objectRange});
        }
        IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{objectRange});
        LOG.debug("Finished reading range {} from path {} ", (Object)range, (Object)this.pathStr);
    }

    private ResponseInputStream<GetObjectResponse> getS3ObjectInputStream(String operationName, long position, int length) throws IOException {
        this.checkIfVectoredIOStopped();
        ResponseInputStream<GetObjectResponse> objectRange = this.getS3Object(operationName, position, length);
        this.checkIfVectoredIOStopped();
        return objectRange;
    }

    private void populateBuffer(int length, ByteBuffer buffer, InputStream objectContent) throws IOException {
        if (buffer.isDirect()) {
            VectoredReadUtils.readInDirectBuffer((int)length, (ByteBuffer)buffer, (position, tmp, offset, currentLength) -> {
                this.readByteArray(objectContent, (byte[])tmp, (int)offset, (int)currentLength);
                return null;
            });
            buffer.flip();
        } else {
            this.readByteArray(objectContent, buffer.array(), 0, length);
        }
        this.incrementBytesRead(length);
    }

    private void readByteArray(InputStream objectContent, byte[] dest, int offset, int length) throws IOException {
        int readBytesCurr;
        for (int readBytes = 0; readBytes < length; readBytes += readBytesCurr) {
            readBytesCurr = objectContent.read(dest, offset + readBytes, length - readBytes);
            if (readBytesCurr >= 0) continue;
            throw new EOFException("End of file reached before reading fully.");
        }
    }

    private ResponseInputStream<GetObjectResponse> getS3Object(String operationName, long position, int length) throws IOException {
        ResponseInputStream objectRange;
        GetObjectRequest request = (GetObjectRequest)((GetObjectRequest.Builder)this.client.newGetRequestBuilder(this.key).range(S3AUtils.formatRange(position, position + (long)length - 1L)).applyMutation(this.changeTracker::maybeApplyConstraint)).build();
        Invoker invoker = this.context.getReadInvoker();
        try (DurationTracker tracker = this.streamStatistics.initiateGetRequest();){
            objectRange = (ResponseInputStream)invoker.retry(operationName, this.pathStr, true, () -> {
                this.checkIfVectoredIOStopped();
                return this.client.getObject(request);
            });
        }
        this.changeTracker.processResponse((GetObjectResponse)objectRange.response(), operationName, position);
        return objectRange;
    }

    private void checkIfVectoredIOStopped() throws InterruptedIOException {
        if (this.stopVectoredIOOperations.get()) {
            throw new InterruptedIOException("Stream closed or unbuffer is called");
        }
    }

    @InterfaceAudience.Private
    @InterfaceStability.Unstable
    @VisibleForTesting
    public S3AInputStreamStatistics getS3AStreamStatistics() {
        return this.streamStatistics;
    }

    public synchronized void setReadahead(Long readahead) {
        this.readahead = S3AInputStream.validateReadahead(readahead);
    }

    public synchronized long getReadahead() {
        return this.readahead;
    }

    static long calculateRequestLimit(S3AInputPolicy inputPolicy, long targetPos, long length, long contentLength, long readahead) {
        long rangeLimit;
        switch (inputPolicy) {
            case Random: {
                rangeLimit = length < 0L ? contentLength : targetPos + Math.max(readahead, length);
                break;
            }
            case Sequential: {
                rangeLimit = contentLength;
                break;
            }
            default: {
                rangeLimit = contentLength;
            }
        }
        rangeLimit = Math.min(contentLength, rangeLimit);
        return rangeLimit;
    }

    public static long validateReadahead(@Nullable Long readahead) {
        if (readahead == null) {
            return 65536L;
        }
        Preconditions.checkArgument((readahead >= 0L ? 1 : 0) != 0, (Object)E_NEGATIVE_READAHEAD_VALUE);
        return readahead;
    }

    public synchronized void unbuffer() {
        try {
            this.stopVectoredIOOperations.set(true);
            this.closeStream("unbuffer()", false, false);
        }
        finally {
            this.streamStatistics.unbuffered();
            if (this.inputPolicy.isAdaptive()) {
                S3AInputPolicy policy = S3AInputPolicy.Random;
                LOG.debug("Switching to seek policy {} after unbuffer() invoked", (Object)policy);
                this.setInputPolicy(policy);
            }
        }
    }

    public boolean hasCapability(String capability) {
        switch (StringUtils.toLowerCase((String)capability)) {
            case "iostatistics": 
            case "fs.capability.iocontext.supported": 
            case "in:readahead": 
            case "in:unbuffer": 
            case "in:readvectored": {
                return true;
            }
        }
        return false;
    }

    @VisibleForTesting
    public boolean isObjectStreamOpen() {
        return this.wrappedStream != null;
    }

    public IOStatistics getIOStatistics() {
        return this.ioStatistics;
    }

    @VisibleForTesting
    public ResponseInputStream<GetObjectResponse> getWrappedStream() {
        return this.wrappedStream;
    }

    public static interface InputStreamCallbacks
    extends Closeable {
        public GetObjectRequest.Builder newGetRequestBuilder(String var1);

        public ResponseInputStream<GetObjectResponse> getObject(GetObjectRequest var1);

        public <T> CompletableFuture<T> submit(CallableRaisingIOE<T> var1);
    }
}

