/*
 * Decompiled with CFR 0.152.
 */
package io.fabric8.kubernetes.client.extended.leaderelection;

import io.fabric8.kubernetes.client.KubernetesClient;
import io.fabric8.kubernetes.client.KubernetesClientException;
import io.fabric8.kubernetes.client.extended.leaderelection.LeaderElectionConfig;
import io.fabric8.kubernetes.client.extended.leaderelection.resourcelock.LeaderElectionRecord;
import io.fabric8.kubernetes.client.extended.leaderelection.resourcelock.Lock;
import io.fabric8.kubernetes.client.extended.leaderelection.resourcelock.LockException;
import io.fabric8.kubernetes.client.utils.Utils;
import java.time.Duration;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.function.Supplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LeaderElector {
    private static final Logger LOGGER = LoggerFactory.getLogger(LeaderElector.class);
    protected static final Double JITTER_FACTOR = 1.2;
    private KubernetesClient kubernetesClient;
    private LeaderElectionConfig leaderElectionConfig;
    private final AtomicReference<LeaderElectionRecord> observedRecord = new AtomicReference();
    private final AtomicReference<LocalDateTime> observedTime = new AtomicReference();
    private final Executor executor;

    public LeaderElector(KubernetesClient kubernetesClient, LeaderElectionConfig leaderElectionConfig, Executor executor) {
        this.kubernetesClient = kubernetesClient;
        this.leaderElectionConfig = leaderElectionConfig;
        this.executor = executor;
    }

    public void run() {
        CompletableFuture<?> acquire = this.start();
        try {
            acquire.get();
        }
        catch (InterruptedException e) {
            acquire.cancel(true);
            Thread.currentThread().interrupt();
        }
        catch (ExecutionException e) {
            LOGGER.error("Exception during leader election", (Throwable)e);
        }
    }

    public CompletableFuture<?> start() {
        LOGGER.debug("Leader election started");
        CompletableFuture result = new CompletableFuture();
        CompletableFuture<Void> acquireFuture = this.acquire();
        acquireFuture.whenComplete((v, t) -> {
            if (t == null) {
                CompletableFuture<Void> renewFuture = this.renewWithTimeout();
                result.whenComplete((v1, t1) -> renewFuture.cancel(true));
                renewFuture.whenComplete((v1, t1) -> {
                    if (t1 != null) {
                        result.completeExceptionally((Throwable)t1);
                    } else {
                        result.complete(null);
                    }
                });
            }
        });
        result.whenComplete((v, t) -> {
            acquireFuture.cancel(true);
            LeaderElectionRecord current = this.observedRecord.get();
            if (current != null && Objects.equals(current.getHolderIdentity(), this.leaderElectionConfig.getLock().identity())) {
                this.leaderElectionConfig.getLeaderCallbacks().onStopLeading();
            }
        });
        return result;
    }

    private CompletableFuture<Void> acquire() {
        String lockDescription = this.leaderElectionConfig.getLock().describe();
        LOGGER.debug("Attempting to acquire leader lease '{}'...", (Object)lockDescription);
        return LeaderElector.loop(completion -> {
            try {
                if (this.tryAcquireOrRenew()) {
                    completion.complete(null);
                }
                LOGGER.debug("Failed to acquire lease '{}' retrying...", (Object)lockDescription);
            }
            catch (KubernetesClientException | LockException exception) {
                LOGGER.error("Exception occurred while acquiring lock '{}'", (Object)lockDescription, (Object)exception);
            }
        }, () -> LeaderElector.jitter(this.leaderElectionConfig.getRetryPeriod(), JITTER_FACTOR).toMillis(), this.executor);
    }

    private CompletableFuture<Void> renewWithTimeout() {
        String lockDescription = this.leaderElectionConfig.getLock().describe();
        LOGGER.debug("Attempting to renew leader lease '{}'...", (Object)lockDescription);
        AtomicLong renewBy = new AtomicLong(System.currentTimeMillis() + this.leaderElectionConfig.getRenewDeadline().toMillis());
        return LeaderElector.loop(completion -> {
            if (System.currentTimeMillis() > renewBy.get()) {
                LOGGER.debug("Renew deadline reached after {} seconds while renewing lock {}", (Object)this.leaderElectionConfig.getRenewDeadline().get(ChronoUnit.SECONDS), (Object)lockDescription);
                completion.complete(null);
                return;
            }
            try {
                if (this.tryAcquireOrRenew()) {
                    renewBy.set(System.currentTimeMillis() + this.leaderElectionConfig.getRenewDeadline().toMillis());
                } else {
                    completion.complete(null);
                }
            }
            catch (KubernetesClientException | LockException exception) {
                LOGGER.debug("Exception occurred while renewing lock: {}", (Object)exception.getMessage(), (Object)exception);
            }
        }, () -> this.leaderElectionConfig.getRetryPeriod().toMillis(), this.executor);
    }

    private boolean tryAcquireOrRenew() throws LockException {
        Lock lock = this.leaderElectionConfig.getLock();
        ZonedDateTime now = LeaderElector.now();
        LeaderElectionRecord oldLeaderElectionRecord = lock.get(this.kubernetesClient);
        if (oldLeaderElectionRecord == null) {
            LeaderElectionRecord newLeaderElectionRecord = new LeaderElectionRecord(lock.identity(), this.leaderElectionConfig.getLeaseDuration(), now, now, 0);
            lock.create(this.kubernetesClient, newLeaderElectionRecord);
            this.updateObserved(newLeaderElectionRecord);
            return true;
        }
        this.updateObserved(oldLeaderElectionRecord);
        boolean isLeader = this.isLeader(oldLeaderElectionRecord);
        if (!isLeader && !this.canBecomeLeader(oldLeaderElectionRecord)) {
            LOGGER.debug("Lock is held by {} and has not yet expired", (Object)oldLeaderElectionRecord.getHolderIdentity());
            return false;
        }
        LeaderElectionRecord newLeaderElectionRecord = new LeaderElectionRecord(lock.identity(), this.leaderElectionConfig.getLeaseDuration(), isLeader ? oldLeaderElectionRecord.getAcquireTime() : now, now, isLeader ? oldLeaderElectionRecord.getLeaderTransitions() + 1 : 0);
        newLeaderElectionRecord.setVersion(oldLeaderElectionRecord.getVersion());
        this.leaderElectionConfig.getLock().update(this.kubernetesClient, newLeaderElectionRecord);
        this.updateObserved(newLeaderElectionRecord);
        return true;
    }

    private void updateObserved(LeaderElectionRecord leaderElectionRecord) {
        LeaderElectionRecord current = this.observedRecord.getAndSet(leaderElectionRecord);
        if (!Objects.equals(leaderElectionRecord, current)) {
            this.observedTime.set(LocalDateTime.now());
            String currentLeader = current == null ? null : current.getHolderIdentity();
            String newLeader = leaderElectionRecord.getHolderIdentity();
            if (!Objects.equals(newLeader, currentLeader)) {
                LOGGER.debug("Leader changed from {} to {}", (Object)currentLeader, (Object)newLeader);
                this.leaderElectionConfig.getLeaderCallbacks().onNewLeader(newLeader);
                if (Objects.equals(currentLeader, this.leaderElectionConfig.getLock().identity())) {
                    this.leaderElectionConfig.getLeaderCallbacks().onStopLeading();
                } else if (Objects.equals(newLeader, this.leaderElectionConfig.getLock().identity())) {
                    LOGGER.debug("Successfully Acquired leader lease '{}'", (Object)this.leaderElectionConfig.getLock().describe());
                    this.leaderElectionConfig.getLeaderCallbacks().onStartLeading();
                }
            }
        }
    }

    protected final boolean isLeader(LeaderElectionRecord leaderElectionRecord) {
        return Objects.equals(this.leaderElectionConfig.getLock().identity(), leaderElectionRecord.getHolderIdentity());
    }

    protected final boolean canBecomeLeader(LeaderElectionRecord leaderElectionRecord) {
        return !leaderElectionRecord.getRenewTime().plus(this.leaderElectionConfig.getLeaseDuration()).isAfter(LeaderElector.now());
    }

    protected static CompletableFuture<Void> loop(Consumer<CompletableFuture<?>> consumer, Supplier<Long> delaySupplier, Executor executor) {
        CompletableFuture<Void> completion = new CompletableFuture<Void>();
        Utils.scheduleWithVariableRate(completion, executor, () -> consumer.accept(completion), 0L, delaySupplier, TimeUnit.MILLISECONDS);
        return completion;
    }

    protected static ZonedDateTime now() {
        return ZonedDateTime.now(ZoneOffset.UTC);
    }

    protected static Duration jitter(Duration duration, double maxFactor) {
        maxFactor = maxFactor > 0.0 ? maxFactor : 1.0;
        return duration.plusMillis(Double.valueOf((double)duration.toMillis() * Math.random() * maxFactor).longValue());
    }
}

