/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.juneau.rest.annotation;

import static org.apache.juneau.commons.utils.CollectionUtils.*;

import java.lang.annotation.*;
import java.nio.charset.*;

import org.apache.juneau.*;
import org.apache.juneau.encoders.*;
import org.apache.juneau.http.*;
import org.apache.juneau.commons.annotation.*;
import org.apache.juneau.commons.reflect.*;
import org.apache.juneau.rest.*;
import org.apache.juneau.rest.converter.*;
import org.apache.juneau.rest.guard.*;
import org.apache.juneau.rest.httppart.*;
import org.apache.juneau.rest.matcher.*;
import org.apache.juneau.serializer.*;
import org.apache.juneau.svl.*;

/**
 * Utility classes and methods for the {@link RestPost @RestPost} annotation.
 *
 * <h5 class='section'>See Also:</h5><ul>
 * 	<li class='link'><a class="doclink" href="https://juneau.apache.org/docs/topics/RestOpAnnotatedMethodBasics">@RestOp-Annotated Method Basics</a>
 * </ul>
 */
public class RestPostAnnotation {
	/**
	 * Builder class.
	 *
	 * <h5 class='section'>See Also:</h5><ul>
	 * 	<li class='jm'>{@link org.apache.juneau.BeanContext.Builder#annotations(Annotation...)}
	 * </ul>
	 */
	@SuppressWarnings("unchecked")
	public static class Builder extends AppliedAnnotationObject.BuilderM {

		private String[] description = {};
		private Class<? extends RestConverter>[] converters = new Class[0];
		private Class<? extends RestGuard>[] guards = new Class[0];
		private Class<? extends RestMatcher>[] matchers = new Class[0];
		private Class<? extends Encoder>[] encoders = new Class[0];
		private Class<? extends Serializer>[] serializers = new Class[0];
		private Class<?>[] parsers = {};
		private OpSwagger swagger = OpSwaggerAnnotation.DEFAULT;
		private String clientVersion = "", debug = "", defaultAccept = "", defaultCharset = "", defaultContentType = "", maxInput = "", rolesDeclared = "", roleGuard = "", summary = "", value = "";
		private String[] consumes = {}, defaultRequestFormData = {}, defaultRequestQueryData = {}, defaultRequestAttributes = {}, defaultRequestHeaders = {}, defaultResponseHeaders = {}, path = {},
			produces = {};

		/**
		 * Constructor.
		 */
		protected Builder() {
			super(RestPost.class);
		}

		/**
		 * Instantiates a new {@link RestPost @RestPost} object initialized with this builder.
		 *
		 * @return A new {@link RestPost @RestPost} object.
		 */
		public RestPost build() {
			return new Object(this);
		}

		/**
		 * Sets the description property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder description(String...value) {
			description = value;
			return this;
		}

		/**
		 * Sets the {@link RestPost#clientVersion()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder clientVersion(String value) {
			clientVersion = value;
			return this;
		}

		/**
		 * Sets the {@link RestPost#consumes()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder consumes(String...value) {
			consumes = value;
			return this;
		}

		/**
		 * Sets the {@link RestPost#converters()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		@SafeVarargs
		public final Builder converters(Class<? extends RestConverter>...value) {
			converters = value;
			return this;
		}

		/**
		 * Sets the {@link RestPost#debug()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder debug(String value) {
			debug = value;
			return this;
		}

		/**
		 * Sets the {@link RestPost#defaultAccept()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder defaultAccept(String value) {
			defaultAccept = value;
			return this;
		}

		/**
		 * Sets the {@link RestPost#defaultCharset()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder defaultCharset(String value) {
			defaultCharset = value;
			return this;
		}

		/**
		 * Sets the {@link RestPost#defaultContentType()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder defaultContentType(String value) {
			defaultContentType = value;
			return this;
		}

		/**
		 * Sets the {@link RestPost#defaultRequestAttributes()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder defaultRequestAttributes(String...value) {
			defaultRequestAttributes = value;
			return this;
		}

		/**
		 * Sets the {@link RestPost#defaultRequestFormData()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder defaultRequestFormData(String...value) {
			defaultRequestFormData = value;
			return this;
		}

		/**
		 * Sets the {@link RestPost#defaultRequestHeaders()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder defaultRequestHeaders(String...value) {
			defaultRequestHeaders = value;
			return this;
		}

		/**
		 * Sets the {@link RestPost#defaultRequestQueryData()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder defaultRequestQueryData(String...value) {
			defaultRequestQueryData = value;
			return this;
		}

		/**
		 * Sets the {@link RestPost#defaultResponseHeaders()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder defaultResponseHeaders(String...value) {
			defaultResponseHeaders = value;
			return this;
		}

		/**
		 * Sets the {@link RestPost#encoders()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		@SafeVarargs
		public final Builder encoders(Class<? extends Encoder>...value) {
			encoders = value;
			return this;
		}

		/**
		 * Sets the {@link RestPost#guards()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		@SafeVarargs
		public final Builder guards(Class<? extends RestGuard>...value) {
			guards = value;
			return this;
		}

		/**
		 * Sets the {@link RestPost#matchers()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		@SafeVarargs
		public final Builder matchers(Class<? extends RestMatcher>...value) {
			matchers = value;
			return this;
		}

		/**
		 * Sets the {@link RestPost#maxInput()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder maxInput(String value) {
			maxInput = value;
			return this;
		}

		/**
		 * Sets the {@link RestPost#parsers()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder parsers(Class<?>...value) {
			parsers = value;
			return this;
		}

		/**
		 * Sets the {@link RestPost#path()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder path(String...value) {
			path = value;
			return this;
		}

		/**
		 * Sets the {@link RestPost#produces()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder produces(String...value) {
			produces = value;
			return this;
		}

		/**
		 * Sets the {@link RestPost#roleGuard()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder roleGuard(String value) {
			roleGuard = value;
			return this;
		}

		/**
		 * Sets the {@link RestPost#rolesDeclared()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder rolesDeclared(String value) {
			rolesDeclared = value;
			return this;
		}

		/**
		 * Sets the {@link RestPost#serializers()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		@SafeVarargs
		public final Builder serializers(Class<? extends Serializer>...value) {
			serializers = value;
			return this;
		}

		/**
		 * Sets the {@link RestPost#summary()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder summary(String value) {
			summary = value;
			return this;
		}

		/**
		 * Sets the {@link RestPost#swagger()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder swagger(OpSwagger value) {
			swagger = value;
			return this;
		}

		/**
		 * Sets the {@link RestPost#value()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder value(String value) {
			this.value = value;
			return this;
		}

		@Override /* Overridden from AppliedAnnotationObject.Builder */
		public Builder on(String...value) {
			super.on(value);
			return this;
		}

		@Override /* Overridden from AppliedAnnotationObject.BuilderM */
		public Builder on(java.lang.reflect.Method...value) {
			super.on(value);
			return this;
		}
	
		@Override /* Overridden from AppliedAnnotationObject.BuilderM */
		public Builder on(MethodInfo...value) {
			super.on(value);
			return this;
		}

	}

	/**
	 * Applies {@link RestPost} annotations to a {@link org.apache.juneau.rest.RestOpContext.Builder}.
	 */
	public static class RestOpContextApply extends AnnotationApplier<RestPost,RestOpContext.Builder> {

		/**
		 * Constructor.
		 *
		 * @param vr The resolver for resolving values in annotations.
		 */
		public RestOpContextApply(VarResolverSession vr) {
			super(RestPost.class, RestOpContext.Builder.class, vr);
		}

		@Override
		public void apply(AnnotationInfo<RestPost> ai, RestOpContext.Builder b) {
			RestPost a = ai.inner();

			b.httpMethod("post");

			classes(a.serializers()).ifPresent(x -> b.serializers().set(x));
			classes(a.parsers()).ifPresent(x -> b.parsers().set(x));
			classes(a.encoders()).ifPresent(x -> b.encoders().set(x));
			stream(a.produces()).map(MediaType::of).forEach(x -> b.produces(x));
			stream(a.consumes()).map(MediaType::of).forEach(x -> b.consumes(x));
			stream(a.defaultRequestHeaders()).map(HttpHeaders::stringHeader).forEach(x -> b.defaultRequestHeaders().setDefault(x));
			stream(a.defaultResponseHeaders()).map(HttpHeaders::stringHeader).forEach(x -> b.defaultResponseHeaders().setDefault(x));
			stream(a.defaultRequestAttributes()).map(BasicNamedAttribute::ofPair).forEach(x -> b.defaultRequestAttributes().add(x));
			stream(a.defaultRequestQueryData()).map(HttpParts::basicPart).forEach(x -> b.defaultRequestQueryData().setDefault(x));
			stream(a.defaultRequestFormData()).map(HttpParts::basicPart).forEach(x -> b.defaultRequestFormData().setDefault(x));
			string(a.defaultAccept()).map(HttpHeaders::accept).ifPresent(x -> b.defaultRequestHeaders().setDefault(x));
			string(a.defaultContentType()).map(HttpHeaders::contentType).ifPresent(x -> b.defaultRequestHeaders().setDefault(x));
			b.converters().append(a.converters());
			b.guards().append(a.guards());
			b.matchers().append(a.matchers());
			string(a.clientVersion()).ifPresent(x -> b.clientVersion(x));
			string(a.defaultCharset()).map(Charset::forName).ifPresent(x -> b.defaultCharset(x));
			string(a.maxInput()).ifPresent(x -> b.maxInput(x));
			stream(a.path()).forEach(x -> b.path(x));
			string(a.value()).ifPresent(x -> b.path(x));
			cdl(a.rolesDeclared()).forEach(x -> b.rolesDeclared(x));
			string(a.roleGuard()).ifPresent(x -> b.roleGuard(x));
			string(a.debug()).map(Enablement::fromString).ifPresent(x -> b.debug(x));
		}

	}

	private static class Object extends AppliedAnnotationObject implements RestPost {

		private final String[] description;
		private final Class<? extends RestConverter>[] converters;
		private final Class<? extends RestGuard>[] guards;
		private final Class<? extends RestMatcher>[] matchers;
		private final Class<? extends Encoder>[] encoders;
		private final Class<? extends Serializer>[] serializers;
		private final Class<?>[] parsers;
		private final OpSwagger swagger;
		private final String clientVersion, debug, defaultAccept, defaultCharset, defaultContentType, maxInput, rolesDeclared, roleGuard, summary, value;
		private final String[] consumes, defaultRequestFormData, defaultRequestQueryData, defaultRequestAttributes, defaultRequestHeaders, defaultResponseHeaders, path, produces;

		Object(RestPostAnnotation.Builder b) {
			super(b);
			description = copyOf(b.description);
			clientVersion = b.clientVersion;
			consumes = copyOf(b.consumes);
			converters = copyOf(b.converters);
			debug = b.debug;
			defaultAccept = b.defaultAccept;
			defaultCharset = b.defaultCharset;
			defaultContentType = b.defaultContentType;
			defaultRequestFormData = copyOf(b.defaultRequestFormData);
			defaultRequestQueryData = copyOf(b.defaultRequestQueryData);
			defaultRequestAttributes = copyOf(b.defaultRequestAttributes);
			defaultRequestHeaders = copyOf(b.defaultRequestHeaders);
			defaultResponseHeaders = copyOf(b.defaultResponseHeaders);
			encoders = copyOf(b.encoders);
			guards = copyOf(b.guards);
			matchers = copyOf(b.matchers);
			maxInput = b.maxInput;
			parsers = copyOf(b.parsers);
			path = copyOf(b.path);
			produces = copyOf(b.produces);
			roleGuard = b.roleGuard;
			rolesDeclared = b.rolesDeclared;
			serializers = copyOf(b.serializers);
			summary = b.summary;
			swagger = b.swagger;
			value = b.value;
		}

		@Override /* Overridden from RestPost */
		public String clientVersion() {
			return clientVersion;
		}

		@Override /* Overridden from RestPost */
		public String[] consumes() {
			return consumes;
		}

		@Override /* Overridden from RestPost */
		public Class<? extends RestConverter>[] converters() {
			return converters;
		}

		@Override /* Overridden from RestPost */
		public String debug() {
			return debug;
		}

		@Override /* Overridden from RestPost */
		public String defaultAccept() {
			return defaultAccept;
		}

		@Override /* Overridden from RestPost */
		public String defaultCharset() {
			return defaultCharset;
		}

		@Override /* Overridden from RestPost */
		public String defaultContentType() {
			return defaultContentType;
		}

		@Override /* Overridden from RestPost */
		public String[] defaultRequestAttributes() {
			return defaultRequestAttributes;
		}

		@Override /* Overridden from RestPost */
		public String[] defaultRequestFormData() {
			return defaultRequestFormData;
		}

		@Override /* Overridden from RestPost */
		public String[] defaultRequestHeaders() {
			return defaultRequestHeaders;
		}

		@Override /* Overridden from RestPost */
		public String[] defaultRequestQueryData() {
			return defaultRequestQueryData;
		}

		@Override /* Overridden from RestPost */
		public String[] defaultResponseHeaders() {
			return defaultResponseHeaders;
		}

		@Override /* Overridden from RestPost */
		public Class<? extends Encoder>[] encoders() {
			return encoders;
		}

		@Override /* Overridden from RestPost */
		public Class<? extends RestGuard>[] guards() {
			return guards;
		}

		@Override /* Overridden from RestPost */
		public Class<? extends RestMatcher>[] matchers() {
			return matchers;
		}

		@Override /* Overridden from RestPost */
		public String maxInput() {
			return maxInput;
		}

		@Override /* Overridden from RestPost */
		public Class<?>[] parsers() {
			return parsers;
		}

		@Override /* Overridden from RestPost */
		public String[] path() {
			return path;
		}

		@Override /* Overridden from RestPost */
		public String[] produces() {
			return produces;
		}

		@Override /* Overridden from RestPost */
		public String roleGuard() {
			return roleGuard;
		}

		@Override /* Overridden from RestPost */
		public String rolesDeclared() {
			return rolesDeclared;
		}

		@Override /* Overridden from RestPost */
		public Class<? extends Serializer>[] serializers() {
			return serializers;
		}

		@Override /* Overridden from RestPost */
		public String summary() {
			return summary;
		}

		@Override /* Overridden from RestPost */
		public OpSwagger swagger() {
			return swagger;
		}

		@Override /* Overridden from RestPost */
		public String value() {
			return value;
		}

		@Override /* Overridden from annotation */
		public String[] description() {
			return description;
		}
	}

	/** Default value */
	public static final RestPost DEFAULT = create().build();

	/**
	 * Instantiates a new builder for this class.
	 *
	 * @return A new builder object.
	 */
	public static Builder create() {
		return new Builder();
	}
}