/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.juneau.transforms;

import static org.apache.juneau.TestUtils.*;

import java.time.*;
import java.time.chrono.*;
import java.time.temporal.*;
import java.util.*;

import org.apache.juneau.*;
import org.apache.juneau.swap.*;
import org.apache.juneau.swaps.*;
import org.junit.jupiter.api.*;

class TemporalSwap_StringSwapTest extends StringSwapTest_Base {

	//------------------------------------------------------------------------------------------------------------------
	// Setup
	//------------------------------------------------------------------------------------------------------------------

	@BeforeAll
	static void beforeClass() {
		setTimeZone("GMT-5");
	}

	@AfterAll
	static void afterClass() {
		unsetTimeZone();
	}

	private static BeanSession
		BS_DEFAULT = BeanContext.DEFAULT_SESSION,
		BS_PST = BeanContext.DEFAULT.createSession().timeZone(TimeZone.getTimeZone("PST")).build();

	private static Temporal
		DT_ZonedDateTime = ZonedDateTime.parse("2012-12-21T12:34:56Z"),
		DT_Instant = Instant.parse("2012-12-21T12:34:56Z"),
		DT_LocalDate = LocalDate.parse("2012-12-21"),
		DT_LocalDateTime = LocalDateTime.parse("2012-12-21T12:34:56"),
		DT_LocalTime = LocalTime.parse("12:34:56"),
		DT_OffsetDateTime = OffsetDateTime.parse("2012-12-21T12:34:56-05:00"),
		DT_OffsetTime = OffsetTime.parse("12:34:56-05:00"),
		DT_Year = Year.parse("2012"),
		DT_YearMonth = YearMonth.parse("2012-12");

	private static Temporal
		DT_HijrahDate = HijrahDate.from(DT_ZonedDateTime),
		DT_JapaneseDate = JapaneseDate.from(DT_ZonedDateTime),
		DT_MinguoDate = MinguoDate.from(DT_ZonedDateTime),
		DT_ThaiBuddhistDate = ThaiBuddhistDate.from(DT_ZonedDateTime);

	private static <T> StringSwap_Tester<T> tester(int index, String label, T object, StringSwap<T> swap, String expected, BeanSession bs) {
		return StringSwap_Tester.create(index, label, object, swap, expected, bs).build();
	}

	//------------------------------------------------------------------------------------------------------------------
	// Parameters
	//------------------------------------------------------------------------------------------------------------------

	private static final StringSwap_Tester<?>[] TESTERS = {
		tester(1, "BasicIsoDate, DT_ZonedDateTime, Default timezone", DT_ZonedDateTime, new TemporalSwap.BasicIsoDate(), "20121221Z", BS_DEFAULT),
		tester(2, "BasicIsoDate, DT_Instant, Default timezone", DT_Instant, new TemporalSwap.BasicIsoDate(), "20121221Z", BS_DEFAULT),
		tester(3, "BasicIsoDate, DT_LocalDate, Default timezone", DT_LocalDate, new TemporalSwap.BasicIsoDate(), "20121221", BS_DEFAULT),
		tester(4, "BasicIsoDate, DT_LocalDateTime, Default timezone", DT_LocalDateTime, new TemporalSwap.BasicIsoDate(), "20121221", BS_DEFAULT),
		tester(5, "BasicIsoDate, DT_LocalTime, Default timezone", DT_LocalTime, new TemporalSwap.BasicIsoDate(), "19700101", BS_DEFAULT),
		tester(6, "BasicIsoDate, DT_OffsetDateTime, Default timezone", DT_OffsetDateTime, new TemporalSwap.BasicIsoDate(), "20121221-0500", BS_DEFAULT),
		tester(7, "BasicIsoDate, DT_OffsetTime, Default timezone", DT_OffsetTime, new TemporalSwap.BasicIsoDate(), "19700101-0500", BS_DEFAULT),
		tester(8, "BasicIsoDate, DT_Year, Default timezone", DT_Year, new TemporalSwap.BasicIsoDate(), "20120101", BS_DEFAULT),
		tester(9, "BasicIsoDate, DT_YearMonth, Default timezone", DT_YearMonth, new TemporalSwap.BasicIsoDate(), "20121201", BS_DEFAULT),
		tester(10, "BasicIsoDate, DT_HijrahDate, Default timezone", DT_HijrahDate, new TemporalSwap.BasicIsoDate(), "20121221", BS_DEFAULT),
		tester(11, "BasicIsoDate, DT_JapaneseDate, Default timezone", DT_JapaneseDate, new TemporalSwap.BasicIsoDate(), "20121221", BS_DEFAULT),
		tester(12, "BasicIsoDate, DT_MinguoDate, Default timezone", DT_MinguoDate, new TemporalSwap.BasicIsoDate(), "20121221", BS_DEFAULT),
		tester(13, "BasicIsoDate, DT_ThaiBuddhistDate, Default timezone", DT_ThaiBuddhistDate, new TemporalSwap.BasicIsoDate(), "20121221", BS_DEFAULT),
		tester(14, "BasicIsoDate, DT_ZonedDateTime, PST timezone", DT_ZonedDateTime, new TemporalSwap.BasicIsoDate(), "20121221Z", BS_PST),
		tester(15, "BasicIsoDate, DT_Instant, PST timezone", DT_Instant, new TemporalSwap.BasicIsoDate(), "20121221Z", BS_PST),
		tester(16, "BasicIsoDate, DT_LocalDate, PST timezone", DT_LocalDate, new TemporalSwap.BasicIsoDate(), "20121221", BS_PST),
		tester(17, "BasicIsoDate, DT_LocalDateTime, PST timezone", DT_LocalDateTime, new TemporalSwap.BasicIsoDate(), "20121221", BS_PST),
		tester(18, "BasicIsoDate, DT_LocalTime, PST timezone", DT_LocalTime, new TemporalSwap.BasicIsoDate(), "19700101", BS_PST),
		tester(19, "BasicIsoDate, DT_OffsetDateTime, PST timezone", DT_OffsetDateTime, new TemporalSwap.BasicIsoDate(), "20121221-0500", BS_PST),
		tester(20, "BasicIsoDate, DT_OffsetTime, PST timezone", DT_OffsetTime, new TemporalSwap.BasicIsoDate(), "19700101-0500", BS_PST),
		tester(21, "BasicIsoDate, DT_Year, PST timezone", DT_Year, new TemporalSwap.BasicIsoDate(), "20120101", BS_PST),
		tester(22, "BasicIsoDate, DT_YearMonth, PST timezone", DT_YearMonth, new TemporalSwap.BasicIsoDate(), "20121201", BS_PST),
		tester(23, "BasicIsoDate, DT_HijrahDate, PST timezone", DT_HijrahDate, new TemporalSwap.BasicIsoDate(), "20121221", BS_PST),
		tester(24, "BasicIsoDate, DT_JapaneseDate, PST timezone", DT_JapaneseDate, new TemporalSwap.BasicIsoDate(), "20121221", BS_PST),
		tester(25, "BasicIsoDate, DT_MinguoDate, PST timezone", DT_MinguoDate, new TemporalSwap.BasicIsoDate(), "20121221", BS_PST),
		tester(26, "BasicIsoDate, DT_ThaiBuddhistDate, PST timezone", DT_ThaiBuddhistDate, new TemporalSwap.BasicIsoDate(), "20121221", BS_PST),
		tester(27, "IsoDate, DT_ZonedDateTime, Default timezone", DT_ZonedDateTime, new TemporalSwap.IsoDate(), "2012-12-21Z", BS_DEFAULT),
		tester(28, "IsoDate, DT_Instant, Default timezone", DT_Instant, new TemporalSwap.IsoDate(), "2012-12-21Z", BS_DEFAULT),
		tester(29, "IsoDate, DT_LocalDate, Default timezone", DT_LocalDate, new TemporalSwap.IsoDate(), "2012-12-21", BS_DEFAULT),
		tester(30, "IsoDate, DT_LocalDateTime, Default timezone", DT_LocalDateTime, new TemporalSwap.IsoDate(), "2012-12-21", BS_DEFAULT),
		tester(31, "IsoDate, DT_LocalTime, Default timezone", DT_LocalTime, new TemporalSwap.IsoDate(), "1970-01-01", BS_DEFAULT),
		tester(32, "IsoDate, DT_OffsetDateTime, Default timezone", DT_OffsetDateTime, new TemporalSwap.IsoDate(), "2012-12-21-05:00", BS_DEFAULT),
		tester(33, "IsoDate, DT_OffsetTime, Default timezone", DT_OffsetTime, new TemporalSwap.IsoDate(), "1970-01-01-05:00", BS_DEFAULT),
		tester(34, "IsoDate, DT_Year, Default timezone", DT_Year, new TemporalSwap.IsoDate(), "2012-01-01", BS_DEFAULT),
		tester(35, "IsoDate, DT_YearMonth, Default timezone", DT_YearMonth, new TemporalSwap.IsoDate(), "2012-12-01", BS_DEFAULT),
		tester(36, "IsoDate, DT_HijrahDate, Default timezone", DT_HijrahDate, new TemporalSwap.IsoDate(), "2012-12-21", BS_DEFAULT),
		tester(37, "IsoDate, DT_JapaneseDate, Default timezone", DT_JapaneseDate, new TemporalSwap.IsoDate(), "2012-12-21", BS_DEFAULT),
		tester(38, "IsoDate, DT_MinguoDate, Default timezone", DT_MinguoDate, new TemporalSwap.IsoDate(), "2012-12-21", BS_DEFAULT),
		tester(39, "IsoDate, DT_ThaiBuddhistDate, Default timezone", DT_ThaiBuddhistDate, new TemporalSwap.IsoDate(), "2012-12-21", BS_DEFAULT),
		tester(40, "IsoDate, DT_ZonedDateTime, PST timezone", DT_ZonedDateTime, new TemporalSwap.IsoDate(), "2012-12-21Z", BS_PST),
		tester(41, "IsoDate, DT_Instant, PST timezone", DT_Instant, new TemporalSwap.IsoDate(), "2012-12-21Z", BS_PST),
		tester(42, "IsoDate, DT_LocalDate, PST timezone", DT_LocalDate, new TemporalSwap.IsoDate(), "2012-12-21", BS_PST),
		tester(43, "IsoDate, DT_LocalDateTime, PST timezone", DT_LocalDateTime, new TemporalSwap.IsoDate(), "2012-12-21", BS_PST),
		tester(44, "IsoDate, DT_LocalTime, PST timezone", DT_LocalTime, new TemporalSwap.IsoDate(), "1970-01-01", BS_PST),
		tester(45, "IsoDate, DT_OffsetDateTime, PST timezone", DT_OffsetDateTime, new TemporalSwap.IsoDate(), "2012-12-21-05:00", BS_PST),
		tester(46, "IsoDate, DT_OffsetTime, PST timezone", DT_OffsetTime, new TemporalSwap.IsoDate(), "1970-01-01-05:00", BS_PST),
		tester(47, "IsoDate, DT_Year, PST timezone", DT_Year, new TemporalSwap.IsoDate(), "2012-01-01", BS_PST),
		tester(48, "IsoDate, DT_YearMonth, PST timezone", DT_YearMonth, new TemporalSwap.IsoDate(), "2012-12-01", BS_PST),
		tester(49, "IsoDate, DT_HijrahDate, PST timezone", DT_HijrahDate, new TemporalSwap.IsoDate(), "2012-12-21", BS_PST),
		tester(50, "IsoDate, DT_JapaneseDate, PST timezone", DT_JapaneseDate, new TemporalSwap.IsoDate(), "2012-12-21", BS_PST),
		tester(51, "IsoDate, DT_MinguoDate, PST timezone", DT_MinguoDate, new TemporalSwap.IsoDate(), "2012-12-21", BS_PST),
		tester(52, "IsoDate, DT_ThaiBuddhistDate, PST timezone", DT_ThaiBuddhistDate, new TemporalSwap.IsoDate(), "2012-12-21", BS_PST),
		tester(53, "IsoDateTime, DT_ZonedDateTime, Default timezone", DT_ZonedDateTime, new TemporalSwap.IsoDateTime(), "2012-12-21T12:34:56Z", BS_DEFAULT),
		tester(54, "IsoDateTime, DT_Instant, Default timezone", DT_Instant, new TemporalSwap.IsoDateTime(), "2012-12-21T12:34:56Z", BS_DEFAULT),
		tester(55, "IsoDateTime, DT_LocalDate, Default timezone", DT_LocalDate, new TemporalSwap.IsoDateTime(), "2012-12-21T00:00:00", BS_DEFAULT),
		tester(56, "IsoDateTime, DT_LocalDateTime, Default timezone", DT_LocalDateTime, new TemporalSwap.IsoDateTime(), "2012-12-21T12:34:56", BS_DEFAULT),
		tester(57, "IsoDateTime, DT_LocalTime, Default timezone", DT_LocalTime, new TemporalSwap.IsoDateTime(), "1970-01-01T12:34:56", BS_DEFAULT),
		tester(58, "IsoDateTime, DT_OffsetDateTime, Default timezone", DT_OffsetDateTime, new TemporalSwap.IsoDateTime(), "2012-12-21T12:34:56-05:00", BS_DEFAULT),
		tester(59, "IsoDateTime, DT_OffsetTime, Default timezone", DT_OffsetTime, new TemporalSwap.IsoDateTime(), "1970-01-01T12:34:56-05:00", BS_DEFAULT),
		tester(60, "IsoDateTime, DT_Year, Default timezone", DT_Year, new TemporalSwap.IsoDateTime(), "2012-01-01T00:00:00", BS_DEFAULT),
		tester(61, "IsoDateTime, DT_YearMonth, Default timezone", DT_YearMonth, new TemporalSwap.IsoDateTime(), "2012-12-01T00:00:00", BS_DEFAULT),
		tester(62, "IsoDateTime, DT_HijrahDate, Default timezone", DT_HijrahDate, new TemporalSwap.IsoDateTime(), "2012-12-21T00:00:00", BS_DEFAULT),
		tester(63, "IsoDateTime, DT_JapaneseDate, Default timezone", DT_JapaneseDate, new TemporalSwap.IsoDateTime(), "2012-12-21T00:00:00", BS_DEFAULT),
		tester(64, "IsoDateTime, DT_MinguoDate, Default timezone", DT_MinguoDate, new TemporalSwap.IsoDateTime(), "2012-12-21T00:00:00", BS_DEFAULT),
		tester(65, "IsoDateTime, DT_ThaiBuddhistDate, Default timezone", DT_ThaiBuddhistDate, new TemporalSwap.IsoDateTime(), "2012-12-21T00:00:00", BS_DEFAULT),
		tester(66, "IsoDateTime, DT_ZonedDateTime, PST timezone", DT_ZonedDateTime, new TemporalSwap.IsoDateTime(), "2012-12-21T12:34:56Z", BS_PST),
		tester(67, "IsoDateTime, DT_Instant, PST timezone", DT_Instant, new TemporalSwap.IsoDateTime(), "2012-12-21T12:34:56Z", BS_PST),
		tester(68, "IsoDateTime, DT_LocalDate, PST timezone", DT_LocalDate, new TemporalSwap.IsoDateTime(), "2012-12-21T00:00:00", BS_PST),
		tester(69, "IsoDateTime, DT_LocalDateTime, PST timezone", DT_LocalDateTime, new TemporalSwap.IsoDateTime(), "2012-12-21T12:34:56", BS_PST),
		tester(70, "IsoDateTime, DT_LocalTime, PST timezone", DT_LocalTime, new TemporalSwap.IsoDateTime(), "1970-01-01T12:34:56", BS_PST),
		tester(71, "IsoDateTime, DT_OffsetDateTime, PST timezone", DT_OffsetDateTime, new TemporalSwap.IsoDateTime(), "2012-12-21T12:34:56-05:00", BS_PST),
		tester(72, "IsoDateTime, DT_OffsetTime, PST timezone", DT_OffsetTime, new TemporalSwap.IsoDateTime(), "1970-01-01T12:34:56-05:00", BS_PST),
		tester(73, "IsoDateTime, DT_Year, PST timezone", DT_Year, new TemporalSwap.IsoDateTime(), "2012-01-01T00:00:00", BS_PST),
		tester(74, "IsoDateTime, DT_YearMonth, PST timezone", DT_YearMonth, new TemporalSwap.IsoDateTime(), "2012-12-01T00:00:00", BS_PST),
		tester(75, "IsoDateTime, DT_HijrahDate, PST timezone", DT_HijrahDate, new TemporalSwap.IsoDateTime(), "2012-12-21T00:00:00", BS_PST),
		tester(76, "IsoDateTime, DT_JapaneseDate, PST timezone", DT_JapaneseDate, new TemporalSwap.IsoDateTime(), "2012-12-21T00:00:00", BS_PST),
		tester(77, "IsoDateTime, DT_MinguoDate, PST timezone", DT_MinguoDate, new TemporalSwap.IsoDateTime(), "2012-12-21T00:00:00", BS_PST),
		tester(78, "IsoDateTime, DT_ThaiBuddhistDate, PST timezone", DT_ThaiBuddhistDate, new TemporalSwap.IsoDateTime(), "2012-12-21T00:00:00", BS_PST),
		tester(79, "IsoInstant, DT_ZonedDateTime, Default timezone", DT_ZonedDateTime, new TemporalSwap.IsoInstant(), "2012-12-21T12:34:56Z", BS_DEFAULT),
		tester(80, "IsoInstant, DT_Instant, Default timezone", DT_Instant, new TemporalSwap.IsoInstant(), "2012-12-21T12:34:56Z", BS_DEFAULT),
		tester(81, "IsoInstant, DT_LocalDate, Default timezone", DT_LocalDate, new TemporalSwap.IsoInstant(), "2012-12-21T05:00:00Z", BS_DEFAULT),
		tester(82, "IsoInstant, DT_LocalDateTime, Default timezone", DT_LocalDateTime, new TemporalSwap.IsoInstant(), "2012-12-21T17:34:56Z", BS_DEFAULT),
		tester(83, "IsoInstant, DT_LocalTime, Default timezone", DT_LocalTime, new TemporalSwap.IsoInstant(), "1970-01-01T17:34:56Z", BS_DEFAULT),
		tester(84, "IsoInstant, DT_OffsetDateTime, Default timezone", DT_OffsetDateTime, new TemporalSwap.IsoInstant(), "2012-12-21T17:34:56Z", BS_DEFAULT),
		tester(85, "IsoInstant, DT_OffsetTime, Default timezone", DT_OffsetTime, new TemporalSwap.IsoInstant(), "1970-01-01T17:34:56Z", BS_DEFAULT),
		tester(86, "IsoInstant, DT_Year, Default timezone", DT_Year, new TemporalSwap.IsoInstant(), "2012-01-01T05:00:00Z", BS_DEFAULT),
		tester(87, "IsoInstant, DT_YearMonth, Default timezone", DT_YearMonth, new TemporalSwap.IsoInstant(), "2012-12-01T05:00:00Z", BS_DEFAULT),
		tester(88, "IsoInstant, DT_HijrahDate, Default timezone", DT_HijrahDate, new TemporalSwap.IsoInstant(), "2012-12-21T05:00:00Z", BS_DEFAULT),
		tester(89, "IsoInstant, DT_JapaneseDate, Default timezone", DT_JapaneseDate, new TemporalSwap.IsoInstant(), "2012-12-21T05:00:00Z", BS_DEFAULT),
		tester(90, "IsoInstant, DT_MinguoDate, Default timezone", DT_MinguoDate, new TemporalSwap.IsoInstant(), "2012-12-21T05:00:00Z", BS_DEFAULT),
		tester(91, "IsoInstant, DT_ThaiBuddhistDate, Default timezone", DT_ThaiBuddhistDate, new TemporalSwap.IsoInstant(), "2012-12-21T05:00:00Z", BS_DEFAULT),
		tester(92, "IsoInstant, DT_ZonedDateTime, PST timezone", DT_ZonedDateTime, new TemporalSwap.IsoInstant(), "2012-12-21T12:34:56Z", BS_PST),
		tester(93, "IsoInstant, DT_Instant, PST timezone", DT_Instant, new TemporalSwap.IsoInstant(), "2012-12-21T12:34:56Z", BS_PST),
		tester(94, "IsoInstant, DT_LocalDate, PST timezone", DT_LocalDate, new TemporalSwap.IsoInstant(), "2012-12-21T08:00:00Z", BS_PST),
		tester(95, "IsoInstant, DT_LocalDateTime, PST timezone", DT_LocalDateTime, new TemporalSwap.IsoInstant(), "2012-12-21T20:34:56Z", BS_PST),
		tester(96, "IsoInstant, DT_LocalTime, PST timezone", DT_LocalTime, new TemporalSwap.IsoInstant(), "1970-01-01T20:34:56Z", BS_PST),
		tester(97, "IsoInstant, DT_OffsetDateTime, PST timezone", DT_OffsetDateTime, new TemporalSwap.IsoInstant(), "2012-12-21T17:34:56Z", BS_PST),
		tester(98, "IsoInstant, DT_OffsetTime, PST timezone", DT_OffsetTime, new TemporalSwap.IsoInstant(), "1970-01-01T17:34:56Z", BS_PST),
		tester(99, "IsoInstant, DT_Year, PST timezone", DT_Year, new TemporalSwap.IsoInstant(), "2012-01-01T08:00:00Z", BS_PST),
		tester(100, "IsoInstant, DT_YearMonth, PST timezone", DT_YearMonth, new TemporalSwap.IsoInstant(), "2012-12-01T08:00:00Z", BS_PST),
		tester(101, "IsoInstant, DT_HijrahDate, PST timezone", DT_HijrahDate, new TemporalSwap.IsoInstant(), "2012-12-21T08:00:00Z", BS_PST),
		tester(102, "IsoInstant, DT_JapaneseDate, PST timezone", DT_JapaneseDate, new TemporalSwap.IsoInstant(), "2012-12-21T08:00:00Z", BS_PST),
		tester(103, "IsoInstant, DT_MinguoDate, PST timezone", DT_MinguoDate, new TemporalSwap.IsoInstant(), "2012-12-21T08:00:00Z", BS_PST),
		tester(104, "IsoInstant, DT_ThaiBuddhistDate, PST timezone", DT_ThaiBuddhistDate, new TemporalSwap.IsoInstant(), "2012-12-21T08:00:00Z", BS_PST),
		tester(105, "IsoLocalDate, DT_ZonedDateTime, Default timezone", DT_ZonedDateTime, new TemporalSwap.IsoLocalDate(), "2012-12-21", BS_DEFAULT),
		tester(106, "IsoLocalDate, DT_Instant, Default timezone", DT_Instant, new TemporalSwap.IsoLocalDate(), "2012-12-21", BS_DEFAULT),
		tester(107, "IsoLocalDate, DT_LocalDate, Default timezone", DT_LocalDate, new TemporalSwap.IsoLocalDate(), "2012-12-21", BS_DEFAULT),
		tester(108, "IsoLocalDate, DT_LocalDateTime, Default timezone", DT_LocalDateTime, new TemporalSwap.IsoLocalDate(), "2012-12-21", BS_DEFAULT),
		tester(109, "IsoLocalDate, DT_LocalTime, Default timezone", DT_LocalTime, new TemporalSwap.IsoLocalDate(), "1970-01-01", BS_DEFAULT),
		tester(110, "IsoLocalDate, DT_OffsetDateTime, Default timezone", DT_OffsetDateTime, new TemporalSwap.IsoLocalDate(), "2012-12-21", BS_DEFAULT),
		tester(111, "IsoLocalDate, DT_OffsetTime, Default timezone", DT_OffsetTime, new TemporalSwap.IsoLocalDate(), "1970-01-01", BS_DEFAULT),
		tester(112, "IsoLocalDate, DT_Year, Default timezone", DT_Year, new TemporalSwap.IsoLocalDate(), "2012-01-01", BS_DEFAULT),
		tester(113, "IsoLocalDate, DT_YearMonth, Default timezone", DT_YearMonth, new TemporalSwap.IsoLocalDate(), "2012-12-01", BS_DEFAULT),
		tester(114, "IsoLocalDate, DT_HijrahDate, Default timezone", DT_HijrahDate, new TemporalSwap.IsoLocalDate(), "2012-12-21", BS_DEFAULT),
		tester(115, "IsoLocalDate, DT_JapaneseDate, Default timezone", DT_JapaneseDate, new TemporalSwap.IsoLocalDate(), "2012-12-21", BS_DEFAULT),
		tester(116, "IsoLocalDate, DT_MinguoDate, Default timezone", DT_MinguoDate, new TemporalSwap.IsoLocalDate(), "2012-12-21", BS_DEFAULT),
		tester(117, "IsoLocalDate, DT_ThaiBuddhistDate, Default timezone", DT_ThaiBuddhistDate, new TemporalSwap.IsoLocalDate(), "2012-12-21", BS_DEFAULT),
		tester(118, "IsoLocalDate, DT_ZonedDateTime, PST timezone", DT_ZonedDateTime, new TemporalSwap.IsoLocalDate(), "2012-12-21", BS_PST),
		tester(119, "IsoLocalDate, DT_Instant, PST timezone", DT_Instant, new TemporalSwap.IsoLocalDate(), "2012-12-21", BS_PST),
		tester(120, "IsoLocalDate, DT_LocalDate, PST timezone", DT_LocalDate, new TemporalSwap.IsoLocalDate(), "2012-12-21", BS_PST),
		tester(121, "IsoLocalDate, DT_LocalDateTime, PST timezone", DT_LocalDateTime, new TemporalSwap.IsoLocalDate(), "2012-12-21", BS_PST),
		tester(122, "IsoLocalDate, DT_LocalTime, PST timezone", DT_LocalTime, new TemporalSwap.IsoLocalDate(), "1970-01-01", BS_PST),
		tester(123, "IsoLocalDate, DT_OffsetDateTime, PST timezone", DT_OffsetDateTime, new TemporalSwap.IsoLocalDate(), "2012-12-21", BS_PST),
		tester(124, "IsoLocalDate, DT_OffsetTime, PST timezone", DT_OffsetTime, new TemporalSwap.IsoLocalDate(), "1970-01-01", BS_PST),
		tester(125, "IsoLocalDate, DT_Year, PST timezone", DT_Year, new TemporalSwap.IsoLocalDate(), "2012-01-01", BS_PST),
		tester(126, "IsoLocalDate, DT_YearMonth, PST timezone", DT_YearMonth, new TemporalSwap.IsoLocalDate(), "2012-12-01", BS_PST),
		tester(127, "IsoLocalDate, DT_HijrahDate, PST timezone", DT_HijrahDate, new TemporalSwap.IsoLocalDate(), "2012-12-21", BS_PST),
		tester(128, "IsoLocalDate, DT_JapaneseDate, PST timezone", DT_JapaneseDate, new TemporalSwap.IsoLocalDate(), "2012-12-21", BS_PST),
		tester(129, "IsoLocalDate, DT_MinguoDate, PST timezone", DT_MinguoDate, new TemporalSwap.IsoLocalDate(), "2012-12-21", BS_PST),
		tester(130, "IsoLocalDate, DT_ThaiBuddhistDate, PST timezone", DT_ThaiBuddhistDate, new TemporalSwap.IsoLocalDate(), "2012-12-21", BS_PST),
		tester(131, "IsoLocalDateTime, DT_ZonedDateTime, Default timezone", DT_ZonedDateTime, new TemporalSwap.IsoLocalDateTime(), "2012-12-21T12:34:56", BS_DEFAULT),
		tester(132, "IsoLocalDateTime, DT_Instant, Default timezone", DT_Instant, new TemporalSwap.IsoLocalDateTime(), "2012-12-21T12:34:56", BS_DEFAULT),
		tester(133, "IsoLocalDateTime, DT_LocalDate, Default timezone", DT_LocalDate, new TemporalSwap.IsoLocalDateTime(), "2012-12-21T00:00:00", BS_DEFAULT),
		tester(134, "IsoLocalDateTime, DT_LocalDateTime, Default timezone", DT_LocalDateTime, new TemporalSwap.IsoLocalDateTime(), "2012-12-21T12:34:56", BS_DEFAULT),
		tester(135, "IsoLocalDateTime, DT_LocalTime, Default timezone", DT_LocalTime, new TemporalSwap.IsoLocalDateTime(), "1970-01-01T12:34:56", BS_DEFAULT),
		tester(136, "IsoLocalDateTime, DT_OffsetDateTime, Default timezone", DT_OffsetDateTime, new TemporalSwap.IsoLocalDateTime(), "2012-12-21T12:34:56", BS_DEFAULT),
		tester(137, "IsoLocalDateTime, DT_OffsetTime, Default timezone", DT_OffsetTime, new TemporalSwap.IsoLocalDateTime(), "1970-01-01T12:34:56", BS_DEFAULT),
		tester(138, "IsoLocalDateTime, DT_Year, Default timezone", DT_Year, new TemporalSwap.IsoLocalDateTime(), "2012-01-01T00:00:00", BS_DEFAULT),
		tester(139, "IsoLocalDateTime, DT_YearMonth, Default timezone", DT_YearMonth, new TemporalSwap.IsoLocalDateTime(), "2012-12-01T00:00:00", BS_DEFAULT),
		tester(140, "IsoLocalDateTime, DT_HijrahDate, Default timezone", DT_HijrahDate, new TemporalSwap.IsoLocalDateTime(), "2012-12-21T00:00:00", BS_DEFAULT),
		tester(141, "IsoLocalDateTime, DT_JapaneseDate, Default timezone", DT_JapaneseDate, new TemporalSwap.IsoLocalDateTime(), "2012-12-21T00:00:00", BS_DEFAULT),
		tester(142, "IsoLocalDateTime, DT_MinguoDate, Default timezone", DT_MinguoDate, new TemporalSwap.IsoLocalDateTime(), "2012-12-21T00:00:00", BS_DEFAULT),
		tester(143, "IsoLocalDateTime, DT_ThaiBuddhistDate, Default timezone", DT_ThaiBuddhistDate, new TemporalSwap.IsoLocalDateTime(), "2012-12-21T00:00:00", BS_DEFAULT),
		tester(144, "IsoLocalDateTime, DT_ZonedDateTime, PST timezone", DT_ZonedDateTime, new TemporalSwap.IsoLocalDateTime(), "2012-12-21T12:34:56", BS_PST),
		tester(145, "IsoLocalDateTime, DT_Instant, PST timezone", DT_Instant, new TemporalSwap.IsoLocalDateTime(), "2012-12-21T12:34:56", BS_PST),
		tester(146, "IsoLocalDateTime, DT_LocalDate, PST timezone", DT_LocalDate, new TemporalSwap.IsoLocalDateTime(), "2012-12-21T00:00:00", BS_PST),
		tester(147, "IsoLocalDateTime, DT_LocalDateTime, PST timezone", DT_LocalDateTime, new TemporalSwap.IsoLocalDateTime(), "2012-12-21T12:34:56", BS_PST),
		tester(148, "IsoLocalDateTime, DT_LocalTime, PST timezone", DT_LocalTime, new TemporalSwap.IsoLocalDateTime(), "1970-01-01T12:34:56", BS_PST),
		tester(149, "IsoLocalDateTime, DT_OffsetDateTime, PST timezone", DT_OffsetDateTime, new TemporalSwap.IsoLocalDateTime(), "2012-12-21T12:34:56", BS_PST),
		tester(150, "IsoLocalDateTime, DT_OffsetTime, PST timezone", DT_OffsetTime, new TemporalSwap.IsoLocalDateTime(), "1970-01-01T12:34:56", BS_PST),
		tester(151, "IsoLocalDateTime, DT_Year, PST timezone", DT_Year, new TemporalSwap.IsoLocalDateTime(), "2012-01-01T00:00:00", BS_PST),
		tester(152, "IsoLocalDateTime, DT_YearMonth, PST timezone", DT_YearMonth, new TemporalSwap.IsoLocalDateTime(), "2012-12-01T00:00:00", BS_PST),
		tester(153, "IsoLocalDateTime, DT_HijrahDate, PST timezone", DT_HijrahDate, new TemporalSwap.IsoLocalDateTime(), "2012-12-21T00:00:00", BS_PST),
		tester(154, "IsoLocalDateTime, DT_JapaneseDate, PST timezone", DT_JapaneseDate, new TemporalSwap.IsoLocalDateTime(), "2012-12-21T00:00:00", BS_PST),
		tester(155, "IsoLocalDateTime, DT_MinguoDate, PST timezone", DT_MinguoDate, new TemporalSwap.IsoLocalDateTime(), "2012-12-21T00:00:00", BS_PST),
		tester(156, "IsoLocalDateTime, DT_ThaiBuddhistDate, PST timezone", DT_ThaiBuddhistDate, new TemporalSwap.IsoLocalDateTime(), "2012-12-21T00:00:00", BS_PST),
		tester(157, "IsoLocalTime, DT_ZonedDateTime, Default timezone", DT_ZonedDateTime, new TemporalSwap.IsoLocalTime(), "12:34:56", BS_DEFAULT),
		tester(158, "IsoLocalTime, DT_Instant, Default timezone", DT_Instant, new TemporalSwap.IsoLocalTime(), "12:34:56", BS_DEFAULT),
		tester(159, "IsoLocalTime, DT_LocalDate, Default timezone", DT_LocalDate, new TemporalSwap.IsoLocalTime(), "00:00:00", BS_DEFAULT),
		tester(160, "IsoLocalTime, DT_LocalDateTime, Default timezone", DT_LocalDateTime, new TemporalSwap.IsoLocalTime(), "12:34:56", BS_DEFAULT),
		tester(161, "IsoLocalTime, DT_LocalTime, Default timezone", DT_LocalTime, new TemporalSwap.IsoLocalTime(), "12:34:56", BS_DEFAULT),
		tester(162, "IsoLocalTime, DT_OffsetDateTime, Default timezone", DT_OffsetDateTime, new TemporalSwap.IsoLocalTime(), "12:34:56", BS_DEFAULT),
		tester(163, "IsoLocalTime, DT_OffsetTime, Default timezone", DT_OffsetTime, new TemporalSwap.IsoLocalTime(), "12:34:56", BS_DEFAULT),
		tester(164, "IsoLocalTime, DT_Year, Default timezone", DT_Year, new TemporalSwap.IsoLocalTime(), "00:00:00", BS_DEFAULT),
		tester(165, "IsoLocalTime, DT_YearMonth, Default timezone", DT_YearMonth, new TemporalSwap.IsoLocalTime(), "00:00:00", BS_DEFAULT),
		tester(166, "IsoLocalTime, DT_HijrahDate, Default timezone", DT_HijrahDate, new TemporalSwap.IsoLocalTime(), "00:00:00", BS_DEFAULT),
		tester(167, "IsoLocalTime, DT_JapaneseDate, Default timezone", DT_JapaneseDate, new TemporalSwap.IsoLocalTime(), "00:00:00", BS_DEFAULT),
		tester(168, "IsoLocalTime, DT_MinguoDate, Default timezone", DT_MinguoDate, new TemporalSwap.IsoLocalTime(), "00:00:00", BS_DEFAULT),
		tester(169, "IsoLocalTime, DT_ThaiBuddhistDate, Default timezone", DT_ThaiBuddhistDate, new TemporalSwap.IsoLocalTime(), "00:00:00", BS_DEFAULT),
		tester(170, "IsoLocalTime, DT_ZonedDateTime, PST timezone", DT_ZonedDateTime, new TemporalSwap.IsoLocalTime(), "12:34:56", BS_PST),
		tester(171, "IsoLocalTime, DT_Instant, PST timezone", DT_Instant, new TemporalSwap.IsoLocalTime(), "12:34:56", BS_PST),
		tester(172, "IsoLocalTime, DT_LocalDate, PST timezone", DT_LocalDate, new TemporalSwap.IsoLocalTime(), "00:00:00", BS_PST),
		tester(173, "IsoLocalTime, DT_LocalDateTime, PST timezone", DT_LocalDateTime, new TemporalSwap.IsoLocalTime(), "12:34:56", BS_PST),
		tester(174, "IsoLocalTime, DT_LocalTime, PST timezone", DT_LocalTime, new TemporalSwap.IsoLocalTime(), "12:34:56", BS_PST),
		tester(175, "IsoLocalTime, DT_OffsetDateTime, PST timezone", DT_OffsetDateTime, new TemporalSwap.IsoLocalTime(), "12:34:56", BS_PST),
		tester(176, "IsoLocalTime, DT_OffsetTime, PST timezone", DT_OffsetTime, new TemporalSwap.IsoLocalTime(), "12:34:56", BS_PST),
		tester(177, "IsoLocalTime, DT_Year, PST timezone", DT_Year, new TemporalSwap.IsoLocalTime(), "00:00:00", BS_PST),
		tester(178, "IsoLocalTime, DT_YearMonth, PST timezone", DT_YearMonth, new TemporalSwap.IsoLocalTime(), "00:00:00", BS_PST),
		tester(179, "IsoLocalTime, DT_HijrahDate, PST timezone", DT_HijrahDate, new TemporalSwap.IsoLocalTime(), "00:00:00", BS_PST),
		tester(180, "IsoLocalTime, DT_JapaneseDate, PST timezone", DT_JapaneseDate, new TemporalSwap.IsoLocalTime(), "00:00:00", BS_PST),
		tester(181, "IsoLocalTime, DT_MinguoDate, PST timezone", DT_MinguoDate, new TemporalSwap.IsoLocalTime(), "00:00:00", BS_PST),
		tester(182, "IsoLocalTime, DT_ThaiBuddhistDate, PST timezone", DT_ThaiBuddhistDate, new TemporalSwap.IsoLocalTime(), "00:00:00", BS_PST),
		tester(183, "IsoOffsetDate, DT_ZonedDateTime, Default timezone", DT_ZonedDateTime, new TemporalSwap.IsoOffsetDate(), "2012-12-21Z", BS_DEFAULT),
		tester(184, "IsoOffsetDate, DT_Instant, Default timezone", DT_Instant, new TemporalSwap.IsoOffsetDate(), "2012-12-21Z", BS_DEFAULT),
		tester(185, "IsoOffsetDate, DT_LocalDate, Default timezone", DT_LocalDate, new TemporalSwap.IsoOffsetDate(), "2012-12-21-05:00", BS_DEFAULT),
		tester(186, "IsoOffsetDate, DT_LocalDateTime, Default timezone", DT_LocalDateTime, new TemporalSwap.IsoOffsetDate(), "2012-12-21-05:00", BS_DEFAULT),
		tester(187, "IsoOffsetDate, DT_LocalTime, Default timezone", DT_LocalTime, new TemporalSwap.IsoOffsetDate(), "1970-01-01-05:00", BS_DEFAULT),
		tester(188, "IsoOffsetDate, DT_OffsetDateTime, Default timezone", DT_OffsetDateTime, new TemporalSwap.IsoOffsetDate(), "2012-12-21-05:00", BS_DEFAULT),
		tester(189, "IsoOffsetDate, DT_OffsetTime, Default timezone", DT_OffsetTime, new TemporalSwap.IsoOffsetDate(), "1970-01-01-05:00", BS_DEFAULT),
		tester(190, "IsoOffsetDate, DT_Year, Default timezone", DT_Year, new TemporalSwap.IsoOffsetDate(), "2012-01-01-05:00", BS_DEFAULT),
		tester(191, "IsoOffsetDate, DT_YearMonth, Default timezone", DT_YearMonth, new TemporalSwap.IsoOffsetDate(), "2012-12-01-05:00", BS_DEFAULT),
		tester(192, "IsoOffsetDate, DT_HijrahDate, Default timezone", DT_HijrahDate, new TemporalSwap.IsoOffsetDate(), "2012-12-21-05:00", BS_DEFAULT),
		tester(193, "IsoOffsetDate, DT_JapaneseDate, Default timezone", DT_JapaneseDate, new TemporalSwap.IsoOffsetDate(), "2012-12-21-05:00", BS_DEFAULT),
		tester(194, "IsoOffsetDate, DT_MinguoDate, Default timezone", DT_MinguoDate, new TemporalSwap.IsoOffsetDate(), "2012-12-21-05:00", BS_DEFAULT),
		tester(195, "IsoOffsetDate, DT_ThaiBuddhistDate, Default timezone", DT_ThaiBuddhistDate, new TemporalSwap.IsoOffsetDate(), "2012-12-21-05:00", BS_DEFAULT),
		tester(196, "IsoOffsetDate, DT_ZonedDateTime, PST timezone", DT_ZonedDateTime, new TemporalSwap.IsoOffsetDate(), "2012-12-21Z", BS_PST),
		tester(197, "IsoOffsetDate, DT_Instant, PST timezone", DT_Instant, new TemporalSwap.IsoOffsetDate(), "2012-12-21Z", BS_PST),
		tester(198, "IsoOffsetDate, DT_LocalDate, PST timezone", DT_LocalDate, new TemporalSwap.IsoOffsetDate(), "2012-12-21-08:00", BS_PST),
		tester(199, "IsoOffsetDate, DT_LocalDateTime, PST timezone", DT_LocalDateTime, new TemporalSwap.IsoOffsetDate(), "2012-12-21-08:00", BS_PST),
		tester(200, "IsoOffsetDate, DT_LocalTime, PST timezone", DT_LocalTime, new TemporalSwap.IsoOffsetDate(), "1970-01-01-08:00", BS_PST),
		tester(201, "IsoOffsetDate, DT_OffsetDateTime, PST timezone", DT_OffsetDateTime, new TemporalSwap.IsoOffsetDate(), "2012-12-21-05:00", BS_PST),
		tester(202, "IsoOffsetDate, DT_OffsetTime, PST timezone", DT_OffsetTime, new TemporalSwap.IsoOffsetDate(), "1970-01-01-05:00", BS_PST),
		tester(203, "IsoOffsetDate, DT_Year, PST timezone", DT_Year, new TemporalSwap.IsoOffsetDate(), "2012-01-01-08:00", BS_PST),
		tester(204, "IsoOffsetDate, DT_YearMonth, PST timezone", DT_YearMonth, new TemporalSwap.IsoOffsetDate(), "2012-12-01-08:00", BS_PST),
		tester(205, "IsoOffsetDate, DT_HijrahDate, PST timezone", DT_HijrahDate, new TemporalSwap.IsoOffsetDate(), "2012-12-21-08:00", BS_PST),
		tester(206, "IsoOffsetDate, DT_JapaneseDate, PST timezone", DT_JapaneseDate, new TemporalSwap.IsoOffsetDate(), "2012-12-21-08:00", BS_PST),
		tester(207, "IsoOffsetDate, DT_MinguoDate, PST timezone", DT_MinguoDate, new TemporalSwap.IsoOffsetDate(), "2012-12-21-08:00", BS_PST),
		tester(208, "IsoOffsetDate, DT_ThaiBuddhistDate, PST timezone", DT_ThaiBuddhistDate, new TemporalSwap.IsoOffsetDate(), "2012-12-21-08:00", BS_PST),
		tester(209, "IsoOffsetDateTime, DT_ZonedDateTime, Default timezone", DT_ZonedDateTime, new TemporalSwap.IsoOffsetDateTime(), "2012-12-21T12:34:56Z", BS_DEFAULT),
		tester(210, "IsoOffsetDateTime, DT_Instant, Default timezone", DT_Instant, new TemporalSwap.IsoOffsetDateTime(), "2012-12-21T12:34:56Z", BS_DEFAULT),
		tester(211, "IsoOffsetDateTime, DT_LocalDate, Default timezone", DT_LocalDate, new TemporalSwap.IsoOffsetDateTime(), "2012-12-21T00:00:00-05:00", BS_DEFAULT),
		tester(212, "IsoOffsetDateTime, DT_LocalDateTime, Default timezone", DT_LocalDateTime, new TemporalSwap.IsoOffsetDateTime(), "2012-12-21T12:34:56-05:00", BS_DEFAULT),
		tester(213, "IsoOffsetDateTime, DT_LocalTime, Default timezone", DT_LocalTime, new TemporalSwap.IsoOffsetDateTime(), "1970-01-01T12:34:56-05:00", BS_DEFAULT),
		tester(214, "IsoOffsetDateTime, DT_OffsetDateTime, Default timezone", DT_OffsetDateTime, new TemporalSwap.IsoOffsetDateTime(), "2012-12-21T12:34:56-05:00", BS_DEFAULT),
		tester(215, "IsoOffsetDateTime, DT_OffsetTime, Default timezone", DT_OffsetTime, new TemporalSwap.IsoOffsetDateTime(), "1970-01-01T12:34:56-05:00", BS_DEFAULT),
		tester(216, "IsoOffsetDateTime, DT_Year, Default timezone", DT_Year, new TemporalSwap.IsoOffsetDateTime(), "2012-01-01T00:00:00-05:00", BS_DEFAULT),
		tester(217, "IsoOffsetDateTime, DT_YearMonth, Default timezone", DT_YearMonth, new TemporalSwap.IsoOffsetDateTime(), "2012-12-01T00:00:00-05:00", BS_DEFAULT),
		tester(218, "IsoOffsetDateTime, DT_HijrahDate, Default timezone", DT_HijrahDate, new TemporalSwap.IsoOffsetDateTime(), "2012-12-21T00:00:00-05:00", BS_DEFAULT),
		tester(219, "IsoOffsetDateTime, DT_JapaneseDate, Default timezone", DT_JapaneseDate, new TemporalSwap.IsoOffsetDateTime(), "2012-12-21T00:00:00-05:00", BS_DEFAULT),
		tester(220, "IsoOffsetDateTime, DT_MinguoDate, Default timezone", DT_MinguoDate, new TemporalSwap.IsoOffsetDateTime(), "2012-12-21T00:00:00-05:00", BS_DEFAULT),
		tester(221, "IsoOffsetDateTime, DT_ThaiBuddhistDate, Default timezone", DT_ThaiBuddhistDate, new TemporalSwap.IsoOffsetDateTime(), "2012-12-21T00:00:00-05:00", BS_DEFAULT),
		tester(222, "IsoOffsetDateTime, DT_ZonedDateTime, PST timezone", DT_ZonedDateTime, new TemporalSwap.IsoOffsetDateTime(), "2012-12-21T12:34:56Z", BS_PST),
		tester(223, "IsoOffsetDateTime, DT_Instant, PST timezone", DT_Instant, new TemporalSwap.IsoOffsetDateTime(), "2012-12-21T12:34:56Z", BS_PST),
		tester(224, "IsoOffsetDateTime, DT_LocalDate, PST timezone", DT_LocalDate, new TemporalSwap.IsoOffsetDateTime(), "2012-12-21T00:00:00-08:00", BS_PST),
		tester(225, "IsoOffsetDateTime, DT_LocalDateTime, PST timezone", DT_LocalDateTime, new TemporalSwap.IsoOffsetDateTime(), "2012-12-21T12:34:56-08:00", BS_PST),
		tester(226, "IsoOffsetDateTime, DT_LocalTime, PST timezone", DT_LocalTime, new TemporalSwap.IsoOffsetDateTime(), "1970-01-01T12:34:56-08:00", BS_PST),
		tester(227, "IsoOffsetDateTime, DT_OffsetDateTime, PST timezone", DT_OffsetDateTime, new TemporalSwap.IsoOffsetDateTime(), "2012-12-21T12:34:56-05:00", BS_PST),
		tester(228, "IsoOffsetDateTime, DT_OffsetTime, PST timezone", DT_OffsetTime, new TemporalSwap.IsoOffsetDateTime(), "1970-01-01T12:34:56-05:00", BS_PST),
		tester(229, "IsoOffsetDateTime, DT_Year, PST timezone", DT_Year, new TemporalSwap.IsoOffsetDateTime(), "2012-01-01T00:00:00-08:00", BS_PST),
		tester(230, "IsoOffsetDateTime, DT_YearMonth, PST timezone", DT_YearMonth, new TemporalSwap.IsoOffsetDateTime(), "2012-12-01T00:00:00-08:00", BS_PST),
		tester(231, "IsoOffsetDateTime, DT_HijrahDate, PST timezone", DT_HijrahDate, new TemporalSwap.IsoOffsetDateTime(), "2012-12-21T00:00:00-08:00", BS_PST),
		tester(232, "IsoOffsetDateTime, DT_JapaneseDate, PST timezone", DT_JapaneseDate, new TemporalSwap.IsoOffsetDateTime(), "2012-12-21T00:00:00-08:00", BS_PST),
		tester(233, "IsoOffsetDateTime, DT_MinguoDate, PST timezone", DT_MinguoDate, new TemporalSwap.IsoOffsetDateTime(), "2012-12-21T00:00:00-08:00", BS_PST),
		tester(234, "IsoOffsetDateTime, DT_ThaiBuddhistDate, PST timezone", DT_ThaiBuddhistDate, new TemporalSwap.IsoOffsetDateTime(), "2012-12-21T00:00:00-08:00", BS_PST),
		tester(235, "IsoOffsetTime, DT_ZonedDateTime, Default timezone", DT_ZonedDateTime, new TemporalSwap.IsoOffsetTime(), "12:34:56Z", BS_DEFAULT),
		tester(236, "IsoOffsetTime, DT_Instant, Default timezone", DT_Instant, new TemporalSwap.IsoOffsetTime(), "12:34:56Z", BS_DEFAULT),
		tester(237, "IsoOffsetTime, DT_LocalDate, Default timezone", DT_LocalDate, new TemporalSwap.IsoOffsetTime(), "00:00:00-05:00", BS_DEFAULT),
		tester(238, "IsoOffsetTime, DT_LocalDateTime, Default timezone", DT_LocalDateTime, new TemporalSwap.IsoOffsetTime(), "12:34:56-05:00", BS_DEFAULT),
		tester(239, "IsoOffsetTime, DT_LocalTime, Default timezone", DT_LocalTime, new TemporalSwap.IsoOffsetTime(), "12:34:56-05:00", BS_DEFAULT),
		tester(240, "IsoOffsetTime, DT_OffsetDateTime, Default timezone", DT_OffsetDateTime, new TemporalSwap.IsoOffsetTime(), "12:34:56-05:00", BS_DEFAULT),
		tester(241, "IsoOffsetTime, DT_OffsetTime, Default timezone", DT_OffsetTime, new TemporalSwap.IsoOffsetTime(), "12:34:56-05:00", BS_DEFAULT),
		tester(242, "IsoOffsetTime, DT_Year, Default timezone", DT_Year, new TemporalSwap.IsoOffsetTime(), "00:00:00-05:00", BS_DEFAULT),
		tester(243, "IsoOffsetTime, DT_YearMonth, Default timezone", DT_YearMonth, new TemporalSwap.IsoOffsetTime(), "00:00:00-05:00", BS_DEFAULT),
		tester(244, "IsoOffsetTime, DT_HijrahDate, Default timezone", DT_HijrahDate, new TemporalSwap.IsoOffsetTime(), "00:00:00-05:00", BS_DEFAULT),
		tester(245, "IsoOffsetTime, DT_JapaneseDate, Default timezone", DT_JapaneseDate, new TemporalSwap.IsoOffsetTime(), "00:00:00-05:00", BS_DEFAULT),
		tester(246, "IsoOffsetTime, DT_MinguoDate, Default timezone", DT_MinguoDate, new TemporalSwap.IsoOffsetTime(), "00:00:00-05:00", BS_DEFAULT),
		tester(247, "IsoOffsetTime, DT_ThaiBuddhistDate, Default timezone", DT_ThaiBuddhistDate, new TemporalSwap.IsoOffsetTime(), "00:00:00-05:00", BS_DEFAULT),
		tester(248, "IsoOffsetTime, DT_ZonedDateTime, PST timezone", DT_ZonedDateTime, new TemporalSwap.IsoOffsetTime(), "12:34:56Z", BS_PST),
		tester(249, "IsoOffsetTime, DT_Instant, PST timezone", DT_Instant, new TemporalSwap.IsoOffsetTime(), "12:34:56Z", BS_PST),
		tester(250, "IsoOffsetTime, DT_LocalDate, PST timezone", DT_LocalDate, new TemporalSwap.IsoOffsetTime(), "00:00:00-08:00", BS_PST),
		tester(251, "IsoOffsetTime, DT_LocalDateTime, PST timezone", DT_LocalDateTime, new TemporalSwap.IsoOffsetTime(), "12:34:56-08:00", BS_PST),
		tester(252, "IsoOffsetTime, DT_LocalTime, PST timezone", DT_LocalTime, new TemporalSwap.IsoOffsetTime(), "12:34:56-08:00", BS_PST),
		tester(253, "IsoOffsetTime, DT_OffsetDateTime, PST timezone", DT_OffsetDateTime, new TemporalSwap.IsoOffsetTime(), "12:34:56-05:00", BS_PST),
		tester(254, "IsoOffsetTime, DT_OffsetTime, PST timezone", DT_OffsetTime, new TemporalSwap.IsoOffsetTime(), "12:34:56-05:00", BS_PST),
		tester(255, "IsoOffsetTime, DT_Year, PST timezone", DT_Year, new TemporalSwap.IsoOffsetTime(), "00:00:00-08:00", BS_PST),
		tester(256, "IsoOffsetTime, DT_YearMonth, PST timezone", DT_YearMonth, new TemporalSwap.IsoOffsetTime(), "00:00:00-08:00", BS_PST),
		tester(257, "IsoOffsetTime, DT_HijrahDate, PST timezone", DT_HijrahDate, new TemporalSwap.IsoOffsetTime(), "00:00:00-08:00", BS_PST),
		tester(258, "IsoOffsetTime, DT_JapaneseDate, PST timezone", DT_JapaneseDate, new TemporalSwap.IsoOffsetTime(), "00:00:00-08:00", BS_PST),
		tester(259, "IsoOffsetTime, DT_MinguoDate, PST timezone", DT_MinguoDate, new TemporalSwap.IsoOffsetTime(), "00:00:00-08:00", BS_PST),
		tester(260, "IsoOffsetTime, DT_ThaiBuddhistDate, PST timezone", DT_ThaiBuddhistDate, new TemporalSwap.IsoOffsetTime(), "00:00:00-08:00", BS_PST),
		tester(261, "IsoOrdinalDate, DT_ZonedDateTime, Default timezone", DT_ZonedDateTime, new TemporalSwap.IsoOrdinalDate(), "2012-356Z", BS_DEFAULT),
		tester(262, "IsoOrdinalDate, DT_Instant, Default timezone", DT_Instant, new TemporalSwap.IsoOrdinalDate(), "2012-356Z", BS_DEFAULT),
		tester(263, "IsoOrdinalDate, DT_LocalDate, Default timezone", DT_LocalDate, new TemporalSwap.IsoOrdinalDate(), "2012-356", BS_DEFAULT),
		tester(264, "IsoOrdinalDate, DT_LocalDateTime, Default timezone", DT_LocalDateTime, new TemporalSwap.IsoOrdinalDate(), "2012-356", BS_DEFAULT),
		tester(265, "IsoOrdinalDate, DT_LocalTime, Default timezone", DT_LocalTime, new TemporalSwap.IsoOrdinalDate(), "1970-001", BS_DEFAULT),
		tester(266, "IsoOrdinalDate, DT_OffsetDateTime, Default timezone", DT_OffsetDateTime, new TemporalSwap.IsoOrdinalDate(), "2012-356-05:00", BS_DEFAULT),
		tester(267, "IsoOrdinalDate, DT_OffsetTime, Default timezone", DT_OffsetTime, new TemporalSwap.IsoOrdinalDate(), "1970-001-05:00", BS_DEFAULT),
		tester(268, "IsoOrdinalDate, DT_Year, Default timezone", DT_Year, new TemporalSwap.IsoOrdinalDate(), "2012-001", BS_DEFAULT),
		tester(269, "IsoOrdinalDate, DT_YearMonth, Default timezone", DT_YearMonth, new TemporalSwap.IsoOrdinalDate(), "2012-336", BS_DEFAULT),
		tester(270, "IsoOrdinalDate, DT_HijrahDate, Default timezone", DT_HijrahDate, new TemporalSwap.IsoOrdinalDate(), "2012-356", BS_DEFAULT),
		tester(271, "IsoOrdinalDate, DT_JapaneseDate, Default timezone", DT_JapaneseDate, new TemporalSwap.IsoOrdinalDate(), "2012-356", BS_DEFAULT),
		tester(272, "IsoOrdinalDate, DT_MinguoDate, Default timezone", DT_MinguoDate, new TemporalSwap.IsoOrdinalDate(), "2012-356", BS_DEFAULT),
		tester(273, "IsoOrdinalDate, DT_ThaiBuddhistDate, Default timezone", DT_ThaiBuddhistDate, new TemporalSwap.IsoOrdinalDate(), "2012-356", BS_DEFAULT),
		tester(274, "IsoOrdinalDate, DT_ZonedDateTime, PST timezone", DT_ZonedDateTime, new TemporalSwap.IsoOrdinalDate(), "2012-356Z", BS_PST),
		tester(275, "IsoOrdinalDate, DT_Instant, PST timezone", DT_Instant, new TemporalSwap.IsoOrdinalDate(), "2012-356Z", BS_PST),
		tester(276, "IsoOrdinalDate, DT_LocalDate, PST timezone", DT_LocalDate, new TemporalSwap.IsoOrdinalDate(), "2012-356", BS_PST),
		tester(277, "IsoOrdinalDate, DT_LocalDateTime, PST timezone", DT_LocalDateTime, new TemporalSwap.IsoOrdinalDate(), "2012-356", BS_PST),
		tester(278, "IsoOrdinalDate, DT_LocalTime, PST timezone", DT_LocalTime, new TemporalSwap.IsoOrdinalDate(), "1970-001", BS_PST),
		tester(279, "IsoOrdinalDate, DT_OffsetDateTime, PST timezone", DT_OffsetDateTime, new TemporalSwap.IsoOrdinalDate(), "2012-356-05:00", BS_PST),
		tester(280, "IsoOrdinalDate, DT_OffsetTime, PST timezone", DT_OffsetTime, new TemporalSwap.IsoOrdinalDate(), "1970-001-05:00", BS_PST),
		tester(281, "IsoOrdinalDate, DT_Year, PST timezone", DT_Year, new TemporalSwap.IsoOrdinalDate(), "2012-001", BS_PST),
		tester(282, "IsoOrdinalDate, DT_YearMonth, PST timezone", DT_YearMonth, new TemporalSwap.IsoOrdinalDate(), "2012-336", BS_PST),
		tester(283, "IsoOrdinalDate, DT_HijrahDate, PST timezone", DT_HijrahDate, new TemporalSwap.IsoOrdinalDate(), "2012-356", BS_PST),
		tester(284, "IsoOrdinalDate, DT_JapaneseDate, PST timezone", DT_JapaneseDate, new TemporalSwap.IsoOrdinalDate(), "2012-356", BS_PST),
		tester(285, "IsoOrdinalDate, DT_MinguoDate, PST timezone", DT_MinguoDate, new TemporalSwap.IsoOrdinalDate(), "2012-356", BS_PST),
		tester(286, "IsoOrdinalDate, DT_ThaiBuddhistDate, PST timezone", DT_ThaiBuddhistDate, new TemporalSwap.IsoOrdinalDate(), "2012-356", BS_PST),
		tester(287, "IsoTime, DT_ZonedDateTime, Default timezone", DT_ZonedDateTime, new TemporalSwap.IsoTime(), "12:34:56Z", BS_DEFAULT),
		tester(288, "IsoTime, DT_Instant, Default timezone", DT_Instant, new TemporalSwap.IsoTime(), "12:34:56Z", BS_DEFAULT),
		tester(289, "IsoTime, DT_LocalDate, Default timezone", DT_LocalDate, new TemporalSwap.IsoTime(), "00:00:00", BS_DEFAULT),
		tester(290, "IsoTime, DT_LocalDateTime, Default timezone", DT_LocalDateTime, new TemporalSwap.IsoTime(), "12:34:56", BS_DEFAULT),
		tester(291, "IsoTime, DT_LocalTime, Default timezone", DT_LocalTime, new TemporalSwap.IsoTime(), "12:34:56", BS_DEFAULT),
		tester(292, "IsoTime, DT_OffsetDateTime, Default timezone", DT_OffsetDateTime, new TemporalSwap.IsoTime(), "12:34:56-05:00", BS_DEFAULT),
		tester(293, "IsoTime, DT_OffsetTime, Default timezone", DT_OffsetTime, new TemporalSwap.IsoTime(), "12:34:56-05:00", BS_DEFAULT),
		tester(294, "IsoTime, DT_Year, Default timezone", DT_Year, new TemporalSwap.IsoTime(), "00:00:00", BS_DEFAULT),
		tester(295, "IsoTime, DT_YearMonth, Default timezone", DT_YearMonth, new TemporalSwap.IsoTime(), "00:00:00", BS_DEFAULT),
		tester(296, "IsoTime, DT_HijrahDate, Default timezone", DT_HijrahDate, new TemporalSwap.IsoTime(), "00:00:00", BS_DEFAULT),
		tester(297, "IsoTime, DT_JapaneseDate, Default timezone", DT_JapaneseDate, new TemporalSwap.IsoTime(), "00:00:00", BS_DEFAULT),
		tester(298, "IsoTime, DT_MinguoDate, Default timezone", DT_MinguoDate, new TemporalSwap.IsoTime(), "00:00:00", BS_DEFAULT),
		tester(299, "IsoTime, DT_ThaiBuddhistDate, Default timezone", DT_ThaiBuddhistDate, new TemporalSwap.IsoTime(), "00:00:00", BS_DEFAULT),
		tester(300, "IsoTime, DT_ZonedDateTime, PST timezone", DT_ZonedDateTime, new TemporalSwap.IsoTime(), "12:34:56Z", BS_PST),
		tester(301, "IsoTime, DT_Instant, PST timezone", DT_Instant, new TemporalSwap.IsoTime(), "12:34:56Z", BS_PST),
		tester(302, "IsoTime, DT_LocalDate, PST timezone", DT_LocalDate, new TemporalSwap.IsoTime(), "00:00:00", BS_PST),
		tester(303, "IsoTime, DT_LocalDateTime, PST timezone", DT_LocalDateTime, new TemporalSwap.IsoTime(), "12:34:56", BS_PST),
		tester(304, "IsoTime, DT_LocalTime, PST timezone", DT_LocalTime, new TemporalSwap.IsoTime(), "12:34:56", BS_PST),
		tester(305, "IsoTime, DT_OffsetDateTime, PST timezone", DT_OffsetDateTime, new TemporalSwap.IsoTime(), "12:34:56-05:00", BS_PST),
		tester(306, "IsoTime, DT_OffsetTime, PST timezone", DT_OffsetTime, new TemporalSwap.IsoTime(), "12:34:56-05:00", BS_PST),
		tester(307, "IsoTime, DT_Year, PST timezone", DT_Year, new TemporalSwap.IsoTime(), "00:00:00", BS_PST),
		tester(308, "IsoTime, DT_YearMonth, PST timezone", DT_YearMonth, new TemporalSwap.IsoTime(), "00:00:00", BS_PST),
		tester(309, "IsoTime, DT_HijrahDate, PST timezone", DT_HijrahDate, new TemporalSwap.IsoTime(), "00:00:00", BS_PST),
		tester(310, "IsoTime, DT_JapaneseDate, PST timezone", DT_JapaneseDate, new TemporalSwap.IsoTime(), "00:00:00", BS_PST),
		tester(311, "IsoTime, DT_MinguoDate, PST timezone", DT_MinguoDate, new TemporalSwap.IsoTime(), "00:00:00", BS_PST),
		tester(312, "IsoTime, DT_ThaiBuddhistDate, PST timezone", DT_ThaiBuddhistDate, new TemporalSwap.IsoTime(), "00:00:00", BS_PST),
		tester(313, "IsoWeekDate, DT_ZonedDateTime, Default timezone", DT_ZonedDateTime, new TemporalSwap.IsoWeekDate(), "2012-W51-5Z", BS_DEFAULT),
		tester(314, "IsoWeekDate, DT_Instant, Default timezone", DT_Instant, new TemporalSwap.IsoWeekDate(), "2012-W51-5Z", BS_DEFAULT),
		tester(315, "IsoWeekDate, DT_LocalDate, Default timezone", DT_LocalDate, new TemporalSwap.IsoWeekDate(), "2012-W51-5", BS_DEFAULT),
		tester(316, "IsoWeekDate, DT_LocalDateTime, Default timezone", DT_LocalDateTime, new TemporalSwap.IsoWeekDate(), "2012-W51-5", BS_DEFAULT),
		tester(317, "IsoWeekDate, DT_LocalTime, Default timezone", DT_LocalTime, new TemporalSwap.IsoWeekDate(), "1970-W01-4", BS_DEFAULT),
		tester(318, "IsoWeekDate, DT_OffsetDateTime, Default timezone", DT_OffsetDateTime, new TemporalSwap.IsoWeekDate(), "2012-W51-5-05:00", BS_DEFAULT),
		tester(319, "IsoWeekDate, DT_OffsetTime, Default timezone", DT_OffsetTime, new TemporalSwap.IsoWeekDate(), "1970-W01-4-05:00", BS_DEFAULT),
		tester(320, "IsoWeekDate, DT_Year, Default timezone", DT_Year, new TemporalSwap.IsoWeekDate(), "2011-W52-7", BS_DEFAULT),
		tester(321, "IsoWeekDate, DT_YearMonth, Default timezone", DT_YearMonth, new TemporalSwap.IsoWeekDate(), "2012-W48-6", BS_DEFAULT),
		tester(322, "IsoWeekDate, DT_HijrahDate, Default timezone", DT_HijrahDate, new TemporalSwap.IsoWeekDate(), "2012-W51-5", BS_DEFAULT),
		tester(323, "IsoWeekDate, DT_JapaneseDate, Default timezone", DT_JapaneseDate, new TemporalSwap.IsoWeekDate(), "2012-W51-5", BS_DEFAULT),
		tester(324, "IsoWeekDate, DT_MinguoDate, Default timezone", DT_MinguoDate, new TemporalSwap.IsoWeekDate(), "2012-W51-5", BS_DEFAULT),
		tester(325, "IsoWeekDate, DT_ThaiBuddhistDate, Default timezone", DT_ThaiBuddhistDate, new TemporalSwap.IsoWeekDate(), "2012-W51-5", BS_DEFAULT),
		tester(326, "IsoWeekDate, DT_ZonedDateTime, PST timezone", DT_ZonedDateTime, new TemporalSwap.IsoWeekDate(), "2012-W51-5Z", BS_PST),
		tester(327, "IsoWeekDate, DT_Instant, PST timezone", DT_Instant, new TemporalSwap.IsoWeekDate(), "2012-W51-5Z", BS_PST),
		tester(328, "IsoWeekDate, DT_LocalDate, PST timezone", DT_LocalDate, new TemporalSwap.IsoWeekDate(), "2012-W51-5", BS_PST),
		tester(329, "IsoWeekDate, DT_LocalDateTime, PST timezone", DT_LocalDateTime, new TemporalSwap.IsoWeekDate(), "2012-W51-5", BS_PST),
		tester(330, "IsoWeekDate, DT_LocalTime, PST timezone", DT_LocalTime, new TemporalSwap.IsoWeekDate(), "1970-W01-4", BS_PST),
		tester(331, "IsoWeekDate, DT_OffsetDateTime, PST timezone", DT_OffsetDateTime, new TemporalSwap.IsoWeekDate(), "2012-W51-5-05:00", BS_PST),
		tester(332, "IsoWeekDate, DT_OffsetTime, PST timezone", DT_OffsetTime, new TemporalSwap.IsoWeekDate(), "1970-W01-4-05:00", BS_PST),
		tester(333, "IsoWeekDate, DT_Year, PST timezone", DT_Year, new TemporalSwap.IsoWeekDate(), "2011-W52-7", BS_PST),
		tester(334, "IsoWeekDate, DT_YearMonth, PST timezone", DT_YearMonth, new TemporalSwap.IsoWeekDate(), "2012-W48-6", BS_PST),
		tester(335, "IsoWeekDate, DT_HijrahDate, PST timezone", DT_HijrahDate, new TemporalSwap.IsoWeekDate(), "2012-W51-5", BS_PST),
		tester(336, "IsoWeekDate, DT_JapaneseDate, PST timezone", DT_JapaneseDate, new TemporalSwap.IsoWeekDate(), "2012-W51-5", BS_PST),
		tester(337, "IsoWeekDate, DT_MinguoDate, PST timezone", DT_MinguoDate, new TemporalSwap.IsoWeekDate(), "2012-W51-5", BS_PST),
		tester(338, "IsoWeekDate, DT_ThaiBuddhistDate, PST timezone", DT_ThaiBuddhistDate, new TemporalSwap.IsoWeekDate(), "2012-W51-5", BS_PST),
		tester(339, "IsoYear, DT_ZonedDateTime, Default timezone", DT_ZonedDateTime, new TemporalSwap.IsoYear(), "2012", BS_DEFAULT),
		tester(340, "IsoYear, DT_Instant, Default timezone", DT_Instant, new TemporalSwap.IsoYear(), "2012", BS_DEFAULT),
		tester(341, "IsoYear, DT_LocalDate, Default timezone", DT_LocalDate, new TemporalSwap.IsoYear(), "2012", BS_DEFAULT),
		tester(342, "IsoYear, DT_LocalDateTime, Default timezone", DT_LocalDateTime, new TemporalSwap.IsoYear(), "2012", BS_DEFAULT),
		tester(343, "IsoYear, DT_LocalTime, Default timezone", DT_LocalTime, new TemporalSwap.IsoYear(), "1970", BS_DEFAULT),
		tester(344, "IsoYear, DT_OffsetDateTime, Default timezone", DT_OffsetDateTime, new TemporalSwap.IsoYear(), "2012", BS_DEFAULT),
		tester(345, "IsoYear, DT_OffsetTime, Default timezone", DT_OffsetTime, new TemporalSwap.IsoYear(), "1970", BS_DEFAULT),
		tester(346, "IsoYear, DT_Year, Default timezone", DT_Year, new TemporalSwap.IsoYear(), "2012", BS_DEFAULT),
		tester(347, "IsoYear, DT_YearMonth, Default timezone", DT_YearMonth, new TemporalSwap.IsoYear(), "2012", BS_DEFAULT),
		tester(348, "IsoYear, DT_HijrahDate, Default timezone", DT_HijrahDate, new TemporalSwap.IsoYear(), "2012", BS_DEFAULT),
		tester(349, "IsoYear, DT_JapaneseDate, Default timezone", DT_JapaneseDate, new TemporalSwap.IsoYear(), "2012", BS_DEFAULT),
		tester(350, "IsoYear, DT_MinguoDate, Default timezone", DT_MinguoDate, new TemporalSwap.IsoYear(), "2012", BS_DEFAULT),
		tester(351, "IsoYear, DT_ThaiBuddhistDate, Default timezone", DT_ThaiBuddhistDate, new TemporalSwap.IsoYear(), "2012", BS_DEFAULT),
		tester(352, "IsoYear, DT_ZonedDateTime, PST timezone", DT_ZonedDateTime, new TemporalSwap.IsoYear(), "2012", BS_PST),
		tester(353, "IsoYear, DT_Instant, PST timezone", DT_Instant, new TemporalSwap.IsoYear(), "2012", BS_PST),
		tester(354, "IsoYear, DT_LocalDate, PST timezone", DT_LocalDate, new TemporalSwap.IsoYear(), "2012", BS_PST),
		tester(355, "IsoYear, DT_LocalDateTime, PST timezone", DT_LocalDateTime, new TemporalSwap.IsoYear(), "2012", BS_PST),
		tester(356, "IsoYear, DT_LocalTime, PST timezone", DT_LocalTime, new TemporalSwap.IsoYear(), "1970", BS_PST),
		tester(357, "IsoYear, DT_OffsetDateTime, PST timezone", DT_OffsetDateTime, new TemporalSwap.IsoYear(), "2012", BS_PST),
		tester(358, "IsoYear, DT_OffsetTime, PST timezone", DT_OffsetTime, new TemporalSwap.IsoYear(), "1970", BS_PST),
		tester(359, "IsoYear, DT_Year, PST timezone", DT_Year, new TemporalSwap.IsoYear(), "2012", BS_PST),
		tester(360, "IsoYear, DT_YearMonth, PST timezone", DT_YearMonth, new TemporalSwap.IsoYear(), "2012", BS_PST),
		tester(361, "IsoYear, DT_HijrahDate, PST timezone", DT_HijrahDate, new TemporalSwap.IsoYear(), "2012", BS_PST),
		tester(362, "IsoYear, DT_JapaneseDate, PST timezone", DT_JapaneseDate, new TemporalSwap.IsoYear(), "2012", BS_PST),
		tester(363, "IsoYear, DT_MinguoDate, PST timezone", DT_MinguoDate, new TemporalSwap.IsoYear(), "2012", BS_PST),
		tester(364, "IsoYear, DT_ThaiBuddhistDate, PST timezone", DT_ThaiBuddhistDate, new TemporalSwap.IsoYear(), "2012", BS_PST),
		tester(365, "IsoYearMonth, DT_ZonedDateTime, Default timezone", DT_ZonedDateTime, new TemporalSwap.IsoYearMonth(), "2012-12", BS_DEFAULT),
		tester(366, "IsoYearMonth, DT_Instant, Default timezone", DT_Instant, new TemporalSwap.IsoYearMonth(), "2012-12", BS_DEFAULT),
		tester(367, "IsoYearMonth, DT_LocalDate, Default timezone", DT_LocalDate, new TemporalSwap.IsoYearMonth(), "2012-12", BS_DEFAULT),
		tester(368, "IsoYearMonth, DT_LocalDateTime, Default timezone", DT_LocalDateTime, new TemporalSwap.IsoYearMonth(), "2012-12", BS_DEFAULT),
		tester(369, "IsoYearMonth, DT_LocalTime, Default timezone", DT_LocalTime, new TemporalSwap.IsoYearMonth(), "1970-01", BS_DEFAULT),
		tester(370, "IsoYearMonth, DT_OffsetDateTime, Default timezone", DT_OffsetDateTime, new TemporalSwap.IsoYearMonth(), "2012-12", BS_DEFAULT),
		tester(371, "IsoYearMonth, DT_OffsetTime, Default timezone", DT_OffsetTime, new TemporalSwap.IsoYearMonth(), "1970-01", BS_DEFAULT),
		tester(372, "IsoYearMonth, DT_Year, Default timezone", DT_Year, new TemporalSwap.IsoYearMonth(), "2012-01", BS_DEFAULT),
		tester(373, "IsoYearMonth, DT_YearMonth, Default timezone", DT_YearMonth, new TemporalSwap.IsoYearMonth(), "2012-12", BS_DEFAULT),
		tester(374, "IsoYearMonth, DT_HijrahDate, Default timezone", DT_HijrahDate, new TemporalSwap.IsoYearMonth(), "2012-12", BS_DEFAULT),
		tester(375, "IsoYearMonth, DT_JapaneseDate, Default timezone", DT_JapaneseDate, new TemporalSwap.IsoYearMonth(), "2012-12", BS_DEFAULT),
		tester(376, "IsoYearMonth, DT_MinguoDate, Default timezone", DT_MinguoDate, new TemporalSwap.IsoYearMonth(), "2012-12", BS_DEFAULT),
		tester(377, "IsoYearMonth, DT_ThaiBuddhistDate, Default timezone", DT_ThaiBuddhistDate, new TemporalSwap.IsoYearMonth(), "2012-12", BS_DEFAULT),
		tester(378, "IsoYearMonth, DT_ZonedDateTime, PST timezone", DT_ZonedDateTime, new TemporalSwap.IsoYearMonth(), "2012-12", BS_PST),
		tester(379, "IsoYearMonth, DT_Instant, PST timezone", DT_Instant, new TemporalSwap.IsoYearMonth(), "2012-12", BS_PST),
		tester(380, "IsoYearMonth, DT_LocalDate, PST timezone", DT_LocalDate, new TemporalSwap.IsoYearMonth(), "2012-12", BS_PST),
		tester(381, "IsoYearMonth, DT_LocalDateTime, PST timezone", DT_LocalDateTime, new TemporalSwap.IsoYearMonth(), "2012-12", BS_PST),
		tester(382, "IsoYearMonth, DT_LocalTime, PST timezone", DT_LocalTime, new TemporalSwap.IsoYearMonth(), "1970-01", BS_PST),
		tester(383, "IsoYearMonth, DT_OffsetDateTime, PST timezone", DT_OffsetDateTime, new TemporalSwap.IsoYearMonth(), "2012-12", BS_PST),
		tester(384, "IsoYearMonth, DT_OffsetTime, PST timezone", DT_OffsetTime, new TemporalSwap.IsoYearMonth(), "1970-01", BS_PST),
		tester(385, "IsoYearMonth, DT_Year, PST timezone", DT_Year, new TemporalSwap.IsoYearMonth(), "2012-01", BS_PST),
		tester(386, "IsoYearMonth, DT_YearMonth, PST timezone", DT_YearMonth, new TemporalSwap.IsoYearMonth(), "2012-12", BS_PST),
		tester(387, "IsoYearMonth, DT_HijrahDate, PST timezone", DT_HijrahDate, new TemporalSwap.IsoYearMonth(), "2012-12", BS_PST),
		tester(388, "IsoYearMonth, DT_JapaneseDate, PST timezone", DT_JapaneseDate, new TemporalSwap.IsoYearMonth(), "2012-12", BS_PST),
		tester(389, "IsoYearMonth, DT_MinguoDate, PST timezone", DT_MinguoDate, new TemporalSwap.IsoYearMonth(), "2012-12", BS_PST),
		tester(390, "IsoYearMonth, DT_ThaiBuddhistDate, PST timezone", DT_ThaiBuddhistDate, new TemporalSwap.IsoYearMonth(), "2012-12", BS_PST),
		tester(391, "IsoZonedDateTime, DT_ZonedDateTime, Default timezone", DT_ZonedDateTime, new TemporalSwap.IsoZonedDateTime(), "2012-12-21T12:34:56Z", BS_DEFAULT),
		tester(392, "IsoZonedDateTime, DT_Instant, Default timezone", DT_Instant, new TemporalSwap.IsoZonedDateTime(), "2012-12-21T12:34:56Z", BS_DEFAULT),
		tester(393, "IsoYearMonth, DT_LocalDate, Default timezone", DT_LocalDate, new TemporalSwap.IsoYearMonth(), "2012-12", BS_DEFAULT),
		tester(394, "IsoZonedDateTime, DT_LocalDateTime, Default timezone", DT_LocalDateTime, new TemporalSwap.IsoZonedDateTime(), "2012-12-21T12:34:56-05:00[GMT-05:00]", BS_DEFAULT),
		tester(395, "IsoZonedDateTime, DT_LocalTime, Default timezone", DT_LocalTime, new TemporalSwap.IsoZonedDateTime(), "1970-01-01T12:34:56-05:00[GMT-05:00]", BS_DEFAULT),
		tester(396, "IsoZonedDateTime, DT_OffsetDateTime, Default timezone", DT_OffsetDateTime, new TemporalSwap.IsoZonedDateTime(), "2012-12-21T12:34:56-05:00", BS_DEFAULT),
		tester(397, "IsoZonedDateTime, DT_OffsetTime, Default timezone", DT_OffsetTime, new TemporalSwap.IsoZonedDateTime(), "1970-01-01T12:34:56-05:00", BS_DEFAULT),
		tester(398, "IsoZonedDateTime, DT_Year, Default timezone", DT_Year, new TemporalSwap.IsoZonedDateTime(), "2012-01-01T00:00:00-05:00[GMT-05:00]", BS_DEFAULT),
		tester(399, "IsoZonedDateTime, DT_YearMonth, Default timezone", DT_YearMonth, new TemporalSwap.IsoZonedDateTime(), "2012-12-01T00:00:00-05:00[GMT-05:00]", BS_DEFAULT),
		tester(400, "IsoZonedDateTime, DT_HijrahDate, Default timezone", DT_HijrahDate, new TemporalSwap.IsoZonedDateTime(), "2012-12-21T00:00:00-05:00[GMT-05:00]", BS_DEFAULT),
		tester(401, "IsoZonedDateTime, DT_JapaneseDate, Default timezone", DT_JapaneseDate, new TemporalSwap.IsoZonedDateTime(), "2012-12-21T00:00:00-05:00[GMT-05:00]", BS_DEFAULT),
		tester(402, "IsoZonedDateTime, DT_MinguoDate, Default timezone", DT_MinguoDate, new TemporalSwap.IsoZonedDateTime(), "2012-12-21T00:00:00-05:00[GMT-05:00]", BS_DEFAULT),
		tester(403, "IsoZonedDateTime, DT_ThaiBuddhistDate, Default timezone", DT_ThaiBuddhistDate, new TemporalSwap.IsoZonedDateTime(), "2012-12-21T00:00:00-05:00[GMT-05:00]", BS_DEFAULT),
		tester(404, "IsoZonedDateTime, DT_ZonedDateTime, PST timezone", DT_ZonedDateTime, new TemporalSwap.IsoZonedDateTime(), "2012-12-21T12:34:56Z", BS_PST),
		tester(405, "IsoZonedDateTime, DT_Instant, PST timezone", DT_Instant, new TemporalSwap.IsoZonedDateTime(), "2012-12-21T12:34:56Z", BS_PST),
		tester(406, "IsoYearMonth, DT_LocalDate, PST timezone", DT_LocalDate, new TemporalSwap.IsoYearMonth(), "2012-12", BS_PST),
		tester(407, "IsoZonedDateTime, DT_LocalDateTime, PST timezone", DT_LocalDateTime, new TemporalSwap.IsoZonedDateTime(), "2012-12-21T12:34:56-08:00[America/Los_Angeles]", BS_PST),
		tester(408, "IsoZonedDateTime, DT_LocalTime, PST timezone", DT_LocalTime, new TemporalSwap.IsoZonedDateTime(), "1970-01-01T12:34:56-08:00[America/Los_Angeles]", BS_PST),
		tester(409, "IsoZonedDateTime, DT_OffsetDateTime, PST timezone", DT_OffsetDateTime, new TemporalSwap.IsoZonedDateTime(), "2012-12-21T12:34:56-05:00", BS_PST),
		tester(410, "IsoZonedDateTime, DT_OffsetTime, PST timezone", DT_OffsetTime, new TemporalSwap.IsoZonedDateTime(), "1970-01-01T12:34:56-05:00", BS_PST),
		tester(411, "IsoZonedDateTime, DT_Year, PST timezone", DT_Year, new TemporalSwap.IsoZonedDateTime(), "2012-01-01T00:00:00-08:00[America/Los_Angeles]", BS_PST),
		tester(412, "IsoZonedDateTime, DT_YearMonth, PST timezone", DT_YearMonth, new TemporalSwap.IsoZonedDateTime(), "2012-12-01T00:00:00-08:00[America/Los_Angeles]", BS_PST),
		tester(413, "IsoZonedDateTime, DT_HijrahDate, PST timezone", DT_HijrahDate, new TemporalSwap.IsoZonedDateTime(), "2012-12-21T00:00:00-08:00[America/Los_Angeles]", BS_PST),
		tester(414, "IsoZonedDateTime, DT_JapaneseDate, PST timezone", DT_JapaneseDate, new TemporalSwap.IsoZonedDateTime(), "2012-12-21T00:00:00-08:00[America/Los_Angeles]", BS_PST),
		tester(415, "IsoZonedDateTime, DT_MinguoDate, PST timezone", DT_MinguoDate, new TemporalSwap.IsoZonedDateTime(), "2012-12-21T00:00:00-08:00[America/Los_Angeles]", BS_PST),
		tester(416, "IsoZonedDateTime, DT_ThaiBuddhistDate, PST timezone", DT_ThaiBuddhistDate, new TemporalSwap.IsoZonedDateTime(), "2012-12-21T00:00:00-08:00[America/Los_Angeles]", BS_PST),
		tester(417, "Rfc1123DateTime, DT_ZonedDateTime, Default timezone", DT_ZonedDateTime, new TemporalSwap.Rfc1123DateTime(), "Fri, 21 Dec 2012 12:34:56 GMT", BS_DEFAULT),
		tester(418, "Rfc1123DateTime, DT_Instant, Default timezone", DT_Instant, new TemporalSwap.Rfc1123DateTime(), "Fri, 21 Dec 2012 12:34:56 GMT", BS_DEFAULT),
		tester(419, "Rfc1123DateTime, DT_LocalDate, Default timezone", DT_LocalDate, new TemporalSwap.Rfc1123DateTime(), "Fri, 21 Dec 2012 00:00:00 -0500", BS_DEFAULT),
		tester(420, "Rfc1123DateTime, DT_LocalDateTime, Default timezone", DT_LocalDateTime, new TemporalSwap.Rfc1123DateTime(), "Fri, 21 Dec 2012 12:34:56 -0500", BS_DEFAULT),
		tester(421, "Rfc1123DateTime, DT_LocalTime, Default timezone", DT_LocalTime, new TemporalSwap.Rfc1123DateTime(), "Thu, 1 Jan 1970 12:34:56 -0500", BS_DEFAULT),
		tester(422, "Rfc1123DateTime, DT_OffsetDateTime, Default timezone", DT_OffsetDateTime, new TemporalSwap.Rfc1123DateTime(), "Fri, 21 Dec 2012 12:34:56 -0500", BS_DEFAULT),
		tester(423, "Rfc1123DateTime, DT_OffsetTime, Default timezone", DT_OffsetTime, new TemporalSwap.Rfc1123DateTime(), "Thu, 1 Jan 1970 12:34:56 -0500", BS_DEFAULT),
		tester(424, "Rfc1123DateTime, DT_Year, Default timezone", DT_Year, new TemporalSwap.Rfc1123DateTime(), "Sun, 1 Jan 2012 00:00:00 -0500", BS_DEFAULT),
		tester(425, "Rfc1123DateTime, DT_YearMonth, Default timezone", DT_YearMonth, new TemporalSwap.Rfc1123DateTime(), "Sat, 1 Dec 2012 00:00:00 -0500", BS_DEFAULT),
		tester(426, "Rfc1123DateTime, DT_HijrahDate, Default timezone", DT_HijrahDate, new TemporalSwap.Rfc1123DateTime(), "Fri, 21 Dec 2012 00:00:00 -0500", BS_DEFAULT),
		tester(427, "Rfc1123DateTime, DT_JapaneseDate, Default timezone", DT_JapaneseDate, new TemporalSwap.Rfc1123DateTime(), "Fri, 21 Dec 2012 00:00:00 -0500", BS_DEFAULT),
		tester(428, "Rfc1123DateTime, DT_MinguoDate, Default timezone", DT_MinguoDate, new TemporalSwap.Rfc1123DateTime(), "Fri, 21 Dec 2012 00:00:00 -0500", BS_DEFAULT),
		tester(429, "Rfc1123DateTime, DT_ThaiBuddhistDate, Default timezone", DT_ThaiBuddhistDate, new TemporalSwap.Rfc1123DateTime(), "Fri, 21 Dec 2012 00:00:00 -0500", BS_DEFAULT),
		tester(430, "Rfc1123DateTime, DT_ZonedDateTime, PST timezone", DT_ZonedDateTime, new TemporalSwap.Rfc1123DateTime(), "Fri, 21 Dec 2012 12:34:56 GMT", BS_PST),
		tester(431, "Rfc1123DateTime, DT_Instant, PST timezone", DT_Instant, new TemporalSwap.Rfc1123DateTime(), "Fri, 21 Dec 2012 12:34:56 GMT", BS_PST),
		tester(432, "Rfc1123DateTime, DT_LocalDate, PST timezone", DT_LocalDate, new TemporalSwap.Rfc1123DateTime(), "Fri, 21 Dec 2012 00:00:00 -0800", BS_PST),
		tester(433, "Rfc1123DateTime, DT_LocalDateTime, PST timezone", DT_LocalDateTime, new TemporalSwap.Rfc1123DateTime(), "Fri, 21 Dec 2012 12:34:56 -0800", BS_PST),
		tester(434, "Rfc1123DateTime, DT_LocalTime, PST timezone", DT_LocalTime, new TemporalSwap.Rfc1123DateTime(), "Thu, 1 Jan 1970 12:34:56 -0800", BS_PST),
		tester(435, "Rfc1123DateTime, DT_OffsetDateTime, PST timezone", DT_OffsetDateTime, new TemporalSwap.Rfc1123DateTime(), "Fri, 21 Dec 2012 12:34:56 -0500", BS_PST),
		tester(436, "Rfc1123DateTime, DT_OffsetTime, PST timezone", DT_OffsetTime, new TemporalSwap.Rfc1123DateTime(), "Thu, 1 Jan 1970 12:34:56 -0500", BS_PST),
		tester(437, "Rfc1123DateTime, DT_Year, PST timezone", DT_Year, new TemporalSwap.Rfc1123DateTime(), "Sun, 1 Jan 2012 00:00:00 -0800", BS_PST),
		tester(438, "Rfc1123DateTime, DT_YearMonth, PST timezone", DT_YearMonth, new TemporalSwap.Rfc1123DateTime(), "Sat, 1 Dec 2012 00:00:00 -0800", BS_PST),
		tester(439, "Rfc1123DateTime, DT_HijrahDate, PST timezone", DT_HijrahDate, new TemporalSwap.Rfc1123DateTime(), "Fri, 21 Dec 2012 00:00:00 -0800", BS_PST),
		tester(440, "Rfc1123DateTime, DT_JapaneseDate, PST timezone", DT_JapaneseDate, new TemporalSwap.Rfc1123DateTime(), "Fri, 21 Dec 2012 00:00:00 -0800", BS_PST),
		tester(441, "Rfc1123DateTime, DT_MinguoDate, PST timezone", DT_MinguoDate, new TemporalSwap.Rfc1123DateTime(), "Fri, 21 Dec 2012 00:00:00 -0800", BS_PST),
		tester(442, "Rfc1123DateTime, DT_ThaiBuddhistDate, PST timezone", DT_ThaiBuddhistDate, new TemporalSwap.Rfc1123DateTime(), "Fri, 21 Dec 2012 00:00:00 -0800", BS_PST),
		tester(443, "BasicIsoDate, null", null, new TemporalSwap.BasicIsoDate(), null, BS_DEFAULT)
	};

	static StringSwap_Tester<?>[] testers() {
		return TESTERS;
	}
}