/*
 * Decompiled with CFR 0.152.
 */
package org.apache.streampark.console.core.task;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import java.util.Date;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.streampark.console.core.entity.Application;
import org.apache.streampark.console.core.entity.Savepoint;
import org.apache.streampark.console.core.enums.CheckPointStatus;
import org.apache.streampark.console.core.enums.FailoverStrategy;
import org.apache.streampark.console.core.metrics.flink.CheckPoints;
import org.apache.streampark.console.core.service.ApplicationService;
import org.apache.streampark.console.core.service.SavepointService;
import org.apache.streampark.console.core.service.alert.AlertService;
import org.apache.streampark.console.core.task.FlinkAppHttpWatcher;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

@Component
public class CheckpointProcessor {
    private static final Byte DEFAULT_FLAG_BYTE = Byte.valueOf("0");
    private static final Integer SAVEPOINT_CACHE_HOUR = 1;
    private final Cache<String, Long> checkPointCache = Caffeine.newBuilder().expireAfterAccess(1L, TimeUnit.DAYS).build();
    private final Cache<String, Byte> savepointedCache = Caffeine.newBuilder().expireAfterWrite((long)SAVEPOINT_CACHE_HOUR.intValue(), TimeUnit.HOURS).build();
    private final Map<Long, Counter> checkPointFailedCache = new ConcurrentHashMap<Long, Counter>(0);
    @Autowired
    private ApplicationService applicationService;
    @Autowired
    private AlertService alertService;
    @Autowired
    private SavepointService savepointService;
    @Autowired
    private FlinkAppHttpWatcher flinkAppHttpWatcher;

    public void process(Application application, @Nonnull CheckPoints checkPoints) {
        checkPoints.getLatestCheckpoint().forEach(checkPoint -> this.process(application, (CheckPoints.CheckPoint)checkPoint));
    }

    private void process(Application application, @Nonnull CheckPoints.CheckPoint checkPoint) {
        String jobID = application.getJobId();
        Long appId = application.getId();
        CheckPointStatus status = checkPoint.getCheckPointStatus();
        CheckPointKey checkPointKey = new CheckPointKey(appId, jobID, checkPoint.getId());
        if (CheckPointStatus.COMPLETED.equals(status)) {
            switch (checkPoint.getCheckPointType()) {
                case SAVEPOINT: {
                    if (!this.checkSaveForSavepoint(checkPointKey, checkPoint)) break;
                    this.savepointedCache.put((Object)checkPointKey.getSavePointId(), (Object)DEFAULT_FLAG_BYTE);
                    this.saveSavepoint(checkPoint, application.getId());
                    this.flinkAppHttpWatcher.cleanSavepoint(application);
                    return;
                }
                case CHECKPOINT: {
                    Long latestChkId = this.getLatestCheckpointId(appId, checkPointKey.getCheckPointId());
                    if (!this.checkSaveForCheckpoint(checkPoint, latestChkId)) break;
                    this.checkPointCache.put((Object)checkPointKey.getCheckPointId(), (Object)checkPoint.getId());
                    this.saveSavepoint(checkPoint, application.getId());
                    break;
                }
            }
            return;
        }
        if (this.shouldProcessFailedTrigger(checkPoint, application.cpFailedTrigger(), status)) {
            Counter counter = this.checkPointFailedCache.get(appId);
            if (counter == null) {
                this.checkPointFailedCache.put(appId, new Counter(checkPoint.getTriggerTimestamp()));
            } else {
                long minute = counter.getDuration(checkPoint.getTriggerTimestamp());
                if (minute <= (long)application.getCpFailureRateInterval().intValue() && counter.getCount() >= application.getCpMaxFailureInterval()) {
                    this.checkPointFailedCache.remove(appId);
                    FailoverStrategy failoverStrategy = FailoverStrategy.of(application.getCpFailureAction());
                    if (failoverStrategy == null) {
                        throw new IllegalArgumentException("Unexpected cpFailureAction: " + application.getCpFailureAction());
                    }
                    switch (failoverStrategy) {
                        case ALERT: {
                            this.alertService.alert(application, CheckPointStatus.FAILED);
                            break;
                        }
                        case RESTART: {
                            try {
                                this.applicationService.restart(application);
                                break;
                            }
                            catch (Exception e) {
                                throw new RuntimeException(e);
                            }
                        }
                    }
                } else {
                    counter.increment();
                }
            }
        }
    }

    private boolean checkSaveForCheckpoint(@Nonnull CheckPoints.CheckPoint checkPoint, Long latestId) {
        return latestId == null || !latestId.equals(checkPoint.getId());
    }

    private boolean checkSaveForSavepoint(CheckPointKey checkPointKey, @Nonnull CheckPoints.CheckPoint checkPoint) {
        return this.savepointedCache.getIfPresent((Object)checkPointKey.getSavePointId()) == null && checkPoint.getTriggerTimestamp() >= System.currentTimeMillis() - TimeUnit.HOURS.toMillis(SAVEPOINT_CACHE_HOUR.intValue());
    }

    @Nullable
    private Long getLatestCheckpointId(Long appId, String checkpointId) {
        return (Long)this.checkPointCache.get((Object)checkpointId, key -> {
            Savepoint savepoint = this.savepointService.getLatest(appId);
            return Optional.ofNullable(savepoint).map(Savepoint::getChkId).orElse(null);
        });
    }

    private boolean shouldProcessFailedTrigger(CheckPoints.CheckPoint checkPoint, boolean cpFailedTrigger, CheckPointStatus status) {
        return CheckPointStatus.FAILED.equals(status) && checkPoint.getIsSavepoint() == false && cpFailedTrigger;
    }

    private void saveSavepoint(CheckPoints.CheckPoint checkPoint, Long appId) {
        Savepoint savepoint = new Savepoint();
        savepoint.setAppId(appId);
        savepoint.setChkId(checkPoint.getId());
        savepoint.setLatest(true);
        savepoint.setType(checkPoint.getCheckPointType().get());
        savepoint.setPath(checkPoint.getExternalPath());
        savepoint.setTriggerTime(new Date(checkPoint.getTriggerTimestamp()));
        savepoint.setCreateTime(new Date());
        this.savepointService.saveSavePoint(savepoint);
    }

    public static class CheckPointKey {
        private Long appId;
        private String jobId;
        private Long checkId;

        public CheckPointKey(Long appId, String jobId, Long checkId) {
            this.appId = appId;
            this.jobId = jobId;
            this.checkId = checkId;
        }

        public String getSavePointId() {
            return String.format("%s_%s_%s", this.appId, this.jobId, this.checkId);
        }

        public String getCheckPointId() {
            return String.format("%s_%s", this.appId, this.jobId);
        }

        public Long getAppId() {
            return this.appId;
        }

        public String getJobId() {
            return this.jobId;
        }

        public Long getCheckId() {
            return this.checkId;
        }

        public void setAppId(Long appId) {
            this.appId = appId;
        }

        public void setJobId(String jobId) {
            this.jobId = jobId;
        }

        public void setCheckId(Long checkId) {
            this.checkId = checkId;
        }
    }

    public static class Counter {
        private final Long timestamp;
        private final AtomicInteger count;

        public Counter(Long timestamp) {
            this.timestamp = timestamp;
            this.count = new AtomicInteger(1);
        }

        public void increment() {
            this.count.incrementAndGet();
        }

        public Integer getCount() {
            return this.count.get();
        }

        public long getDuration(Long currentTimestamp) {
            return (currentTimestamp - this.timestamp) / 1000L / 60L;
        }
    }
}

