/*
 *  This file is part of RawTherapee.
 *
 *  Copyright (c) 2004-2010 Gabor Horvath <hgabor@rawtherapee.com>
 *
 *  RawTherapee is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  RawTherapee is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with RawTherapee.  If not, see <https://www.gnu.org/licenses/>.
 */

#include <glibmm/ustring.h>

#include "colortemp.h"
#include "iccmatrices.h"
#include "rt_math.h"
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include "sleef.h"
#include "settings.h"
#include "iccstore.h"


namespace rtengine
{

static const double cie_colour_match_jd2[97][3] = {//350nm to 830nm   5 nm J.Desmis 2° Standard Observer.
    {0.0000000, 0.000000, 0.000000}, {0.0000000, 0.000000, 0.000000}, {0.0001299, 0.0003917, 0.0006061},
    {0.0002321, 0.000006965, 0.001086}, {0.0004149, 0.00001239, 0.001946}, {0.0007416, 0.00002202, 0.003846},
    {0.001368, 0.000039, 0.006450001}, {0.002236, 0.000064, 0.01054999}, {0.004243, 0.000120, 0.02005001},
    {0.007650, 0.000217, 0.036210}, {0.014310, 0.000396, 0.06785001}, {0.023190, 0.000640, 0.110200},
    {0.043510, 0.001210, 0.207400}, {0.077630, 0.002180, 0.371300}, {0.134380, 0.004000, 0.645600},
    {0.214770, 0.007300, 1.0390501}, {0.283900, 0.011600, 1.385600}, {0.328500, 0.016840, 1.622960},
    {0.348280, 0.023000, 1.747060}, {0.348060, 0.029800, 1.782600}, {0.336200, 0.038000, 1.772110},
    {0.318700, 0.048000, 1.744100}, {0.290800, 0.060000, 1.669200}, {0.251100, 0.073900, 1.528100},
    {0.195360, 0.090980, 1.287640}, {0.142100, 0.112600, 1.041900}, {0.095640, 0.139020, 0.8129501},
    {0.05795001, 0.169300, 0.616200}, {0.032010, 0.208020, 0.465180}, {0.014700, 0.258600, 0.353300},
    {0.004900, 0.323000, 0.272000}, {0.002400, 0.407300, 0.212300}, {0.009300, 0.503000, 0.158200},
    {0.029100, 0.608200, 0.111700}, {0.063270, 0.710000, 0.07824999}, {0.109600, 0.793200, 0.05725001},
    {0.165500, 0.862000, 0.042160}, {0.2257499, 0.9148501, 0.029840}, {0.290400, 0.954000, 0.020300},
    {0.359700, 0.980300, 0.013400}, {0.43344990, 0.9949501, 0.008749999}, {0.5120501, 1.000000, 0.005749999},
    {0.594500, 0.995000, 0.003900}, {0.678400, 0.978600, 0.002749999}, {0.762100, 0.952000, 0.002100},
    {0.842500, 0.915400, 0.001800}, {0.916300, 0.870000, 0.001650001}, {0.978600, 0.816300, 0.001400},
    {1.026300, 0.757000, 0.001100}, {1.056700, 0.694900, 0.001000}, {1.062200, 0.631000, 0.000800},
    {1.045600, 0.566800, 0.000600}, {1.002600, 0.503000, 0.000340}, {0.938400, 0.441200, 0.000240},
    {0.8544499, 0.381000, 0.000190}, {0.751400, 0.321000, 0.000100}, {0.642400, 0.265000, 0.00004999999},
    {0.541900, 0.217000, 0.000030}, {0.447900, 0.175000, 0.000020}, {0.360800, 0.138200, 0.000010},
    {0.283500, 0.107000, 0.000000}, {0.218700, 0.081600, 0.000000}, {0.164900, 0.061000, 0.000000},
    {0.121200, 0.044580, 0.000000}, {0.087400, 0.032000, 0.000000}, {0.063600, 0.023200, 0.000000},
    {0.046770, 0.017000, 0.000000}, {0.032900, 0.011920, 0.000000}, {0.022700, 0.008210, 0.000000},
    {0.015840, 0.005723, 0.000000}, {0.01135916, 0.004102, 0.000000}, {0.008110916, 0.002929, 0.000000},
    {0.005790346, 0.002091, 0.000000}, {0.004109457, 0.001484, 0.000000}, {0.002899327, 0.001047, 0.000000},
    {0.00204919, 0.000740, 0.000000}, {0.001439971, 0.000520, 0.000000}, {0.0009999493, 0.0003611, 0.000000},
    {0.0006900786, 0.0002492, 0.000000}, {0.0004760213, 0.0001719, 0.000000}, {0.0003323011, 0.000120, 0.000000},
    {0.0002348261, 0.0000848, 0.000000}, {0.0001661505, 0.000060, 0.000000}, {0.000117413, 0.0000424, 0.000000},
    {0.00008307527, 0.000030, 0.000000}, {0.00005870652, 0.0000212, 0.000000}, {0.00004150994, 0.00001499, 0.000000},
    {0.00002935326, 0.0000106, 0.000000}, {0.00002067383, 0.0000074657, 0.000000}, {0.00001455977, 0.0000052578, 0.000000},
    {0.00001025398, 0.0000037029, 0.000000}, {0.000007221456, 0.00000260778, 0.000000}, {0.000005085868, 0.0000018366, 0.000000},
    {0.000003581652, 0.0000012934, 0.000000}, {0.000002522525, 0.00000091093, 0.000000}, {0.000001776509, 0.00000064153, 0.000000},
    {0.000001251141, 0.00000045181, 0.000000}
};


static double cie_colour_match_jd[97][3] = {//350nm to 830nm   5 nm J.Desmis 10° Standard Observer.
{0.000000000000, 0.000000000000, 0.000000000000},
{0.000000000000, 0.000000000000, 0.000000000000},
{0.000000122200, 0.000000013398, 0.000000535027},
{0.000000919270, 0.000000100650, 0.000004028300},
{0.000005958600, 0.000000651100, 0.000026143700},
{0.000033266000, 0.000003625000, 0.000146220000},
{0.000159952000, 0.000017364000, 0.000704776000},
{0.000662440000, 0.000071560000, 0.002927800000},
{0.002361600000, 0.000253400000, 0.010482200000},
{0.007242300000, 0.000768500000, 0.032344000000},
{0.019109700000, 0.002004400000, 0.086010900000},
{0.043400000000, 0.004509000000, 0.197120000000},
{0.084736000000, 0.008756000000, 0.389366000000},
{0.140638000000, 0.014456000000, 0.656760000000},
{0.204492000000, 0.021391000000, 0.972542000000},
{0.264737000000, 0.029497000000, 1.282500000000},
{0.314679000000, 0.038676000000, 1.553480000000},
{0.357719000000, 0.049602000000, 1.798500000000},
{0.383734000000, 0.062077000000, 1.967280000000},
{0.386726000000, 0.074704000000, 2.027300000000},
{0.370702000000, 0.089456000000, 1.994800000000},
{0.342957000000, 0.106256000000, 1.900700000000},
{0.302273000000, 0.128201000000, 1.745370000000},
{0.254085000000, 0.152761000000, 1.554900000000},
{0.195618000000, 0.185190000000, 1.317560000000},
{0.132349000000, 0.219940000000, 1.030200000000},
{0.080507000000, 0.253589000000, 0.772125000000},
{0.041072000000, 0.297665000000, 0.570060000000},
{0.016172000000, 0.339133000000, 0.415254000000},
{0.005132000000, 0.395379000000, 0.302356000000},
{0.003816000000, 0.460777000000, 0.218502000000},
{0.015444000000, 0.531360000000, 0.159249000000},
{0.037465000000, 0.606741000000, 0.112044000000},
{0.071358000000, 0.685660000000, 0.082248000000},
{0.117749000000, 0.761757000000, 0.060709000000},
{0.172953000000, 0.823330000000, 0.043050000000},
{0.236491000000, 0.875211000000, 0.030451000000},
{0.304213000000, 0.923810000000, 0.020584000000},
{0.376772000000, 0.961988000000, 0.013676000000},
{0.451584000000, 0.982200000000, 0.007918000000},
{0.529826000000, 0.991761000000, 0.003988000000},
{0.616053000000, 0.999110000000, 0.001091000000},
{0.793832000000, 0.982380000000, 0.000000000000},
{0.878655000000, 0.955552000000, 0.000000000000},
{0.951162000000, 0.915175000000, 0.000000000000},
{1.014160000000, 0.868934000000, 0.000000000000},
{1.074300000000, 0.825623000000, 0.000000000000},
{1.118520000000, 0.777405000000, 0.000000000000},
{1.134300000000, 0.720353000000, 0.000000000000},
{1.123990000000, 0.658341000000, 0.000000000000},
{1.089100000000, 0.593878000000, 0.000000000000},
{1.030480000000, 0.527963000000, 0.000000000000},
{0.950740000000, 0.461834000000, 0.000000000000},
{0.856297000000, 0.398057000000, 0.000000000000},
{0.754930000000, 0.339554000000, 0.000000000000},
{0.647467000000, 0.283493000000, 0.000000000000},
{0.535110000000, 0.228254000000, 0.000000000000},
{0.431567000000, 0.179828000000, 0.000000000000},
{0.343690000000, 0.140211000000, 0.000000000000},
{0.268329000000, 0.107633000000, 0.000000000000},
{0.204300000000, 0.081187000000, 0.000000000000},
{0.152568000000, 0.060281000000, 0.000000000000},
{0.112210000000, 0.044096000000, 0.000000000000},
{0.081260600000, 0.031800400000, 0.000000000000},
{0.057930000000, 0.022601700000, 0.000000000000},
{0.040850800000, 0.015905100000, 0.000000000000},
{0.028623000000, 0.011130300000, 0.000000000000},
{0.019941300000, 0.007748800000, 0.000000000000},
{0.013842000000, 0.005375100000, 0.000000000000},
{0.009576880000, 0.003717740000, 0.000000000000},
{0.006605200000, 0.002564560000, 0.000000000000},
{0.004552630000, 0.001768470000, 0.000000000000},
{0.003144700000, 0.001222390000, 0.000000000000},
{0.002174960000, 0.000846190000, 0.000000000000},
{0.001505700000, 0.000586440000, 0.000000000000},
{0.001044760000, 0.000407410000, 0.000000000000},
{0.000727450000, 0.000284041000, 0.000000000000},
{0.000508258000, 0.000198730000, 0.000000000000},
{0.000356380000, 0.000139550000, 0.000000000000},
{0.000250969000, 0.000098428000, 0.000000000000},
{0.000177730000, 0.000069819000, 0.000000000000},
{0.000126390000, 0.000049737000, 0.000000000000},
{0.000090151000, 0.000035540500, 0.000000000000},
{0.000064525800, 0.000025486000, 0.000000000000},
{0.000046339000, 0.000018338400, 0.000000000000},
{0.000033411700, 0.000013249000, 0.000000000000},
{0.000024209000, 0.000009619600, 0.000000000000},
{0.000017611500, 0.000007012800, 0.000000000000},
{0.000012855000, 0.000005129800, 0.000000000000},
{0.000009413630, 0.000003764730, 0.000000000000},
{0.000006913000, 0.000002770810, 0.000000000000},
{0.000005093470, 0.000002046130, 0.000000000000},
{0.000003767100, 0.000001516770, 0.000000000000},
{0.000002795310, 0.000001128090, 0.000000000000},
{0.000002082000, 0.000000842160, 0.000000000000},
{0.000001553140, 0.000000629700, 0.000000000000}
};



ColorTemp::ColorTemp (double t, double g, double e, const std::string &m) : temp(t), green(g), equal(e), method(m)
{
    clip (temp, green, equal);
}

void ColorTemp::clip (double &temp, double &green)
{
    temp = rtengine::LIM(temp, MINTEMP, MAXTEMP);
    green = rtengine::LIM(green, MINGREEN, MAXGREEN);
}

void ColorTemp::clip (double &temp, double &green, double &equal)
{
    temp = rtengine::LIM(temp, MINTEMP, MAXTEMP);
    green = rtengine::LIM(green, MINGREEN, MAXGREEN);
    equal = rtengine::LIM(equal, MINEQUAL, MAXEQUAL);
}

ColorTemp::ColorTemp (double mulr, double mulg, double mulb, double e) : equal(e), method("Custom")
{
    mul2temp (mulr, mulg, mulb, equal, temp, green);
}

void ColorTemp::mul2temp (const double rmul, const double gmul, const double bmul, const double equal, double& temp, double& green) const
{

    double maxtemp = MAXTEMP, mintemp = MINTEMP;
    double tmpr, tmpg, tmpb;
    temp = (maxtemp + mintemp) / 2;

    while (maxtemp - mintemp > 1) {
        temp2mul (temp, 1.0, equal, tmpr, tmpg, tmpb);

        if (tmpb / tmpr > bmul / rmul) {
            maxtemp = temp;
        } else {
            mintemp = temp;
        }

        temp = (maxtemp + mintemp) / 2;
    }

    green = (tmpg / tmpr) / (gmul / rmul);
    clip (temp, green);
}


// spectral data for Daylight direct Sun: I have choose 5300K because Nikon=5200K, Olympus=5300K, Panasonic=5500K, Leica=5400K, Minolta=5100K
const double ColorTemp::Daylight5300_spect[97] = {
    24.82, 26.27, 27.72, 28.97, 30.22, 29.71, 29.19, 31.95, 34.71, 45.49, 56.26, 59.97, 63.68, 65.30, 66.92, 65.39, 63.86, 72.59, 81.32, 87.53, 93.73, 95.15, 96.56, 96.55, 96.54, 98.13, 99.73, 97.70, 95.66, 97.19, 98.72,
    98.90, 99.08, 98.98, 98.87, 101.13, 103.39, 102.48, 101.57, 102.14, 102.71, 101.36, 100.00, 98.71, 97.42, 97.81, 98.21, 95.20, 92.20, 93.92, 95.63, 96.15, 96.67, 96.34, 96.01, 94.21, 92.41, 93.58, 94.74, 93.05, 91.36, 92.29,
    93.21, 95.25, 97.28, 95.30, 93.32, 87.92, 82.51, 84.29, 86.06, 86.94, 87.81, 80.24, 72.68, 77.32, 81.96, 84.88, 87.79, 81.01, 74.22, 64.41, 54.60, 66.55, 78.51, 76.35, 74.20, 74.79, 75.38, 72.48, 69.58, 65.11, 60.64,
    63.88, 67.13, 68.85, 70.57
};

//spectral data for Daylight Cloudy: I have choose 6200K because Nikon=6000K, Olympus=6000K, Panasonic=6200K, Leica=6400K, Minolta=6500K
const double ColorTemp::Cloudy6200_spect[97] = {
    39.50, 40.57, 41.63, 43.85, 46.08, 45.38, 44.69, 47.20, 49.71, 63.06, 76.41, 80.59, 84.77, 85.91, 87.05, 84.14, 81.23, 90.29, 99.35, 105.47, 111.58, 112.23, 112.87, 111.74, 110.62, 111.41, 112.20, 108.98, 105.76, 106.32,
    106.89, 106.34, 105.79, 104.62, 103.45, 105.09, 106.72, 105.24, 103.76, 103.75, 103.75, 101.87, 100.00, 98.29, 96.58, 96.46, 96.34, 92.85, 89.37, 90.25, 91.12, 91.06, 90.99, 90.17, 89.35, 87.22, 85.10, 85.48, 85.85,
    84.03, 82.20, 82.45, 82.69, 83.92, 85.15, 83.14, 81.13, 76.65, 72.17, 73.27, 74.36, 75.65, 76.95, 70.34, 63.74, 67.98, 72.22, 74.88, 77.54, 71.59, 65.65, 56.82, 47.99, 58.53, 69.06, 67.27, 65.47, 65.96, 66.44, 63.92, 61.41, 57.52,
    53.63, 56.47, 59.31, 60.80, 62.29
};

//spectral data for Daylight Shade: I have choose 7600K because Nikon=8000K, Olympus=7500K, Panasonic=7500K, Leica=7500K, Minolta=7500K
const double ColorTemp::Shade7600_spect[97] = {
    64.42, 64.46, 64.51, 68.35, 72.20, 70.22, 68.24, 69.79, 71.35, 87.49, 103.64, 108.68, 113.72, 114.12, 114.53, 109.54, 104.55, 113.59, 122.63, 128.52, 134.41, 134.02, 133.63, 131.02, 128.41, 128.08, 127.75, 123.16,
    118.57, 117.89, 117.22, 115.72, 114.22, 111.60, 108.99, 109.84, 110.68, 108.57, 106.45, 105.71, 104.98, 102.49, 100.00, 97.78, 95.55, 94.82, 94.08, 90.47, 86.87, 86.94, 87.01, 86.45, 85.88, 84.57, 83.27, 80.83, 78.40, 78.21,
    78.03, 76.22, 74.42, 74.15, 73.89, 74.41, 74.92, 73.01, 71.09, 67.26, 63.42, 64.01, 64.60, 66.10, 67.60, 61.83, 56.06, 59.94, 63.82, 66.27, 68.71, 63.49, 58.26, 50.30, 42.34, 51.64, 60.95, 59.45, 57.95, 58.35, 58.76, 56.57,
    54.38, 51.00, 47.62, 50.10, 52.58, 53.88, 55.19
};

//spectral data for tungsten - incandescent 2856K
const double ColorTemp::A2856_spect[97] = {
    4.75, 5.42, 6.15, 6.95, 7.83, 8.78, 9.80, 10.91, 12.09, 13.36, 14.72, 16.16, 17.69, 19.30, 21.01, 22.80, 24.68, 26.65, 28.71, 30.86, 33.10, 35.42, 37.82, 40.31, 42.88, 45.53, 48.25, 51.05, 53.92, 56.87, 59.87, 62.94, 66.07, 69.26, 72.50,
    75.80, 79.14, 82.53, 85.95, 89.42, 92.91, 96.44, 100.00, 103.58, 107.18, 110.80, 114.43, 118.07, 121.72, 125.38, 129.03, 132.68, 136.33, 139.97, 143.60, 147.21, 150.81, 154.39, 157.95, 161.48, 164.99, 168.47, 171.92, 175.34,
    178.72, 182.07, 185.38, 188.65, 191.88, 195.06, 198.20, 201.30, 204.34, 207.34, 210.29, 213.19, 216.04, 218.84, 221.58, 224.28, 226.91, 229.49, 232.02, 234.49, 236.91, 239.27, 241.57, 243.82, 246.01, 248.14, 250.21, 252.23, 254.19,
    256.10, 257.95, 259.74, 261.47
};

//spectral data for fluo  F1 Daylight 6430K
const double ColorTemp::FluoF1_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.87, 2.36, 2.94, 3.47, 5.17, 19.49, 6.13, 6.24, 7.01, 7.79, 8.56, 43.67, 16.94, 10.72, 11.35, 11.89, 12.37, 12.75, 13.00, 13.15, 13.23, 13.17, 13.13, 12.85, 12.52,
    12.20, 11.83, 11.50, 11.22, 11.05, 11.03, 11.18, 11.53, 27.74, 17.05, 13.55, 14.33, 15.01, 15.52, 18.29, 19.55, 15.48, 14.91, 14.15, 13.22, 12.19, 11.12, 10.03, 8.95, 7.96, 7.02, 6.20, 5.42, 4.73, 4.15, 3.64, 3.20, 2.81,
    2.47, 2.18, 1.93, 1.72, 1.67, 1.43, 1.29, 1.19, 1.08, 0.96, 0.88, 0.81, 0.77, 0.75, 0.73, 0.68, 0.69, 0.64, 0.68, 0.69, 0.61, 0.52, 0.43, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data for fluo  F2 Cool white 4230K
const double ColorTemp::FluoF2_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.18, 1.48, 1.84, 2.15, 3.44, 15.69, 3.85, 3.74, 4.19, 4.62, 5.06, 34.98, 11.81, 6.27, 6.63, 6.93, 7.19, 7.40, 7.54, 7.62, 7.65, 7.62, 7.62, 7.45, 7.28, 7.15, 7.05, 7.04, 7.16, 7.47, 8.04, 8.88, 10.01, 24.88, 16.64, 14.59, 16.16, 17.60, 18.62, 21.47, 22.79, 19.29, 18.66, 17.73, 16.54, 15.21, 13.80, 12.36, 10.95, 9.65, 8.40, 7.32, 6.31, 5.43, 4.68, 4.02, 3.45,
    2.96, 2.55, 2.19, 1.89, 1.64, 1.53, 1.27, 1.10, 0.99, 0.88, 0.76, 0.68, 0.61, 0.56, 0.54, 0.51, 0.47, 0.47, 0.43, 0.46, 0.47, 0.40, 0.33, 0.27, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data for fluo  F3 White 3450K
const double ColorTemp::FluoF3_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.82, 1.02, 1.26, 1.44, 2.57, 14.36, 2.70, 2.45, 2.73, 3.00, 3.28, 31.85, 9.47, 4.02, 4.25, 4.44, 4.59, 4.72, 4.80, 4.86, 4.87, 4.85, 4.88, 4.77, 4.67, 4.62, 4.62, 4.73, 4.99, 5.48, 6.25,
    7.34, 8.78, 23.82, 16.14, 14.59, 16.63, 18.49, 19.95, 23.11, 24.69, 21.41, 20.85, 19.93, 18.67, 17.22, 15.65, 14.04, 12.45, 10.95, 9.51, 8.27, 7.11, 6.09, 5.22, 4.45, 3.80, 3.23, 2.75, 2.33, 1.99, 1.70, 1.55,
    1.27, 1.09, 0.96, 0.83, 0.71, 0.62, 0.54, 0.49, 0.46, 0.43, 0.39, 0.39, 0.35, 0.38, 0.39, 0.33, 0.28, 0.21, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data for fluo F4 Warm white 2940K
const double ColorTemp::FluoF4_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.57, 0.70, 0.87, 0.98, 2.01, 13.75, 1.95, 1.59, 1.76, 1.93, 2.10, 30.28, 8.03, 2.55, 2.70, 2.82, 2.91, 2.99, 3.04, 3.08, 3.09, 3.09, 3.14, 3.06, 3.00, 2.98, 3.01,
    3.14, 3.41, 3.90, 4.69, 5.81, 7.32, 22.59, 15.11, 13.88, 16.33, 18.68, 20.64, 24.28, 26.26, 23.28, 22.94, 22.14, 20.91, 19.43, 17.74, 16.00, 14.42, 12.56, 10.93, 9.52, 8.18, 7.01, 6.00, 5.11, 4.36, 3.69, 3.13, 2.64,
    2.24, 1.91, 1.70, 1.39, 1.18, 1.03, 0.88, 0.74, 0.64, 0.54, 0.49, 0.46, 0.42, 0.37, 0.37, 0.33, 0.35, 0.36, 0.31, 0.26, 0.19, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data for fluo F5 Daylight 6350K
const double ColorTemp::FluoF5_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.87, 2.35, 2.92, 3.45, 5.10, 18.91, 6.00, 6.11, 6.85, 7.58, 8.31, 40.76, 16.06, 10.32, 10.91, 11.40, 11.83, 12.17, 12.40, 12.54, 12.58, 12.52, 12.47, 12.20, 11.89,
    11.61, 11.33, 11.10, 10.96, 10.97, 11.16, 11.54, 12.12, 27.78, 17.73, 14.47, 15.20, 15.77, 16.10, 18.54, 19.50, 15.39, 14.64, 13.72, 12.69, 11.57, 10.45, 9.35, 8.29, 7.32, 6.41, 5.63, 4.90, 4.26,
    3.72, 3.25, 2.83, 2.49, 2.19, 1.93, 1.71, 1.52, 1.43, 1.26, 1.13, 1.05, 0.96, 0.85, 0.78, 0.72, 0.68, 0.67, 0.65, 0.61, 0.62, 0.59, 0.62, 0.64, 0.55, 0.47, 0.40, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data for fluo F6 Lite white 4150K
const double ColorTemp::FluoF6_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.05, 1.31, 1.63, 1.90, 3.11, 14.8, 3.43, 3.30, 3.68, 4.07, 4.45, 32.61, 10.74, 5.48, 5.78, 6.03, 6.25, 6.41, 6.52, 6.58, 6.59, 6.56, 6.56, 6.42, 6.28, 6.20, 6.19, 6.30, 6.60, 7.12, 7.94, 9.07, 10.49, 25.22, 17.46, 15.63, 17.22, 18.53,
    19.43, 21.97, 23.01, 19.41, 18.56, 17.42, 16.09, 14.64, 13.15, 11.68, 10.25, 8.96, 7.74, 6.69, 5.71, 4.87, 4.16, 3.55, 3.02, 2.57, 2.20, 1.87, 1.60, 1.37, 1.29, 1.05, 0.91, 0.81, 0.71, 0.61, 0.54, 0.48, 0.44,
    0.43, 0.40, 0.37, 0.38, 0.35, 0.39, 0.41, 0.33, 0.26, 0.21, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data for fluo F7 D65 Daylight simulator 6500K
const double ColorTemp::FluoF7_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.56, 3.18, 3.84, 4.53, 6.15, 19.37, 7.37, 7.05, 7.71, 8.41, 9.15, 44.14, 17.52, 11.35, 12.00, 12.58, 13.08, 13.45, 13.71, 13.88, 13.95, 13.93, 13.82, 13.64, 13.43, 13.25, 13.08, 12.93, 12.78, 12.60,
    12.44, 12.33, 12.26, 29.52, 17.05, 12.44, 12.58, 12.72, 12.83, 15.46, 16.75, 12.83, 12.67, 12.43, 12.19, 11.89, 11.60, 11.35, 11.12, 10.95, 10.76, 10.42, 10.11, 10.04, 10.02, 10.11, 9.87, 8.65, 7.27, 6.44, 5.83, 5.41,
    5.04, 4.57, 4.12, 3.77, 3.46, 3.08, 2.73, 2.47, 2.25, 2.06, 1.90, 1.75, 1.62, 1.54, 1.45, 1.32, 1.17, 0.99, 0.81, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data for fluo F8 D50 simulator Sylvania F40 Design 5000K
const double ColorTemp::FluoF8_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.21, 1.5, 1.81, 2.13, 3.17, 13.08, 3.83, 3.45, 3.86, 4.42, 5.09, 34.1, 12.42, 7.68, 8.60, 9.46, 10.24, 10.84, 11.33, 11.71, 11.98, 12.17, 12.28, 12.32, 12.35, 12.44, 12.55, 12.68, 12.77, 12.72,
    12.60, 12.43, 12.22, 28.96, 16.51, 11.79, 11.76, 11.77, 11.84, 14.61, 16.11, 12.34, 13.61, 13.87, 14.07, 14.20, 14.16, 14.13, 14.34, 14.50, 14.46, 14.00, 12.58, 10.99, 9.98, 9.22, 8.62, 8.07, 7.39, 6.71, 6.16, 5.63, 5.03, 4.46, 4.02, 3.66,
    3.36, 3.09, 2.85, 2.65, 2.51, 2.37, 2.15, 1.89, 1.61, 1.32, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data for fluo F9 Cool white deluxe 4150K
const double ColorTemp::FluoF9_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.9, 1.12, 1.36, 1.60, 2.59, 12.8, 3.05, 2.56, 2.86, 3.30, 3.82, 32.62, 10.77, 5.84, 6.57, 7.25, 7.86, 8.35, 8.75, 9.06, 9.31, 9.48, 9.61, 9.68, 10.04, 10.26, 10.48, 10.63, 10.76, 10.96,
    11.18, 27.71, 16.29, 12.28, 12.74, 13.21, 13.65, 16.57, 18.14, 14.55, 14.65, 14.66, 14.61, 14.50, 14.39, 14.40, 14.47, 14.62, 14.72, 14.55, 14.4, 14.58, 14.88, 15.51, 15.47, 13.20, 10.57, 9.18, 8.25, 7.57, 7.03,
    6.35, 5.72, 5.25, 4.80, 4.29, 3.80, 3.43, 3.12, 2.86, 2.64, 2.43, 2.26, 2.14, 2.02, 1.83, 1.61, 1.38, 1.12, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data for fluo F10 Philips TL85 - 5000K
const double ColorTemp::FluoF10_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.11, 0.63, 0.62, 0.57, 1.48, 12.16, 2.12, 2.70, 3.74, 5.14, 6.75, 34.39, 14.86, 10.4, 10.76, 10.11, 9.27, 8.29, 7.29, 7.91, 16.64, 16.73, 10.44, 5.94, 3.34, 2.35, 1.88, 1.59, 1.47,
    1.80, 5.71, 40.98, 73.69, 33.61, 8.24, 3.38, 2.47, 4.86, 11.45, 14.79, 12.16, 8.97, 6.52, 8.81, 44.12, 34.55, 12.09, 12.15, 10.52, 4.43, 1.95, 2.19, 3.19, 2.77, 2.29, 2.00, 1.52, 1.35, 1.47, 1.79, 1.74, 1.02, 1.14,
    3.32, 4.49, 2.05, 0.49, 0.24, 0.21, 0.21, 0.24, 0.24, 0.21, 0.17, 0.21, 0.22, 0.17, 0.12, 0.09, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data for fluo F11 Philips TL84 4150K
const double ColorTemp::FluoF11_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.91, 0.63, 0.46, 0.37, 1.29, 12.68, 1.59, 1.79, 2.46, 3.38, 4.49, 33.94, 12.13, 6.95, 7.19, 7.12, 6.72, 6.13, 5.46, 4.79, 5.66, 14.29, 14.96, 8.97, 4.72, 2.33, 1.47, 1.10, 0.89, 0.83, 1.18, 4.90, 39.59,
    72.84, 32.61, 7.52, 2.83, 1.96, 1.67, 4.43, 11.28, 14.76, 12.73, 9.74, 7.33, 9.72, 55.27, 42.58, 13.18, 13.16, 12.26, 5.11, 2.07, 2.34, 3.58, 3.01, 2.48, 2.14, 1.54, 1.33, 1.46, 1.94, 2.00, 1.20, 1.35, 4.10, 5.58,
    2.51, 0.57, 0.27, 0.23, 0.21, 0.24, 0.24, 0.20, 0.24, 0.32, 0.26, 0.16, 0.12, 0.09, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data for fluo F12 Philips TL83 3000K
const double ColorTemp::FluoF12_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.96, 0.64, 0.45, 0.33, 1.19, 12.48, 1.12, 0.94, 1.08, 1.37, 1.78, 29.05, 7.90, 2.65, 2.71, 2.65, 2.49, 2.33, 2.10, 1.91, 3.01, 10.83, 11.88, 6.88, 3.43, 1.49, 0.92, 0.71, 0.60, 0.63, 1.10, 4.56, 34.4, 65.40, 29.48,
    7.16, 3.08, 2.47, 2.27, 5.09, 11.96, 15.32, 14.27, 11.86, 9.28, 12.31, 68.53, 53.02, 14.67, 14.38, 14.71, 6.46, 2.57, 2.75, 4.18, 3.44, 2.81, 2.42, 1.64, 1.36, 1.49, 2.14, 2.34, 1.42, 1.61, 5.04, 6.98, 3.19, 0.71, 0.30, 0.26, 0.23, 0.28, 0.28, 0.21,
    0.17, 0.21, 0.19, 0.15, 0.10, 0.05, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data for HMI lamp studio "Osram" 4800K (for film, spectacle, studio...)
const double ColorTemp::HMI_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 9.66, 11.45, 13.24, 14.93, 16.63, 17.90, 19.20, 20.12, 21.03, 23.84, 26.65, 26.38, 26.12, 26.52, 27.92, 31.15, 34.37, 34.98, 35.61, 35.71, 35.81, 34.90, 34.02, 34.06, 34.08, 34.68, 35.28, 34.72, 34.20, 33.63,
    33.05, 34.70, 36.35, 38.01, 39.48, 37.29, 35.10, 36.22, 37.28, 38.76, 40.24, 39.56, 38.90, 39.35, 39.79, 38.90, 38.01, 38.05, 38.10, 37.45, 36.64, 35.82, 35.00, 35.06, 33.13, 33.85, 34.55, 35.26, 35.77, 34.92,
    34.09, 33.40, 32.72, 32.08, 31.45, 26.83, 22.23, 21.50, 20.79, 21.41, 22.03, 11.01, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data for GTI lamp : Graphiclite & ColorMatch for Photography 5000K
const double ColorTemp::GTI_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.26, 4.71, 6.17, 12.71, 19.27, 20.53, 21.80, 19.15, 16.53, 28.25, 39.97, 48.52, 57.06, 43.66, 30.27, 30.22, 30.16, 31.48, 32.98, 34.01, 35.04, 35.83, 36.62, 37.12, 37.62, 37.99, 38.19, 38.29, 38.48,
    38.82, 39.16, 45.40, 51.63, 51.83, 62.04, 52.41, 42.80, 42.95, 43.09, 45.64, 48.20, 46.23, 44.27, 43.74, 43.22, 43.30, 43.41, 43.10, 42.78, 42.03, 41.29, 40.29, 39.29, 37.89, 36.58, 34.92, 33.27, 31.47, 29.68, 27.90,
    26.13, 24.55, 22.98, 21.42, 19.86, 18.40, 16.92, 14.46, 13.99, 12.36, 11.73, 5.86, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data for JudgeIII Lamp D50
const double ColorTemp::JudgeIII_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 4.08, 4.25, 4.43, 6.90, 9.40, 9.75, 10.11, 9.30, 8.54, 14.90, 21.16, 26.01, 30.83, 24.90, 19.00, 19.00, 19.00, 19.56, 20.13, 20.28, 20.44, 20.64, 20.85, 21.05, 21.24, 21.65, 22.11, 22.85, 23.58, 24.00, 24.43,
    27.75, 31.27, 33.90, 36.59, 30.90, 25.32, 25.05, 24.76, 26.03, 27.31, 25.90, 24.48, 23.85, 23.29, 23.10, 22.94, 23.24, 23.53, 24.02, 24.52, 23.80, 23.13, 24.51, 25.76, 27.90, 29.15, 24.70, 20.25, 16.60, 12.98, 11.63, 10.27, 9.30, 8.34,
    7.60, 6.91, 6.25, 5.67, 5.15, 4.68, 2.34, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data for Solux lamp : 3500K
const double ColorTemp::Solux3500_spect[97] = {
    0.5268, 0.93, 1.3278, 1.51, 1.6987, 2.65, 3.6100, 3.80, 3.9927, 6.08, 8.1680, 11.02, 13.863, 15.66, 17.4600, 18.78, 20.130, 21.43, 22.749, 24.02, 25.290, 27.40, 29.504, 31.77, 34.031, 36.35, 38.672, 40.55, 42.426, 44.15, 45.865, 47.37, 48.879,
    49.71, 50.531, 51.2, 51.872, 51.9, 51.928, 52.97, 54.015, 55.93, 57.846, 60.25, 62.650, 64.36, 66.065, 66.72, 67.369, 68.81, 70.260, 71.37, 72.487, 72.53, 72.578, 72.51, 72.447, 72.46, 72.471,
    72.76, 73.047, 74.25, 75.449, 76.5, 77.543, 78.79, 80.040, 80.72, 81.394, 82.12, 82.840, 83.23, 83.614, 83.36, 83.100, 82.36, 81.615, 80.11, 78.606, 75.91, 73.221, 69.61, 66.006, 62.43, 58.844, 56.07, 53.292,
    51.07, 48.839, 46.93, 45.013, 43.54, 42.070, 40.61, 39.150, 37.79, 36.425
};

//spectral data for Solux lamp : 4100K
const double ColorTemp::Solux4100_spect[97] = {
    0.5717, 0.70, 0.8286, 1.16, 1.522, 2.01, 2.384, 3.45, 4.57, 6.46, 8.4548, 11.31, 14.205, 16.10, 17.949, 19.51, 21.068, 22.60, 24.197, 25.37, 26.566, 28.15, 29.742, 30.90, 32.060, 33.26, 34.481, 34.80, 35.130, 35.42, 35.697, 36.20, 36.763,
    37.90, 39.004, 40.26, 41.494, 43.10, 44.690, 45.80, 46.900, 47.45, 47.885, 47.75, 47.635, 47.00, 46.410, 46.22, 46.058, 46.70, 47.344, 48.65, 50.005, 51.02, 52.045, 53.55, 55.075, 55.98, 56.823,
    56.85, 56.884, 56.15, 55.523, 54.60, 53.732, 52.55, 51.425, 50.30, 49.1830, 48.76, 48.273, 48.22, 48.169, 49.92, 49.915, 51.90, 53.099, 54.95, 56.852, 58.45, 60.090, 61.67, 63.2530, 63.55, 63.834, 63.55, 63.468,
    62.40, 61.373, 59.75, 58.1810, 56.25, 54.395, 51.90, 49.496, 47.05, 44.620
};

//spectral data for Solux lamp : near Daylight (for example  "Musée d'Orsay") - 4700K
const double ColorTemp::Solux4700_spect[97] = {
    0.4590, 0.83, 1.2011, 1.53, 1.8647, 2.15, 2.5338, 3.06, 3.5809, 3.99, 4.4137, 4.82, 5.2228, 5.63, 6.0387, 6.53, 6.9944, 7.55, 8.0266, 8.475, 8.9276, 8.90, 9.7840, 10.20, 10.6390, 11.00, 11.3600, 11.75, 12.1340, 12.36, 12.5880, 12.74, 12.8790,
    13.07, 13.2560, 13.38, 13.5220, 13.41, 13.3070, 13.35, 13.3990, 13.37, 13.3420, 13.39, 13.4220, 13.65, 13.2710, 13.25, 13.2330, 13.12, 13.0110, 12.93, 12.8470, 12.805, 12.7630, 12.66, 12.5760, 12.563, 12.5490,
    12.59, 12.6330, 12.617, 12.6010, 12.616, 12.6310, 12.6275, 12.6240, 12.70, 12.7710, 12.776, 12.7810, 12.786, 12.7950, 12.74, 12.6850, 12.64, 12.5950, 12.55, 12.5420, 12.43, 12.3180, 12.07, 11.8340, 11.72, 11.6190, 11.55, 11.5020,
    11.32, 11.1510, 11.05, 10.9530, 10.80, 10.6550, 10.495, 10.4390, 10.31, 10.1790
};

//spectral data for Solux lamp : near Daylight 4400K - test National Gallery
const double ColorTemp::NG_Solux4700_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 139, 152, 170, 185, 202, 223, 240, 257, 270, 282, 293, 305, 317, 329, 342, 355, 367, 378, 387, 395, 401, 405, 408, 411, 414, 415, 416, 415, 414, 414, 416, 419, 423, 427, 432, 437, 442, 447, 452,
    456, 461, 467, 475, 483, 488, 490, 491, 490, 487, 485, 481, 477, 472, 466, 461, 455, 449, 442, 434, 427, 419, 411, 403, 395, 386, 377, 367, 359, 351, 343, 335, 327, 322, 316, 312, 306, 305, 301, 299, 299, 298,
    0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.0
};

//spectral data for LED LSI Lumelex 2040 - test National Gallery
const double ColorTemp::NG_LEDLSI2040_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.5, 1.2, 0.5, 0.7, 0.6, 1.6, 1.7, 7.0, 16.6, 35.5, 64, 106, 162.5, 230.5, 272.2, 249, 213.4, 214, 227.6, 231.9, 233, 235.2, 241.4, 253.7, 270.3, 288.5, 306.2, 322.3, 337.6, 352.5, 367.2, 381.7, 395.9, 409.6, 416.2, 423.2, 429.7, 435.8, 442.8,
    451.7, 464.2, 480.3, 501, 526.3, 555.9, 587.5, 625.4, 655.1, 681.7, 705.3, 721.5, 728.5, 729, 719.8, 702.5, 676.7, 646.2, 611.5, 571.7, 530.3, 488.3, 445.9, 404, 365.2, 326.8, 290.8, 257.6, 226.9, 199.8, 175.2, 154.2, 133.8, 116.4, 101.5, 88.5, 76.6, 67.3, 57.9, 50.7, 44.2, 38.2,
    0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.0
};

//spectral data for LED CRS SP12 WWMR16 - test National Gallery
const double ColorTemp::NG_CRSSP12WWMR16_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0., 0., 0., 0., 0., 0.14, 0.33, 1.31, 3.34, 7.9, 17.4, 36.5, 72.6, 145.4, 260.5, 359.2, 365.3, 303.1, 256.1, 221.7, 193.6, 185.8, 191.4, 207.3, 232.8, 257.5, 285.1, 310.5, 333.4, 351.5, 368.8, 383.7, 398.8, 411.6, 424.7, 435.6, 447.9, 459.7, 471.7,
    484.6, 497.9, 512.3, 531.1, 548.9, 567.9, 587.5, 608.6, 625.3, 640.1, 648.6, 654.0, 654.3, 647.2, 633.9, 616.1, 590.5, 561.5, 526.5, 494.8, 457.9, 420.8, 382.4, 347.3, 309.9, 280.5, 249.2, 220.0, 194.9, 170.8, 149.1, 130.0, 112.3, 97.5, 84.9, 73.2, 63.1, 54.1, 45.6, 39.0, 32.6, 27.4,
    0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.0
};


//Data for flash :
// in theory, should be the spectral data of each owner flash (Nikon, Canon, Sony ...) or studio flash (Profoto. ..)
// but: 1) I did not, 2) the data vary depending on the power used ... so by default, although this is not true, I chose the values "Daylight" for temp...
// CRI for flash near of 95 - 97 !!

//spectral data for Flash daylight 5500K (Leica...)
const double ColorTemp::Flash5500_spect[97] = {
    27.77, 29.17, 30.58, 32.02, 33.47, 33.00, 32.53, 35.28, 38.04, 49.46, 60.88, 64.68, 68.48, 69.99, 71.51, 69.68, 67.85, 76.70, 85.54, 91.74, 97.93, 99.17, 100.41, 100.14, 99.86, 101.28, 102.70, 100.37, 98.04, 99.35, 100.65,
    100.66, 100.67, 100.32, 99.97, 102.08, 104.20, 103.15, 102.09, 102.53, 102.96, 101.48, 100.00, 98.61, 97.22, 97.49, 97.76, 94.60, 91.44, 92.94, 94.44, 94.80, 95.16, 94.70, 94.25, 92.36, 90.48, 91.42, 92.36, 90.63,
    88.89, 89.62, 90.36, 92.18, 94.00, 92.00, 90.00, 84.86, 79.72, 81.30, 82.88, 83.88, 84.89, 77.58, 70.27, 74.80, 82.19, 85.03, 78.47, 71.91, 62.37, 52.82, 64.39, 75.96, 73.91, 71.85, 72.41, 72.97, 70.17, 67.38, 63.07,
    58.75, 61.89, 65.02, 66.68, 68.34
};

//spectral data for Flash daylight 6000K (Canon, Pentax, Olympus,...Standard)
const double ColorTemp::Flash6000_spect[97] = {
    36.00, 37.18, 38.36, 40.36, 42.35, 41.77, 41.19, 43.79, 46.40, 59.24, 72.09, 76.15, 80.22, 81.47, 82.71, 80.12, 77.52, 86.54, 95.56, 101.70, 107.85, 108.66, 109.46, 108.57, 107.68, 108.65, 109.61, 106.63, 103.65, 104.42,
    105.19, 104.79, 104.39, 103.45, 102.51, 104.28, 106.05, 104.68, 103.31, 103.42, 103.54, 101.77, 100.00, 98.38, 96.75, 96.74, 96.72, 93.30, 89.89, 90.92, 91.95, 91.99, 92.03, 91.30, 90.57, 88.51, 86.45, 86.96,
    87.47, 85.66, 83.85, 84.21, 84.57, 85.95, 87.32, 85.31, 83.30, 78.66, 74.03, 75.24, 76.45, 77.68, 78.91, 72.13, 65.35, 69.67, 73.98, 76.68, 79.39, 73.29, 67.19, 58.19, 49.19, 59.98, 70.77, 68.91, 67.05, 67.55, 68.05,
    65.47, 62.88, 58.89, 54.90, 57.81, 60.72, 62.25, 63.78
};

//spectral data for Flash daylight 6500K (Nikon, Minolta, Panasonic, Sony...)
const double ColorTemp::Flash6500_spect[97] = {
    44.86, 45.72, 46.59, 49.16, 51.74, 50.83, 49.92, 52.26, 54.60, 68.65, 82.69, 87.06, 91.42, 92.39, 93.37, 90.00, 86.63, 95.72, 104.81, 110.88, 116.96, 117.36, 117.76, 116.29, 114.82, 115.35, 115.89, 112.33, 108.78, 109.06,
    109.33, 108.56, 107.78, 106.28, 104.78, 106.23, 107.68, 106.04, 104.40, 104.22, 104.04, 102.02, 100.00, 98.17, 96.34, 96.06, 95.79, 92.24, 88.69, 89.35, 90.02, 89.81, 89.61, 88.66, 87.71, 85.51, 83.30, 83.51, 83.72,
    81.88, 80.05, 80.14, 80.24, 81.27, 82.30, 80.31, 78.31, 74.03, 69.74, 70.69, 71.63, 73.00, 74.37, 68.00, 61.62, 65.76, 69.91, 72.51, 75.11, 69.36, 63.61, 55.02, 46.43, 56.63, 66.83, 65.11, 63.40, 63.86, 64.32, 61.90, 59.47,
    55.72, 51.97, 54.72, 57.46, 58.89, 60.33
};

const std::map<std::string,const double *> ColorTemp::spectMap = {
           {"Daylight", Daylight5300_spect},
           {"Cloudy", Cloudy6200_spect},
           {"Shade", Shade7600_spect},
           {"Tungsten", A2856_spect},
           {"Fluo F1", FluoF1_spect},
           {"Fluo F2", FluoF2_spect},
           {"Fluo F3", FluoF3_spect},
           {"Fluo F4", FluoF4_spect},
           {"Fluo F5", FluoF5_spect},
           {"Fluo F6", FluoF6_spect},
           {"Fluo F7", FluoF7_spect},
           {"Fluo F8", FluoF8_spect},
           {"Fluo F9", FluoF9_spect},
           {"Fluo F10", FluoF10_spect},
           {"Fluo F11", FluoF11_spect},
           {"Fluo F12", FluoF12_spect},
           {"HMI Lamp", HMI_spect},
           {"GTI Lamp", GTI_spect},
           {"JudgeIII Lamp", JudgeIII_spect},
           {"Solux Lamp 3500K", Solux3500_spect},
           {"Solux Lamp 4100K", Solux4100_spect},
           {"Solux Lamp 4700K", Solux4700_spect},
           {"NG Solux Lamp 4700K", NG_Solux4700_spect},
           {"LED LSI Lumelex 2040", NG_LEDLSI2040_spect},
           {"LED CRS SP12 WWMR16", NG_CRSSP12WWMR16_spect},
           {"Flash 5500K", Flash5500_spect},
           {"Flash 6000K", Flash6000_spect},
           {"Flash 6500K", Flash6500_spect}
      };

// Data for Color ==> CRI (Color Rendering Index and Palette
// actually 20 color that must be good enough for CRI

// I think 40  color for palette (Skin, Sky, gray)

//spectral data Colorchecker24 : Red C3
const double  ColorTemp::ColorchechredC3_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0478, 0.0476, 0.0474, 0.0472, 0.0470, 0.0466, 0.0461, 0.0460, 0.0459, 0.0456, 0.0453, 0.0451, 0.0449, 0.0448, 0.0447, 0.0446, 0.0445, 0.0441, 0.0437, 0.0432, 0.0427, 0.0424, 0.0421, 0.0419,
    0.0417, 0.0415, 0.0412, 0.0412, 0.0412, 0.0413, 0.0413, 0.0415, 0.0416, 0.0421, 0.0426, 0.0436, 0.0446, 0.0469, 0.0491, 0.0549, 0.0607, 0.0773, 0.0939, 0.1376, 0.1812, 0.2568, 0.3323, 0.4070, 0.4816, 0.5308,
    0.5800, 0.6059, 0.6317, 0.6447, 0.6577, 0.6653, 0.6728, 0.6761, 0.6793, 0.6820, 0.6847, 0.6878, 0.6909, 0.6945, 0.6980, 0.7013, 0.7046, 0.7065, 0.7084, 0.7107, 0.7129, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data Colorchecker24 : Orange A2
const double ColorTemp::ColorchechOraA2_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0520, 0.0530, 0.0540, 0.0535, 0.0530, 0.0532, 0.0534, 0.0532, 0.0529, 0.0529, 0.0528, 0.0530, 0.0532, 0.0537, 0.0542, 0.0550, 0.0557, 0.0565, 0.0573, 0.0585, 0.0597, 0.0613, 0.0628, 0.0656, 0.0683, 0.0793,
    0.0902, 0.1085, 0.1268, 0.1414, 0.1559, 0.1645, 0.1730, 0.1837, 0.1944, 0.2184, 0.2424, 0.2877, 0.3329, 0.3923, 0.4517, 0.5021, 0.5525, 0.5739, 0.5952, 0.5967, 0.5982, 0.5962, 0.5942, 0.5932, 0.5922, 0.5927,
    0.5932, 0.5938, 0.5944, 0.5988, 0.6032, 0.6105, 0.6178, 0.6284, 0.6389, 0.6498, 0.6607, 0.6699, 0.6791, 0.6839, 0.6886, 0.6879, 0.6871, 0.6886, 0.6901, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data Colorchecker24 :yellow D3
const double ColorTemp::ColorchechYelD3_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0476, 0.0482, 0.0488, 0.0492, 0.0496, 0.0498, 0.0499, 0.0498, 0.0496, 0.0501, 0.0506, 0.0516, 0.0526, 0.0547, 0.0567, 0.0610, 0.0652, 0.0733, 0.0813, 0.0962, 0.1110, 0.1333, 0.1556, 0.1884, 0.2211,
    0.2782, 0.3353, 0.4023, 0.4692, 0.5198, 0.5703, 0.5976, 0.6249, 0.6400, 0.6551, 0.6667, 0.6783, 0.6901, 0.7018, 0.7095, 0.7171, 0.7231, 0.7290, 0.7329, 0.7367, 0.7395, 0.7423, 0.7447, 0.7471, 0.7490, 0.7508,
    0.7533, 0.7558, 0.7578, 0.7598, 0.7623, 0.7647, 0.7654, 0.7661, 0.7677, 0.7693, 0.7720, 0.7746, 0.7780, 0.7814, 0.7845, 0.7876, 0.7889, 0.7902, 0.7920, 0.7938,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data Colorchecker24 : Green E2
const double ColorTemp::ColorchechGreE2_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0560, 0.0583, 0.0605, 0.0626, 0.0646, 0.0650, 0.0653, 0.0657, 0.0661, 0.0669, 0.0676, 0.0692, 0.0708, 0.0737, 0.0765, 0.0816, 0.0867, 0.0956, 0.1044, 0.1194, 0.1344, 0.1581, 0.1818, 0.2196, 0.2574, 0.3166, 0.3757,
    0.4297, 0.4837, 0.5142, 0.5446, 0.5541, 0.5636, 0.5608, 0.5579, 0.5480, 0.5381, 0.5258, 0.5135, 0.4959, 0.4783, 0.4570, 0.4356, 0.4124, 0.3891, 0.3710, 0.3529, 0.3425, 0.3320, 0.3266, 0.3211, 0.3180, 0.3149,
    0.3129, 0.3108, 0.3123, 0.3137, 0.3193, 0.3248, 0.3335, 0.3422, 0.3518, 0.3613, 0.3693, 0.3772, 0.3810, 0.3847, 0.3838, 0.3829, 0.3838, 0.3847, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::ColorGreenM25_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0758, 0.0614, 0.0662, 0.0591, 0.0648, 0.0633, 0.0671, 0.0613, 0.0621, 0.0610, 0.0680, 0.0690, 0.0784, 0.0830, 0.0920, 0.1070, 0.1231, 0.1423, 0.1607, 0.1731, 0.1816, 0.1911, 0.1951, 0.1986, 0.1915, 0.1889,
    0.1758, 0.1673, 0.1606, 0.1505, 0.1384, 0.1317, 0.1230, 0.1149, 0.1081, 0.0992, 0.0882, 0.0785, 0.0709, 0.0629, 0.0550, 0.0502, 0.0486, 0.0474, 0.0445, 0.0434, 0.0429, 0.0423, 0.0411, 0.0405, 0.0397, 0.0387, 0.0399, 0.0398, 0.0398, 0.0407, 0.0408, 0.0426, 0.0445, 0.0443, 0.0468,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
//0 658 45 0 0 131 98 84 123 104 131 112 121 121 139 128 148 199 296 389 476 689 945 1132 1326 1490 1674 1741 1775 1868 1914 1928 1961 1972 1992 2045 2064 2053 2048 2072 2086 2081 2069 2056 2073 2096 2114 2067 2089 2100 2061 2019 1983 1971 1961 2016 1956 1946 1922 1983 1991
const double ColorTemp::ColorYellowkeltano_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0, 0.0658, 0.0045,  0.0,  0.0, 0.0131,  0.0098,  0.0084, 0.0123, 0.0104, 0.0131, 0.0112, 0.0121, 0.0121, 0.0139,
    0.0128, 0.0148, 0.0199, 0.0296, 0.0389, 0.0476, 0.0689, 0.0945, 0.1132, 0.1326, 0.1490,  0.1674, 0.1741, 0.1775, 0.1868,
    0.1914, 0.1928, 0.1961, 0.1972, 0.1992, 0.2045, 0.2064, 0.2053, 0.2048, 0.2072, 0.2086, 0.2081, 0.2069, 0.2056,  0.2073,
    0.2096, 0.2114, 0.2067, 0.2089, 0.2100, 0.2061, 0.2019, 0.1983, 0.1971, 0.1961, 0.2016, 0.1956, 0.1946, 0.1922, 0.1983, 0.1991,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorGreenalsi_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1732,  0.0292,  0.0315,  0.0533,  0.0469,  0.0359,  0.0404,  0.0356,  0.0345,  0.0374,  0.0348,  0.0336,  0.0332,  0.0333,  0.0342,
    0.0330,  0.0350,  0.0343,  0.0354,  0.0371,  0.0399,  0.0426,  0.0497,  0.0541,  0.0616,  0.0701,  0.0750,  0.0764,  0.0783,  0.0794,
    0.0784,  0.0732,  0.0708,  0.0652,  0.0612,  0.0595,  0.0570,  0.0531,  0.0507,  0.0501,  0.0488,  0.0458,  0.0437,  0.0420,  0.0436,
    0.0424,  0.0417,  0.0389,  0.0380,  0.0378,  0.0371,  0.0350,  0.0333,  0.0350,  0.0394,  0.0379,  0.0446,  0.0491,  0.0575,  0.0734,  0.0953,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
//1890 1097 855 899 987 881 807 804 787 691 643 549 465 404 385 302 244 195 165 159 123 129 108 111 114 126 126 134 162 170 213 248 279 351 412 566 752 909 1069 1270 1526
//1707 1858 1999 2112 2293 2422 2471 2611 2718 2710 2778 2807 2825 2856 2909 2901 2974 3042 3044 3075
const double ColorTemp::ColorRedpetunia_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1890, 0.1097,  0.0855,  0.0899,  0.0987,  0.0881,  0.0807,  0.0804,  0.0787,  0.0691,  0.0643,  0.0549,  0.0465,  0.0404,  0.0385,
    0.0302,  0.0244,  0.0195,  0.0165,  0.0159,  0.0123,  0.0129,  0.0108,  0.0111,  0.0114,  0.0126,  0.0126,  0.0134,  0.0162,  0.0170,
    0.0213,  0.0248,  0.0279,  0.0351,  0.0412,  0.0566,  0.0752,  0.0909, 0.1069, 0.1270, 0.1526, 0.1707, 0.1858, 0.1999, 0.2112,
    0.2293, 0.2422, 0.2471, 0.2611, 0.2718, 0.2710, 0.2778, 0.2807, 0.2825, 0.2856, 0.2909, 0.2901, 0.2974, 0.3042, 0.3044, 0.3075,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//0.1588    0.1980  0.2789  0.4585  0.6059  0.6609  0.6674  0.6599  0.6551  0.6472  0.6423  0.6485  0.6515  0.6379  0.6193  0.6121  0.6026  0.5678  0.5310  0.5245  0.5305  0.5324  0.5262  0.5219  0.5247  0.5312  0.5436  0.5634  0.5832  0.5943  0.5953  0.5902  0.5805  0.5754  0.5901  0.6262

const double ColorTemp::JDC468_B14_75Redspect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1588, 0.1780, 0.1980, 0.2310, 0.2789, 0.3213, 0.4585, 0.5324, 0.6059, 0.6334, 0.6609, 0.6630, 0.6674, 0.6640, 0.6599, 0.6600, 0.6551, 0.6532, 0.6472, 0.6450,
    0.6423, 0.6467, 0.6485, 0.6500, 0.6515, 0.6456, 0.6379, 0.6285, 0.6193, 0.6150, 0.6121, 0.6056, 0.6026, 0.5812, 0.5678, 0.5490, 0.5310, 0.5307, 0.5245, 0.5300,
    0.5305, 0.5310, 0.5324, 0.5298, 0.5262, 0.5230, 0.5219, 0.5231, 0.5247, 0.5280, 0.5312, 0.5364, 0.5436, 0.5550, 0.5634, 0.5731, 0.5832, 0.5901, 0.5943, 0.5950,
    0.5953, 0.5926, 0.5902, 0.5850, 0.5805, 0.5770, 0.5754, 0.6012, 0.6262,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


const double ColorTemp::ColorRedkurttu_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0979, 0.1536,  0.0745, 0.1145,  0.0937,  0.0996,  0.0972,  0.0924,  0.0875,  0.0878,  0.0826,  0.0766,  0.0658,  0.0599,  0.0545,
    0.0509,  0.0443,  0.0381,  0.0356,  0.0327,  0.0318,  0.0304,  0.0295,  0.0288,  0.0285,  0.0271,  0.0281,  0.0282,  0.0278,  0.0280,
    0.0296,  0.0309,  0.0324,  0.0342,  0.0343,  0.0376,  0.0419,  0.0464,  0.0520,  0.0634,  0.0747,  0.0905, 0.1093, 0.1167, 0.1264,
    0.1439, 0.1650, 0.1928, 0.2183, 0.2380, 0.2537, 0.2754, 0.2893, 0.3009, 0.3115, 0.3213, 0.3173, 0.3222, 0.3237, 0.3192, 0.3210,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//#light red flower (lupiini)
//1792 1553 1684 1882 1909 1847 2053 2084 2045 2052 2039 2084 2041 2044 2007 1984 1906 1876 1886 1855 1859 1875 1816 1800 1811 1780 1802 1816 1838 1915 1973 2018 2083 2114 2133 2226 2304 2385 2458 2494 2571 2689 2738 2774 2734 2759 2781 2831 2844 2857 2878 2876 2884 2920 2932 2860 2894 2934 2925 2928 2921
const double ColorTemp::ColorRedlupiini_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1792, 0.1553, 0.1684, 0.1882, 0.1909, 0.1847, 0.2053, 0.2084, 0.2045, 0.2052, 0.2039, 0.2084, 0.2041, 0.2044, 0.2007, 0.1984, 0.1906, 0.1876,
    0.1886, 0.1855, 0.1859, 0.1875, 0.1816, 0.1800, 0.1811, 0.1780, 0.1802, 0.1816, 0.1838, 0.1915, 0.1973, 0.2018, 0.2083, 0.2114, 0.2133, 0.2226,
    0.2304, 0.2385, 0.2458, 0.2494, 0.2571, 0.2689, 0.2738, 0.2774, 0.2734, 0.2759, 0.2781, 0.2831, 0.2844, 0.2857, 0.2878, 0.2876, 0.2884, 0.2920,
    0.2932, 0.2860, 0.2894, 0.2934, 0.2925, 0.2928, 0.2921,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
//#red flower (hevosminttu)
//1280  706  612  462  391  339  253  285  261  264  239  208  201  186  185  161  156  149  146  148  161  144  143  151  147  146  139  148  173  185  185  197  222  238  283  322  384  439  519  633  792  922 1061 1186 1235 1342 1538 1691 1839 1974 2024 2098 2128 2187 2204 2217 2267 2299 2339 2331 2322
const double ColorTemp::ColorRedhevosminttu_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1280,  0.0706,  0.0612,  0.0462,  0.0391,  0.0339,  0.0253,  0.0285,  0.0261,  0.0264,  0.0239,  0.0208,  0.0201,  0.0186,  0.0185,  0.0161,  0.0156,  0.0149,
    0.0146,  0.0148,  0.0161,  0.0144,  0.0143,  0.0151,  0.0147,  0.0146,  0.0139,  0.0148,  0.0173,  0.0185,  0.0185,  0.0197,  0.0222,  0.0238,  0.0283,  0.0322,
    0.0384,  0.0439,  0.0519,  0.0633,  0.0792,  0.0922,  0.1061, 0.1186, 0.1235, 0.1342, 0.1538, 0.1691, 0.1839, 0.1974, 0.2024, 0.2098, 0.2128, 0.2187, 0.2204,
    0.2217, 0.2267, 0.2299, 0.2339, 0.2331, 0.2322,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
//#red flower (neilikka)
//0    0  394    0  245   95  174  149  194  171  181  175  172  167  147  137  107  108  100   87   93   87   83   77   80   67   72   64   83   84   88   90   91   94  114  133  178  241  309  419  612  823  992 1153 1222 1366 1503 1658 1767 1841 1884 1992 2035 2007 2009 2045 2065 2229 2290 2395 2449
const double ColorTemp::ColorRedneilikka_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0,    0.0,  0.0394,    0.0,  0.0245,   0.0095,  0.0174,  0.0149,  0.0194,  0.0171,  0.0181,  0.0175,  0.0172,  0.0167,  0.0147,  0.0137,  0.0107,  0.0108,  0.0100,
    0.0087,   0.0093,   0.0087,   0.0083,   0.0077,   0.0080,   0.0067,   0.0072,   0.0064,   0.0083,   0.0084,   0.0088,   0.0090,   0.0091,   0.0094,  0.0114,  0.0133,
    0.0178,  0.0241,  0.0309,  0.0419,  0.0612,  0.0823,  0.0992, 0.1153, 0.1222, 0.1366, 0.1503, 0.1658, 0.1767, 0.1841, 0.1884, 0.1992, 0.2035, 0.2007, 0.2009,
    0.2045, 0.2065, 0.2229, 0.2290, 0.2395, 0.2449,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//#red petal (pelargonia)
//2529 2048 2087 2698 2452 2372 2531 2475 2296 2294 2159 2111 1986 1898 1854 1729 1586 1501 1392 1332 1343 1255 1217 1182 1183 1203 1230 1277 1381 1474 1615 1762 1876 2028 2214 2464 2657 2919 3051 3172 3293 3421 3395 3494 3438 3495 3506 3490 3454 3487 3431 3452 3484 3438 3422 3368 3325 3441 3356 3432 3320
const double ColorTemp::ColorRedpelagornia_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2529, 0.2048, 0.2087, 0.2698, 0.2452, 0.2372, 0.2531, 0.2475, 0.2296, 0.2294, 0.2159, 0.2111, 0.1986, 0.1898, 0.1854, 0.1729, 0.1586, 0.1501, 0.1392, 0.1332, 0.1343,
    0.1255, 0.1217, 0.1182, 0.1183, 0.1203, 0.1230, 0.1277, 0.1381, 0.1474, 0.1615, 0.1762, 0.1876, 0.2028, 0.2214, 0.2464, 0.2657, 0.2919, 0.3051, 0.3172, 0.3293, 0.3421,
    0.3395, 0.3494, 0.3438, 0.3495, 0.3506, 0.3490, 0.3454, 0.3487, 0.3431, 0.3452, 0.3484, 0.3438, 0.3422, 0.3368, 0.3325, 0.3441, 0.3356, 0.3432, 0.3320,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//#red flower (talvio)
//3131 2199 2559 2540 2844 2530 2694 2765 2594 2673 2617 2629 2491 2384 2308 2256 2081 1973 1857 1752 1719 1652 1527 1477 1459 1386 1341 1283 1318 1334 1354 1424 1495 1543 1634 1773 1950 2129 2272 2431 2642 2827 2941 3045 3082 3158 3216 3307 3364 3388 3387 3517 3573 3501 3499 3523 3495 3606 3493 3518 3522
const double ColorTemp::ColorRedtalvio_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.3131, 0.2199, 0.2559, 0.2540, 0.2844, 0.2530, 0.2694, 0.2765, 0.2594, 0.2673, 0.2617, 0.2629, 0.2491, 0.2384, 0.2308, 0.2256, 0.2081, 0.1973, 0.1857, 0.1752, 0.1719,
    0.1652, 0.1527, 0.1477, 0.1459, 0.1386, 0.1341, 0.1283, 0.1318, 0.1334, 0.1354, 0.1424, 0.1495, 0.1543, 0.1634, 0.1773, 0.1950, 0.2129, 0.2272, 0.2431, 0.2642, 0.2827,
    0.2941, 0.3045, 0.3082, 0.3158, 0.3216, 0.3307, 0.3364, 0.3388, 0.3387, 0.3517, 0.3573, 0.3501, 0.3499, 0.3523, 0.3495, 0.3606, 0.3493, 0.3518, 0.3522,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0

};

//#brown dry leaf (poimulehti)
//964  520  223  244  261  247  196  199  200  207  202  198  209  204  207  222  205  218  213  212  224  218  230  235  251  250  245  250  263  273  271  275  281  264  274  288  287  307  303  307  323  304  335  335  346  345  347  348  370  364  380  393  384  407  419  421  419  433  431  461  465
//RIs 67
const double ColorTemp::ColorBrownpoimulehti_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0964,  0.0520,  0.0223,  0.0244,  0.0261,  0.0247,  0.0196,  0.0199,  0.0200,  0.0207,  0.0202,  0.0198,  0.0209,  0.0204,  0.0207,  0.0222,  0.0205,  0.0218, 0.0213,
    0.0212,  0.0224,  0.0218,  0.0230,  0.0235,  0.0251,  0.0250,  0.0245,  0.0250,  0.0263,  0.0273,  0.0271,  0.0275,  0.0281,  0.0264,  0.0274,  0.0288, 0.0287,  0.0307,
    0.0303,  0.0307,  0.0323,  0.0304,  0.0335,  0.0335,  0.0346,  0.0345,  0.0347,  0.0348,  0.0370,  0.0364,  0.0380,  0.0393,  0.0384,  0.0407,
    0.0419,  0.0421,  0.0419,  0.0433,  0.0431,  0.0461,  0.0465,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//#orange leaf (koristepensas, tuntematon)
//241  195  223  489  574  565  634  605  574  613  645  636  644  628  621  603  614  654  676  719  776  795  862  879  918  918  955  980 1013 1055 1132 1225 1258 1362 1427 1579 1796 1936 2079 2258 2440 2597 2728 2790 2777 2857 2923 2991 3031 3040 3037 3094 3066 3023 3093 3044 3082 3085 3147 3226 3192
const double ColorTemp::ColorOrangetuntematon_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0241,  0.0195,  0.0223,  0.0489,  0.0574,  0.0565,  0.0634,  0.0605,  0.0574,  0.0613,  0.0645,  0.0636,  0.0644,  0.0628,  0.0621,  0.0603,  0.0614,  0.0654,  0.0676,
    0.0719,  0.0776,  0.0795,  0.0862,  0.0879,  0.0918,  0.0918,  0.0955,  0.0980, 0.1013, 0.1055, 0.1132, 0.1225, 0.1258, 0.1362, 0.1427, 0.1579, 0.1796, 0.1936, 0.2079,
    0.2258, 0.2440, 0.2597, 0.2728, 0.2790, 0.2777, 0.2857, 0.2923, 0.2991, 0.3031, 0.3040, 0.3037, 0.3094, 0.3066, 0.3023, 0.3093, 0.3044, 0.3082, 0.3085, 0.3147, 0.3226, 0.3192,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
//,#orange leaf (lehmus)
//1062  866  443  544  496  485  492  458  450  425  458  477  497  461  451  481  454  500  515  538  529  593  638  670  686  711  718  729  741  760  796  833  895  958 1016 1128 1246 1344 1450 1505 1596 1636 1621 1631 1627 1628 1658 1583 1486 1415 1322 1265 1159 1062  975  974 1063 1326 1736 2141 2568
const double ColorTemp::ColorOrangetlehmus_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1062,  0.0866,  0.0443,  0.0544,  0.0496,  0.0485,  0.0492,  0.0458,  0.0450,  0.0425,  0.0458,  0.0477,  0.0497,  0.0461,  0.0451,  0.0481,  0.0454,  0.0500,  0.0515,
    0.0538,  0.0529,  0.0593,  0.0638,  0.0670,  0.0686,  0.0711,  0.0718,  0.0729,  0.0741,  0.0760,  0.0796,  0.0833,  0.0895,  0.0958, 0.1016, 0.1128, 0.1246, 0.1344, 0.1450,
    0.1505, 0.1596, 0.1636, 0.1621, 0.1631, 0.1627, 0.1628, 0.1658, 0.1583, 0.1486, 0.1415, 0.1322, 0.1265, 0.1159, 0.1062,  0.0975,  0.0974, 0.1063, 0.1326, 0.1736, 0.2141, 0.2568,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//#orange leaf (vaahtera)
//1517  551  664  659  521  585  460  385  424  389  375  374  359  380  371  373  379  387  378  394  405  416  463  496  536  542  577  579  619  642  678  710  777  829  894 1035 1174 1334 1484 1611 1798 1941 2012 2065 2135 2229 2286 2317 2332 2357 2323 2330 2292 2236 2137 2093 2180 2240 2368 2487 2528
const double ColorTemp::ColorOrangvaahtera_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1517,  0.0551,  0.0664,  0.0659,  0.0521,  0.0585,  0.0460,  0.0385,  0.0424,  0.0389,  0.0375,  0.0374,  0.0359,  0.0380,  0.0371,  0.0373,  0.0379,  0.0387,  0.0378,
    0.0394,  0.0405,  0.0416,  0.0463,  0.0496,  0.0536,  0.0542,  0.0577,  0.0579,  0.0619,  0.0642,  0.0678,  0.0710,  0.0777,  0.0829,  0.0894, 0.1035, 0.1174, 0.1334, 0.1484,
    0.1611, 0.1798, 0.1941, 0.2012, 0.2065, 0.2135, 0.2229, 0.2286, 0.2317, 0.2332, 0.2357, 0.2323, 0.2330, 0.2292, 0.2236, 0.2137, 0.2093, 0.2180, 0.2240, 0.2368, 0.2487, 0.2528,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//#brown dry leaf (lehmus)
//758  236  148  430  347  435  438  495  439  454  472  471  461  459  458  479  492  482  499  513  520  545  567  594  623  647  698  717  744  792  803  834  864  876  916  932  963 1013 1025 1060 1099 1118 1153 1175 1207 1242 1268 1266 1284 1305 1305 1304 1353 1360 1330 1332 1413 1502 1610 1682 1737
const double ColorTemp::ColorBrownlehmus_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0758,  0.0236,  0.0148,  0.0430,  0.0347,  0.0435,  0.0438,  0.0495,  0.0439,  0.0454,  0.0472,  0.0471,  0.0461,  0.0459,  0.0458,  0.0479,  0.0492,  0.0482,  0.0499,  0.0513,
    0.0520,  0.0545,  0.0567,  0.0594,  0.0623,  0.0647,  0.0698,  0.0717,  0.0744,  0.0792,  0.0803,  0.0834,  0.0864,  0.0876,  0.0916,  0.0932,  0.0963, 0.1013, 0.1025, 0.1060,
    0.1099, 0.1118, 0.1153, 0.1175, 0.1207, 0.1242, 0.1268, 0.1266, 0.1284, 0.1305, 0.1305, 0.1304, 0.1353, 0.1360, 0.1330, 0.1332, 0.1413, 0.1502, 0.1610, 0.1682, 0.1737,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


//#brown moss (nuotiosammal)
//482  260  178   92  104   88   92   40   43   52   58   64   70   63   67   67   62   76   82   82   91   96  104  116  135  141  142  155  168  179  198  199  193  201  212  218  226  240  242  238  255  265  277  266  265  283  289  275  289  277  291  288  277  252  262  260  264  299  375  411  446
const double ColorTemp::ColorBrownuotiosammal_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0482,  0.0260,  0.0178, 0.0092,  0.0104, 0.0088, 0.0092, 0.0040, 0.0043, 0.0052, 0.0058, 0.0064, 0.0070, 0.0063, 0.0067, 0.0067, 0.0062, 0.0076, 0.0082, 0.0082, 0.0091, 0.0096,
    0.0104,  0.0116,  0.0135,  0.0141, 0.0142, 0.0155,  0.0168,  0.0179,  0.0198,  0.0199,  0.0193,  0.0201,  0.0212,  0.0218,  0.0226,  0.0240,  0.0242,  0.0238,  0.0255,  0.0265,
    0.0277,  0.0266,  0.0265,  0.0283,  0.0289,  0.0275,  0.0289,  0.0277,  0.0291,  0.0288,  0.0277,  0.0252,  0.0262,  0.0260,  0.0264,  0.0299,  0.0375,  0.0411,  0.0446,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//#black soil
//0    0   89  122   52   53  104  127  130  134  137  137  134  136  138  139  134  140  142  148  154  153  152  150  151  156  153  166  154  171  163  163  166  166  169  169  166  174  174  170  170  168  176  177  176  174  179  180  180  183  177  193  178  187  194  193  182  196  184  195  195
const double ColorTemp::ColorBlacksoil_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0,    0.0,   0.0089,  0.0122, 0.0052, 0.0053, 0.0104, 0.0127, 0.0130, 0.0134, 0.0137, 0.0137, 0.0134, 0.0136, 0.0138, 0.0139, 0.0134, 0.0140, 0.0142, 0.0148, 0.0154, 0.0153,
    0.0152,  0.0150, 0.0151, 0.0156, 0.0153, 0.0166, 0.0154, 0.0171, 0.0163, 0.0163, 0.0166, 0.0166, 0.0169, 0.0169, 0.0166, 0.0174, 0.0174, 0.0170, 0.0170, 0.0168, 0.0176, 0.0177,
    0.0176,  0.0174,  0.0179,  0.0180,  0.0180, 0.0183, 0.0177, 0.0193, 0.0178, 0.0187, 0.0194, 0.0193, 0.0182, 0.0196, 0.0184, 0.0195, 0.0195,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//#black, dry leaf (pihlaja)
//0   79  111  172  174  201  214  211  207  207  191  200  196  206  196  194  203  207  204  208  210  212  211  208  209  219  222  224  231  241  232  244  249  250  267  264  262  269  282  277  289  284  279  302  289  308  313  315  310  325  313  319  356  340  331  347  356  352  364  373  352
const double ColorTemp::ColorBlackpihlaja[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0, 0.0079, 0.0111, 0.0172, 0.0174, 0.0201, 0.0214, 0.0211, 0.0207, 0.0207, 0.0191, 0.0200, 0.0196, 0.0206, 0.0196, 0.0194,  0.0203, 0.0207, 0.0204, 0.0208, 0.0210, 0.0212,
    0.0211, 0.0208, 0.0209, 0.0219, 0.0222, 0.0224, 0.0231, 0.0241, 0.0232, 0.0244, 0.0249, 0.0250, 0.0267, 0.0264, 0.0262, 0.0269, 0.0282, 0.0277, 0.0289, 0.0284, 0.0279, 0.0302,
    0.0289, 0.0308, 0.0313, 0.0315, 0.0310, 0.0325, 0.0313, 0.0319, 0.0356, 0.0340, 0.0331, 0.0347, 0.0356, 0.0352, 0.0364, 0.0373, 0.0352,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};



//#gray lichen (nahkajaekaelae)
//1204  585 1113  733  600  653  715  685  726  682  713  691  719  691  683  693  711  715  701  700  720  697  706  696  723  714  726  738  729  735  737  739  742  746  746  761  743  735  722  717  728  749  721  712  705  737  733  758  780  785  775  771  755  744  743  742  755  779  849  940 1042
//RIS 74
const double ColorTemp::ColorGraynahjajaekaelae_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1204,  0.0585, 0.1113,  0.0733,  0.0600,  0.0653,  0.0715,  0.0685,  0.0726,  0.0682,  0.0713,  0.0691,  0.0719,  0.0691,  0.0683,  0.0693,  0.0711,  0.0715,  0.0701,  0.0700,
    0.0720,  0.0697,  0.0706,  0.0696,  0.0723,  0.0714,  0.0726,  0.0738,  0.0729,  0.0735,  0.0737,  0.0739,  0.0742,  0.0746,  0.0746,  0.0761,  0.0743,  0.0735,  0.0722,  0.0717,
    0.0728,  0.0749,  0.0721,  0.0712,  0.0705,  0.0737,  0.0733,  0.0758,  0.0780,  0.0785,  0.0775,  0.0771,  0.0755,  0.0744,  0.0743,  0.0742,  0.0755,  0.0779,  0.0849,  0.0940, 0.1042,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//#green moss (nuotiosammal)
//120   65  134   31  209  124  104   96   97   95   76   79   83   93   83   95   95  104  117  127  140  161  214  252  290  310  328  343  347  373  365  351  347  343  311  301  285  283  263  256  255  251  257  235  227  224  233  208  194  186  165  160  151  149  157  161  185  243  309  425  543
const double ColorTemp::ColorGreennuotisammal_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0120, 0.0065, 0.0134, 0.0031, 0.0209, 0.0124, 0.0104, 0.0096,  0.0097, 0.0095, 0.0076, 0.0079, 0.0083, 0.0093, 0.0083, 0.0095, 0.0095, 0.0104, 0.0117, 0.0127, 0.0140, 0.0161,
    0.0214,  0.0252,  0.0290,  0.0310,  0.0328,  0.0343, 0.0347,  0.0373, 0.0365, 0.0351,  0.0347,  0.0343,  0.0311,  0.0301,  0.0285,  0.0283,  0.0263,  0.0256,  0.0255,  0.0251,
    0.0257,  0.0235,  0.0227,  0.0224,  0.0233,  0.0208,  0.0194,  0.0186,  0.0165,  0.0160,  0.0151,  0.0149,  0.0157,  0.0161,  0.0185,  0.0243,  0.0309,  0.0425,  0.0543,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


//#green leaf (leskenlehti)
//525  273    0  378  318  164  224  276  316  266  303  290  305  286  290  303  323  323  352  383  405  482  614  743  920 1015 1139 1192 1175 1216 1195 1145 1116 1009  947  867  802  754  741  709  675  625  574  579  561  565  557  511  471  419  399  372  365  395  375  382  458  555  716 1002 1407
const double ColorTemp::ColorGreenleskenlehti_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0525,  0.0273, 0.0, 0.0378,  0.0318, 0.0164, 0.0224, 0.0276, 0.0316, 0.0266, 0.0303, 0.0290, 0.0305, 0.0286, 0.0290, 0.0303, 0.0323, 0.0323, 0.0352, 0.0383,  0.0405, 0.0482,
    0.0614,  0.0743,  0.0920, 0.1015, 0.1139, 0.1192, 0.1175, 0.1216, 0.1195, 0.1145, 0.1116, 0.1009,  0.0947,  0.0867,  0.0802,  0.0754,  0.0741,  0.0709,  0.0675,  0.0625,
    0.0574,  0.0579,  0.0561,  0.0565,  0.0557,  0.0511,  0.0471,  0.0419,  0.0399,  0.0372,  0.0365,  0.0395,  0.0375,  0.0382,  0.0458,  0.0555,  0.0716, 0.1002, 0.1407,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//#green leaf (linnunkaali)
//602    0  267  306  339  335  300  247  292  289  295  298  292  318  312  289  299  307  310  320  350  375  446  499  574  634  698  725  736  754  736  702  668  633  590  551  514  499  467  460  445  424  415  409  399  412  393  380  370  362  366  343  342  350  333  350  364  418  494  670  914
//RIS 77
const double ColorTemp::ColorGreenlinnunkaali_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0602, 0.0, 0.0267,  0.0306,  0.0339,  0.0335,  0.0300,  0.0247,  0.0292, 0.0289,  0.0295,  0.0298,  0.0292,  0.0318,  0.0312,  0.0289,  0.0299,  0.0307, 0.0310, 0.0320,
    0.0350,  0.0375,  0.0446,  0.0499,  0.0574,  0.0634,  0.0698,  0.0725,  0.0736,  0.0754,  0.0736,  0.0702,  0.0668, 0.0633, 0.0590,  0.0551,  0.0514,  0.0499,  0.0467,  0.0460,
    0.0445,  0.0424,  0.0415,  0.0409,  0.0399,  0.0412,  0.0393,  0.0380,  0.0370,  0.0362,  0.0366, 0.0343,  0.0342,  0.0350,  0.0333, 0.0350, 0.0364, 0.0418, 0.0494,  0.0670,  0.0914,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
//#green leaf (pelto-ohake)
//0  366  360  233  173  179  157  175  206  205  180  179  173  178  187  189  184  171  195  204  193  219  253  297  365  431  467  489  493  516  500  466  426  406  380  343  316  295  276  282  265  253  239  228  226  229  238  237  216  221  219  217  212  219  229  258  284  309  375  487  732
const double ColorTemp::ColorGreenpelto_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0, 0.0366, 0.0360, 0.0233, 0.0173, 0.0179, 0.0157, 0.0175, 0.0206, 0.0205, 0.0180, 0.0179, 0.0173, 0.0178, 0.0187, 0.0189,  0.0184, 0.0171,  0.0195, 0.0204, 0.0193, 0.0219,
    0.0253, 0.0297,  0.0365,  0.0431, 0.0467,  0.0489,  0.0493,  0.0516,  0.0500,  0.0466, 0.0426, 0.0406, 0.0380, 0.0343, 0.0316, 0.0295, 0.0276, 0.0282, 0.0265,  0.0253,  0.0239,
    0.0228, 0.0226, 0.0229,  0.0238,  0.0237,  0.0216,  0.0221,  0.0219,  0.0217,  0.0212,  0.0219,  0.0229,  0.0258,  0.0284,  0.0309,  0.0375,  0.0487,  0.0732,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//#green rod (voikukka),
//2205 1755 1710 1365 1159 1207 1024 1118 1127 1141 1134 1125 1149 1140 1120 1128 1139 1156 1212 1273 1262 1359 1461 1519 1568 1599 1660 1668 1680 1718 1697 1690 1672 1675 1663 1644 1642 1652 1626 1623 1653 1621 1614 1590 1625 1609 1615 1576 1509 1483 1418 1391 1324 1294 1267 1220 1315 1417 1650 1861 2006
const double ColorTemp::ColorGreenrodvoikukka[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2205, 0.1755, 0.1710, 0.1365, 0.1159, 0.1207, 0.1024, 0.1118, 0.1127, 0.1141, 0.1134, 0.1125, 0.1149, 0.1140, 0.1120, 0.1128, 0.1139, 0.1156, 0.1212, 0.1273, 0.1262, 0.1359,
    0.1461, 0.1519, 0.1568, 0.1599, 0.1660, 0.1668, 0.1680, 0.1718, 0.1697, 0.1690, 0.1672, 0.1675, 0.1663, 0.1644, 0.1642, 0.1652, 0.1626, 0.1623, 0.1653, 0.1621, 0.1614, 0.1590,
    0.1625, 0.1609, 0.1615, 0.1576, 0.1509, 0.1483, 0.1418, 0.1391, 0.1324, 0.1294, 0.1267, 0.1220, 0.1315, 0.1417, 0.1650, 0.1861, 0.2006,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//#green leaf (lehmus)
//2362 1024  945  666  617  604  591  580  648  631  656  607  616  653  643  626  643  656  710  753  801  929 1105 1277 1437 1601 1742 1774 1798 1848 1832 1820 1787 1730 1663 1593 1541 1461 1446 1419 1335 1298 1247 1192 1197 1199 1156 1072 1007  942  899  832  824  793  755  801  860 1031 1305 1809 2260
const double ColorTemp::ColorGreenlehmus[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2362, 0.1024,  0.0945,  0.0666,  0.0617,  0.0604, 0.0591, 0.0580,  0.0648,  0.0631,  0.0656,  0.0607,  0.0616,  0.0653,  0.0643,  0.0626,  0.0643,  0.0656,  0.0710,  0.0753,
    0.0801,  0.0929, 0.1105, 0.1277, 0.1437, 0.1601, 0.1742, 0.1774, 0.1798, 0.1848, 0.1832, 0.1820, 0.1787, 0.1730, 0.1663, 0.1593, 0.1541, 0.1461, 0.1446, 0.1419, 0.1335, 0.1298,
    0.1247, 0.1192, 0.1197, 0.1199, 0.1156, 0.1072, 0.1007,  0.0942, 0.0899, 0.0832, 0.0824, 0.0793, 0.0755, 0.0801, 0.0860, 0.1031, 0.1305, 0.1809, 0.2260,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//#green leaf (koristeherukka)
//945  292  315  433  321  294  295  321  278  261  282  272  270  278  285  274  277  268  269  283  275  309  325  389  450  493  551  557  587  585  567  554  515  487  460  424  409  387  353  349  353  333  309  309  312  315  321  298  304  304  281  273  293  311  314  333  355  392  439  595  811
const double ColorTemp::ColorGreenkoriste[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0945, 0.0292, 0.0315, 0.0433, 0.0321, 0.0294, 0.0295, 0.0321, 0.0278,  0.0261, 0.0282, 0.0272, 0.0270,  0.0278, 0.0285, 0.0274, 0.0277, 0.0268, 0.0269, 0.0283,
    0.0275, 0.0309, 0.0325, 0.0389, 0.0450, 0.0493, 0.0551, 0.0557, 0.0587, 0.0585, 0.0567, 0.0554, 0.0515, 0.0487, 0.0460, 0.0424, 0.0409, 0.0387, 0.0353, 0.0349,
    0.0353, 0.0333, 0.0309, 0.0309, 0.0312, 0.0315, 0.0321, 0.0298, 0.0304, 0.0304, 0.0281, 0.0273, 0.0293, 0.0311, 0.0314, 0.0333, 0.0355, 0.0392, 0.0439, 0.0595, 0.0811,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


//#green leaf (poimulehti)
//1102  146  630  266  247  261  285  238  273  281  272  260  262  254  274  263  273  278  296  309  322  388  493  607  712  840  953  986 1006 1034  999  981  918  855  794  711  649  627  604  563  531  515  467  450  448  466  445  421  402  385  369  345  346  319  330  359  378  439  578  835 1177
const double ColorTemp::ColorGreenpoimulehti[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1102, 0.0146, 0.0630, 0.0266, 0.0247, 0.0261, 0.0285, 0.0238, 0.0273, 0.0281, 0.0272, 0.0260, 0.0262, 0.0254, 0.0274, 0.0263, 0.0273, 0.0278, 0.0296, 0.0309,  0.0322,
    0.0388, 0.0493, 0.0607, 0.0712, 0.0840, 0.0953, 0.0986, 0.1006, 0.1034, 0.0999, 0.0981, 0.0918, 0.0855, 0.0794, 0.0711, 0.0649, 0.0627, 0.0604, 0.0563, 0.0531, 0.0515,
    0.0467, 0.0450, 0.0448, 0.0466, 0.0445, 0.0421, 0.0402, 0.0385, 0.0369, 0.0345, 0.0346, 0.0319, 0.0330, 0.0359, 0.0378, 0.0439, 0.0578, 0.0835, 0.1177,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


//#green leaf (hopeapaju)
//787  512 1260 1032  765  881  994  908  983  985  941  985  971  967  964  937  928  959  973  992 1004 1017 1053 1102 1180 1227 1281 1309 1317 1328 1318 1271 1238 1222 1179 1152 1131 1092 1086 1078 1083 1020 1015 1000 1027 1037 1028  970  962  977  952  963  955  935  980  979  963 1028 1059 1228 1401
const double ColorTemp::ColorGreenhopeapaju[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0787, 0.0512, 0.1260, 0.1032, 0.0765, 0.0881, 0.0994, 0.0908, 0.0983, 0.0985, 0.0941, 0.0985, 0.0971, 0.0967, 0.0964, 0.0937, 0.0928, 0.0959, 0.0973, 0.0992, 0.1004,
    0.1017, 0.1053, 0.1102, 0.1180, 0.1227, 0.1281, 0.1309, 0.1317, 0.1328, 0.1318, 0.1271, 0.1238, 0.1222, 0.1179, 0.1152, 0.1131, 0.1092, 0.1086, 0.1078, 0.1083, 0.1020,
    0.1015, 0.1000, 0.1027, 0.1037, 0.1028, 0.0970, 0.0962, 0.0977,  0.0952, 0.0963, 0.0955, 0.0935, 0.0980, 0.0979, 0.0963, 0.1028, 0.1059, 0.1228, 0.1401,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//#red flower (uuden guinean liisa)
//2288 1861 2364 2229 2783 2842 2842 2923 2902 2990 2828 2871 2772 2723 2639 2558 2424 2315 2169 2094 2064 1964 1865 1739 1680 1624 1548 1457 1424 1408 1434 1451 1492 1528 1597 1755 1951 2147 2367 2648 2986 3236 3393 3596 3665 3786 3879 3915 3926 3994 3987 4017 4026 4112 4067 4125 4139 4121 4050 4040 4095
const double ColorTemp::ColorReduuden[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2288, 0.1861, 0.2364, 0.2229, 0.2783, 0.2842, 0.2842, 0.2923, 0.2902, 0.2990, 0.2828, 0.2871, 0.2772, 0.2723, 0.2639, 0.2558, 0.2424, 0.2315, 0.2169, 0.2094, 0.2064,
    0.1964, 0.1865, 0.1739, 0.1680, 0.1624, 0.1548, 0.1457, 0.1424, 0.1408, 0.1434, 0.1451, 0.1492, 0.1528, 0.1597, 0.1755, 0.1951, 0.2147, 0.2367, 0.2648, 0.2986, 0.3236,
    0.3393, 0.3596, 0.3665, 0.3786, 0.3879, 0.3915, 0.3926, 0.3994, 0.3987, 0.4017, 0.4026, 0.4112, 0.4067, 0.4125, 0.4139, 0.4121, 0.4050, 0.4040, 0.4095,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


//#red flower (pajuangervo)
//445 1024  605  833  937  959 1052 1028 1049 1029 1017  975  948  882  865  812  757  718  658  638  628  597  554  523  509  509  485  475  469  492  479  477  490  525  555  597  641  704  756  846  948 1055 1164 1221 1266 1339 1393 1491 1553 1604 1608 1650 1643 1652 1655 1658 1651 1739 1813 1818 1938
const double ColorTemp::ColorRedpajuan[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0445, 0.1024, 0.0605, 0.0833, 0.0937, 0.0959, 0.1052, 0.1028, 0.1049, 0.1029, 0.1017, 0.0975, 0.0948, 0.0882, 0.0865, 0.0812, 0.0757, 0.0718, 0.0658, 0.0638, 0.0628, 0.0597,
    0.0554, 0.0523, 0.0509, 0.0509, 0.0485, 0.0475, 0.0469, 0.0492, 0.0479, 0.0477, 0.0490, 0.0525, 0.0555, 0.0597, 0.0641, 0.0704, 0.0756, 0.0846, 0.0948, 0.1055, 0.1164, 0.1221,
    0.1266, 0.1339, 0.1393, 0.1491, 0.1553, 0.1604, 0.1608, 0.1650, 0.1643, 0.1652, 0.1655, 0.1658, 0.1651, 0.1739, 0.1813, 0.1818, 0.1938,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


//#red flower (jaloangervo)
//120  152  512  635  662  538  749  713  743  792  777  785  733  726  728  749  709  674  661  657  645  635  598  570  553  544  545  538  546  514  540  567  585  577  602  651  690  765  836  907  980 1089 1147 1188 1212 1253 1318 1371 1412 1473 1459 1478 1548 1582 1564 1590 1595 1714 1728 1814 1837
const double ColorTemp::ColorRedjaloan[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0120, 0.0152, 0.0512, 0.0635, 0.0662, 0.0538, 0.0749, 0.0713, 0.0743, 0.0792, 0.0777, 0.0785, 0.0733, 0.0726, 0.0728, 0.0749, 0.0709, 0.0674, 0.0661, 0.0657, 0.0645, 0.0635,
    0.0598, 0.0570, 0.0553, 0.0544, 0.0545, 0.0538, 0.0546, 0.0514, 0.0540, 0.0567, 0.0585, 0.0577, 0.0602, 0.0651, 0.0690, 0.0765, 0.0836, 0.0907, 0.0980, 0.1089, 0.1147, 0.1188,
    0.1212, 0.1253, 0.1318, 0.1371, 0.1412, 0.1473, 0.1459, 0.1478, 0.1548, 0.1582, 0.1564, 0.1590, 0.1595, 0.1714, 0.1728, 0.1814, 0.1837,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


//#blue flower (ukonhattu)
//801  682 1070 1319 1311 1420 1453 1394 1318 1292 1268 1179 1132 1054 1015  948  846  780  731  709  705  667  621  598  555  522  505  493  498  500  494  471  479  463  450  461  487  515  546  574  555  562  539  558  546  552  567  626  715  807  862  978 1086 1199 1313 1323 1350 1366 1358 1320 1365
const double ColorTemp::ColorBlueukon[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0801, 0.0682, 0.1070, 0.1319, 0.1311, 0.1420, 0.1453, 0.1394, 0.1318, 0.1292, 0.1268, 0.1179, 0.1132, 0.1054, 0.1015, 0.0948, 0.0846, 0.0780, 0.0731, 0.0709, 0.0705, 0.0667,
    0.0621, 0.0598, 0.0555, 0.0522, 0.0505, 0.0493, 0.0498, 0.0500, 0.0494, 0.0471, 0.0479, 0.0463, 0.0450, 0.0461, 0.0487, 0.0515, 0.0546, 0.0574, 0.0555, 0.0562, 0.0539, 0.0558,
    0.0546, 0.0552, 0.0567, 0.0626, 0.0715, 0.0807, 0.0862, 0.0978, 0.1086, 0.1199, 0.1313, 0.1323, 0.1350, 0.1366, 0.1358, 0.1320, 0.1365,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


//#blue flower (orvokki)
//292  528  645 1000  932 1439 1752 1947 2077 2158 2169 2153 2164 2132 2091 1993 1916 1876 1803 1702 1659 1554 1503 1425 1330 1229 1186 1134 1065 1031 1014  993  989  980  939  936  945  995 1055 1104 1180 1247 1284 1343 1349 1403 1458 1538 1634 1790 1880 2006 2218 2396 2556 2612 2735 2811 2765 2840 2877
const double ColorTemp::ColorBlueorvokki[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0292, 0.0528, 0.0645, 0.1000, 0.0932, 0.1439, 0.1752, 0.1947, 0.2077, 0.2158, 0.2169, 0.2153, 0.2164, 0.2132, 0.2091, 0.1993, 0.1916, 0.1876, 0.1803, 0.1702, 0.1659, 0.1554,
    0.1503, 0.1425, 0.1330, 0.1229, 0.1186, 0.1134, 0.1065, 0.1031, 0.1014, 0.0993, 0.0989, 0.0980, 0.0939, 0.0936, 0.0945, 0.0995, 0.1055, 0.1104, 0.1180, 0.1247, 0.1284, 0.1343,
    0.1349, 0.1403, 0.1458, 0.1538, 0.1634, 0.1790, 0.1880, 0.2006, 0.2218, 0.2396, 0.2556, 0.2612, 0.2735, 0.2811, 0.2765, 0.2840, 0.2877,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


//#blue flower (malvikki)
//1062  528  749  571  512  538  455  445  431  384  353  299  249  212  190  162  123  105   90   81   83   75   78   72   59   56   61   54   71   69   70   62   63   65   70   74   78   73   76   87   90  104  119  119  131  145  156  184  225  255  314  414  538  669  849 1068 1247 1467 1701 1885 2032
const double ColorTemp::ColorBluemalvikki[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1062, 0.0528, 0.0749, 0.0571, 0.0512, 0.0538, 0.0455, 0.0445, 0.0431, 0.0384, 0.0353, 0.0299, 0.0249, 0.0212, 0.0190, 0.0162, 0.0123, 0.0105, 0.0090, 0.0081, 0.0083, 0.0075,
    0.0078, 0.0072, 0.0059, 0.0056, 0.0061, 0.0054, 0.0071, 0.0069, 0.0070, 0.0062, 0.0063, 0.0065, 0.0070, 0.0074, 0.0078, 0.0073, 0.0076, 0.0087, 0.0090, 0.0104, 0.0119, 0.0119,
    0.0131, 0.0145, 0.0156, 0.0184, 0.0225, 0.0255, 0.0314, 0.0414, 0.0538, 0.0669, 0.0849, 0.1068, 0.1247, 0.1467, 0.1701, 0.1885, 0.2032,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//#black dry leaf (maitohorsma)
//256    0  172  356  213  270  203  203  195  208  202  201  210  210  203  204  209  203  209  201  205  201  194  210  206  197  203  198  207  201  204  202  198  200  198  197  186  203  202  198  200  208  206  231  235  223  244  254  278  289  297  309  338  335  338  368  412  524  686  926 1185
const double ColorTemp::ColorBlackmaito[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0256,  0.0, 0.0172, 0.0356, 0.0213, 0.0270, 0.0203, 0.0203, 0.0195, 0.0208, 0.0202, 0.0201, 0.0210, 0.0210, 0.0203, 0.0204, 0.0209, 0.0203, 0.0209, 0.0201, 0.0205, 0.0201,
    0.0194, 0.0210, 0.0206, 0.0197, 0.0203, 0.0198, 0.0207, 0.0201, 0.0204, 0.0202, 0.0198, 0.0200, 0.0198, 0.0197, 0.0186, 0.0203, 0.0202, 0.0198, 0.0200, 0.0208, 0.0206, 0.0231,
    0.0235, 0.0223, 0.0244, 0.0254, 0.0278, 0.0289, 0.0297, 0.0309, 0.0338, 0.0335, 0.0338, 0.0368, 0.0412, 0.0524, 0.0686, 0.0926, 0.1185,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


//#orange berry (pihlaja)
//945  731  585  433  247  408  266  314  293  305  289  288  280  297  262  298  277  274  291  293  285  303  300  310  324  336  364  377  426  465  499  561  602  667  741  890 1028 1164 1275 1465 1602 1640 1695 1744 1812 1837 1859 1805 1791 1822 1796 1751 1715 1655 1575 1600 1560 1618 1666 1740 1838
const double ColorTemp::ColorOrangpihlaja[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0945, 0.0731, 0.0585, 0.0433, 0.0247, 0.0408, 0.0266, 0.0314, 0.0293, 0.0305, 0.0289, 0.0288, 0.0280, 0.0297, 0.0262, 0.0298, 0.0277, 0.0274, 0.0291, 0.0293, 0.0285, 0.0303,
    0.0300, 0.0310, 0.0324, 0.0336, 0.0364, 0.0377, 0.0426, 0.0465, 0.0499, 0.0561, 0.0602, 0.0667, 0.0741, 0.0890, 0.1028, 0.1164, 0.1275, 0.1465, 0.1602, 0.1640, 0.1695, 0.1744,
    0.1812, 0.1837, 0.1859, 0.1805, 0.1791, 0.1822, 0.1796, 0.1751, 0.1715, 0.1655, 0.1575, 0.1600, 0.1560, 0.1618, 0.1666, 0.1740, 0.1838,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//#green flower (lehmus=linden)
//2677 1682 1170 1032 1085  816  728  755  833  832  813  845  857  884  855  882  914  997 1084 1179 1231 1437 1661 1873 2048 2209 2378 2408 2442 2509 2503 2452 2457 2418 2383 2348 2277 2213 2221 2169 2146 2048 1977 1960 2000 1993 1961 1899 1784 1748 1625 1517 1389 1260 1165 1143 1244 1522 1870 2324 2586
//RIS 81
const double ColorTemp::ColorGreenlinden[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2677, 0.1682, 0.1170, 0.1032, 0.1085, 0.0816, 0.0728, 0.0755, 0.0833, 0.0832, 0.0813, 0.0845, 0.0857, 0.0884, 0.0855, 0.0882, 0.0914, 0.0997, 0.1084, 0.1179, 0.1231, 0.1437,
    0.1661, 0.1873, 0.2048, 0.2209, 0.2378, 0.2408, 0.2442, 0.2509, 0.2503, 0.2452, 0.2457, 0.2418, 0.2383, 0.2348, 0.2277, 0.2213, 0.2221, 0.2169, 0.2146, 0.2048, 0.1977, 0.1960,
    0.2000, 0.1993, 0.1961, 0.1899, 0.1784, 0.1748, 0.1625, 0.1517, 0.1389, 0.1260, 0.1165, 0.1143, 0.1244, 0.1522, 0.1870, 0.2324, 0.2586,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//#yellow petal (lehmus)
//1890 1097  900  832  814  799  758  853  803  808  833  862  916  943  960  969 1039 1162 1283 1370 1427 1529 1689 1781 1894 1950 2105 2118 2140 2185 2191 2199 2234 2266 2263 2297 2328 2312 2298 2332 2344 2312 2288 2347 2384 2390 2358 2280 2306 2315 2310 2253 2274 2271 2242 2292 2254 2208 2319 2314 2264
const double ColorTemp::ColorYellowlehmus[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1890, 0.1097,  0.0900, 0.0832, 0.0814, 0.0799, 0.0758, 0.0853,  0.0803,  0.0808,  0.0833,  0.0862,  0.0916,  0.0943,  0.0960,  0.0969, 0.1039, 0.1162, 0.1283, 0.1370, 0.1427,
    0.1529, 0.1689, 0.1781, 0.1894, 0.1950, 0.2105, 0.2118, 0.2140, 0.2185, 0.2191, 0.2199, 0.2234, 0.2266, 0.2263, 0.2297, 0.2328, 0.2312, 0.2298, 0.2332, 0.2344, 0.2312, 0.2288,
    0.2347, 0.2384, 0.2390, 0.2358, 0.2280, 0.2306, 0.2315, 0.2310, 0.2253, 0.2274, 0.2271, 0.2242, 0.2292, 0.2254, 0.2208, 0.2319, 0.2314, 0.2264,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//#yellow flower (suikeroalpi)
//2048 1666 1140 1210  699  680  615  566  567  561  609  585  614  572  599  575  636  730  982 1194 1360 1766 2222 2558 2849 3048 3201 3395 3395 3484 3576 3623 3606 3672 3651 3634 3647 3669 3715 3660 3720 3692 3704 3784 3683 3731 3681 3697 3635 3694 3617 3610 3632 3663 3616 3595 3599 3584 3588 3613 3527
const double ColorTemp::ColorYellowsuikeroalpi[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2048, 0.1666, 0.1140, 0.1210, 0.0699, 0.0680, 0.0615, 0.0566, 0.0567, 0.0561, 0.0609, 0.0585, 0.0614, 0.0572, 0.0599, 0.0575, 0.0636, 0.0730, 0.0982, 0.1194, 0.1360, 0.1766,
    0.2222, 0.2558, 0.2849, 0.3048, 0.3201, 0.3395, 0.3395, 0.3484, 0.3576, 0.3623, 0.3606, 0.3672, 0.3651, 0.3634, 0.3647, 0.3669, 0.3715, 0.3660, 0.3720, 0.3692, 0.3704, 0.3784,
    0.3683, 0.3731, 0.3681, 0.3697, 0.3635, 0.3694, 0.3617, 0.3610, 0.3632, 0.3663, 0.3616, 0.3595, 0.3599, 0.3584, 0.3588, 0.3613, 0.3527,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//#yellow flower (pensashanhikki)
//356 1365 1024  902  535  387  355  247  365  307  321  330  319  332  317  336  408  487  709  963 1235 1631 2111 2436 2718 2950 3151 3262 3313 3420 3448 3475 3491 3534 3520 3565 3622 3631 3626 3657 3640 3607 3641 3627 3601 3591 3588 3667 3618 3601 3630 3613 3592 3609 3569 3590 3568 3563 3588 3480 3471
const double ColorTemp::ColorYellowpensashanhikki1[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0356, 0.1365, 0.1024,  0.0902, 0.0535, 0.0387, 0.0355, 0.0247, 0.0365, 0.0307, 0.0321, 0.0330,  0.0319,  0.0332,  0.0317,  0.0336, 0.0408,  0.0487, 0.0709,  0.0963, 0.1235,
    0.1631, 0.2111, 0.2436, 0.2718, 0.2950, 0.3151, 0.3262, 0.3313, 0.3420, 0.3448, 0.3475, 0.3491, 0.3534, 0.3520, 0.3565, 0.3622, 0.3631, 0.3626, 0.3657, 0.3640, 0.3607,
    0.3641, 0.3627, 0.3601, 0.3591, 0.3588, 0.3667, 0.3618, 0.3601, 0.3630, 0.3613, 0.3592, 0.3609, 0.3569, 0.3590, 0.3568, 0.3563, 0.3588, 0.3480, 0.3471,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//#yellow sepal (pensashanhikki)
//1068  427  326  416  428  590  503  470  539  526  546  540  539  526  497  546  555  603  753  903 1010 1268 1563 1868 2068 2226 2429 2495 2560 2625 2636 2610 2655 2667 2635 2630 2612 2560 2597 2588 2543 2478 2499 2472 2438 2431 2379 2406 2361 2319 2264 2174 2128 2010 1942 1912 1930 2148 2334 2585 2764
const double ColorTemp::ColorYellowpensashanhikki2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1068, 0.0427, 0.0326, 0.0416, 0.0428, 0.0590, 0.0503, 0.0470, 0.0539, 0.0526, 0.0546, 0.0540, 0.0539, 0.0526, 0.0497, 0.0546, 0.0555, 0.0603, 0.0753, 0.0903, 0.1010, 0.1268,
    0.1563, 0.1868, 0.2068, 0.2226, 0.2429, 0.2495, 0.2560, 0.2625, 0.2636, 0.2610, 0.2655, 0.2667, 0.2635, 0.2630, 0.2612, 0.2560, 0.2597, 0.2588, 0.2543, 0.2478, 0.2499, 0.2472,
    0.2438, 0.2431, 0.2379, 0.2406, 0.2361, 0.2319, 0.2264, 0.2174, 0.2128, 0.2010, 0.1942, 0.1912, 0.1930, 0.2148, 0.2334, 0.2585, 0.2764,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//#blue flower (hiidenvirna)
//315  512  675  832  765  865  807  867  911  904  852  826  780  753  711  661  595  528  513  476  431  391  361  331  305  276  240  229  237  223  212  208  215  205  203  195  209  212  222  266  296  322  356  352  388  391  411  425  473  532  550  630  669  748  823  879  904  917  930  950  942
const double ColorTemp::ColorBluehiidenvirna[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0315, 0.0512, 0.0675, 0.0832, 0.0765, 0.0865, 0.0807, 0.0867, 0.0911, 0.0904, 0.0852, 0.0826, 0.0780, 0.0753, 0.0711, 0.0661, 0.0595, 0.0528, 0.0513, 0.0476, 0.0431, 0.0391,
    0.0361, 0.0331, 0.0305, 0.0276, 0.0240, 0.0229, 0.0237, 0.0223, 0.0212, 0.0208, 0.0215, 0.0205, 0.0203, 0.0195, 0.0209, 0.0212, 0.0222, 0.0266, 0.0296, 0.0322, 0.0356,  0.0352,
    0.0388, 0.0391, 0.0411, 0.0425, 0.0473, 0.0532, 0.0550, 0.0630, 0.0669, 0.0748, 0.0823, 0.0879, 0.0904, 0.0917, 0.0930, 0.0950, 0.0942,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//#blue flower (kurkkuyrtti)
//2687 1553 2181 2246 2209 2263 2442 2347 2261 2353 2292 2230 2095 2008 1896 1782 1569 1443 1333 1223 1177 1074  992  902  813  755  701  626  577  548  525  498  469  445  456  448  428  441  448  447  455  467  496  534  527  586  668  798  966 1126 1289 1469 1679 1870 2013 2040 2060 2077 2104 2155 2119
//RIS 87
const double ColorTemp::ColorBluekurkkuyrtti[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2687, 0.1553, 0.2181, 0.2246, 0.2209, 0.2263, 0.2442, 0.2347, 0.2261, 0.2353, 0.2292, 0.2230, 0.2095, 0.2008, 0.1896, 0.1782, 0.1569, 0.1443, 0.1333, 0.1223, 0.1177, 0.1074,
    0.0992, 0.0902, 0.0813, 0.0755, 0.0701, 0.0626, 0.0577, 0.0548, 0.0525, 0.0498, 0.0469, 0.0445, 0.0456, 0.0448, 0.0428, 0.0441, 0.0448, 0.0447, 0.0455, 0.0467, 0.0496, 0.0534,
    0.0527, 0.0586, 0.0668, 0.0798, 0.0966, 0.1126, 0.1289, 0.1469, 0.1679, 0.1870, 0.2013, 0.2040, 0.2060, 0.2077, 0.2104, 0.2155, 0.2119,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//#pink (siankaersaemoe)
//585  859  758 1094  780 1012  987 1067 1059 1034 1098 1110 1097 1040 1058 1048 1028 1014 1068 1024 1023 1025 1032 1029 1011 1007  986  973  946  906  949  923  943  949  956  998 1051 1107 1166 1242 1284 1355 1394 1438 1451 1543 1589 1588 1612 1616 1562 1534 1562 1541 1494 1492 1518 1650 1749 1907 1991
const double ColorTemp::ColorPinksiankaersaemoe[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0585, 0.0859, 0.0758, 0.1094, 0.0780, 0.1012, 0.0987, 0.1067, 0.1059, 0.1034, 0.1098, 0.1110, 0.1097, 0.1040, 0.1058, 0.1048, 0.1028, 0.1014, 0.1068, 0.1024, 0.1023, 0.1025,
    0.1032, 0.1029, 0.1011, 0.1007, 0.0986, 0.0973, 0.0946, 0.0906, 0.0949, 0.0923, 0.0943, 0.0949, 0.0956, 0.0998, 0.1051, 0.1107, 0.1166, 0.1242, 0.1284, 0.1355, 0.1394, 0.1438,
    0.1451, 0.1543, 0.1589, 0.1588, 0.1612, 0.1616, 0.1562, 0.1534, 0.1562, 0.1541, 0.1494, 0.1492, 0.1518, 0.1650, 0.1749, 0.1907, 0.1991,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


//#violet flower (harakankello)
//2520 1462 1890 1898 1751 1713 1555 1516 1471 1403 1282 1209 1144 1135 1069  976  895  823  782  762  713  685  661  635  603  559  551  550  541  567  562  574  580  589  586  620  670  690  718  801  786  769  773  739  800  806  837  845  971 1043 1102 1241 1359 1502 1611 1726 1793 1859 1909 1969 2014
//RIS 89
const double ColorTemp::ColorVioletharakankello[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2520, 0.1462, 0.1890, 0.1898, 0.1751, 0.1713, 0.1555, 0.1516, 0.1471, 0.1403, 0.1282, 0.1209, 0.1144, 0.1135, 0.1069, 0.0976, 0.0895, 0.0823, 0.0782, 0.0762, 0.0713,
    0.0685, 0.0661, 0.0635, 0.0603, 0.0559, 0.0551, 0.0550, 0.0541, 0.0567, 0.0562, 0.0574, 0.0580, 0.0589, 0.0586, 0.0620, 0.0670, 0.0690, 0.0718, 0.0801, 0.0786, 0.0769,
    0.0773, 0.0739, 0.0800, 0.0806, 0.0837, 0.0845, 0.0971, 0.1043, 0.1102, 0.1241, 0.1359, 0.1502, 0.1611, 0.1726, 0.1793, 0.1859, 0.1909, 0.1969, 0.2014,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//#violet flower (alsikeapila)
//1260  585  765 1065  937  881  847  943 1075 1053 1020  994 1008 1026 1015  980  962  949  925  908  880  864  843  814  802  749  698  691  677  660  653  660  631  633  644  692  743  809  889 1005 1160 1325 1396 1450 1526 1583 1655 1674 1689 1707 1675 1674 1624 1576 1564 1591 1613 1717 1851 1962 2033
const double ColorTemp::ColorVioletalsikeapila[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1260, 0.0585, 0.0765, 0.1065, 0.0937, 0.0881, 0.0847, 0.0943, 0.1075, 0.1053, 0.1020, 0.0994, 0.1008, 0.1026, 0.1015,  0.0980,  0.0962,  0.0949,  0.0925, 0.0908, 0.0880, 0.0864,
    0.0843, 0.0814, 0.0802, 0.0749, 0.0698, 0.0691, 0.0677, 0.0660, 0.0653, 0.0660, 0.0631, 0.0633, 0.0644, 0.0692, 0.0743, 0.0809, 0.0889, 0.1005, 0.1160, 0.1325, 0.1396, 0.1450,
    0.1526, 0.1583, 0.1655, 0.1674, 0.1689, 0.1707, 0.1675, 0.1674, 0.1624, 0.1576, 0.1564, 0.1591, 0.1613, 0.1717, 0.1851, 0.1962, 0.2033,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//#violet flower (akilleija)
//843 1340 1393 1254 1290 1452 1508 1519 1454 1384 1301 1256 1178 1113 1056  985  884  827  743  720  691  664  605  578  540  507  499  475  485  494  492  479  487  493  471  495  559  595  645  689  720  732  716  723  734  750  804  849  948 1041 1169 1362 1525 1693 1761 1935 2071 2235 2376 2493 2604
const double ColorTemp::ColorVioletakilleija[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0843, 0.1340, 0.1393, 0.1254, 0.1290, 0.1452, 0.1508, 0.1519, 0.1454, 0.1384, 0.1301, 0.1256, 0.1178, 0.1113, 0.1056, 0.0985, 0.0884, 0.0827, 0.0743, 0.0720, 0.0691, 0.0664,
    0.0605, 0.0578, 0.0540, 0.0507, 0.0499, 0.0475, 0.0485, 0.0494, 0.0492, 0.0479, 0.0487, 0.0493, 0.0471, 0.0495, 0.0559, 0.0595, 0.0645, 0.0689, 0.0720, 0.0732, 0.0716, 0.0723,
    0.0734, 0.0750, 0.0804, 0.0849, 0.0948, 0.1041, 0.1169, 0.1362, 0.1525, 0.1693, 0.1761, 0.1935, 0.2071, 0.2235, 0.2376, 0.2493, 0.2604,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//#orange flower (kehaekukka)
//0    0    0    0    0    0  102   58  128  125  137  122  122  134  123  136  159  138  163  151  167  178  192  177  206  226  315  451  707 1045 1446 1707 1944 2131 2276 2524 2719 2841 2968 3052 3199 3264 3282 3429 3451 3454 3477 3556 3478 3565 3595 3569 3582 3582 3559 3610 3626 3668 3733 3692 3722
const double ColorTemp::ColorOrangekehaekukka[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0,  0.0,  0.0,  0.0,  0.0,  0.0,  0.0102, 0.0058, 0.0128, 0.0125, 0.0137, 0.0122, 0.0122, 0.0134, 0.0123, 0.0136, 0.0159, 0.0138, 0.0163, 0.0151, 0.0167, 0.0178, 0.0192,
    0.0177, 0.0206, 0.0226, 0.0315, 0.0451, 0.0707, 0.1045, 0.1446, 0.1707, 0.1944, 0.2131, 0.2276, 0.2524, 0.2719, 0.2841, 0.2968, 0.3052, 0.3199, 0.3264, 0.3282, 0.3429,
    0.3451, 0.3454, 0.3477, 0.3556, 0.3478, 0.3565, 0.3595, 0.3569, 0.3582, 0.3582, 0.3559, 0.3610, 0.3626, 0.3668, 0.3733, 0.3692, 0.3722,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//#red berry (pihlaja)
//0    0    0    0   25   84  128   87  165  130  167  147  155  146  148  165  158  159  164  160  158  158  157  157  173  173  179  195  210  234  264  302  349  386  461  572  735  886 1038 1216 1376 1521 1607 1691 1728 1769 1842 1843 1865 1910 1881 1920 1909 1909 1891 1879 1915 1879 1878 1843 1832
const double ColorTemp::ColorRedpihlaja[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0,  0.0,  0.0,  0.0,  0.0025,  0.0084, 0.0128,  0.0087, 0.0165, 0.0130, 0.0167, 0.0147, 0.0155, 0.0146, 0.0148, 0.0165, 0.0158, 0.0159, 0.0164, 0.0160, 0.0158, 0.0158,
    0.0157, 0.0157, 0.0173, 0.0173, 0.0179, 0.0195, 0.0210, 0.0234, 0.0264, 0.0302, 0.0349, 0.0386, 0.0461, 0.0572, 0.0735, 0.0886, 0.1038, 0.1216, 0.1376, 0.1521, 0.1607, 0.1691,
    0.1728, 0.1769, 0.1842, 0.1843, 0.1865, 0.1910, 0.1881, 0.1920, 0.1909, 0.1909, 0.1891, 0.1879, 0.1915, 0.1879, 0.1878, 0.1843, 0.1832,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
//#violet flower (petunia)
//292   66  227  313  325  332  310  319  300  268  229  193  164  137  127  104   67   50   49   37   34   34   44   32   33   31   38   41   33   34   45   44   37   42   44   49   49   67   80   89  110  130  137  145  153  171  194  223  275  321  391  464  580  720  907 1055 1230 1436 1548 1777 1933
//RIS 94
const double ColorTemp::ColorVioletpetunia[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0292, 0.0066, 0.0227, 0.0313, 0.0325, 0.0332, 0.0310, 0.0319, 0.0300, 0.0268, 0.0229, 0.0193, 0.0164, 0.0137, 0.0127, 0.0104, 0.0067, 0.0050, 0.0049, 0.0037, 0.0034, 0.0034,
    0.0044, 0.0032, 0.0033, 0.0031, 0.0038, 0.0041, 0.0033, 0.0034, 0.0045, 0.0044, 0.0037, 0.0042, 0.0044, 0.0049, 0.0049, 0.0067, 0.0080, 0.0089, 0.0110, 0.0130, 0.0137, 0.0145,
    0.0153, 0.0171, 0.0194, 0.0223, 0.0275, 0.0321, 0.0391, 0.0464, 0.0580, 0.0720, 0.0907, 0.1055, 0.1230, 0.1436, 0.1548, 0.1777, 0.1933,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//#violet flower (orvokki)
//195    0  152   31   22   32   77   69   45   20   27   26   13   12   14   11   15   23   16   18   16   12   16   10   16   15   13   15   15   16   14   20   14   17   15   17   15   17   17   17   23   24   29   38   36   38   37   43   58   65   70   86  113  155  222  285  405  506  645  817 1035
const double ColorTemp::ColorVioletorvokki[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0195,  0.0, 0.0152, 0.0031, 0.0022, 0.0032, 0.0077, 0.0069, 0.0045, 0.0020, 0.0027, 0.0026, 0.0013, 0.0012, 0.0014, 0.0011, 0.0015, 0.0023, 0.0016, 0.0018, 0.0016, 0.0012, 0.0016,
    0.0010, 0.0016, 0.0015, 0.0013, 0.0015, 0.0015, 0.0016, 0.0014, 0.0020, 0.0014, 0.0017, 0.0015, 0.0017, 0.0015, 0.0017, 0.0017, 0.0017, 0.0023, 0.0024, 0.0029, 0.0038, 0.0036,
    0.0038, 0.0037, 0.0043, 0.0058, 0.0065, 0.0070, 0.0086, 0.0113, 0.0155, 0.0222, 0.0285, 0.0405, 0.0506, 0.0645, 0.0817, 0.1035,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


//#blue flower (sinisievikki)
//801 1109 1861 2325 2329 2380 2562 2565 2558 2611 2517 2567 2475 2397 2337 2294 2195 2001 1881 1892 1854 1746 1668 1580 1491 1362 1229 1178 1110 1094 1072 1019  994  960  928  879  836  859  863  951 1046 1102 1154 1193 1174 1166 1153 1199 1275 1316 1376 1550 1739 1918 2104 2228 2364 2377 2423 2394 2334
const double ColorTemp::ColorBluesinisievikki[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0801, 0.1109, 0.1861, 0.2325, 0.2329, 0.2380, 0.2562, 0.2565, 0.2558, 0.2611, 0.2517, 0.2567, 0.2475, 0.2397, 0.2337, 0.2294, 0.2195, 0.2001, 0.1881, 0.1892, 0.1854, 0.1746,
    0.1668, 0.1580, 0.1491, 0.1362, 0.1229, 0.1178, 0.1110, 0.1094, 0.1072, 0.1019,  0.0994, 0.0960, 0.0928, 0.0879, 0.0836, 0.0859, 0.0863, 0.0951, 0.1046, 0.1102, 0.1154, 0.1193,
    0.1174, 0.1166, 0.1153, 0.1199, 0.1275, 0.1316, 0.1376, 0.1550, 0.1739, 0.1918, 0.2104, 0.2228, 0.2364, 0.2377, 0.2423, 0.2394, 0.2334,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//#blue flower (iisoppi)
//623   85  605  833  776  756  755  781  774  775  697  724  697  654  617  575  536  494  460  469  442  436  400  393  380  358  369  352  342  368  357  360  342  342  341  335  355  353  365  376  382  392  412  412  407  414  420  449  487  504  517  571  651  734  806  885  968 1088 1210 1296 1411
const double ColorTemp::ColorBlueiisoppi[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0623, 0.0085, 0.0605, 0.0833, 0.0776, 0.0756, 0.0755, 0.0781, 0.0774, 0.0775, 0.0697, 0.0724, 0.0697, 0.0654, 0.0617, 0.0575, 0.0536, 0.0494, 0.0460, 0.0469, 0.0442, 0.0436,
    0.0400, 0.0393, 0.0380, 0.0358, 0.0369, 0.0352, 0.0342, 0.0368, 0.0357, 0.0360, 0.0342, 0.0342, 0.0341, 0.0335, 0.0355, 0.0353, 0.0365, 0.0376, 0.0382, 0.0392, 0.0412, 0.0412,
    0.0407, 0.0414, 0.0420, 0.0449, 0.0487, 0.0504, 0.0517, 0.0571, 0.0651, 0.0734, 0.0806, 0.0885, 0.0968, 0.1088, 0.1210, 0.1296, 0.1411,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
//#white petal (ojakaersaemoe)
//1732  951 1800 1365 1801 1697 1762 2103 2243 2218 2200 2206 2255 2254 2269 2261 2272 2251 2254 2260 2256 2266 2247 2269 2310 2273 2345 2312 2301 2323 2302 2314 2362 2355 2348 2362 2396 2374 2362 2381 2396 2440 2383 2347 2422 2419 2472 2423 2406 2425 2377 2381 2380 2398 2390 2404 2370 2375 2364 2411 2417
const double ColorTemp::ColorWhiteojaka[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1732, 0.0951, 0.1800, 0.1365, 0.1801, 0.1697, 0.1762, 0.2103, 0.2243, 0.2218, 0.2200, 0.2206, 0.2255, 0.2254, 0.2269, 0.2261, 0.2272, 0.2251, 0.2254, 0.2260, 0.2256, 0.2266,
    0.2247, 0.2269, 0.2310, 0.2273, 0.2345, 0.2312, 0.2301, 0.2323, 0.2302, 0.2314, 0.2362, 0.2355, 0.2348, 0.2362, 0.2396, 0.2374, 0.2362, 0.2381, 0.2396, 0.2440, 0.2383, 0.2347,
    0.2422, 0.2419, 0.2472, 0.2423, 0.2406, 0.2425, 0.2377, 0.2381, 0.2380, 0.2398, 0.2390, 0.2404, 0.2370, 0.2375, 0.2364, 0.2411, 0.2417,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


//#white flower (petunia)
//4095 4022 4410 4095 4095 4095 4193 4207 4388 4328 4223 4168 4221 4304 4245 4210 4212 4192 4181 4233 4207 4224 4197 4262 4243 4241 4274 4257 4204 4285 4265 4241 4267 4275 4245 4276 4260 4217 4217 4244 4240 4186 4160 4156 4227 4286 4237 4137 4202 4187 4100 4112 4103 4090 4125 4115 4098 4036 4047 4105 4050
const double ColorTemp::ColorWhitepetunia[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.4095, 0.4022, 0.4410, 0.4095, 0.4095, 0.4095, 0.4193, 0.4207, 0.4388, 0.4328, 0.4223, 0.4168, 0.4221, 0.4304, 0.4245, 0.4210, 0.4212, 0.4192, 0.4181, 0.4233, 0.4207, 0.4224,
    0.4197, 0.4262, 0.4243, 0.4241, 0.4274, 0.4257, 0.4204, 0.4285, 0.4265, 0.4241, 0.4267, 0.4275, 0.4245, 0.4276, 0.4260, 0.4217, 0.4217, 0.4244, 0.4240, 0.4186, 0.4160, 0.4156,
    0.4227, 0.4286, 0.4237, 0.4137, 0.4202, 0.4187, 0.4100, 0.4112, 0.4103, 0.4090, 0.4125, 0.4115, 0.4098, 0.4036, 0.4047, 0.4105, 0.4050,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};



//#blue flower (lobelia dortmanna)
//0  660 1277 1544 1612 1961 1909 1950 1901 1907 1809 1785 1685 1622 1522 1377 1178 1054  931  898  850  732  610  508  434  370  343  329  303  265  232  199  183  169  172  177  200  233  214  214  199  186  199  228  249  321  435  684 1006 1345 1703 2082 2432 2661 2843 2936 3079 3015 3003 3045 3038
//RIS 98
const double ColorTemp::ColorBluelobelia[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0, 0.0660, 0.1277, 0.1544, 0.1612, 0.1961, 0.1909, 0.1950, 0.1901, 0.1907, 0.1809, 0.1785, 0.1685, 0.1622, 0.1522, 0.1377, 0.1178, 0.1054, 0.0931, 0.0898, 0.0850, 0.0732,
    0.0610, 0.0508, 0.0434, 0.0370, 0.0343, 0.0329, 0.0303, 0.0265, 0.0232, 0.0199, 0.0183, 0.0169, 0.0172, 0.0177, 0.0200, 0.0233, 0.0214, 0.0214, 0.0199, 0.0186, 0.0199,
    0.0228, 0.0249, 0.0321, 0.0435, 0.0684, 0.1006, 0.1345, 0.1703, 0.2082, 0.2432, 0.2661, 0.2843, 0.2936, 0.3079, 0.3015, 0.3003, 0.3045, 0.3038,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//#white petal (pelargonia)
//3493 2882 2284 2730 2869 2609 2781 2869 2861 2869 2795 2810 2740 2716 2650 2631 2539 2554 2450 2453 2447 2451 2343 2408 2404 2367 2343 2401 2474 2549 2668 2759 2843 2883 2989 3106 3209 3344 3383 3404 3453 3521 3495 3571 3521 3548 3582 3557 3581 3539 3563 3589 3597 3579 3502 3546 3507 3554 3490 3561 3518
const double ColorTemp::ColorWhitepelargonia[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.3493, 0.2882, 0.2284, 0.2730, 0.2869, 0.2609, 0.2781, 0.2869, 0.2861, 0.2869, 0.2795, 0.2810, 0.2740, 0.2716, 0.2650, 0.2631, 0.2539, 0.2554, 0.2450, 0.2453, 0.2447,
    0.2451, 0.2343, 0.2408, 0.2404, 0.2367, 0.2343, 0.2401, 0.2474, 0.2549, 0.2668, 0.2759, 0.2843, 0.2883, 0.2989, 0.3106, 0.3209, 0.3344, 0.3383, 0.3404, 0.3453, 0.3521,
    0.3495, 0.3571, 0.3521, 0.3548, 0.3582, 0.3557, 0.3581, 0.3539, 0.3563, 0.3589, 0.3597, 0.3579, 0.3502, 0.3546, 0.3507, 0.3554, 0.3490, 0.3561, 0.3518,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

/*
#white petal (paeivaenkakkara)
2168 1365 1969 2095 2231 2530 2944 3092 3107 3148 3188 3207 3195 3216 3225 3261 3211 3228 3260 3237 3258 3276 3265 3316 3327 3291 3315 3324 3355 3255 3264 3308 3324 3328 3282 3253 3220 3257 3289 3265 3245 3297 3284 3292 3228 3312 3290 3277 3278 3284 3182 3244 3273 3291 3212 3256 3154 3243 3306 3234 3155
*/
const double ColorTemp::ColorWhitepaeivaen[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2168, 0.1365, 0.1969, 0.2095, 0.2231, 0.2530, 0.2944, 0.3092, 0.3107, 0.3148, 0.3188, 0.3207, 0.3195, 0.3216, 0.3225, 0.3261, 0.3211, 0.3228, 0.3260, 0.3237, 0.3258,
    0.3276, 0.3265, 0.3316, 0.3327, 0.3291, 0.3315, 0.3324, 0.3355, 0.3255, 0.3264, 0.3308, 0.3324, 0.3328, 0.3282, 0.3253, 0.3220, 0.3257, 0.3289, 0.3265, 0.3245, 0.3297,
    0.3284, 0.3292, 0.3228, 0.3312, 0.3290, 0.3277, 0.3278, 0.3284, 0.3182, 0.3244, 0.3273, 0.3291, 0.3212, 0.3256, 0.3154, 0.3243, 0.3306, 0.3234, 0.3155,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


//spectral data Colorchecker24 : Green B3
const double ColorTemp::ColorchechGreB3_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0531, 0.0545, 0.0559, 0.0563, 0.0566, 0.0571, 0.0576, 0.0576, 0.0575, 0.0581, 0.0586, 0.0596, 0.0606, 0.0629, 0.0652, 0.0699, 0.0745, 0.0839, 0.0932, 0.1101, 0.1270, 0.1521, 0.1771, 0.2098, 0.2424,
    0.2789, 0.3154, 0.3312, 0.3470, 0.3431, 0.3392, 0.3303, 0.3213, 0.3089, 0.2964, 0.2788, 0.2612, 0.2442, 0.2271, 0.2117, 0.1962, 0.1815, 0.1667, 0.1527, 0.1386, 0.1284, 0.1182, 0.1124, 0.1066, 0.1035, 0.1003,
    0.0987, 0.0971, 0.0961, 0.0950, 0.0950, 0.0950, 0.0962, 0.0973, 0.0994, 0.1014, 0.1045, 0.1075, 0.1106, 0.1137, 0.1157, 0.1176, 0.1175, 0.1173, 0.1173, 0.1173,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data Colorchecker24 : Cyan F3
const double ColorTemp::ColorchechCyaF3_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0813, 0.1048, 0.1282, 0.1611, 0.1940, 0.2198, 0.2456, 0.2575, 0.2693, 0.2807, 0.2921, 0.3079, 0.3237, 0.3424, 0.3611, 0.3820, 0.4029, 0.4234, 0.4439, 0.4547, 0.4654, 0.4638, 0.4621, 0.4482, 0.4342, 0.4119, 0.3895,
    0.3656, 0.3417, 0.3160, 0.2903, 0.2654, 0.2404, 0.2167, 0.1929, 0.1720, 0.1510, 0.1368, 0.1226, 0.1138, 0.1049, 0.0993, 0.0936, 0.0890, 0.0844, 0.0810, 0.0776, 0.0759, 0.0742, 0.0733, 0.0724, 0.0723, 0.0722, 0.0727,
    0.0732, 0.0745, 0.0757, 0.0763, 0.0768, 0.0764, 0.0759, 0.0748, 0.0736, 0.0723, 0.0710, 0.0703, 0.0696, 0.0707, 0.0718, 0.0756, 0.0793, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data Colorchecker24 : Purple D2
const double ColorTemp::ColorchechPurD2_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0854, 0.1047, 0.1240, 0.1468, 0.1696, 0.1826, 0.1955, 0.1963, 0.1970, 0.1910, 0.1849, 0.1750, 0.1651, 0.1541, 0.1430, 0.1322, 0.1213, 0.1117, 0.1020, 0.0944, 0.0868, 0.0809, 0.0750, 0.0703, 0.0655,
    0.0627, 0.0599, 0.0583, 0.0567, 0.0550, 0.0533, 0.0525, 0.0517, 0.0518, 0.0519, 0.0523, 0.0526, 0.0525, 0.0524, 0.0520, 0.0516, 0.0523, 0.0529, 0.0560, 0.0591, 0.0662, 0.0732, 0.0828, 0.0924, 0.1021,
    0.1117, 0.1222, 0.1327, 0.1469, 0.1610, 0.1796, 0.1981, 0.2173, 0.2365, 0.2532, 0.2698, 0.2826, 0.2953, 0.3022, 0.3090, 0.3126, 0.3161, 0.3238, 0.3314, 0.3504, 0.3694, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data Colorchecker24 : Magenta E3
const double ColorTemp::ColorchechMagE3_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1112, 0.1438, 0.1763, 0.2294, 0.2824, 0.3188, 0.3552, 0.3623, 0.3693, 0.3653, 0.3612, 0.3510, 0.3407, 0.3269, 0.3130, 0.2981, 0.2832, 0.2686, 0.2539, 0.2385, 0.2230, 0.2083, 0.1935, 0.1818, 0.1700, 0.1600, 0.1499,
    0.1394, 0.1288, 0.1188, 0.1088, 0.1051, 0.1014, 0.1026, 0.1038, 0.1041, 0.1043, 0.1064, 0.1085, 0.1225, 0.1364, 0.1701, 0.2037, 0.2532, 0.3027, 0.3587, 0.4147, 0.4683, 0.5219, 0.5672, 0.6124, 0.6455, 0.6785, 0.7009,
    0.7232, 0.7391, 0.7550, 0.7629, 0.7707, 0.7737, 0.7766, 0.7778, 0.7790, 0.7803, 0.7815, 0.7835, 0.7854, 0.7896, 0.7937, 0.8026, 0.8114, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data Colorchecker24 : Skin A1
//use also for palette WB
const double ColorTemp::ColorchechSkiA138_13_14_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0479, 0.051, 0.0553, 0.058, 0.0610, 0.062, 0.0626, 0.0622, 0.0619, 0.0617, 0.0616, 0.0615, 0.0614, 0.0614, 0.0615, 0.0617, 0.0618, 0.0618, 0.0619, 0.0618, 0.0618, 0.062, 0.0622, 0.063, 0.0638, 0.066, 0.0696,
    0.073, 0.0767, 0.078, 0.0801, 0.0807, 0.0817, 0.0831, 0.0845, 0.0870, 0.0902, 0.0955, 0.1017, 0.1096, 0.1175, 0.1250, 0.1336, 0.1385, 0.1435, 0.1455, 0.1479, 0.1490, 0.1514, 0.1547, 0.1580, 0.1625, 0.1675,
    0.173, 0.1772, 0.181, 0.1842, 0.1846, 0.1853, 0.1831, 0.1811, 0.1788, 0.1765, 0.1769, 0.1773, 0.181, 0.1834, 0.1874, 0.1914, 0.1965, 0.2018, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data Colorchecker24 : Gray C4 L=67
//use also for palette WB
const double ColorTemp::ColorchechGraC4_67_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1074, 0.1380, 0.1704, 0.22, 0.2705, 0.305, 0.3409, 0.35, 0.3601, 0.3628, 0.3655, 0.3675, 0.3698, 0.371, 0.3724, 0.373, 0.3733, 0.3725, 0.3715, 0.3705, 0.3692,
    0.369, 0.3689, 0.368, 0.3673, 0.3678, 0.3684, 0.37, 0.3711, 0.3712, 0.3714, 0.3714, 0.3714, 0.371, 0.3707, 0.37, 0.3694, 0.3697, 0.3703, 0.3697, 0.3692, 0.3688, 0.3685, 0.3675, 0.3669, 0.3657, 0.3647, 0.3635, 0.3625, 0.361,
    0.3596, 0.3585, 0.3579, 0.357, 0.3560, 0.3555, 0.3548, 0.3535, 0.3526, 0.3513, 0.3500, 0.349, 0.3475, 0.3467, 0.3460, 0.3452, 0.3444, 0.3431, 0.3421, 0.3411, 0.3403, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Fictif_61greyspect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1,
    0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1,
    0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::JDC468_K15_87greyspect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1984, 0.2223, 0.2448, 0.2934, 0.3415, 0.4425, 0.5707, 0.6609, 0.7619, 0.7956, 0.8275, 0.8280, 0.8292, 0.8223, 0.8156, 0.8112, 0.8076, 0.8040, 0.7982, 0.7970, 0.7954, 0.8013, 0.8083, 0.8141, 0.8184, 0.8167,
    0.8137, 0.8080, 0.8026, 0.8013, 0.7988, 0.7963, 0.7942, 0.7855, 0.7765, 0.7680, 0.7603, 0.7640, 0.7681, 0.7750, 0.7827, 0.7876, 0.7923, 0.7935, 0.7945, 0.7955, 0.7964, 0.7975, 0.7982, 0.8000, 0.8017, 0.8051,
    0.8090, 0.8145, 0.8191, 0.8234, 0.8269, 0.8300, 0.8327, 0.8342, 0.8359, 0.8375, 0.8390, 0.8405, 0.8421, 0.8436, 0.8452, 0.8480, 0.8504, 0.8564, 0.8611,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0

};
//K15 275   275 0.1984  0.2448  0.3415  0.5707  0.7619  0.8275  0.8292  0.8156  0.8076  0.7982  0.7954  0.8083  0.8184  0.8137  0.8026  0.7988  0.7942  0.7765  0.7603  0.7681  0.7827  0.7923  0.7945  0.7964  0.7982  0.8017  0.8090  0.8191  0.8269  0.8327  0.8359  0.8390  0.8421  0.8452  0.8504  0.8611

//spectral data Colorchecker24 : Skin B1
//use also for palette WB
const double ColorTemp::ColorchechSkiB166_18_18_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0962, 0.114, 0.1328, 0.152, 0.1706, 0.1755, 0.1877, 0.189, 0.1903, 0.1913, 0.1923, 0.1946, 0.1971, 0.2015, 0.2064, 0.215, 0.2245, 0.239, 0.2535, 0.273, 0.2922, 0.31, 0.3274, 0.337, 0.3473,
    0.348, 0.3489, 0.335, 0.3224, 0.303, 0.2835, 0.275, 0.2671, 0.27, 0.2728, 0.2735, 0.2741, 0.279, 0.2836, 0.308, 0.3334, 0.375, 0.4183, 0.457, 0.4950, 0.516, 0.5409, 0.5515, 0.5625, 0.568, 0.5731, 0.5786,
    0.5820, 0.586, 0.5902, 0.596, 0.6025, 0.611, 0.6174, 0.627, 0.6375, 0.65, 0.6626, 0.677, 0.6910, 0.704, 0.7171, 0.723, 0.7339, 0.741, 0.7475,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data Colorchecker24 : blue sky C1
//use also for palette WB
const double ColorTemp::ColorchechBluC150_m5_m22_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1053, 0.134, 0.1633, 0.2075, 0.2518, 0.283, 0.3163, 0.324, 0.3325, 0.334, 0.3355, 0.3352, 0.3349, 0.332, 0.3294, 0.325, 0.3199, 0.3127, 0.3055, 0.2955, 0.2863, 0.28, 0.2737, 0.267, 0.2612, 0.249, 0.2378, 0.228, 0.2199,
    0.218, 0.2173, 0.2146, 0.2118, 0.20, 0.1884, 0.178, 0.1682, 0.166, 0.1639, 0.162, 0.1613, 0.158, 0.1550, 0.1504, 0.1458, 0.1415, 0.1375, 0.135, 0.1327, 0.1316, 0.1305, 0.1304, 0.1302, 0.131, 0.1322, 0.1342, 0.1362,
    0.1367, 0.1372, 0.1356, 0.1340, 0.1311, 0.1288, 0.1253, 0.1227, 0.1205, 0.1187, 0.1195, 0.1205, 0.1255, 0.1303, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorcheckerDC : blue sky N8
//use also for palette WB
const double ColorTemp::ColorchechDCBluN881_m7_m14_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1371, 0.17, 0.2029, 0.291, 0.3790, 0.495, 0.6100, 0.67, 0.7249, 0.737, 0.7501, 0.7545, 0.7597, 0.764, 0.7677, 0.7685, 0.7693, 0.7677, 0.7662, 0.763, 0.7593, 0.753, 0.7471, 0.737, 0.7289, 0.718, 0.7077, 0.705, 0.6819, 0.666, 0.6515, 0.636, 0.6244,
    0.61, 0.5948, 0.577, 0.5581, 0.544, 0.5293, 0.522, 0.5147, 0.512, 0.5091, 0.506, 0.5029, 0.499, 0.4950, 0.494, 0.4931, 0.497, 0.5007, 0.508, 0.5176, 0.527, 0.5359, 0.542, 0.5487, 0.549, 0.5494, 0.544,
    0.5375, 0.531, 0.5244, 0.522, 0.5207, 0.524, 0.5264, 0.532, 0.5369, 0.542, 0.5505, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorcheckerSG : Skin F7
//use also for palette WB
const double ColorTemp::ColorchechSGSkiF763_14_26_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0508, 0.064, 0.0776, 0.903, 0.1099, 0.1128, 0.1256, 0.128, 0.1307, 0.133, 0.1357, 0.139, 0.1425, 0.148, 0.1523, 0.159, 0.1669, 0.177, 0.1871, 0.20, 0.2118, 0.2235, 0.2355, 0.2445, 0.2537, 0.259, 0.2655, 0.268,
    0.2700, 0.2708, 0.2716, 0.2743, 0.2770, 0.2803, 0.2827, 0.283, 0.2832, 0.283, 0.2828, 0.295, 0.3079, 0.344, 0.3803, 0.4105, 0.4409, 0.455, 0.4694, 0.477, 0.4851, 0.4896, 0.4962, 0.501, 0.5066, 0.511, 0.5160, 0.521,
    0.5256, 0.529, 0.5318, 0.535, 0.5383, 0.541, 0.5451, 0.549, 0.5524, 0.556, 0.5597, 0.562, 0.5650, 0.568, 0.5709, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorcheckerSG : Skin K2 85 11 17
//use also for palette WB
const double ColorTemp::ColorchechSGSkiK285_11_17_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1122, 0.149, 0.1866, 0.259, 0.3318, 0.393, 0.4547, 0.469, 0.4846, 0.4845, 0.4844, 0.4838, 0.4834, 0.4837, 0.4840, 0.4847, 0.4854, 0.4852, 0.4849, 0.4842, 0.4835, 0.4832, 0.4828, 0.485,
    0.4874, 0.501, 0.5150, 0.536, 0.5572, 0.5685, 0.5798, 0.586, 0.5932, 0.5987, 0.6142, 0.6342, 0.6541, 0.683, 0.7119, 0.734, 0.7571, 0.769, 0.7829, 0.788, 0.7932, 0.795, 0.7968, 0.7973,
    0.7977, 0.7974, 0.7969, 0.797, 0.7972, 0.7973, 0.7975, 0.7983, 0.7990, 0.7978, 0.7965, 0.7957, 0.7949, 0.7944, 0.7940, 0.794, 0.7941, 0.7943, 0.7946, 0.7938, 0.7930, 0.7929, 0.7928,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data Colorcheck24 : White A4 L=96
//use also for palette WB
const double ColorTemp::ColorchechWhiA496_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1267, 0.172, 0.2179, 0.317, 0.4164, 0.505, 0.6780, 0.758, 0.8397, 0.865, 0.8911, 0.897, 0.9035, 0.9062, 0.9092, 0.9124, 0.9154, 0.9167, 0.9180, 0.9187, 0.9194, 0.92, 0.9225, 0.9217, 0.9209, 0.921,
    0.9212, 0.9227, 0.9242, 0.9235, 0.9227, 0.9232, 0.9238, 0.9243, 0.9248, 0.9237, 0.9227, 0.9239, 0.9252, 0.924, 0.9233, 0.9238, 0.9242, 0.924, 0.9239, 0.9239, 0.9239, 0.924, 0.9242, 0.9239,
    0.9237, 0.925, 0.9264, 0.9276, 0.9288, 0.9298, 0.9308, 0.9296, 0.9284, 0.928, 0.9276, 0.928, 0.9284, 0.9294, 0.9304, 0.9316, 0.9328, 0.9328, 0.9328, 0.9337, 0.9345,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data Colorcheck24 : foliage Green D1
const double ColorTemp::ColorchechGreD1_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0477, 0.0492, 0.0507, 0.0517, 0.0527, 0.0532, 0.0537, 0.054, 0.0544, 0.0554, 0.0563, 0.0573, 0.0584, 0.0592, 0.0601, 0.0607, 0.0611, 0.0613, 0.0619, 0.626, 0.0634, 0.0646, 0.0659, 0.069,
    0.0721, 0.0837, 0.0958, 0.117, 0.1386, 0.156, 0.1748, 0.1802, 0.1855, 0.1795, 0.1742, 0.1636, 0.1529, 0.144, 0.1351, 0.13, 0.1239, 0.1202, 0.1171, 0.1138, 0.1106, 0.108, 0.1048, 0.1035,
    0.1022, 0.1025, 0.1030, 0.1041, 0.1052, 0.106, 0.1068, 0.107, 0.1073, 0.1066, 0.1060, 0.1047, 0.1035, 0.1028, 0.1021, 0.1029, 0.1034, 0.105, 0.1069, 0.1086, 0.1104, 0.1127, 0.1150,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorchecSG : black N3 L=6
//use also for palette WB
const double ColorTemp::ColorchechSGBlaN3_6_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0066, 0.0069, 0.0071, 0.0072, 0.0074, 0.0073, 0.0072, 0.0073, 0.0074, 0.0074, 0.0074, 0.0074, 0.0074, 0.0073, 0.0073, 0.0073, 0.0073, 0.0072, 0.0072, 0.0072, 0.0072, 0.0071, 0.0071, 0.0071,
    0.0071, 0.0071, 0.0071, 0.0071, 0.0071, 0.0071, 0.0071, 0.0071, 0.0071, 0.0071, 0.0071, 0.0071, 0.0070, 0.0070, 0.0071, 0.0071, 0.0071, 0.0071, 0.0071, 0.0071, 0.0071, 0.0070,
    0.0070, 0.0070, 0.0070, 0.0069, 0.0069, 0.0069, 0.0068, 0.0068, 0.0068, 0.0068, 0.0068, 0.0068, 0.0067, 0.0067, 0.0067, 0.0067, 0.0066, 0.0066, 0.0066, 0.0066, 0.0066, 0.0066, 0.0066, 0.0067,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data 468 color : gray K14 L=44
//use also for palette WB
const double ColorTemp::JDC468_GraK14_44_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.04240, 0.0485, 0.05500, 0.0624, 0.06930, 0.084, 0.09820, 0.109, 0.12160, 0.127, 0.13300, 0.13490, 0.13690, 0.1379, 0.13890, 0.1396, 0.14060, 0.1407, 0.14080, 0.1423, 0.14380, 0.1488, 0.15370, 0.157, 0.16040,
    0.157, 0.15360, 0.1482, 0.14290, 0.1436, 0.14470, 0.1454, 0.14620, 0.137, 0.12870, 0.1205, 0.11250, 0.116, 0.11930, 0.1261, 0.13350, 0.1367, 0.13990, 0.139, 0.13810, 0.1371, 0.13610, 0.1372, 0.13820,
    0.1408, 0.14330, 0.1475, 0.15170, 0.1583, 0.16500, 0.172, 0.17940, 0.1836, 0.18780, 0.188, 0.18820, 0.186, 0.18430, 0.1801, 0.17620, 0.1741, 0.17210, 0.179, 0.18420, 0.1991, 0.21430,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::JDC468_BluM5_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1510, 0.1802, 0.2069, 0.2550,  0.3047, 0.4055, 0.5069, 0.590, 0.6747, 0.701,   0.7351, 0.7345, 0.7338, 0.7195, 0.7063, 0.693,  0.6732, 0.6490, 0.6261, 0.5993, 0.5723, 0.5560,
    0.5401, 0.526, 0.5106, 0.4805,  0.4504, 0.42, 0.3907, 0.385, 0.3799, 0.3750,    0.3695, 0.3340,  0.3005, 0.2692, 0.2382, 0.2387, 0.2389, 0.2501, 0.2610, 0.2635, 0.2662, 0.2601,  0.2541,
    0.2450, 0.2426, 0.2430, 0.2434, 0.2490, 0.2523, 0.2612, 0.2692, 0.2694, 0.2996, 0.3145, 0.3329, 0.3413, 0.3498, 0.3467, 0.3442, 0.3355, 0.3266, 0.3131, 0.2996, 0.2911, 0.2831, 0.2950, 0.3070, 0.3430, 0.3799,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//m5    317 //0.1510    0.2069  0.3047  0.5069  0.6747  0.7351  0.7338  0.7063  0.6732  0.6261  0.5723  0.5401
//  0.5106  0.4504  0.3907  0.3799  0.3695  0.3005  0.2382  0.2389  0.2610  0.2662  0.2541
//  0.2426  0.2434  0.2523  0.2692  0.2996  0.3329  0.3498  0.3442  0.3266  0.2996  0.2831  0.3070  0.3799

const double ColorTemp::JDC468_RedG21va_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1207, 0.141, 0.1585, 0.1810, 0.2073, 0.2529,  0.2959,  0.3210, 0.3476, 0.3350, 0.3232, 0.2845,  0.2564, 0.2140,   0.1823, 0.1523, 0.1266, 0.1001, 0.0792, 0.061,  0.0439, 0.0349,  0.0295, 0.0260,  0.0222,
    0.0180, 0.0135, 0.0111,  0.0087, 0.0090,  0.0094, 0.0101,  0.0109, 0.0093,  0.0086, 0.0090,  0.0091, 0.0061,  0.0321, 0.0086,  0.1368, 0.2312,  0.3256, 0.4112,  0.4958, 0.5444,  0.5884, 0.6002,  0.6264,  0.6323, 0.6473,
    0.6546, 0.6659, 0.6775,  0.6881, 0.6982,  0.7081, 0.7150,  0.7201, 0.7217,  0.7232, 0.7222,  0.7215,  0.7187, 0.7157, 0.7144,  0.7131, 0.7196,  0.7269, 0.7303, 0.7599,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
//g21 177   0.1207  0.1585  0.2073  0.2959  0.3476  0.3232  0.2564  0.1823  0.1266  0.0792  0.0439  0.0295  0.0222  0.0135  0.0087  0.0094  0.0109  0.0086  0.0091  0.0321
//  0.1368  0.3256  0.4958  0.5884  0.6264  0.6473  0.6659  0.6881  0.7081  0.7201  0.7232  0.7215  0.7157  0.7131  0.7269  0.7599
const double ColorTemp::JDC468_RedI9_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0258, 0.023,  0.0220, 0.0205,  0.0189, 0.0183,  0.0174, 0.0168,  0.0162, 0.0152,  0.0148, 0.0145,  0.0139, 0.0136,  0.0133, 0.0130,  0.0127, 0.0130,  0.0133, 0.0151,  0.0168, 0.0218,  0.0268, 0.0317,  0.0367, 0.0330,
    0.0313, 0.0270,  0.0227, 0.0240,  0.0255, 0.0280,  0.0302, 0.0280,  0.0225, 0.0215,  0.0209, 0.0424,  0.0639, 0.1401,  0.2131, 0.3250,  0.4369, 0.5210,  0.6265, 0.6795,  0.7336, 0.7551,  0.7784, 0.7890,  0.7994, 0.8070,
    0.8146, 0.8210,  0.8277, 0.8321,  0.8362, 0.8398,  0.8439, 0.8470,  0.8504, 0.8530,  0.8572,  0.8612, 0.8653, 0.8689,  0.8715, 0.8730,  0.8747, 0.8766,  0.8788,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//I9 RED 217    0.0258  0.0220  0.0189  0.0174  0.0162  0.0148  0.0139  0.0133  0.0127  0.0133  0.0168  0.0268  0.0367  0.0313  0.0227  0.0255  0.0302  0.0225  0.0209  0.0639  0.2131  0.4369  0.6265  0.7336  0.7784  0.7994  0.8146  0.8277  0.8362  0.8439  0.8504  0.8572  0.8653  0.8715  0.8747  0.8788

const double ColorTemp::JDC468_YelN10_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0531, 0.0520,  0.0504, 0.0510,  0.0518, 0.0608,  0.0628, 0.0669,  0.0699, 0.0705,  0.0716, 0.0720,  0.0735, 0.0755,  0.0775, 0.0800,  0.0825, 0.0896,  0.0969, 0.1260, 0.1563, 0.2312,  0.3096, 0.4132,  0.5177, 0.5905,  0.6637,
    0.7251, 0.7350,  0.7458, 0.7480,  0.7507, 0.7460,  0.7414, 0.7356,  0.7301, 0.7320,  0.7347, 0.7390,  0.7438, 0.7472,  0.7500, 0.7508,  0.7515, 0.7528,  0.7538, 0.7550,  0.7563, 0.7581,  0.7607, 0.7642,  0.7686, 0.7710,
    0.7791, 0.7840,  0.7872, 0.7902,  0.7935, 0.7955,  0.7979, 0.7995,  0.8021, 0.8035,  0.8058, 0.8072,  0.8090, 0.8110,  0.8143, 0.8198,  0.8259,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


//n10   348 0.0531  0.0504  0.0518  0.0628  0.0699  0.0716  0.0735  0.0775  0.0825  0.0969  0.1563  0.3096  0.5177  0.6637  0.7251  0.7458  0.7507  0.7414  0.7301  0.7347  0.7438  0.7500  0.7515  0.7538  0.7563  0.7607  0.7686  0.7791  0.7872  0.7935  0.7979  0.8021  0.8058  0.8090  0.8143  0.8259
const double ColorTemp::JDC468_GreN7_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0112, 0.0102,  0.0094, 0.0096,  0.0099, 0.0100,  0.0100, 0.0100,  0.0100, 0.0099,  0.0099, 0.0099,  0.0099, 0.0099,  0.0099, 0.0100,  0.0100, 0.0103,  0.0107, 0.0129,  0.0151, 0.0312,  0.0462, 0.1015,  0.1571, 0.2270,  0.2977,
    0.3558, 0.3441,  0.3321, 0.3020,  0.2710, 0.2312,  0.1954, 0.1602,  0.1251, 0.1003,  0.0794, 0.0672,  0.0563, 0.0513,  0.0452,  0.0418, 0.0378, 0.0356,  0.0337, 0.0336,  0.0335, 0.0345,  0.0358, 0.0383,  0.0405, 0.0445,  0.0497,
    0.0612, 0.0647,  0.0670, 0.0660,  0.0644, 0.0620,  0.0574, 0.0525,  0.0483, 0.0460,  0.0436, 0.0484,  0.0532, 0.0690,  0.0870,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//n7    345 0.0112  0.0094  0.0099  0.0100  0.0100  0.0099  0.0099  0.0099  0.0100  0.0107  0.0151  0.0462  0.1571  0.2977  0.3558  0.3321  0.2710  0.1954  0.1251  0.0794  0.0563  0.0452  0.0378  0.0337  0.0335  0.0358  0.0405  0.0497  0.0612  0.0670  0.0644  0.0574  0.0483  0.0436  0.0532  0.0870

const double ColorTemp::JDC468_GreA10_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0958, 0.1010,  0.1090, 0.1172,  0.1352, 0.1954,  0.1957, 0.2178,  0.2402, 0.2477,  0.2553, 0.2594,  0.2622, 0.2667,  0.2707, 0.2760,  0.2805, 0.2913,  0.3023, 0.3376,  0.3715, 0.4345,  0.5030, 0.5702,  0.6376, 0.6724,  0.7072,
    0.7216, 0.7160,  0.7110, 0.6990,  0.6865, 0.6667,  0.6446, 0.6174,  0.5921, 0.5727,  0.5511, 0.5386,  0.5238, 0.5134,  0.5070, 0.4980,  0.4918, 0.4867,  0.4830, 0.4834,  0.4838, 0.4889,  0.4906, 0.4976,  0.5046, 0.5162,  0.5279,
    0.5519, 0.5589,  0.5649, 0.5645,  0.5639, 0.5576,  0.5552, 0.5480,  0.5407, 0.5377,  0.5326, 0.5387,  0.5498, 0.5732,  0.5966,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_GreQ7_spect[97] = { //468 Q7
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0149, 0.0139, 0.0133, 0.0128, 0.0123, 0.01185, 0.0114, 0.011, 0.0106, 0.01045,    0.0103, 0.0104, 0.0105, 0.01065,    0.0108, 0.0109, 0.0110, 0.0117, 0.0124, 0.0205, 0.0283, 0.071,  0.1122, 0.2051,
    0.3017, 0.385, 0.4690, 0.4880,   0.5069, 0.4811, 0.4561, 0.411,  0.3687, 0.3185, 0.2673, 0.2190, 0.1703, 0.1371, 0.1042, 0.086,  0.0695, 0.598,  0.0527, 0.0475, 0.04210, 0.039, 0.03600, 0.0357,    0.0355, 0.037,
    0.0381, 0.0405, 0.0438, 0.0499, 0.0555, 0.0635, 0.0708, 0.074,  0.0789, 0.077,  0.0751, 0.071,  0.0658, 0.059,  0.0530, 0.049,  0.0458, 0.052,  0.0570, 0.077,  0.0982,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//a10  Green 10  0.0958  0.1090  0.1352  0.1957  0.2402  0.2553  0.2622  0.2707  0.2805  0.3023  0.3715  0.5030  0.6376  0.7072  0.7216  0.7110  0.6865  0.6446  0.5921  0.5511  0.5238  0.5070  0.4918  0.4830  0.4838  0.4906  0.5046  0.5279  0.5519  0.5649  0.5639  0.5552  0.5407  0.5326  0.5498  0.5966
const double ColorTemp::JDC468_GreK7_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0114, 0.0111,  0.0109, 0.0107,  0.0105, 0.0106,  0.0108, 0.0107,  0.0106, 0.0105,  0.0104, 0.0103,  0.0103, 0.0106,  0.0109, 0.0112,  0.0118, 0.0135,  0.0153, 0.0244,  0.0334, 0.0666,  0.0984, 0.1534,  0.2082, 0.2412,  0.2835,
    0.2959, 0.2843,  0.2735, 0.2516,  0.2305, 0.2012,  0.1728, 0.1435,  0.1156, 0.0964,  0.0772, 0.0671,  0.0570, 0.0518,  0.0468, 0.0436,  0.0397, 0.0380,  0.0354, 0.0354,  0.0355, 0.0367,  0.0380, 0.0402,  0.0426, 0.0481,  0.0523,
    0.0643, 0.0678,  0.0704, 0.0693,  0.0676, 0.0639,  0.0609, 0.0567,  0.0514, 0.0487,  0.0468, 0.0518,  0.0567, 0.0730,  0.0902,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//k7  Green  267 0.0114  0.0109  0.0105  0.0108  0.0106  0.0104  0.0103  0.0109  0.0118  0.0153  0.0334  0.0984  0.2082  0.2835  0.2959  0.2735  0.2305  0.1728  0.1156  0.0772  0.0570  0.0468  0.0397  0.0354  0.0355  0.0380  0.0426  0.0523  0.0643  0.0704  0.0676  0.0609  0.0514  0.0468  0.0567  0.0902

const double ColorTemp::JDC468_PurE24_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0677, 0.901,  0.1043, 0.1298,  0.1534, 0.1913,  0.2297, 0.2553,  0.2756, 0.2789,  0.2620, 0.2380,  0.2135, 0.1837,  0.1536, 0.1312,  0.1068, 0.0867,  0.0663, 0.0517,  0.0368, 0.0309,  0.0247, 0.0214,  0.0186, 0.0151,  0.0116,
    0.0077, 0.0079,  0.0079, 0.0083,  0.0086, 0.0077,  0.0071, 0.0071,  0.0072, 0.0107,  0.0147, 0.0298,  0.0440, 0.0661,  0.0880, 0.1010,  0.1152, 0.1193,  0.1236, 0.1260,  0.1287, 0.1326,  0.1366, 0.1428,  0.1489, 0.1596,  0.1697,
    0.1936, 0.1996,  0.2057, 0.2036,  0.2015, 0.1954, 0.1890, 0.1798,  0.1706, 0.1651,  0.1603, 0.1692,  0.1788, 0.2075,  0.2363,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
//E24  Pur 128 0.0677  0.1043  0.1534  0.2297  0.2756  0.2620  0.2135  0.1536  0.1068  0.0663  0.0368  0.0247  0.0186  0.0116  0.0077  0.0079  0.0086  0.0071  0.0072  0.0147  0.0440  0.0880  0.1152  0.1236  0.1287  0.1366  0.1489  0.1697  0.1936  0.2057  0.2015  0.1890  0.1706  0.1603  0.1788  0.2363


//spectral data 468 color : Blue H10 - Gamut > WidegamutRGB
const double ColorTemp::JDC468_BluH10_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.01590, 0.028, 0.03970, 0.0697, 0.09970, 0.1526, 0.20550, 0.253, 0.30110, 0.3412, 0.38180, 0.423, 0.46610, 0.4683, 0.51030, 0.5005, 0.49950, 0.4785, 0.45810, 0.429, 0.39950, 0.374, 0.35010, 0.3135, 0.29630,
    0.2587, 0.22070, 0.182, 0.14450, 0.1125, 0.09060, 0.072, 0.04810, 0.033, 0.01740, 0.0113, 0.00520, 0.004, 0.00290, 0.0028, 0.00270, 0.0027, 0.00270, 0.0027, 0.00280, 0.0027, 0.00270, 0.0028, 0.00280,
    0.0029, 0.00300, 0.0029, 0.00290, 0.0029, 0.0029, 0.00290, 0.0029, 0.00290, 0.0029, 0.00290, 0.0029, 0.00290, 0.0029, 0.00290, 0.0029, 0.0031, 0.00320, 0.0035, 0.00380, 0.0047, 0.00560,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
//0.0159, 0.028,    0.0397, 0.0697, 0.0997, 0.1526, 0.2055, 0.253, 0.3011,  0.3412, 0.3818, 0.423,  0.4661, 0.5103  0.4995  0.4581  0.3995  0.3501  0.2963
//0.2207    0.1445  0.0906  0.0481  0.0174  0.0052  0.0029  0.0027  0.0027  0.0028  0.0027  0.0028  0.0030  0.0029  0.0029  0.0029  0.0029  0.0029  0.0029
//0.0029    0.0032  0.0038  0.0056

const double ColorTemp::JDC468_BluD6_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1127, 0.143, 0.1773, 0.223,   0.2813, 0.3987, 0.4782, 0.5665, 0.6470, 0.6870, 0.7270, 0.7403, 0.7593, 0.7592, 0.7591, 0.7480, 0.7402, 0.7234, 0.7054, 0.6876, 0.6617, 0.6512, 0.6302, 0.6124, 0.5962, 0.5660,
    0.5352, 0.5009, 0.4655, 0.4356, 0.4191, 0.3923, 0.3619, 0.3145, 0.2653, 0.2245, 0.1744, 0.1499, 0.1255, 0.1124, 0.1014, 0.0972, 0.0855, 0.0786, 0.0715, 0.0659, 0.0626, 0.0625, 0.0624, 0.0645, 0.0670, 0.0714,
    0.0769, 0.0865, 0.0964, 0.1086, 0.1200, 0.123,  0.1327, 0.1309, 0.1281, 0.1214, 0.1146, 0.1023, 0.0950, 0.0901, 0.0839, 0.0918, 0.1009, 0.1260, 0.1597,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0


};
//d6 blue   84  0.1127  0.1773  0.2813  0.4782  0.6470  0.7270  0.7593  0.7591  0.7402  0.7054  0.6617  0.6302  0.5962  0.5352  0.4655  0.4191  0.3619  0.2653  0.1744  0.1255  0.1014  0.0855  0.0715  0.0626  0.0624  0.0670  0.0769  0.0964  0.1200  0.1327  0.1281  0.1146  0.0950  0.0839  0.1009  0.1597
const double ColorTemp::JDC468_BluF4_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0180, 0.0270, 0.0324, 0.0453, 0.0611, 0.0845, 0.1066, 0.1234, 0.1446, 0.1567, 0.1718, 0.1867, 0.1954, 0.2024, 0.2083, 0.2090, 0.2096, 0.2060, 0.2036, 0.1990, 0.1947, 0.1920, 0.1901, 0.1856, 0.1794, 0.1667, 0.1516, 0.1321,
    0.1167, 0.1032, 0.0876, 0.0730, 0.0584, 0.0445, 0.0296, 0.0212, 0.0125, 0.0099, 0.0069, 0.0060, 0.0053, 0.0050, 0.0049, 0.0047, 0.0046, 0.0045, 0.0044, 0.0043, 0.0043, 0.0043, 0.0043, 0.0046, 0.0049, 0.0050, 0.0052, 0.0057,
    0.0063, 0.0066, 0.0069, 0.0067, 0.0066, 0.0063, 0.0059, 0.0056, 0.0053, 0.0054, 0.0055, 0.0062, 0.0069, 0.0099, 0.0122,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0

};

// f4 blue  134 0.0180  0.0324  0.0611  0.1066  0.1446  0.1718  0.1954  0.2083  0.2096  0.2036  0.1947  0.1901  0.1794  0.1516  0.1167  0.0876  0.0584  0.0296  0.0125  0.0069  0.0053  0.0049  0.0046  0.0044  0.0043  0.0043  0.0049  0.0052  0.0063  0.0069  0.0066  0.0059  0.0053  0.0055  0.0069  0.0122
const double ColorTemp::JDC468_GreI8_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0230, 0.0232, 0.0234, 0.0254, 0.0263, 0.0298, 0.0329, 0.0367, 0.0377, 0.0388, 0.0399, 0.0410, 0.0421, 0.0440, 0.0460, 0.0481, 0.0496, 0.0523, 0.0559, 0.0645, 0.0727, 0.0878, 0.1020, 0.1156, 0.1288, 0.1334, 0.1394, 0.1398,
    0.1402, 0.1407, 0.1413, 0.1409, 0.1396, 0.1334, 0.1276, 0.1200, 0.1129, 0.1095, 0.1064, 0.1053, 0.1043, 0.1031, 0.1021, 0.1001, 0.0980, 0.0970, 0.0952, 0.0963, 0.0967, 0.0990, 0.1009, 0.1042, 0.1078, 0.1130, 0.1188, 0.1251,
    0.1307, 0.1335, 0.1374, 0.1376, 0.1378, 0.1362, 0.1345, 0.1312, 0.1278, 0.1257, 0.1240, 0.1290, 0.1345, 0.1476, 0.1615,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0

};

// i8 green215  215 0.0230  0.0234  0.0263  0.0329  0.0377  0.0399  0.0421  0.0460  0.0496  0.0559  0.0727  0.1020  0.1288  0.1394  0.1402  0.1413  0.1396  0.1276  0.1129  0.1064  0.1043  0.1021  0.0980  0.0952  0.0967  0.1009  0.1078  0.1188  0.1307  0.1374  0.1378  0.1345  0.1278  0.1240  0.1345  0.1615

const double ColorTemp::JDC468_OraO18_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0826, 0.0830, 0.0832, 0.0861, 0.0892, 0.0993, 0.1108, 0.1180, 0.1248, 0.1253, 0.1263, 0.1261, 0.1259, 0.1267, 0.1289, 0.1304, 0.1319, 0.1370, 0.1419, 0.1631, 0.1851, 0.2311, 0.2743, 0.3131, 0.3536, 0.3551, 0.3585, 0.3488, 0.3322,
    0.3470, 0.3575, 0.3680, 0.3498, 0.3316, 0.3224, 0.3129, 0.3578, 0.4013, 0.4734, 0.5454, 0.5978, 0.6502, 0.6745, 0.6982, 0.7080, 0.7182, 0.7273, 0.7269, 0.7308, 0.7342, 0.7393, 0.7436, 0.7498, 0.7550, 0.7597, 0.7640, 0.7680, 0.7713,
    0.7766, 0.7786, 0.7816, 0.7841, 0.7863, 0.7889, 0.7902, 0.7931, 0.7957, 0.7997, 0.8068,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0

};
// o18 ora 382  382 0.0826  0.0832  0.0892  0.1108  0.1248  0.1263  0.1259  0.1289  0.1319  0.1419  0.1851  0.2743  0.3536  0.3585  0.3322  0.3470  0.3680  0.3316  0.3129  0.4013  0.5454  0.6502  0.6982  0.7182  0.7269  0.7342  0.7436  0.7550  0.7640  0.7713  0.7766  0.7816  0.7863  0.7902  0.7957  0.8068
const double ColorTemp::JDC468_OraD17_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0462, 0.0442,  0.0422, 0.0401,  0.0383, 0.0390,  0.0396, 0.0396,  0.0395, 0.0388,  0.0380, 0.0378,  0.0376, 0.0381,  0.0384, 0.0391,  0.0399, 0.0421,  0.0451, 0.0561,  0.0676,  0.0934, 0.1189, 0.1432,  0.1671, 0.1650,  0.1632, 0.1512,  0.1402, 0.1456,
    0.1521, 0.1613,  0.1696, 0.1552,  0.1409,  0.1342, 0.1283, 0.1689,  0.2084, 0.2845,  0.3575, 0.4183,  0.4797, 0.5090,  0.5389, 0.5498,  0.5617, 0.5667,  0.5728, 0.5788,  0.5822, 0.5889,  0.5938, 0.6011,  0.6081, 0.6145,  0.6212, 0.6267,  0.6304, 0.6331,
    0.6352, 0.6361,  0.6373, 0.6372,  0.6370, 0.6376,  0.6384, 0.6413,  0.6483, 0.6523,  0.6668,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0

};

// d17 ora 95   95  0.0462  0.0422  0.0383  0.0396  0.0395  0.0380  0.0376  0.0384  0.0399  0.0451  0.0676  0.1189  0.1671  0.1632  0.1402  0.1521  0.1696  0.1409  0.1283  0.2084  0.3575  0.4797  0.5389  0.5617  0.5728  0.5822  0.5938  0.6081  0.6212  0.6304  0.6352  0.6373  0.6370  0.6384  0.6483  0.6668

//spectral data ColorLab : Skin 35 15 17
const double ColorTemp::ColabSkin35_15_17_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0211, 0.022,   0.0225, 0.0234,  0.0244, 0.0294,  0.0349, 0.038,   0.0411, 0.0425,  0.0441, 0.0455,  0.0472, 0.0473,  0.0475, 0.0463,  0.0452, 0.0435,  0.0417, 0.0397,  0.0377,
    0.0375, 0.0372, 0.0412,  0.0452, 0.052,   0.0603, 0.0655,  0.0707, 0.0722,  0.0736, 0.0738,  0.0741, 0.0737,  0.0731, 0.0721,  0.0711, 0.0707,  0.0704, 0.071,   0.0717,
    0.0782, 0.0846, 0.099,   0.1159, 0.1298,  0.1432, 0.1497,  0.1581, 0.1603,  0.1644, 0.1659,  0.1673, 0.1679,  0.1690, 0.1696,  0.1702, 0.1704,  0.1705, 0.1706,  0.1707, 0.1707,  0.1707, 0.1707,  0.1706, 0.1706,  0.1707, 0.1707,  0.1706, 0.1712,  0.1719,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorLab : Skin 57 22 18
const double ColorTemp::ColabSkin57_22_18_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0647, 0.0677,  0.0709, 0.0754,  0.0797, 0.099,   0.1181, 0.1296,  0.1409, 0.1469,  0.1529, 0.1594,  0.1657, 0.1672,  0.1683, 0.1648,  0.1615, 0.1561,  0.1506, 0.144,   0.1375, 0.136,   0.1339,
    0.1437, 0.1531, 0.172,   0.1911, 0.2032,  0.2153, 0.2171,  0.2190, 0.2176,  0.2164, 0.213,   0.2095, 0.2048,  0.2005, 0.1985,  0.1965, 0.198,   0.1997, 0.2196,  0.2396, 0.288,   0.3362, 0.378,
    0.4209, 0.444,   0.4671, 0.477,   0.4865, 0.491,   0.4955, 0.498,   0.5007, 0.5027,  0.5048, 0.5055,  0.5061, 0.5063,  0.5066, 0.5065,  0.5063, 0.506,   0.5057, 0.5056,  0.5055, 0.5056,  0.5057, 0.5078,  0.5099,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorLab : Skin 40 17 17
const double ColorTemp::ColabSkin40_17_17_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0296, 0.0306,  0.0317, 0.0332,  0.0346, 0.042,   0.0498, 0.0543,  0.0588, 0.061,   0.0632, 0.0624,  0.0678, 0.068,   0.0682, 0.0663,  0.0649, 0.0625,  0.0598, 0.057,   0.0540, 0.0535,  0.0529, 0.057,
    0.0631, 0.072,   0.0827, 0.089,   0.0959, 0.0975,  0.0994, 0.0996,  0.0997, 0.0988,  0.0980, 0.0966,  0.0951, 0.0945,  0.0939, 0.0948,  0.0957, 0.105,   0.1143, 0.136,   0.1589, 0.178,   0.1980,
    0.2095, 0.2194, 0.224,   0.2283, 0.2302,  0.2325, 0.2337,  0.2348, 0.2357,  0.2366, 0.2369,  0.2371, 0.2372,  0.2373, 0.2373,  0.2373, 0.2373,  0.2372, 0.2372,  0.2372, 0.2372,  0.2372, 0.238,   0.2389,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorLab : Skin 91 4 14
const double ColorTemp::ColabSkin91_4_14_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1430, 0.16,    0.1778, 0.202,   0.2303, 0.301,   0.3813, 0.4245,  0.4692, 0.499,   0.5287, 0.5635,  0.5977, 0.6175,  0.6372, 0.6394,  0.6418, 0.638,   0.6341, 0.6228,  0.6117, 0.6121,  0.6125,
    0.646,  0.6807, 0.742,   0.8032, 0.8355,  0.8687, 0.8595,  0.8510, 0.828,   0.8059, 0.778,   0.7490, 0.721,   0.6914, 0.676,   0.6608, 0.657,   0.6530, 0.6565,  0.7001, 0.7609,  0.8219,
    0.876,  0.9297, 0.9598,  0.9901, 1.003,   1.0156, 1.021,   1.0289, 1.0346,  1.0396, 1.045,   1.0513, 1.0538,  1.0561, 1.0559,  1.0557, 1.054,   1.0523, 1.049,   1.0466, 1.045,   1.0436, 1.0445,  1.0455, 1.053,   1.0605,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorLab : Skin 87 8 8
const double ColorTemp::ColabSkin87_8_8_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1433, 0.161,   0.1780, 0.204,   0.2305, 0.306,   0.3828, 0.428,   0.4722, 0.502,   0.5317, 0.5645,  0.5997, 0.618,   0.6366, 0.6368,  0.6370, 0.631,   0.6251, 0.6120,  0.5994, 0.596,
    0.5931, 0.618,   0.6420, 0.705,   0.7347, 0.757,   0.7785, 0.765,   0.7532, 0.71,    0.7062, 0.677,   0.6491, 0.62,    0.5922, 0.577,   0.5619, 0.5586,  0.5556, 0.579,   0.6121,
    0.684,  0.7563, 0.82,    0.8837, 0.918,   0.9545, 0.969,   0.9843, 0.992,   0.9991, 1.005,   1.0104, 1.016,   1.0223, 1.0248,  1.0273, 1.0272,  1.0271, 1.025,   1.0238, 1.02,    1.0182, 1.017,   1.0151, 1.016,   1.0171, 1.024,   1.0321,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorLab : Skin 89 8 21
const double ColorTemp::ColabSkin89_8_21_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1394, 0.152,   0.1659, 0.1855,  0.2052, 0.266,   0.3277, 0.363,   0.3988, 0.422,   0.4450, 0.472,   0.4984, 0.512,   0.5270, 0.5274,  0.5278, 0.522,   0.5177, 0.5065,  0.4960, 0.4975,
    0.4995, 0.535,   0.5721, 0.637,   0.7016, 0.7395,  0.7786, 0.7777,  0.7767, 0.763,   0.7485, 0.728,   0.7081, 0.687,   0.6649, 0.653,   0.6425, 0.641,   0.6391, 0.665,   0.6925, 0.76,
    0.8266, 0.885,   0.9447, 0.975,   1.0106, 1.025,   1.0383, 1.045,   1.0525, 1.058,   1.0628, 1.068,   1.0730, 1.075,   1.0768, 1.0768,  1.0769, 1.0755,  1.0744, 1.0724,  1.0704, 1.069,   1.0685, 1.0691,  1.0697, 1.075,   1.0823,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorLab : Skin 75 8 4
const double ColorTemp::ColabSkin75_8_4_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1030, 0.116,   0.1294, 0.1495,  0.1696, 0.227,   0.2847, 0.319,   0.3524, 0.375,   0.3977, 0.423,   0.4492, 0.462,   0.4770, 0.4768,  0.4767, 0.471,   0.4675, 0.458,   0.4480, 0.444,   0.4408,
    0.455,  0.4688, 0.496,   0.5243, 0.535,   0.5465, 0.534,   0.5228, 0.503,   0.4851, 0.463,   0.4408, 0.419,   0.3974, 0.385,   0.3741, 0.371,   0.3692, 0.391,   0.4110, 0.464,   0.5180,
    0.565,  0.6126, 0.638,   0.6651, 0.676,   0.6871, 0.692,   0.6979, 0.702,   0.7062, 0.71,    0.7151, 0.717,   0.7189, 0.7188,  0.7187, 0.717,   0.7162, 0.714,   0.7119, 0.7105,  0.7095, 0.7097,  0.7110, 0.716,   0.7223,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorLab : Skin 75 10 33
const double ColorTemp::ColabSkin75_10_33_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0873, 0.091,   0.0967, 0.103,   0.1097, 0.135,   0.1617, 0.177,   0.1913, 0.198,   0.2086, 0.218,   0.2289, 0.234,   0.2383, 0.2375,  0.2370, 0.2335,  0.2299, 0.223,   0.2180, 0.222,
    0.2259, 0.256,   0.2860, 0.338,   0.3905, 0.426,   0.4613, 0.47,    0.4786, 0.478,   0.4772, 0.471,   0.4668, 0.459,   0.4522, 0.449,   0.4454, 0.446,   0.4467, 0.464,   0.4834, 0.527,
    0.5727, 0.609,   0.6511, 0.673,   0.6946, 0.703,   0.7130, 0.718,   0.7224, 0.725,   0.7285, 0.731,   0.7337, 0.7345,  0.7351, 0.7353,  0.7355, 0.735,   0.7348, 0.7342,  0.7337, 0.7336,  0.7335, 0.7335,  0.7336, 0.737,   0.7395,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorLab : Skin 65 33 11
const double ColorTemp::ColabSkin65_33_11_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1067, 0.113,   0.1182, 0.126,   0.1346, 0.165,   0.2033, 0.224,   0.2448, 0.259,   0.2666, 0.277,   0.2891, 0.291,   0.2927, 0.285,   0.2783, 0.268,   0.2569, 0.244,   0.2323, 0.225,   0.2195,
    0.225,  0.2323, 0.248,   0.2655, 0.275,   0.2832, 0.281,   0.2797, 0.275,   0.2708, 0.264,   0.2567, 0.248,   0.2403, 0.236,   0.2326, 0.235,   0.2386, 0.274,   0.3116, 0.40,    0.4885, 0.56,
    0.6435, 0.688,   0.7279, 0.745,   0.7633, 0.772,   0.7791, 0.783,   0.7883, 0.792,   0.7955, 0.7965,  0.7978, 0.7982,  0.7989, 0.7985,  0.7983, 0.7975,  0.7971, 0.7968,  0.7966, 0.7968,  0.7970, 0.801,   0.8043,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorLab : Skin 65 7 24
const double ColorTemp::ColabSkin65_7_24_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0619, 0.066,   0.0710, 0.077,   0.0840, 0.106,   0.1288, 0.142,   0.1546, 0.163,   0.1706, 0.179,   0.1893, 0.194,   0.1989, 0.1988,  0.1987, 0.196,   0.1941, 0.189,   0.1853, 0.188,   0.1894,
    0.209,  0.2282, 0.262,   0.2962, 0.318,   0.3402, 0.343,   0.3469, 0.343,   0.3407, 0.334,   0.3285, 0.321,   0.3140, 0.31,    0.3069, 0.308,   0.3066, 0.319,   0.3311, 0.362,   0.3918, 0.418,
    0.4451, 0.459,   0.4747, 0.481,   0.4873, 0.491,   0.4937, 0.496,   0.4981, 0.501,   0.5022, 0.503,   0.5035, 0.5035,  0.5036, 0.5032,  0.5029, 0.5022,  0.5016, 0.5013,  0.5011, 0.5013,  0.5014, 0.504,   0.5063,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorLab : Skin 57 19 6
const double ColorTemp::ColabSkin57_19_6_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0662, 0.071,   0.0773, 0.085,   0.0939, 0.115,   0.1491, 0.165,   0.1821, 0.192,   0.2019, 0.214,   0.2236, 0.228,   0.2321, 0.2298,  0.2266, 0.221,   0.2161, 0.208,   0.2019, 0.199,
    0.1951, 0.201,   0.2066, 0.219,   0.2325, 0.239,   0.2443, 0.241,   0.2366, 0.23,    0.2235, 0.215,   0.2068, 0.199,   0.1895, 0.185,   0.1806, 0.1811,  0.1816, 0.20,    0.2197, 0.267,   0.3135,
    0.355,  0.3960, 0.418,   0.4411, 0.449,   0.4600, 0.4643,  0.4687, 0.471,   0.4743, 0.477,   0.4792, 0.48,    0.4811, 0.4813,  0.4815, 0.481,   0.4806, 0.4798,  0.4790, 0.4786,  0.4782, 0.4786,  0.4788, 0.481,   0.4844,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorLab : Skin 57 4 19
const double ColorTemp::ColabSkin57_4_19_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0430, 0.047,   0.0505, 0.056,   0.0614, 0.077,   0.0963, 0.1063,  0.1164, 0.123,   0.1294, 0.137,   0.1448, 0.149,   0.1533, 0.154,   0.1544, 0.153,   0.1521, 0.149,   0.1463, 0.148,
    0.1496, 0.164,   0.1780, 0.202,   0.2273, 0.242,   0.2585, 0.26,    0.2616, 0.258,   0.2550, 0.2495,  0.2442, 0.238,   0.2320, 0.229,   0.2258, 0.2253,  0.2247, 0.232,   0.2394,
    0.258,  0.2763, 0.292,   0.3087, 0.318,   0.3269, 0.331,   0.3346, 0.337,   0.3387, 0.341,   0.3417, 0.343,   0.3447, 0.345,   0.3457, 0.3457,  0.3457, 0.3455,  0.3451, 0.3445,  0.3439, 0.3437,  0.3435, 0.3437,  0.3438, 0.346,   0.3475,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorLab : Skin 57 10 28
const double ColorTemp::ColabSkin57_10_28_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0463, 0.048,   0.0505, 0.053,   0.0563, 0.069,   0.0816, 0.088,   0.0961, 0.102,   0.1041, 0.1085,  0.1135, 0.1155,  0.1174, 0.1168,  0.1161, 0.114,   0.1118, 0.1085,  0.1054, 0.1074,  0.1094, 0.124,
    0.1406, 0.168,   0.1951, 0.223,   0.2325, 0.238,   0.2426, 0.243,   0.2432, 0.242,   0.2391, 0.239,   0.2326, 0.231,   0.2297, 0.234,   0.2309, 0.242,   0.2516, 0.277,   0.3017, 0.324,
    0.3456, 0.358,   0.3700, 0.375,   0.3802, 0.3827,  0.3854, 0.387,   0.3887, 0.39,    0.3913, 0.3916,  0.3920, 0.3921,  0.3923, 0.3921,  0.3920, 0.3918,  0.3916, 0.3916,  0.3915, 0.3915,  0.3915, 0.393,   0.3945,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorLab : Skin 40 7 19
const double ColorTemp::ColabSkin40_7_19_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0215, 0.023,   0.0240, 0.026,   0.0275, 0.033,   0.0409, 0.044,   0.0487, 0.051,   0.0532, 0.056,   0.0585, 0.0595,  0.0608, 0.0605,  0.0602, 0.059,   0.0581, 0.057,   0.0549, 0.0555,  0.0562,
    0.061,  0.0692, 0.08,    0.0922, 0.099,   0.1075, 0.109,   0.1107, 0.11,    0.1098, 0.1082,  0.1069, 0.1045,  0.1031, 0.102,   0.1013, 0.1015,  0.1016, 0.106,   0.1112, 0.123,   0.1348, 0.145,
    0.1554, 0.161,   0.1668, 0.169,   0.1716, 0.1728,  0.1741, 0.175,   0.1756, 0.1763,  0.1769, 0.1771,  0.1773, 0.1773,  0.1774, 0.1773,  0.1772, 0.177,   0.1769, 0.1769,  0.1769, 0.1769,  0.1769, 0.1777,  0.1784,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorLab : Skin 40 17 6
const double ColorTemp::ColabSkin40_17_6_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0314, 0.033,   0.0359, 0.039,   0.0427, 0.054,   0.0668, 0.074,   0.0812, 0.085,   0.0895, 0.094,   0.0985, 0.10,    0.1015, 0.0991,  0.0984, 0.096,   0.0930, 0.089,   0.0861, 0.085,   0.0828,
    0.085,  0.0878, 0.094,   0.0995, 0.103,   0.1052, 0.1035,  0.1026, 0.10,    0.0976, 0.094,   0.0911, 0.088,   0.0840, 0.083,   0.0805, 0.081,   0.0814, 0.09,    0.1006, 0.124,
    0.1474, 0.1685,  0.1885, 0.1995,  0.2110, 0.216,   0.2204, 0.223,   0.2247, 0.226,   0.2273, 0.2284,  0.2296, 0.230,   0.2304, 0.2305,  0.2306, 0.2305,  0.2303, 0.23,    0.2297, 0.2296,  0.2294, 0.2295,  0.2296, 0.231,   0.2321,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorLab : Skin 40 4 11
const double ColorTemp::ColabSkin40_4_11_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0209, 0.023,   0.0250, 0.028,   0.0310, 0.039,   0.0497, 0.056,   0.0605, 0.064,   0.0675, 0.072,   0.0758, 0.078,   0.0802, 0.0803,  0.0804, 0.0797,  0.0790, 0.078,   0.0758, 0.076,   0.0764, 0.082,
    0.0875, 0.098,   0.1072, 0.113,   0.1189, 0.1187,  0.1185, 0.116,   0.1141, 0.111,   0.1078, 0.104,   0.1012, 0.099,   0.0977, 0.098,   0.0971, 0.101,   0.1049, 0.115,   0.1245, 0.133,
    0.1417, 0.147,   0.1513, 0.153,   0.1554, 0.1564,  0.1575, 0.158,   0.1590, 0.1598,  0.1606, 0.1608,  0.1611, 0.1611,  0.1611, 0.1609,  0.1608, 0.1604,  0.1601, 0.160,   0.1598, 0.1599,  0.1600, 0.1609,  0.1619,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorLab : Skin 33 6 15
const double ColorTemp::ColabSkin33_6_15_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0143, 0.015,   0.0162, 0.0175,  0.0189, 0.023,   0.0286, 0.031,   0.0342, 0.036,   0.0376, 0.039,   0.0415, 0.0425,  0.0434, 0.0432,  0.0431, 0.0425,  0.0418, 0.041,   0.0396, 0.04,
    0.0404, 0.0444,  0.0488, 0.056,   0.0638, 0.0689,  0.0735, 0.074,   0.0752, 0.0745,  0.0741, 0.073,   0.0717, 0.070,   0.0688, 0.0681,  0.0673, 0.0673,  0.0674, 0.0710,  0.0737, 0.0810,
    0.0889, 0.0960,  0.1023, 0.1065,  0.1098, 0.1120,  0.1129, 0.1135,  0.1145, 0.1150,  0.1155, 0.1160,  0.1164, 0.1165,  0.1167, 0.1167,  0.1168, 0.1167,  0.1166, 0.1165,  0.1164, 0.1164,  0.1163, 0.1163,  0.1164, 0.1170,  0.1174,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorLab : Skin 33 15 5
const double ColorTemp::ColabSkin33_15_5_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0212, 0.023,   0.0243, 0.0265,  0.0289, 0.037,   0.0451, 0.051,   0.0549, 0.058,   0.0605, 0.063,   0.0666, 0.0675,  0.0686, 0.0672,  0.0664, 0.065,   0.0627, 0.0061,  0.0580, 0.0565,  0.0557,
    0.057,  0.0590, 0.063,   0.0666, 0.069,   0.0703, 0.0694,  0.0684, 0.0666,  0.0651, 0.063,   0.0607, 0.0585,  0.0559, 0.0545,  0.0535, 0.0540,  0.0542, 0.0610,  0.0672, 0.0832,
    0.0992, 0.1132,  0.1272, 0.1345,  0.1425, 0.1455,  0.1489, 0.1505,  0.1518, 0.1527,  0.1536, 0.1545,  0.1552, 0.1555,  0.1557, 0.1558,  0.1559, 0.1558,  0.1557, 0.1155,  0.1552, 0.1551,  0.1550, 0.1551,  0.1552, 0.1560,  0.1569,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
//spectral data ColorLab : Skin 33 10 15
const double ColorTemp::ColabSkin33_10_15_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0166, 0.0175,  0.0183, 0.0194,  0.0207, 0.0260,  0.0306, 0.033,   0.0364, 0.0380,  0.0396, 0.0415,  0.0431, 0.0437,  0.0443, 0.0438,  0.0432, 0.0420,  0.0409, 0.0395,  0.0380, 0.0380,
    0.0381, 0.0415,  0.0456, 0.0525,  0.0595, 0.0645,  0.0686, 0.0695,  0.0705, 0.0702,  0.0700, 0.0690,  0.0681, 0.0667,  0.0655, 0.065,   0.0644, 0.0648,  0.0650, 0.0695,  0.0739, 0.0852,
    0.0955, 0.1040,  0.1145, 0.1196,  0.1249, 0.1271,  0.1293, 0.1305,  0.1314, 0.1322,  0.1327, 0.1332,  0.1337, 0.1338,  0.1340, 0.1340,  0.1341, 0.1340,  0.1340, 0.1339,  0.1338, 0.1338,  0.1338, 0.1338,  0.1338, 0.1345,  0.1349,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorLab : Skin 24 5 6
const double ColorTemp::ColabSkin24_5_6_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0086, 0.0095,  0.0102, 0.0112,  0.0127, 0.0167,  0.0203, 0.0225,  0.0248, 0.0265,  0.0277, 0.0295,  0.0309, 0.0315,  0.0325, 0.0324,  0.0323, 0.0319,  0.0315, 0.0307,  0.0299, 0.0298,
    0.0297, 0.0315,  0.0330, 0.0365,  0.0392, 0.0412,  0.0427, 0.0424,  0.0421, 0.0410,  0.0402, 0.0390,  0.0377, 0.0365,  0.0351, 0.0345,  0.0337, 0.0337,  0.0336, 0.0356,  0.0374, 0.0415,
    0.0470, 0.0512,  0.0554, 0.0575,  0.0601, 0.0610,  0.0620, 0.0625,  0.0630, 0.0634,  0.0637, 0.0640,  0.0643, 0.0645,  0.0646, 0.0646,  0.0646, 0.0645,  0.0644, 0.0643,  0.0642, 0.0642,  0.0641, 0.0641,  0.0641, 0.0645,  0.0649,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorLab : Skin 26 18 18
const double ColorTemp::ColabSkin26_18_18_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0135, 0.0137,  0.0138, 0.0139,  0.0140, 0.0163,  0.0187, 0.0202,  0.0215, 0.0220,  0.0224, 0.0228,  0.0231, 0.0227,  0.0222, 0.0212,  0.0202, 0.0189,  0.0174, 0.0161,  0.0146, 0.0143,
    0.0140, 0.0163,  0.0184, 0.0224,  0.0268, 0.0291,  0.0331, 0.0348,  0.0358, 0.0366,  0.0374, 0.0378,  0.0380, 0.0380,  0.0379, 0.0380,  0.0381, 0.0388,  0.0394, 0.0440,  0.0490,
    0.0605, 0.0720, 0.0821,  0.0921, 0.0976,  0.1030, 0.1056,  0.1076, 0.1087,  0.1097, 0.1103,  0.1108, 0.1111,  0.1115, 0.1115,  0.1116, 0.1117,  0.1118, 0.1118,  0.1119, 0.1119,  0.1120, 0.1120,  0.1121, 0.1121,  0.1120, 0.1123,  0.1126,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorLab : Skin 24 7 5
const double ColorTemp::ColabSkin24_7_5_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0093, 0.0105,  0.0111, 0.0125,  0.0137, 0.0180,  0.0221, 0.0245,  0.0270, 0.0285,  0.0301, 0.0316,  0.0336, 0.0345,  0.0353, 0.0350,  0.0349, 0.0343,  0.0338, 0.0329,  0.0320, 0.0317,  0.0315,
    0.0328, 0.0342, 0.0368,  0.0397, 0.0412,  0.0424, 0.0420,  0.0415, 0.0404,  0.0393, 0.0379,  0.0366, 0.0352,  0.0337, 0.0330,  0.0323, 0.0322,  0.0322, 0.0344,  0.0367, 0.0422,  0.0479, 0.0529,
    0.0578, 0.0606,  0.0633, 0.0644,  0.0656, 0.0662,  0.0667, 0.0670,  0.0674, 0.0678,  0.0681, 0.0683,  0.0684, 0.0684,  0.0684, 0.0683,  0.0683, 0.0682,  0.0680, 0.0679,  0.0678, 0.0678,  0.0679, 0.0683,  0.0688,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorLab : Skin 24 4 2
const double ColorTemp::ColabSkin20_4_2_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0064, 0.0074,  0.0080, 0.00903, 0.0104, 0.0139,  0.0174, 0.0189,  0.0216, 0.0222,  0.0243, 0.0258,  0.0274, 0.0282,  0.0291, 0.0290,  0.0290, 0.0288,  0.0284, 0.0278,  0.0272, 0.0270,  0.0267, 0.0276,
    0.0285, 0.0302,  0.0320, 0.0327,  0.0335, 0.0328,  0.0321, 0.0311,  0.0299, 0.0280,  0.0272, 0.0259,  0.0246, 0.0239,  0.0232, 0.0230,  0.0229, 0.0243,  0.0256, 0.0291,  0.0324, 0.0354,
    0.0385, 0.0401,  0.0418, 0.0425,  0.0432, 0.0435,  0.0439, 0.0442,  0.0444, 0.0447,  0.0450, 0.0451,  0.0452, 0.0452,  0.0452, 0.0451,  0.0450, 0.0449,  0.0448, 0.0447,  0.0446, 0.0447,  0.0447, 0.0450,  0.0454,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorLab : Skin 98 -2 10
const double ColorTemp::ColabSkin98_m2_10_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1627, 0.1870,  0.2115, 0.2480,  0.2860, 0.3870,  0.4878, 0.5460,  0.6050, 0.6460,  0.6874, 0.7355,  0.7836, 0.8130,  0.8424, 0.8494,  0.8543, 0.8520,  0.8508, 0.8390,  0.8267, 0.8274,  0.8280,
    0.8680, 0.9076, 0.9600,  1.0497, 1.089,   1.1190, 1.10,    1.0836, 1.045,   1.0140, 0.975,   0.9305, 0.884,   0.8486, 0.826,   0.8042, 0.7980,  0.7895, 0.8093,  0.8292, 0.884,   0.9376, 0.987,
    1.0341, 1.059,   1.0892, 1.104,   1.1125, 1.1253,  1.1255, 1.131,   1.1375, 1.145,   1.1520, 1.155,   1.1585, 1.158,   1.1574, 1.1548,  1.1523, 1.148,   1.1439, 1.141,   1.1394, 1.141,   1.1423, 1.151,   1.1619,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorLab : Skin 90 -1 20
const double ColorTemp::ColabSkin90_m1_20_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1228, 0.138,   0.1532, 0.175,   0.1987, 0.261,   0.3279, 0.365,   0.4022, 0.428,   0.4537, 0.4842,  0.5147, 0.5337,  0.5521, 0.557,   0.5611, 0.5602,  0.5593, 0.551,   0.5438, 0.548,
    0.5527, 0.593,   0.6334, 0.703,   0.7732, 0.8135,  0.8543, 0.851,   0.8474, 0.829,   0.8105, 0.786,   0.7613, 0.736,   0.7105, 0.697,   0.6835, 0.679,   0.6750, 0.6895,  0.7045, 0.743,
    0.7832, 0.818,   0.8530, 0.873,   0.8929, 0.899,   0.9099, 0.914,   0.9197, 0.924,   0.9282, 0.933,   0.9380, 0.9395,  0.9419, 0.9416,  0.9413, 0.9398,  0.9382, 0.9357,  0.9332, 0.9322,  0.9306, 0.9315,  0.9322, 0.939,   0.9452,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorLab : Skin 95 0 4
const double ColorTemp::ColabSkin95_0_4_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1614, 0.1865,  0.2118, 0.2495,  0.2889, 0.392,   0.4969, 0.557,   0.6185, 0.6605,  0.7035, 0.749,   0.8018, 0.832,   0.8605, 0.865,   0.8696, 0.866,   0.8633, 0.849,   0.8365, 0.834,
    0.8308, 0.861,   0.8911, 0.946,   1.0030, 1.025,   1.0490, 1.025,   1.0030, 0.964,   0.9278, 0.884,   0.8407, 0.7985,  0.7565, 0.734,   0.7107, 0.6985,  0.6962, 0.718,   0.7416, 0.803,
    0.8642, 0.919,   0.9733, 1.001,   1.0349, 1.051,   1.0609, 1.068,   1.0747, 1.081,   1.0872, 1.095,   1.1021, 1.105,   1.1089, 1.1085,  1.1079, 1.1055,  1.1027, 1.098,   1.0940, 1.091,   1.0892, 1.0905,  1.0923, 1.102,   1.1123,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorLab : Skin 81 2 14
const double ColorTemp::ColabSkin81_2_14_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1029, 0.116,   0.1285, 0.148,   0.1672, 0.222,   0.2774, 0.311,   0.3412, 0.362,   0.3849, 0.410,   0.4359, 0.451,   0.4659, 0.468,   0.4706, 0.4685,  0.4664, 0.4685,  0.4512, 0.4525,  0.4536,
    0.461,  0.5076, 0.551,   0.6035, 0.6295,  0.6559, 0.6495,  0.6442, 0.627,   0.6112, 0.5905,  0.5691, 0.537,   0.5266, 0.515,   0.5039, 0.501,   0.4975, 0.5125,  0.5283, 0.568,   0.6087, 0.643,
    0.6799, 0.700,   0.7200, 0.729,   0.7370, 0.7415,  0.7461, 0.750,   0.7536, 0.759,   0.7620, 0.764,   0.7655, 0.7653,  0.7651, 0.764,   0.7626, 0.761,   0.7583, 0.7572,  0.7561, 0.7567,  0.7575, 0.758,   0.7685,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorLab : Skin 87 3 10
const double ColorTemp::ColabSkin87_3_10_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1295, 0.146,   0.1639, 0.190,   0.2160, 0.291,   0.3626, 0.405,   0.4480, 0.476,   0.5066, 0.541,   0.5743, 0.593,   0.6136, 0.616,   0.6186, 0.614,   0.6119, 0.601,   0.5911, 0.5905,
    0.5897, 0.623,   0.6460, 0.697,   0.7483, 0.773,   0.7992, 0.788,   0.7759, 0.752,   0.7287, 0.699,   0.6712, 0.642,   0.6141, 0.598,   0.5835, 0.579,   0.5750, 0.596,   0.6162, 0.669,
    0.7239, 0.772,   0.8193, 0.845,   0.8728, 0.888,   0.8954, 0.901,   0.9073, 0.912,   0.9171, 0.922,   0.9281, 0.931,   0.9328, 0.9325,  0.9323, 0.931,   0.9289, 0.926,   0.9232, 0.9215,  0.9201, 0.921,   0.9220, 0.929,   0.9364,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorLab : Skin 77 12 21
const double ColorTemp::ColabSkin77_12_21_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1039, 0.111,   0.1205, 0.132,   0.1448, 0.185,   0.2261, 0.249,   0.2734, 0.287,   0.3028, 0.318,   0.3364, 0.345,   0.3525, 0.351,   0.3499, 0.345,   0.3397, 0.3295,  0.3224, 0.329,   0.3234,
    0.349,  0.3729, 0.418,   0.4625, 0.490,   0.5173, 0.5185,  0.5196, 0.511,   0.5045, 0.492,   0.4807, 0.467,   0.4543, 0.447,   0.4410, 0.4409,  0.4407, 0.464,   0.4872, 0.544,   0.6020, 0.6522,
    0.7029, 0.731,   0.7588, 0.771,   0.7823, 0.787,   0.7939, 0.798,   0.8017, 0.805,   0.8090, 0.8103,  0.8115, 0.8117,  0.8118, 0.8111,  0.8104, 0.8193,  0.8081, 0.8076,  0.8070, 0.8073,  0.8077, 0.812,   0.8162,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorLab : Skin 70 7 32
const double ColorTemp::ColabSkin70_7_32_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0695, 0.074,   0.0777, 0.084,   0.0890, 0.104,   0.1321, 0.144,   0.1565, 0.164,   0.1713, 0.1795,  0.1889, 0.194,   0.1978, 0.198,   0.1983, 0.196,   0.1939, 0.189,   0.1853, 0.189,
    0.1933, 0.219,   0.2458, 0.291,   0.3362, 0.367,   0.3974, 0.405,   0.4120, 0.411,   0.4101, 0.406,   0.4007, 0.394,   0.3877, 0.385,   0.3816, 0.3817,  0.3819, 0.395,   0.4081, 0.440,
    0.4721, 0.498,   0.5284, 0.544,   0.5598, 0.566,   0.5730, 0.577,   0.5801, 0.5825,  0.5848, 0.587,   0.5890, 0.5895,  0.5901, 0.5903,  0.5903, 0.59,    0.5897, 0.5892,  0.5887, 0.5885,  0.5884, 0.5885,  0.5886, 0.5896,  0.5934,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorLab : Sky 60 0 -31
const double ColorTemp::ColabSky60_0_m31_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0752, 0.094,   0.1121, 0.141,   0.1699, 0.243,   0.3150, 0.357,   0.4015, 0.432,   0.4631, 0.497,   0.5325, 0.553,   0.5730, 0.574,   0.5758, 0.572,   0.5695, 0.559,   0.5503, 0.539,
    0.5284, 0.5175,  0.5066, 0.493,   0.4800, 0.459,   0.4336, 0.401,   0.3684, 0.333,   0.3003, 0.265,   0.2313, 0.199,   0.1695, 0.167,   0.1349, 0.129,   0.1234, 0.136,   0.1489, 0.184,
    0.2212, 0.253,   0.2858, 0.303,   0.3218, 0.329,   0.3370, 0.341,   0.3440, 0.348,   0.3512, 0.355,   0.3606, 0.363,   0.3658, 0.3653,  0.3649, 0.3625,  0.3611, 0.358,   0.3544, 0.352,   0.3502, 0.3512,  0.3529, 0.359,   0.3660,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data ColorLab : Sky 42 0 -24
const double ColorTemp::ColabSky42_0_m24_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0336, 0.041,   0.0501, 0.063,   0.0761, 0.103,   0.1412, 0.151,   0.1799, 0.193,   0.2076, 0.223,   0.2387, 0.248,   0.2569, 0.2575,  0.2581, 0.256,   0.2553, 0.250,   0.2466, 0.2411,
    0.2368, 0.2318,  0.2268, 0.2205,  0.2145, 0.204,   0.1935, 0.179,   0.1641, 0.149,   0.1335, 0.118,   0.1025, 0.087,   0.0748, 0.067,   0.0593, 0.056,   0.0541, 0.059,   0.0655, 0.081,
    0.0979, 0.112,   0.1269, 0.134,   0.1430, 0.147,   0.1497, 0.151,   0.1529, 0.1545,  0.1561, 0.158,   0.1603, 0.1616,  0.1627, 0.1625,  0.1623, 0.1614,  0.1605, 0.159,   0.1575, 0.1567,  0.1557, 0.1563,  0.1569, 0.159,   0.1627,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
//spectral data ColorLab : blue 77 -44 -50

const double ColorTemp::Colorblue_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0316, 0.0480, 0.0658, 0.0925, 0.1199, 0.1812, 0.2424, 0.2770, 0.3145, 0.3430, 0.3702, 0.4022, 0.4346, 0.4560, 0.4778, 0.4843, 0.4902, 0.4940, 0.4960, 0.4920, 0.4889, 0.4820,
    0.4764, 0.4685, 0.4606, 0.4486, 0.4379, 0.4160, 0.3955, 0.3640, 0.3330, 0.2990, 0.2660, 0.2291, 0.1991, 0.1705, 0.1403, 0.1220, 0.1067, 0.0967, 0.0907, 0.0846, 0.0785, 0.0698,
    0.0601, 0.0525, 0.0455, 0.0423, 0.0386, 0.0370, 0.0358, 0.0354, 0.0351, 0.0368, 0.0382, 0.0413, 0.0449, 0.0474, 0.0492, 0.0484, 0.0477, 0.0460, 0.0437, 0.0402, 0.0371, 0.0349,
    0.0329, 0.0341, 0.0356, 0.0410, 0.0462,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0

};
//0.1571    0.2150  0.3040  0.3684  0.3952  0.3965  0.3782  0.3418  0.2995  0.2543  0.2043  0.1686  0.1420  0.1070  0.0785  0.0725  0.0755  0.0695  0.0680  0.0914  0.1379  0.1833  0.2038  0.2065  0.2079  0.2110  0.2176  0.2319  0.2518  0.2632  0.2616  0.2522  0.2380  0.2290  0.2432  0.2901

const double ColorTemp::ColorViolA1_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1571, 0.185, 0.2150, 0.255, 0.3040, 0.332, 0.3684, 0.381, 0.3952, 0.3956, 0.3965, 0.387, 0.3782, 0.362, 0.3418, 0.322, 0.2995, 0.275, 0.2543, 0.229, 0.2043, 0.185, 0.1686,
    0.155, 0.1420, 0.131, 0.1070, 0.093, 0.0785, 0.075, 0.0725, 0.074, 0.0755, 0.072, 0.0695, 0.069, 0.0680, 0.083, 0.0914, 0.115, 0.1379, 0.162, 0.1833, 0.193, 0.2038, 0.205,
    0.2065, 0.207, 0.2079, 0.209, 0.2110, 0.214, 0.2176, 0.226, 0.2319, 0.242, 0.2518, 0.258, 0.2632, 0.263, 0.2616, 0.256, 0.2522, 0.246, 0.2380, 0.233, 0.2290, 0.235, 0.2432, 0.265, 0.2901,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
//0.2270    0.2413  0.3287  0.4079  0.4469  0.4594  0.4535  0.4268  0.3886  0.3427  0.2866  0.2433  0.2087  0.1604  0.1181  0.1069  0.1098  0.0985  0.0916  0.1130  0.1496  0.1746  0.1783  0.1742  0.1738  0.1763  0.1831  0.1975  0.2169  0.2274  0.2247  0.2140  0.1990  0.1897  0.2039  0.2508

const double ColorTemp::ColorViolA4_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2270, 0231, 0.2413, 0.286, 0.3287, 0.367, 0.4079, 0.427, 0.4469, 0.452, 0.4594, 0.456, 0.4535, 0.432, 0.4268, 0.403, 0.3886, 0.368, 0.3427, 0.314, 0.2866, 0.265, 0.2433, 0.222,
    0.2087, 0.183, 0.1604, 0.130, 0.1181, 0.112, 0.1069, 0.108, 0.1098, 0.103, 0.0985, 0.094, 0.0916, 0.104, 0.1130, 0.131, 0.1496, 0.152, 0.1746, 0.176, 0.1783, 0.176, 0.1742, 0.174,
    0.1738, 0.175, 0.1763, 0.181, 0.1831, 0.192, 0.1975, 0.206, 0.2169, 0.222, 0.2274, 0.226, 0.2247, 0.219, 0.2140, 0.206, 0.1990, 0.195, 0.1897, 0.196, 0.2039, 0.221, 0.2508,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//0.1426    0.2660  0.3556  0.4259  0.4459  0.4317  0.3942  0.3425  0.2917  0.2413  0.1885  0.1524  0.1267  0.0948  0.0700  0.0661  0.0708  0.0671  0.0699  0.1092  0.2099  0.3582  0.4857  0.5583  0.5950  0.6146  0.6307  0.6495  0.6720  0.6825  0.6809  0.6718  0.6593  0.6517  0.6649  0.7066
const double ColorTemp::ColorViolA6_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1426, 0.203, 0.2660, 0.315, 0.3556, 0.392, 0.4259, 0.435, 0.4459, 0.437, 0.4317, 0.417, 0.3942, 0.365, 0.3425, 0.317, 0.2917, 0.266, 0.2413, 0.218, 0.1885, 0.172, 0.1524,  0.141,
    0.1267, 0.112, 0.0948, 0.083, 0.0700, 0.068, 0.0661, 0.068, 0.0708, 0.069, 0.0671, 0.068, 0.0699, 0.900, 0.1092, 0.159, 0.2099, 0.284, 0.3582, 0.441, 0.4857, 0.525, 0.5583, 0.567,
    0.5950, 0.605, 0.6146, 0.623, 0.6307, 0.638, 0.6495, 0.661, 0.6720, 0.679, 0.6825, 0.681, 0.6809, 0.675, 0.6718, 0.667, 0.6593, 0.655, 0.6517, 0.658, 0.6649, 0.681, 0.7066,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//0.4939    0.3859  0.4198  0.4780  0.5328  0.5672  0.5880  0.5994  0.6029  0.5981  0.5808  0.5618  0.5369  0.4819  0.4190  0.3921  0.3815  0.3400  0.2991  0.2977  0.3090  0.3088  0.2930  0.2753  0.2660  0.2636  0.2678  0.2811  0.2995  0.3125  0.3153  0.3111  0.3006  0.2952  0.3116  0.3584
const double ColorTemp::ColorBlueSkyK3_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.4939, 0.435, 0.3859, 0.403, 0.4198, 0.446, 0.4780, 0.505, 0.5328, 0.552, 0.5672, 0.578, 0.5880, 0.595, 0.5994, 0.602, 0.6029, 0.600, 0.5981, 0.588, 0.5808, 0.571, 0.5618, 0.551,
    0.5369, 0.503, 0.4819, 0.452, 0.4190, 0.404, 0.3921, 0.386, 0.3815, 0.364, 0.3400, 0.321, 0.2991, 0.298, 0.2977, 0.304, 0.3090, 0.309, 0.3088, 0.302, 0.2930, 0.284, 0.2753, 0.271,
    0.2660, 0.265, 0.2636, 0.266, 0.2678, 0.275, 0.2811, 0.290, 0.2995, 0.306, 0.3125, 0.314, 0.3153, 0.313, 0.3111, 0.307, 0.3006, 0.298, .2952, 0.306, 0.3116, 0.325, 0.3584,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//0.4058    0.4734  0.5372  0.6051  0.6698  0.6992  0.7118  0.7135  0.7071  0.6938  0.6702  0.6511  0.6282  0.5732  0.5103  0.4913  0.4926  0.4604  0.4341  0.4648  0.5111  0.5335  0.5283  0.5154  0.5098  0.5093  0.5151  0.5309  0.5520  0.5642  0.5657  0.5598  0.5489  0.5430  0.5601  0.6067

const double ColorTemp::ColorBlueSkyK9_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.4058, 0.441, 0.4734, 0.502, 0.5372, 0.585, 0.6051, 0.643, 0.6698, 0.685, 0.6992, 0.705, 0.7118, 0.712, 0.7135, 0.711, 0.7071, 0.702, 0.6938, 0.681, 0.6702, 0.663, 0.6511, 0.642,
    0.6282, 0.604, 0.5732, 0.542, 0.5103, 0.499, 0.4913, 0.492, 0.4926, 0.475, 0.4604, 0.452, 0.4341, 0.453, 0.4648, 0.496, 0.5111, 0.525, 0.5335, 0.531, 0.5283, 0.522, 0.5154, 0.512,
    0.5098, 0.509, 0.5093, 0.513, 0.5151, 0.523, 0.5309, 0.544, 0.5520, 0.562, 0.5642, 0.565, 0.5657, 0.562, 0.5598, 0.554, 0.5489, 0.546, 0.5430, 0.553, 0.5601, 0.576, 0.6067,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//42    C4  0.3280  0.2611  0.3781  0.4646  0.5292  0.5732  0.6112  0.6307  0.6310  0.6181  0.5847  0.5488  0.5066  0.4358  0.3585  0.3151  0.2855  0.2309  0.1786  0.1546  0.1443  0.1359  0.1245  0.1151  0.1120  0.1127  0.1169  0.1275  0.1421  0.1504  0.1488  0.1416  0.1303  0.1241  0.1355  0.1739
const double ColorTemp::ColorBlueSkyC4_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.3280, 0.2950, 0.2611, 0.304, 0.3781, 0.423, 0.4646, 0.498, 0.5292, 0.555, 0.5732, 0.591, 0.6112, 0.6221, 0.6307, 0.631, 0.6310, 0.625, 0.6181, 0.607, 0.5847, 0.563, 0.5488, 0.524,
    0.5066, 0.465, 0.4358, 0.398, 0.3585, 0.336, 0.3151, 0.302, 0.2855, 0.254, 0.2309, 0.203, 0.1786, 0.166, 0.1546, 0.149, 0.1443, 0.143, 0.1359, 0.131, 0.1245, 0.123, 0.115, 0.114,
    0.1120, 0.112, 0.1127, 0.114, 0.1169, 0.122, 0.1275, 0.133, 0.1421, 0.147, 0.1504, 0.149, 0.1488, 0.145, 0.1416, 0.136, 0.1303, 0.127, 0.1241, 0.132, 0.1355, 0.155, 0.1739,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


//52    C14 0.5697  0.4660  0.5000  0.5560  0.6072  0.6402  0.6632  0.6850  0.7069  0.7292  0.7488  0.7678  0.7786  0.7721  0.7544  0.7394  0.7232  0.6889  0.6446  0.6171  0.5966  0.5743  0.5425  0.5093  0.4884  0.4784  0.4774  0.4822  0.4944  0.5076  0.5186  0.5268  0.5303  0.5332  0.5454  0.5760
const double ColorTemp::ColorBlueSkyC14_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.5697, 0.511, 0.4660, 0.481, 0.5000, 0.528, 0.5560, 0.583, 0.6072, 0.622, 0.6402, 0.653, 0.6632, 0.674, 0.6850, 0.699, 0.7069, 0.717, 0.7292, 0.735, 0.7488, 0.757, 0.7678, 0.773,
    0.7786, 0.776, 0.7721, 0.765, 0.7544, 0.746, 0.7394, 0.731, 0.7232, 0.704, 0.6889, 0.674, 0.6446, 0.631, 0.6171, 0.606, 0.5966, 0.585, 0.5743, 0.5570, 0.5425, 0.529, 0.5093, 0.498,
    0.4884, 0.482, 0.4784, 0.478, 0.4774, 0.481, 0.4822, 0.487, 0.4944, 0.503, 0.5076, 0.512, 0.5186, 0.522, 0.5268, 0.529, 0.5303, 0.532, 0.5332, 0.539, 0.5454, 0.565, 0.5760,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


//80    E4  0.1483  0.1756  0.2536  0.3084  0.3665  0.4189  0.4746  0.5127  0.5239  0.5193  0.4917  0.4569  0.4123  0.3422  0.2672  0.2179  0.1820  0.1356  0.0972  0.0784  0.0698  0.0646  0.0592  0.0556  0.0546  0.0551  0.0571  0.0611  0.0670  0.0701  0.0692  0.0661  0.0620  0.0606  0.0663  0.0834
const double ColorTemp::ColorBlueSkyE4_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1483, 0.161, 0.1756, 0.213, 0.2536, 0.283, 0.3084, 0.331, 0.3665, 0.387, 0.4189, 0.445, 0.4746, 0.496, 0.5127, 0.519, 0.5239, 0.522, 0.5193, 0.508, 0.4917, 0.476, 0.4569, 0.431,
    0.4123, 0.375, 0.3422, 0.309, 0.2672, 0.242, 0.2179, 0.208, 0.1820, 0.162, 0.1356, 0.113, 0.0972, 0.091, 0.0784, 0.073, 0.0698, 0.066, 0.0646, 0.062, 0.0592, 0.057, 0.0556, 0.055,
    0.0546, 0.055, 0.0551, 0.056, 0.0571, 0.059, 0.0611, 0.064, 0.0670, 0.069, 0.0701, 0.070, 0.0692, 0.067, 0.0661, 0.065, 0.0620, 0.061, 0.0606, 0.063, 0.0663, 0.072, 0.0834,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


//229   M1  0.3100  0.2922  0.3514  0.4042  0.4443  0.4769  0.5002  0.5133  0.5187  0.5179  0.5057  0.4928  0.4729  0.4235  0.3643  0.3371  0.3234  0.2827  0.2418  0.2338  0.2370  0.2329  0.2184  0.2028  0.1958  0.1937  0.1973  0.2084  0.2244  0.2351  0.2372  0.2331  0.2239  0.2178  0.2319  0.2731
const double ColorTemp::ColorBlueSkyM1_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.3100, 0.303, 0.2922, 0.322, 0.3514, 0.376, 0.4042, 0.424, 0.4443, 0.457, 0.4769, 0.497, 0.5002, 0.507, 0.5133, 0.516, 0.5187, 0.518, 0.5179, 0.511, 0.5057, 0.497, 0.4928, 0.483,
    0.4729, 0.454, 0.4235, 0.398, 0.3643, 0.346, 0.3371, 0.329, 0.3234, 0.301, 0.2827, 0.263, 0.2418, 0.235, 0.2338, 0.235, 0.2370, 0.236, 0.2329, 0.226, 0.2184, 0.213, 0.2028, 0.198,
    0.1958, 0.194, 0.1937, 0.196, 0.1973, 0.203, 0.2084, 0.212, 0.2244, 0.233, 0.2351, 0.236, 0.2372, 0.234, 0.2331, 0.229, 0.2239, 0.222, 0.2178, 0.224, 0.2319, 0.251, 0.2731,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


//514   2B1 0.5277  0.4431  0.4972  0.5820  0.6387  0.6750  0.7001  0.7140  0.7202  0.7193  0.7053  0.6891  0.6657  0.6181  0.5614  0.5312  0.5101  0.4589  0.4045  0.3857  0.3826  0.3751  0.3574  0.3393  0.3314  0.3304  0.3368  0.3523  0.3742  0.3874  0.3883  0.3818  0.3693  0.3616  0.3800  0.4324
const double ColorTemp::ColorBlueSky2B1_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.5277, 0.485, 0.4431, 0.476, 0.4972, 0.539, 0.5820, 0.607, 0.6387, 0.653, 0.6750, 0.691, 0.7001, 0.707, 0.7140, 0.718, 0.7202, 0.720, 0.7193, 0.713, 0.7053, 0.695, 0.6891, 0.674,
    0.6657, 0.632, 0.6181, 0.587, 0.5614, 0.543, 0.5312, 0.521, 0.5101, 0.483, 0.4589, 0.431, 0.4045, 0.398, 0.3857, 0.385, 0.3826, 0.376, 0.3751, 0.364, 0.3574, 0.346, 0.3393, 0.335,
    0.3314, 0.331, 0.3304, 0.333, 0.3368, 0.346, 0.3523, 0.363, 0.3742, 0.382, 0.3874, 0.385, 0.3883, 0.384, 0.3818, 0.375, 0.3693, 0.364, 0.3616, 0.374, 0.3800, 0.396, 0.4324,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


//368   T7  0.1943  0.3199  0.4536  0.5443  0.6043  0.6499  0.6839  0.7125  0.7329  0.7482  0.7527  0.7514  0.7383  0.7028  0.6526  0.6034  0.5500  0.4708  0.3848  0.3268  0.2929  0.2712  0.2493  0.2316  0.2243  0.2234  0.2288  0.2436  0.2640  0.2762  0.2767  0.2693  0.2566  0.2489  0.2665  0.3165
const double ColorTemp::ColorBlueSkyT7_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1943, 0.256, 0.3199, 0.376, 0.4536, 0.494, 0.5443, 0.572, 0.6043, 0.631, 0.6499, 0.664, 0.6839, 0.698, 0.7125, 0.726, 0.7329, 0.741, 0.7482, 0.751, 0.7527, 0.752, 0.7514, 0.745,
    0.7383, 0.721, 0.7028, 0.675, 0.6526, 0.631, 0.6034, 0.589, 0.5500, 0.512, 0.4708, 0.432, 0.3848, 0.342, 0.3268, 0.311, 0.2929, 0.282, 0.2712, 0.261, 0.2493, 0.236, 0.2316, 0.227,
    0.2243, 0.223, 0.2234, 0.229, 0.2288, 0.235, 0.2436, 0.255, 0.2640, 0.268, 0.2762, 0.277, 0.2767, 0.272, 0.2693, 0.263, 0.2566, 0.254, 0.2489, 0.255, 0.2665, 0.275, 0.3165,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};



//399   U19 0.5829  0.4865  0.4927  0.5690  0.6221  0.6532  0.6728  0.6832  0.6889  0.6884  0.6771  0.6648  0.6465  0.6038  0.5524  0.5297  0.5194  0.4797  0.4387  0.4356  0.4455  0.4444  0.4282  0.4094  0.4009  0.3992  0.4046  0.4185  0.4385  0.4515  0.4545  0.4505  0.4411  0.4368  0.4539  0.5013
const double ColorTemp::ColorBlueSkyU19_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.5829, 0.534, 0.4865, 0.489, 0.4927, 0.532, 0.5690, 0.593, 0.6221, 0.641, 0.6532, 0.662, 0.6728, 0.674, 0.6832, 0.687, 0.6889, 0.688, 0.6884, 0.683, 0.6771, 0.671, 0.6648, 0.665,
    0.6465, 0.622, 0.6038, 0.583, 0.5524, 0.542, 0.5297, 0.523, 0.5194, 0.492, 0.4797, 0.451, 0.4387, 0.436, 0.4356, 0.442, 0.4455, 0.445, 0.4444, 0.432, 0.4282, 0.413, 0.4094, 0.404,
    0.4009, 0.400, 0.3992, 0.402, 0.4046, 0.411, 0.4185, 0.426, 0.4385, 0.446, 0.4515, 0.452, 0.4545, 0.452, 0.4505, 0.446, 0.4411, 0.438, 0.4368, 0.443, 0.4539, 0.467, 0.5013,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


//382   U2  0.3594  0.3425  0.3214  0.3654  0.4097  0.4360  0.4590  0.4793  0.5002  0.5234  0.5476  0.5745  0.5940  0.5901  0.5703  0.5545  0.5384  0.5029  0.4592  0.4334  0.4149  0.3947  0.3657  0.3363  0.3177  0.3087  0.3077  0.3123  0.3231  0.3351  0.3454  0.3520  0.3545  0.3562  0.3674  0.3976
const double ColorTemp::ColorBlueSkyU2_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.3594, 0.345, 0.3425, 0.333, 0.3214, 0.346, 0.3654, 0.387, 0.4097, 0.424, 0.4360, 0.446, 0.4590, 0.467, 0.4793, 0.494, 0.5002, 0.517, 0.5234, 0.538, 0.5476, 0.564, 0.5745, 0.583,
    0.5940, 0.593, 0.5901, 0.580, 0.5703, 0.563, 0.5545, 0.546, 0.5384, 0.521, 0.5029, 0.478, 0.4592, 0.444, 0.4334, 0.421, 0.4149, 0.408, 0.3947, 0.378, 0.3657, 0.352, 0.3363, 0.324,
    0.3177, 0.313, 0.3087, 0.308, 0.3077, 0.310, 0.3123, 0.317, 0.3231, 0.329, 0.3351, 0.339, 0.3454, 0.348, 0.3520, 0.353, 0.3545, 0.355, 0.3562, 0.359, 0.3674, 0.375, 0.3976,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};



//378   T17 0.4213  0.3332  0.4205  0.5078  0.5650  0.6025  0.6223  0.6279  0.6195  0.5981  0.5578  0.5197  0.4785  0.4074  0.3325  0.3030  0.2918  0.2511  0.2125  0.2105  0.2198  0.2199  0.2083  0.1945  0.1895  0.1898  0.1954  0.2094  0.2288  0.2406  0.2399  0.2317  0.2189  0.2108  0.2261  0.2755
const double ColorTemp::ColorBlueSkyT17_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.4213, 0.376, 0.3332, 0.387, 0.4205, 0.467, 0.5078, 0.532, 0.5650, 0.587, 0.6025, 0.611, 0.6223, 0.624, 0.6279, 0.623, 0.6195, 0.607, 0.5981, 0.576, 0.5578, 0.534, 0.5197, 0.499,
    0.4785, 0.435, 0.4074, 0.376, 0.3325, 0.317, 0.3030, 0.295, 0.2918, 0.272, 0.2511, 0.235, 0.2125, 0.212, 0.2105, 0.214, 0.2198, 0.219, 0.2199, 0.215, 0.2083, 0.202, 0.1945, 0.191,
    0.1895, 0.189, 0.1898, 0.193, 0.1954, 0.202, 0.2094, 0.215, 0.2288, 0.236, 0.2406, 0.240, 0.2399, 0.236, 0.2317, 0.225, 0.2189, 0.214, 0.2108, 0.218, 0.2261, 0.245, 0.2755,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


//236   M8  -0.0593 0.0128  0.0031  0.0149  0.0197  0.0221  0.0223  0.0230  0.0232  0.0237  0.0243  0.0247  0.0249  0.0249  0.0248  0.0247  0.0244  0.0240  0.0240  0.0242  0.0244  0.0247  0.0252  0.0254  0.0262  0.0269  0.0271  0.0273  0.0278  0.0280  0.0276  0.0275  0.0282  0.0288  0.0295  0.0303
const double ColorTemp::ColorBlackM8_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0593, 0.040, 0.0128, 0.023, 0.0031, 0.024, 0.0149, 0.017, 0.0197, 0.021, 0.0221, 0.022, 0.0223, 0.023, 0.0230, 0.023, 0.0232, 0.023, 0.0237, 0.024, 0.0243, 0.024, 0.0247, 0.024,
    0.0249, 0.025, 0.0249, 0.025, 0.0248, 0.025, 0.0247, 0.024, 0.0244, 0.024, 0.0240, 0.024, 0.0240, 0.024, 0.0242, 0.024, 0.0244, 0.024, 0.0247, 0.025, 0.0252, 0.025, 0.0254, 0.026,
    0.0262, 0.027, 0.0269, 0.027, 0.0271, 0.027, 0.0273, 0.027, 0.0278, 0.028, 0.0280, 0.028, 0.0276, 0.028, 0.0275, 0.028, 0.0282, 0.028, 0.0288, 0.029, 0.0295, 0.030, 0.0303,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


//240   M12 -0.0377 0.0146  0.0167  0.0222  0.0257  0.0261  0.0270  0.0271  0.0274  0.0281  0.0287  0.0299  0.0306  0.0304  0.0297  0.0285  0.0277  0.0267  0.0257  0.0257  0.0258  0.0259  0.0263  0.0276  0.0279  0.0288  0.0297  0.0302  0.0304  0.0303  0.0302  0.0303  0.0310  0.0321  0.0337  0.0356
const double ColorTemp::ColorBlackM12_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0377, 0.022, 0.0146, 0.015, 0.0167, 0.020, 0.0222, 0.023, 0.0257, 0.026, 0.0261, 0.027, 0.0270, 0.027, 0.0271, 0.027, 0.0274, 0.028, 0.0281, 0.028, 0.0287, 0.029, 0.0299, 0.030,
    0.0306, 0.030, 0.0304, 0.030, 0.0297, 0.029, 0.0285, 0.028, 0.0277, 0.027, 0.0267, 0.026, 0.0257, 0.026, 0.0257, 0.026, 0.0258, 0.026, 0.0259, 0.026, 0.0263, 0.027, 0.0276, 0.028,
    0.0279, 0.028, 0.0288, 0.029, 0.0297, 0.030, 0.0302, 0.030, 0.0304, 0.030, 0.0303, 0.030, 0.0302, 0.030, 0.0303, 0.031, 0.0310, 0.031, 0.0321, 0.033, 0.0337, 0.035, 0.0356,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


//241   M13 0.2961  0.0487  0.0261  0.0330  0.0296  0.0305  0.0305  0.0314  0.0320  0.0322  0.0335  0.0348  0.0352  0.0347  0.0336  0.0325  0.0314  0.0299  0.0284  0.0284  0.0286  0.0288  0.0290  0.0299  0.0306  0.0314  0.0320  0.0328  0.0332  0.0331  0.0330  0.0328  0.0337  0.0350  0.0366  0.0385
const double ColorTemp::ColorBlackM13_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2961, 0.100, 0.0487, 0.035, 0.0261, 0.030, 0.0330, 0.031, 0.0296, 0.030, 0.0305, 0.030, 0.0305, 0.031, 0.0314, 0.032, 0.0320, 0.032, 0.0322, 0.033, 0.0335, 0.034, 0.0348, 0.035,
    0.0352, 0.035, 0.0347, 0.034, 0.0336, 0.033, 0.0325, 0.032, 0.0314, 0.030, 0.0299, 0.029, 0.0284, 0.028, 0.0284, 0.029, 0.0286, 0.029, 0.0288, 0.029, 0.0290, 0.030, 0.0299, 0.030,
    0.0306, 0.031, 0.0314, 0.032, 0.0320, 0.032, 0.0328, 0.033, 0.0332, 0.033, 0.0331, 0.033, 0.0330, 0.033, 0.0328, 0.033, 0.0337, 0.034, 0.0350, 0.036, 0.0366, 0.037, 0.0385,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//525   2B12    0.3848  0.4127  0.4780  0.5375  0.5917  0.6194  0.6353  0.6485  0.6625  0.6784  0.7010  0.7367  0.7706  0.7847  0.7843  0.7826  0.7799  0.7689  0.7521  0.7463  0.7401  0.7314  0.7168  0.7008  0.6904  0.6860  0.6861  0.6896  0.6986  0.7064  0.7138  0.7185  0.7233  0.7281  0.7338  0.7498
const double ColorTemp::ColorWhite2B12_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.3848, 0.405, 0.4127, 0.435, 0.4780, 0.501, 0.5375, 0.572, 0.5917, 0.606, 0.6194, 0.627, 0.6353, 0.642, 0.6485, 0.653, 0.6625, 0.669, 0.6784, 0.692, 0.7010, 0.072, 0.7367, 0.0760,
    0.7706, 0.775, 0.7847, 0.784, 0.7843, 0.784, 0.7826, 0.781, 0.7799, 0.775, 0.7689, 0.762, 0.7521, 0.748, 0.7463, 0.743, 0.7401, 0.731, 0.7314, 0.723, 0.7168, 0.705, 0.7008, 0.701,
    0.6904, 0.689, 0.6860, 0.686, 0.6861, 0.687, 0.6896, 0.691, 0.6986, 0.701, 0.7064, 0.711, 0.7138, 0.715, 0.7185, 0.721, 0.7233, 0.725, 0.7281, 0.731, 0.7338, 0.741, 0.7498,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


//527   2B14    0.6608  0.5900  0.6114  0.6747  0.7329  0.7599  0.7706  0.7776  0.7831  0.7890  0.7952  0.8074  0.8185  0.8214  0.8201  0.8218  0.8244  0.8229  0.8192  0.8281  0.8344  0.8388  0.8415  0.8445  0.8495  0.8521  0.8556  0.8604  0.8666  0.8687  0.8687  0.8672  0.8682  0.8703  0.8720  0.8803
const double ColorTemp::ColorWhite2B14_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.6608, 0.631, 0.5900, 0.605, 0.6114, 0.634, 0.6747, 0.698, 0.7329, 0.745, 0.7599, 0.765, 0.7706, 0.775, 0.7776, 0.781, 0.7831, 0.786, 0.7890, 0.792, 0.7952, 0.797, 0.8074, 0.810,
    0.8185, 0.821, 0.8214, 0.821, 0.8201, 0.821, 0.8218, 0.822, 0.8244, 0.823, 0.8229, 0.822, 0.8192, 0.824, 0.8281, 0.831, 0.8344, 0.836, 0.8388, 0.840, 0.8415, 0.843, 0.8445, 0.847,
    0.8495, 0.851, 0.8521, 0.853, 0.8556, 0.858, 0.8604, 0.863, 0.8666, 0.867, 0.8687, 0.869, 0.8687, 0.867, 0.8672, 0.868, 0.8682, 0.870, 0.8703, 0.871, 0.8720, 0.874, 0.8803,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//97    97  0.0031  0.0047  0.0056  0.0064  0.0070  0.0070  0.0071  0.0066  0.0065  0.0065  0.0064  0.0063  0.0063  0.0058  0.0052  0.0052  0.0053  0.0047  0.0044  0.0052  0.0066  0.0077  0.0081  0.0082  0.0084  0.0089  0.0096  0.0106  0.0121  0.0127  0.0128  0.0124  0.0115  0.0117  0.0133  0.0169
const double ColorTemp::JDC468_Blackred97_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0031, 0.0035, 0.0047, 0.0050, 0.0056, 0.0060, 0.0064, 0.0065, 0.0070, 0.007, 0.0070, 0.007, 0.0071, 0.007, 0.0066, 0.007, 0.0065, 0.006, 0.0065, 0.006, 0.0064, 0.006, 0.0063,
    0.006, 0.0063, 0.006, 0.0058, 0.005, 0.0052, 0.005, 0.0052, 0.005, 0.0053, 0.005, 0.0047, 0.005, 0.0044, 0.005, 0.0052, 0.006, 0.0066, 0.007, 0.0077, 0.008, 0.0081, 0.008, 0.008,
    0.008, 0.0084, 0.009, 0.0089, 0.009, 0.0096, 0.01, 0.0106, 0.011, 0.0121, 0.012, 0.0127, 0.012, 0.0128, 0.012, 0.0124, 0.012, 0.0115, 0.012, 0.0117, 0.013, 0.0133, 0.015, 0.0169,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//443   443 0.0067  0.0068  0.0067  0.0074  0.0083  0.0088  0.0092  0.0093  0.0098  0.0099  0.0101  0.0106  0.0109  0.0104  0.0094  0.0086  0.0075  0.0058  0.0044  0.0039  0.0037  0.0038  0.0036  0.0035  0.0036  0.0033  0.0033  0.0036  0.0038  0.0042  0.0041  0.0036  0.0033  0.0035  0.0042  0.0062
const double ColorTemp::JDC468_Blackredbl443_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0067, 0.0067, 0.0068, 0.0068, 0.0067, 0.0070, 0.0074, 0.008, 0.0083, 0.0085, 0.0088, 0.009, 0.0092, 0.0092, 0.0093, 0.0096, 0.0098, 0.0098, 0.0099, 0.01, 0.0101, 0.0104, 0.0106,
    0.0106, 0.0109, 0.0105, 0.0104, 0.01, 0.0094, 0.09, 0.0086, 0.08, 0.0075, 0.06, 0.0058, 0.05, 0.0044, 0.04, 0.0039, 0.0038, 0.0037, 0.0037, 0.0038, 0.0037, 0.0036, 0.0036,
    0.0035, 0.0036, 0.0036, 0.0034, 0.0033, 0.0033, 0.0033, 0.0035, 0.0036, 0.0037, 0.0038, 0.004, 0.0042, 0.004, 0.0041, 0.004, 0.0036, 0.0034, 0.0033, 0.0034, 0.0035, 0.004, 0.0042, 0.005, 0.0062,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


//27    27  0.0060  0.0064  0.0070  0.0076  0.0086  0.0086  0.0088  0.0089  0.0089  0.0083  0.0079  0.0078  0.0077  0.0067  0.0059  0.0057  0.0056  0.0046  0.0041  0.0045  0.0050  0.0056  0.0054  0.0053  0.0054  0.0055  0.0057  0.0065  0.0073  0.0079  0.0080  0.0075  0.0067  0.0068  0.0081  0.0114
const double ColorTemp::JDC468_Blackbl27_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0060, 0.0062, 0.0064, 0.0066, 0.0070, 0.0073, 0.0076, 0.008, 0.0086, 0.0086, 0.0086, 0.0087, 0.0088, 0.0088, 0.0089, 0.0089, 0.0089, 0.0085, 0.0083, 0.008, 0.0079, 0.0078, 0.0078,
    0.0078, 0.0077, 0.007, 0.0067, 0.006, 0.0059, 0.0057, 0.0057, 0.0057, 0.0056, 0.005, 0.0046, 0.0045, 0.0041, 0.0043, 0.0045, 0.0047, 0.0050, 0.0053, 0.0056, 0.0055, 0.0054, 0.0055,
    0.0053, 0.0053, 0.0054, 0.0054, 0.0055, 0.0056, 0.0057, 0.006, 0.0065, 0.007, 0.0073, 0.0075, 0.0079, 0.008, 0.0080, 0.008, 0.0075, 0.007, 0.0067, 0.0067, 0.0068, 0.007, 0.0081, 0.01, 0.0114,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//28    28  0.0092  0.0118  0.0155  0.0218  0.0267  0.0296  0.0312  0.0306  0.0282  0.0244  0.0205  0.0186  0.0167  0.0126  0.0091  0.0080  0.0071  0.0050  0.0042  0.0042  0.0044  0.0045  0.0045  0.0042  0.0041  0.0039  0.0040  0.0043  0.0048  0.0050  0.0047  0.0042  0.0041  0.0042  0.0049  0.0074
const double ColorTemp::JDC468_Blackbl28_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0092, 0.01, 0.0118, 0.012, 0.0155, 0.016, 0.0218, 0.025, 0.0267, 0.028, 0.0296, 0.03, 0.0312, 0.031, 0.0306, 0.03, 0.0282, 0.026, 0.0244, 0.022, 0.0205, 0.02, 0.0186, 0.017, 0.0167,
    0.015, 0.0126, 0.01, 0.0091, 0.0085, 0.0080, 0.0075, 0.0071, 0.006, 0.0050, 0.05, 0.0042, 0.0042, 0.0042, 0.0043, 0.0044, 0.0044, 0.0045, 0.0045, 0.0045, 0.0043, 0.0042, 0.0042,
    0.0041, 0.004, 0.0039, 0.004, 0.0040, 0.004, 0.0043, 0.0044, 0.0048, 0.005, 0.0050, 0.0048, 0.0047, 0.0045, 0.0042, 0.0041, 0.0041, 0.0041, 0.0042, 0.0045, 0.0049, 0.006, 0.0074,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//214   214 0.0072  0.0062  0.0061  0.0064  0.0069  0.0071  0.0075  0.0081  0.0082  0.0088  0.0103  0.0126  0.0145  0.0146  0.0140  0.0133  0.0118  0.0094  0.0070  0.0059  0.0051  0.0046  0.0045  0.0043  0.0045  0.0045  0.0048  0.0056  0.0065  0.0072  0.0067  0.0065  0.0057  0.0058  0.0068  0.0103
const double ColorTemp::JDC468_Blackgr214_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0072, 0.007, 0.0062, 0.0062, 0.0061, 0.0063, 0.0064, 0.0067, 0.0069, 0.007, 0.0071, 0.0072, 0.0075, 0.008, 0.0081, 0.008, 0.0082, 0.0085, 0.0088, 0.009, 0.0103, 0.011, 0.0126,
    0.012, 0.0145, 0.0145, 0.0146, 0.014, 0.0140, 0.014, 0.0133, 0.012, 0.0118, 0.01, 0.0094, 0.008, 0.0070, 0.006, 0.0059, 0.0055, 0.0051, 0.005, 0.0046, 0.0045, 0.0045, 0.0044,
    0.0043, 0.0044, 0.0045, 0.0045, 0.0045, 0.0046, 0.0048, 0.005, 0.0056, 0.006, 0.0065, 0.007, 0.0072, 0.007, 0.0067, 0.0066, 0.0065, 0.006, 0.0057, 0.0056, 0.0058, 0.006, 0.0068, 0.008, 0.0103,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};



//436   436 0.0095  0.0123  0.0173  0.0242  0.0293  0.0317  0.0310  0.0283  0.0246  0.0197  0.0152  0.0129  0.0112  0.0083  0.0063  0.0059  0.0056  0.0045  0.0043  0.0050  0.0059  0.0064  0.0062  0.0060  0.0060  0.0062  0.0066  0.0075  0.0086  0.0093  0.0089  0.0082  0.0073  0.0072  0.0088  0.0139
const double ColorTemp::JDC468_Blackbl436_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0095, 0.01, 0.0123, 0.013, 0.0173, 0.02, 0.0242, 0.026, 0.0293, 0.03, 0.0317, 0.031, 0.0310, 0.03, 0.0283, 0.025, 0.0246, 0.02, 0.0197, 0.018, 0.0152, 0.014, 0.0129, 0.012,
    0.0112, 0.01, 0.0083, 0.007, 0.0063, 0.006, 0.0059, 0.0058, 0.0056, 0.005, 0.0045, 0.0045, 0.0043, 0.005, 0.0050, 0.0055, 0.0059, 0.006, 0.0064, 0.0063, 0.0062, 0.006, 0.0060,
    0.006, 0.0060, 0.006, 0.0062, 0.0064, 0.0066, 0.007, 0.0075, 0.008, 0.0086, 0.009, 0.0093, 0.009, 0.0089, 0.0086, 0.0082, 0.008, 0.0073, 0.0072, 0.0072, 0.008, 0.0088, 0.01, 0.0139,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};



//455   455 0.1560  0.1943  0.2724  0.4469  0.5896  0.6393  0.6390  0.6239  0.6096  0.5895  0.5698  0.5662  0.5603  0.5304  0.4977  0.4975  0.4998  0.4601  0.4261  0.4520  0.4986  0.5237  0.5270  0.5260  0.5297  0.5375  0.5504  0.5711  0.5910  0.6023  0.6029  0.5977  0.5880  0.5830  0.5978  0.6345
const double ColorTemp::JDC468_Whitebl455_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1560, 0.18, 0.1943, 0.25, 0.2724, 0.40, 0.4469, 0.50, 0.5896, 0.60, 0.6393, 0.639, 0.6390, 0.625, 0.6239, 0.61, 0.6096, 0.60, 0.5895, 0.57, 0.5698, 0.567, 0.5662, 0.56,
    0.5603, 0.55, 0.5304, 0.51, 0.4977, 0.498, 0.4975, 0.498, 0.4998, 0.47, 0.4601, 0.44, 0.4261, 0.43, 0.4520, 0.47, 0.4986, 0.51, 0.5237, 0.525, 0.5270, 0.526, 0.5260, 0.528,
    0.5297, 0.53, 0.5375, 0.54, 0.5504, 0.56, 0.5711, 0.58, 0.5910, 0.595, 0.6023, 0.602, 0.6029, 0.600, 0.5977, 0.59, 0.5880, 0.585, 0.5830, 0.59, 0.5978, 0.61, 0.6345,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


//101   101 0.0076  0.0094  0.0116  0.0139  0.0158  0.0158  0.0152  0.0142  0.0130  0.0111  0.0094  0.0090  0.0085  0.0069  0.0058  0.0057  0.0058  0.0048  0.0044  0.0060  0.0094  0.0126  0.0141  0.0141  0.0148  0.0157  0.0172  0.0197  0.0227  0.0245  0.0242  0.0233  0.0215  0.0205  0.0240  0.0321
const double ColorTemp::JDC468_Blackvio101_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0076, 0.008, 0.0094, 0.01, 0.0116, 0.012, 0.0139, 0.015, 0.0158, 0.0158, 0.0158, 0.0156, 0.0152, 0.015, 0.0142, 0.014, 0.0130, 0.012, 0.0111, 0.01, 0.0094, 0.009, 0.0090, 0.009,
    0.0085, 0.008, 0.0069, 0.006, 0.0058, 0.0057, 0.0057, 0.0058, 0.0058, 0.005, 0.0048, 0.0045, 0.0044, 0.005, 0.0060, 0.008, 0.0094, 0.011, 0.0126, 0.013, 0.0141, 0.014, 0.0141, 0.0145, 0.0148, 0.015, 0.0157, 0.016, 0.0172, 0.018, 0.0197, 0.021, 0.0227, 0.023, 0.0245, 0.0241,
    0.0242, 0.0235, 0.0233, 0.022, 0.0215, 0.021, 0.0205, 0.022, 0.0240, 0.026, 0.0321,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//92    92  0.1652  0.2036  0.2848  0.4708  0.6230  0.6758  0.6761  0.6619  0.6502  0.6343  0.6195  0.6194  0.6169  0.5939  0.5677  0.5678  0.5695  0.5373  0.5105  0.5356  0.5778  0.6020  0.6073  0.6090  0.6124  0.6187  0.6295  0.6456  0.6603  0.6693  0.6716  0.6704  0.6660  0.6640  0.6749  0.7003
const double ColorTemp::JDC468_Whitebl92_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1652, 0.19, 0.2036, 0.24, 0.2848, 0.35, 0.4708, 0.55, 0.6230, 0.65, 0.6758, 0.6759, 0.6761, 0.665, 0.6619, 0.655, 0.6502, 0.64, 0.6343, 0.62, 0.6195, 0.6194, 0.6194, 0.618,
    0.6169, 0.605, 0.5939, 0.58, 0.5677, 0.5677, 0.5678, 0.568, 0.5695, 0.555, 0.5373, 0.52, 0.5105, 0.52, 0.5356, 0.55, 0.5778, 0.60, 0.6020, 0.605, 0.6073, 0.608, 0.6090, 0.61,
    0.6124, 0.615, 0.6187, 0.62, 0.6295, 0.64, 0.6456, 0.65, 0.6603, 0.665, 0.6693, 0.67, 0.6716, 0.671, 0.6704, 0.669, 0.6660, 0.665, 0.6640, 0.67, 0.6749, 0.69, 0.7003,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//94    94  0.0506  0.0569  0.0678  0.0906  0.1088  0.1160  0.1184  0.1195  0.1203  0.1215  0.1293  0.1474  0.1609  0.1538  0.1415  0.1466  0.1535  0.1364  0.1248  0.1494  0.1871  0.2098  0.2163  0.2179  0.2217  0.2282  0.2372  0.2500  0.2631  0.2713  0.2739  0.2725  0.2678  0.2660  0.2770  0.3025
const double ColorTemp::JDC468_Greyredbl94_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0506, 0.053, 0.0569, 0.06, 0.0678, 0.08, 0.0906, 0.1, 0.1088, 0.11, 0.1160, 0.117, 0.1184, 0.119, 0.1195, 0.12, 0.1203, 0.1205, 0.1215, 0.125, 0.1293, 0.133, 0.1474, 0.15,
    0.1609, 0.155, 0.1538, 0.145, 0.1415, 0.143, 0.1466, 0.15, 0.1535, 0.14, 0.1364, 0.13, 0.1248, 0.14, 0.1494, 0.17, 0.1871, 0.200, 0.2098, 0.21, 0.2163, 0.217, 0.2179, 0.22,
    0.2217, 0.222, 0.2282, 0.23, 0.2372, 0.24, 0.2500, 0.256, 0.2631, 0.27, 0.2713, 0.272, 0.2739, 0.273, 0.2725, 0.27, 0.2678, 0.267, 0.2660, 0.272, 0.2770, 0.28, 0.3025,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};




//32    32  0.0933  0.1333  0.1967  0.3132  0.4096  0.4556  0.4763  0.4845  0.4870  0.4852  0.4868  0.4959  0.4971  0.4755  0.4407  0.4085  0.3643  0.2922  0.2161  0.1658  0.1370  0.1190  0.1040  0.0947  0.0949  0.1001  0.1106  0.1306  0.1536  0.1652  0.1609  0.1480  0.1291  0.1188  0.1370  0.1933
const double ColorTemp::JDC468_Blue32_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0933, 0.11,   0.1333, 0.17, 0.1967, 0.265, 0.3132, 0.35, 0.4096, 0.43, 0.4556, 0.465, 0.4763, 0.48, 0.4845, 0.486, .4870, 0.486, 0.4852, 0.486, 0.4868, 0.49, 0.4959, 0.496,
    0.4971, 0.48, 0.4755,  0.46, 0.4407, 0.42, 0.4085, 0.38, 0.3643, 0.32, 0.2922, 0.25, 0.2161, 0.19, 0.1658, 0.15, 0.1370, 0.12, 0.1190, 0.11,  0.1040, 0.10, 0.0947, 0.0948,
    0.0949, 0.097, 0.1001, 0.105, 0.1106, 0.12, 0.1306, 0.14, 0.1536, 0.16, 0.1652,  0.162, 0.1609, 0.154, 0.1480,  0.135, 0.1291, 0.12, 0.1188, 0.125, 0.1370, 0.16, 0.1933,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};



//236   236 0.0165  0.0259  0.0432  0.0684  0.0858  0.0895  0.0828  0.0683  0.0527  0.0365  0.0235  0.0177  0.0143  0.0096  0.0067  0.0063  0.0060  0.0052  0.0053  0.0061  0.0074  0.0083  0.0085  0.0080  0.0078  0.0076  0.0081  0.0095  0.0112  0.0121  0.0114  0.0100  0.0086  0.0083  0.0102  0.0180
const double ColorTemp::JDC468_Blue236_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0165, 0.021, 0.0259, 0.035, 0.0432, 0.05, 0.0684, 0.075, 0.0858, 0.087, 0.0895, 0.085, 0.0828, 0.075, 0.0683, 0.058, 0.0527, 0.045, 0.0365, 0.031, 0.0235, 0.021, 0.0177, 0.016,
    0.0143, 0.012, 0.0096, 0.0085, 0.0067, 0.0065, 0.0063, 0.0062, 0.0060, 0.006,  0.0052, 0.0052,  0.0053, 0.006,  0.0061, 0.007, 0.0074, 0.008, 0.0083, 0.0084, 0.0085, 0.0083,
    0.0080, 0.008,  0.0078, 0.0077, 0.0076, 0.008,  0.0081, 0.009, 0.0095, 0.01, 0.0112, 0.012, 0.0121, 0.012, 0.0114, 0.011, 0.0100, 0.009, 0.0086, 0.0085, 0.0083, 0.009, 0.0102, 0.0015, 0.0180,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


//300   300 0.0079  0.0085  0.0087  0.0092  0.0094  0.0093  0.0093  0.0095  0.0097  0.0110  0.0158  0.0544  0.1907  0.3753  0.4883  0.5181  0.5044  0.4631  0.4070  0.3594  0.3262  0.3054  0.2873  0.2772  0.2790  0.2875  0.3029  0.3292  0.3571  0.3719  0.3693  0.3573  0.3371  0.3253  0.3471  0.4087
const double ColorTemp::JDC468_Gre300_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0079, 0.008, 0.0085, 0.0086, 0.0087, 0.009,  0.0092, 0.0093, 0.0094, 0.0094,  0.0093, 0.0093, 0.0093, 0.0094, 0.0095, 0.0096, 0.0097, 0.01, 0.0110, 0.012, 0.0158, 0.045,
    0.0544, 0.134, 0.1907, 0.25, 0.3753, 0.435, 0.4883, 0.505, 0.5181, 0.512,  0.5044, 0.485, 0.4631, 0.433, 0.4070, 0.387,  0.3594, 0.334, 0.3262, 0.31,  0.3054,  0.295, 0.2873, 0.292,
    0.2772, 0.278, 0.2790, 0.281,  0.2875, 0.291,  0.3029, 0.3121,  0.3292, 0.333,  0.3571, 0.365,  0.3719,  0.365, 0.3693,  0.362, 0.3573, 0.342,  0.3371, 0.325,  0.3253, 0.336,  0.3471, 0.367,  0.4087,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//340   340 0.0175  0.0320  0.0587  0.0989  0.1267  0.1322  0.1210  0.0978  0.0732  0.0487  0.0297  0.0212  0.0167  0.0106  0.0069  0.0064  0.0062  0.0051  0.0052  0.0061  0.0073  0.0081  0.0080  0.0076  0.0075  0.0072  0.0077  0.0088  0.0105  0.0112  0.0104  0.0092  0.0079  0.0075  0.0092  0.0167
const double ColorTemp::JDC468_Blue340_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0175, 0.02, 0.0320, 0.04, 0.0587, 0.08, 0.0989, 0.11, 0.1267, 0.13,  0.1322, 0.125, 0.1210, 0.111, 0.0978, 0.08, 0.0732, 0.06, 0.0487, 0.04, 0.0297, 0.025, 0.0212, 0.02,
    0.0167, 0.012, 0.0106, 0.008,  0.0069, 0.0065, 0.0064, 0.0063,  0.0062, 0.0055, 0.0051, 0.0051,  0.0052, 0.006, 0.0061, 0.007, 0.0073, 0.008, 0.0081, 0.008, 0.0080, 0.0078,
    0.0076, 0.0076, 0.0075, 0.0074, 0.0072, 0.0075, 0.0077, 0.008, 0.0088, 0.0092, 0.0105, 0.011, 0.0112, 0.0108, 0.0104, 0.01, 0.0092, 0.008,  0.0079, 0.0075, 0.0075, 0.008,
    0.0092, 0.01,  0.0167,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


//110   110 0.0954  0.1234  0.1702  0.2631  0.3363  0.3664  0.3799  0.3905  0.4002  0.4160  0.4582  0.5262  0.5798  0.5915  0.5742  0.5465  0.5035  0.4364  0.3581  0.2977  0.2589  0.2349  0.2152  0.2030  0.2032  0.2106  0.2249  0.2511  0.2799  0.2951  0.2915  0.2776  0.2552  0.2419  0.2638  0.3287
const double ColorTemp::JDC468_Gree110_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0954, 0.11, 0.1234, 0.15, 0.1702, 0.22, 0.2631, 0.312, 0.3363, 0.352, 0.3664, 0.372, 0.3799, 0.385, 0.3905, 0.395, 0.4002, 0.41, 0.4160, 0.43, 0.4582, 0.511,  0.5262, 0.544,
    0.5798, 0.585, 0.5915, 0.585, 0.5742, 0.563, 0.5465, 0.523, 0.5035, 0.465, 0.4364, 0.389, 0.3581, 0.3334, 0.2977, 0.275, 0.2589, 0.245,  0.2349, 0.223, 0.2152, 0.211,  0.2030, 0.204,
    0.2032, 0.206, 0.2106, 0.221, 0.2249, 0.243, 0.2511, 0.271,  0.2799, 0.286, 0.2951, 0.294, 0.2915, 0.2876, 0.2776, 0.2657, 0.2552, 0.2456, 0.2419, 0.253,  0.2638, 0.275,  0.3287,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//457   457 0.0127  0.0126  0.0112  0.0110  0.0109  0.0106  0.0105  0.0104  0.0110  0.0136  0.0318  0.1246  0.3262  0.5051  0.5566  0.5181  0.4406  0.3429  0.2411  0.1647  0.1202  0.0968  0.0804  0.0709  0.0703  0.0747  0.0840  0.1023  0.1243  0.1355  0.1311  0.1179  0.0993  0.0884  0.1043  0.1590
const double ColorTemp::JDC468_Gree457_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0127, 0.0126, 0.0126, 0.012, 0.0112, 0.011, 0.0110, 0.011,  0.0109, 0.0107, 0.0106, 0.0105,  0.0105, 0.0105,  0.0104, 0.0107,  0.0110, 0.0124,  0.0136, 0.0234,  0.0318, 0.09,
    0.1246, 0.22,  0.3262, 0.43, 0.5051, 0.52, 0.5566, 0.54, 0.5181, 0.476, 0.4406, 0.398, 0.3429, 0.296, 0.2411, 0.203, 0.1647, 0.142, 0.1202, 0.1,  0.0968, 0.09,  0.0804, 0.08,
    0.0709, 0.0708, 0.0703, 0.0723,  0.0747, 0.08, 0.0840, 0.09,  0.1023, 0.11, 0.1243, 0.13, 0.1355, 0.132,  0.1311, 0.12, 0.1179, 0.10,  0.0993, 0.09,  0.0884, 0.09, 0.1043, 0.12,  0.1590,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//241   241 0.0134  0.0106  0.0110  0.0108  0.0111  0.0114  0.0114  0.0115  0.0114  0.0122  0.0192  0.0731  0.2455  0.4689  0.6183  0.6852  0.7107  0.7112  0.7059  0.7177  0.7335  0.7445  0.7487  0.7523  0.7555  0.7606  0.7683  0.7779  0.7855  0.7915  0.7964  0.8011  0.8056  0.8097  0.8144  0.8239
const double ColorTemp::JDC468_Yel241_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0134, 0.012, 0.0106, 0.011, 0.0110, 0.011, 0.0108, 0.011, 0.0111, 0.0112, 0.0114, 0.0114, 0.0114, 0.0114, 0.0115, 0.0114, 0.0114, 0.012, 0.0122, 0.017, 0.0192, 0.05, 0.0731, 0.12,
    0.2455, 0.355, 0.4689, 0.556, 0.6183, 0.645, 0.6852, 0.698, 0.7107, 0.711, 0.7112, 0.708, 0.7059, 0.712, 0.7177, 0.724, 0.7335, 0.742, 0.7445, 0.746, 0.7487, 0.751,  0.7523, 0.753,
    0.7555, 0.758, 0.760, 0.7626, 0.7683, 0.771,  0.7779, 0.782, 0.7855, 0.791, 0.7915, 0.794, 0.7964, 0.799, 0.8011, 0.804, 0.8056, 0.807,  0.8097, 0.811,  0.8144, 0.820,  0.8239,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


//321   321 0.0247  0.0203  0.0182  0.0183  0.0182  0.0179  0.0182  0.0188  0.0199  0.0249  0.0529  0.1519  0.3116  0.4138  0.4410  0.4679  0.4906  0.4655  0.4517  0.5203  0.6238  0.6952  0.7270  0.7406  0.7469  0.7527  0.7607  0.7708  0.7786  0.7849  0.7897  0.7940  0.7984  0.8025  0.8069  0.8160
const double ColorTemp::JDC468_Ora321_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0247, 0.022, 0.0203, 0.019, 0.0182, 0.0182, 0.0183, 0.0182,  0.0182, 0.018, 0.0179, 0.018, 0.0182, 0.0185, 0.0188, 0.019, 0.0199, 0.022, 0.0249, 0.035, 0.0529, 0.112,
    0.1519, 0.231, 0.3116, 0.365, 0.4138, 0.421, 0.4410, 0.451, 0.4679, 0.485, 0.4906, 0.473, 0.4655, 0.455, 0.4517, 0.487, 0.5203, 0.578, 0.6238, 0.667, 0.6952, 0.711, 0.7270, 0.735,
    0.7406, 0.743, 0.7469, .751, 0.7527, 0.756, 0.7607, 0.765, 0.7708, 0.774, 0.7786, 0.782, 0.7849, 0.786, 0.7897, 0.79, 0.7940, 0.794, 0.7984, 0.799, 0.8025, 0.805, 0.8069, 0.811, 0.8160,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
//353   353 0.0260  0.0294  0.0331  0.0420  0.0490  0.0517  0.0541  0.0572  0.0608  0.0664  0.0813  0.1070  0.1291  0.1344  0.1317  0.1356  0.1390  0.1289  0.1195  0.1278  0.1420  0.1493  0.1500  0.1500  0.1527  0.1576  0.1651  0.1759  0.1870  0.1942  0.1963  0.1952  0.1910  0.1898  0.1995  0.2209
const double ColorTemp::JDC468_Yellow353_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0260, 0.027, 0.0294, 0.031, 0.0331, 0.037, 0.0420, 0.045, 0.0490, 0.051, 0.0517, 0.053, 0.0541, 0.056, 0.0572, 0.059, 0.0608, 0.063, 0.0664, 0.072, 0.0813, 0.096, 0.1070, 0.112,
    0.1291, 0.132, 0.1344, 0.133, 0.1317, 0.134, 0.1356, 0.137, 0.1390, 0.134, 0.1289, 0.123, 0.1195, 0.122, 0.1278, 0.134, 0.1420, 0.145, 0.1493, 0.15, 0.1500, 0.15, 0.1500, 0.153,
    0.1527, 0.154, 0.1576, 0.162, 0.1651, 0.172, 0.1759, 0.182, 0.1870, 0.191,  0.1942, 0.195, 0.1963, 0.196, 0.1952, 0.193, 0.1910, 0.19, 0.1898, 0.195, 0.1995, 0.21, 0.2209,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


//465   465 0.0388  0.0654  0.1020  0.1557  0.1880  0.1783  0.1434  0.1013  0.0684  0.0410  0.0219  0.0149  0.0117  0.0080  0.0062  0.0062  0.0062  0.0056  0.0063  0.0098  0.0230  0.0440  0.0577  0.0617  0.0645  0.0690  0.0766  0.0903  0.1064  0.1144  0.1113  0.1022  0.0898  0.0833  0.0962  0.1377
const double ColorTemp::JDC468_Mag465_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0388, 0.05, 0.0654, 0.09, 0.1020, 0.12, 0.1557, 0.17, 0.1880, 0.182, 0.1783, 0.162, 0.1434, 0.12, 0.1013, 0.09, 0.0684, 0.05, 0.0410, 0.03, 0.0219, 0.02, 0.0149, 0.012,
    0.0117, 0.009, 0.0080, 0.007, 0.0062, 0.0062, 0.0062, 0.0062, 0.0062, 0.006, 0.0056, 0.006,  0.0063, 0.008, 0.0098, 0.018, 0.0230, 0.03, 0.0440, 0.05, 0.0577, 0.06, 0.0617, 0.062,
    0.0645, 0.065, 0.0690, 0.07, 0.0766, 0.08,  0.0903, 0.1, 0.1064, 0.111,  0.1144, 0.112, 0.1113, 0.104, 0.1022, 0.09, 0.0898, 0.088, 0.0833, 0.09, 0.0962, 0.111,  0.1377,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


//333   333 0.1030  0.1508  0.2133  0.3191  0.3910  0.3895  0.3424  0.2770  0.2184  0.1593  0.1074  0.0825  0.0669  0.0430  0.0265  0.0278  0.0315  0.0212  0.0163  0.0355  0.0861  0.1365  0.1565  0.1589  0.1629  0.1713  0.1852  0.2099  0.2378  0.2517  0.2469  0.2322  0.2102  0.1973  0.2191  0.2855
const double ColorTemp::JDC468_Mag333_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1030, 0.12, 0.1508, 0.18, 0.2133, 0.26, 0.3191, 0.364, 0.3910, 0.39, 0.3895, 0.375, 0.3424, 0.312, 0.2770, 0.251, 0.2184, 0.183, 0.1593, 0.122, 0.1074, 0.1, 0.0825, 0.07,
    0.0669, 0.05, 0.0430, 0.03, 0.0265, 0.027, 0.0278, 0.03, 0.0315, 0.025, 0.0212, 0.02, 0.0163, 0.03, 0.0355, 0.05, 0.0861, 0.112, 0.1365, 0.143, 0.1565, 0.157, 0.1589, 0.16,
    0.1629, 0.165, 0.1713, 0.175, 0.1852, 0.19, 0.2099, 0.221, 0.2378, 0.245, 0.2517, 0.25, 0.2469, 0.238, 0.2322, 0.224, 0.2102, 0.206, 0.1973, 0.209, 0.2191, 0.231, 0.2855,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


//203   203 0.0833  0.1329  0.2005  0.3099  0.3855  0.3916  0.3530  0.2926  0.2346  0.1741  0.1201  0.0934  0.0759  0.0495  0.0306  0.0308  0.0330  0.0214  0.0150  0.0256  0.0510  0.0723  0.0769  0.0748  0.0761  0.0813  0.0911  0.1087  0.1295  0.1399  0.1353  0.1232  0.1064  0.0971  0.1137  0.1677
const double ColorTemp::JDC468_Mag203_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0833, 0.11, 0.1329, 0.16, 0.2005, 0.25, 0.3099, 0.342, 0.3855, 0.39, 0.3916, 0.374, 0.3530, 0.321, 0.2926, 0.267, 0.2346, 0.21, 0.1741, 0.153, 0.1201, 0.1, 0.0934, 0.08,
    0.0759, 0.06, 0.0495, 0.04, 0.0306, 0.0307, 0.0308, 0.032, 0.0330, 0.025, 0.0214, 0.018, 0.0150, 0.02, 0.0256, 0.04, 0.0510, 0.06, 0.0723, 0.074, 0.0769, 0.075,  0.0748, 0.076,
    0.0761, 0.08, 0.0813, 0.09, 0.0911, 0.1, 0.1087, 0.115, 0.1295, 0.134, 0.1399, 0.136, 0.1353, 0.131, 0.1232, 0.114, 0.1064, 0.1, 0.0971, 0.105, 0.1137, 0.123, 0.1677,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
//blue cyan
const double ColorTemp::J570_BlueB6_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1611, 0.18, 0.1947, 0.23, 0.2639, 0.31, 0.3488, 0.37, 0.4022, 0.43, 0.4517, 0.49, 0.501, 0.52, 0.5317, 0.534, 0.5367, 0.53, 0.5246, 0.51, 0.4905, 0.47, 0.4510, 0.43, 0.4059,
    0.37, 0.3351, 0.31, 0.2612, 0.24, 0.2177, 0.205, 0.1883, 0.16, 0.1444, 0.123, 0.1065, 0.1, 0.0889, 0.085, 0.0811, 0.08, 0.0757, 0.071, 0.0695, 0.067, 0.0648, 0.064, 0.0634,
    0.063, 0.0637,  0.065, 0.0662, 0.068, 0.0714, 0.075, 0.0787, 0.08,  0.0828, 0.082,  0.0822, 0.08, 0.0781, 0.075, 0.0726,  0.071, 0.0698, 0.075, 0.0770, 0.08,   0.0973,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueB15_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2207, 0.2, 0.1897, 0.21,  0.2274, 0.25, 0.2805, 0.30, 0.3216, 0.34, 0.3596, 0.38, 0.4003, 0.42, 0.4438, 0.47, 0.4909, 0.52, 0.5469, 0.57, 0.6009, 0.62, 0.6414, 0.65, 0.6557,
    0.64, 0.6275, 0.59, 0.5675, 0.53, 0.5022, 0.48, 0.4348, 0.39, 0.3473, 0.31, 0.2613, 0.24, 0.2039, 0.18, 0.1696, 0.15, 0.1465, 0.13, 0.1243, 0.11,  0.1055, 0.10, 0.0952, 0.094,
    0.0911, 0.091, 0.0906, 0.092, 0.0940, 0.1, 0.1006, 0.104, 0.1075, 0.11, 0.1125, 0.113,  0.1146, 0.113, 0.1129, 0.112, 0.1120, 0.115, 0.1198, 0.13,  0.1431,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueC2_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2534, 0.25, 0.2454, 0.28, 0.3101, 0.33, 0.3567, 0.37, 0.3981, 0.41, 0.4258, 0.43, 0.4472, 0.45, 0.4574, 0.455, 0.4571, 0.45,  0.4468, 0.43, 0.4205, 0.41, 0.3947, 0.38,
    0.3647, 0.33, 0.3074, 0.27, 0.2471, 0.23, 0.2221, 0.22, 0.2128, 0.20,  0.1810, 0.17, 0.1510, 0.15,  0.1486, 0.15, 0.1546, 0.154, 0.1532, 0.15, 0.1426, 0.135, 0.1310, 0.125,
    0.1257, 0.125,  0.1246, 0.126,  0.1276, 0.13, 0.1360, 0.14, 0.1483, 0.15, 0.1566, 0.157, 0.1575, 0.155, 0.1535, 0.15, 0.1453, 0.143,  0.1403, 0.146, 0.1510, 0.16,  0.1855,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueC14_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.5697, 0.51, 0.4660, 0.49, 0.5000, 0.53, 0.5560, 0.58,  0.6072, 0.62, 0.6402, 0.65, 0.6632, 0.67, 0.6850, 0.69, 0.7069, 0.71, 0.7292, 0.74, 0.7488, 0.75, 0.7678, 0.77, 0.7786,
    0.775, 0.7721, 0.76, 0.7544, 0.75,  0.7394, 0.73, 0.7232, 0.70, 0.6889, 0.66, 0.6446, 0.63, 0.6171, 0.61, 0.5966, 0.58, 0.5743, 0.56, 0.5425, 0.53, 0.5093, 0.50,  0.4884, 0.48,
    0.4784, 0.475, 0.4774, 0.48, 0.4822, 0.49, 0.4944, 0.50, 0.5076, 0.51,  0.5186, 0.52, 0.5268, 0.53, 0.5303, 0.532, 0.5332, 0.54, 0.5454, 0.56,  0.5760,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueC16_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1909, 0.10, 0.0635, 0.06, 0.0554, 0.056,  0.0571, 0.057,  0.0580, 0.06, 0.0608, 0.063, 0.0662, 0.07,  0.0712, 0.072,  0.0742, 0.08,   0.0811, 0.09, 0.0985, 0.11, 0.1363, 0.15,
    0.1743, 0.173, 0.1720, 0.15, 0.1372, 0.12,  0.1005, 0.09, 0.0731, 0.06, 0.0509, 0.04,  0.0374, 0.036,   0.0322, 0.032,  0.0308, 0.031,  0.0309, 0.031,  0.0319, 0.033,
    0.0333, 0.034,  0.0349, 0.035,  0.0364, 0.037,  0.0377, 0.038,  0.0386, 0.0386, 0.0386, 0.0384, 0.0383, 0.038,  0.0377, 0.0377, 0.0378, 0.038,  0.0386, 0.04,   0.0417, 0.043,
    0.0461, 0.049,  0.0514,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueF1_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0060, 0.04,  0.0975, 0.11, 0.1297, 0.13, 0.1426, 0.15,   0.1538, 0.16,   0.1624, 0.165,  0.1684, 0.17,   0.1727, 0.175,  0.1761, 0.178,  0.1798, 0.18,   0.1863, 0.19,   0.1993, 0.20,
    0.2091, 0.205, 0.2041, 0.20,    0.1902, 0.19,   0.1836, 0.182,  0.1809, 0.17,   0.1675, 0.16,   0.1527, 0.151,  0.1511, 0.153,  0.1541, 0.154,  0.1545, 0.153,  0.1516, 0.15,
    0.1484, 0.149,  0.1489, 0.15,   0.1516, 0.153,  0.1563, 0.16,   0.1654, 0.17,   0.1766, 0.18,   0.1827, 0.182,  0.1817, 0.18,   0.1766, 0.17,   0.1693, 0.166,  0.1643, 0.17,   0.1726, 0.18,   0.1978,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueF2_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1613, 0.20, 0.2384, 0.30, 0.3288, 0.35,   0.4026, 0.43, 0.4630, 0.48, 0.4992, 0.51, 0.5294, 0.53, 0.5377, 0.53, 0.5261, 0.51, 0.4995, 0.48, 0.4513, 0.43, 0.4064, 0.38,
    0.3620, 0.31, 0.2935, 0.25, 0.2263, 0.21, 0.1977, 0.19, 0.1851, 0.17, 0.1519, 0.14, 0.1213, 0.12, 0.1149, 0.113, 0.1162, 0.114, 0.1133, 0.11, 0.1053, 0.10, 0.0976, 0.095,
    0.0947, 0.095,  0.0953, 0.097,  0.0985, 0.1, 0.1070, 0.11,  0.1188, 0.12,   0.1256, 0.124,  0.1246, 0.12,   0.1187, 0.11,   0.1099, 0.105,  0.1048, 0.11,   0.1145, 0.12,   0.145,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueF10_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.3003, 0.20,   0.1512, 0.20,   0.2149, 0.25,   0.2709, 0.30, 0.3127, 0.32, 0.3534, 0.37, 0.4001, 0.42, 0.4456, 0.46, 0.4882, 0.51, 0.5343, 0.55,   0.5728, 0.58,   0.5968, 0.595,
    0.5932, 0.56, 0.5490, 0.51, 0.4756, 0.43, 0.3989, 0.37, 0.3256, 0.28, 0.2424, 0.21, 0.1691, 0.15,   0.1239, 0.10,   0.0998, 0.09,   0.0866, 0.08, 0.0752, 0.07, 0.0664, 0.065,
    0.0623, 0.062,  0.0616, 0.062,  0.0626, 0.064,  0.0662, 0.07,   0.0720, 0.075, 0.0767, 0.077, 0.0784, 0.078,  0.0770, 0.075,  0.0734, 0.073,  0.0716, 0.075,  0.0779, 0.08,   0.0968,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueF13_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.3075, 0.22,   0.1514, 0.13,   0.1216, 0.125,  0.1264, 0.13,   0.1312, 0.14,   0.1448, 0.15,   0.1665, 0.17,   0.1850, 0.19,   0.1922, 0.195,  0.1970, 0.20,   0.2036, 0.21,
    0.2136, 0.213,  0.2122, 0.19,   0.1832, 0.16,   0.1412, 0.12,   0.1066, 0.09,   0.0810, 0.07,   0.0579, 0.05,   0.0422, 0.04,   0.0358, 0.035,  0.0336, 0.033,  0.0335, 0.0335,
    0.0339, 0.034,  0.0348, 0.035,  0.0358, 0.036,  0.0371, 0.038,  0.0384, 0.039,  0.0394, 0.04,   0.0405, 0.0404, 0.0405, 0.04,   0.0399, 0.039,  0.0387, 0.039,
    0.0396, 0.04,   0.0422, 0.045,  0.0467, 0.05,   0.0527,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueG9_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0937, 0.09,   0.0891, 0.07,   0.0621, 0.07,   0.0735, 0.08,   0.0842, 0.09,   0.0918, 0.1,    0.1009, 0.105,  0.1091, 0.11,   0.1158, 0.12,   0.1254, 0.13,   0.1452, 0.16,
    0.1805, 0.20,   0.2126, 0.213,  0.2128, 0.20,   0.1863, 0.17,   0.1550, 0.14,   0.1270, 0.1,    0.0973, 0.08,   0.0725, 0.06,   0.0585, 0.055,  0.0521, 0.05,   0.0486, 0.047,
    0.0458, 0.045,  0.0441, 0.044,  0.0439, 0.044,  0.0443, 0.045,  0.0455, 0.046,  0.0478, 0.049,  0.0500, 0.051,  0.0512, 0.051,  0.0511, 0.05,   0.0498, 0.049,
    0.0487, 0.049,  0.0495, 0.05,   0.0531, 0.055,  0.0620,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueG19_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1198, 0.12,   0.1376, 0.15,   0.1735, 0.18,   0.1903, 0.20,   0.2116, 0.22,   0.2322, 0.24,   0.2555, 0.27,   0.2802, 0.29,   0.3077, 0.32,   0.3420, 0.36,   0.3822, 0.41,
    0.4261, 0.43,   0.4565, 0.45, 0.4481, 0.43, 0.4109, 0.39,   0.3736, 0.35,   0.3368, 0.30,   0.2811, 0.25,   0.2228, 0.19,   0.1839, 0.17,   0.1592, 0.15,   0.1394, 0.12,
    0.1183, 0.1,    0.0998, 0.09,   0.0897, 0.085,  0.0851, 0.085,  0.0844, 0.085,  0.0863, 0.09,   0.0912, 0.095,  0.0977, 0.1,    0.1027, 0.105,  0.1064, 0.106,  0.1064, 0.106,
    0.1067, 0.11,   0.1135, 0.12,   0.1319,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueI5_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1096, 012, 0.1546, 0.16,  0.1855, 0.2,    0.2350, 0.25,   0.2720, 0.29,   0.3065, 0.32,   0.3404, 0.36,   0.3782, 0.4,    0.4229, 0.45,   0.4801, 0.51,   0.5416, 0.57,
    0.5962, 0.61,   0.6281, 0.62,   0.6144, 0.58,   0.5680, 0.55,   0.5211, 0.5,    0.4726, 0.43,   0.3991, 0.37,   0.3209, 0.29,   0.2674, 0.25,   0.2311, 0.22,   0.2000, 0.18,
    0.1656, 0.15,   0.1349, 0.12,   0.1161, 0.11,   0.1078, 0.105,  0.1046, 0.105,  0.1049, 0.107,  0.1097, 0.11,   0.1179, 0.12,   0.1264, 0.13,   0.1337, 0.135,  0.1378, 0.14,   0.1402, 0.145,  0.1474, 0.15,   0.1662,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueH15_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1047, 0.105,  0.1078, 0.12,   0.1550, 0.20,   0.2110, 0.23,   0.2487, 0.27,   0.2893, 0.31,   0.3386, 0.35,   0.3723, 0.375,  0.3773, 0.37,   0.3665, 0.35,   0.3373, 0.32,
    0.3037, 0.28,   0.2663, 0.24,   0.2106, 0.17,   0.1540, 0.13,   0.1200, 0.11,   0.0980, 0.08,   0.0721, 0.06,   0.0527, 0.05,   0.0448, 0.043,  0.0418, 0.041,  0.0403, 0.04,
    0.0390, 0.039,  0.0386, 0.039,  0.0392, 0.04,   0.0401, 0.041,  0.0413, 0.042,  0.0435, 0.044,  0.0456, 0.046,  0.0465, 0.046,  0.0457, 0.045,  0.0447, 0.044,  0.0435, 0.044,
    0.0442, 0.046,  0.0485, 0.05,   0.0576,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueI3_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1759, 0.21,   0.2647, 0.31,   0.3209, 0.36,   0.4249, 0.45,   0.4957, 0.52,   0.5536, 0.57,   0.6073, 0.62,   0.6431, 0.65,   0.6538, 0.653,  0.6506, 0.64,   0.6282, 0.61,
    0.5973, 0.57,   0.5559, 0.52,   0.4856, 0.43,   0.4033, 0.37,   0.3417, 0.32,   0.2901, 0.26,   0.2211, 0.19,   0.1585, 0.14,   0.1236, 0.11,   0.1062, 0.1,    0.0959, 0.09,
    0.0865, 0.085,  0.0801, 0.08,   0.0780, 0.079,  0.0788, 0.08,   0.0819, 0.09,   0.0903, 0.1,    0.1009, 0.105,  0.1073, 0.106,  0.1055, 0.1,    0.0998, 0.095,  0.0914, 0.09,   0.0863, 0.09,   0.0961, 0.11,   0.1250,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueI19_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.3031, 0.28,   0.2588, 0.25,   0.2472, 0.26,   0.2738, 0.29,   0.3035, 0.31,   0.3271, 0.33,   0.3496, 0.36,   0.3724, 0.38,   0.3968, 0.41,   0.4252, 0.44,   0.4562, 0.47,
    0.4899, 0.5,    0.5147, 0.512,  0.5100, 0.49,   0.4841, 0.47,   0.4608, 0.45,   0.4373, 0.42,   0.3934, 0.37,   0.3420, 0.32,   0.3086, 0.30,   0.2850, 0.27,   0.2622, 0.25,
    0.2334, 0.22,   0.2054, 0.19,   0.1887, 0.185,  0.1806, 0.18,   0.1791, 0.18,   0.1825, 0.185,  0.1909, 0.2,    0.2018, 0.21,   0.2108, 0.213,  0.2164, 0.217,  0.2180, 0.219,
    0.2198, 0.22,   0.2295, 0.24,   0.2562,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueJ4_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0290, 0.02,  0.0598, 0.1,    0.1278, 0.15,   0.1764, 0.19,   0.2052, 0.21,   0.2331, 0.25,   0.2642, 0.28,   0.2906, 0.3,    0.3093, 0.31,   0.3263, 0.33,   0.3413, 0.35,
    0.3550, 0.354,  0.3532, 0.33,   0.3206, 0.3,    0.2685, 0.24,   0.2212, 0.2,    0.1802, 0.16,   0.1346, 0.1,    0.0965, 0.08,   0.0751, 0.07,   0.0646, 0.06,   0.0586, 0.055,
    0.0535, 0.052,  0.0504, 0.05,   0.0491, 0.0495, 0.0493, 0.05,   0.0504, 0.051,  0.0529, 0.053,  0.0565, 0.057,  0.0591, 0.059,  0.0593, 0.058,  0.0576, 0.056,  0.0556, 0.055,
    0.0553, 0.06,   0.0602, 0.065,  0.0727,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueJ6_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.3039, 0.3,    0.2571, 0.27,   0.2960, 0.31,   0.3446, 0.37,   0.3937, 0.41,   0.4405, 0.47,   0.4884, 0.51,   0.5337, 0.56,   0.5771, 0.59,   0.6221, 0.64,   0.6589, 0.67,
    0.6808, 0.68,   0.6770, 0.65,   0.6375, 0.59,   0.5693, 0.53,   0.4933, 0.45,   0.4160, 0.39,   0.3223, 0.26,   0.2338, 0.21,   0.1761, 0.16,   0.1447, 0.13,   0.1258, 0.12,
    0.1090, 0.1,    0.0961, 0.09,   0.0899, 0.089,  0.0877, 0.088,  0.0892, 0.09,   0.0949, 0.1,    0.1045, 0.11,   0.1115, 0.112,  0.1131, 0.112,  0.1115, 0.11,   0.1065, 0.105,
    0.1033, 0.11,   0.1122, 0.12,   0.1400,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueJ11_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1274, 0.1,    0.091, 0.08,    0.0656, 0.064,  0.0604, 0.06,   0.0570, 0.06,   0.0604, 0.061,  0.0644, 0.065,  0.0668, 0.07,   0.0700, 0.072,  0.0754, 0.08,   0.0874, 0.1,
    0.1111, 0.12,   0.1327, 0.132,  0.1313, 0.12,   0.1127, 0.1,    0.0931, 0.08,   0.0758, 0.06,   0.0580, 0.05,   0.0449, 0.04,   0.0385, 0.036,  0.0360, 0.035,  0.0351, 0.035,
    0.0351, 0.035,  0.0355, 0.036,  0.0371, 0.0375, 0.0379, 0.038,  0.0388, 0.04,   0.0406, 0.041,  0.0414, 0.0415, 0.0416, 0.041,  0.0409, 0.04,
    0.0398, 0.04,   0.0397, 0.04,   0.0424, 0.043,  0.0458, 0.05,   0.0522,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::J570_BlueJ13_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2802, 0.28,   0.2820, 0.3,    0.3461, 0.4,    0.4357, 0.45,   0.5027, 0.53,   0.5528, 0.58,   0.6001, 0.62,   0.6402, 0.66,   0.6730, 0.69,   0.7019, 0.71,   0.7216, 0.725,
    0.7288, 0.72,   0.7172, 0.7,    0.6779, 0.65,   0.6160, 0.58,   0.5478, 0.52,   0.4751, 0.43,   0.3816, 0.35,   0.2882, 0.25,   0.2260, 0.2,    0.1905, 0.18,   0.1699, 0.16,
    0.1514, 0.14,   0.1361, 0.13,   0.1295, 0.129,  0.1281, 0.13,   0.1309, 0.135,  0.1408, 0.145,  0.1546, 0.16,   0.1644, 0.165, 0.1656, 0.162,
    0.1606, 0.155,  0.1516, 0.15,   0.1461, 0.15,   0.1585, 0.16,   0.1975,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::J570_BlueK5_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1727, 0.18,   0.1814, 0.19,   0.1931, 0.21,   0.2379, 0.25,   0.2728, 0.29,   0.3054, 0.31,   0.3396, 0.36,   0.3780, 0.4,    0.4243, 0.45,   0.4829, 0.51,   0.5464, 0.57,
    0.6024, 0.62,   0.6348, 0.63,   0.6209, 0.59,   0.5741, 0.55,   0.5258, 0.49,   0.4755, 0.43,   0.4003, 0.37,   0.3204, 0.3,    0.2656, 0.25,   0.2282, 0.21,   0.1966, 0.17,
    0.1624, 0.15,   0.1322, 0.12,   0.1137, 0.11,   0.1052, 0.105,  0.1025, 0.102,  0.1028, 0.105,  0.1075, 0.11,   0.1155, 0.12,   0.1237, 0.13,   0.1310, 0.133,
    0.1352, 0.136,  0.1370, 0.14,   0.1444, 0.15,   0.1632,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::J570_BlueN1_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1936, 0.194,  0.1942, 0.22,  0.2692, 0.29,  0.3102, 0.33,  0.3535, 0.37,  0.3885, 0.4,  0.4230, 0.44,  0.4594, 0.48,  0.5011, 0.52,  0.5510, 0.58,  0.6022, 0.62,  0.6479, 0.66,
    0.6748, 0.67,  0.6640, 0.65,  0.6268, 0.61,  0.5919, 0.58,  0.5549, 0.53,  0.4921, 0.46,  0.4209, 0.39,  0.3723, 0.35,  0.3364, 0.32,  0.3005, 0.28,  0.2569, 0.23,  0.2154, 0.19,
    0.1897, 0.18,  0.1768, 0.175,  0.1723, 0.173,  0.1722, 0.175,  0.1785, 0.18,  0.1892, 0.19,  0.2017, 0.21,  0.2134, 0.22,  0.2202, 0.222,  0.2247, 0.23,  0.2333, 0.24,  0.2566,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::J570_BlueN4_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.3138, 0.25,   0.2038, 0.19,   0.1787, 0.18,   0.1840, 0.2,    0.2109, 0.23,   0.2400, 0.26,   0.2715, 0.29,   0.3075, 0.33,   0.3534, 0.39,   0.4149, 0.45,   0.4863, 0.52,
    0.5520, 0.57,   0.5904, 0.58,   0.5750, 0.55,   0.5198, 0.48,   0.4602, 0.43,   0.3997, 0.36,   0.3199, 0.27,   0.2411, 0.2,    0.1883, 0.17,   0.1553, 0.14,   0.1307, 0.12,
    0.1062, 0.1,    0.0859, 0.08,   0.0746, 0.07,   0.0697, 0.069,  0.0678, 0.068,  0.0683, 0.07, 0.0715, 0.075,  0.0765, 0.08,   0.0818, 0.085,  0.0862, 0.087,  0.0885, 0.089,
    0.0899, 0.09,   0.0951, 0.1,    0.1090,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::J570_BlueO19_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1231, 0.13,  0.1348, 0.12,  0.1152, 0.12,  0.1368, 0.15,  0.1627, 0.17,  0.1860, 0.2,  0.2130, 0.23,  0.2462, 0.26,  0.2859, 0.31,  0.3425, 0.38,  0.4152, 0.45,  0.4902, 0.51,
    0.5386, 0.53,  0.5259, 0.49,  0.4652, 0.43,  0.3950, 0.35,  0.3266, 0.28,  0.2469, 0.21,  0.1750, 0.15,  0.1293, 0.12,  0.1033, 0.09,  0.0859, 0.08,  0.0703, 0.06,  0.0578, 0.055,
    0.0521, 0.05,  0.0494, 0.049,  0.0492, 0.049,  0.0498, 0.05,  0.0522, 0.053,  0.0558, 0.058,  0.0590, 0.06,  0.0615, 0.062,  0.0625, 0.063,  0.0634, 0.065,  0.0670, 0.07,  0.0776,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::J570_BlueU8_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0776, 0.1,  0.1703, 0.23,  0.2942, 0.35,  0.4022, 0.45,  0.4795, 0.51,  0.5389, 0.56,  0.5968, 0.6,  0.6362, 0.64,  0.6512, 0.652,  0.6523, 0.64,  0.6340, 0.62,  0.6052, 0.58,
    0.5645, 0.53,  0.4952, 0.45,  0.4124, 0.38,  0.3457, 0.32,  0.2883, 0.25,  0.2164, 0.19,  0.1529, 0.14,  0.1168, 0.1,  0.0983, 0.09,  0.0883, 0.08,  0.0798, 0.075,  0.0735, 0.073,
    0.0720, 0.072,  0.0726, 0.074,  0.0757, 0.08,  0.0830, 0.09,  0.0930, 0.095,  0.0989, 0.097,  0.0971, 0.095,  0.0911, 0.085,  0.0832, 0.08,  0.0795, 0.082,  0.0877, 0.09,  0.1140,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//N8
const double ColorTemp::J570_NeuN8_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0632, -0.04, -0.0244, 0.01,  0.0125, 0.02,   0.0294, 0.03,   0.0326, 0.033,  0.0352, 0.036,  0.0361, 0.037,  0.0374, 0.0374, 0.0373, 0.038,  0.0378, 0.039,  0.0397, 0.04,
    0.0421, 0.043,  0.0431, 0.042,  0.0417, 0.04,   0.0391, 0.038,  0.0378, 0.037,  0.0368, 0.035,  0.0347, 0.034,  0.0335, 0.034,  0.0341, 0.035,  0.0350, 0.035,  0.0355, 0.036,
    0.0357, 0.0357, 0.0358, 0.036,  0.0369, 0.037,  0.0372, 0.0376, 0.0378, 0.038,  0.0388, 0.039,  0.0397, 0.04,   0.0400, 0.0395, 0.0394, 0.039,  0.0386, 0.0385, 0.0384, 0.039,
    0.0395, 0.04,   0.0415, 0.043,  0.0448,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::J570_NeuN9_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0573, 0.054,  0.0516, 0.053,  0.0548, 0.05,   0.0401, 0.041,  0.0424, 0.043,  0.0449, 0.046,  0.0467, 0.047,  0.0473, 0.0473, 0.0474, 0.048,  0.0483, 0.05,   0.0508, 0.053,
    0.0558, 0.057,  0.0584, 0.058,  0.0550, 0.05,   0.0495, 0.048,  0.0468, 0.0465, 0.0460, 0.044,  0.0430, 0.042,  0.0411, 0.042,  0.0425, 0.044,  0.0457, 0.046,  0.0473, 0.0474,
    0.0475, 0.0475, 0.0474, 0.0475, 0.0476, 0.048,  0.0487, 0.049,  0.0499, 0.05,   0.0515, 0.052,  0.0533, 0.054,  0.0544, 0.054,  0.0539, 0.053,  0.0526, 0.052,  0.0512, 0.0515,
    0.0515, 0.053,  0.0538, 0.056,  0.0597,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::J570_NeuO8_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0014, 0.02,  0.0806, 0.07,   0.0673, 0.07,   0.0854, 0.09,   0.0901, 0.095,  0.0960, 0.098,  0.0992, 0.1,    0.1017, 0.102,  0.1030, 0.104,  0.1052, 0.107,  0.1098, 0.11,
    0.1176, 0.12,   0.1230, 0.12,   0.1176, 0.11,   0.1071, 0.105,  0.1032, 0.103,  0.1032, 0.1,    0.0963, 0.09,   0.0899, 0.09,   0.0939, 0.095,  0.1007, 0.102,  0.1037, 0.104,
    0.1029, 0.102,  0.1014, 0.102,  0.1020, 0.103,  0.1039, 0.105,  0.1072, 0.11,   0.1134, 0.12,   0.1207, 0.122,  0.1245, 0.123,  0.1236, 0.121,  0.1205, 0.12,   0.1158, 0.115,
    0.1132, 0.115,  0.1185, 0.12,   0.1345,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::J570_NeuO11_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2926, 0.2,    0.1863, 0.16,   0.1428, 0.14,   0.1322, 0.134,  0.1396, 0.14,   0.1450, 0.146,  0.1498, 0.15,   0.1527, 0.155,  0.1554, 0.157,  0.1583, 0.16,   0.1631, 0.17,
    0.1754, 0.18,   0.1841, 0.18,   0.1761, 0.17,   0.1600, 0.155,  0.1549, 0.155,  0.1555, 0.15,   0.1449, 0.14,   0.1352, 0.14,   0.1414, 0.15,   0.1519, 0.153,  0.1568, 0.156,
    0.1556, 0.154,  0.1534, 0.154,  0.1547, 0.156,  0.1573, 0.16,   0.1622, 0.17,   0.1713, 0.18,   0.1823, 0.185,  0.1886, 0.188,  0.1873, 0.183,  0.1829, 0.18,   0.1753, 0.174,
    0.1716, 0.175,  0.1800, 0.2,    0.2039,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::J570_NeuD5_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0840, 0.1,    0.1627, 0.18,   0.1934, 0.2,    0.2234, 0.23,   0.2430, 0.25,   0.2547, 0.26,   0.2618, 0.264,  0.2651, 0.265,  0.2655, 0.2655, 0.2659, 0.266,  0.2674, 0.27,
    0.2776, 0.28,   0.2841, 0.27,   0.2654, 0.25,   0.2351, 0.23,   0.2246, 0.225,  0.2247, 0.22,   0.2074, 0.20,   0.1913, 0.20,   0.2029, 0.21,   0.2231, 0.23,   0.2337, 0.233,
    0.2327, 0.23,   0.2291, 0.23,   0.2305, 0.232,  0.2344, 0.24,   0.2417, 0.25,   0.2553, 0.26,   0.2724, 0.28,   0.2816, 0.28,   0.2797, 0.276,  0.2720, 0.27,
    0.2603, 0.26,   0.2536, 0.26,   0.2660, 0.28,   0.3027,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_NeuE11_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1699, 0.18,  0.1971, 0.21,  0.2276, 0.23,  0.2483, 0.25,  0.2690, 0.28,  0.2820, 0.29,  0.2916, 0.295,  0.2992, 0.3,  0.3064, 0.31,  0.3151, 0.32,  0.3301, 0.34,  0.3593, 0.37,
    0.3873, 0.39,  0.3913, 0.38,  0.3793, 0.375,  0.3723, 0.37,  0.3678, 0.35,  0.3482, 0.33,  0.3249, 0.32,  0.3188, 0.319,  0.3188, 0.318,  0.3179, 0.315,  0.3128, 0.31,
    0.3086, 0.31,  0.3105, 0.312,  0.3148, 0.313,  0.3222, 0.33,  0.3364, 0.34,  0.3535, 0.36,  0.3629, 0.362,  0.3621, 0.36,  0.3549, 0.35,  0.3444, 0.34,  0.3394, 0.35,  0.3511, 0.36,  0.3862,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_NeuK16_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.5837, 0.45,  0.4117, 0.43,  0.4427, 0.47,  0.5098, 0.52,  0.5451, 0.55,  0.5698, 0.57,  0.5828, 0.59,  0.5939, 0.6,  0.6045, 0.607,  0.6140, 0.62,  0.6219, 0.63,  0.6330, 0.64,
    0.6419, 0.643,  0.6440, 0.642,  0.6417, 0.64,  0.6379, 0.631,  0.6309, 0.62,  0.6154, 0.6,  0.5911, 0.58,  0.5736, 0.57,  0.5612, 0.56,  0.5539, 0.55,  0.5462, 0.543,
    0.5406, 0.542,  0.5418, 0.543,  0.5452, 0.55,  0.5529, 0.56,  0.5654, 0.57,  0.5806, 0.584,  0.5888, 0.589,  0.5898, 0.586,  0.5858, 0.58,  0.5796, 0.577,  0.5770, 0.58,  0.5883, 0.59,  0.6190,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::J570_NeuM3_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2659, 0.255,  0.2526, 0.26,  0.2616, 0.27,  0.2854, 0.29,  0.3088, 0.31,  0.3231, 0.33,  0.3336, 0.34,  0.3421, 0.345,  0.347, 0.35,  0.3542, 0.36,  0.3647, 0.37,
    0.3854, 0.4,  0.4041, 0.402,  0.4012, 0.39,  0.3856, 0.38,  0.3769, 0.375,  0.3725, 0.36,  0.3525, 0.34,  0.3286, 0.325,  0.3247, 0.326,  0.3279, 0.328,  0.3285, 0.325,
    0.3240, 0.322,  0.3202, 0.321,  0.3220, 0.323,  0.3267, 0.33,  0.3342, 0.34,  0.3487, 0.35,  0.3667, 0.37,  0.3761, 0.375,  0.3746, 0.37,  0.3670, 0.36,  0.3559, 0.35,  0.3498, 0.35,  0.3630, 0.37,  0.3998,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::J570_NeuN18_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1284, 0.11,  0.1090, 0.13,  0.1573, 0.17,  0.1837, 0.19,  0.1971, 0.2,  0.2059, 0.21,  0.2143, 0.22,  0.2213, 0.225,  0.2271, 0.23,  0.2341, 0.24,  0.2487, 0.26,  0.2764, 0.29,
    0.3025, 0.303,  0.3052, 0.3,  0.2919, 0.29,  0.2843, 0.283,  0.2800, 0.27,  0.2612, 0.24,  0.2394, 0.235,  0.2339, 0.234,  0.2340, 0.233,  0.2326, 0.23,  0.2277, 0.225,
    0.2235, 0.224,  0.2246, 0.226,  0.2282, 0.23,  0.2349, 0.24,  0.2477, 0.25,  0.2632, 0.27,  0.2714, 0.271,  0.2702, 0.27,  0.2637, 0.26,  0.2538, 0.25,  0.2479, 0.25,  0.2589, 0.26,  0.2918,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::J570_NeuQ1_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0610, 0.06,  0.0592, 0.04,  0.0339, 0.04,  0.0338, 0.034,  0.0350, 0.036,  0.0363, 0.037,  0.0380, 0.038,  0.0383, 0.0383,  0.0385, 0.04,  0.0408, 0.042,  0.0451, 0.05,
    0.0524, 0.055,  0.0589, 0.058,  0.0585, 0.055,  0.0529, 0.05,  0.0456, 0.04,  0.0390, 0.035,  0.0330, 0.03,  0.0286, 0.028,  0.0275, 0.0276,  0.0275, 0.0278,  0.0279, 0.028,
    0.0289, 0.03,  0.0304, 0.031,  0.0320, 0.032,  0.0328, 0.033,  0.0341, 0.0345,  0.0346, 0.0346,  0.0347, 0.034,  0.0341, 0.034,  0.0336, 0.034,  0.0340, 0.035,
    0.0351, 0.036,  0.0373, 0.038,  0.0411, 0.042,  0.0446,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::J570_NeuS7_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1447, 0.06,  0.0448, 0.042,  0.0411, 0.03,  0.0282, 0.028,  0.0270, 0.029,  0.0298, 0.03,  0.0319, 0.032,  0.0331, 0.0333,  0.0335, 0.036,  0.0361, 0.038,  0.0403, 0.045,
    0.0493, 0.05,  0.0599, 0.06,  0.0636, 0.062,  0.0606, 0.06,  0.0547, 0.05,  0.0488, 0.045,  0.0421, 0.04,  0.0366, 0.035,  0.0335, 0.033,  0.0323, 0.032,  0.0320, 0.032,
    0.0321, 0.0322,  0.0324, 0.033,  0.0333, 0.034,  0.0345, 0.035,  0.0356, 0.036,  0.0364, 0.037,  0.0372, 0.037,  0.0367, 0.0365,  0.0363, 0.0364,
    0.0361, 0.0362,  0.0363, 0.037,  0.0376, 0.04,  0.0412, 0.042,  0.0450,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::J570_NeuV10_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1439, 0.07,  0.0583, 0.04,  0.0372, 0.037,  0.0362, 0.035,  0.0344, 0.034,  0.0340, 0.035,  0.0351, 0.036,  0.0361, 0.0361,  0.0361, 0.037,  0.0377, 0.039,  0.0404, 0.042,
    0.0450, 0.047,  0.0483, 0.048,  0.0475, 0.045,  0.0436, 0.04,  0.0387, 0.036,  0.0343, 0.03,  0.0299, 0.028,  0.0271, 0.027,  0.0262, 0.0262,  0.0262, 0.0267,  0.0269, 0.027,
    0.0283, 0.029,  0.0299, 0.03,  0.0308, 0.031,  0.0319, 0.032,  0.0331, 0.0333,  0.0337, 0.0337,  0.0337, 0.0333,  0.0332, 0.0332,  0.0331, 0.0331,  0.0331, 0.034,
    0.0341, 0.035,  0.0371, 0.038,  0.0399, 0.041,  0.0432,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::J570_NeuW18_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.4685, 0.45,  0.4262, 0.5,  0.5061, 0.55,  0.5898, 0.6,  0.6487, 0.66,  0.6781, 0.68,  0.6947, 0.7,  0.7070, 0.71,  0.7185, 0.72,  0.7294, 0.73,  0.7383, 0.74,  0.7499, 0.75,
    0.7582, 0.758,  0.7582, 0.755,  0.7531, 0.75,  0.7484, 0.745,  0.7422, 0.73,  0.7263, 0.72,  0.7033, 0.7,  0.6913, 0.69,  0.6820, 0.68,  0.6738, 0.67,  0.6628, 0.66,
    0.6512, 0.65,  0.6462, 0.645,  0.6448, 0.645,  0.6485, 0.65,  0.6569, 0.66,  0.6698, 0.67,  0.6781, 0.68,  0.6822, 0.682,  0.6820, 0.682,  0.6815, 0.682,  0.6820, 0.69,  0.6907, 0.7,  0.7152,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::J570_NeuZ14_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2765, 0.2,  0.1352, 0.13,  0.1222, 0.121,  0.1206, 0.13,  0.1300, 0.134,  0.1357, 0.14,  0.1407, 0.142,  0.1455, 0.147,  0.1485, 0.15,  0.1539, 0.16,  0.1648, 0.17,
    0.1844, 0.2,  0.2015, 0.202,  0.2024, 0.2,  0.1922, 0.19,  0.1868, 0.185,  0.1841, 0.18,  0.1715, 0.16,  0.1566, 0.155,  0.1536, 0.154,  0.1545, 0.154,  0.1536, 0.151,
    0.1500, 0.148,  0.1471, 0.1472,  0.1478, 0.15,  0.1505, 0.153,  0.1552, 0.16,  0.1641, 0.17,  0.1751, 0.18,  0.1813, 0.181,  0.1801, 0.18,
    0.1757, 0.17,  0.1683, 0.165,  0.1642, 0.17,  0.1728, 0.18,  0.1970,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_NeuC18_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0555, 0.055,  0.0545, 0.055,  0.0585, 0.058,  0.0577, 0.056,  0.0554, 0.056,  0.0564, 0.058,  0.0590, 0.06,   0.0611, 0.062,  0.0638, 0.065,  0.0685, 0.07,   0.0797, 0.09,
    0.1009, 0.11,   0.1222, 0.124,  0.1298, 0.127,  0.1257, 0.123,  0.1208, 0.12,   0.1164, 0.11,   0.1067, 0.1,    0.0954, 0.09,   0.0895, 0.088,  0.0862, 0.085,  0.0834, 0.082,
    0.0806, 0.08,   0.0782, 0.078,  0.0780, 0.078,  0.0789, 0.08,   0.0813, 0.084,  0.0858, 0.09,   0.0911, 0.092,  0.0944, 0.093,  0.0938, 0.092,  0.0914, 0.09,   0.0878, 0.086,
    0.0858, 0.09,   0.0903, 0.1,    0.1037,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::J570_NeuD17_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1980, 0.1,    0.0793, 0.06,   0.0578, 0.05,   0.0476, 0.046,  0.0454, 0.046,  0.0471, 0.048,  0.0499, 0.05,   0.0518, 0.052,  0.0533, 0.055,  0.0574, 0.06,   0.0676, 0.07,
    0.0897, 0.1,    0.1129, 0.113,  0.1140, 0.1,    0.0958, 0.08,   0.0743, 0.06,   0.0566, 0.05,   0.0422, 0.04, 0.0332, 0.03,   0.0297, 0.0295, 0.0292, 0.0293, 0.0294, 0.03,
    0.0306, 0.031,  0.0319, 0.032,  0.0339, 0.034,  0.0353, 0.036,  0.0363, 0.037,  0.0370, 0.037,  0.0372, 0.037,  0.0368, 0.0365, 0.0363, 0.036,
    0.0360, 0.037,  0.0376, 0.039,  0.0406, 0.042,  0.0448, 0.046,  0.0499,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::J570_NeuJ11_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1274, 0.1,    0.0916, 0.08,   0.0656, 0.061,  0.0604, 0.06,   0.0570, 0.06,   0.0604, 0.062,  0.0644, 0.065,  0.0668, 0.069,  0.0700, 0.072,  0.0754, 0.08,   0.0874, 0.1,
    0.1111, 0.12,   0.1327, 0.132,  0.1313, 0.12,   0.1127, 0.1,    0.0931, 0.08,   0.0758, 0.06,   0.0580, 0.05,   0.0449, 0.04,   0.0385, 0.037,  0.0360, 0.036,  0.0351, 0.035,
    0.0351, 0.0354, 0.0355, 0.036,  0.0371, 0.0375, 0.0379, 0.038,  0.0388, 0.04,   0.0406, 0.041,  0.0414, 0.0415, 0.0416, 0.041,  0.0409, 0.04,   0.0398, 0.0397, 0.0397, 0.04,
    0.0424, 0.043,  0.0458, 0.048,  0.0522,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::J570_NeuL4_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0348, 0.05,   0.0700, 0.09,   0.1043, 0.11,   0.1320, 0.14,   0.1505, 0.16,   0.1622, 0.17,   0.1721, 0.18,   0.1805, 0.185,  0.1877, 0.19,   0.1955, 0.2,    0.2068, 0.21,
    0.2226, 0.23,   0.2350, 0.235,  0.2352, 0.23,   0.2251, 0.22,   0.2128, 0.2,    0.1990, 0.18,   0.1761, 0.16,   0.1494, 0.13,   0.1296, 0.12,   0.1171, 0.11,   0.1089, 0.105,
    0.1010, 0.1,    0.0949, 0.093,  0.0926, 0.093,  0.0937, 0.095,  0.0961, 0.1,    0.1020, 0.11,   0.1104, 0.112,  0.1150, 0.115,  0.1155, 0.113,  0.1123, 0.11,
    0.1070, 0.105,  0.1040, 0.11,   0.1110, 0.12,   0.1323,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_n72_n2_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0116, 0.01,  0.0171, 0.05,   0.0625, 0.08,   0.1486, 0.16,   0.1963, 0.2,    0.2409, 0.26,   0.2974, 0.31,   0.3468, 0.36,   0.3790, 0.39,   0.4075, 0.41,   0.4216, 0.43,
    0.4399,  0.47,  0.4878, 0.50,   0.5589, 0.57,   0.5882, 0.57,   0.5566, 0.52,   0.5030, 0.46,   0.4451, 0.42,   0.3928, 0.37,   0.3625, 0.35,   0.3396, 0.29,   0.2670, 0.15,
    0.1028, 0.05,   -0.0397, -0.08, -0.1151, -0.12, -0.1464, -0.15, -0.1582, -0.16, -0.1609, -0.16, -0.1581, -0.155,    -0.1556, -0.156,
    -0.1582, -0.16, -0.1621, -0.165,    -0.1683, -0.17, -0.1719, -0.17, -0.1696, -0.165,    -0.1623,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Colorlab_10_n70_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0964, 0.1,    0.1534, 0.2,    0.2437, 0.33,   0.4663, 0.5,    0.6005, 0.65,   0.6958, 0.75,   0.8010, 0.83,   0.8598, 0.858,  0.8579, 0.85,   0.8432, 0.83,   0.8102, 0.79,
    0.7607, 0.7,    0.6760, 0.6,    0.5530, 0.5,    0.4212, 0.3,    0.2974, 0.2,    0.1839, 0.1,    0.0743, 0.03,   -0.0208, -0.05, -0.0747, -0.08, -0.0913, -0.05, -0.0458, 0.03,
    0.0806, 0.1,    0.1936, 0.2,    0.2556, 0.27,   0.2816, 0.29,   0.2925, 0.3,    0.3033, 0.31,   0.3175, 0.32,   0.3257, 0.325,  0.3246, 0.32,
    0.3187, 0.31,   0.3082, 0.305,  0.3014, 0.304,  0.3059, 0.31,   0.3253,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Colorlab_n33_n70_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0615, 0.1,    0.1219, 0.2,    0.2179, 0.3,    0.4397, 0.5,    0.5722, 0.6,    0.6714, 0.7,    0.7834, 0.8,    0.8535, 0.86,   0.8647, 0.864,  0.8642, 0.864,  0.8429, 0.82,
    0.8035, 0.75,   0.7316, 0.7,    0.6238, 0.55,   0.4996, 0.45,   0.3717, 0.3,    0.2487, 0.2,    0.1299, 0.06,   0.0272, 0.01,   -0.0315, -0.04, -0.0557, -0.051,
    -0.0519, -0.03, -0.0234, 0.001, 0.0041, 0.01,   0.0201, 0.021,  0.0269, 0.028,  0.0298, 0.03,   0.0371, 0.04,   0.0497, 0.05,   0.0578, 0.056,
    0.0557, 0.05,   0.0490, 0.04,   0.0374, 0.03,   0.0299, 0.03,   0.0348, 0.04,   0.0537,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Colorlab_n8_n74_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0603, 0.08,   0.1069, 0.15,   0.1810, 0.25,   0.3573, 0.4,    0.4634, 0.5,    0.5406, 0.6,    0.6266, 0.65,   0.6772, 0.679,  0.6800, 0.68,   0.6732, 0.66,   0.6511, 0.63,
    0.6136, 0.6,    0.5444, 0.5,    0.4415, 0.4,    0.3317, 0.3,    0.2286, 0.2,    0.1336, 0.1,    0.0425, 0.01,   -0.0360, -0.05, -0.0807, -0.09, -0.0967, -0.08,
    -0.0761, -0.05, -0.0119, 0.02,  0.0462, 0.05,   0.0784, 0.08,   0.0919, 0.095,  0.0974, 0.1,    0.1046, 0.11,   0.1152, 0.12,   0.1217, 0.121,
    0.1205, 0.12,   0.1155, 0.11,   0.1067, 0.104,  0.1010, 0.103,  0.1047, 0.11,   0.1199,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Colorlab_19_n69_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1281, 0.16,   0.1941, 0.24,   0.2982, 0.4,    0.5607, 0.65,   0.7192, 0.8,    0.8299, 0.9,    0.9517, 1.0,    1.0174, 1.014,  1.0115, 1.0,    0.9899, 0.96,   0.9475, 0.91,
    0.8877, 0.82,   0.7902, 0.7,    0.6508, 0.57,   0.5008, 0.45,   0.3595, 0.3,    0.2302, 0.15,   0.1049, 0.02,   -0.0045, -0.03, -0.0662, -0.07, -0.0832, -0.05, -0.0161, 0.05,
    0.1648, 0.2,    0.3257, 0.37,   0.4137, 0.43,   0.4506, 0.46,   0.4661, 0.47,   0.4801, 0.49,   0.4974, 0.5,    0.5072, 0.505,  0.5062, 0.5,    0.4995, 0.49,   0.4876, 0.48,
    0.4799, 0.48,   0.4849, 0.49,   0.5081,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Colorlab_n80_10_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0247, -0.01, -0.0057, 0.01,  0.0242, 0.05,   0.0736, 0.08,   0.0996, 0.1,    0.1281, 0.14,   0.1660, 0.19,   0.2037, 0.21,   0.2337, 0.25,   0.2618, 0.27,   0.2793, 0.29,
    0.3024, 0.32,   0.3564, 0.4,    0.4360, 0.44,   0.4788, 0.47,   0.4655, 0.45,   0.4308, 0.42,   0.3920, 0.37,   0.3564, 0.34,   0.3357, 0.32,   0.3167, 0.28,   0.2434, 0.1,
    0.0745, 0.01,   -0.0725, -0.09, -0.1506, -0.18, -0.1831, -0.19, -0.1955, -0.195,    -0.1995, -0.199,    -0.1989, -0.198,
    -0.1976, -0.198,    -0.1999, -0.2,  -0.2027, -0.208,    -0.2070, -0.208,    -0.2094, -0.208,    -0.2080, -0.207,    -0.2038,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Colorlab_n80_26_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0325, -0.03, -0.0203, -0.01, -0.0012, 0.01,  0.0228, 0.03,   0.0332, 0.04,   0.0506, 0.06,   0.0762, 0.09,   0.1069, 0.12,   0.1370, 0.15,   0.1666, 0.17,   0.1877, 0.2,
    0.2174, 0.25,   0.2849, 0.33,   0.3849, 0.4,    0.4477, 0.45,   0.4509, 0.44,   0.4305, 0.42,   0.4049, 0.39,   0.3806, 0.37,   0.3664, 0.35,   0.3495, 0.29,   0.2724, 0.1,
    0.0921, 0.01,   -0.0651, -0.09, -0.1489, -0.15, -0.1837, -0.19, -0.1970, -0.2,  -0.2020, -0.202,    -0.2029, -0.203,    -0.2026, -0.204,
    -0.2047, -0.205,    -0.2069, -0.208,    -0.2099, -0.21, -0.2115, -0.21, -0.2106, -0.209,    -0.2086,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
/*
//0,1767000	0,2207000	0,3142000	0,5269000	0,7018000	0,7605000	0,7580000	0,7366000	0,7182000	0,6929000	0,6661000	0,6542000	
//0,6420000	0,6085000	0,5752000	0,5728000	0,5723000	0,5318000	0,4982000	0,5226000	0,5670000	0,5929000	0,5977000	0,5975000	
//0,6002000	0,6065000	0,6177000	0,6352000	0,6526000	0,6623000	0,6633000	0,6593000	0,6517000	0,6479000	0,6607000	0,6908000																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																							

const double ColorTemp::JDC468_greyc14_66_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1767, 0.19, 0.2207, 0.25, 0.3142, 0.40, 0.5269, 0.63, 0.7018, 0.73, 0.7605, 0.76, 0.7580, 0.74, 0.7366, 0.725, 0.7182, 0.705, 0.6929, 0.68, 0.6661, 0.66, 0.6542, 0.65,
    0.642, 0.62, 0.6085, 0.585, 0.5752, 0.573, 0.5728, 0.573, 0.5723, 0.555, 0.5318, 0.51, 0.4982, 0.51, 0.5226, 0.54, 0.5670, 0.58, 0.5929, 0.594, 0.5977, 0.597, 0.5975, 0.6,
    0.6002, 0.602, 0.6065, 0.61, 0.6177, 0.62, 0.6352, 0.64, 0.6526, 0.66, 0.6623, 0.662, 0.6633, 0.66, 0.6593, 0.653, 0.6517, 0.65, 0.6479, 0.65, 0.6607, 0.69, 0.6908,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
//0,1325000	0,1637000	0,2222000	0,3492000	0,4523000	0,4897000	0,4918000	0,4840000	0,4761000	0,4638000	0,4538000	0,4582000
//	0,4588000	0,4360000	0,4091000	0,4101000	0,4128000	0,3785000	0,3494000	0,3720000	0,4122000	0,4339000	0,4362000	0,4355000
//	0,4395000	0,4475000	0,4606000	0,4807000	0,5006000	0,5125000	0,5145000	0,5112000	0,5029000	0,4992000	0,5150000	0,5526000																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																							

const double ColorTemp::JDC468_greym13_325_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1325, 0.15, 0.1637, 0.2, 0.2222, 0.28, 0.3492, 0.40, 0.4523, 0.47, 0.4897, 0.49, 0.4918, 0.49, 0.4840, 0.48, 0.4761, 0.47, 0.4638, 0.46, 0.4538, 0.455, 0.4582, 0.458,
    0.4588, 0.45, 0.4360, 0.42, 0.4091, 0.41, 0.4101, 0.411, 0.4128, 0.405, 0.3785, 0.36, 0.3494, 0.36, 0.3720, 0.41, 0.4122, 0.425, 0.4339, 0.435, 0.4362, 0.597, 0.4355, 0.437,
    0.4395, 0.44, 0.4475, 0.45, 0.4606, 0.47, 0.4807, 0.49, 0.5006, 0.51, 0.5125, 0.513, 0.5145, 0.512, 0.5112, 0.51, 0.5029, 0.5, 0.4992, 0.51, 0.5150, 0.53, 0.5526,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//0,0823000	0,1036000	0,1337000	0,1966000	0,2468000	0,2679000	0,2728000	0,2726000	0,2724000	0,2698000	0,2705000	0,2810000
//	0,2879000	0,2756000	0,2586000	0,2601000	0,2617000	0,2357000	0,2124000	0,2241000	0,2471000	0,2581000	0,2569000	0,2548000
//	0,2579000	0,2653000	0,2765000	0,2941000	0,3126000	0,3230000	0,3238000	0,3189000	0,3091000	0,3043000	0,3200000	0,3579000																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																																							

const double ColorTemp::JDC468_greyf26_156_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0823, 0.1, 0.1036, 0.11, 0.1337, 0.16, 0.1966, 0.22, 0.2468, 0.255, 0.2679, 0.27, 0.2728, 0.273, 0.2726, 0.273, 0.2724, 0.271, 0.2698, 0.27, 0.2705, 0.275, 0.2810, 0.285,
    0.2879, 0.28, 0.2756, 0.26, 0.2586, 0.26, 0.2601, 0.261, 0.2617, 0.25, 0.2357, 0.22, 0.2124, 0.22, 0.2241, 0.23, 0.2471, 0.25, 0.2581, 0.278, 0.2569, 0.255, 0.2548, 0.255,
    0.2579, 0.26, 0.2653, 0.27, 0.2765, 0.28, 0.2941, 0.30, 0.3126, 0.32, 0.3230, 0.323, 0.3238, 0.32, 0.3189, 0.31, 0.3091, 0.302, 0.3043, 0.31, 0.3200, 0.34, 0.3579,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
*/
//A1	0.0912	0.1228	0.1712	0.2978	0.3713	0.4241	0.4861	0.5255	0.5355	0.5363	0.5237	0.5251	
// 0.5722	0.6554	0.6936	0.6675	0.6203	0.5651	0.5116	0.4825	0.4714	0.4866	0.5320	0.5729	
// 0.5968	0.6069	0.6131	0.6198	0.6285	0.6325	0.6316	0.6282	0.6227	0.6196	0.6215	0.6337

const double ColorTemp::Colorlab_n80_5_9_5_9spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0912, 0.1, 0.1228, 0.15, 0.1712, 0.2, 0.2978, 0.32, 0.3713, 0.41, 0.4241, 0.44, 0.4861, 0.51, 0.5255, 0.53, 0.5355, 0.534, 0.5363, 0.53, 0.5237, 0.524, 0.5251, 0.56,
    0.5722, 0.6, 0.6554, 0.67, 0.6936, 0.67, 0.6675, 0.65, 0.6203, 0.6, 0.5651, 0.54, 0.5116, 0.5, 0.4825, 0.48, 0.4714, 0.48, 0.4866, 0.5, 0.5320, 0.55, 0.5729, 0.58,
    0.5968, 0.6, 0.6069, 0.61, 0.6131, 0.615, 0.6198, 0.62, 0.6285, 0.63, 0.6325, 0.632, 0.6316, 0.63, 0.6282, 0.625, 0.6227, 0.62, 0.6196, 0.62, 0.6215, 0.63, 0.6337,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
/*
//A2	0.0385	0.0514	0.0711	0.1229	0.1528	0.1744	0.1999	0.2163	0.2209	0.2216	0.2167	0.2185	
//0.2414	0.2813	0.3012	0.2922	0.2734	0.2511	0.2292	0.2173	0.2127	0.2183	0.2354	0.2508	
//0.2599	0.2637	0.2662	0.2689	0.2725	0.2742	0.2738	0.2724	0.2701	0.2689	0.2697	0.2747

const double ColorTemp::Colorlab_n57_5_6_9spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0385, 0.04, 0.0514, 0.06, 0.0711, 0.1, 0.1229, 0.14, 0.1528, 0.16, 0.1744, 0.18, 0.1999, 0.2, 0.2163, 0.22, 0.2209, 0.221, 0.2216, 0.22, 0.2167, 0.216, 0.2185, 0.23,
    0.2414, 0.26, 0.2813, 0.3, 0.3012, 0.3, 0.2922, 0.28, 0.2734, 0.26, 0.2511, 0.24, 0.2292, 0.22, 0.2173, 0.215, 0.2127, 0.215, 0.2183, 0.22, 0.2354, 0.24, 0.2508, 0.255,
    0.2599, 0.26, 0.2637, 0.263, 0.2662, 0.267, 0.2689, 0.27, 0.2725, 0.273, 0.2742, 0.274, 0.2738, 0.273, 0.2724, 0.271, 0.2701, 0.27, 0.2689, 0.269, 0.2697, 0.27, 0.2747,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
*/
/*
 *      Name:   XYZtoCorColorTemp.c
 *
 *      Author: Bruce Justin Lindbloom
 *
 *      Copyright (c) 2003 Bruce Justin Lindbloom. All rights reserved.
 *
 *
 *      Description:
 *              This is an implementation of Robertson's method of computing the correlated color
 *              temperature of an XYZ color. It can compute correlated color temperatures in the
 *              range [1666.7K, infinity].
 *
 *      Reference:
 *              "Color Science: Concepts and Methods, Quantitative Data and Formulae", Second Edition,
 *              Gunter Wyszecki and W. S. Stiles, John Wiley & Sons, 1982, pp. 227, 228.
 */
//adaptation to RT by J.Desmis

int ColorTemp::XYZtoCorColorTemp(double x0, double y0, double z0, double &temp) const
{

    typedef struct UVT {
        double  u;
        double  v;
        double  t;
    } UVT;

    double rt[31] = {       /* reciprocal temperature (K) */
        DBL_MIN,  10.0e-6,  20.0e-6,  30.0e-6,  40.0e-6,  50.0e-6,
        60.0e-6,  70.0e-6,  80.0e-6,  90.0e-6, 100.0e-6, 125.0e-6,
        150.0e-6, 175.0e-6, 200.0e-6, 225.0e-6, 250.0e-6, 275.0e-6,
        300.0e-6, 325.0e-6, 350.0e-6, 375.0e-6, 400.0e-6, 425.0e-6,
        450.0e-6, 475.0e-6, 500.0e-6, 525.0e-6, 550.0e-6, 575.0e-6,
        600.0e-6
    };

    UVT uvt[31] = {
        {0.18006, 0.26352, -0.24341},
        {0.18066, 0.26589, -0.25479},
        {0.18133, 0.26846, -0.26876},
        {0.18208, 0.27119, -0.28539},
        {0.18293, 0.27407, -0.30470},
        {0.18388, 0.27709, -0.32675},
        {0.18494, 0.28021, -0.35156},
        {0.18611, 0.28342, -0.37915},
        {0.18740, 0.28668, -0.40955},
        {0.18880, 0.28997, -0.44278},
        {0.19032, 0.29326, -0.47888},
        {0.19462, 0.30141, -0.58204},
        {0.19962, 0.30921, -0.70471},
        {0.20525, 0.31647, -0.84901},
        {0.21142, 0.32312, -1.0182},
        {0.21807, 0.32909, -1.2168},
        {0.22511, 0.33439, -1.4512},
        {0.23247, 0.33904, -1.7298},
        {0.24010, 0.34308, -2.0637},
        {0.24792, 0.34655, -2.4681},
        {0.25591, 0.34951, -2.9641},
        {0.26400, 0.35200, -3.5814},
        {0.27218, 0.35407, -4.3633},
        {0.28039, 0.35577, -5.3762},
        {0.28863, 0.35714, -6.7262},
        {0.29685, 0.35823, -8.5955},
        {0.30505, 0.35907, -11.324},
        {0.31320, 0.35968, -15.628},
        {0.32129, 0.36011, -23.325},
        {0.32931, 0.36038, -40.770},
        {0.33724, 0.36051, -116.45}
    };

    double us, vs, p, di, dm;
    int i;

    if ((x0 < 1.0e-20) && (y0 < 1.0e-20) && (z0 < 1.0e-20)) {
        return -1;    /* protect against possible divide-by-zero failure */
    }

    us = (4.0 * x0) / (x0 + 15.0 * y0 + 3.0 * z0);
    vs = (6.0 * y0) / (x0 + 15.0 * y0 + 3.0 * z0);
    dm = 0.0;

    for (i = 0; i < 31; i++) {
        di = (vs - uvt[i].v) - uvt[i].t * (us - uvt[i].u);

        if ((i > 0) && (((di < 0.0) && (dm >= 0.0)) || ((di >= 0.0) && (dm < 0.0)))) {
            break;    /* found lines bounding (us, vs) : i-1 and i */
        }

        dm = di;
    }

    if (i == 31) {
        return -1;    /* bad XYZ input, color temp would be less than minimum of 1666.7 degrees, or too far towards blue */
    }

    di = di / sqrt(1.0 + uvt[i    ].t * uvt[i    ].t);
    dm = dm / sqrt(1.0 + uvt[i - 1].t * uvt[i - 1].t);
    p = dm / (dm - di);     /* p = interpolation parameter, 0.0 : i-1, 1.0 : i */
    p = 1.0 / rtengine::intp(p, rt[i], rt[i - 1]);
    temp = p;
    return 0;      /* success */
}

void ColorTemp::cieCAT02(double Xw, double Yw, double Zw, double &CAM02BB00, double &CAM02BB01, double &CAM02BB02, double &CAM02BB10, double &CAM02BB11, double &CAM02BB12, double &CAM02BB20, double &CAM02BB21, double &CAM02BB22, double adap )
{

// CIECAT02  - J.Desmis January 2012 review September 2012
    const double whiteD50p[3] = {0.9646019585, 1.0, 0.8244507152}; //calculate with these tools

    double cam_dest[3] = {0., 0., 0.};
    double cam_orig[3] = {0., 0., 0.};
    const double CAT02[3][3] = {{0.7328,   0.4296,  -0.1624},//CAT02 2002
        { -0.7036,  1.6975,   0.0061},
        {0.0030,   0.0136,   0.9834}
    };
    const double INVCAT02[3][3] = {{1.09612382083551,      -0.278869000218287,    0.182745179382773},  //Inverse CAT02
        {0.454369041975359,      0.4735331543070412,   0.0720978037172291},
        { -0.009627608738442936, -0.00569803121611342,  1.01532563995454}
    };

    double inv_white_orig[3][3] = {{0.,  0.,  0.},
        {0.,  0.,  0.},
        {0.,  0.,  0.}
    };
    double intermed[3][3] = {{0.,  0.,  0.},
        {0.,  0.,  0.},
        {0.,  0.,  0.}
    };

    double intermed_2[3][3] = {{0.,  0.,  0.},
        {0.,  0.,  0.},
        {0.,  0.,  0.}
    };
    double CAM02[3][3] = {{0.,  0.,  0.},
        {0.,  0.,  0.},
        {0.,  0.,  0.}
    };
    double D = adap;

    //white destination Wd : RT use always D50
    cam_dest[0] = CAT02[0][0] * whiteD50p[0] + CAT02[0][1] * whiteD50p[1] + CAT02[0][2] * whiteD50p[2]; //Cone response RoD
    cam_dest[1] = CAT02[1][0] * whiteD50p[0] + CAT02[1][1] * whiteD50p[1] + CAT02[1][2] * whiteD50p[2]; //GaD
    cam_dest[2] = CAT02[2][0] * whiteD50p[0] + CAT02[2][1] * whiteD50p[1] + CAT02[2][2] * whiteD50p[2]; //BeD

    //origin white Ws : A, D65, custom, etc.
    cam_orig[0] = CAT02[0][0] * Xw + CAT02[0][1] * Yw + CAT02[0][2] * Zw; //Cone response RoS
    cam_orig[1] = CAT02[1][0] * Xw + CAT02[1][1] * Yw + CAT02[1][2] * Zw;
    cam_orig[2] = CAT02[2][0] * Xw + CAT02[2][1] * Yw + CAT02[2][2] * Zw;

    //inverse white
    inv_white_orig[0][0] = 1. / cam_orig[0]; // 1/RoS
    inv_white_orig[1][1] = 1. / cam_orig[1]; // 1/GaS
    inv_white_orig[2][2] = 1. / cam_orig[2]; // 1/BeS

    //intermediates computation
    for(int i = 0; i < 3; i++)
        for(int j = 0; j < 3 ; j++) {
            intermed[i][j] = inv_white_orig[i][i] * CAT02[i][j];
        }

    for(int i = 0; i < 3; i++)
        for(int j = 0; j < 3 ; j++) {
            intermed_2[i][j] = cam_dest[i] * intermed[i][j];
        }

    //and CAM02
    for(int i = 0; i < 3; i++)
        for(int j = 0; j < 3; j++)
            for(int k = 0; k < 3; k++) {
                CAM02[i][j] += INVCAT02[i][k] * intermed_2[k][j];
            }

    //adaptation jdc : slightly different from CIECAM02 : Rc=(Yw(D/Rw)+(1-D))*R , but it's work !   true at 0 and 1
    CAM02[1][1] = (CAM02[1][1] - 1.0) * D + 1.0;
    CAM02[0][0] = (CAM02[0][0] - 1.0) * D + 1.0;
    CAM02[2][2] = (CAM02[2][2] - 1.0) * D + 1.0;
    CAM02[0][1] *= D;
    CAM02[0][2] *= D;
    CAM02[1][0] *= D;
    CAM02[1][2] *= D;
    CAM02[2][0] *= D;
    CAM02[2][1] *= D;
    //CAT02  matrix with D adaptation
    CAM02BB00 = CAM02[0][0];
    CAM02BB01 = CAM02[0][1];
    CAM02BB02 = CAM02[0][2];
    CAM02BB10 = CAM02[1][0];
    CAM02BB11 = CAM02[1][1];
    CAM02BB12 = CAM02[1][2];
    CAM02BB20 = CAM02[2][0];
    CAM02BB21 = CAM02[2][1];
    CAM02BB22 = CAM02[2][2];

}

void ColorTemp::cieCAT02float(float Xw, float Yw, float Zw, float &CAM02BB00, float &CAM02BB01, float &CAM02BB02, float &CAM02BB10, float &CAM02BB11, float &CAM02BB12, float &CAM02BB20, float &CAM02BB21, float &CAM02BB22, float adap)
{

// CIECAT02  - J.Desmis January 2012 review September 2012
    const float whiteD50p[3] = {0.9646019585, 1.0, 0.8244507152}; //calculate with these tools

    float cam_dest[3] = {0., 0., 0.};
    float cam_orig[3] = {0., 0., 0.};
    const float CAT02[3][3] = {{0.7328,   0.4296,  -0.1624},//CAT02 2002
        { -0.7036,  1.6975,   0.0061},
        {0.0030,   0.0136,   0.9834}
    };
    const float INVCAT02[3][3] = {{1.09612382083551,      -0.278869000218287,    0.182745179382773},  //Inverse CAT02
        {0.454369041975359,      0.4735331543070412,   0.0720978037172291},
        { -0.009627608738442936, -0.00569803121611342,  1.01532563995454}
    };

    float inv_white_orig[3][3] = {{0.,  0.,  0.},
        {0.,  0.,  0.},
        {0.,  0.,  0.}
    };
    float intermed[3][3] = {{0.,  0.,  0.},
        {0.,  0.,  0.},
        {0.,  0.,  0.}
    };

    float intermed_2[3][3] = {{0.,  0.,  0.},
        {0.,  0.,  0.},
        {0.,  0.,  0.}
    };
    float CAM02[3][3] = {{0.,  0.,  0.},
        {0.,  0.,  0.},
        {0.,  0.,  0.}
    };
    float D = adap;

    //white destination Wd : RT use always D50
    cam_dest[0] = CAT02[0][0] * whiteD50p[0] + CAT02[0][1] * whiteD50p[1] + CAT02[0][2] * whiteD50p[2]; //Cone response RoD
    cam_dest[1] = CAT02[1][0] * whiteD50p[0] + CAT02[1][1] * whiteD50p[1] + CAT02[1][2] * whiteD50p[2]; //GaD
    cam_dest[2] = CAT02[2][0] * whiteD50p[0] + CAT02[2][1] * whiteD50p[1] + CAT02[2][2] * whiteD50p[2]; //BeD

    //origin white Ws : A, D65, custom, etc.
    cam_orig[0] = CAT02[0][0] * Xw + CAT02[0][1] * Yw + CAT02[0][2] * Zw; //Cone response RoS
    cam_orig[1] = CAT02[1][0] * Xw + CAT02[1][1] * Yw + CAT02[1][2] * Zw;
    cam_orig[2] = CAT02[2][0] * Xw + CAT02[2][1] * Yw + CAT02[2][2] * Zw;

    //inverse white
    inv_white_orig[0][0] = 1. / cam_orig[0]; // 1/RoS
    inv_white_orig[1][1] = 1. / cam_orig[1]; // 1/GaS
    inv_white_orig[2][2] = 1. / cam_orig[2]; // 1/BeS

    //intermediates computation
    for (int i = 0; i < 3; i++)
        for (int j = 0; j < 3 ; j++) {
            intermed[i][j] = inv_white_orig[i][i] * CAT02[i][j];
        }

    for (int i = 0; i < 3; i++)
        for (int j = 0; j < 3 ; j++) {
            intermed_2[i][j] = cam_dest[i] * intermed[i][j];
        }

    //and CAM02
    for (int i = 0; i < 3; i++)
        for (int j = 0; j < 3; j++)
            for (int k = 0; k < 3; k++) {
                CAM02[i][j] += INVCAT02[i][k] * intermed_2[k][j];
            }

    //adaptation jdc : slightly different from CIECAM02 : Rc=(Yw(D/Rw)+(1-D))*R , but it's work !   true at 0 and 1
    CAM02[1][1] = (CAM02[1][1] - 1.0) * D + 1.0;
    CAM02[0][0] = (CAM02[0][0] - 1.0) * D + 1.0;
    CAM02[2][2] = (CAM02[2][2] - 1.0) * D + 1.0;
//   CAM02[1][1] *= D;
//   CAM02[0][0] *= D;
//   CAM02[2][2] *= D;
    CAM02[0][1] *= D;
    CAM02[0][2] *= D;
    CAM02[1][0] *= D;
    CAM02[1][2] *= D;
    CAM02[2][0] *= D;
    CAM02[2][1] *= D;
    //CAT02  matrix with D adaptation
    CAM02BB00 = CAM02[0][0];
    CAM02BB01 = CAM02[0][1];
    CAM02BB02 = CAM02[0][2];
    CAM02BB10 = CAM02[1][0];
    CAM02BB11 = CAM02[1][1];
    CAM02BB12 = CAM02[1][2];
    CAM02BB20 = CAM02[2][0];
    CAM02BB21 = CAM02[2][1];
    CAM02BB22 = CAM02[2][2];

}

void ColorTemp::icieCAT02float(float Xw, float Yw, float Zw, float &iCAM02BB00, float &iCAM02BB01, float &iCAM02BB02, float &iCAM02BB10, float &iCAM02BB11, float &iCAM02BB12, float &iCAM02BB20, float &iCAM02BB21, float &iCAM02BB22, float adap)
{

// CIECAT02  - J.Desmis January 2012 review September 2017
    const float whiteD50p[3] = {0.9646019585, 1.0, 0.8244507152}; //calculate with these tools

    float cam_dest[3] = {0., 0., 0.};
    float cam_orig[3] = {0., 0., 0.};
    const float CAT02[3][3] = {{0.7328,   0.4296,  -0.1624},//CAT02 2002
        { -0.7036,  1.6975,   0.0061},
        {0.0030,   0.0136,   0.9834}
    };
    const float INVCAT02[3][3] = {{1.09612382083551,      -0.278869000218287,    0.182745179382773},  //Inverse CAT02
        {0.454369041975359,      0.4735331543070412,   0.0720978037172291},
        { -0.009627608738442936, -0.00569803121611342,  1.01532563995454}
    };

    float inv_white_orig[3][3] = {{0.,  0.,  0.},
        {0.,  0.,  0.},
        {0.,  0.,  0.}
    };
    float intermed[3][3] = {{0.,  0.,  0.},
        {0.,  0.,  0.},
        {0.,  0.,  0.}
    };

    float intermed_2[3][3] = {{0.,  0.,  0.},
        {0.,  0.,  0.},
        {0.,  0.,  0.}
    };
    float INVCAM02[3][3] = {{0.,  0.,  0.},
        {0.,  0.,  0.},
        {0.,  0.,  0.}
    };
    float D = adap / 2.;

    //white destination Wd : RT use always D50
    cam_dest[0] = INVCAT02[0][0] * whiteD50p[0] + INVCAT02[0][1] * whiteD50p[1] + INVCAT02[0][2] * whiteD50p[2]; //Cone response RoD
    cam_dest[1] = INVCAT02[1][0] * whiteD50p[0] + INVCAT02[1][1] * whiteD50p[1] + INVCAT02[1][2] * whiteD50p[2]; //GaD
    cam_dest[2] = INVCAT02[2][0] * whiteD50p[0] + INVCAT02[2][1] * whiteD50p[1] + INVCAT02[2][2] * whiteD50p[2]; //BeD

    //origin white Ws : A, D65, custom, etc.
    cam_orig[0] = INVCAT02[0][0] * Xw + INVCAT02[0][1] * Yw + INVCAT02[0][2] * Zw; //Cone response RoS
    cam_orig[1] = INVCAT02[1][0] * Xw + INVCAT02[1][1] * Yw + INVCAT02[1][2] * Zw;
    cam_orig[2] = INVCAT02[2][0] * Xw + INVCAT02[2][1] * Yw + INVCAT02[2][2] * Zw;
//   cam_orig[0] = CAT02[0][0] * Xw + CAT02[0][1] * Yw + CAT02[0][2] * Zw; //Cone response RoS
//   cam_orig[1] = CAT02[1][0] * Xw + CAT02[1][1] * Yw + CAT02[1][2] * Zw;
    //  cam_orig[2] = CAT02[2][0] * Xw + CAT02[2][1] * Yw + CAT02[2][2] * Zw;

    //inverse white
    inv_white_orig[0][0] = 1. / cam_orig[0]; // 1/RoS
    inv_white_orig[1][1] = 1. / cam_orig[1]; // 1/GaS
    inv_white_orig[2][2] = 1. / cam_orig[2]; // 1/BeS

    //intermediates computation
    for (int i = 0; i < 3; i++)
        for (int j = 0; j < 3 ; j++) {
            // intermed[i][j] = inv_white_orig[i][i] * INVCAT02[i][j];
            intermed[i][j] = inv_white_orig[i][i] * CAT02[i][j];
        }

    for (int i = 0; i < 3; i++)
        for (int j = 0; j < 3 ; j++) {
            intermed_2[i][j] = cam_dest[i] * intermed[i][j];
        }

    //and CAM02
    for (int i = 0; i < 3; i++)
        for (int j = 0; j < 3; j++)
            for (int k = 0; k < 3; k++) {
                INVCAM02[i][j] += INVCAT02[i][k] * intermed_2[k][j];
            }

    //adaptation jdc : slightly different from CIECAM02 : Rc=(Yw(D/Rw)+(1-D))*R , but it's work !   true at 0 and 1
    INVCAM02[0][0] = (INVCAM02[0][0] - 1.0) * D + 1.0;
    INVCAM02[2][2] = (INVCAM02[2][2] - 1.0) * D + 1.0;
    INVCAM02[0][1] *= D;
    INVCAM02[0][2] *= D;
    INVCAM02[1][0] *= D;
    INVCAM02[1][2] *= D;
    INVCAM02[2][0] *= D;
    INVCAM02[2][1] *= D;
    //CAT02  matrix with D adaptation
    iCAM02BB00 = INVCAM02[0][0];
    iCAM02BB01 = INVCAM02[0][1];
    iCAM02BB02 = INVCAM02[0][2];
    iCAM02BB10 = INVCAM02[1][0];
    iCAM02BB11 = INVCAM02[1][1];
    iCAM02BB12 = INVCAM02[1][2];
    iCAM02BB20 = INVCAM02[2][0];
    iCAM02BB21 = INVCAM02[2][1];
    iCAM02BB22 = INVCAM02[2][2];

}


void ColorTemp::temp2mulxyz (double temp, const std::string &method, double &Xxyz, double &Zxyz)
{
    double x, y, z;

    // We first test for specially handled methods
    const auto iterator = spectMap.find(method);

    if (iterator != spectMap.end()) {
        spectrum_to_xyz_preset(iterator->second, x, y, z);
    } else {
        // otherwise we use the Temp+Green generic solution
        if (temp <= INITIALBLACKBODY) {
            // if temperature is between 2000K and 4000K we use blackbody, because there will be no Daylight reference below 4000K...
            // of course, the previous version of RT used the "magical" but wrong formula of U.Fuchs (Ufraw).
            spectrum_to_xyz_blackbody(temp, x, y, z);
        } else {
            // from 4000K up to 25000K: using the D illuminant (daylight) which is standard
            double x_D, y_D;

            if (temp <= 7000) {
                x_D = -4.6070e9 / (temp * temp * temp) + 2.9678e6 / (temp * temp) + 0.09911e3 / temp + 0.244063;
            } else if (temp <= 25000) {
                x_D = -2.0064e9 / (temp * temp * temp) + 1.9018e6 / (temp * temp) + 0.24748e3 / temp + 0.237040;
            } else /*if (temp > 25000)*/ {
                x_D = -2.0064e9 / (temp * temp * temp) + 1.9018e6 / (temp * temp) + 0.24748e3 / temp + 0.237040 - ((temp - 25000) / 25000) * 0.025;    //Jacques empirical adjustment for very high temp (underwater !)
            }

            y_D = -3.0 * x_D * x_D + 2.87 * x_D - 0.275; //modify blue / red action
            //calculate D -daylight in function of s0, s1, s2 and temp ==> x_D y_D
            //S(lamda)=So(lambda)+m1*s1(lambda)+m2*s2(lambda)
            double interm = 0.0241 + 0.2562 * x_D - 0.734 * y_D;
            double m1 = (-1.3515 - 1.7703 * x_D + 5.9114 * y_D) / interm;
            double m2 = (0.03 - 31.4424 * x_D + 30.0717 * y_D) / interm;
            spectrum_to_xyz_daylight(m1, m2, x, y, z);
        }
    }

    Xxyz = x / y;
    Zxyz = (1.0 - x - y) / y;
}

void ColorTemp::temp2mul (double temp, double green, double equal, double& rmul, double& gmul, double& bmul) const
{
    clip(temp, green, equal);
    double Xwb, Zwb;
    temp2mulxyz(temp, method, Xwb, Zwb);

    double adj = 1.0;

    if(equal < 0.9999 || equal > 1.0001 ) {
        adj = (100.0 + ( 1000.0 - (1000.0 * equal) ) / 20.0) / 100.0;
    }


    /*if (isRaw) {
        rmul = sRGB_xyz[0][0]*X + sRGB_xyz[0][1]*Y + sRGB_xyz[0][2]*Z;
        gmul = sRGB_xyz[1][0]*X + sRGB_xyz[1][1]*Y + sRGB_xyz[1][2]*Z;
        bmul = sRGB_xyz[2][0]*X + sRGB_xyz[2][1]*Y + sRGB_xyz[2][2]*Z;
    } else {*/
    //recalculate channels multipliers with new values of XYZ tue to whitebalance
    rmul = sRGBd65_xyz[0][0] * Xwb * adj + sRGBd65_xyz[0][1] + sRGBd65_xyz[0][2] * Zwb / adj; // Jacques' empirical modification 5/2013
    gmul = sRGBd65_xyz[1][0] * Xwb       + sRGBd65_xyz[1][1] + sRGBd65_xyz[1][2] * Zwb;
    bmul = sRGBd65_xyz[2][0] * Xwb * adj + sRGBd65_xyz[2][1] + sRGBd65_xyz[2][2] * Zwb / adj;
    //};
    gmul /= green;
    //printf("rmul=%f gmul=%f bmul=%f\n",rmul, gmul, bmul);
    double maxRGB = rtengine::max(rmul, gmul, bmul);

    rmul /= maxRGB;
    gmul /= maxRGB;
    bmul /= maxRGB;


    if(settings->CRI_color != 0) { //activate if CRi_color !=0
        // begin CRI_RT : color rendering index RT - adaptation of CRI by J.Desmis
        // CRI = 100 for Blackbody and Daylight
        // calculate from spectral data  values X, Y, Z , for  color of colorchecker24 , SG, DC, JDC_468
        // only for lamp different of tungstene
        // first calcul with illuminant (choice)
        // and calcul with : blackbody at equivalent temp of lamp
        // CRI_color-1 = display Lab values of color CRI_color -1
        const double whiteD50[3] = {0.9646019585, 1.0, 0.8244507152}; //calculate with this tool : spect 5nm
        double Xchk[50], Ychk[50], Zchk[50]; //50 : I think it's a good limit for number of color : for CRI and Palette
        double Xcam02[50], Ycam02[50], Zcam02[50];

        double XchkLamp[50], YchkLamp[50], ZchkLamp[50];
        double Xcam02Lamp[50], Ycam02Lamp[50], Zcam02Lamp[50];
        const double epsilon = 0.008856; //Lab

        double xr[50], yr[50], zr[50];

        int illum;
        int numero_color = settings->CRI_color - 1;

        //spectral data illuminant (actually 21): only those necessary (lamp, fluorescent, LED) others CRI=100 (not Flash...)
        const double* spec_color[] = {
            ColorchechredC3_spect, ColorchechOraA2_spect, ColorchechYelD3_spect, ColorchechGreE2_spect, ColorchechGreB3_spect,
            ColorchechCyaF3_spect, ColorchechPurD2_spect, ColorchechMagE3_spect, ColorchechSkiA138_13_14_spect, ColorchechGraC4_67_spect,
            ColorchechSkiB166_18_18_spect, ColorchechBluC150_m5_m22_spect, ColorchechDCBluN881_m7_m14_spect, ColorchechSGSkiF763_14_26_spect,
            ColorchechSGSkiK285_11_17_spect, ColorchechWhiA496_spect, ColorchechGreD1_spect, ColorchechSGBlaN3_6_spect,
            JDC468_GraK14_44_spect, JDC468_BluH10_spect
        };

        int N_c = sizeof(spec_color) / sizeof(spec_color[0]); //number of color

        bool CRI_type = false;
        double tempw;

        if      (method == "Fluo F1")              {
            CRI_type = true;
            tempw = 6430;
            illum = 1;
        } else if (method == "Fluo F2")              {
            CRI_type = true;
            tempw = 4230;
            illum = 2;
        } else if (method == "Fluo F3")              {
            CRI_type = true;
            tempw = 3450;
            illum = 3;
        } else if (method == "Fluo F4")              {
            CRI_type = true;
            tempw = 2940;
            illum = 4;
        } else if (method == "Fluo F5")              {
            CRI_type = true;
            tempw = 6350;
            illum = 5;
        } else if (method == "Fluo F6")              {
            CRI_type = true;
            tempw = 4150;
            illum = 6;
        } else if (method == "Fluo F7")              {
            CRI_type = true;
            tempw = 6500;
            illum = 7;
        } else if (method == "Fluo F8")              {
            CRI_type = true;
            tempw = 5020;
            illum = 8;
        } else if (method == "Fluo F9")              {
            CRI_type = true;
            tempw = 4330;
            illum = 9;
        } else if (method == "Fluo F10")             {
            CRI_type = true;
            tempw = 5300;
            illum = 10;
        } else if (method == "Fluo F11")             {
            CRI_type = true;
            tempw = 4000;
            illum = 11;
        } else if (method == "Fluo F12")             {
            CRI_type = true;
            tempw = 3000;
            illum = 12;
        } else if (method == "HMI Lamp")             {
            CRI_type = true;
            tempw = 4760;
            illum = 13;
        } else if (method == "GTI Lamp")             {
            CRI_type = true;
            tempw = 5000;
            illum = 14;
        } else if (method == "JudgeIII Lamp")        {
            CRI_type = true;
            tempw = 5100;
            illum = 15;
        } else if (method == "Solux Lamp 3500K")     {
            CRI_type = true;
            tempw = 3480;
            illum = 16;
        } else if (method == "Solux Lamp 4100K")     {
            CRI_type = true;
            tempw = 3930;
            illum = 17;
        } else if (method == "Solux Lamp 4700K"   )  {
            CRI_type = true;
            tempw = 4700;
            illum = 18;
        } else if (method == "NG Solux Lamp 4700K")  {
            CRI_type = true;
            tempw = 4480;
            illum = 19;
        } else if (method == "LED LSI Lumelex 2040") {
            CRI_type = true;
            tempw = 2970;
            illum = 20;
        } else if (method == "LED CRS SP12 WWMR16")  {
            CRI_type = true;
            tempw = 3050;
            illum = 21;
        } else                                       {
            CRI_type = false;
        }

        if (CRI_type) {
            double x, y, z;
            double Ywb = 1.0;

            const double* spect_illum[] = {
                Daylight5300_spect, Cloudy6200_spect, Shade7600_spect, A2856_spect, FluoF1_spect, FluoF2_spect, FluoF3_spect,
                FluoF4_spect, FluoF5_spect, FluoF6_spect, FluoF7_spect, FluoF8_spect, FluoF9_spect, FluoF10_spect, FluoF11_spect,
                FluoF12_spect, HMI_spect, GTI_spect, JudgeIII_spect, Solux3500_spect, Solux4100_spect, Solux4700_spect,
                NG_Solux4700_spect, NG_CRSSP12WWMR16_spect, NG_CRSSP12WWMR16_spect
            };

            float DeltaE[50], DeltaEs[8];
            float quadCRI = 0.0f, quadCRIs = 0.0f;
            float CRI_RT = 0.0, CRI[50];
            float CRI_RTs = 0.0, CRIs[8];

            for(int i = 0; i < N_c; i++) {
                spectrum_to_color_xyz_preset(spec_color[i], spect_illum[illum + 3], XchkLamp[i], YchkLamp[i], ZchkLamp[i]);
            }

            //calculate XYZ for each color : for Blackbody and Daylight at tempw
            if(tempw <= INITIALBLACKBODY) {
                for(int i = 0; i < N_c; i++) {
                    spectrum_to_color_xyz_blackbody(spec_color[i], tempw, Xchk[i], Ychk[i], Zchk[i]);
                }

                spectrum_to_xyz_blackbody(tempw, x, y, z);//for white point
            } else { // after 6600K (arbitrary) I use daylight...because ...but there is no lamp...
                double m11, m22, x_DD, y_DD, interm2;

                if (tempw <= 7000) {
                    x_DD = -4.6070e9 / (tempw * tempw * tempw) + 2.9678e6 / (tempw * tempw) + 0.09911e3 / tempw + 0.244063;
                } else {
                    x_DD = -2.0064e9 / (tempw * tempw * tempw) + 1.9018e6 / (tempw * tempw) + 0.24748e3 / tempw + 0.237040;
                }

                y_DD = -3.0 * x_DD * x_DD + 2.87 * x_DD - 0.275;
                //calculate D -daylight in function of s0, s1, s2 and temp ==> x_D y_D
                //S(lamda)=So(lambda)+m1*s1(lambda)+m2*s2(lambda)
                interm2 = (0.0241 + 0.2562 * x_DD - 0.734 * y_DD);
                m11 = (-1.3515 - 1.7703 * x_DD + 5.9114 * y_DD) / interm2;
                m22 = (0.03 - 31.4424 * x_DD + 30.0717 * y_DD) / interm2;

                for(int i = 0; i < N_c; i++) {
                    spectrum_to_color_xyz_daylight(spec_color[i], m11, m22, Xchk[i], Ychk[i], Zchk[i]);
                }

                spectrum_to_xyz_daylight(m11, m22, x, y, z);
            }

            if (settings->verbose) {
                double correl_temp = 0.0;
                XYZtoCorColorTemp(Xwb, Ywb, Zwb, correl_temp);
                printf("Correlated temperature (lamp)=%i\n", (int) correl_temp); //use only for lamp...otherwise It give an information!!
            }

            double Xwb_bb = x / y; //white balance for blackbody
            double Ywb_bb = 1.0;
            double Zwb_bb = (1.0 - x - y) / y;

            //calculate Matrix CAM02 : better than Von Kries and Bradford==> for Lamp
            double  adap = 1.0;
            double CAM02BB00, CAM02BB01, CAM02BB02, CAM02BB10, CAM02BB11, CAM02BB12, CAM02BB20, CAM02BB21, CAM02BB22; //for CIECAT02
            cieCAT02(Xwb, Ywb, Zwb, CAM02BB00, CAM02BB01, CAM02BB02, CAM02BB10, CAM02BB11, CAM02BB12, CAM02BB20, CAM02BB21, CAM02BB22, adap);

            //here new value of X,Y,Z for lamp with chromatic CAM02 adaptation
            for(int i = 0; i < N_c; i++) {
                Xcam02Lamp[i] = CAM02BB00 * XchkLamp[i] + CAM02BB01 * YchkLamp[i] + CAM02BB02 * ZchkLamp[i] ;
                Ycam02Lamp[i] = CAM02BB10 * XchkLamp[i] + CAM02BB11 * YchkLamp[i] + CAM02BB12 * ZchkLamp[i] ;
                Zcam02Lamp[i] = CAM02BB20 * XchkLamp[i] + CAM02BB21 * YchkLamp[i] + CAM02BB22 * ZchkLamp[i] ;
            }

            //now calculate CAM02 for Blackbody (or Daylight) at tempx

            cieCAT02(Xwb_bb, Ywb_bb, Zwb_bb, CAM02BB00, CAM02BB01, CAM02BB02, CAM02BB10, CAM02BB11, CAM02BB12, CAM02BB20, CAM02BB21, CAM02BB22, adap);

            //here new value of X,Y,Z for blackbody with chromatic CAM02 adaptation

            for(int i = 0; i < N_c; i++) {
                Xcam02[i] = CAM02BB00 * Xchk[i] + CAM02BB01 * Ychk[i] + CAM02BB02 * Zchk[i] ;
                Ycam02[i] = CAM02BB10 * Xchk[i] + CAM02BB11 * Ychk[i] + CAM02BB12 * Zchk[i] ;
                Zcam02[i] = CAM02BB20 * Xchk[i] + CAM02BB21 * Ychk[i] + CAM02BB22 * Zchk[i] ;
                //  printf("CoulXYZ %i X %f Y %f Z %f\n", i, Xchk[i],Ychk[i],Zchk[i]);
                //  printf("CoulCAM %i X %f Y %f Z %f\n", i, Xcam02[i],Ycam02[i],Zcam02[i]);
            }

            //now conversion to Lab
            // Lamp
            double fx[50], fy[50], fz[50];

            for(int i = 0; i < N_c; i++) {
                xr[i] = Xcam02Lamp[i] / whiteD50[0];
                yr[i] = Ycam02Lamp[i] / whiteD50[1];
                zr[i] = Zcam02Lamp[i] / whiteD50[2];

                // xr, yr , zr > epsilon
                if(xr[i] > epsilon) {
                    fx[i] = std::cbrt(xr[i]);
                } else {
                    fx[i] = (903.3 * xr[i] + 16.0) / 116.0;
                }

                if(yr[i] > epsilon) {
                    fy[i] = std::cbrt(yr[i]);
                } else {
                    fy[i] = (903.3 * yr[i] + 16.0) / 116.0;
                }

                if(zr[i] > epsilon) {
                    fz[i] = std::cbrt(zr[i]);
                } else {
                    fz[i] = (903.3 * zr[i] + 16.0) / 116.0;
                }
            }

            double Llamp[50], alamp[50], blamp[50];
            for(int i = 0; i < N_c; i++) {
                Llamp[i] = 116.0 * fy[i] - 16.0;
                alamp[i] = 500.0 * (fx[i] - fy[i]);
                blamp[i] = 200.0 * (fy[i] - fz[i]);
            }

            //blackbody at tempx
            for(int i = 0; i < N_c; i++) {
                xr[i] = Xcam02[i] / whiteD50[0];
                yr[i] = Ycam02[i] / whiteD50[1];
                zr[i] = Zcam02[i] / whiteD50[2];

                //
                if(xr[i] > epsilon) {
                    fx[i] = std::cbrt(xr[i]);
                } else {
                    fx[i] = (903.3 * xr[i] + 16.0) / 116.0;
                }

                if(yr[i] > epsilon) {
                    fy[i] = std::cbrt(yr[i]);
                } else {
                    fy[i] = (903.3 * yr[i] + 16.0) / 116.0;
                }

                if(zr[i] > epsilon) {
                    fz[i] = std::cbrt(zr[i]);
                } else {
                    fz[i] = (903.3 * zr[i] + 16.0) / 116.0;
                }
            }

            double Lbb[50], abb[50], bbb[50];

            for(int i = 0; i < N_c; i++) {
                Lbb[i] = 116.*fy[i] - 16.;
                abb[i] = 500.*(fx[i] - fy[i]);
                bbb[i] = 200.*(fy[i] - fz[i]);
            }

            //display value to verify calculus
            if(settings->CRI_color != 0) {
                printf("Color Number %i\n", numero_color);
                printf("L_refer=%2.2f a=%2.2f b=%2.2f\n", Lbb[numero_color], abb[numero_color], bbb[numero_color]);
                printf("L_lamp=%2.2f al=%2.2f bl=%2.2f\n", Llamp[numero_color], alamp[numero_color], blamp[numero_color]);
            }

            //then calculate DeltaE CIE 1976
            for(int i = 0; i < 8; i++) {
                DeltaEs[i] = sqrt((Lbb[i] - Llamp[i]) * (Lbb[i] - Llamp[i]) + (abb[i] - alamp[i]) * (abb[i] - alamp[i]) + (bbb[i] - blamp[i]) * (bbb[i] - blamp[i]));
            }

            for(int i = 0; i < 8; i++) {
                CRIs[i] = 100 - 3.f * DeltaEs[i];    //3.0 coef to adapt ==> same results than CRI "official"
            }

            for(int i = 0; i < 8; i++) {
                CRI_RTs += CRIs[i];
            }

            CRI_RTs /= 8;

            for(int i = 0; i < 8; i++) {
                quadCRIs += (CRIs[i] - CRI_RTs) * (CRIs[i] - CRI_RTs);
            }

            quadCRIs /= 8;

            for(int i = 0; i < N_c; i++) {
                DeltaE[i] = sqrt((Lbb[i] - Llamp[i]) * (Lbb[i] - Llamp[i]) + (abb[i] - alamp[i]) * (abb[i] - alamp[i]) + (bbb[i] - blamp[i]) * (bbb[i] - blamp[i]));
            }

            for(int i = 0; i < N_c; i++) {
                CRI[i] = 100 - 3.f * DeltaE[i];    //3.0 coef to adapt ==> same results than CRI "official"
            }

            for(int i = 0; i < N_c; i++) {
                CRI_RT += CRI[i];
            }

            CRI_RT /= N_c;

            for(int i = 0; i < N_c; i++) {
                quadCRI += (CRI[i] - CRI_RT) * (CRI[i] - CRI_RT);
            }

            quadCRI /= N_c;

            if(settings->CRI_color != 0) {
                printf("CRI_standard=%i CRI:1->8=%i %i %i %i %i %i %i %i  Sigma=%2.1f\n", (int) CRI_RTs, (int) CRIs[0], (int) CRIs[1], (int) CRIs[2], (int) CRIs[3], (int) CRIs[4], (int) CRIs[5], (int) CRIs[6], (int) CRIs[7], sqrt(static_cast<double>(quadCRIs)));
                printf("CRI_RT_exten=%i CRI:9->20=%i %i %i %i %i %i %i %i %i %i %i %i Sigma=%2.1f\n", (int) CRI_RT, (int) CRI[8], (int) CRI[9], (int) CRI[10], (int) CRI[11], (int) CRI[12], (int) CRI[13], (int) CRI[14], (int) CRI[15], (int) CRI[16], (int) CRI[17], (int) CRI[18], (int) CRI[19], static_cast<double>(sqrt(quadCRI)));
            }
        }
    }
}

/*
    Calculate Planck's radiation
*/
//calculate spectral data for blackbody at temp!
double ColorTemp::blackbody_spect(double wavelength, double temperature)
{
    const double wlm = wavelength * 1e-9;   /* Wavelength in meters */
    return (3.7417715247e-16 / rtengine::pow5(wlm)) /              //3.7417..= c1 = 2*Pi*h*c2  where h=Planck constant, c=velocity of light
           (xexp(1.438786e-2 / (wlm * temperature)) - 1.0); //1.4387..= c2 = h*c/k  where k=Boltzmann constant
}

/*
The next 3 methods are inspired from:

   a) Colour Rendering of Spectra  by John Walker
      http://www.fourmilab.ch/
      This program is in the public domain.

   b) Bruce Lindbloom
      Adapted to RawTherapee by J.Desmis

this values are often called xBar yBar zBar and are characteristics of a color / illuminant

values cie_colour_match[][3] = 2° Standard Observer x2, y2, z2
E.g. for 380nm: x2=0.001368  y2=0.000039  z2=0.006451  round in J.Walker to 0.0014  0.0000 0.0065 above
I have increase precision used by J.Walker  and pass to 350nm to 830nm
And also add 10°  standard observer
*/

void ColorTemp::spectrum_to_xyz_daylight(double _m1, double _m2, double &x, double &y, double &z)
{
    int i;
    double lambda, X = 0, Y = 0, Z = 0, XYZ;

    for (i = 0, lambda = 350.; lambda < 830.1; i++, lambda += 5.) {
        double Me = daylight_spect(lambda, _m1, _m2);
        X += Me * cie_colour_match_jd[i][0];
        Y += Me * cie_colour_match_jd[i][1];
        Z += Me * cie_colour_match_jd[i][2];
    }

    XYZ = (X + Y + Z);
    x = X / XYZ;
    y = Y / XYZ;
    z = Z / XYZ;
}

void ColorTemp::spectrum_to_xyz_blackbody(double _temp, double &x, double &y, double &z)
{
    int i;
    double lambda, X = 0, Y = 0, Z = 0, XYZ;

    for (i = 0, lambda = 350.; lambda < 830.1; i++, lambda += 5.) {
        double Me = blackbody_spect(lambda, _temp);
        X += Me * cie_colour_match_jd[i][0];
        Y += Me * cie_colour_match_jd[i][1];
        Z += Me * cie_colour_match_jd[i][2];
    }

    XYZ = (X + Y + Z);
    x = X / XYZ;
    y = Y / XYZ;
    z = Z / XYZ;
}

void ColorTemp::spectrum_to_xyz_preset(const double* spec_intens, double &x, double &y, double &z)
{
    int i;
    double lambda, X = 0, Y = 0, Z = 0, XYZ;

    /*
    Inspired from:

       a) Colour Rendering of Spectra  by John Walker
          http://www.fourmilab.ch/
          This program is in the public domain.

       b) Bruce Lindbloom
          Adapted to RawTherapee by J.Desmis

    this values are often called xBar yBar zBar and are characteristics of a color / illuminant

    values cie_colour_match[][3] = 2° Standard Observer x2, y2, z2
    E.g. for 380nm: x2=0.001368  y2=0.000039  z2=0.006451  round in J.Walker to 0.0014  0.0000 0.0065 above
    I have increased the precision used by J.Walker and pass from 350nm to 830nm
    And also add standard observer 10°
    */
    for (i = 0, lambda = 350.; lambda < 830.1; i++, lambda += 5.) {
        double Me = get_spectral_color(lambda, spec_intens);
        X += Me * cie_colour_match_jd[i][0];
        Y += Me * cie_colour_match_jd[i][1];
        Z += Me * cie_colour_match_jd[i][2];
    }

    XYZ = (X + Y + Z);
    x = X / XYZ;
    y = Y / XYZ;
    z = Z / XYZ;
}

//calculate XYZ from spectrum data (color) and illuminant : J.Desmis December 2011
void ColorTemp::spectrum_to_color_xyz_preset(const double* spec_color, const double* spec_intens, double &xx, double &yy, double &zz)
{
    int i;
    double lambda, X = 0, Y = 0, Z = 0, Yo = 0;

    for (i = 0, lambda = 350; lambda < 830.1; i++, lambda += 5) {

        double Me;
        double Mc;

        Me = get_spectral_color(lambda, spec_color);
        Mc = get_spectral_color(lambda, spec_intens);
        X += Mc * cie_colour_match_jd[i][0] * Me;
        Y += Mc * cie_colour_match_jd[i][1] * Me;
        Z += Mc * cie_colour_match_jd[i][2] * Me;
    }

    for (i = 0, lambda = 350; lambda < 830.1; i++, lambda += 5) {

        double Ms;

        Ms = get_spectral_color(lambda, spec_intens);
        Yo += cie_colour_match_jd[i][1] * Ms;
    }

    xx = X / Yo;
    yy = Y / Yo;
    zz = Z / Yo;
}

//calculate XYZ from spectrum data (color) and illuminant : J.Desmis december 2011
void ColorTemp::spectrum_to_color_xyz_daylight(const double* spec_color, double _m1, double _m2, double &xx, double &yy, double &zz)
{
    int i;
    double lambda, X = 0, Y = 0, Z = 0;

    for (i = 0, lambda = 350; lambda < 830.1; i++, lambda += 5) {
        const double Me = spec_color[i];
        const double Mc = daylight_spect(lambda, _m1, _m2);
        X += Mc * cie_colour_match_jd[i][0] * Me;
        Y += Mc * cie_colour_match_jd[i][1] * Me;
        Z += Mc * cie_colour_match_jd[i][2] * Me;
    }

    xx = X / Y;
    yy = 1.0;
    zz = Z / Y;
}

//calculate XYZ from spectrum data (color) and illuminant : J.Desmis december 2011
void ColorTemp::spectrum_to_color_xyz_blackbody(const double* spec_color, double _temp, double &xx, double &yy, double &zz)
{
    int i;
    double lambda, X = 0, Y = 0, Z = 0;

    for (i = 0, lambda = 350; lambda < 830.1; i++, lambda += 5) {
        const double Me = spec_color[i];
        const double Mc = blackbody_spect(lambda, _temp);
        X += Mc * cie_colour_match_jd[i][0] * Me;
        Y += Mc * cie_colour_match_jd[i][1] * Me;
        Z += Mc * cie_colour_match_jd[i][2] * Me;
    }

    xx = X / Y;
    yy = 1.0;
    zz = Z / Y;
}

double ColorTemp::daylight_spect(double wavelength, double m1, double m2)
{
    //Values for Daylight illuminant: s0 s1 s2
    //s0
    static const double s0[97] = {61.80, 61.65, 61.50, 65.15, 68.80, 66.10, 63.40, 64.60, 65.80, 80.30, 94.80, 99.80, 104.80, 105.35, 105.90, 101.35, 96.80, 105.35, 113.90, 119.75, 125.60, 125.55, 125.50, 123.40, 121.30, 121.30, 121.30, 117.40, 113.50, 113.30,
                                  113.10, 111.95, 110.80, 108.65, 106.50, 107.65, 108.80, 107.05, 105.30, 104.85, 104.40, 102.20, 100.00, 98.00, 96.00, 95.55, 95.10, 92.10, 89.10, 89.80, 90.50, 90.40, 90.30, 89.35, 88.40, 86.20, 84.00, 84.55, 85.10,
                                  83.50, 81.90, 82.25, 82.60, 83.75, 84.90, 83.10, 81.30, 76.60, 71.90, 73.10, 74.30, 75.35, 76.40, 69.85, 63.30, 67.50, 71.70, 74.35, 77.00, 71.10, 65.20, 56.45, 47.70, 58.15, 68.60, 66.80, 65.00, 65.50, 66.00, 63.50, 61.00, 57.15,
                                  53.30, 56.10, 58.90, 60.40, 61.90
                                 };
    //s1
    static const double s1[97] = {41.60, 39.80, 38.00, 40.20, 42.40, 40.45, 38.50, 36.75, 35.00, 39.20, 43.40, 44.85, 46.30, 45.10, 43.90, 40.50, 37.10, 36.90, 36.70, 36.30, 35.90, 34.25, 32.60, 30.25, 27.90, 26.10, 24.30, 22.20, 20.10, 18.15, 16.20, 14.70,
                                  13.20, 10.90, 8.60, 7.35, 6.10, 5.15, 4.20, 3.05, 1.90, 0.95, 0.00, -0.80, -1.60, -2.55, -3.50, -3.50, -3.50, -4.65, -5.80, -6.50, -7.20, -7.90, -8.60, -9.05, -9.50, -10.20, -10.90, -10.80, -10.70, -11.35, -12.00, -13.00, -14.00,
                                  -13.80, -13.60, -12.80, -12.00, -12.65, -13.30, -13.10, -12.90, -11.75, -10.60, -11.10, -11.60, -11.90, -12.20, -11.20, -10.20, -9.00, -7.80, -9.50, -11.20, -10.80, -10.40, -10.50, -10.60, -10.15, -9.70, -9.00, -8.30,
                                  -8.80, -9.30, -9.55, -9.80
                                 };
    //s2
    static const double s2[97] = {6.70, 6.00, 5.30, 5.70, 6.10, 4.55, 3.00, 2.10, 1.20, 0.05, -1.10, -0.80, -0.50, -0.60, -0.70, -0.95, -1.20, -1.90, -2.60, -2.75, -2.90, -2.85, -2.80, -2.70, -2.60, -2.60, -2.60, -2.20, -1.80, -1.65, -1.50, -1.40, -1.30,
                                  -1.25, -1.20, -1.10, -1.00, -0.75, -0.50, -0.40, -0.30, -0.15, 0.00, 0.10, 0.20, 0.35, 0.50, 1.30, 2.10, 2.65, 3.20, 3.65, 4.10, 4.40, 4.70, 4.90, 5.10, 5.90, 6.70, 7.00, 7.30, 7.95, 8.60, 9.20, 9.80, 10.00, 10.20, 9.25, 8.30, 8.95,
                                  9.60, 9.05, 8.50, 7.75, 7.00, 7.30, 7.60, 7.80, 8.00, 7.35, 6.70, 5.95, 5.20, 6.30, 7.40, 7.10, 6.80, 6.90, 7.00, 6.70, 6.40, 5.95, 5.50, 5.80, 6.10, 6.30, 6.50
                                 };

    int wlm = (int) ((wavelength - 350.) / 5.);
    return (s0[wlm] + m1 * s1[wlm] + m2 * s2[wlm]);
}

//tempxy : return x and y of xyY for 200 or more refreence color, and for T temperature from 2000K to 12000K
// we can change step for temperature and increase number  for T > 7500K if necessary
//these values Temp, x, y are references for all calculations and very precise.
//copyright J.Desmis august  2017 and june 2018
void ColorTemp::tempxy(bool separated, int repref, float **Tx, float **Ty, float **Tz, float **Ta, float **Tb, float **TL, double *TX, double *TY, double *TZ, const procparams::WBParams & wbpar)
{
    const double* spec_colorforxcyc[] = {//color references
        JDC468_BluH10_spect, JDC468_BluD6_spect, ColorchechCyaF3_spect, JDC468_BluM5_spect, // 0 3
        ColorGreenM25_spect,   JDC468_GreK7_spect, ColabSky42_0_m24_spect, ColabSky60_0_m31_spect,  ColorchechBluC150_m5_m22_spect,//8
        JDC468_GreQ7_spect, ColorchechDCBluN881_m7_m14_spect, ColorchechGreB3_spect, ColorchechPurD2_spect,  //12
        ColorchechSGBlaN3_6_spect, ColorchechGraC4_67_spect, JDC468_K15_87greyspect,//15
        JDC468_GraK14_44_spect, ColorGreenalsi_spect, Fictif_61greyspect, ColorchechGreD1_spect,//19
        ColorchechWhiA496_spect, JDC468_GreA10_spect, JDC468_GreI8_spect,//22
        ColabSkin91_4_14_spect, JDC468_PurE24_spect, //24
        ColorchechSGSkiK285_11_17_spect,  ColorchechGreE2_spect, ColorchechMagE3_spect, //27
        ColorchechSkiB166_18_18_spect, ColabSkin70_7_32_spect, ColorchechSGSkiF763_14_26_spect,//30
        ColorchechSkiA138_13_14_spect, ColabSkin57_22_18_spect, JDC468_YelN10_spect,//33
        ColabSkin35_15_17_spect, ColabSkin40_17_17_spect, ColorRedkurttu_spect, ColorYellowkeltano_spect,  ColorchechYelD3_spect, JDC468_OraO18_spect,//39
        JDC468_GreN7_spect, JDC468_RedG21va_spect, JDC468_OraD17_spect,//42
        ColorchechredC3_spect, JDC468_RedI9_spect, ColorRedpetunia_spect, ColorchechOraA2_spect,//46
        ColabSkin87_8_8_spect, ColabSkin89_8_21_spect, ColabSkin75_8_4_spect, ColabSkin75_10_33_spect,//50
        ColabSkin65_33_11_spect, ColabSkin65_7_24_spect, ColabSkin57_19_6_spect, ColabSkin57_4_19_spect, ColabSkin57_10_28_spect, ColabSkin40_17_6_spect,//56
        ColabSkin26_18_18_spect, ColabSkin90_m1_20_spect, ColorRedlupiini_spect, ColorRedhevosminttu_spect, //60
        ColorRedneilikka_spect, ColorRedpelagornia_spect, ColorRedtalvio_spect, ColorBrownpoimulehti_spect, ColorOrangetuntematon_spect,//65
        ColorOrangetlehmus_spect, ColorOrangvaahtera_spect, ColorBrownlehmus_spect, ColorBrownuotiosammal_spect,//69
        ColorBlacksoil_spect, ColorGraynahjajaekaelae_spect, //71
        ColorGreennuotisammal_spect, ColorGreenleskenlehti_spect, ColorGreenlinnunkaali_spect, //74
        ColorGreenpelto_spect, ColorGreenrodvoikukka, ColorGreenlehmus, ColorGreenlinden, ColorYellowlehmus, ColorYellowsuikeroalpi, //80
        ColorYellowpensashanhikki1, ColorYellowpensashanhikki2,  ColorBluehiidenvirna,  ColorBluekurkkuyrtti, //84
        ColorPinksiankaersaemoe, ColorVioletharakankello, ColorVioletalsikeapila, ColorVioletakilleija, ColorOrangekehaekukka,//89
        ColorRedpihlaja, ColorVioletpetunia, ColorVioletorvokki, ColorBluesinisievikki, ColorBlueiisoppi,  ColorBluelobelia, //95
        ColorWhiteojaka, ColorWhitepetunia, ColorWhitepelargonia, ColorWhitepaeivaen, JDC468_B14_75Redspect,//100
        ColorGreenkoriste, ColorGreenpoimulehti, ColorGreenhopeapaju, //103
        ColorReduuden, ColorRedpajuan, ColorRedjaloan, ColorBlueukon, ColorBlueorvokki, ColorBluemalvikki, //109
        ColorBlackmaito, ColorOrangpihlaja, ColorBlackpihlaja, //112
        ColorViolA1_spect, ColorViolA4_spect, ColorViolA6_spect, ColorBlueSkyK3_spect, ColorBlueSkyK9_spect, //117
        ColorBlueSkyC4_spect, ColorBlueSkyC14_spect, ColorBlueSkyE4_spect, //120
        ColorBlueSkyM1_spect, ColorBlueSky2B1_spect, ColorBlueSkyT7_spect, //123
        ColorBlueSkyU19_spect, ColorBlueSkyU2_spect, ColorBlueSkyT17_spect, //126
        ColorBlackM8_spect, ColorBlackM12_spect, ColorBlackM13_spect, ColorWhite2B12_spect, ColorWhite2B14_spect, //131
        JDC468_Blackred97_spect, JDC468_Blackredbl443_spect, JDC468_Blackbl27_spect, JDC468_Blackbl28_spect, //135
        JDC468_Blackgr214_spect, JDC468_Blackbl436_spect, JDC468_Whitebl455_spect, JDC468_Blackvio101_spect, JDC468_Whitebl92_spect, JDC468_Greyredbl94_spect, //141
        JDC468_Blue32_spect, JDC468_Blue236_spect, JDC468_Gre300_spect, //144
        JDC468_Blue340_spect, JDC468_Gree110_spect, JDC468_Gree457_spect, JDC468_Yel241_spect, JDC468_Ora321_spect, JDC468_Yellow353_spect, JDC468_Mag465_spect, //151
        JDC468_Mag333_spect, JDC468_Mag203_spect, J570_BlueB6_spect, J570_BlueB15_spect, J570_BlueC2_spect, J570_BlueC14_spect, J570_BlueC16_spect,//158
        J570_BlueF1_spect, J570_BlueF2_spect, J570_BlueF10_spect, J570_BlueF13_spect, J570_BlueG9_spect, J570_BlueG19_spect, J570_BlueI5_spect,//165
        J570_BlueI3_spect, J570_BlueI19_spect, J570_BlueJ4_spect, J570_BlueJ6_spect, J570_BlueJ11_spect, J570_BlueK5_spect, //171
        J570_BlueN1_spect, J570_BlueN4_spect, J570_BlueO19_spect, J570_BlueU8_spect, J570_NeuN8_spect,//176
        J570_NeuN9_spect, J570_NeuO8_spect, J570_NeuO11_spect, J570_NeuD5_spect,//180
        J570_NeuE11_spect, J570_NeuK16_spect, J570_NeuM3_spect, J570_NeuN18_spect,
        J570_NeuQ1_spect, J570_NeuS7_spect,
        J570_NeuV10_spect, J570_NeuW18_spect, J570_NeuZ14_spect, //189
        J570_NeuC18_spect, J570_NeuD17_spect, J570_NeuJ11_spect, J570_NeuL4_spect, Colorlab_n72_n2_spect,
        Colorlab_10_n70_spect, Colorlab_n33_n70_spect, Colorlab_n8_n74_spect, Colorlab_19_n69_spect, Colorlab_n80_10_spect, Colorlab_n80_26_spect,
        Colorlab_n80_5_9_5_9spect //, Colorlab_n57_5_6_9spect

        /*JDC468_greyc14_66_spect, JDC468_greym13_325_spect, JDC468_greyf26_156_spect*/
    };


    typedef struct WbTxyz {
        double Tem;
        double XX;
        double ZZ;
    } WbTxyz;
    //probably can be "passed" with rawimagesource.cc but I don't know how to do this.
    constexpr WbTxyz Txyz[118] = {//temperature Xwb Zwb 118 values - same table as in Rawimagesource.cc  x wb and y wb are calculated after
        {2001., 1.273842, 0.145295},
        {2101., 1.244008, 0.167533},
        {2201., 1.217338, 0.190697},
        {2301., 1.193444, 0.214632},
        {2401., 1.171996, 0.239195},
        {2501., 1.152883, 0.264539},
        {2605., 1.134667, 0.290722},
        {2655., 1.126659, 0.303556},
        {2705., 1.119049, 0.316446},
        {2755., 1.111814, 0.329381},
        {2803., 1.105381, 0.342193},
        {2856., 1.098258, 0.355599},
        {2910., 1.091550, 0.369645},
        {2960., 1.085649, 0.382655},
        {3003., 1.080982, 0.394258},
        {3050., 1.075727, 0.406057},
        {3103., 1.070277, 0.419815},
        {3153., 1.065384, 0.432769},
        {3203., 1.060906, 0.446161},
        {3250., 1.056535, 0.457806},
        {3303., 1.052034, 0.471422},
        {3353., 1.047990, 0.484218},
        {3400., 1.044547, 0.496719},
        {3450., 1.040667, 0.508891},
        {3500., 1.037145, 0.521523},
        {3550., 1.033783, 0.534090},
        {3600., 1.030574, 0.546590},
        {3650., 1.027510, 0.559020},
        {3699., 1.024834, 0.571722},
        {3801., 1.019072, 0.596102},
        {3851., 1.016527, 0.608221},
        {3902., 1.014244, 0.621136},
        {3952., 1.011729, 0.632447},
        {4002., 0.996153, 0.609518},
        {4052., 0.993720, 0.620805},
        {4102., 0.993908, 0.631520},
        {4152., 0.989179, 0.643262},
        {4202., 0.989283, 0.653999},
        {4252., 0.985039, 0.665536},
        {4302., 0.985067, 0.676288},
        {4352., 0.981271, 0.687599},
        {4402., 0.981228, 0.698349},
        {4452., 0.977843, 0.709425},
        {4502., 0.977736, 0.720159},
        {4552., 0.974728, 0.730993},
        {4602., 0.974562, 0.741698},
        {4652., 0.971899, 0.752284},
        {4702., 0.971681, 0.762949},
        {4752., 0.969335, 0.773285},
        {4802., 0.969069, 0.783899},
        {4827., 0.967570, 0.788836},
        {4852., 0.967011, 0.793982},
        {4877., 0.966465, 0.799108},
        {4902., 0.965933, 0.804214},
        {4927., 0.965414, 0.809229},
        {4952., 0.964908, 0.814366},
        {4977., 0.964415, 0.819412},
        {5002., 0.963934, 0.824438},
        {5027., 0.963465, 0.829444},
        {5052., 0.963008, 0.834429},
        {5077., 0.962563, 0.839395},
        {5102., 0.962129, 0.844339},
        {5127., 0.961706, 0.849263},
        {5152., 0.961294, 0.854166},
        {5177., 0.960893, 0.859049},
        {5202., 0.960501, 0.863911},
        {5252., 0.959749, 0.873572},
        {5302., 0.959313, 0.883815},
        {5352., 0.958361, 0.892644},
        {5402., 0.957903, 0.902793},
        {5452., 0.957116, 0.911379},
        {5502., 0.956639, 0.921431},
        {5552., 0.956002, 0.929779},
        {5602., 0.955509, 0.939728},
        {5652., 0.955008, 0.947842},
        {5702., 0.954502, 0.957685},
        {5752., 0.954124, 0.965569},
        {5802., 0.953608, 0.975303},
        {5852., 0.953342, 0.982963},
        {5902., 0.952818, 0.992584},
        {5952., 0.952652, 1.000025},
        {6002., 0.952122, 1.009532},
        {6052., 0.952047, 1.016759},
        {6102., 0.951514, 1.026149},
        {6152., 0.951520, 1.033168},
        {6202., 0.950985, 1.042439},
        {6252., 0.951064, 1.049256},
        {6302., 0.950530, 1.058406},
        {6352., 0.950674, 1.065027},
        {6402., 0.950143, 1.074055},
        {6452., 0.950345, 1.080484},
        {6502., 0.950201, 1.088097},
        {6552., 0.950070, 1.095633},
        {6602., 0.949952, 1.103094},
        {6652., 0.949846, 1.110479},
        {6702., 0.949752, 1.119138},
        {6752., 0.949668, 1.125027},
        {6802., 0.949596, 1.132190},
        {6902., 0.949033, 1.147691},
        {7002., 0.949402, 1.160129},
        {7152., 0.949348, 1.180429},
        {7301., 0.948896, 1.201432},
        {7451., 0.949434, 1.219076},
        {7601., 0.949099, 1.239061},
        {7751., 0.949729, 1.255559},
        {7901., 0.949498, 1.274460},
        {8151., 0.950361, 1.300912},
        {8301., 0.950253, 1.318464},
        {8451., 0.950966, 1.332651},
        {8601., 0.950941, 1.349261},
        {8801., 0.951772, 1.367421},
        {9001., 0.951969, 1.387639},
        {9201., 0.952784, 1.404422},
        {9401., 0.953081, 1.423213},
        {9901., 0.954537, 1.464134},
        {10501., 0.956321, 1.508623},
        {11001., 0.957747, 1.541281},
        {12001., 0.960440, 1.601019}
    };

    int N_c = sizeof(spec_colorforxcyc) / sizeof(spec_colorforxcyc[0]);   //number of color
    int N_t = sizeof(Txyz) / sizeof(Txyz[0]);   //number of temperature White point
    typedef struct XYZref {
        double Xref;
        double Yref;
        double Zref;
    } XYZref;
    XYZref Refxyz[N_c + 1];

    for (int i = 0; i < N_c; i++) {
        Refxyz[i].Xref = 0.f;
        Refxyz[i].Yref = 0.f;
        Refxyz[i].Zref = 0.f;
    }

    if (settings->verbose) {
        if (settings->itcwb_stdobserver10 == false) {
            printf("Use standard observer 2°\n");
        } else {
            printf("Use standard observer 10°\n");
        }
    }

    if (settings->itcwb_stdobserver10 == false) {
        for (int i = 0; i < 97; i++) {
            cie_colour_match_jd[i][0] = cie_colour_match_jd2[i][0];
            cie_colour_match_jd[i][1] = cie_colour_match_jd2[i][1];;
            cie_colour_match_jd[i][2] = cie_colour_match_jd2[i][2];
        }
    }

    if (separated) {
        const double tempw = Txyz[repref].Tem;

        if (tempw <= INITIALBLACKBODY) {
            for (int i = 0; i < N_c; i++) {
                spectrum_to_color_xyz_blackbody(spec_colorforxcyc[i], tempw, TX[i], TY[i], TZ[i]);
            }
        } else {
            double m11, m22, x_DD, y_DD, interm2;

            if (tempw <= 7000) {
                x_DD = -4.6070e9 / (tempw * tempw * tempw) + 2.9678e6 / (tempw * tempw) + 0.09911e3 / tempw + 0.244063;
            } else {
                x_DD = -2.0064e9 / (tempw * tempw * tempw) + 1.9018e6 / (tempw * tempw) + 0.24748e3 / tempw + 0.237040;
            }

            y_DD = -3.0 * x_DD * x_DD + 2.87 * x_DD - 0.275;
            //calculate D -daylight in function of s0, s1, s2 and temp ==> x_D y_D
            //S(lamda)=So(lambda)+m1*s1(lambda)+m2*s2(lambda)
            interm2 = (0.0241 + 0.2562 * x_DD - 0.734 * y_DD);
            m11 = (-1.3515 - 1.7703 * x_DD + 5.9114 * y_DD) / interm2;
            m22 = (0.03 - 31.4424 * x_DD + 30.0717 * y_DD) / interm2;

            for (int i = 0; i < N_c; i++) {
                spectrum_to_color_xyz_daylight(spec_colorforxcyc[i], m11, m22, TX[i], TY[i], TZ[i]);
            }
        }
    } else {
        for (int tt = 0; tt < N_t; tt++) {
            const double tempw = Txyz[tt].Tem;

            if (tempw <= INITIALBLACKBODY) {
                for (int i = 0; i < N_c; i++) {
                    spectrum_to_color_xyz_blackbody(spec_colorforxcyc[i], tempw, Refxyz[i].Xref, Refxyz[i].Yref, Refxyz[i].Zref);
                }
            } else {
                double x_DD;

                if (tempw <= 7000) {
                    x_DD = -4.6070e9 / (tempw * tempw * tempw) + 2.9678e6 / (tempw * tempw) + 0.09911e3 / tempw + 0.244063;
                } else {
                    x_DD = -2.0064e9 / (tempw * tempw * tempw) + 1.9018e6 / (tempw * tempw) + 0.24748e3 / tempw + 0.237040;
                }

                const double y_DD = -3.0 * x_DD * x_DD + 2.87 * x_DD - 0.275;
                //calculate D -daylight in function of s0, s1, s2 and temp ==> x_D y_D
                //S(lamda)=So(lambda)+m1*s1(lambda)+m2*s2(lambda)
                const double interm2 = (0.0241 + 0.2562 * x_DD - 0.734 * y_DD);
                const double m11 = (-1.3515 - 1.7703 * x_DD + 5.9114 * y_DD) / interm2;
                const double m22 = (0.03 - 31.4424 * x_DD + 30.0717 * y_DD) / interm2;

                for (int i = 0; i < N_c; i++) {
                    spectrum_to_color_xyz_daylight(spec_colorforxcyc[i], m11, m22, Refxyz[i].Xref, Refxyz[i].Yref, Refxyz[i].Zref);
                }
            }

            for (int i = 0; i < N_c; i++) {
                Tx[i][tt] = Refxyz[i].Xref;
                Ty[i][tt] = Refxyz[i].Yref;
                Tz[i][tt] = Refxyz[i].Zref;
            }
        }
    }
}


}

