/*
mpfrcx_treefromroots

high-level functions working with trees of real polynomials, the leaves
of which are linear or quadratic polynomials obtained from complex roots

Copyright (C) 2018 Andreas Enge

This file is part of the MPFRCX Library.

The MPFRCX Library is free software; you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation; either version 3 of the License, or (at your
option) any later version.

The MPFRCX Library is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
License for more details.

You should have received a copy of the GNU Lesser General Public License
along with the MPFRCX library; see the file COPYING.LESSER.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
MA 02111-1307, USA.
*/

#include "mpfrcx-impl.h"

/**************************************************************************/

void mpfrcx_init_set_quadratic (mpfrx_ptr h, mpc_srcptr z)
   /* Initialise h to the precision of z and set it to
      (X-z)*(X-zbar), where zbar is the complex conjugate of z. */

{
   mpfr_prec_t pr, pi, prec;

   mpc_get_prec2 (&pr, &pi, z);
   prec = MIN (pr, pi);

   mpfrx_init (h, 3, prec);
   mpfrx_set_deg (h, 2);
   mpfr_set_ui (h->coeff [2], 1, MPFR_RNDN);
   mpfr_mul_si (h->coeff [1], mpc_realref (z), -2, MPFR_RNDN);
   mpc_norm (h->coeff [0], z, MPFR_RNDN);
}

/**************************************************************************/

void mpfrcx_reconstruct_from_roots (mpfrx_ptr rop, mpc_t *roots,
   int *conjugates, int no)
   /* Takes an array of no complex elements and computes the polynomial
      having these as roots, that is, prod_{i=1}^n (X - roots[i]); the
      real and imaginary part of each element of roots must have the same
      precision. It is assumed that the resulting polynomial is real, that
      is, the roots are either real or come in complex conjugate pairs.
      This information is passed via the array conjugates:
      conjugates [i] == j when roots [j] is the complex conjugate
      of roots [i]; only the first element of a complex conjugate pair
      needs to be filled in. */

{
   int i;
   mpfrx_t *fac;
   int k1, k2, k; /* number of real roots, pairs of complex conjugate roots,
                     and their sum; we have no = k1 + 2*k2 */

   k1 = 0;
   k2 = 0;
   for (i = 0; i < no; i++)
      if (conjugates [i] == i)
         k1++;
      else if (conjugates [i] > i)
         k2++;
   k = k1 + k2;

   fac = (mpfrx_t *) malloc (k * sizeof (mpfrx_t));
   k = 0;
   for (i = 0; i < no; i++)
      if (conjugates [i] == i) {
         mpfrx_init_set_linear (fac [k], mpc_realref (roots [i]));
         k++;
      }
      else if (conjugates [i] > i) {
         mpfrcx_init_set_quadratic (fac [k], roots [i]);
         k++;
      }

   mpfrx_reconstruct (rop, fac, k);

   for (i = 0; i < k; i++)
      mpfrx_clear (fac [i]);
   free (fac);
}

/**************************************************************************/

void mpfrcx_subproducttree_from_roots (mpfrx_tree_ptr rop, mpc_t *roots,
   int *conjugates, int no)
   /* Initialises a subproduct tree from the linear and quadratic
      polynomials associated with the elements in roots, paired up as
      indicated by conjugates. All roots are assumed to be of the same
      precision, which is used for the initialisation. The exact size
      of the tree depends on the number of real and complex conjugate
      pairs of roots; the root of the tree will have degree no. */

{
   int i;
   int n1, n2, n;
      /* number of real and pairs of complex roots and their sum */
   mpfrx_t *fac;
   mpfr_prec_t prec = mpc_get_prec (roots [0]);

   n1 = 0;
   n2 = 0;
   for (i = 0; i < no; i++)
      if (conjugates [i] == i)
         n1++;
      else if (conjugates [i] > i)
         n2++;
   n = n1 + n2;

   mpfrx_tree_init (rop, n, prec);
   fac = (mpfrx_t *) malloc (n * sizeof (mpfrx_t));
   n = 0;
   for (i = 0; i < no; i++) {
      if (conjugates [i] == i) {
         mpfrx_init_set_linear (fac [n], mpc_realref (roots [i]));
         n++;
      }
      else if (conjugates [i] > i) {
         mpfrcx_init_set_quadratic (fac [n], roots [i]);
         n++;
      }
   }

   mpfrx_subproducttree (rop, fac);

   for (i = 0; i < n; i++)
      mpfrx_clear (fac [i]);
   free (fac);
}

/**************************************************************************/

void mpfrcx_hecke_from_roots (mpfrx_ptr rop, mpfrx_tree_srcptr subproducts,
   mpc_t *vals, mpc_t *roots, int *conjugates)
   /* The function works exactly as mpfrx_hecke, but assumes that the
      subproduct tree has been generated by a call to
      mpfrcx_subproducttree_from_roots; then vals is an array of real
      or complex-conjugate values of the polynomial rop in the correspond-
      ing elements of roots, both paired up according to the same
      conjugates. The function requires the complex-conjugate values of
      roots to fill in the leaves of a tree with linear real polynomials
      interpolating a pair of complex-conjugate values. */

{
   int i;
   int no = subproducts->node [subproducts->levels-1][0]->deg;
   int n = subproducts->no;
   mpfr_prec_t prec = mpc_get_prec (vals [0]);
   mpfrx_t *fac;
   mpc_t coeff;

   fac = (mpfrx_t *) malloc (n * sizeof (mpfrx_t));
   mpc_init2 (coeff, prec);
   n = 0;
   for (i = 0; i < no; i++) {
      if (conjugates [i] == i) {
         mpfrx_init (fac [n], 1, prec);
         mpfrx_set_deg (fac [n], 0);
         mpfrx_set_coeff (fac [n], 0, mpc_realref (vals [i]));
         n++;
      }
      else if (conjugates [i] > i) {
         mpfrx_init (fac [n], 2, prec);
         mpfrx_set_deg (fac [n], 1);
         mpfr_mul_2ui (fac [n]->coeff [1], mpc_realref (vals [i]), 1, MPFR_RNDN);
         mpc_conj (coeff, roots [i], MPC_RNDNN);
         mpc_mul (coeff, coeff, vals [i], MPC_RNDNN);
         mpfr_mul_si (fac [n]->coeff [0], mpc_realref (coeff), -2, MPFR_RNDN);
         n++;
      }
   }
   mpc_clear (coeff);

   mpfrx_hecke (rop, subproducts, fac);

   for (i = 0; i < n; i++)
      mpfrx_clear (fac [i]);
   free (fac);
}

/**************************************************************************/

void mpfrcx_product_and_hecke_from_roots (mpfrx_t *rop, mpc_t **vals,
   int *conjugates, int no_pols, int no_factors)
   /* The entries of val are expected to have the same precision. */

{
   int i, j;
   mpfr_prec_t prec = mpc_get_prec (vals [0][0]);
   int n;
   mpfrx_t **valx;
   mpc_t coeff;

   n = 0;
   for (i = 0; i < no_factors; i++)
      if (conjugates [i] >= i)
         n++;

   valx = (mpfrx_t **) malloc (no_pols * sizeof (mpfrx_t *));
   for (i = 0; i < no_pols; i++)
      valx [i] = (mpfrx_t *) malloc (n * sizeof (mpfrx_t));
   mpc_init2 (coeff, prec);
   n = 0;
   for (j = 0; j < no_factors; j++) {
      if (conjugates [j] == j) {
         mpfrx_init_set_linear (valx [0][n], mpc_realref (vals [0][j]));
         for (i = 1; i < no_pols; i++) {
            mpfrx_init (valx [i][n], 1, prec);
            mpfrx_set_deg (valx [i][n], 0);
            mpfrx_set_coeff (valx [i][n], 0, mpc_realref (vals [i][j]));
         }
         n++;
      }
      else if (conjugates [j] > j) {
         mpfrcx_init_set_quadratic (valx [0][n], vals [0][j]);
         for (i = 1; i < no_pols; i++) {
            mpfrx_init (valx [i][n], 2, prec);
            mpfrx_set_deg (valx [i][n], 1);
            mpfr_mul_2ui (valx [i][n]->coeff [1], mpc_realref (vals [i][j]),
               1, MPFR_RNDN);
            mpc_conj (coeff, vals [0][j], MPC_RNDNN);
            mpc_mul (coeff, coeff, vals [i][j], MPC_RNDNN);
            mpfr_mul_si (valx [i][n]->coeff [0], mpc_realref (coeff), -2,
               MPFR_RNDN);
         }
         n++;
      }
   }
   mpc_clear (coeff);

   mpfrx_product_and_hecke (rop, valx, no_pols, n);

   for (i = 0; i < no_pols; i++) {
      for (j = 0; j < n; j++)
         mpfrx_clear (valx [i][j]);
      free (valx [i]);
   }
   free (valx);
}

/**************************************************************************/
