/*
 * xpath.c: interface for XML Path Language implementation
 *
 * Reference: W3C Working Draft 5 July 1999
 *            http://www.w3.org/Style/XSL/Group/1999/07/xpath-19990705.html
 *
 * See COPYRIGHT for the status of this software
 *
 * Author: daniel@veillard.com
 */

#ifndef __XML_XPATH_H__
#define __XML_XPATH_H__

#include <libxml/tree.h>
#include <libxml/hash.h>

#ifdef __cplusplus
extern "C" {
#endif

typedef struct _xmlXPathContext xmlXPathContext;
typedef xmlXPathContext *xmlXPathContextPtr;
typedef struct _xmlXPathParserContext xmlXPathParserContext;
typedef xmlXPathParserContext *xmlXPathParserContextPtr;

/**
 * The set of XPath error codes
 */

typedef enum {
    XPATH_EXPRESSION_OK = 0,
    XPATH_NUMBER_ERROR,
    XPATH_UNFINISHED_LITERAL_ERROR,
    XPATH_START_LITERAL_ERROR,
    XPATH_VARIABLE_REF_ERROR,
    XPATH_UNDEF_VARIABLE_ERROR,
    XPATH_INVALID_PREDICATE_ERROR,
    XPATH_EXPR_ERROR,
    XPATH_UNCLOSED_ERROR,
    XPATH_UNKNOWN_FUNC_ERROR,
    XPATH_INVALID_OPERAND,
    XPATH_INVALID_TYPE,
    XPATH_INVALID_ARITY,
    XPATH_INVALID_CTXT_SIZE,
    XPATH_INVALID_CTXT_POSITION,
    XPATH_MEMORY_ERROR,
    XPTR_SYNTAX_ERROR,
    XPTR_RESOURCE_ERROR,
    XPTR_SUB_RESOURCE_ERROR,
    XPATH_UNDEF_PREFIX_ERROR,
    XPATH_ENCODING_ERROR,
    XPATH_INVALID_CHAR_ERROR
} xmlXPathError;

/*
 * A node-set (an unordered collection of nodes without duplicates) 
 */
typedef struct _xmlNodeSet xmlNodeSet;
typedef xmlNodeSet *xmlNodeSetPtr;
struct _xmlNodeSet {
    int nodeNr;			/* number of nodes in the set */
    int nodeMax;		/* size of the array as allocated */
    xmlNodePtr *nodeTab;	/* array of nodes in no particular order */
};

/*
 * An expression is evaluated to yield an object, which
 * has one of the following four basic types:
 *   - node-set
 *   - boolean
 *   - number
 *   - string
 *
 * @@ XPointer will add more types !
 */

typedef enum {
    XPATH_UNDEFINED = 0,
    XPATH_NODESET = 1,
    XPATH_BOOLEAN = 2,
    XPATH_NUMBER = 3,
    XPATH_STRING = 4,
    XPATH_POINT = 5,
    XPATH_RANGE = 6,
    XPATH_LOCATIONSET = 7,
    XPATH_USERS = 8,
    XPATH_XSLT_TREE = 9  /* An XSLT value tree, non modifiable */
} xmlXPathObjectType;

typedef struct _xmlXPathObject xmlXPathObject;
typedef xmlXPathObject *xmlXPathObjectPtr;
struct _xmlXPathObject {
    xmlXPathObjectType type;
    xmlNodeSetPtr nodesetval;
    int boolval;
    double floatval;
    xmlChar *stringval;
    void *user;
    int index;
    void *user2;
    int index2;
};

/*
 * A conversion function is associated to a type and used to cast
 * the new type to primitive values.
 */
typedef int (*xmlXPathConvertFunc) (xmlXPathObjectPtr obj, int type);

/*
 * Extra type: a name and a conversion function.
 */

typedef struct _xmlXPathType xmlXPathType;
typedef xmlXPathType *xmlXPathTypePtr;
struct _xmlXPathType {
    const xmlChar         *name;		/* the type name */
    xmlXPathConvertFunc func;		/* the conversion function */
};

/*
 * Extra variable: a name and a value.
 */

typedef struct _xmlXPathVariable xmlXPathVariable;
typedef xmlXPathVariable *xmlXPathVariablePtr;
struct _xmlXPathVariable {
    const xmlChar       *name;		/* the variable name */
    xmlXPathObjectPtr value;		/* the value */
};

/**
 * xmlXPathEvalFunc:
 * @ctxt: an XPath parser context
 * @nargs: the number of arguments passed to the function
 *
 * an XPath evaluation function, the parameters are on thei XPath context stack
 */

typedef void (*xmlXPathEvalFunc)(xmlXPathParserContextPtr ctxt,
	                         int nargs);

/*
 * Extra function: a name and a evaluation function.
 */

typedef struct _xmlXPathFunct xmlXPathFunct;
typedef xmlXPathFunct *xmlXPathFuncPtr;
struct _xmlXPathFunct {
    const xmlChar      *name;		/* the function name */
    xmlXPathEvalFunc func;		/* the evaluation function */
};

/*
 * An axis traversal function. To traverse an axis, the engine calls
 * the first time with cur == NULL and repeat until the function returns
 * NULL indicating the end of the axis traversal.
 */

typedef xmlXPathObjectPtr (*xmlXPathAxisFunc)	(xmlXPathParserContextPtr ctxt,
						 xmlXPathObjectPtr cur);

/*
 * Extra axis: a name and an axis function.
 */

typedef struct _xmlXPathAxis xmlXPathAxis;
typedef xmlXPathAxis *xmlXPathAxisPtr;
struct _xmlXPathAxis {
    const xmlChar      *name;		/* the axis name */
    xmlXPathAxisFunc func;		/* the search function */
};

/**
 * xmlXPathContext:
 *
 * Expression evaluation occurs with respect to a context.
 * he context consists of:
 *    - a node (the context node) 
 *    - a node list (the context node list) 
 *    - a set of variable bindings 
 *    - a function library 
 *    - the set of namespace declarations in scope for the expression 
 * Following the switch to hash tables, this need to be trimmed up at
 * the next binary incompatible release.
 */

struct _xmlXPathContext {
    xmlDocPtr doc;			/* The current document */
    xmlNodePtr node;			/* The current node */

    int nb_variables_unused;		/* unused (hash table) */
    int max_variables_unused;		/* unused (hash table) */
    xmlHashTablePtr varHash;		/* Hash table of defined variables */

    int nb_types;			/* number of defined types */
    int max_types;			/* max number of types */
    xmlXPathTypePtr types;		/* Array of defined types */

    int nb_funcs_unused;		/* unused (hash table) */
    int max_funcs_unused;		/* unused (hash table) */
    xmlHashTablePtr funcHash;		/* Hash table of defined funcs */

    int nb_axis;			/* number of defined axis */
    int max_axis;			/* max number of axis */
    xmlXPathAxisPtr axis;		/* Array of defined axis */

    /* the namespace nodes of the context node */
    xmlNsPtr *namespaces;		/* Array of namespaces */
    int nsNr;				/* number of namespace in scope */
    void *user;				/* function to free */

    /* extra variables */
    int contextSize;			/* the context size */
    int proximityPosition;		/* the proximity position */

    /* extra stuff for XPointer */
    int xptr;				/* it this an XPointer context */
    xmlNodePtr here;			/* for here() */
    xmlNodePtr origin;			/* for origin() */

    /* the set of namespace declarations in scope for the expression */
    xmlHashTablePtr nsHash;		/* The namespaces hash table */
    void *varLookupFunc;		/* variable lookup func */
    void *varLookupData;		/* variable lookup data */

    /* Possibility to link in an extra item */
    void *extra;                        /* needed for XSLT */

    /* The function name and URI when calling a function */
    const xmlChar *function;
    const xmlChar *functionURI;
};

/*
 * The structure of a compiled expression form is not public
 */

typedef struct _xmlXPathCompExpr xmlXPathCompExpr;
typedef xmlXPathCompExpr *xmlXPathCompExprPtr;

/**
 * xmlXPathParserContext:
 *
 * An XPath parser context, it contains pure parsing informations,
 * an xmlXPathContext, and the stack of objects.
 */
struct _xmlXPathParserContext {
    const xmlChar *cur;			/* the current char being parsed */
    const xmlChar *base;			/* the full expression */

    int error;				/* error code */

    xmlXPathContextPtr  context;	/* the evaluation context */
    xmlXPathObjectPtr     value;	/* the current value */
    int                 valueNr;	/* number of values stacked */
    int                valueMax;	/* max number of values stacked */
    xmlXPathObjectPtr *valueTab;	/* stack of values */

    xmlXPathCompExprPtr comp;		/* the precompiled expression */
    int xptr;				/* it this an XPointer expression */
    xmlNodePtr         ancestor;	/* used for walking preceding axis */
};

/**
 * xmlXPathFunction:
 *
 * An XPath function
 * The arguments (if any) are popped out of the context stack
 * and the result is pushed on the stack.
 */

typedef void (*xmlXPathFunction) (xmlXPathParserContextPtr ctxt, int nargs);

/************************************************************************
 *									*
 *			Public API					*
 *									*
 ************************************************************************/

/**
 * Objects and Nodesets handling
 */

/* These macros may later turn into functions */
#define xmlXPathNodeSetGetLength(ns) ((ns) ? (ns)->nodeNr : 0)
#define xmlXPathNodeSetItem(ns, index)				\
		((((ns) != NULL) && 				\
		  ((index) > 0) && ((index) <= (ns)->nodeNr)) ?	\
		 (ns)->nodeTab[(index)]				\
		 : NULL)


void		   xmlXPathFreeObject		(xmlXPathObjectPtr obj);
xmlNodeSetPtr	   xmlXPathNodeSetCreate	(xmlNodePtr val);
void		   xmlXPathFreeNodeSetList	(xmlXPathObjectPtr obj);
void		   xmlXPathFreeNodeSet		(xmlNodeSetPtr obj);
xmlXPathObjectPtr  xmlXPathObjectCopy		(xmlXPathObjectPtr val);
int		   xmlXPathCmpNodes		(xmlNodePtr node1,
						 xmlNodePtr node2);
/**
 * Conversion functions to basic types
 */
int		   xmlXPathCastNumberToBoolean	(double val);
int		   xmlXPathCastStringToBoolean	(const xmlChar * val);
int		   xmlXPathCastNodeSetToBoolean	(xmlNodeSetPtr ns);
int		   xmlXPathCastToBoolean	(xmlXPathObjectPtr val);

double		   xmlXPathCastBooleanToNumber	(int val);
double		   xmlXPathCastStringToNumber	(const xmlChar * val);
double		   xmlXPathCastNodeToNumber	(xmlNodePtr node);
double		   xmlXPathCastNodeSetToNumber	(xmlNodeSetPtr ns);
double		   xmlXPathCastToNumber		(xmlXPathObjectPtr val);

xmlChar *	   xmlXPathCastBooleanToString	(int val);
xmlChar *	   xmlXPathCastNumberToString	(double val);
xmlChar *	   xmlXPathCastNodeToString	(xmlNodePtr node);
xmlChar *	   xmlXPathCastNodeSetToString	(xmlNodeSetPtr ns);
xmlChar *	   xmlXPathCastToString		(xmlXPathObjectPtr val);

xmlXPathObjectPtr  xmlXPathConvertBoolean	(xmlXPathObjectPtr val);
xmlXPathObjectPtr  xmlXPathConvertNumber	(xmlXPathObjectPtr val);
xmlXPathObjectPtr  xmlXPathConvertString	(xmlXPathObjectPtr val);

/**
 * Context handling
 */
void		   xmlXPathInit			(void);
xmlXPathContextPtr xmlXPathNewContext		(xmlDocPtr doc);
void		   xmlXPathFreeContext		(xmlXPathContextPtr ctxt);

/**
 * Evaluation functions.
 */
xmlXPathObjectPtr  xmlXPathEval			(const xmlChar *str,
						 xmlXPathContextPtr ctxt);
xmlXPathObjectPtr  xmlXPathEvalXPtrExpr		(const xmlChar *str,
						 xmlXPathContextPtr ctxt);
xmlXPathObjectPtr  xmlXPathEvalExpression	(const xmlChar *str,
						 xmlXPathContextPtr ctxt);
int                xmlXPathEvalPredicate	(xmlXPathContextPtr ctxt,
						 xmlXPathObjectPtr res);
/**
 * Separate compilation/evaluation entry points
 */
xmlXPathCompExprPtr xmlXPathCompile		(const xmlChar *str);
xmlXPathObjectPtr   xmlXPathCompiledEval	(xmlXPathCompExprPtr comp,
						 xmlXPathContextPtr ctx);
void                xmlXPathFreeCompExpr	(xmlXPathCompExprPtr comp);
#ifdef __cplusplus
}
#endif
#endif /* ! __XML_XPATH_H__ */
