/*
 * Cogl
 *
 * A Low Level GPU Graphics and Utilities API
 *
 * Copyright (C) 2011 Intel Corporation.
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#pragma once

#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "cogl/cogl-renderer.h"

G_BEGIN_DECLS

/**
 * cogl_xlib_renderer_handle_event: (skip)
 * @renderer: a #CoglRenderer
 * @event: pointer to an XEvent structure
 *
 * This function processes a single event; it can be used to hook into
 * external event retrieval (for example that done by Clutter or
 * GDK).
 *
 * Return value: #CoglFilterReturn. %COGL_FILTER_REMOVE indicates that
 * Cogl has internally handled the event and the caller should do no
 * further processing. %COGL_FILTER_CONTINUE indicates that Cogl is
 * either not interested in the event, or has used the event to update
 * internal state without taking any exclusive action.
 */
COGL_EXPORT CoglFilterReturn
cogl_xlib_renderer_handle_event (CoglRenderer *renderer,
                                 XEvent *event);

/*
 * CoglXlibFilterFunc:
 * @event: pointer to an XEvent structure
 * @data: the data that was given when the filter was added
 *
 * A callback function that can be registered with
 * cogl_xlib_renderer_add_filter(). The function should return
 * %COGL_FILTER_REMOVE if it wants to prevent further processing or
 * %COGL_FILTER_CONTINUE otherwise.
 */
typedef CoglFilterReturn (* CoglXlibFilterFunc) (XEvent *event,
                                                 void *data);

/**
 * cogl_xlib_renderer_add_filter: (skip)
 * @renderer: a #CoglRenderer
 * @func: the callback function
 * @data: user data passed to @func when called
 *
 * Adds a callback function that will receive all native events. The
 * function can stop further processing of the event by return
 * %COGL_FILTER_REMOVE.
 */
COGL_EXPORT void
cogl_xlib_renderer_add_filter (CoglRenderer *renderer,
                               CoglXlibFilterFunc func,
                               void *data);

/**
 * cogl_xlib_renderer_remove_filter: (skip)
 * @renderer: a #CoglRenderer
 * @func: the callback function
 * @data: user data given when the callback was installed
 *
 * Removes a callback that was previously added with
 * cogl_xlib_renderer_add_filter().
 */
COGL_EXPORT void
cogl_xlib_renderer_remove_filter (CoglRenderer *renderer,
                                  CoglXlibFilterFunc func,
                                  void *data);

/**
 * cogl_xlib_renderer_set_foreign_display: (skip)
 * @renderer: a #CoglRenderer
 *
 * Sets a foreign Xlib display that Cogl will use for and Xlib based winsys
 * backend.
 *
 * Note that calling this function will automatically disable Cogl's
 * event retrieval. Cogl still needs to see all of the X events so the
 * application should also use cogl_xlib_renderer_handle_event() if it
 * uses this function.
 */
COGL_EXPORT void
cogl_xlib_renderer_set_foreign_display (CoglRenderer *renderer,
                                        Display *display);

/**
 * cogl_xlib_renderer_request_reset_on_video_memory_purge: (skip)
 * @renderer: a #CoglRenderer
 * @enable: The new value
 *
 * Sets whether Cogl should make use of the
 * NV_robustness_video_memory_purge extension, if exposed by the
 * driver, by initializing the GLX context appropriately.
 *
 * The extension is only useful when running on certain versions of
 * the NVIDIA driver. Quoting from the spec:
 *
 * "The NVIDIA OpenGL driver architecture on Linux has a limitation:
 *  resources located in video memory are not persistent across certain
 *  events. VT switches, suspend/resume events, and mode switching
 *  events may erase the contents of video memory. Any resource that
 *  is located exclusively in video memory, such as framebuffer objects
 *  (FBOs), will be lost."
 *
 * "This extension provides a way for applications to discover when video
 *  memory content has been lost, so that the application can re-populate
 *  the video memory content as necessary."
 *
 * "Any driver that exposes this extension is a driver that considers
 *  video memory to be volatile. Once the driver stack has been
 *  improved, the extension will no longer be exposed."
 *
 * cogl_get_graphics_reset_status() needs to be called at least once
 * every frame to find out if video memory was purged.
 *
 * Note that this doesn't cause Cogl to enable robust buffer access
 * but other context reset errors may still happen and be reported via
 * cogl_get_graphics_reset_status() if external factors cause the
 * driver to trigger them.
 *
 * This defaults to %FALSE and is effective only if called before
 * cogl_display_setup() .
 */
COGL_EXPORT void
cogl_xlib_renderer_request_reset_on_video_memory_purge (CoglRenderer *renderer,
                                                        gboolean enable);
G_END_DECLS
