/*
 * Electric(tm) VLSI Design System
 *
 * File: iolefo.c
 * Input/output aid: LEF output
 * Written by: Steven M. Rubin, Static Free Software
 *
 * Copyright (c) 2000 Static Free Software.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Static Free Software
 * 4119 Alpine Road
 * Portola Valley, California 94028
 * info@staticfreesoft.com
 */

/*
 * Note that this reader was built by examining LEF files and reverse-engineering them.
 * It does not claim to be compliant with the LEF specification, but it also does not
 * claim to define a new specification.  It is merely incomplete.
 */

#include "config.h"
#if IOLEF

#include "global.h"
#include "eio.h"
#include "efunction.h"

void io_lefoutfacet(FILE *out, NODEPROTO *facet, INTSML top, XARRAY trans);
char *io_lefoutlayername(TECHNOLOGY *tech, INTSML layer);
void io_lefoutspread(FILE *out, NODEPROTO *facet, NETWORK *net, NODEINST *ignore);
void io_lefwritepoly(FILE *out, POLYGON *poly, XARRAY trans, TECHNOLOGY *tech);

char io_lefoutcurlayer[200];

INTSML io_writeleflibrary(LIBRARY *lib)
{
	char file[100], *truename;
	REGISTER char *name;
	REGISTER INTSML i, tot;
	REGISTER float xs, ys;
	REGISTER NODEPROTO *np;
	REGISTER NODEINST *rni, *ni;
	REGISTER ARCINST *ai;
	XARRAY trans, xform, temp;
	REGISTER PORTPROTO *pp, *rpp;
	static POLYGON *poly = NOPOLYGON;

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocstaticpolygon(4, io_aid->cluster);

	/* create the proper disk file for the LEF */
	np = lib->curnodeproto;
	if (np == NONODEPROTO)
	{
		ttyputerr("Must be editing a facet to generate LEF output");
		return(1);
	}
	(void)strcpy(file, np->cell->cellname);
	(void)strcat(file, ".lef");
	name = truepath(file);
	io_fileout = xcreate(name, FILETYPELEF, "LEF File", &truename);
	if (io_fileout == NULL)
	{
		if (truename != 0) ttyputerr("Cannot write %s", truename);
		return(1);
	}

	/* write header information */
	xprintf(io_fileout, "# %s\n\n", timetostring(getcurrenttime()));
	xprintf(io_fileout, "NAMESCASESENSITIVE ON ;\n");
	xprintf(io_fileout, "UNITS\n");
	xprintf(io_fileout, "  DATABASE MICRONS 1 ;\n");
	xprintf(io_fileout, "END UNITS\n\n");

	/* write layer information */
	for(i=0; i<8; i++)
	{
		xprintf(io_fileout, "LAYER METAL%ld\n", i+1);
		xprintf(io_fileout, "  TYPE ROUTING ;\n");
		xprintf(io_fileout, "END METAL%ld\n\n", i+1);
	}
	xprintf(io_fileout, "LAYER CONT\n");
	xprintf(io_fileout, "  TYPE CUT ;\n");
	xprintf(io_fileout, "END CONT\n\n");
	for(i=0; i<3; i++)
	{
		xprintf(io_fileout, "LAYER VIA%ld%ld\n", i+1, i+2);
		xprintf(io_fileout, "  TYPE CUT ;\n");
		xprintf(io_fileout, "END VIA%ld%l\n\n", i+1, i+2);
	}
	for(i=0; i<3; i++)
	{
		xprintf(io_fileout, "LAYER POLY%ld\n", i+1);
		xprintf(io_fileout, "  TYPE MASTERSLICE ;\n");
		xprintf(io_fileout, "END POLY%l\n\n", i+1);
	}
	xprintf(io_fileout, "LAYER PDIFF\n");
	xprintf(io_fileout, "  TYPE MASTERSLICE ;\n");
	xprintf(io_fileout, "END PDIFF\n\n");
	xprintf(io_fileout, "LAYER NDIFF\n");
	xprintf(io_fileout, "  TYPE MASTERSLICE ;\n");
	xprintf(io_fileout, "END NDIFF\n\n");

	/* write main facet header */
	xprintf(io_fileout, "MACRO %s\n", np->cell->cellname);
	xprintf(io_fileout, "  FOREIGN %s ;\n", np->cell->cellname);
	xs = scaletodispunit(np->highx-np->lowx, DISPUNITMIC);
	ys = scaletodispunit(np->highy-np->lowy, DISPUNITMIC);
	xprintf(io_fileout, "  SIZE %g BY %g ;\n", xs, ys);
	xprintf(io_fileout, "  SITE %s ;\n", np->cell->cellname);

	/* write all of the metal geometry and ports */
	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst) ni->temp1 = 0;
	for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst) ai->temp1 = 0;
	for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
	{
		if (pp != np->firstportproto) xprintf(io_fileout, "\n");
		xprintf(io_fileout, "  PIN %s\n", pp->protoname);

		transid(trans);
		rpp = pp->subportproto;
		rni = pp->subnodeinst;
		makerot(rni, xform);   transmult(xform, trans, temp);  transcpy(temp, trans);
		while (rni->proto->primindex == 0)
		{
			rni = rpp->subnodeinst;
			rpp = rpp->subportproto;
			maketrans(rni, xform);  transmult(xform, trans, temp);
			makerot(rni, xform);    transmult(xform, temp, trans);
		}
		xprintf(io_fileout, "    PORT\n");
		io_lefoutcurlayer[0] = 0;
		tot = nodeEpolys(rni);
		for(i=0; i<tot; i++)
		{
			shapeEnodepoly(rni, i, poly);
			if (poly->portproto != rpp) continue;
			io_lefwritepoly(io_fileout, poly, trans, rni->proto->tech);
		}
		io_lefoutspread(io_fileout, np, pp->network, ni);
		xprintf(io_fileout, "    END\n");
		if ((pp->userbits&STATEBITS) == PWRPORT)
			xprintf(io_fileout, "    USE POWER ;\n");
		if ((pp->userbits&STATEBITS) == GNDPORT)
			xprintf(io_fileout, "    USE GROUND ;\n");
		xprintf(io_fileout, "  END %s\n", pp->protoname);
	}

	/* write the obstructions (all of the metal) */
	xprintf(io_fileout, "\n  OBS\n");
	io_lefoutcurlayer[0] = 0;
	io_lefoutfacet(io_fileout, np, 1, el_matid);
	xprintf(io_fileout, "  END\n\n");

	/* clean up */
	xprintf(io_fileout, "END %s\n\n", np->cell->cellname);
	xprintf(io_fileout, "END LIBRARY\n");
	xclose(io_fileout);

	/* tell the user that the file is written */
	ttyputmsg("%s written", truename);
	return(0);
}

/*
 * Routine to write all geometry in facet "facet" that is on network "net"
 * to file "out".  Does not write node "ignore".
 */
void io_lefoutspread(FILE *out, NODEPROTO *facet, NETWORK *net, NODEINST *ignore)
{
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER PORTARCINST *pi;
	REGISTER INTSML i, tot, fun;
	char *extra;
	XARRAY trans;
	static POLYGON *poly = NOPOLYGON;

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocstaticpolygon(4, io_aid->cluster);
	for(ni = facet->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		if (ni->proto->primindex == 0) continue;
		if (ni == ignore) continue;
		fun = nodefunction(ni, &extra);
		if (fun != NPPIN && fun != NPCONTACT && fun != NPNODE && fun != NPCONNECT)
			continue;
		for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
			if (pi->conarcinst->network != net) break;
		if (pi != NOPORTARCINST) continue;

		/* write all layers on this node */
		ni->temp1 = 1;
		makerot(ni, trans);
		tot = nodepolys(ni);
		for(i=0; i<tot; i++)
		{
			shapenodepoly(ni, i, poly);
			io_lefwritepoly(out, poly, trans, ni->proto->tech);
		}
	}

	/* write metal layers for all arcs */
	for(ai = facet->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
	{
		if (ai->network != net) continue;
		ai->temp1 = 1;
		tot = arcpolys(ai);
		for(i=0; i<tot; i++)
		{
			shapearcpoly(ai, i, poly);
			io_lefwritepoly(out, poly, trans, ai->proto->tech);
		}
	}
}

/*
 * Routine to write all geometry in facet "facet" to file "out".  The
 * hierarchy is flattened.  The current facet is transformed by "trans".
 */
void io_lefoutfacet(FILE *out, NODEPROTO *facet, INTSML top, XARRAY trans)
{
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER INTSML i, tot;
	XARRAY localtran, subrot, localrot, thisrot;
	static POLYGON *poly = NOPOLYGON;

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocstaticpolygon(4, io_aid->cluster);
	for(ni = facet->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		if (top != 0 && ni->temp1 != 0) continue;
		makerot(ni, localrot);   transmult(localrot, trans, thisrot);
		if (ni->proto->primindex == 0)
		{
			/* subfacet: recurse */
			maketrans(ni, localtran);
			transmult(localtran, thisrot, subrot);
			io_lefoutfacet(out, ni->proto, 0, subrot);
		} else
		{
			/* primitive: find the metal layers */
			tot = nodepolys(ni);
			for(i=0; i<tot; i++)
			{
				shapenodepoly(ni, i, poly);
				io_lefwritepoly(out, poly, trans, ni->proto->tech);
			}
		}
	}

	/* write metal layers for all arcs */
	for(ai = facet->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
	{
		if (top != 0 && ai->temp1 != 0) continue;
		tot = arcpolys(ai);
		for(i=0; i<tot; i++)
		{
			shapearcpoly(ai, i, poly);
			io_lefwritepoly(out, poly, trans, ai->proto->tech);
		}
	}
}

/*
 * Routine to write polygon "poly" from technology "tech", transformed by "trans",
 * to "out".
 */
void io_lefwritepoly(FILE *out, POLYGON *poly, XARRAY trans, TECHNOLOGY *tech)
{
	INTBIG lx, hx, ly, hy;
	REGISTER float flx, fhx, fly, fhy;
	REGISTER char *layername;

	layername = io_lefoutlayername(tech, poly->layer);
	if (*layername == 0) return;
	xformpoly(poly, trans);
	if (isbox(poly, &lx, &hx, &ly, &hy) == 0) return;
	flx = scaletodispunit(lx, DISPUNITMIC);
	fhx = scaletodispunit(hx, DISPUNITMIC);
	fly = scaletodispunit(ly, DISPUNITMIC);
	fhy = scaletodispunit(hy, DISPUNITMIC);
	if (strcmp(layername, io_lefoutcurlayer) != 0)
	{
		xprintf(out, "    LAYER %s ;\n", layername);
		strcpy(io_lefoutcurlayer, layername);
	}
	xprintf(out, "    RECT %g %g %g %g ;\n", flx, fly, fhx, fhy);
}

char *io_lefoutlayername(TECHNOLOGY *tech, INTSML layer)
{
	REGISTER INTBIG fun;

	fun = layerfunction(tech, layer);
	switch (fun&LFTYPE)
	{
		case LFMETAL1:   return("METAL1");
		case LFMETAL2:   return("METAL2");
		case LFMETAL3:   return("METAL3");
		case LFMETAL4:   return("METAL4");
		case LFMETAL5:   return("METAL5");
		case LFMETAL6:   return("METAL6");
		case LFMETAL7:   return("METAL7");
		case LFMETAL8:   return("METAL8");
		case LFCONTACT1: return("CONT");
		case LFCONTACT2: return("VIA12");
		case LFCONTACT3: return("VIA23");
		case LFCONTACT4: return("VIA34");
		case LFCONTACT5: return("VIA45");
		case LFCONTACT6: return("VIA56");
		case LFPOLY1:    return("POLY1");
		case LFPOLY2:    return("POLY2");
		case LFPOLY3:    return("POLY3");
		case LFDIFF:
			if ((fun&LFPTYPE) != 0) return("PDIFF");
			if ((fun&LFNTYPE) != 0) return("NDIFF");
			return("DIFF");
	}
	return("");
}

#endif  /* IOLEF - at top */
