/*
 This file is part of GNU Taler
 (C) 2021-2023 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * @author Nic Eigel
 * @author Sebastian Javier Marchano (sebasjm)
 */

import {
  ErrorType,
  HttpError,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { h, VNode } from "preact";
import { useState } from "preact/hooks";
import { Loading } from "../../components/exception/loading.js";
import { NotificationCard } from "../../components/menu/index.js";
import { AuditorBackend, WithId } from "../../declaration.js";
import { Notification } from "../../utils/types.js";
import { CardTable } from "./Table.js";
import { HttpStatusCode } from "@gnu-taler/taler-util";
import { EntityDataContextProvider, useEntityContext } from "../../context/entity.js";
import { getEntityList, useEntityAPI } from "../../hooks/entity.js";
import { useMemo } from "preact/hooks";
import { ConfirmModal, DeleteModal } from "../../components/modal/index.js";
import { route } from "preact-router";
import { Paths } from "../../InstanceRoutes.js";


interface Props {
  onNotFound: () => VNode;
  onLoadError: (e: HttpError<AuditorBackend.ErrorDetail>) => VNode;
}

export default function DefaultList({
                                      onLoadError,
                                      onNotFound,
                                    }: Props): VNode {
  const { endpoint, entity } = useEntityContext();
  const result = getEntityList({ endpoint, entity });
  const { updateEntity } = useEntityAPI();
  const [suppressing, setSuppressing] =
    useState<typeof entity & WithId | null>(null);
  const [notif, setNotif] = useState<Notification | undefined>(undefined);
  const { i18n } = useTranslationContext();

  if (result.loading) return <Loading />;
  if (!result.ok) {
    if (
      result.type === ErrorType.CLIENT &&
      result.status === HttpStatusCode.Unauthorized
    )
      return onNotFound();
    return onLoadError(result);
  }

  let data = result.data;
  const value = useMemo(
    () => ({ data }),
    [data],
  );

  function onReturn(): void {
    route(Paths.detail_view);
  }

  return (

    <section class="section is-main-section">
      <button
        class="button is-fullwidth"
        onClick={onReturn}
      >Back
      </button><br />

      <NotificationCard notification={notif} />

      <EntityDataContextProvider value={value}>
        <CardTable
          onSuppress={(e: typeof entity & WithId) =>
            setSuppressing(e)
          }
        />
      </EntityDataContextProvider>

      {suppressing && (
        <ConfirmModal
          label={`Suppress row`}
          description={`Suppress the row`}
          danger
          active
          onCancel={() => setSuppressing(null)}
          onConfirm={async (): Promise<void> => {
            try {
              await updateEntity(suppressing);
              setNotif({
                message: i18n.str`Entity row  with id: ${suppressing} has been suppressed`,
                type: "SUCCESS",
              });
            } catch (error) {
              setNotif({
                message: i18n.str`Failed to suppress row`,
                type: "ERROR",
                description: error instanceof Error ? error.message : undefined,
              });
            }
            setSuppressing(null);
          }}
        >
          <p class="warning">
            Suppressing a row <b>cannot be undone</b> in this GUI.
          </p>
        </ConfirmModal>
      )}
    </section>
  );
}
