/*******************************************************************
 * This file is part of the Emulex Linux Device Driver for         *
 * Enterprise Fibre Channel Host Bus Adapters.                     *
 * Refer to the README file included with this package for         *
 * driver version and adapter support.                             *
 * Copyright (C) 2003 Emulex Corporation.                          *
 * www.emulex.com                                                  *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the   *
 * GNU General Public License for more details, a copy of which    *
 * can be found in the file COPYING included with this package.    *
 *******************************************************************/

#ifndef _H_FC
#define _H_FC

/* Open Source defines */
#define DFC_SUBSYSTEM    1        /* Include dfc subsystem */

#include "fcdiag.h"
#include "fcdds.h"

#define LONGW_ALIGN       2             /* longword align for xmalloc */
#define FC_MAX_SEQUENCE   65536         /* maximum fc sequence size */
#define FC_MIN_SEQUENCE   0             /* minimum fc sequence size */
#define FC_MAX_IP_VECS    16            /* Max scatter list for mapping */
#define RING_TMO_DFT      30            /* default cmd timeout for IOCB rings */
#define MBOX_TMO_DFT      30            /* dft mailbox timeout for mbox cmds */
#define FC_CAP_AUTOSENSE  0x0400        /* SCSI capability for autosense */
#define FC_MAX_HOLD_RSCN  32            /* max number of deferred RSCNs */
#define FC_MAX_NS_RSP     65536         /* max size NameServer rsp */

/* Definitions for Binding Entry Type for fc_parse_binding_entry()  */
#define FC_BIND_WW_NN_PN   0
#define FC_BIND_DID        1

#define FC_DMX_ID       0x100
#define FL_DMX_ID       0x101

/*
 * Debug printf event ids.
 */

/* Check if WWN is 0 */
#define isWWNzero(wwn) ((wwn.nameType == 0) && (wwn.IEEE[0] == 0) && (wwn.IEEE[1] == 0) && (wwn.IEEE[2] == 0) && (wwn.IEEE[3] == 0) && (wwn.IEEE[4] == 0) && (wwn.IEEE[5] == 0))

#define ERRID_NOTICE            0x100
#define ERRID_ERROR             0x200
#define ERRID_PANIC             0x400
#define ERRID_MASK              0xf00

#define ERRID_VERBOSE           0x10ff

/* These are verbose logging masks and debug printf masks */
#define DBG_ELS                 0x1     /* ELS events */
#define DBG_DISCOVERY           0x2     /* Link discovery events */
#define DBG_MBOX                0x4     /* Mailbox events */
#define DBG_INIT                0x8     /* Initialization events */
#define DBG_LINK_EVENT          0x10    /* link events */
#define DBG_IP                  0x20    /* IP traffic history */
#define DBG_FCP                 0x40    /* FCP traffic history */
#define DBG_NODE                0x80    /* Node Table events */
#define DBG_CHK_COND        0x1000  /* FCP Check condition flag */

/* These are debug printf masks */
#define DBG_XRI                 0x1000  /* Exchange events */
#define DBG_IP_DATA             0x2000  /* IP traffic history */
#define DBG_INTR                0x4000  /* Interrupts */
#define DBG_IOCB_RSP            0x8000  /* IOCB Response ring events */
#define DBG_IOCB_RSP_DATA       0x10000 /* IOCB Response ring events */
#define DBG_IOCB_CMD            0x20000 /* IOCB Command ring events */
#define DBG_IOCB_CMD_DATA       0x40000 /* IOCB Command ring events */
#define DBG_FCP_DATA            0x100000/* FCP traffic history */
#define DBG_ERROR               0x800000/* ERROR events */


/* 
 * These definitions define SYNTAX errors that occur during the parsing
 * of binding config lines.
 */
#define FC_SYNTAX_OK                      0
#define FC_SYNTAX_OK_BUT_NOT_THIS_BRD     1
#define FC_SYNTAX_ERR_ASC_CONVERT         2
#define FC_SYNTAX_ERR_EXP_COLON           3
#define FC_SYNTAX_ERR_EXP_LPFC            4
#define FC_SYNTAX_ERR_INV_LPFC_NUM        5
#define FC_SYNTAX_ERR_EXP_T               6
#define FC_SYNTAX_ERR_INV_TARGET_NUM      7
#define FC_SYNTAX_ERR_EXP_D               8
#define FC_SYNTAX_ERR_INV_DEVICE_NUM      9
#define FC_SYNTAX_ERR_EXP_NULL_TERM      13

/*****************************************************************************/
/*                      device states                                        */
/*****************************************************************************/

#define CLOSED          0               /* initial device state */
#define DEAD            1               /* fatal hardware error encountered */
#define LIMBO           2               /* error recovery period */
#define OPEN_PENDING    3               /* open initiated */
#define OPENED          4               /* opened successfully, functioning */
#define CLOSE_PENDING   5               /* close initiated */

#define NORMAL_OPEN     0x0               /* opened in normal mode */
#define DIAG_OPEN       0x1               /* opened in diagnostics mode */

/*****************************************************************************/
/*      This is the board information structure for the fc device            */
/*****************************************************************************/

struct fc_q {
   uchar        *q_first;       /* queue first element */
   uchar        *q_last;        /* queue last element */
   ushort       q_cnt;          /* current length of queue */
   ushort       q_max;          /* max length queue can get */
};
typedef struct fc_q Q;

typedef struct fclink {
   struct fclink *_f;
   struct fclink *_b;
} FCLINK;

/*
*** fc_enque - enqueue element 'x' after element 'p' in
*** a queue without protection for critical sections.
*/
#define fc_enque(x,p)   {(((FCLINK *)x)->_f     = ((FCLINK *)p)->_f,   \
      ((FCLINK *)x)->_b     = ((FCLINK *)p),    \
      ((FCLINK *)p)->_f->_b = ((FCLINK *)x),    \
      ((FCLINK *)p)->_f     = ((FCLINK *)x));}

/*
*** fc_deque - dequeue element 'x' (the user must make
*** sure its not the queue header
*/
#define fc_deque(x)     {(((FCLINK *)x)->_b->_f = ((FCLINK *)x)->_f,   \
      ((FCLINK *)x)->_f->_b = ((FCLINK *)x)->_b,    \
      ((FCLINK *)x)->_b     = 0,     \
      ((FCLINK *)x)->_f     = 0);}

/* This structure is used when allocating a buffer pool.
 */
typedef struct mbuf_info {
   int  size;   /* Specifies the number of bytes to allocate. */
   int  align;  /* The desired address boundary. */

   int  flags;
#define FC_MBUF_DMA        0x1 /* blocks are for DMA */ 
#define FC_MBUF_PHYSONLY   0x2 /* For malloc - map a given virtual address
                                * to physical (skip the malloc). For free - 
                                * just unmap the given physical address
                                * (skip the free).
                                */ 
#define FC_MBUF_IOCTL      0x4 /* called from dfc_ioctl */
#define FC_MBUF_UNLOCK     0x8 /* called with driver unlocked */
   void  * virt; /* specifies the virtual buffer pointer */
   void  * phys; /* specifies the physical buffer pointer */
   ulong * data_handle;
   ulong * dma_handle;
} MBUF_INFO;


struct fc_match {
   uchar         * fc_mptr;
   uchar         * virt;        /* virtual address ptr */
   uchar         * phys;        /* mapped address */
   ulong         * data_handle;
   ulong         * dma_handle;
};
typedef struct fc_match MATCHMAP;

struct dfc_match {
   MATCHMAP        dfc;
   uint32          dfc_size;
   int             dfc_flag;
};
typedef struct dfc_match DMATCHMAP;

/* Kernel level Event structure */
struct fcEvent {
   uint32 evt_handle;
   uint32 evt_mask;
   uint32 evt_type;
   uint32 evt_data0;
   ushort evt_sleep;
   ushort evt_flags;
   void  *evt_next;
   void  *evt_data1;
   void  *evt_data2;
};
typedef struct fcEvent fcEvent;

/* Define for e_mode */
#define E_SLEEPING_MODE     0x0001

/* Define for e_flag */
#define E_GET_EVENT_ACTIVE  0x0001

/* Kernel level Event Header */
struct fcEvent_header {
  uint32      e_handle;
  uint32      e_mask;
  ushort      e_mode;
  ushort      e_flag;
  fcEvent   * e_head;
  fcEvent   * e_tail;
  void      * e_next_header;
/* Add something here */
};
typedef struct fcEvent_header fcEvent_header;

/* Structures using for clock / timeout handling */
typedef struct fcclock {
   struct fcclock *cl_fw;  /* forward linkage */
   union {
      struct {
         ushort cl_soft_arg;
         ushort cl_soft_cmd;
      } c1;
      struct fcclock *cl_bw;  /* backward linkage */
   } un;
   uint32 cl_tix;        /* differential number of clock ticks */
   void   (*cl_func)(void *, void *, void *);
   void   * cl_p_dev_ctl;
   void   * cl_arg1;       /* argument 1 to function */
   void   * cl_arg2;       /* argument 2 to function */
} FCCLOCK;

#define cl_bw         un.cl_bw

typedef struct clkhdr {
   FCCLOCK *cl_f;
   FCCLOCK * cl_b;
   uint32 count;             /* number of clock blocks in list */
} CLKHDR;

#define FC_NUM_GLBL_CLK 4  /* number of global clock blocks */

typedef struct fcclock_info {
   CLKHDR fc_clkhdr;       /* fc_clock queue head */
   uint32 ticks;           /* elapsed time since initialization */
   uint32 Tmr_ct;          /* Timer expired count */
   uint32 timestamp[2];    /* SMT 64 bit timestamp */
   void   * clktimer;      /* used for scheduling clock routine */
   Simple_lock  clk_slock; /* clock routine lock */
   FCCLOCK clk_block[FC_NUM_GLBL_CLK];   /* global clock blocks */
} FCCLOCK_INFO;


/* Structure used to access adapter rings */
struct fc_ring {
   IOCBQ        * fc_iocbhd;    /* ptr to head iocb rsp list for ring */
   IOCBQ        * fc_iocbtl;    /* ptr to tail iocb rsp list for ring */
   uchar        fc_numCiocb;    /* number of command iocb's per ring */
   uchar        fc_numRiocb;    /* number of rsp iocb's per ring */
   uchar        fc_rspidx;      /* current index in response ring */
   uchar        fc_cmdidx;      /* current index in command ring */
   uchar        fc_ringno;      /* ring number */
   uchar        fc_xmitstate;   /* state needed for xmit */
   void         * fc_cmdringaddr; /* virtual offset for cmd rings */
   void         * fc_rspringaddr; /* virtual offset for rsp rings */
   ushort       fc_iotag;       /* used to identify I/Os */

   ushort       fc_missbufcnt;  /* buf cnt we need to repost */
   ushort       fc_wdt_inited;  /* timer is inited */
   ushort       fc_bufcnt;      /* cnt of buffers posted */
   uchar        * fc_mpon;      /* index ptr for match structure */
   uchar        * fc_mpoff;     /* index ptr for match structure */
   uchar        * fc_binfo;     /* ptr to FC_BRD_INFO for ring */
   Q            fc_tx;          /* iocb command queue */
   Q            fc_txp;         /* iocb pending queue */
   FCCLOCK      * fc_wdt;       /* timer for ring activity */
   int          fc_ringtmo;     /* timer timeout value */
};
typedef struct fc_ring RING;

/* Defines for nlp_state (uchar) */
#define NLP_UNUSED      0       /* unused NL_PORT entry */
#define NLP_LIMBO       0x1     /* entry needs to hang around for wwpn / sid */
#define NLP_LOGOUT      0x2     /* NL_PORT is not logged in - entry is cached */
#define NLP_PLOGI       0x3     /* PLOGI was sent to NL_PORT */
#define NLP_LOGIN       0x4     /* NL_PORT is logged in / login REG_LOGINed */
#define NLP_PRLI        0x5     /* PRLI was sent to NL_PORT */
#define NLP_ALLOC       0x6     /* NL_PORT is  ready to initiate adapter I/O */
#define NLP_SEED        0x7     /* seed scsi id bind in table */

/* Defines for nlp_flag (uint32) */
#define NLP_RPI_XRI        0x1          /* creating xri for entry */
#define NLP_REQ_SND        0x2          /* sent ELS request for this entry */
#define NLP_RM_ENTRY       0x4          /* Remove this entry */
#define NLP_FARP_SND       0x8          /* sent FARP request for this entry */
#define NLP_NS_NODE        0x10         /* Authenticated entry by NameServer */
#define NLP_NODEV_TMO      0x20         /* nodev timeout is running for node */
#define NLP_REG_INP        0x40         /* Reglogin in progress for node */
#define NLP_UNREG_LOGO     0x80         /* Perform LOGO after unreglogin */
#define NLP_RCV_PLOGI      0x100        /* Rcv'ed PLOGI from remote system */
#define NLP_MAPPED         0x200        /* Node is now mapped */
#define NLP_UNMAPPED       0x400        /* Node is now unmapped */
#define NLP_BIND           0x800        /* Node is now bound */
#define NLP_LIST_MASK      0xe00        /* mask to see what list node is on */
#define NLP_SND_PLOGI      0x1000       /* Flg to indicate send PLOGI */
#define NLP_REQ_SND_PRLI   0x2000       /* Send PRLI ELS command */
#define NLP_REQ_SND_ADISC  0x2000       /* Send ADISC ELS command */
#define NLP_REQ_SND_PDISC  0x2000       /* Send PDISC ELS command */
#define NLP_NS_REMOVED     0x4000       /* Node removed from NameServer */

/* Defines for nlp_action (uchar) */
#define NLP_DO_ADDR_AUTH   0x1          /* Authenticating addr for entry */
#define NLP_DO_DISC_START  0x2          /* start discovery on this entry */
#define NLP_DO_RSCN        0x4          /* Authenticate entry for by RSCN */
#define NLP_DO_RNID        0x8          /* Authenticate entry for by RSCN */
#define NLP_DO_SCSICMD     0x10         /* Authenticate entry for by RSCN */
#define NLP_DO_CT_USR      0x20         /* Authenticate entry for by RSCN */
#define NLP_DO_CT_DRVR     0x40         /* Authenticate entry for by RSCN */

/* Defines for nlp_type (uchar) */
#define NLP_FABRIC         0x1          /* this entry represents the Fabric */
#define NLP_FCP_TARGET     0x2          /* this entry is an FCP target */
#define NLP_IP_NODE        0x4          /* this entry is an IP node */
#define NLP_SEED_WWPN      0x10         /* Entry scsi id is seeded for WWPN */
#define NLP_SEED_WWNN      0x20         /* Entry scsi id is seeded for WWNN */
#define NLP_SEED_DID       0x40         /* Entry scsi id is seeded for DID */
#define NLP_SEED_MASK      0x70         /* mask for seeded flags */
#define NLP_AUTOMAP        0x80         /* Entry was automap'ed */

/* Defines for list searchs */
#define NLP_SEARCH_MAPPED    0x1        /* search mapped */
#define NLP_SEARCH_UNMAPPED  0x2        /* search unmapped */
#define NLP_SEARCH_BIND      0x4        /* search bind */
#define NLP_SEARCH_ALL       0x7        /* search all lists */

/* Defines for nlp_fcp_info */
#define NLP_FCP_2_DEVICE   0x10         /* FCP-2 device */

struct nlp_nodeList {            /* NOTE: any changes to this structure
                                  * must be dup'ed in fcdds.h, cnode_t.
                                  */
   void     * nlp_listp_next;    /* Node table ptr bind / map / unmap list */
   void     * nlp_listp_prev;    /* Node table ptr bind / map / unmap list */
   uchar     nlp_state;          /* state transition indicator */
   uchar     nlp_action;         /* Action being performed on node */
   uchar     nlp_type;           /* node type identifier */
   uchar     nlp_alpa;           /* SCSI device AL_PA */
   ushort    nlp_Rpi;            /* login id returned by REG_LOGIN */
   ushort    nlp_Xri;            /* output exchange id for RPI */
   ushort    capabilities;
   ushort    sync;
   uint32    target_scsi_options;
   uint32    nlp_flag;           /* entry  flags */
   uint32    nlp_DID;            /* fibre channel D_ID of entry */
   uint32    nlp_time;           /* timestamp */
   uint32    nlp_oldDID;
   NAME_TYPE    nlp_portname;    /* port name */
   NAME_TYPE    nlp_nodename;    /* node name */
   struct {                      /* device id - for FCP */
      uchar  nlp_pan;            /* pseudo adapter number */
      uchar  nlp_sid;            /* scsi id */
      uchar  nlp_fcp_info;   /* Remote class info */
      uchar  nlp_ip_info;    /* Remote class info */
   } id;
   uchar    * nlp_bp;            /* save buffer ptr - for IP */
   uchar    * nlp_targetp;       /* Node table ptr for target */
};
typedef struct nlp_nodeList NODELIST;

/* For now stick fc_lun_t in here, 
 * should move to fc_os.h eventually.
 */
typedef uint32 fc_lun_t;

#define mapLun(di)      ((di)->lun_id)

#define NLP_MAXREQ      32      /* max num of outstanding NODELIST requests */
#define NLP_MAXSID      16      /* max number of scsi devices / adapter */
#define NLP_MAXPAN      32      /* max number of pseudo adapters */
#define PA_MASK         0x1f    /* mask devno to get pseudo adapter number */
#define DABS            5       /* convert devno to adapter number bit shift */
#define FC_MIN_QFULL    1   /* lowest we can decrement throttle
                                   to on qfull */

#define FC_SCSID(pan, sid)      ((uint32)((pan << 16) | sid)) /* For logging */

/* Max number of  fibre channel devices supported in network */
#define NLP_MAXRPI        512    /* firmware supports 512 rpis [0-511] */

#define FC_MAXLOOP       126    /* max devices supported on a single fc loop */
#define FC_MAX_MCAST     16     /* max number of multicast addresses */
#define MULTI_BIT_MASK   (0x01) /* Multicast Bit Mask */
#define FC_MAX_ADPTMSG   (8*28) /* max size of a msg from adapter */
#define FC_INIT_RING_BUF 12

struct fc_networkhdr {
   NAME_TYPE fc_destname;  /* destination port name */
   NAME_TYPE fc_srcname;   /* source port name */
};
typedef struct fc_networkhdr NETHDR;

#define MEM_NLP          0      /* memory segment to hold node list entries */
#define MEM_IOCB         1      /* memory segment to hold iocb commands */
#define MEM_CLOCK        1      /* memory segment to hold clock blocks */
#define MEM_MBOX         2      /* memory segment to hold mailbox cmds  */
#define MEM_BUF          3      /* memory segment to hold buffer data   */
#define MEM_BPL          3      /* and to hold buffer ptr lists - SLI2   */
#define FC_MAX_SEG      4 

#define MEM_SEG_MASK    0xff    /* mask used to mask off the priority bit */
#define MEM_PRI         0x100   /* Priority bit: set to exceed low water */

#define MIN_CLK_BLKS    256

struct fc_memseg {
   uchar        *fc_memptr;     /* ptr to memory blocks */
   uchar        *fc_endmemptr;  /* ptr to last memory block */
   uchar        *fc_memhi;      /* highest address in pool */
   uchar        *fc_memlo;      /* lowest address in pool */
   ushort       fc_memsize;     /* size of memory blocks */
   ushort       fc_numblks;     /* number of memory blocks */
   ushort       fc_free;        /* number of free memory blocks */
   ushort       fc_memflag;     /* what to do when list is exhausted */
   ushort       fc_lowmem;      /* low water mark, used w/MEM_PRI flag */
};
typedef struct fc_memseg MEMSEG;

#define FC_MEM_ERR      1       /* return error memflag */
#define FC_MEM_GETMORE  2       /* get more memory memflag */
#define FC_MEM_DMA      4       /* blocks are for DMA */
#define FC_MEM_LOWHIT   8       /* low water mark was hit */

#define FC_MEMPAD       16      /* offset used for a FC_MEM_DMA buffer */

/*
 * Board stat counters
 */
struct fc_stats {
   uint32 chipRingFree;
   uint32 cmdCreateXri;
   uint32 cmdQbuf;
   uint32 elsCmdIocbInval;
   uint32 elsCmdPktInval;
   uint32 elsLogiCol;
   uint32 elsRetryExceeded;
   uint32 elsStrayXmitCmpl;
   uint32 elsXmitCmpl;
   uint32 elsXmitErr;
   uint32 elsXmitFrame;
   uint32 elsXmitRetry;
   uint32 elsRcvDrop;
   uint32 elsRcvFrame;
   uint32 elsRcvRSCN;
   uint32 elsRcvFARP;
   uint32 elsRcvFARPR;
   uint32 elsRcvFLOGI;
   uint32 elsRcvPLOGI;
   uint32 elsRcvADISC;
   uint32 elsRcvPDISC;
   uint32 elsRcvFAN;
   uint32 elsRcvLOGO;
   uint32 elsRcvPRLO;
   uint32 elsRcvRRQ;
   uint32 frameRcvBcast;
   uint32 frameRcvMulti;
   uint32 hostRingFree;
   uint32 iocbCmdInval;
   uint32 iocbRingBusy;
   uint32 IssueIocb;
   uint32 iocbRsp;
   uint32 issueMboxCmd;
   uint32 linkEvent;
   uint32 xmitnoroom;
   uint32 NoIssueIocb;
   uint32 mapPageErr;
   uint32 mboxCmdBusy;
   uint32 mboxCmdInval;
   uint32 mboxEvent;
   uint32 mboxStatErr;
   uint32 memAllocErr;
   uint32 noRpiList;
   uint32 noVirtPtr;
   uint32 ringEvent;
   uint32 strayXmitCmpl;
   uint32 frameXmitDelay;
   uint32 xriCmdCmpl;
   uint32 xriStatErr;
   uint32 mbufcopy;
   uint32 LinkUp;
   uint32 LinkDown;
   uint32 LinkMultiEvent;
   uint32 NoRcvBuf;
   uint32 fcpCmd;
   uint32 fcpCmpl;
   uint32 fcpStrayCmpl;
   uint32 fcpFirstCheck;
   uint32 fcpGood;
   uint32 fcpRspErr;
   uint32 fcpRemoteStop;
   uint32 fcpLocalErr;
   uint32 fcpLocalTmo;
   uint32 fcpLocalNores;
   uint32 fcpLocalBufShort;
   uint32 fcpLocalSfw;
   uint32 fcpLocalTxDMA;
   uint32 fcpLocalRxDMA;
   uint32 fcpLocalinternal;
   uint32 fcpLocalCorrupt;
   uint32 fcpLocalIllFrm;
   uint32 fcpLocalDupFrm;
   uint32 fcpLocalLnkCtlFrm;
   uint32 fcpLocalLoopOpen;
   uint32 fcpLocalInvalRpi;
   uint32 fcpLocalLinkDown;
   uint32 fcpLocalOOO;
   uint32 fcpLocalAbtInp;
   uint32 fcpLocalAbtReq;
   uint32 fcpLocal;
   uint32 fcpPortRjt;
   uint32 fcpPortBusy;
   uint32 fcpError;
   uint32 fcpScsiTmo;
   uint32 fcpSense;
   uint32 fcpNoDevice;
   uint32 fcMallocCnt;
   uint32 fcMallocByte;
   uint32 fcFreeCnt;
   uint32 fcFreeByte;
   uint32 fcMapCnt;
   uint32 fcUnMapCnt;
   uint32 fcpRsvd0;
   uint32 fcpRsvd1;
   uint32 fcpRsvd2;
   uint32 fcpRsvd3;
   uint32 fcpRsvd4;
   uint32 fcpRsvd5;
   uint32 fcpRsvd6;
   uint32 fcpRsvd7;
   uint32 fcpRsvd8;
};
typedef struct fc_stats fc_stat_t;


/* Defines / Structures used to support IP profile */

#define FC_MIN_MTU      0               /* minimum size FC message */
#define FC_MAX_MTU      65280           /* maximum size FC message */

/* structure for MAC header */
typedef struct {
   uchar        dest_addr[MACADDR_LEN]; /* 48 bit unique address */
   uchar        src_addr[MACADDR_LEN];  /* 48 bit unique address */
   ushort       llc_len;                /* length of LLC data */
} emac_t;
#define HDR_LEN         14              /* MAC header size */

/* structure for LLC/SNAP header */
typedef struct {
   unsigned char        dsap;           /* DSAP                         */
   unsigned char        ssap;           /* SSAP                         */
   unsigned char        ctrl;           /* control field                */
   unsigned char        prot_id[3];     /* protocol id                  */
   unsigned short       type;           /* type field                   */
} snaphdr_t;

struct fc_hdr {
   emac_t       mac;
   snaphdr_t    llc;
};

struct fc_nethdr {
   NETHDR       fcnet;
   snaphdr_t    llc;
};

#define FC_LLC_SSAP             0xaa    /* specifies LLC SNAP header */
#define FC_LLC_DSAP             0xaa    /* specifies LLC SNAP header */
#define FC_LLC_CTRL             3       /* UI */


/*
 * The fc_buf structure is used to communicate SCSI commands to the adapter
 */
typedef struct sc_buf   T_SCSIBUF;  
#define SET_ADAPTER_STATUS(bp, val) bp->general_card_status = val;

#define P_DEPTH         ((FC_MAX_TRANSFER/PAGESIZE) + 2)

typedef struct fc_buf {
   FCP_CMND             fcp_cmd;        /* FCP command - This MUST be first */
   FCP_RSP              fcp_rsp;        /* FCP response - This MUST be next */
   struct fc_buf        *fc_fwd;        /* forward list pointer */
   struct fc_buf        *fc_bkwd;       /* backward list pointer */
   char                 *phys_adr;      /* physical address of this fc_buf */
   T_SCSIBUF            *sc_bufp;       /* pointer to sc_buf for this cmd */
   struct dev_info      *dev_ptr;       /* pointer to SCSI device structure */
   uint32               timeout;        /* Fill in how OS represents a time stamp */
                                        /* Fill in any OS specific members */
   int                  offset;
   ulong *              fc_cmd_dma_handle;
   ushort               iotag;          /* iotag for this cmd */
   ushort               flags;          /* flags for this cmd */
#define DATA_MAPPED     0x0001          /* data buffer has been D_MAPed */
#define FCBUF_ABTS      0x0002          /* ABTS has been sent for this cmd */
#define FCBUF_ABTS2     0x0004          /* ABTS has been sent twice */
#define FCBUF_INTERNAL  0x0008          /* Internal generated driver command */

    /*
     * Save the buffer pointer list for later use.
     * In SLI2, the fc_deq_fcbuf_active uses this pointer to 
     * free up MEM_BPL buffer
     */ 
   MATCHMAP              *bmp;
} fc_buf_t;

#define FCP_CONTINUE    0x01            /* flag for issue_fcp_cmd */
#define FCP_REQUEUE     0x02            /* flag for issue_fcp_cmd */
#define FCP_EXIT        0x04            /* flag for issue_fcp_cmd */

/*
 * The fcp_table structure is used to relate FCP iotags to an fc_buf
 */

typedef struct fcp_table {
   fc_buf_t     *fcp_array[MAX_FCP_CMDS];/* fc_buf pointers indexed by iotag */
} FCPTBL;


/*
 * SCSI node structure for each open Fibre Channel node
 */

typedef struct scsi_node {
   struct fc_dev_ctl * ap;              /* adapter structure ptr */
   struct dev_info   * lunlist;         /* LUN structure list for this node */
   NODELIST          * nlp;             /* nlp structure ptr */
   struct dev_info   * last_dev;        /* The last device had an I/O */
   FCCLOCK           * nodev_tmr;       /* Timer for nodev-tmo */
   int                 devno;           /* pseudo adapter major/minor number */
   int                 max_lun;         /* max number of luns */
   ushort              tgt_queue_depth; /* Max throttle of this node */
   ushort              num_active_io;   /* Total number of active I/O */
   ushort              rpi;             /* Device rpi */
   ushort              last_good_rpi;   /* Last known good device rpi */
   ushort              scsi_id;         /* SCSI ID of this device */
   ushort              flags;
#define FC_NODEV_TMO        0x1         /* nodev-tmo tmr started and expired */
#define FC_FCP2_RECOVERY    0x2         /* set FCP2 Recovery for commands */
#define RETRY_RPTLUN        0x4     /* Report Lun has been retried */
   ushort          addr_mode;   /* SCSI address method */
#define PERIPHERAL_DEVICE_ADDRESSING    0
#define VOLUME_SET_ADDRESSING       1
#define LOGICAL_UNIT_ADDRESSING     2
   ushort          rptlunstate;     /* For report lun SCSI command */
#define REPORT_LUN_REQUIRED     0
#define REPORT_LUN_ONGOING      1
#define REPORT_LUN_COMPLETE     2
   void           *virtRptLunData; 
   void           *physRptLunData; 
} node_t;

/* Values for node_flag and fcp_mapping are in fcdds.h */

/*
 * SCSI device structure for each open LUN
 */

#define MAX_FCBUF_PAGES 6               /* This value may need to change when
                                         * lun-queue-depth > 256 in lpfc.conf 
                                         */

typedef struct dev_info {
   node_t          *nodep;              /* Pointer to the node structure */
   struct dev_info *next;               /* Used for list of LUNs on this node */
   fc_lun_t        lun_id;              /* LUN ID of this device */
   uchar           first_check;         /* flag for first check condition */
#define FIRST_CHECK_COND	0x1
#define FIRST_IO		0x2   

   uchar           opened;
   uchar           ioctl_wakeup;        /* wakeup sleeping ioctl call */
   int             ioctl_event;
   int             ioctl_errno;
   int             stop_event;
   int             active_io_count;

   struct dev_info *DEVICE_WAITING_fwd;
   struct dev_info *ABORT_BDR_fwd;
   struct dev_info *ABORT_BDR_bkwd;

   long           qfullcnt;
    /* Fill in any OS specific members */
   T_SCSIBUF     *scp;
   void          *scsi_dev;
   long           scpcnt;
   long           qcmdcnt;
   long           iodonecnt;
   long           errorcnt;
    /*
     *  A command lives in a pending queue until it is sent to the HBA.
     *  Throttling constraints apply:
     *          No more than N commands total to a single target
     *          No more than M commands total to a single LUN on that target
     *
     *  A command that has left the pending queue and been sent to the HBA
     *  is an "underway" command.  We count underway commands, per-LUN,
     *  to obey the LUN throttling constraint.
     *
     *  Because we only allocate enough fc_buf_t structures to handle N
     *  commands, per target, we implicitly obey the target throttling
     *  constraint by being unable to send a command when we run out of
     *  free fc_buf_t structures.
     *
     *  We count the number of pending commands to determine whether the
     *  target has I/O to be issued at all.
     *
     *  We use next_pending to rotor through the LUNs, issuing one I/O at
     *  a time for each LUN.  This mechanism guarantees a fair distribution
     *  of I/Os across LUNs in the face of a target queue_depth lower than
     *  #LUNs*fcp_lun_queue_depth.
     */
   T_SCSIBUF     *standby_queue_head;   /* ptr to retry command queue */
   T_SCSIBUF     *standby_queue_tail;   /* ptr to retry command queue */
   uint32       standby_count;          /* # of I/Os on standby queue */
   /* END: added by andy kong for SCSI */

   ushort       fcp_cur_queue_depth;    /* Current maximum # cmds outstanding 
                                         * to dev; */ 
   ushort       fcp_lun_queue_depth;    /* maximum # cmds to each lun */
   T_SCSIBUF     *pend_head;            /* ptr to pending cmd queue */
   T_SCSIBUF     *pend_tail;            /* ptr to pending cmd queue */
   uint32       pend_count;
#define     QUEUE_HEAD      1
#define     QUEUE_TAIL      0
   struct buf *clear_head;              /* ptr to bufs to iodone after clear */
   uint32       clear_count;

   uchar        numfcbufs;              /* number of free fc_bufs */
   uchar        stop_send_io;           /* stop sending any io to this dev */


#define ACTIVE                  0
#define STOPPING                1
#define HALTED                  2
#define RESTART_WHEN_READY      3       
#define ACTIVE_PASSTHRU         4
#define WAIT_RESUME     8
#define WAIT_INFO       10
#define WAIT_ACA        11
#define WAIT_FLUSH      12
#define WAIT_HEAD_RESUME    13
   uchar         queue_state;           /* device general queue state */
                                        /* ACTIVE, STOPPING, or HALTED */

#define SCSI_TQ_HALTED        0x0001    /* The transaction Q is halted */
#define SCSI_TQ_CLEARING      0x0002    /* The transaction Q is clearing */
#define SCSI_TQ_CLEAR_ACA     0x0004    /* a CLEAR_ACA is PENDING      */
#define SCSI_LUN_RESET        0x0008    /* sent LUN_RESET not of TARGET_RESET */
#define SCSI_ABORT_TSET       0x0010    /* BDR requested but not yet sent */
#define SCSI_TARGET_RESET     0x0020    /* a SCSI BDR is active for device */
#define CHK_SCSI_ABDR         0x0038    /* value used to check tm flags */
#define QUEUED_FOR_ABDR       0x0040    /* dev_ptr is on ABORT_BDR queue */
#define NORPI_RESET_DONE      0x0100    /* BOGUS_RPI Bus Reset attempted */
#define DONT_LOG_INVALID_RPI  0x0200    /* if flag is set, the I/O issuing */
                                        /* to an invalid RPI won't be logged */ 
#define SCSI_IOCTL_INPROGRESS 0x0400    /* An ioctl is in progress  */
#define SCSI_SEND_INQUIRY_SN  0x1000    /* Serial number inq should be sent */
#define SCSI_INQUIRY_SN       0x2000    /* Serial number inq has been sent */
#define SCSI_INQUIRY_P0       0x4000    /* Page 0 inq has been sent */
#define SCSI_INQUIRY_CMD      0x6000    /* Serial number or Page 0 inq sent */
#define SCSI_DEV_RESET        0x8000    /* device is in process of resetting */
   ushort           flags;              /* flags for the drive */

   struct dio vlist;                    /* virtual address of fc_bufs */
   struct dio blist;                    /* physical addresses of fc_bufs */
   fc_buf_t      * fcbuf_head;          /* head ptr to list of free fc_bufs */
   fc_buf_t      * fcbuf_tail;          /* tail ptr to list of free fc_bufs */

   uchar        sense[MAX_FCP_SNS];     /* Temporary request sense buffer */
   uchar        sense_valid;            /* flag to indicate new sense data */
   uchar        sizeSN;                 /* size of InquirySN */
   uint32       sense_length;           /* new sense data length */

#define MAX_QFULL_RETRIES   255
#define MAX_QFULL_RETRY_INTERVAL 1000   /* 1000 (ms) */
   short        qfull_retries;          /* number of retries on a qfull condition */
   short        qfull_retry_interval;   /* the interval for qfull retry */
   void         * qfull_tmo_id;
   T_SCSIBUF     scbuf;                 /* sc_buf for task management cmds */

} dvi_t;


typedef struct node_info_hash {
   node_t       *node_ptr;              /* SCSI device node pointer */
   uint32        node_flag;             /* match node on WWPN WWNN or DID */
   union {
      NAME_TYPE   dev_nodename;         /* SCSI node name */
      NAME_TYPE   dev_portname;         /* SCSI port name */
      uint32      dev_did;              /* SCSI did */
   } un;
} nodeh_t;

/* 
 * LONGWAIT is used to define a default scsi_timeout value in seconds.
 */
#define LONGWAIT        30


/*
*** Board Information Data Structure
*/

struct fc_brd_info {
   /* Configuration Parameters */
   int  fc_ffnumrings;          /* number of FF rings being used */
   NAME_TYPE fc_nodename;               /* fc nodename */
   NAME_TYPE fc_portname;               /* fc portname */
   uint32       fc_pref_DID;            /* preferred D_ID */
   uchar        fc_pref_ALPA;           /* preferred AL_PA */
   uchar        fc_deferip;             /* defer IP processing */
   uchar        fc_nummask[4];          /* number of masks/rings being used */
   uchar        fc_rval[6];             /* rctl for ring assume mask is 0xff */
   uchar        fc_tval[6];             /* type for ring assume mask is 0xff */
   uchar        ipAddr[16];             /* For RNID support */
   ushort       ipVersion;              /* For RNID support */
   ushort       UDPport;                /* For RNID support */
   uint32       fc_edtov;               /* E_D_TOV timer value */
   uint32       fc_arbtov;              /* ARB_TOV timer value */
   uint32       fc_ratov;               /* R_A_TOV timer value */
   uint32       fc_rttov;               /* R_T_TOV timer value */
   uint32       fc_altov;               /* AL_TOV timer value */
   uint32       fc_crtov;               /* C_R_TOV timer value */
   uint32       fc_citov;               /* C_I_TOV timer value */
   uint32       fc_myDID;               /* fibre channel S_ID */
   uint32       fc_prevDID;             /* previous fibre channel S_ID */

   /* The next three structures get DMA'ed directly into,
    * so they must be in the first page of the adapter structure!
    */
   volatile SERV_PARM fc_sparam;        /* buffer for our service parameters */
   volatile SERV_PARM fc_fabparam;      /* fabric service parameters buffer */
   volatile uchar     alpa_map[128];    /* AL_PA map from READ_LA */

   uchar   fc_mbox_active;              /* flag for mailbox in use */
   uchar   fc_process_LA;               /* flag to process Link Attention */
   uchar   fc_ns_retry;                 /* retries for fabric nameserver */
   uchar   fc_sli;                      /* configured SLI, 1 or 2 */
   int     fc_nlp_cnt;                  /* cnt outstanding NODELIST requests */
   int     fc_open_count;               /* count of devices opened */
   int     fc_rscn_id_cnt;              /* count of RSCNs dids in list */
   uint32  fc_rscn_id_list[FC_MAX_HOLD_RSCN];
   Q       fc_plogi;                    /* ELS PLOGI cmd queue */
   Q       fc_mbox;                     /* mailbox cmd queue */
   Q       fc_rscn;                     /* RSCN cmd queue */
   Q       fc_defer_rscn;               /* deferred RSCN cmd queue */
   uchar   * fc_mbbp;                   /* buffer pointer for mbox command */
   uchar   * fc_p_dev_ctl;              /* pointer to driver device ctl  */

   /* Board dependent variables */
   int  fc_flag;                        /* FC flags */
   int  fc_brd_no;                      /* FC board number */
   int  fc_ints_disabled;               /* DEBUG: interrupts disabled */
   volatile int fc_ffstate;             /* Current state of FF init process */
   int  fc_interrupts;                  /* number of fc interrupts */
   int  fc_cnt;                         /* generic counter for board */
   int  fc_topology;                    /* link topology, from LINK INIT */
   int  fc_firstopen;                   /* First open to driver flag */
   int  fc_msgidx;                      /* current index to adapter msg buf */
   uint32       fc_eventTag;            /* event tag for link attention */
   ulong        fc_fabrictmo;           /* timeout for fabric timer */
   uchar        fc_multi;               /* number of multicast addresses */
   uchar        fc_linkspeed;           /* Link speed after last READ_LA */
   uchar        fc_max_data_rate;       /* max_data_rate                 */
   
   void         * physaddr[FC_MAX_IP_VECS]; /* used in mbuf_to_iocb for */
   uint32   cntaddr[FC_MAX_IP_VECS];        /* phys mapping */

   uchar        fc_busflag;             /* bus access flags */
#define FC_HOSTPTR      2               /* Default is ring pointers in SLIM */

   volatile uint32      * fc_mboxaddr;  /* virtual offset for mailbox/SLIM */
   volatile uint32      fc_BCregaddr;   /* virtual offset for BIU config reg */
   volatile uint32      fc_HAregaddr;   /* virtual offset for host attn reg */
   volatile uint32      fc_HCregaddr;   /* virtual offset for host ctl reg */
   volatile uint32      fc_FFregaddr;   /* virtual offset for FF attn reg */
   volatile uint32      fc_STATregaddr; /* virtual offset for status reg */


   MATCHMAP     fc_slim2;               /* pointers to slim for SLI-2 */

  void  *fc_iomap_io;                   /* starting address for registers */
  void   *fc_iomap_mem;                 /* starting address for SLIM */
                                        /* Fill in any OS specific members */
                                        /* dma handle, mem map, pci config access */





   FCCLOCK      * fc_mbox_wdt;          /* timer for mailbox   */
   FCCLOCK      * fc_fabric_wdt;        /* timer for fabric    */
   FCCLOCK      * fc_rscn_disc_wdt;     /* timer for RSCN discovery */
   fc_stat_t       fc_stats;            /* fc driver generic statistics */

   NAME_TYPE fc_multiaddr[FC_MAX_MCAST];/* multicast adrs for interface */
   NODELIST  * fc_nlpbind_start;        /* ptr to bind list */
   NODELIST  * fc_nlpbind_end;          /* ptr to bind list */
   NODELIST  * fc_nlpunmap_start;       /* ptr to unmap list */
   NODELIST  * fc_nlpunmap_end;         /* ptr to unmap list */
   NODELIST  * fc_nlpmap_start;         /* ptr to map list */
   NODELIST  * fc_nlpmap_end;           /* ptr to map list */
   ushort    fc_bind_cnt;
   ushort    fc_unmap_cnt;
   ushort    fc_map_cnt;
   ushort    fc_rpi_used;
   NODELIST  * fc_nlplookup[NLP_MAXRPI]; /* ptr to active D_ID / RPIs */
   NODELIST  fc_fcpnodev;               /* nodelist entry for no device */
   uint32    nlptimer;                  /* timestamp for nlplist entry */
   ushort    fc_capabilities;       /* default value for NODELIST caps */
   ushort    fc_sync;           /* default value for NODELIST sync */

   nodeh_t   device_queue_hash[MAX_FC_TARGETS]; /* SCSI node pointers */
   FCPTBL    * fc_table;                /* FCP iotag table pointer */
   IOCBQ     * fc_delayxmit;            /* List of IOCBs for delayed xmit */


   char      fc_adaptermsg[FC_MAX_ADPTMSG];   /* adapter printf messages */
   char      fc_SerialNumber[32];       /* adapter Serial Number */
   char      fc_OptionROMVersion[32];   /* adapter BIOS / Fcode version */
   MEMSEG    fc_memseg[FC_MAX_SEG];     /* memory for buffers / structures */
   RING fc_ring[MAX_RINGS];
};

typedef struct fc_brd_info FC_BRD_INFO;


/* Host Attn reg */
#define FC_HA_REG(binfo,sa)     ((volatile uint32 *)((volatile char *)sa + (binfo->fc_HAregaddr)))
#define FC_FF_REG(binfo,sa)     ((volatile uint32 *)((volatile char *)sa + (binfo->fc_FFregaddr)))

/* Host Status reg */
#define FC_STAT_REG(binfo,sa)   ((volatile uint32 *)((volatile char *)sa +(binfo->fc_STATregaddr)))

/* Host Cntl reg */
#define FC_HC_REG(binfo,sa)     ((volatile uint32 *)((volatile char *)sa + (binfo->fc_HCregaddr)))

/* BIU Configuration reg */
#define FC_BC_REG(binfo,sa)     ((volatile uint32 *)((volatile char *)sa + (binfo->fc_BCregaddr)))

/* SLIM defines for SLI-1 */
#define FC_MAILBOX(binfo,sa)    ((MAILBOX *)((volatile char *)sa + ((uint32)((ulong)binfo->fc_mboxaddr))))

/* SLIM defines for SLI-2 */
#define FC_SLI2_MAILBOX(binfo)  ((MAILBOX *)(binfo->fc_mboxaddr))

#define FC_IOCB(binfo,sa)       ((volatile uchar *)((volatile char *)sa + ((uint32)binfo->fc_mboxaddr + 0x100)))

#define FC_RING(ringoff,sa)     ((volatile uchar *)((volatile char *)sa + (ulong)(ringoff)))



/* Write 32-bit value to CSR register pointed to by regp */
#define WRITE_CSR_REG(binfo, regp, val)  fc_writel((uint32 *)(regp), (uint32)val)

/* Write 32-bit value to SLIM address pointed to by regp */
#define WRITE_SLIM_ADDR(binfo, regp, val)  fc_writel((uint32 *)(regp), (uint32)val)

/* Read 32-bit value from CSR register pointed to by regp */
#define READ_CSR_REG(binfo, regp)   fc_readl((uint32 *)(regp))

/* Read 32-bit value from SLIM address pointed to by regp */
#define READ_SLIM_ADDR(binfo, regp)  fc_readl((uint32 *)(regp))

/* Write wcnt 32-bit words to SLIM address pointed to by slimp */
#define WRITE_SLIM_COPY(binfo, bufp, slimp, wcnt)                       \
      fc_write_toio((uint32*)bufp, (uint32*)slimp, (sizeof(uint32)*(wcnt)))

/* Read wcnt 32-bit words from SLIM address pointed to by slimp */
#define READ_SLIM_COPY(binfo, bufp, slimp, wcnt)                        \
    fc_read_fromio((uint32*)slimp, (uint32*)bufp, (sizeof(uint32)*(wcnt)));\

#define WRITE_FLASH_COPY(binfo, bufp, flashp, wcnt)                     \
    fc_write_toio(bufp, flashp ,(sizeof(uint32)*(wcnt)))

#define READ_FLASH_COPY(binfo, bufp, flashp, wcnt)                      \
    fc_read_fromio(flashp, bufp, (sizeof(uint32)*(wcnt)))





/* defines for fc_open_count */
#define FC_LAN_OPEN 0x1         /* LAN open completed */
#define FC_FCP_OPEN 0x2         /* FCP open completed */

/* defines for fc_flag */
#define FC_FCP_WWNN             0x0   /* Match FCP targets on WWNN */
#define FC_FCP_WWPN             0x1   /* Match FCP targets on WWPN */
#define FC_FCP_DID              0x2   /* Match FCP targets on DID */
#define FC_FCP_MATCH            0x3   /* Mask for match FCP targets */
#define FC_PENDING_RING0        0x4   /* Defer ring 0 IOCB processing */
#define FC_LNK_DOWN             0x8   /* Link is down */
#define FC_PT2PT                0x10  /* pt2pt with no fabric */
#define FC_PT2PT_PLOGI          0x20  /* pt2pt initiate PLOGI */
#define FC_DELAY_DISC           0x40  /* Delay discovery till after cfglnk */
#define FC_PUBLIC_LOOP          0x80  /* Public loop */
#define FC_INTR_THREAD          0x100    /* In interrupt code */
#define FC_LBIT                 0x200    /* LOGIN bit in loopinit set */
#define FC_RSCN_MODE            0x400    /* RSCN cmd rcv'ed */
#define FC_RSCN_DISC_TMR        0x800    /* wait edtov before processing RSCN */
#define FC_NLP_MORE             0x1000   /* More node to process in node tbl */
#define FC_OFFLINE_MODE         0x2000   /* Interface is offline for diag */
#define FC_LD_TIMER             0x4000   /* Linkdown timer has been started */
#define FC_LD_TIMEOUT           0x8000   /* Linkdown timeout has occurred */
#define FC_FABRIC               0x10000  /* We are fabric attached */
#define FC_DELAY_PLOGI          0x20000  /* Delay login till unreglogin */
#define FC_SLI2                 0x40000  /* SLI-2 CONFIG_PORT cmd completed */
#define FC_INTR_WORK            0x80000  /* Was there work last intr */
#define FC_NO_ROOM_IP           0x100000   /* No room on IP xmit queue */
#define FC_NO_RCV_BUF           0x200000   /* No Rcv Buffers posted IP ring */
#define FC_BUS_RESET            0x400000   /* SCSI BUS RESET */
#define FC_ESTABLISH_LINK       0x800000   /* Reestablish Link */ 
#define FC_SCSI_RLIP            0x1000000  /* SCSI rlip routine called */
#define FC_DELAY_NSLOGI         0x2000000  /* Delay NameServer till ureglogin */
#define FC_NSLOGI_TMR           0x4000000  /* NameServer in process of logout */
#define FC_DELAY_RSCN           0x8000000  /* Delay RSCN till ureg/reg login */
#define FC_RSCN_DISCOVERY       0x10000000 /* Authenticate all devices after RSCN */
#define FC_2G_CAPABLE           0x20000000 /* HBA is 2 Gig capable */
#define FC_POLL_MODE        0x40000000 /* [SYNC] I/O is in the polling mode */
#define FC_BYPASSED_MODE        0x80000000 /* Interface is offline for diag */

/* defines for fc_ffstate */
#define FC_INIT_START           1
#define FC_INIT_NVPARAMS        2
#define FC_INIT_REV             3
#define FC_INIT_PARTSLIM        4
#define FC_INIT_CFGRING         5
#define FC_INIT_INITLINK        6
#define FC_LINK_DOWN            7
#define FC_LINK_UP              8
#define FC_INIT_SPARAM          9
#define FC_CFG_LINK             10
#define FC_FLOGI                11
#define FC_LOOP_DISC            12
#define FC_NS_REG               13
#define FC_NS_QRY               14
#define FC_NODE_DISC            15
#define FC_REG_LOGIN            16
#define FC_CLEAR_LA             17
#define FC_READY                32
#define FC_ERROR                0xff

#define NADDR_LEN       6       /* MAC network address length */

/* This should correspond with the HBA API event structure */
struct fc_hba_event {
   uint32   fc_eventcode;
   uint32   fc_evdata1;
   uint32   fc_evdata2;
   uint32   fc_evdata3;
   uint32   fc_evdata4;
};

typedef struct fc_hba_event HBAEVENT;
#define MAX_HBAEVENT   32

/***************************************************************************/
/*
 * This is the whole device control area for the adapter
 */
/***************************************************************************/

struct fc_dev_ctl {             /* NOTE: struct intr must be FIRST */
   struct intr  ihs;            /* interrupt handler control struct */
   ndd_t        ndd;            /* ndd for NS ndd chain */
   struct fc_dev_ctl *next;     /* point to the next device */
   uchar  phys_addr[NADDR_LEN]; /* actual network address in use */
   Simple_lock  cmd_slock;      /* adapter command lock */
   void       * ctl_correlator;/* point to the dd_ctl table */
   uchar        device_state;   /* main state of the device */
   uchar        open_state;     /* open state of the device */
   uchar        intr_inited;    /* flag for interrupt registration */
   uchar        fcp_mapping;    /* Map FCP devices based on WWNN WWPN or DID */
   ulong        fc_ipri;        /* save priority */
   int          power_up;
   uint32       dev_flag;       /* device flags */
#define FC_SCHED_CFG_INIT       2  /* schedule a call to fc_cfg_init() */
#define FC_FULL_INFO_CALL   4  /* set if fc_info() can return full info */ 
#define FC_NEEDS_DPC      0x10

   uchar        * devinfo;      /* point to the device info */
   uchar        * dip;          /* point to device information */
   uchar        * tran;         /* point to device information */
   FCCLOCK      * fc_estabtmo;  /* link establishment timer */
   FCCLOCK      * fc_waitflogi;  /* link establishment timer */
   fc_dds_t     dds;            /* device dependent structure */
   fc_vpd_t     vpd;            /* vital product data */
   FC_BRD_INFO  info;           /* device specific info */
   uchar        * mbufl_head;   /* mbuf for offlevel intr handler */
   uchar        * mbufl_tail;   /* mbuf for offlevel intr handler */
   void         * fc_evt_head;  /* waiting for event queue */
   void         * fc_evt_tail;  /* waiting for event queue */

   dvi_t        * DEVICE_WAITING_head;
   dvi_t        * DEVICE_WAITING_tail;
   dvi_t        * ABORT_BDR_head;
   dvi_t        * ABORT_BDR_tail;
   struct buf   * timeout_head; /* bufs to iodone after RLIP done */

   ushort       timeout_count;
   ushort       init_eventTag;  /* initial READ_LA eventtag from cfg */
   ushort       hba_event_put;  /* hbaevent event put word anchor */
   ushort       hba_event_get;  /* hbaevent event get word anchor */
   int          hba_event_missed;/* hbaevent missed event word anchor */
   uchar        pan_cnt;        /* pseudo adapter number counter */
   uchar        sid_cnt;        /* SCSI ID counter */
   uchar        adapter_state[NLP_MAXPAN];
   /* open/close state for pseudo adapters */

   Simple_lock  iostrat_lock;   /* lock for ioctl IOSTRAT */
   int          iostrat_event;  /* iostrat event word anchor */
   struct buf   * iostrat_head; /* head ptr to list of returned bufs */
   struct buf   * iostrat_tail; /* tail ptr to list of returned bufs */
   HBAEVENT     hbaevent[MAX_HBAEVENT];
   uint32       vendor_flag;
   uint32       dfcmb[MAILBOX_CMD_WSIZE];
  /* Fill in any OS specific members */
  struct Scsi_Host *host;
  struct pci_dev *pcidev;
  struct buf *iodone_head;
  struct buf *iodone_list;
  void       *dfc_kernel_buf;
  void       *abort_head;
  void       *abort_list;
  void       *rdev_head;
  void       *rdev_list;
  void       *rbus_head;
  void       *rbus_list;
  void       *rhst_head;
  void       *rhst_list;
  void       *qcmd_head;
  void       *qcmd_list;
  void       *qclk_head;
  void       *qclk_list;
  uint32      dpc_ha_copy;    /* copy of Host Attention Reg for DPC */
  uint32      dpc_hstatus;    /* copy of Host Status Reg for DPC */
  uint32      dpc_cnt;
  uint32      save_dpc_cnt;
  ulong       iflg;
  ulong       siflg;
  WAIT_QUEUE linkwq;
  WAIT_QUEUE rscnwq;
  WAIT_QUEUE ctwq;
};

typedef struct fc_dev_ctl fc_dev_ctl_t;


/***************************************************************************/
/*
 * This is the global device driver control structure
 */
/***************************************************************************/

struct fc_dd_ctl {
   FCCLOCK_INFO    fc_clock_info; /* clock setup */
   FCCLOCK         * fc_scsitmo;  /* scsi timeout timer */
   fc_dev_ctl_t    * p_dev[MAX_FC_BRDS]; /* device array */
   void            * p_config[MAX_FC_BRDS];
   ushort            num_devs;    /* count of devices configed */

   spinlock_t      smp_lock; /* keep this at end */
};

typedef struct fc_dd_ctl fc_dd_ctl_t;

/*
 * Macros for accessing device control area. The pointer to this area has to 
 * be named p_dev_ctl for using these macros.
 */

#define DD_CTL          fc_dd_ctl
#define CMD_LOCK        p_dev_ctl->cmd_slock
#define IOCTL_SLP_LOCK  ioctl_slp_lock
#define CLOCK_LOCK      clock_info->clk_slock
#define IOSTRAT_LOCK    p_dev_ctl->iostrat_lock
#define SCSI_TMO        DD_CTL.fc_scsitmo
#define CLOCKWDT clock_info->clktimer

#define IHS     p_dev_ctl->ihs
#define NDD     p_dev_ctl->ndd
#define NDDSTAT p_dev_ctl->ndd.ndd_genstats
#define VPD     p_dev_ctl->vpd
#define DDS     p_dev_ctl->dds
#define BINFO   p_dev_ctl->info
#define RINGTMO rp->fc_wdt
#define MBOXTMO binfo->fc_mbox_wdt
#define FABRICTMO binfo->fc_fabric_wdt
#define FCSTATCTR binfo->fc_stats

/* 
 * Lock class registration number for lock instrumentation. 
 * These numbers should be unique on the system and they should be 
 * controlled by the lock registration procedure set up for the lock 
 * instrumentations.
 */
#define FC_CMD_LOCK             47      
#define FC_IOSTRAT_LOCK         48      
#define FC_CFG_LOCK             49      
#define FC_CLOCK_LOCK       50
#define FC_IOCTL_SLP_LOCK       51

#ifndef LITTLE_ENDIAN_HOST
#if defined(i386)
#define LITTLE_ENDIAN_HOST      1
#endif

#endif
#if LITTLE_ENDIAN_HOST
#define SWAP_SHORT(x)   (x)
#define SWAP_LONG(x)    (x)
#define SWAP_DATA(x)    ((((x) & 0xFF)<<24) | (((x) & 0xFF00)<<8) | \
                        (((x) & 0xFF0000)>>8) | (((x) & 0xFF000000)>>24))
#define SWAP_DATA16(x)  ((((x) & 0xFF) << 8) | ((x) >> 8))
#define PCIMEM_SHORT(x) SWAP_SHORT(x)
#define PCIMEM_LONG(x)  SWAP_LONG(x)
#define PCIMEM_DATA(x)  SWAP_DATA(x)

#else   /* BIG_ENDIAN_HOST */

#define SWAP_SHORT(x)   ((((x) & 0xFF) << 8) | ((x) >> 8))
#define SWAP_LONG(x)    ((((x) & 0xFF)<<24) | (((x) & 0xFF00)<<8) | \
                        (((x) & 0xFF0000)>>8) | (((x) & 0xFF000000)>>24))
#define SWAP_DATA(x)    (x)
#define SWAP_DATA16(x)  (x)

#ifdef BIU_BSE   /* This feature only makes sense for Big Endian */
#define PCIMEM_SHORT(x) (x)
#define PCIMEM_LONG(x)  (x)
#define PCIMEM_DATA(x)  ((((x) & 0xFF)<<24) | (((x) & 0xFF00)<<8) | \
                        (((x) & 0xFF0000)>>8) | (((x) & 0xFF000000)>>24))
#else
#define PCIMEM_SHORT(x) SWAP_SHORT(x)
#define PCIMEM_LONG(x)  SWAP_LONG(x)
#define PCIMEM_DATA(x)  SWAP_DATA(x)
#endif
#endif

#define SWAP_ALWAYS(x)  ((((x) & 0xFF)<<24) | (((x) & 0xFF00)<<8) | \
                        (((x) & 0xFF0000)>>8) | (((x) & 0xFF000000)>>24))
#define SWAP_ALWAYS16(x) ((((x) & 0xFF) << 8) | ((x) >> 8))

/*
 * For PCI configuration
 */
#define ADDR_LO(addr)   ((int)(addr) & 0xffff)          /* low 16 bits */
#define ADDR_HI(addr)   (((int)(addr) >> 16) & 0xffff)  /* high 16 bits */

#endif /* _H_FC */
