#ifdef MODVERSIONS
#include <linux/modversions.h>
#endif

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/ctype.h>
#include <linux/time.h>
#include <linux/kdev_t.h>

#include "shfs.h"
#include "shfs_dcache.h"
#include "shfs_proc.h"
#include "shfs_proto.h"

static unsigned int
shfs_get_this_year(void)
{
	unsigned long s_per_year = 365*24*3600;
	unsigned long delta_s = 24*3600;

	unsigned int year = CURRENT_TIME / (s_per_year + delta_s/4);
	return 1970 + year; 
}

static unsigned int
shfs_get_this_month(void) 
{
	int month = -1;
	long secs = CURRENT_TIME % (365*24*3600+24*3600/4);
	static long days_month[] = { 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};
	if (shfs_get_this_year() % 4) {
		days_month[1] = 28;
	} else {
		days_month[1] = 29;
	}
	while (month < 11 && secs >= 0) {
		month++;
		secs-=24*3600*days_month[month];
	}
	return (unsigned int)month;
}

static unsigned int
shfs_get_month(char *mon)
{
	static char* months[] = {"Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};
	unsigned int i;

	for (i = 0; i<12; i++)
		if(strcmp(mon, months[i]) == 0) {
			DEBUG("%s: %u\n", mon, i);
			return i;
		}

	return 0;
}

static int
shfs_parse_dir(char *s, char **col)
{
	int c = 0;
	int is_space = 1;
	int device = 0;

	while (*s) {
		if (c == DIR_COLS)
			break;
		
		if (*s == ' ') {
			if (!is_space) {
				if ((c-1 == DIR_SIZE) && device) {
					while (*s && (*s == ' '))
						s++;
					while (*s && (*s != ' '))
						s++;
				}
				*s = '\0';
				is_space = 1;
			}
		} else {
			if (is_space) {
				/* (b)lock/(c)haracter device hack */
				col[c++] = s;
				is_space = 0;
				if ((c-1 == DIR_PERM) && ((*s == 'b')||(*s == 'c'))) {
					device = 1;
				}

			}
		}
		s++;
	}
	return c;
}

static int
shfs_loaddir(struct shfs_sb_info *info, char *name, struct shfs_dir_entry **head)
{
	char line[SHFS_LINE_MAX];
	char path[SHFS_PATH_MAX];
	char *col[DIR_COLS];
	char *b, *s;
	struct shfs_dir_entry *p, *q;
	umode_t mode;
	unsigned int year, mon, day, hour, min, this_year, this_month;
	int device; 
	int namelen;
	int res;
	
	DEBUG("reading %s\n", name);

	*head = NULL;
	q = NULL;		/* shut up gcc :) */

	this_year = shfs_get_this_year();
	this_month = shfs_get_this_month();
	
	if (strlen(name) + 1 > SHFS_PATH_MAX) {
		VERBOSE("Path too long!\n");
		return -ENAMETOOLONG;
	}

	if ((res = do_lsdir(info, name, info->mnt.stable_symlinks)) < 0) {
		VERBOSE("ls request failed!\n");
		return res;
	}
	
	while ((res = shfs_pipe_readln(info, line, SHFS_LINE_MAX)) > 0) {
		switch (reply(line)) {
		case REP_COMPLETE:
			return 0;
		case REP_EPERM:
			return -EPERM;
		case REP_ENOENT:
			return -ENOENT;
		case REP_ERROR:
			return -EIO;
		}

		res = shfs_parse_dir(line, col);
		if (res != DIR_COLS)
			continue;		/* skip `total xx' line */
		
		s = col[DIR_NAME];
		if ((strcmp(s, ".") != 0) && (strcmp(s, "..") != 0)) {
			p = (struct shfs_dir_entry*)KMEM_ALLOC("entry", dir_entry_cache, GFP_KERNEL);
			if (!p) {
				VERBOSE(" slab cache alloc error!\n");
				res = -ENOMEM;
				goto out;
			}
			memset(p, 0, sizeof(struct shfs_dir_entry));
			namelen = strlen(s) + 1;
			if (strstr(s, "->")) {
				char *ss = strstr(s, "->");
				if (*(ss + 3) == '/')
					namelen += strlen(info->mnt.mount_point);
			}
			if (namelen + 1 <= sizeof(p->ename)) {
				p->name = p->ename;
			} else {
				if (namelen + 1 > SHFS_PATH_MAX) {
					VERBOSE("too long entry!\n");
					KMEM_FREE("entry", dir_entry_cache, p);
					res = -ENAMETOOLONG;
					goto out;
				}
				p->name = (char *)KMEM_ALLOC("name", dir_name_cache, GFP_KERNEL);
				if (!p->name) {
					VERBOSE("slab cache alloc error!\n");
					KMEM_FREE("entry", dir_entry_cache, p);
					res = -ENOMEM;
					goto out;
				}
			}
			strcpy(p->name, s);

			s = col[DIR_PERM];
			mode = 0; device = 0;
			switch(s[0]) {
			case 'b':
				device = 1;
				if ((info->mnt.fmask & S_IFMT) & S_IFBLK)
					mode = S_IFBLK;
				else
					mode = S_IFREG;
				break;
			case 'c':
				device = 1;
				if ((info->mnt.fmask & S_IFMT) & S_IFCHR)
					mode = S_IFCHR;
				else
					mode = S_IFREG;
				break;
			case 's':
			case 'S':			/* IRIX64 socket */
				mode = S_IFSOCK;
				break;
			case 'd':
				mode = S_IFDIR;
				break;
			case 'l':
				mode = S_IFLNK;
				break;
			case '-':
				mode = S_IFREG;
				break;
			case 'p':
				mode = S_IFIFO;
				break;
			}
			if (s[1] == 'r') mode |= S_IRUSR;
			if (s[2] == 'w') mode |= S_IWUSR;
			if (s[3] == 'x') mode |= S_IXUSR;
			if (s[3] == 's') mode |= S_IXUSR | S_ISUID;
			if (s[3] == 'S') mode |= S_ISUID;
			if (s[4] == 'r') mode |= S_IRGRP;
			if (s[5] == 'w') mode |= S_IWGRP;
			if (s[6] == 'x') mode |= S_IXGRP;
			if (s[6] == 's') mode |= S_IXGRP | S_ISGID;
			if (s[6] == 'S') mode |= S_ISGID;
			if (s[7] == 'r') mode |= S_IROTH;
			if (s[8] == 'w') mode |= S_IWOTH;
			if (s[9] == 'x') mode |= S_IXOTH;
			if (s[9] == 't') mode |= S_ISVTX | S_IXOTH;
			if (s[9] == 'T') mode |= S_ISVTX;
			p->mode = S_ISREG(mode) ? mode & info->mnt.fmask : mode;

			p->ino = simple_strtoul(col[DIR_INODE], NULL, 10);
			p->uid = simple_strtoul(col[DIR_UID], NULL, 10);
			p->gid = simple_strtoul(col[DIR_GID], NULL, 10);
			
			if (!device) {
				p->size = simple_strtoul(col[DIR_SIZE], NULL, 10);
			} else {
				unsigned short major, minor;
				p->size = 0;
				major = (unsigned short) simple_strtoul(col[DIR_SIZE], &s, 10);
				while (*s && (!isdigit(*s)))
					s++;
				minor = (unsigned short) simple_strtoul(s, NULL, 10);
				p->rdev = MKDEV(major, minor);
			}
			p->nlink = simple_strtoul(col[DIR_NLINK], NULL, 10);
			p->blocksize = 4096;
			p->blocks = (p->size + 4095) >> 11;

			mon = shfs_get_month(col[DIR_MONTH]);
			day = simple_strtoul(col[DIR_DAY], NULL, 10);
			
			s = col[DIR_YEAR];
			if (!strchr(s, ':')) {
				year = simple_strtoul(s, NULL, 10);
				hour = 12;
				min = 0;
			} else {
				year = this_year;
				if (mon > this_month) 
					year--;
				b = strchr(s, ':');
				*b = 0;
				hour = simple_strtoul(s, NULL, 10);
				min = simple_strtoul(++b, NULL, 10);
			}
			p->atime = mktime(year, mon + 1, day, hour, min, 0);

			/* symlink hack */
			if (S_ISLNK(mode) && (strstr(p->name, "->"))) {
				char *ss = strstr(p->name, "->");

				*(ss - 1) = 0;
				if ((*(ss + 3) == '/') && (strlen(info->mnt.mount_point) + strlen(ss + 3) + 1 < SHFS_PATH_MAX)) {
					strcpy(path, ss+3);
					sprintf(ss+3, "%s%s", info->mnt.mount_point, path);
				}
			}
			DEBUG("Name: %s, mode: %o, inode: %lu, size: %lu, nlink: %d, month: %d, day: %d, year: %d, hour: %d, min: %d (time: %lu)\n", p->name, p->mode, p->ino, p->size, p->nlink, mon, day, year, hour, min, p->atime);

			p->next = NULL;
			if (!*head)
				*head = p;
			else
				q->next = p;
			q = p;
		}
	}
out:
	for (p = *head; p; p = q) {
		q = p->next;
		if (p->name != p->ename)
			KMEM_FREE("name", dir_name_cache, p->name);
		KMEM_FREE("entry", dir_entry_cache, p);
	}
	info->mnt.garbage_read = 0;
	info->mnt.garbage = 1;
	return res;
}

static int
dcache_add(struct shfs_sb_info *info, struct dentry *dentry)
{
	char name[SHFS_PATH_MAX];
	struct inode *inode = dentry->d_inode;
	struct shfs_dir_head *head;
	int error = 0;
	
	DEBUG("loading dir %s...\n", dentry->d_name.name);
        if (shfs_get_name(dentry, name) < 0) {
                VERBOSE("Name too long!\n");
                return -ENAMETOOLONG;
        }
	if (!inode || !S_ISDIR(inode->i_mode)) {
		VERBOSE("invalid inode\n");
		return -ENOENT;
	}
 	
	head = (struct shfs_dir_head*)KMEM_ALLOC("head", dir_head_cache, GFP_KERNEL);
	if (!head) {
		VERBOSE("slab cache alloc error!\n");
		return -ENOMEM;
	}
	head->first = NULL;
	head->age = jiffies;
	error = shfs_loaddir(info, name, &head->first);
	if (error < 0) {
		VERBOSE("Couldn't load directory (%d)!\n", error);
		KMEM_FREE("head", dir_head_cache, head);
		head = NULL;
	}
	(struct shfs_dir_head *)inode->u.generic_ip = head;
	return error;
}

static void
dcache_del(struct inode *inode)
{
	struct shfs_dir_entry *p, *q;
	struct shfs_dir_head *head;

	DEBUG("deleting a cache entry\n");
	if (!inode || !S_ISDIR(inode->i_mode)) {
		VERBOSE("invalid inode\n");
		return;
	}
	head = (struct shfs_dir_head *)inode->u.generic_ip;
	if (!head) {
		DEBUG("no head\n");
		return;
	}
	p = head->first;
	while (p) {
		q = p->next;
		if (p->name != p->ename)
			KMEM_FREE("name", dir_name_cache, p->name);
		KMEM_FREE("entry", dir_entry_cache, p);
		p = q;
	}
	inode->u.generic_ip = NULL;
	KMEM_FREE("head", dir_head_cache, head);
	DEBUG("done\n");
	return;
}
#if 0
static void
dcache_invalidate_dircache_entries(struct dentry *parent)
{
        struct list_head *next;
        struct dentry *dentry;

        spin_lock(&dcache_lock);
        next = parent->d_subdirs.next;
        while (next != &parent->d_subdirs) {
                dentry = list_entry(next, struct dentry, d_child);
		dget(dentry);
		d_drop(dentry);
		dput(dentry);
                next = next->next;
        }
        spin_unlock(&dcache_lock);
}
#endif
int
shfs_dcache_get_dir(struct dentry *dentry, struct shfs_dir_entry **dir)
{
	struct shfs_sb_info *info = (struct shfs_sb_info*)dentry->d_sb->u.generic_sbp;
	struct shfs_dir_head *head;
	struct inode *inode = dentry->d_inode;
	int error = 0;

	DEBUG("%s\n", dentry->d_name.name);
	if (dir)
		*dir = NULL;
	if (!inode || !S_ISDIR(inode->i_mode)) {
		VERBOSE("invalid inode\n");
		return -ENOENT;
	}

	head = (struct shfs_dir_head *)inode->u.generic_ip;
	if (!head) {
		DEBUG("no head\n");
		error = dcache_add(info, dentry);
		head = (struct shfs_dir_head *)inode->u.generic_ip;
		goto out;
	}

	if (jiffies - head->age >= SHFS_MAX_AGE(info)) {
		DEBUG("entry too old\n");
		dcache_del(dentry->d_inode);
//		dcache_invalidate_dircache_entries(dentry);
		DEBUG("deleted\n");
		error = dcache_add(info, dentry);
		goto out;
	}
out:
	if (error) {
		DEBUG("error: %d\n", error);
		return error;
	}
	if (dir)
		*dir = head->first;
	return 0;
}

int
shfs_dcache_get_entry(struct dentry *dentry, struct shfs_dir_entry **dir)
{
	struct shfs_dir_entry *p;
	int error = 0;
	
#if DEBUG_LEVEL >= 3
	{
		char name[SHFS_PATH_MAX];
	        if (shfs_get_name(dentry, name) >= 0)
			DEBUG("path: %s\n", name);
	}
#endif
	
	if (dir)
		*dir = NULL;
	if (IS_ROOT(dentry)) {
		DEBUG("root\n");
		return 0;
	}

	error = shfs_dcache_get_dir(dentry->d_parent, &p);
	if (error) {
		DEBUG("error: %d\n", error);
		return error;
	}

	while (p) {
                if (!strcmp(dentry->d_name.name, p->name)) {
			if (dir)
				*dir = p;
			return 0;
		}
		p = p->next;
        }
        return -ENOENT;
}

void
shfs_dcache_invalidate_dir(struct inode *dir)
{
	struct shfs_sb_info *info = (struct shfs_sb_info*)dir->i_sb->u.generic_sbp;
	struct shfs_dir_head *head;

	DEBUG("\n");
	head = (struct shfs_dir_head *)dir->u.generic_ip;
	if (!head)
		return;

	/* make sure next dcache_get_dir() will reload entire dir */
	head->age = jiffies - SHFS_MAX_AGE(info);
}

void
shfs_dcache_clear_inode(struct inode *inode)
{
	if (!inode->u.generic_ip)
		return;
	dcache_del(inode);
}

void
shfs_dcache_update_fattr(struct dentry *dentry, struct iattr *attr)
{
	struct shfs_dir_entry *entry;
	int error;

	DEBUG("%s\n", dentry->d_name.name);
	error = shfs_dcache_get_entry(dentry, &entry);
	if (error < 0 || !entry)
		return;
	
	if (attr->ia_valid & ATTR_MODE)
		entry->mode = attr->ia_mode;
	if (attr->ia_valid & ATTR_UID)
		entry->uid = attr->ia_uid;
	if (attr->ia_valid & ATTR_GID)
		entry->gid = attr->ia_gid;
	if (attr->ia_valid & ATTR_ATIME)
		entry->atime = attr->ia_atime;
	if (attr->ia_valid & ATTR_MTIME)
		entry->atime = attr->ia_mtime;
	if (attr->ia_valid & ATTR_CTIME)
		entry->atime = attr->ia_ctime;
	if (attr->ia_valid & ATTR_SIZE) {
		entry->size = attr->ia_size;
		entry->blocksize = 4096;
		entry->blocks = (attr->ia_size + 4095) >> 11;
		DEBUG("size changed to: %llu\n", attr->ia_size);
	}
}

static void
set_inode_attr(struct inode *inode, struct shfs_dir_entry *entry)
{
	time_t last_time = inode->i_mtime;
	loff_t last_size = inode->i_size;

	DEBUG("\n");
	inode->i_mode = entry->mode;
	inode->i_uid = entry->uid;
	inode->i_gid = entry->gid;
	inode->i_size = entry->size;
	inode->i_blksize = entry->blocksize;
	inode->i_blocks = entry->blocks;
	inode->i_ctime = entry->atime;
	inode->i_atime = entry->atime;
	inode->i_mtime = entry->atime;

	if (inode->i_mtime != last_time || inode->i_size != last_size) {
		DEBUG("inode changed\n");
		invalidate_inode_pages(inode);
		shfs_fcache_clear(inode);
	}
}

int
shfs_dcache_revalidate_entry(struct dentry *dentry)
{
	struct inode *inode = dentry->d_inode;
	struct shfs_dir_entry *p;
	int error;
	
	DEBUG("%s\n", dentry->d_name.name);
	if (!inode) {
		DEBUG("no inode\n");
		return 0;
	}
	if (IS_ROOT(dentry)) {
		DEBUG("is root\n");
		return 1;
	}

	error = shfs_dcache_get_entry(dentry, &p);
	if (error < 0)
		return error;
	
	set_inode_attr(inode, p);
	return 1;
}

