#ifndef _SHFS_PROTO_H_
#define _SHFS_PROTO_H_

#include <linux/types.h>
#include <linux/fs.h>

/* response code */
#define REP_PRELIM	100
#define REP_COMPLETE	200
#define REP_NOP 	201
#define REP_NOTEMPTY	202		/* file with zero size but not empty */
#define REP_CONTINUE	300
#define REP_TRANSIENT	400
#define REP_ERROR	500
#define REP_EPERM	501
#define REP_ENOSPC	502
#define REP_ENOENT	503

/* directory list fields (ls -lani) */
#define DIR_COLS   10
#define DIR_INODE  0
#define DIR_PERM   1
#define DIR_NLINK  2
#define DIR_UID    3
#define DIR_GID    4
#define DIR_SIZE   5
#define DIR_MONTH  6
#define DIR_DAY    7
#define DIR_YEAR   8
#define DIR_NAME   9

#define SHFS_OPS   20

extern char **shfs_system_identify[];
extern char **shfs_system_ops[][SHFS_OPS];

int do_identify(struct shfs_sb_info *, int);
int do_init(struct shfs_sb_info *, int);
int do_ping(struct shfs_sb_info *, unsigned long);

static inline int 
do_finish(struct shfs_sb_info *info)
{
	return shfs_pipe_printf(info, "s_finish\n");
}

static inline int 
do_lsdir(struct shfs_sb_info *info, char *dir, int stable)
{
	return shfs_pipe_printf(info, "s_lsdir '%s' %c\n", dir, stable ? 'L' : ' ');
}

static inline int 
do_open(struct shfs_sb_info *info, char *path, char *mode)
{
	return shfs_pipe_printf(info, "s_open '%s' %s\n", path, mode);
}

static inline int 
do_read(struct shfs_sb_info *info, char *path, unsigned offset, unsigned count)
{
	unsigned bs = 1, offset2 = offset, count2 = count;

	/* read speedup if possible */
	if (count && !(offset % count)) {
		bs = count;
		offset2 = offset / count;
		count2 = 1;
	}
	return shfs_pipe_printf(info, "s_read '%s' %u %u %u %u %u\n", path, offset, count, bs, offset2, count2);
}

static inline int 
do_slow_read(struct shfs_sb_info *info, char *path, unsigned offset, unsigned count, unsigned long ino)
{
	unsigned bs = 1, offset2 = offset, count2 = count;

	/* read speedup if possible */
	if (count && !(offset % count)) {
		bs = count;
		offset2 = offset / count;
		count2 = 1;
	}
	return shfs_pipe_printf(info, "s_sread '%s' %u %u %u %u %u %lu\n", path, offset, count, bs, offset2, count2, ino);
}

static inline int
do_write(struct shfs_sb_info *info, char *path, unsigned offset, unsigned count, unsigned long ino)
{
	unsigned offset2 = offset, bs = 1;

	if (offset % SHFS_FCACHE_SIZE == 0) {
		offset2 = offset / SHFS_FCACHE_SIZE;
		bs = SHFS_FCACHE_SIZE;
	}
	return shfs_pipe_printf(info, "s_write '%s' %u %u %u %u %lu\n", path, offset, count, bs, offset2, ino);
}

static inline int
do_mkdir(struct shfs_sb_info *info, char *path)
{
	return shfs_pipe_printf(info, "s_mkdir '%s'\n", path);
}

static inline int 
do_rmdir(struct shfs_sb_info *info, char *path)
{
	return shfs_pipe_printf(info, "s_rmdir '%s'\n", path);
}

static inline int
do_mv(struct shfs_sb_info *info, char *old, char *new)
{
	return shfs_pipe_printf(info, "s_mv '%s' '%s'\n", old, new);
}

static inline int
do_rm(struct shfs_sb_info *info, char *path)
{
	return shfs_pipe_printf(info, "s_rm '%s'\n", path);
}

static inline int 
do_creat(struct shfs_sb_info *info, char *path, umode_t mode)
{
	return shfs_pipe_printf(info, "s_creat '%s' %o\n", path, mode);
}

static inline int
do_ln(struct shfs_sb_info *info, char *old, char *new)
{
	return shfs_pipe_printf(info, "s_ln '%s' '%s'\n", old, new);
}

static inline int
do_sln(struct shfs_sb_info *info, const char *old, const char *new)
{
	return shfs_pipe_printf(info, "s_sln '%s' '%s'\n", old, new);
}

static inline int 
do_chmod(struct shfs_sb_info *info, char *path, umode_t mode)
{
	return shfs_pipe_printf(info, "s_chmod '%s' %o\n", path, mode);
}

static inline int 
do_chown(struct shfs_sb_info *info, char *path, uid_t uid)
{
	return shfs_pipe_printf(info, "s_chown '%s' %u\n", path, uid);
}

static inline int 
do_chgrp(struct shfs_sb_info *info, char *path, gid_t gid)
{
	return shfs_pipe_printf(info, "s_chgrp '%s' %u\n", path, gid);
}

static inline int 
do_trunc(struct shfs_sb_info *info, char *path, loff_t size)
{
	unsigned seek = 1;

	/* dd doesn't like bs=0 */
	if (size == 0) {
		seek = 0;
		size = 1;
	}
	return shfs_pipe_printf(info, "s_trunc '%s' %u %u\n", path, (unsigned) size, seek);
}
	
static inline int
do_settime(struct shfs_sb_info *info, char *path, int atime, char *time)
{
	return shfs_pipe_printf(info, "s_settime '%s' %s %s\n", path, atime ? "a" : "m", time);
}

static inline int 
do_statfs(struct shfs_sb_info *info)
{
	return shfs_pipe_printf(info, "s_statfs\n");
}

#endif	/* _SHFS_PROTO_H_ */
