/*
   SCSI upper level driver that permits user applications to monitor
   the state of SCSI devices and hosts. The original purpose of this
   driver is to support hotplugging notification.

 *       Copyright (C) 2001 Douglas Gilbert
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 */
#include <linux/config.h>
/* static char * scsimon_version_str = "Version: 0.0.92 (20010313)"; */
 static int scsimon_version_num = 92; /* 2 digits for each component */

/*
 *  For more information contact:
 *  D. P. Gilbert (dgilbert@interlog.com, dougg@triode.net.au)
 */
#include <linux/module.h>

#include <linux/fs.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/string.h>
#include <linux/mm.h>
#include <linux/errno.h>
#include <linux/ioctl.h>
#include <linux/fcntl.h>
#include <linux/init.h>
#include <linux/poll.h>
#include <linux/smp_lock.h>
#include <linux/time.h>
#include <linux/miscdevice.h>
#include <linux/kmod.h>

#include <asm/io.h>
#include <asm/uaccess.h>
#include <asm/system.h>

#include <linux/blk.h>
#include "scsi.h"
#include "hosts.h"
#include <scsi/scsi_ioctl.h>
#include <scsi/scsimon.h>

#ifndef LINUX_VERSION_CODE
#include <linux/version.h>
#endif /* LINUX_VERSION_CODE */


static int scsimon_init(void);
static int scsimon_attach(Scsi_Device *);
static void scsimon_finish(void);
static int scsimon_detect(Scsi_Device *);
static void scsimon_detach(Scsi_Device *);

static struct Scsi_Device_Template scsimon_template =
{
	name:"scsimon",
	tag:"scsimon",
	scsi_type:0xff,
	major:MISC_MAJOR,
	detect:scsimon_detect,
	init:scsimon_init,
	finish:scsimon_finish,
	attach:scsimon_attach,
	detach:scsimon_detach
};


static int scsimon_open(struct inode * inode, struct file * filp);
static int scsimon_release(struct inode * inode, struct file * filp);
static ssize_t scsimon_read(struct file * filp, char * buf,
			     size_t count, loff_t *ppos);
static int scsimon_ioctl(struct inode * inode, struct file * filp,
			  unsigned int cmd_in, unsigned long arg);
static unsigned int scsimon_poll(struct file * filp, poll_table * wait);
static int scsimon_fasync(int fd, struct file * filp, int mode);

static struct file_operations scsimon_fops = {
	owner:		THIS_MODULE,
	read:		scsimon_read,
	poll:		scsimon_poll,
	ioctl:		scsimon_ioctl,
	open:		scsimon_open,
	release:	scsimon_release,
	fasync:		scsimon_fasync,
};

static struct miscdevice scsimon_miscdev = {
	SCSIMON_MINOR,
	"scsimon",
	&scsimon_fops
};


typedef struct scsimon_fd /* holds the state of a file descriptor */
{
	struct scsimon_fd * nextfp; /* NULL when last opened fd */
	struct fasync_struct * async_qp; /* for asynchronous notification */
} Sm_fd;

typedef struct scsimon_attached 
{
	struct scsimon_attached * next_att; /* NULL if last */
	Scsi_Device * device;
	unsigned int host;
	unsigned int bus;
	unsigned int target;
	unsigned int lun;
	unsigned long time_attached;	/* jiffy count when attached */
	unsigned long event_count;
} Sm_attached;

typedef struct scsimon_detached 
{
	struct scsimon_detached * next_det; /* NULL if last */
	unsigned int host;
	unsigned int bus;
	unsigned int target;
	unsigned int lun;
	char vendor[8];
	char model[16];
	char rev[4];
	char host_name[16]; 
	char host_info[80];
	char scsi_type;
	char scsi_level;	/* as in SCSI field in INQ */
	char removable;
	char emulated;
	unsigned long time_detached;	/* jiffy count when detached */
	unsigned long event_count;	/* event count when detached */
	unsigned long event_attached;	/* event count when attached */
} Sm_detached;

typedef struct scsimon_hotinfo
{
	int action;		/* 0 -> attach, 1 -> detach */
	unsigned int host;
	unsigned int bus;
	unsigned int target;
	unsigned int lun;
	unsigned long event_attached;
	unsigned long event_detached;
	char vendor[8];
	char model[16];
	char rev[4];
	char scsi_type;
	char removable;
	char emulated;
} Sm_hotinfo;

static rwlock_t scsimon_lock = RW_LOCK_UNLOCKED;

/* static Sm_fd * sm_fd_rootp; */
static Sm_attached * sm_att_rootp;
static Sm_detached * sm_det_rootp;
static int sm_locked_down; /* hack to keep module loaded */
static unsigned long sm_event_count; /* bumped on each change */
static unsigned long sm_init_time; 
static int sm_max_detached_elems = 10; /* forgets oldest */
static int sm_flag_attached;
static int sm_flag_detached;

static void sm_trunc_alist(int fromPos); 
static void sm_trunc_dlist(int fromPos); 
static Sm_fd * sm_add_sfp(void);
static void sm_remove_sfp(Sm_fd *);
static void sm_read_state(struct scsimon_state *);
static void sm_read_att_list(struct scsimon_att_list *);
static void sm_read_det_list(struct scsimon_det_list *);
static void sm_read_host_list(struct scsimon_host_list *);
static void sm_call_policy (Sm_hotinfo *);


static int scsimon_open(struct inode * inode, struct file * filp)
{
	Sm_fd * sfp;

	SCSI_LOG_TIMEOUT(3, printk("scsimon_open: flags=0x%x\n", 
				   filp->f_flags));
	if ((sfp = sm_add_sfp()))
		filp->private_data = sfp;
	else
		return -ENOMEM;
	return 0;
}

static int scsimon_release(struct inode * inode, struct file * filp)
{
    Sm_fd * sfp;

	lock_kernel();
	if (! (sfp = (Sm_fd *)filp->private_data)) {
		unlock_kernel();
		return -ENXIO;
	}
	SCSI_LOG_TIMEOUT(3, printk("scsimon_release:\n"));
	scsimon_fasync(-1, filp, 0);   /* remove from async notify list */
	filp->private_data = NULL;
	sm_remove_sfp(sfp);
	unlock_kernel();
	return 0;
}

static ssize_t scsimon_read(struct file * filp, char * buf,
                       size_t count, loff_t *ppos)
{
	Sm_fd * sfp;
	unsigned long iflags;

	SCSI_LOG_TIMEOUT(3, printk("scsimon_read: count=%d\n", (int)count));
	if (! (sfp = (Sm_fd *)filp->private_data))
		return -ENXIO;
	write_lock_irqsave(&scsimon_lock, iflags);
	sm_flag_attached = 0;
	sm_flag_detached = 0;
	write_unlock_irqrestore(&scsimon_lock, iflags);
	return count;
}

static int scsimon_ioctl(struct inode * inode, struct file * filp,
                          unsigned int cmd_in, unsigned long arg)
{
	int val, result, read_only;
	Sm_fd * sfp;

	if (! (sfp = (Sm_fd *)filp->private_data))
		return -ENXIO;
	SCSI_LOG_TIMEOUT(3, printk("scsimon_ioctl: cmd=0x%x\n", (int)cmd_in));
	read_only = (O_RDWR != (filp->f_flags & O_ACCMODE));

	switch(cmd_in)
	{
	case SCSIMON_LOCK_MOD:
		if (0 == sm_locked_down) {
			sm_locked_down++;
			MOD_INC_USE_COUNT;
		}
		return 0;
	case SCSIMON_UNLOCK_MOD:
		if (sm_locked_down > 0) {
			sm_locked_down--;
			MOD_DEC_USE_COUNT;
		}
		return 0;
	case SCSIMON_GET_VERSION_NUM:
		return put_user(scsimon_version_num, (int *)arg);
	case SCSIMON_SET_MX_DLIST_LEN:
		result = get_user(val, (int *)arg);
		if (result) return result;
			sm_max_detached_elems = val;
			return 0;
	case SCSIMON_GET_MX_DLIST_LEN:
		return put_user(sm_max_detached_elems, (int *)arg);
    	case SCSIMON_GET_STATE:
		result = verify_area(VERIFY_WRITE, (void *)arg, 
				     sizeof(struct scsimon_state));
		if (result) return result;
		else {
			struct scsimon_state ss;

			sm_read_state(&ss);
			__copy_to_user((void *)arg, &ss, 
				       sizeof(struct scsimon_state));
			return 0;
		}
	case SCSIMON_GET_ATT_LIST:
		{
			struct scsimon_att_list * sm_alp = 
					(struct scsimon_att_list *)arg;
			struct scsimon_att_list * sm_alop; 
			size_t sz = sizeof(struct scsimon_att_list);

			result = verify_area(VERIFY_WRITE, (void *)arg, sz);
			if (result) return result;
			__get_user(val, &sm_alp->max_num);
			if (val > 1) {
				sz += ((val - 1) * 
				       sizeof(struct scsimon_att_dev));
				result = verify_area(VERIFY_WRITE, 
						     (void *)arg, sz);
				if (result) return result;
			}
			sm_alop = (struct scsimon_att_list *)
				  kmalloc(sz, GFP_ATOMIC);
			if (NULL == sm_alop)
				return -ENOMEM;
			memset(sm_alop, 0, sz);
			sm_alop->max_num = val;
			__get_user(sm_alop->match_event, &sm_alp->match_event);
			__get_user(sm_alop->flags, &sm_alp->flags);
			sm_read_att_list(sm_alop);
			__copy_to_user((void *)arg, sm_alop, sz);
			kfree((char *)sm_alop);
			return 0;
		}
	case SCSIMON_GET_DET_LIST:
		{
			struct scsimon_det_list * sm_dlp = 
					(struct scsimon_det_list *)arg;
			struct scsimon_det_list * sm_dlop; 
			size_t sz = sizeof(struct scsimon_det_list);

			result = verify_area(VERIFY_WRITE, (void *)arg, sz);
			if (result) return result;
			__get_user(val, &sm_dlp->max_num);
			if (val > 1) {
				sz += ((val - 1) * 
				       sizeof(struct scsimon_det_dev));
				result = verify_area(VERIFY_WRITE, 
						     (void *)arg, sz);
				if (result) return result;
			}
	    		sm_dlop = (struct scsimon_det_list *)
				  kmalloc(sz, GFP_ATOMIC);
			if (NULL == sm_dlop)
				return -ENOMEM;
			memset(sm_dlop, 0, sz);
			sm_dlop->max_num = val;
			__get_user(sm_dlop->match_event, &sm_dlp->match_event);
			__get_user(sm_dlop->flags, &sm_dlp->flags);
			sm_read_det_list(sm_dlop);
			__copy_to_user((void *)arg, sm_dlop, sz);
			kfree((char *)sm_dlop);
			return 0;
		}
	case SCSIMON_GET_HOST_LIST:
		{
			struct scsimon_host_list * sm_hlp = 
					(struct scsimon_host_list *)arg;
			struct scsimon_host_list * sm_hlop; 
			size_t sz = sizeof(struct scsimon_host_list);

			result = verify_area(VERIFY_WRITE, (void *)arg, sz);
				if (result) return result;
	    		__get_user(val, &sm_hlp->max_num);
			if (val > 1) {
				sz += ((val - 1) * 
				      sizeof(struct scsimon_host));
				result = verify_area(VERIFY_WRITE, 
						     (void *)arg, sz);
				if (result) return result;
			}
			sm_hlop = (struct scsimon_host_list *)
				  kmalloc(sz, GFP_ATOMIC);
			if (NULL == sm_hlop)
				return -ENOMEM;
			memset(sm_hlop, 0, sz);
			sm_hlop->max_num = val;
			__get_user(sm_hlop->host, &sm_hlp->host);
			__get_user(sm_hlop->flags, &sm_hlp->flags);
			sm_read_host_list(sm_hlop);
			__copy_to_user((void *)arg, sm_hlop, sz);
			kfree((char *)sm_hlop);
			return 0;
		}
	default:
		return -EINVAL;
	}
}

/* Use POLLIN for attach, POLLHUP for detach and a read() clears flags */
static unsigned int scsimon_poll(struct file * filp, poll_table * wait)
{
	unsigned int res = 0;
	Sm_fd * sfp;

	if (! (sfp = (Sm_fd *)filp->private_data))
		return POLLERR;
	if (sm_flag_attached)
		res = POLLIN | POLLRDNORM;
	if (sm_flag_detached)
		res |= POLLHUP;
	SCSI_LOG_TIMEOUT(3, printk("scsimon_poll: res=0x%x\n", (int)res));
	return res;
}

static int scsimon_fasync(int fd, struct file * filp, int mode)
{
	int retval;
	Sm_fd * sfp;

	SCSI_LOG_TIMEOUT(3, printk("scsimon_fasync: mode=%d\n", mode));
	if (! (sfp = (Sm_fd *)filp->private_data))
		return -ENXIO;
	retval = fasync_helper(fd, filp, mode, &sfp->async_qp);
	return (retval < 0) ? retval : 0;
}

static int scsimon_detect(Scsi_Device * scsidp)
{
	return 1;
}

/* Driver initialization */
static int scsimon_init()
{
	SCSI_LOG_TIMEOUT(3, printk("scsimon_init\n"));
	return 0;
}

static int scsimon_attach(Scsi_Device * scsidp)
{
	unsigned long iflags;
	Sm_attached * sap;
	Sm_hotinfo shi;
	Scsi_Device * sdp;

	SCSI_LOG_TIMEOUT(3, printk("scsimon_attach: scsidp=%p\n", scsidp));
	write_lock_irqsave(&scsimon_lock, iflags);
	sap = (Sm_attached *)kmalloc(sizeof(Sm_attached), GFP_ATOMIC);
	if (NULL == sap) {
		scsidp->attached--;
		write_unlock_irqrestore(&scsimon_lock, iflags);
		printk(KERN_WARNING "scsimon_attach: no kernel memory\n");
		return 1;
	}
	memset(sap, 0, sizeof(Sm_attached));
	memset(&shi, 0, sizeof(Sm_hotinfo));
	sm_event_count++;
	sap->event_count = sm_event_count;
	shi.action = 0;		/* attach */
	shi.event_attached = sap->event_count;
	sap->time_attached = jiffies;
	sap->device = scsidp;
	sap->host = scsidp->host->host_no;
	shi.host = sap->host;
	sap->bus = scsidp->channel;
	shi.bus = sap->bus;
	sap->target = scsidp->id;
	shi.target = sap->target;
	sap->lun = scsidp->lun;
	shi.lun = sap->lun;
	sdp = sap->device;
	if (sdp->vendor)
		memcpy(shi.vendor, sdp->vendor, sizeof(shi.vendor));
	if (sdp->model)
		memcpy(shi.model, sdp->model, sizeof(shi.model));
	if (sdp->rev)
		memcpy(shi.rev, sdp->rev, sizeof(shi.rev));
	shi.scsi_type = sdp->type;
	shi.removable = sdp->removable;
	shi.emulated = sdp->host->hostt->emulated;

	sap->next_att = sm_att_rootp;
	sm_att_rootp = sap;   /* prepend to attached list */

	sm_flag_attached = 1;
	scsimon_template.nr_dev++;	/* currently attached devices */
	scsimon_template.dev_noticed++;/* total attachment count */
	write_unlock_irqrestore(&scsimon_lock, iflags);
	sm_call_policy(&shi);
	return 0;
}

static void scsimon_finish(void)
{
	SCSI_LOG_TIMEOUT(3, printk("scsimon_finish\n"));
}

static void scsimon_detach(Scsi_Device * scsidp)
{
	unsigned long iflags;
	Sm_attached * sap;
	Sm_attached * psap = NULL;
	Sm_detached * sep;
	Scsi_Device * sdp;
	Sm_hotinfo shi;

	SCSI_LOG_TIMEOUT(3, printk("scsimon_detach: scsidp=%p\n", scsidp));
	write_lock_irqsave(&scsimon_lock, iflags);
	for (sap = sm_att_rootp; sap; sap = sap->next_att) {
		if (scsidp == sap->device)
			break;
		psap = sap;
	}
	sep = (Sm_detached *)kmalloc(sizeof(Sm_detached), GFP_ATOMIC);
	if ((NULL == sap) || (NULL == sep)) {
		write_unlock_irqrestore(&scsimon_lock, iflags);
		printk(KERN_WARNING "scsimon_detach: missing attach or "
			"no kernel memory\n");
		return;
	}
	memset(sep, 0, sizeof(Sm_detached));
	memset(&shi, 0, sizeof(Sm_hotinfo));
	shi.action = 1;		/* flags detach */
	sm_event_count++;
	/* xfer from alist element to new dlist element */
	sep->host = sap->host;
	shi.host = sep->host;
	sep->bus = sap->bus;
	shi.bus = sep->bus;
	sep->target = sap->target;
	shi.target = sep->target;
	sep->lun = sap->lun;
	shi.lun = sep->lun;
	sdp = sap->device;
	if (sdp->vendor)
		memcpy(sep->vendor, sdp->vendor, sizeof(sep->vendor));
	memcpy(shi.vendor, sep->vendor, sizeof(shi.vendor));
	if (sdp->model)
		memcpy(sep->model, sdp->model, sizeof(sep->model));
	memcpy(shi.model, sep->model, sizeof(shi.model));
	if (sdp->rev)
		memcpy(sep->rev, sdp->rev, sizeof(sep->rev));
	memcpy(shi.rev, sep->rev, sizeof(shi.rev));
	if (sdp->host->hostt->name)
		strncpy(sep->host_name, sdp->host->hostt->name, 
			sizeof(sep->host_name));
	if (sdp->host->hostt->info)
		strncpy(sep->host_info, sdp->host->hostt->info(sdp->host),
			sizeof(sep->host_info));
	sep->scsi_type = sdp->type;
	shi.scsi_type = sep->scsi_type;
	sep->scsi_level = sdp->scsi_level;
	if (sep->scsi_level > 1)
		--sep->scsi_level;
	sep->removable = sdp->removable;
	shi.removable = sep->removable;
	sep->emulated = sdp->host->hostt->emulated;
	shi.emulated = sep->emulated;
	sep->event_count = sm_event_count;
	shi.event_detached = sep->event_count;
	sep->event_attached = sap->event_count;
	shi.event_attached = sep->event_attached;
	sep->time_detached = jiffies;

	if (NULL == psap)
		sm_att_rootp = sap->next_att;
	else
		psap->next_att = sap->next_att;
	kfree((char *)sap);
	sep->next_det = sm_det_rootp;
	sm_det_rootp = sep;   /* prepend to detached list */

	sm_flag_detached = 1;
	scsidp->attached--;
	scsimon_template.nr_dev--;	/* currently attached devices */
	scsimon_template.dev_max++;	/* total detachment count */

	sm_trunc_dlist(sm_max_detached_elems);
	write_unlock_irqrestore(&scsimon_lock, iflags);
	sm_call_policy(&shi);
	return;
}

static void sm_trunc_alist(int fromPos) 
{
	Sm_attached * sap = sm_att_rootp;
	Sm_attached * psap = NULL;
	Sm_attached * nsap;
	int k;

	for (k = 0; ((k < fromPos) && sap); ++k, sap = sap->next_att)
		psap = sap;
	if (sap) {
		while (sap) {
    			nsap = sap->next_att;
    			kfree((char *)sap);
    			sap = nsap;
		}
		if (NULL == psap)
    			sm_att_rootp = NULL;
		else
    			psap->next_att = NULL;
	}
}

static void sm_trunc_dlist(int fromPos)
{
	Sm_detached * sep = sm_det_rootp;
	Sm_detached * psep = NULL;
	Sm_detached * nsep;
	int k;

	for (k = 0; ((k < fromPos) && sep); ++k, sep = sep->next_det)
		psep = sep;
	if (sep) {
		while (sep) {
			nsep = sep->next_det;
			kfree((char *)sep);
			sep = nsep;
		}
		if (NULL == psep)
			sm_det_rootp = NULL;
		else
			psep->next_det = NULL;
	}
}

static Sm_fd * sm_add_sfp()
{
	Sm_fd * sfp;

	if ((sfp = (Sm_fd *)kmalloc(sizeof(Sm_fd), GFP_ATOMIC)))
		memset(sfp, 0, sizeof(Sm_fd));
	return sfp;
}

static void sm_remove_sfp(Sm_fd * sfp)
{
	kfree((char *)sfp);
}

static void sm_read_state(struct scsimon_state * ssp)
{
	Sm_detached * sep;
	struct Scsi_Host * shp;
	int k;
	unsigned int high_host = 0;
	unsigned long iflags;

	read_lock_irqsave(&scsimon_lock, iflags);
	jiffies_to_timespec(sm_init_time, &ssp->init_time);
	ssp->devices_attached = scsimon_template.nr_dev;
	ssp->total_attachs = scsimon_template.dev_noticed;
	ssp->total_detachs = scsimon_template.dev_max;
	for (k = 0, sep = sm_det_rootp; sep; k++, sep = sep->next_det)
		;
	ssp->detach_list_len = (unsigned int)k;
	ssp->event_count = sm_event_count;
	ssp->lock_mod_state = sm_locked_down;
	for (k = 0, shp = scsi_hostlist; shp; shp = shp->next, ++k) {
		if (shp->host_no > high_host)
			high_host = shp->host_no;
	}
	ssp->count_hosts = k;
	ssp->high_host = high_host;
	read_unlock_irqrestore(&scsimon_lock, iflags);
}

static void sm_read_att_elem(struct scsimon_att_dev * sm_ap, 
			     Sm_attached * sap)
{
	Scsi_Device * sdp = sap->device;

	sm_ap->att_event = sap->event_count;
	jiffies_to_timespec(sap->time_attached, &sm_ap->att_time);
	sm_ap->d.host = sap->host;
	sm_ap->d.bus = sap->bus;
	sm_ap->d.target = sap->target;
	sm_ap->d.lun = sap->lun;
	memcpy(sm_ap->d.vendor, sdp->vendor, sizeof(sm_ap->d.vendor));
	memcpy(sm_ap->d.model, sdp->model, sizeof(sm_ap->d.model));
	memcpy(sm_ap->d.rev, sdp->rev, sizeof(sm_ap->d.rev));
	sm_ap->d.scsi_type = sdp->type;
	sm_ap->d.scsi_level = sdp->scsi_level;
	if (sm_ap->d.scsi_level > 1)
		--sm_ap->d.scsi_level;
	sm_ap->d.removable = sdp->removable;
	sm_ap->d.emulated = sdp->host->hostt->emulated;
}

static void sm_read_att_list(struct scsimon_att_list * alp)
{
	Sm_attached * sap;
	int k;
	unsigned long iflags;

	read_lock_irqsave(&scsimon_lock, iflags);
	alp->curr_event = sm_event_count;
	for (k = 0, sap = sm_att_rootp; (k < alp->max_num) && sap; 
	     sap = sap->next_att) {
		if (alp->match_event > 0) {
			if (alp->match_event == sap->event_count) {
				k = 1;
				sm_read_att_elem(&alp->arr[0], sap);
				break;
			}
			continue;
		}
		sm_read_att_elem(&alp->arr[k], sap);
		k++;
	}
	alp->num_out = k;
	read_unlock_irqrestore(&scsimon_lock, iflags);
}

static void sm_read_det_elem(struct scsimon_det_dev * sm_dp, 
			     Sm_detached * sep)
{
	sm_dp->att_event = sep->event_attached;
	sm_dp->det_event = sep->event_count;
	jiffies_to_timespec(sep->time_detached, &sm_dp->det_time);
	memcpy(sm_dp->host_name, sep->host_name, sizeof(sm_dp->host_name));
	memcpy(sm_dp->host_info, sep->host_info, sizeof(sm_dp->host_info));
	sm_dp->d.host = sep->host;
	sm_dp->d.bus = sep->bus;
	sm_dp->d.target = sep->target;
	sm_dp->d.lun = sep->lun;
	memcpy(sm_dp->d.vendor, sep->vendor, sizeof(sm_dp->d.vendor));
	memcpy(sm_dp->d.model, sep->model, sizeof(sm_dp->d.model));
	memcpy(sm_dp->d.rev, sep->rev, sizeof(sm_dp->d.rev));
	sm_dp->d.scsi_type = sep->scsi_type;
	sm_dp->d.scsi_level = sep->scsi_level;
	sm_dp->d.removable = sep->removable;
	sm_dp->d.emulated = sep->emulated;
}

static void sm_read_det_list(struct scsimon_det_list * dlp)
{
	Sm_detached * sep;
	int k;
	unsigned long iflags;

	read_lock_irqsave(&scsimon_lock, iflags);
	dlp->curr_event = sm_event_count;
	for (k = 0, sep = sm_det_rootp; (k < dlp->max_num) && sep; 
	     sep = sep->next_det) {
		if (dlp->match_event > 0) {
			if (dlp->match_event == sep->event_attached) {
				k = 1;
				sm_read_det_elem(&dlp->arr[0], sep);
				break;
			}
			continue;
		}
		sm_read_det_elem(&dlp->arr[k], sep);
		k++;
	}
	dlp->num_out = k;
	read_unlock_irqrestore(&scsimon_lock, iflags);
}

static void sm_read_host_elem(struct scsimon_host * sm_hp, 
			      struct Scsi_Host * shp)
{
	sm_hp->host = shp->host_no;
	sm_hp->host_scsi_id = shp->this_id;
	sm_hp->emulated = shp->hostt->emulated;
	sm_hp->is_module = shp->loaded_as_module;
	if (shp->hostt->name)
		strncpy(sm_hp->host_name, shp->hostt->name, 
			sizeof(sm_hp->host_name));
	if (shp->hostt->info)
		strncpy(sm_hp->host_info, shp->hostt->info(shp), 
			sizeof(sm_hp->host_info));
}

static void sm_read_host_list(struct scsimon_host_list * hlp)
{
	struct Scsi_Host * shp;
	int k;
	unsigned long iflags;
	int match;

	match = (SCSIMON_HOST_FLAG_MATCH & hlp->flags) ? 1 : 0;
	read_lock_irqsave(&scsimon_lock, iflags);
	hlp->curr_event = sm_event_count;
	for (k = 0, shp = scsi_hostlist; shp && (k < hlp->max_num); 
	     shp = shp->next) {
		if (match) {
			if (shp->host_no == hlp->host) {
				sm_read_host_elem(&hlp->arr[0], shp);
				k = 1;
				break;
			}
			continue;
		}
		sm_read_host_elem(&hlp->arr[k], shp);
		k++;
	}
	hlp->num_out = k;
	read_unlock_irqrestore(&scsimon_lock, iflags);
}

#ifdef CONFIG_HOTPLUG 

static void sm_call_policy(Sm_hotinfo * hip)
{
	char *argv [3], **envp, *buf, *scratch;
	int k = 0, value;

	if (!hotplug_path [0])
		return;
	if (in_interrupt ()) {
		SCSI_LOG_TIMEOUT(1, printk("sm_call_policy: in_interrupt\n"));
		return;
	}
	if (! current->fs->root) {
		SCSI_LOG_TIMEOUT(1, printk("sm_call_policy: no root fs\n"));
		return;
	}
	if (!(envp = (char **) kmalloc (20 * sizeof (char *), GFP_KERNEL))) {
		SCSI_LOG_TIMEOUT(1, printk("sm_call_policy: ENOMEM\n"));
		return;
	}
	if (!(buf = kmalloc (512, GFP_KERNEL))) {
		kfree (envp);
		SCSI_LOG_TIMEOUT(1, printk("sm_call_policy: ENOMEM\n"));
		return;
	}

	/* only one standardized param to hotplug command: type */
	argv [0] = hotplug_path;
	argv [1] = "scsi";
	argv [2] = 0;

	/* minimal command environment */
	envp [k++] = "HOME=/";
	envp [k++] = "PATH=/sbin:/bin:/usr/sbin:/usr/bin";

	/* extensible set of named bus-specific parameters,
	 * supporting multiple driver selection algorithms.
	 */
	scratch = buf;

	/* action:  add, remove */
	envp [k++] = scratch;
	scratch += sprintf (scratch, "ACTION=%s", 
			    hip->action ? "remove" : "add") + 1;

	/* per-device configuration hacks are common */
	envp [k++] = scratch;
	scratch += sprintf (scratch, "SCSI_HOST=%u", hip->host) + 1;
	envp [k++] = scratch;
	scratch += sprintf (scratch, "SCSI_BUS=%u", hip->bus) + 1;
	envp [k++] = scratch;
	scratch += sprintf (scratch, "SCSI_TARGET=%u", hip->target) + 1;
	envp [k++] = scratch;
	scratch += sprintf (scratch, "SCSI_LUN=%u", hip->lun) + 1;
	envp [k++] = scratch;
	scratch += sprintf (scratch, "SCSI_VENDOR=%.8s", hip->vendor) + 1;
	envp [k++] = scratch;
	scratch += sprintf (scratch, "SCSI_MODEL=%.16s", hip->model) + 1;
	envp [k++] = scratch;
	scratch += sprintf (scratch, "SCSI_REV=%.4s", hip->rev) + 1;
	envp [k++] = scratch;
	scratch += sprintf (scratch, "SCSI_TYPE=%d", hip->scsi_type) + 1;
	envp [k++] = scratch;
	scratch += sprintf (scratch, "SCSI_REMOVABLE=%d", hip->removable) + 1;
	envp [k++] = scratch;
	scratch += sprintf (scratch, "SCSI_EMULATED=%d", hip->emulated) + 1;
	envp [k++] = scratch;
	scratch += sprintf (scratch, "SCSI_EVENT_ATTACHED=%lu", 
			    hip->event_attached) + 1;
	if (1 == hip->action) {
		envp [k++] = scratch;
		scratch += sprintf (scratch, "SCSI_EVENT_DETACHED=%lu", 
				    hip->event_detached) + 1;
	}

	envp [k++] = 0;
	/* assert: (scratch - buf) < sizeof buf */

	/* NOTE: user mode daemons can call the agents too */

	value = call_usermodehelper (argv [0], argv, envp);
	kfree (buf);
	kfree (envp);
	if (value != 0) {
		SCSI_LOG_TIMEOUT(1, printk("sm_call_policy: "
			"call_usermodehelper returned %d\n", value));
	}
}

#else

static void sm_call_policy (Sm_hotinfo * hip) { }

#endif

MODULE_AUTHOR("Douglas Gilbert");
MODULE_DESCRIPTION("SCSI monitor (scsimon) driver");

static int __init init_scsimon(void) 
{
	int res;

	sm_init_time = jiffies;
	scsimon_template.module = THIS_MODULE;
	res = scsi_register_module(MODULE_SCSI_DEV, &scsimon_template);
	misc_register(&scsimon_miscdev);
	return res;
}

static void __exit exit_scsimon( void)
{
	misc_deregister(&scsimon_miscdev);
	scsi_unregister_module(MODULE_SCSI_DEV, &scsimon_template);
	sm_trunc_alist(0);
	sm_trunc_dlist(0);
}

module_init(init_scsimon);
module_exit(exit_scsimon);

MODULE_LICENSE("GPL");
