/*
 *  linux/fs/supermount/proc.c
 *
 *  Initial version for kernel 2.4.21 (C) 2003 Andrey Borzenkov
 *                                             (arvidjaar@mail.ru)
 *
 *  $Id: proc.c,v 1.8 2003/08/09 11:11:36 bor Exp $
 */

#include "supermount.h"

#ifdef CONFIG_PROC_FS

/*
 * we use semaphore because we have to lock subfs inside and it can
 * sleep. Unlocking procfs list would mean adding generic usage count
 * management to sbi and this is far too fetching
 */

static DECLARE_MUTEX(supermount_proc_sem);
static struct proc_dir_entry *supermount_proc_root;
static struct proc_dir_entry *supermount_proc_subfs;
static struct proc_dir_entry *supermount_proc_version;
static struct supermount_sb_info *supermount_list_head;

#define SKIP_BLANKS(s)	while(*s == ' ' || *s == '\t' || *s == '\n') s++
#define CHECK_COUNT do { \
	size += n; \
	count -= n; \
	if (count <= 0) { \
		subfs_unlock(sbi->host); \
		break; \
	} \
} while (0)

/* iterator; shamelessly copied over from pci.c */
static void *supermount_seq_start(struct seq_file *sf, loff_t *pos)
{
        struct supermount_sb_info *sbi;
        loff_t n = *pos;

	down(&supermount_proc_sem);

        sbi = supermount_list_head;
        while (n-- && sbi)
                sbi = sbi->next;

        return sbi;
}

static void *supermount_seq_next(struct seq_file *sf, void *v, loff_t *pos)
{
        struct supermount_sb_info *sbi = v;
        (*pos)++;
        return sbi->next;
}

static void supermount_seq_stop(struct seq_file *sf, void *v)
{
	up(&supermount_proc_sem);
}


static int
supermount_show_sbi(struct seq_file *sf, void *v)
{
	struct supermount_sb_info *sbi = v;


	subfs_lock(sbi->host);
	seq_puts(sf, sbi->devname);
	if (sbi->disabled)
		seq_puts(sf, " disabled\n");
	else if (subfs_is_mounted(sbi->host))
		seq_printf(sf, " mounted %d %d\n",
				sbi->readcount, sbi->writecount);
	else
		seq_puts(sf, " unmounted\n");
	subfs_unlock(sbi->host);

	return 0;
}

static struct seq_operations supermount_proc_subfs_op = {
        start:  supermount_seq_start,
        next:   supermount_seq_next,
        stop:   supermount_seq_stop,
        show:   supermount_show_sbi
};

static int supermount_proc_subfs_open(struct inode *inode, struct file *file)
{
        return seq_open(file, &supermount_proc_subfs_op);
}

/*
 * mostly copied over from drivers/scsi/scsi.c:proc_scsi_gen_write()
 */
static int
supermount_proc_subfs_write(struct file *file, const char *buf, size_t length, loff_t *offset)
{
	char *buffer, *s, *dev = 0;
	int disable = 0, enable = 0, release = 0, force = 0;
	struct supermount_sb_info *sbi;
	size_t rc;

	rc = -EINVAL;
	if (!buf || length > PAGE_SIZE)
		goto out;

	rc = -ENOMEM;
	if (!(s = buffer = (char *)__get_free_page(GFP_KERNEL)))
		goto out;

	rc =-EFAULT;
	if(copy_from_user(buffer, buf, length))
		goto free_buffer;

	rc = -EINVAL;
	if (length < PAGE_SIZE)
		buffer[length] = '\0';
	else if (buffer[PAGE_SIZE-1])
		goto free_buffer;

	/*
	 * echo "/dev/cdrom [enable|disable] [release [force]]" > \
	 * 				/proc/fs/supermount/subfs
	 */

	do {
		char *p;

		SKIP_BLANKS(s);
		p = strpbrk(s, " \t\n");
		if (p)
			*p++ = '\0';
		if (!dev)
			dev = s;
		else if (!strcmp(s, "disable"))
			disable = 1;
		else if (!strcmp(s, "enable"))
			enable = 1;
		else if (!strcmp(s, "release"))
			release = 1;
		else if (!strcmp(s, "force"))
			force = 1;
		else
			goto free_buffer;

		s = p;
	} while (s && *s);

	if ((enable && disable) || (force && !release))
		goto free_buffer;

	down(&supermount_proc_sem);
	for(sbi = supermount_list_head; sbi; sbi = sbi->next) {
		if (strcmp(sbi->devname, dev))
			continue;

		subfs_lock(sbi->host);

		rc = length;
		if (release && subfs_is_mounted(sbi->host)) {
			if (!subfs_is_busy(sbi->host) || force)
				subfs_umount(sbi->host, SUBFS_UMNT_USER);
			else
				rc = -EBUSY;
		}
		
		if (disable && subfs_is_mounted(sbi->host))
			rc = -EBUSY;

		if (rc >= 0) {
			if (disable)
				sbi->disabled = 1;
			else if (enable)
				sbi->disabled = 0;
		}

		subfs_unlock(sbi->host);
		break;
	}
	up(&supermount_proc_sem);

free_buffer:
	free_page((unsigned long)buffer);
out:
	return rc;

}

static struct file_operations supermount_proc_subfs_operations = {
        open:           supermount_proc_subfs_open,
        read:           seq_read,
        llseek:         seq_lseek,
        release:        seq_release,
	write:		supermount_proc_subfs_write,
};

static int
supermount_proc_version_read(char *page, char **start, off_t pos, int count, int *eof, void *data)
{
	int rc;

	rc = snprintf(page, count, "Supermount version %s for kernel 2.4\n",
				    SUPERMOUNT_VERSION);
	*eof = 1;
	return rc;
}

void
supermount_proc_register(void)
{
	supermount_proc_root = proc_mkdir("fs/supermount", 0);
	if (!supermount_proc_root) {
		printk(KERN_ERR "SUPERMOUNT failed to create /proc/fs/supermount");
		return;
	}
	SET_MODULE_OWNER(supermount_proc_root);

	supermount_proc_version = create_proc_read_entry("version",
                                        S_IFREG | S_IRUGO | S_IWUSR,
                                        supermount_proc_root,
					supermount_proc_version_read,
					0);

	if (supermount_proc_version)
		SET_MODULE_OWNER(supermount_proc_version);
	else
		printk(KERN_ERR
		"SUPERMOUNT failed to create /proc/fs/supermount/version");

	supermount_proc_subfs = create_proc_entry("subfs",
                                        S_IFREG | S_IRUGO,
                                        supermount_proc_root);

	if (supermount_proc_subfs) {
		supermount_proc_subfs->proc_fops =
			&supermount_proc_subfs_operations;
		SET_MODULE_OWNER(supermount_proc_subfs);
	} else
		printk(KERN_ERR
		"SUPERMOUNT failed to create /proc/fs/supermount/subfs");

}

void
supermount_proc_unregister(void)
{
	remove_proc_entry("fs/supermount/subfs", 0);
	remove_proc_entry("fs/supermount/version", 0);
	remove_proc_entry("fs/supermount", 0);
}

void
supermount_proc_insert(struct supermount_sb_info *sbi)
{

	down(&supermount_proc_sem);

	sbi->next = supermount_list_head;
	supermount_list_head = sbi;

	up(&supermount_proc_sem);
}

void
supermount_proc_remove(struct supermount_sb_info *sbi)
{
	struct supermount_sb_info **p, *q;

	down(&supermount_proc_sem);

	for(p = &supermount_list_head, q = supermount_list_head;
			q; p = &q->next, q = q->next)
		if (q == sbi)
			break;

	if (q)
		*p = q->next;

	up(&supermount_proc_sem);

	SUPERMOUNT_BUG_ON(!q);
}
#endif /* CONFIG_PROC_FS */
