#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/mm.h>
#include <linux/vmalloc.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/tty.h>
#include <linux/pagemap.h>
#include <linux/times.h>
#include <linux/bitops.h>
#include <linux/version.h>
#include <linux/list.h>
#include <asm/uaccess.h>


#include <linux/procinfo.h>

#define PROCINFO_MAJOR  250

#define BITS_PER_PAGE (PAGE_SIZE * 8)
#define PAGE_NR (64*1024 / BITS_PER_PAGE)
#define MAX_TASKS 512
#define BITS_PER_PAGE (PAGE_SIZE * 8)


#ifndef next_task
#define next_task(p) (p->next_task)
#endif

#ifndef for_each_process
#define for_each_process(p) for_each_task(p)
#endif

#ifdef DEF_COUNTER
#define task_nice(p) (p->nice)
#define task_prio(p) (20 - (p->counter * 10 + DEF_COUNTER / 2) / DEF_COUNTER)
#endif


#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0)
#define SIGLOCK(p) (p->sig->siglock)
#else
#define SIGLOCK(p) (p->sigmask_lock)
#endif


struct procinfo_kfilter {
        struct list_head list;
        struct procinfo_filter filter;
};


struct procinfo_session {        
        int time;
        int reclen;
        int cmdline_len;
        int env_len;
        pid_t task[MAX_TASKS];
        int tasks;
        int last_pid;
        int last_started;        
        unsigned long flags;
        char * buf;
        unsigned char * bitmap[PAGE_NR];
        int lookups, last_missed;
        struct list_head filter;
};


int pi_access_process_vm (struct mm_struct * mm, unsigned long addr, void *buf, int len, int write)
{
	struct vm_area_struct *vma;
	struct page *page;
	void *old_buf = buf;

	/* ignore errors, just check how much was sucessfully transfered */
	while (len) {
		int bytes, ret, offset;
		void *maddr;

		ret = get_user_pages (current, mm, addr, 1, write, 1, &page, &vma);
		if (ret <= 0)
			break;

		bytes = len;
		offset = addr & (PAGE_SIZE-1);
		if (bytes > PAGE_SIZE-offset)
			bytes = PAGE_SIZE-offset;

		flush_cache_page(vma, addr);

		maddr = kmap(page);
		if (write) {
			memcpy(maddr + offset, buf, bytes);
			flush_page_to_ram(page);
			flush_icache_user_range(vma, page, addr, len);
		} else {
			memcpy(buf, maddr + offset, bytes);
			flush_page_to_ram(page);
		}
		kunmap(page);
		put_page(page);
		len -= bytes;
		buf += bytes;
		addr += bytes;
	}	
	return buf - old_buf;
}


static int one_filter_task (struct procinfo_kfilter * f, struct task_struct * p)
{
        switch (f->filter.type) {
                        
        case PROCINFO_FLT_COMM:
                if (!strcmp (f->filter.arg.comm, p->comm))
                        return 1;
                break;

        case PROCINFO_FLT_RGID:
                if (f->filter.arg.uint16 == p->gid)
                        return 1;
                break;
                        
        case PROCINFO_FLT_RUID:
                if (f->filter.arg.uint16 == p->uid)
                        return 1;
                break;
                        
        case PROCINFO_FLT_EGID:
                if (f->filter.arg.uint16 == p->egid)
                        return 1;
                break;
                        
        case PROCINFO_FLT_EUID:
                if (f->filter.arg.uint16 == p->euid)
                        return 1;
                break;
                        
        case PROCINFO_FLT_SID:
                if (f->filter.arg.uint16 == p->session)
                        return 1;
                break;
                        
        case PROCINFO_FLT_PID:
                if (f->filter.arg.uint16 == p->pid)
                        return 1;
                break;
                        
        case PROCINFO_FLT_TTY:
                if (p->tty != NULL && f->filter.arg.uint16 == p->tty->device)
                        return 1;
                break;
                        
        default:
                printk (KERN_ALERT "procinfo: unknown filter %u\n", f->filter.type);
        }

        return 0;
}

static int filter_task_or (struct procinfo_session * pis, struct task_struct * p)
{
        struct list_head * cur;
        struct procinfo_kfilter * f;

        if (list_empty (&pis->filter))
                return 1;
        
        list_for_each (cur, &pis->filter) {
                f = list_entry (cur, struct procinfo_kfilter, list);
                
                if (one_filter_task (f, p))
                        return 1;
        }
        
        return 0;
}


static int filter_task_and (struct procinfo_session * pis, struct task_struct * p)
{
        struct list_head * cur;
        struct procinfo_kfilter * f;

        if (list_empty (&pis->filter))
                return 0;
        
        list_for_each (cur, &pis->filter) {
                f = list_entry (cur, struct procinfo_kfilter, list);
                
                if (!one_filter_task (f, p))
                        return 0;
        }
        
        return 1;
}


static int filter_task (struct procinfo_session * pis, struct task_struct * p)
{
        int res;
        
        if (pis->flags & PI_AND_FILTER)
                res = filter_task_and (pis, p);
        else
                res = filter_task_or (pis, p);

        if ((pis->flags & PI_RUN_FILTER) && (p->state != TASK_RUNNING))
                res = 0;
        
        if (pis->flags & PI_NEGATE_FILTER)
                res = !res;
        
        return res;
}


inline int is_task_retreived (struct procinfo_session * pis, int pid)
{
        int page_nr = pid / BITS_PER_PAGE;
        int bit_nr = pid % BITS_PER_PAGE;

        return test_bit (bit_nr, (unsigned long *) pis->bitmap[page_nr]);
}


inline void set_task_retreived (struct procinfo_session * pis, int pid)
{
        int page_nr = pid / BITS_PER_PAGE;
        int bit_nr = pid % BITS_PER_PAGE;

        if (test_bit (bit_nr, (unsigned long *) pis->bitmap[page_nr]))
                printk ("BIT ALREADY SET for %d\n", pid);
        set_bit (bit_nr, (unsigned long *) pis->bitmap[page_nr]);
}


struct task_struct * look_for_next_pid (struct procinfo_session * pis, int last, unsigned long start)
{
        struct task_struct * p;

        pis->lookups++;
        
        p = find_task_by_pid (last);
        if (p != NULL && p->start_time == start) {
                if (next_task (p) == &init_task)
                        return NULL;
                return next_task (p);
        }

        pis->last_missed++;
        
        /* looking for last timestamp */
        for_each_process (p) {
                if (p->start_time >= start)
                        break;
        }

        /* there is no fresh tasks */
        if (p == &init_task)
                return NULL;

        /* scanning for non-retreived task */
        while (p != &init_task) {
                if (! is_task_retreived (pis, p->pid))
                        return p;
                p = next_task (p);
        }
        
        /* there is no fresh tasks */
        return NULL;
}


static void fill_vm_info (struct procinfo_session * pis, struct mm_struct * mm, struct procinfo_process * pi_p)
{
        struct vm_area_struct * vma;

        if (mm == NULL)
                return;
        
        down_read(&mm->mmap_sem);
                        
        if ((pis->flags & PI_GET_CMDLINE_INFO) && (pis->cmdline_len > 0)) {
                int res, len = mm->arg_end - mm->arg_start;
                char * cmdline = ((char *) pi_p) + sizeof (*pi_p);
                                
                if (len > (PAGE_SIZE - sizeof (*pi_p)))
                        len = PAGE_SIZE - sizeof (*pi_p);
                res = pi_access_process_vm (mm, mm->arg_start, cmdline, len, 0);
                if (res > 0 && cmdline[res-1] != '\0') {
                        len = strnlen (cmdline, res);
                        if ( len < res )
                                res = len;
                        else {
                                len = mm->env_end - mm->env_start;
                                if (len > pis->cmdline_len - res)
                                        len = pis->cmdline_len - res;
                                res += pi_access_process_vm (mm, mm->env_start, cmdline+res, len, 0);
                        }
                }
                        
                pi_p->cmdline_len = pis->cmdline_len < res ? pis->cmdline_len : res;
        }
                        
        if (pis->flags & (PI_GET_VM_INFO | PI_GET_PAGE_INFO | PI_GET_ENV_INFO)) {                                
                for (vma = mm->mmap; vma; vma = vma->vm_next) {
                        unsigned long len = (vma->vm_end - vma->vm_start) >> 10;
                        if (!vma->vm_file) {
                                pi_p->vmdata += len;
                                if (vma->vm_flags & VM_GROWSDOWN)
                                        pi_p->vmstack += len;
                                continue;
                        }
                        if (vma->vm_flags & VM_WRITE)
                                continue;
                        if (vma->vm_flags & VM_EXEC) {
                                pi_p->vmexec += len;
                                if (vma->vm_flags & VM_EXECUTABLE)
                                        continue;
                                pi_p->vmlib += len;
                        }
                }
                pi_p->vmsize = mm->total_vm << (PAGE_SHIFT-10);
                pi_p->vmlocked = mm->locked_vm << (PAGE_SHIFT-10);
                pi_p->vmrss = mm->rss << (PAGE_SHIFT-10);
                pi_p->vmexec -= pi_p->vmlib;
                pi_p->vmdata -= pi_p->vmstack;
        }

        up_read (&mm->mmap_sem);
        mmput (mm);
}


static int fill_info (struct procinfo_session * pis, struct task_struct * p, char * buf)
{
        struct procinfo_process * pi_p;;

        pi_p = (struct procinfo_process *) pis->buf;

        pi_p->state = p->state;
        pi_p->flags = p->flags;
        pi_p->ptrace = p->ptrace;
        pi_p->nice = task_nice (p);
        pi_p->priority = task_prio (p);
        pi_p->pid = p->pid;
        
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0)
        pi_p->ppid = p->pid ? p->real_parent->pid : 0;
        pi_p->utime = jiffies_to_clock_t (p->utime);
        pi_p->stime = jiffies_to_clock_t (p->stime);
        pi_p->tty = p->tty? kdev_val (p->tty->device) : 0;
#else
        pi_p->ppid = p->pid ? p->p_opptr->pid : 0;
        pi_p->stime = p->times.tms_stime;
        pi_p->utime = p->times.tms_utime;
        pi_p->tty = p->tty ? p->tty->device : 0;
#endif
        pi_p->pgrp = p->pgrp;
        pi_p->session = p->session;
        pi_p->tgid = p->tgid;
        pi_p->leader = p->leader;
        pi_p->start_time = p->start_time;

        pi_p->it_real_value = p->it_real_value;
                
        pi_p->min_flt = p->min_flt;
        pi_p->maj_flt = p->maj_flt;
        pi_p->rss_lim = p->rlim[RLIMIT_RSS].rlim_cur;

        pi_p->wchan = get_wchan (p);
        pi_p->eip = KSTK_EIP (p);
        pi_p->esp = KSTK_ESP (p);
        pi_p->stack = p->mm != NULL ? p->mm->start_stack : 0;
                
        pi_p->uid = p->uid;
        pi_p->euid = p->euid;
        pi_p->suid = p->suid;
        pi_p->fsuid = p->fsuid;

        pi_p->gid = p->gid;
        pi_p->egid = p->egid;
        pi_p->sgid = p->sgid;
        pi_p->fsgid = p->fsgid;

        pi_p->ngroups = p->ngroups;
        memcpy (pi_p->groups, p->groups, sizeof (pi_p->groups));

        memcpy (pi_p->comm, p->comm, sizeof (pi_p->comm));

        pi_p->ignored = 0;
        pi_p->catch = 0;
        pi_p->blocked = 0;
        pi_p->pending = 0;

        if (pis->flags & PI_GET_SIGNAL_INFO) {
                struct k_sigaction *k;
                int i;
                                
                pi_p->blocked = p->blocked.sig[0] & 0x7fffffffUL;
                pi_p->pending = p->pending.signal.sig[0] & 0x7fffffffUL;
                                
                spin_lock_irq (&SIGLOCK(p));                                
                if (p->sig) {
                        k = p->sig->action;
                        for (i = 1; i <= _NSIG; ++i, ++k) {
                                if (k->sa.sa_handler == SIG_IGN)
                                        pi_p->ignored |= (1 << (i-1));
                                else if (k->sa.sa_handler != SIG_DFL)
                                        pi_p->catch |= (1 << (i-1));
                        }
                }
                spin_unlock_irq (&SIGLOCK(p));                                
        }

        pi_p->cmdline_len = 0;
                
        pi_p->vmsize = 0;
        pi_p->vmlocked = 0;
        pi_p->vmrss = 0;
        pi_p->vmrss = 0;
        pi_p->vmdata = 0;
        pi_p->vmstack = 0;
        pi_p->vmexec = 0;
        pi_p->vmlib = 0;

        if (p->mm != NULL && (pis->flags & (PI_GET_VM_INFO | PI_GET_PAGE_INFO | PI_GET_CMDLINE_INFO | PI_GET_ENV_INFO))) {
                atomic_inc(&p->mm->mm_users);

                task_unlock (p);

                fill_vm_info (pis, p->mm, pi_p);
        } else
                task_unlock (p);

        if (copy_to_user (buf, pi_p, pis->reclen))
                return -EFAULT;
        
        return 0;
}


static ssize_t procinfo_read(struct file * file, char * buf, size_t count, loff_t *ppos)
{
        unsigned long pnum, offs = 0, i;
        int copied = 0, res;
        struct procinfo_session * pis = file->private_data;
	struct task_struct *p;
        struct procinfo_process * pi_p;
        int last;
        unsigned long start;
        
        if (pis == NULL)
                return -EINVAL;

        last = pis->last_pid;
        start = pis->last_started;
        
        pnum = (unsigned long) *ppos / pis->reclen;
        offs = (unsigned long) *ppos % pis->reclen;        
        pi_p = (struct procinfo_process *) pis->buf;
        
        pis->tasks = 0;
        
	read_lock (&tasklist_lock);
        
        p = look_for_next_pid (pis, last, start);
        if (p == NULL) {
                /* no more tasks */
                read_unlock(&tasklist_lock);
                *ppos += copied;
                return copied;
        }

        /* collect cache */
        for (pis->tasks = 0; pis->tasks < (count / pis->reclen) && p != &init_task && pis->tasks < MAX_TASKS; pis->tasks++) {
                pis->task[pis->tasks] = p->pid;
                p = next_task (p);
        }
        
        for (i = 0; i < pis->tasks; i++) {
                p = find_task_by_pid (pis->task[i]);
                if (p != NULL) {
                        task_lock (p);
                        
                        last = p->pid;
                        start = p->start_time;
                        
                        read_unlock (&tasklist_lock);

                        if (filter_task (pis, p)) {
                                res = fill_info (pis, p, buf);
                                if (res)
                                        return res;

                                copied += pis->reclen;
                                buf += pis->reclen;
                        } else
                                task_unlock (p);
                        
                        set_task_retreived (pis, last);
                        
                        read_lock (&tasklist_lock);
                }
        }

        read_unlock (&tasklist_lock);

        pis->last_pid = last;
        pis->last_started = start;
        
        *ppos += copied;
        return copied;
}


static int procinfo_open(struct inode * inode, struct file * filp)
{
        struct procinfo_session * pis;
        int i;
        
        filp->private_data = pis = kmalloc (sizeof (struct procinfo_session), GFP_KERNEL);
        if (pis == NULL)
                return -ENOMEM;
        
        pis->buf = (char *) __get_free_page (GFP_KERNEL);
        if (pis->buf == NULL) {
                kfree (filp->private_data);
                return -ENOMEM;
        }

        for (i = 0; i < PAGE_NR; i++) {
                pis->bitmap[i] = (char *) __get_free_page (GFP_KERNEL);
                if (pis->bitmap[i] == NULL) {
                        while (--i >= 0)
                                free_page ((unsigned long) pis->bitmap[i]);
                        kfree (filp->private_data);
                        return -ENOMEM;
                }
                memset (pis->bitmap[i], 0, PAGE_SIZE);
        }
                
        pis->cmdline_len = 0;
        pis->env_len = 0;
        pis->reclen = sizeof (struct procinfo_process) + pis->cmdline_len + pis->env_len;
        
        pis->flags = 0;        
        pis->last_pid = 0;
        pis->last_started = 0;

        pis->lookups = 0;
        pis->last_missed = 0;

        INIT_LIST_HEAD (&pis->filter);
        
        filp->f_pos = 0;

        return 0;
}


static void procinfo_release_filters (struct procinfo_session * pis)
{
        struct list_head * cur, * tmp;
        list_for_each_safe (cur, tmp, &pis->filter) {
                struct procinfo_kfilter * f = list_entry (cur, struct procinfo_kfilter, list);                
                kfree (f);
        }
        INIT_LIST_HEAD (&pis->filter);
}


static int procinfo_release (struct inode * inode, struct file * filp)
{
        struct procinfo_session * pis = (struct procinfo_session *) filp->private_data;
        int i;

        if (pis == NULL)
                return 0;

        procinfo_release_filters (pis);
        
        //printk ("%d lookups, %d last missed\n", pis->lookups, pis->last_missed);
        for (i = 0; i < PAGE_NR; i++)
                free_page ((unsigned long) pis->bitmap[i]);
        free_page ((unsigned long) pis->buf);
        kfree (pis);
        filp->private_data = NULL;

        return 0;
}


static int procinfo_add_filter (struct procinfo_session * pis, unsigned long arg)
{
        struct procinfo_filter flt;
        struct procinfo_kfilter * f;

        if (copy_from_user(&flt, (void *) arg, sizeof (flt)))
                return -EFAULT;

        if (flt.type > PROCINFO_FLT_MAX)
                return -EINVAL;

        f = kmalloc (sizeof (struct procinfo_kfilter), GFP_USER);
        if (f == NULL)
                return -ENOMEM;

        memcpy (&f->filter, &flt, sizeof (struct procinfo_filter));
        list_add (&f->list, &pis->filter);

        return 0;
}


static int procinfo_ioctl (struct inode * inode, struct file * filp, unsigned int cmd, unsigned long arg)
{
        struct procinfo_session * pis = (struct procinfo_session *) filp->private_data;

        if (pis == NULL)
                return -EBADFD;
        
        if (cmd == PROCINFO_IOC_GET_PARAMS) {
                struct procinfo_params prl;
                
                prl.reclen = pis->reclen;
                prl.cmdline_len = pis->cmdline_len;
                prl.env_len = pis->env_len;
                prl.flags = pis->flags;
                
                if (copy_to_user((void *) arg, &prl, sizeof (prl)))
                        return -EFAULT;
                
        } else if (cmd == PROCINFO_IOC_SET_PARAMS) {
                struct procinfo_params prl;
                
                if (copy_from_user(&prl, (void *) arg, sizeof (prl)))
                        return -EFAULT;

                if (prl.reclen != sizeof (struct procinfo_process))
                        return -EINVAL;
                if (prl.cmdline_len > 4096 || prl.cmdline_len < 0)
                        return -EINVAL;
                if (prl.env_len > 4096 || prl.env_len < 0)
                        return -EINVAL;
                
                pis->cmdline_len = prl.cmdline_len;
                pis->env_len = prl.env_len;
                pis->reclen = prl.reclen + pis->cmdline_len + pis->env_len;
                pis->flags = prl.flags;
                
        } else if (cmd == PROCINFO_IOC_ADD_FILTER)
                return procinfo_add_filter (pis, arg);
        else if (cmd == PROCINFO_IOC_RESET_FILTERS)
                procinfo_release_filters (pis);
        else
                return -EINVAL;

        return 0;
}


static struct file_operations procinfo_fops = {
        open:           procinfo_open,
        release:        procinfo_release,
	read:		procinfo_read,
        ioctl:          procinfo_ioctl,
};


static int init (void)
{

        if (register_chrdev (PROCINFO_MAJOR, "procinfo", &procinfo_fops)) {
		printk("unable to get major %d for memory devs\n", PROCINFO_MAJOR);
                return -1;
        }

        return 0;
}

void release (void)
{
        unregister_chrdev (PROCINFO_MAJOR, "procinfo");
}

module_init (init);
module_exit (release);


#ifdef MODULE
MODULE_AUTHOR("Alex Tomas <bzzz@tmi.comex.ru>");
MODULE_DESCRIPTION("Driver for fast access to the process list (http://tmi.comex.ru/fps/");
MODULE_LICENSE("GPL");
#endif
