#! /bin/perl -w
#
# Nisplus package for Nistool 
#
# Nistool is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either version 2 of the
# License, or (at your option) any later version.
#
# Nistool is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
# 
# You should have received a copy of the GNU General Public
# License along with the NIS+ Server; see the file COPYING.  If
# not, write to the Free Software Foundation, Inc., 675 Mass Ave,
# Cambridge, MA 02139, USA.
#
# Version: 0.1alpha
# Author: Ralf Lehmann <ralfl@darwin.muc.de>
#

package Nisplus;
use strict;

my $ERROR = 'error';
my $NAME = 'Name';
my $TYPE = 'Type';
my $OWNER='Owner';
my $GROUP='Group';
my $ACCESS='Access';
my $TTL='TTL';
my $MEMBERS='Members';
my $SERVERS='Servers';
my $MASTER='Master';
my %DEFAULTS;

## What we expect from niscat -o
my $T_TABLE='TABLE';
my $T_DIR='DIRECTORY';
my $T_GROUP='GROUP';
my $T_LINK='LINK';

##
## new ($nis_object)
##
sub new {
	my $type = shift;
	my $object = shift;
	
	my $self = {};
	$self->{$NAME} = $object;

	bless($self, $type);
}

##
## get name
##
sub get_name {
	my $self = shift;
	return $self->{$NAME};
}

##
## get an error
##
sub get_error{
	my $self = shift;

	if (defined($self->{$ERROR}) && $self->{$ERROR} ne "") {
		return $self->{$ERROR};
	}
}

##
## clear an error
##
sub clear_error{
	my $self = shift;

	undef($self->{$ERROR});
	return 0;
}

##
## get attributes 
##
sub getattr {
	my $self = shift;
	my @NISC;
	my $j;

	@NISC = `niscat -o $self->{$NAME}`;
	if (! defined($NISC[0])) {
		$self->{$ERROR} = "Can't get NIS+ Object: $self->{$NAME}";
		return 1;
	}
	while (@NISC) {
		my $i = shift @NISC;
		if ($i =~ /^Owner.*:\s*(.*)\n$/) { $self->{$OWNER} = $1; }
		if ($i =~ /^Group.*:\s*(.*)\n$/) { $self->{$GROUP} = $1; }
		if ($i =~ /^Access.*:\s*(.*)\n$/) { $self->{$ACCESS} = $1; }
		if ($i =~ /^Time.*?:\s*(.*)\n$/) { $self->{$TTL} = $1; }
		if ($i =~ /^Object Type.*:\s*(.*)\n$/) { $self->{$TYPE} = $1; }
		if (defined ($self->{$TYPE}) && $self->{$TYPE} eq $T_TABLE) {
			if ($i =~ /^Number.*?:\s*(.*)\n$/) { $self->{'Colnum'} = $1; }
			if ($i =~ /^Character.*?:\s(.*)\n$/) { $self->{'Sep'} = $1; }
			if ($i =~ /^Columns.*?:\s*(.*)\n$/) { 
				for ($j=0; $j<$self->{'Colnum'}; $j++) {
					my $Name; my $Attr; my $Access;
					$i = shift @NISC;
					if ($i =~ /.*Name\s*:\s*(\w*)/) {$Name = $1;}
					$i = shift @NISC;
					if ($i =~ /.*Attributes\s*:\s*(.*)/) {$Attr = $1;}
					$i = shift @NISC;
					if ($i =~ /.*Access.*:\s*([\w-]*)/) {$Access = $1;}
					$self->{"Col$j"} = {$NAME => $Name,
						'Attr' => $Attr,
						$ACCESS => $Access};
				}
			}
		}
		if (defined($self->{$TYPE}) && $self->{$TYPE} eq $T_GROUP){
			if ($i =~ /^Group Members\s*:\s(.*)/) { # now the gang members
				$self->{$MEMBERS} = "";
				while (@NISC){
					$i = shift @NISC;
					$i =~ /\s*(.*)\n$/;
					$self->{$MEMBERS} .= "$1 ";
				}
			}
		}
		if (defined($self->{$TYPE}) && $self->{$TYPE} eq $T_DIR){
			if ($i =~ /^Master Server\s*:\s(.*)/) { # now the servers
				$self->{$SERVERS} = "";
				while (@NISC){
					$i = shift @NISC;
					if ($i =~ /\s*Name\s*:\s(.*)\n$/){
						$self->{$SERVERS} .= "$1 ";
					}
				}
				($self->{$MASTER}) = split(/ /, $self->{$SERVERS}, 1);
			}
		}
	}
	return 0;
}

##
## List a directory object
##
sub list {
	my $self = shift;
	my $i;
	my @list = `nisls $self->{$NAME}`;
	shift @list; # Get rid of the domainname
	foreach $i (@list) {chop $i; $i .= '.' . $self->{$NAME};}
	return @list;
}

##
## new directory
##
sub mkdir{
	my $self = shift;
	my $master = shift;

	if (!$self->getattr()){
		$self->{$ERROR} = "Object \"$self->{$NAME}\" already exists!\n";
		return 1;
	}
	$self->clear_error();
	if (has_descredentials($master)) {
		print STDERR "Error: $master does not have credentials\n";
		$self->{$ERROR} = "Error: $master does not have credentials\n";
		return 1;
	}
	if (runs_nisd($master)) {
		print STDERR "Error: $master does not run rpc.nisd\n";
		return 1;
	}
	print `nismkdir -m $master $self->{$NAME}`;
	return 0;
}


##
## get entry of a table
##
sub get_entry{
	my $self = shift;
	my $line = shift;
	my %ret;
	chop ($line) if ($line =~ /.*\n/);
	my $sep = $self->{'Sep'};
	my @elements = split(/$sep/, $line, $self->{'Colnum'});
	my $i;
	for ($i=0; $i<$self->{'Colnum'}; $i++) {
		$ret{"Col$i"} = $elements[$i];
	}
	my $args = $self->indexedname(\%ret);
	my @out = `niscat -o $args`;
	foreach (@out){
		if (/\s*Owner\s*:\s(.*)\n$/) {$ret{$OWNER} = $1;}
		if (/\s*Group\s*:\s(.*)\n$/) {$ret{$GROUP} = $1;}
		if (/\s*Access\s*.*:\s(.*)\n$/) {$ret{$ACCESS} = $1;}
		if (/\s*\[(\d+)\].*'(.*)'\n$/) {
			$ret{"Col$1"} = $2;
			if ($ret{"Col$1"} eq '(nil)'){ $ret{"Col$1"} = '';}
		}

	}
	return \%ret;
}

##
## build an empty table entry
##
sub emptyentry{
	my $self = shift;
	my %ret;
	$ret{'Owner'}="";
	$ret{'Group'}="";
	$ret{'Access'}="----------------";
	my $i;
	for ($i=0; $i<$self->{'Colnum'}; $i++) {
		$ret{"Col$i"} = "";
	}
	return \%ret;
}

##
## is it a key (seachable) column
##
sub iskeycol{
	my $self = shift;
	my $n = shift;

	if ($self->{"Col$n"}->{'Attr'} =~ /SEARCHABLE/) {return 1;}
	return 0;
}

##
## return the indexed name of this entry
##
sub indexedname{
	my $self = shift;
	my $entry = shift;
	my $i=0;
	my $indexedname = "\'[";
	my $colname; my $value;
	while (defined($entry->{"Col$i"})){
		if ($self->iskeycol($i)){
			$colname = $self->{"Col$i"}->{'Name'};
			$value = $entry->{"Col$i"};
			$indexedname .= "$colname=\"$value\",";
		}
		$i++;
	}
	$indexedname .= "]\'$self->{'Name'}";
	return $indexedname;
}


##
## create a new domain 
## create($master, $netpass)
sub create_domain {
	my $self = shift;
	my $master = shift;
	my $netpass = shift;
	my $ret;
	my $domain = `domainname`;

	if (!$self->getattr()){
		$self->{$ERROR} = "Object \"$self->{$NAME}\" already exists!\n";
		return 1;
	}
	$self->clear_error();

	if (!is_rootdomain($self->{$NAME})) { # this is becomming the root domain
		if (defined ($domain) && $domain ne 'noname') {
			my $d = Nisplus->new($domain);
			if (!$d->getattr()){ # there is already a domain
				print STDERR "Error: can't create domain $self->{$NAME}\n";
				return 1;
			}
		}
		$ret = $self->create_root_domain($master, $netpass);
		if (defined($self->{$ERROR})) { 
			print STDERR "Error: $self->{$ERROR}\n";
		}
		return $ret;
	}
	else { # this is a subdomain
		if (has_descredentials($master)) {
			print STDERR "Error: $master does not have credentials\n";
			$self->{$ERROR} = "Error: $master does not have credentials\n";
			return 1;
		}
		if (runs_nisd($master)) {
			print STDERR "Error: $master does not run rpc.nisd\n";
			return 1;
		}
		print `nismkdir -m $master $self->{$NAME}`;
		print `nissetup $self->{$NAME}`;
		return 0;
	}		
}


##
## test server for credentials
##
sub has_descredentials{
	my $principal = shift;
	my $domain = strip_first($principal);
	my $success = `nismatch cname=$principal auth_type=DES cred.org_dir.$domain`;
	if (defined($success)){
		return 0;
	}
	return 1;
}

##
## test servers nisd
##
sub runs_nisd{
	my $server = shift;
	my $success = `rpcinfo -p $server | grep 100300`;
	if (defined($success) && $success =~ /100300/){
		return 0;
	}
	return 1;
}


##
## create the rootdomain
## 
sub create_root_domain {
	my $self = shift;
	my $master = shift;
	my $netpass = shift;
	my $OUT;
	my $RET;
	
	if ($< != 0 ) { 
		$self->{'error'} = "You need to be superuser to create a rootdomain!\n";
		return 1;
	}
	my $d = $self->{$NAME};
	my $domain = $d; chop ($domain);
	# Start building the domain
	`domainname $domain`;
	`domainname > /etc/defaultdomain`;
	($RET, $OUT) = adjust_nsswitch();
	if ($RET) { $self->{$ERROR} = $OUT; return 1;}
	print `nisinit -r`;
	kill_proc('keyserv');
	`keyserv`;
	kill_proc('rpc.nisd');
	`rpc.nisd -S0`;
	print `/usr/lib/nis/nissetup $d`;
	print `nisaddcred -l $netpass des`;
	print `nisupdkeys $d`;
	print `nisupdkeys org_dir.$d`;
	print `nisupdkeys groups_dir.$d`;
	kill_proc('rpc.nisd');
	`rpc.nisd`;
	return 0;	
}


##
## destroy a table or directory
##
sub destroy{
	my $self = shift;

	if (!defined($self->{$TYPE})) { $self->getattr();}
	if ($self->{$TYPE} eq $T_TABLE) {
		print `nistbladm -d $self->{$NAME}`;
		}
	elsif ($self->{$TYPE} eq $T_DIR){
		print `nisrmdir $self->{$NAME}`;
	}
}


##
## change an owner
##
sub chown{
	my $self = shift;
	my $owner = shift;
	my @res; my $RETSTR;
	print `nischown $owner $self->{$NAME}`;
}

##
## change group
##
sub chgrp{
	my $self = shift;
	my $grp = shift;
	my @res; my $RETSTR;
	print `nischgrp $grp $self->{$NAME}`;
}
##
## change mod
##
sub chmod{
	my $self = shift;
	my $mod = shift;
	my @res; my $RETSTR;
	print `nischmod $mod $self->{$NAME}`;
}

##
## return the name of the object
##
sub name{
	my $self = shift;
	return ($self->{$NAME});
}

##
## return the groupname
##
sub groupname{
	my $self = shift;
	my $g = $self->name();
	my $d = strip_first($g); 
	return if (get_first($d) ne 'groups_dir');
	$d = strip_first($d);
	$g = get_first($g);
	return $g . '.' . $d;
}


##
## return the owner of a NIS+ object
##
sub owner{
	my $self = shift;
	return($self->{$OWNER});
}

##
## 
## Helper functions
##
##

##
## see if domain is the rootdomain
##
sub is_rootdomain {
	my $domain = shift;
	$domain = strip_first($domain);
	my $d = Nisplus->new($domain);
	return(!$d->getattr());
}


##
## kill a process
##
sub kill_proc {
	$_ = shift;

	my @p = `ps -ef |grep $_ | grep -v grep`;
	while (@p) {
		my $p = shift (@p);
		my @ps = split(/ +/, $p);
		`kill $ps[2]`;
	}
}


##
##
## strip the first name of a NIS+ name
##
sub strip_first {
	$_ = shift;
	/.*?\.(.*)/;
	return $1;
}
##
## get the first part of the NIS+ name
##
sub get_first {
	$_ = shift;
	/(.*?)\..*/;
	return $1;
}

##
## add the minimum support for NIS+ to the file /etc/nsswitch.conf
##
sub adjust_nsswitch {
	my $FILE = "/etc/nsswitch.conf";

	open (F, $FILE) || return (1, "Can't open file: $FILE $!");
	my @l = <F>;
	close F;
	`cp $FILE $FILE.bak`;
	open (O , ">$FILE");
	while (@l) {
		$_ = shift (@l);
		if (/hosts:/) {
			if (! (/nisplus/)) { chop; $_ .= " nisplus\n"; }
		}
		if (/publickey:/) {
			$_ = "publickey: nisplus\n";
		}
		print O;
	}
	close O;
	return 0;
}

##
## make an absolut name (trailing dot)
##
sub absname{
	my $n = shift;
	if (!($n =~ /\.$/)){ return ($n . ".");}
	return $n;
}


##
## return the principal name of my machine
##
sub mymachinename{
	my $m = `cat /etc/nodename`; chop($m);
	my $d = `domainname`; chop($d); $d = absname($d);
	return ($m . "." . $d);
}

##
## check a domainname
##
sub check_domainname {
	my $d = shift;
	my $i;
	my %R;

	my @leaves = split (/\./, $d);
	if ($#leaves < 1) { 
		$R{"RET"}=1; $R{"TEXT"}="Error: Directory must have at least 2 leaves.\n";
		return %R;
	}
	for ($i=0; $i<=$#leaves; $i++) {
		if ($leaves[$i] =~ /.*\W+.*/) { # no whitespaces
			$R{"RET"}=1; $R{"TEXT"}="Error: No whitespaces in directory name!\n";
			return %R;
		}
	}
	$R{"RET"} = 0;
	return %R;
}

##
## check the existance of a master
##
sub check_server{
	my $serv = shift;
	my $domain;
	my $ret;
	my $retstr;
	my %R;

	$R{"RET"} = 0;
	if ($serv eq mymachinename()){  return %R;}
	%R = check_domainname(strip_first($serv));
	if ($R{"RET"}) { $R{"TEXT"} .=  $retstr; return (%R);}
	if (!runs_nisd($serv)) { return %R;}
	$R{"RET"}=1; $R{"TEXT"}="Error: Server \"$serv\" does not run rpc.nisd\n";
	return (%R);
}

##
## fill up the defaults
##
sub get_defaults{
	my @def = `nisdefaults`;
	foreach (@def) {
		if ( /.*Principal.*: (.+)/) {$DEFAULTS{'Principal'}=$1;}
		if ( /.*Domain.*: (.+)/) {$DEFAULTS{'Domain'}=$1;}
		if (/.*Host.*: (.+)/) {$DEFAULTS{'Host'}=$1;}
		if (/.*Group.*: (.+)/) {$DEFAULTS{'Group'}=$1;}
		if (/.*Rights.*: (.+)/) {$DEFAULTS{'Rights'}=$1;}
		if (/.*Time.*: (.+)/) {$DEFAULTS{'TTL'}=$1;}
		if (/.*Search.*: (.+)/) {$DEFAULTS{'Search'}=$1;}
	}
}

##
## return the default rights
##
sub get_default_rights{
	if (!defined(%DEFAULTS)){ get_defaults();}
	return $DEFAULTS{'Rights'};
}


##
## debuging print hash
##
sub printhash {
	my $h = shift;
	my $key;
	my $val;
	foreach $key ( keys %{$h}) {
		$val = $h->{"$key"};
		print "$key: $val\n";
	}
}

##
## find error messages in an array
##
sub find_errs {
	my $a = shift;
	my $errs="";

	foreach (@{$a}){
		if (/Error/ || /error/ || /cannot/ || /can't/ || /not/) { $errs .= $_ ; }
	}
	return $errs;
}


1;
