/*--------------------------------------------------------------------
 * FILE:
 *     probe_exec.c
 *
 * NOTE:
 *     This file is composed of the probe process 
 *     Low level I/O functions that called by in these functions are 
 *     contained in 'replicate_com.c'.
 *
 *--------------------------------------------------------------------
 */
#include "postgres.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <sys/wait.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/file.h>

#ifdef MULTIBYTE
#include "mb/pg_wchar.h"
#endif

#include "pgc_admin.h"
static char Exec_Result[512];

int PGC_Probe_Exec(SSL_Info * ssl_tbl, Probe_Header * header, char * packet);
char * PGC_Exec_Server(Exec_Info * exec);

static uint16_t probe_exec_myself(Probe_Header * header, char * packet);

int
PGC_Probe_Exec(SSL_Info * ssl_tbl, Probe_Header * header, char * packet)
{
	char * func = "PGC_Probe_Exec()";
	uint16_t status = STATUS_OK;
	Probe_Header r_header;
	Probe_Header h_data;

	switch (ntohs(header->serverType))
	{
		case SERVER_TYPE_PGLB:
			status = PGC_Probe_Exec_Pglb(header, packet);
			show_debug("%s;PGC_Probe_Exec_Pglb status[%d]\n",func,status);
			break;
		case SERVER_TYPE_CLUSTER:
			status = PGC_Probe_Exec_Cluster(header, packet);
			show_debug("%s;PGC_Probe_Exec_Cluster status[%d]\n",func,status);
			break;
		case SERVER_TYPE_PGRP:
			status = PGC_Probe_Exec_Pgrp(header, packet);
			show_debug("%s;PGC_Probe_Exec_Pgrp status[%d]\n",func,status);
			break;
		case SERVER_TYPE_PROBE:
			status = probe_exec_myself(header, packet);
			break;
		default :
			break;
	}

	if (ntohs(header->packet_no) == STATUS_REQ_PKT)
	{
		show_debug("%s:STATUS_ANS_PKT return[%d] as response",func,ntohs(header->serverType));
		h_data.packet_no = STATUS_ANS_PKT;
		h_data.serverType = ntohs(header->serverType);
		h_data.body_length = 0;
		h_data.rec_num = 0;
		h_data.status = status;
		PGC_Set_Packet_Header(&r_header, &h_data);
		status = PGC_Send_Status_Packet(ssl_tbl, &r_header,NULL);
	}
	return status;
}

char *
PGC_Exec_Server(Exec_Info * exec)
{
	char * func = "PGC_Exec_Server()";
	int pfd[2];
	int status;
	char *args[16];
	int i = 0;
	int j = 0;
	pid_t pid;
	char bin_path[PATH_MAX_LENGTH *2];
	int r_size = 0;

	if (pipe(pfd) == -1)
	{
		show_error("%s:pipe open error:%s",func,strerror(errno));
		return NULL;
	}
	if (exec == NULL)
	{
		show_error("%s:exec is null",func);
		return NULL;
	}
	args[i++] = exec->command;
	args[i++] = "-D";
	args[i++] = exec->workPath;
	j = 0;
	while (exec->options[j][0] != '\0')
	{
		args[i++] = exec->options[j++];
	}
	args[i++] = NULL;
	memset(bin_path,0 ,sizeof(bin_path));
	sprintf(bin_path,"%s/%s",exec->binPath, exec->command);
	show_debug("%s:bin_path[%s]\n",func, bin_path);
i = 0;
while(args[i] != NULL)
{
	show_debug("[%s]\n",args[i]);
	i++;
}
	pid = fork();
	if (pid == 0)
	{
		sleep(Fork_Delay_Time);
		close(STDOUT_FILENO);
		dup2(pfd[1], STDOUT_FILENO);
		close(pfd[0]);
		status = execv(bin_path,args);
		exit(3);
	}
	else if (pid > 0)
	{
		int result;
		close(pfd[1]);
		while ((result = waitpid(pid, &status, 0)) < 0)
		{
			if (errno == EINTR)
				continue;
			show_error("%s:result[%d] (%s)",func,result,strerror(errno));
			if (WIFEXITED(status) == 0 && WEXITSTATUS(status) != 0)
			{
				show_error("%s:WIFEXITED[%d]  WEXITSTATUS[%d]",func,WIFEXITED(status), WEXITSTATUS(status));
				return NULL;
			}
		}
		memset(Exec_Result, 0, sizeof(Exec_Result));
		i = 0;
		while  (( (r_size = read (pfd[0], &Exec_Result[i], sizeof(Exec_Result)-i)) > 0) && (errno == EINTR))
		{
			i += r_size;
		}
		close(pfd[0]);
	}
	else
	{
		return NULL;
	}
	return Exec_Result;
}

static uint16_t 
probe_exec_myself(Probe_Header * header, char * packet)
{
	char * func = "probe_exec_myself()";
	uint16_t status = DATA_ERR;

	if (header == NULL)
	{
		show_error("%s:table is not allocated",func);
		return DATA_ERR;
	}
	switch (ntohs(header->packet_no))
	{
		case STOP_REQ_PKT:
			PGC_Stop_Process(PGR_Write_Path,PGC_PROBE_PID_FILE);
			status = DATA_USE;
			break;
		case STATUS_REQ_PKT:
			status = DATA_USE;
			break;
		default:
			break;
	}
	return status;
}
