/* $Id: numeric.c,v 1.4 2005/12/14 08:22:32 jwp Exp $
 *
 * Copyright 2005, PostgresPy Project.
 * http://python.projects.postgresql.org
 *
 *//*
 * Interface to "numeric" types
 */
#include <setjmp.h>
#include <postgres.h>
#include <fmgr.h>
#include <access/heapam.h>
#include <access/htup.h>
#include <access/tupdesc.h>
#include <catalog/pg_proc.h>
#include <catalog/pg_type.h>
#include <catalog/pg_namespace.h>
#include <catalog/pg_conversion.h>
#include <catalog/pg_operator.h>
#include <catalog/pg_opclass.h>
#include <catalog/namespace.h>
#include <nodes/params.h>
#include <parser/parse_func.h>
#include <tcop/dest.h>
#include <tcop/tcopprot.h>
#include <utils/array.h>
#include <utils/datum.h>
#include <utils/elog.h>
#include <utils/palloc.h>
#include <utils/builtins.h>
#include <utils/syscache.h>
#include <utils/typcache.h>
#include <utils/relcache.h>
#include <pypg/environment.h>
#include <pypg/postgres.h>

#include <Python.h>
#include <structmember.h>
#include <pypg/python.h>

#include <pypg/externs.h>
#include <pypg/type.h>
#include <pypg/type/object.h>
#include <pypg/type/numeric.h>
#include <pypg/error.h>

#ifdef PGTYPE_INT2
static int
int2_nonzero(PyObj self)
{
	return(PyPgObject_FetchDatum(self) != 0 ? 1 : 0);
}

static PyObj
int2_int(PyObj self)
{
	PyObj rob;
	rob = PyInt_FromLong((long) DatumGetInt16(PyPgObject_FetchDatum(self)));
	return(rob);
}

static PyObj
int2_long(PyObj self)
{
	PyObj rob;
	rob = PyLong_FromLong((long) DatumGetInt16(PyPgObject_FetchDatum(self)));
	return(rob);
}

static PyObj
int2_float(PyObj self)
{
	PyObj rob;
	int2 src;
	src = DatumGetInt16(PyPgObject_FetchDatum(self));
	rob = PyFloat_FromDouble((double) src);
	return(rob);
}

static PyObj
int2_oct(PyObj self)
{
	PyObj ob, rob = NULL;
	ob = int2_int(self);
	if (ob)
	{
		rob = ob->ob_type->tp_as_number->nb_oct(ob);
		Py_DECREF(ob);
	}
	return(rob);
}

static PyObj
int2_hex(PyObj self)
{
	PyObj ob, rob = NULL;
	ob = int2_int(self);
	if (ob)
	{
		rob = ob->ob_type->tp_as_number->nb_hex(ob);
		Py_DECREF(ob);
	}
	return(rob);
}

static PyNumberMethods int2_as_number = {
	NULL,					/* nb_add */
	NULL,					/* nb_subtract */
	NULL,					/* nb_multiply */
	NULL,					/* nb_divide */
	NULL,					/* nb_remainder */
	NULL,					/* nb_divmod */
	NULL,					/* nb_power */
	NULL,					/* nb_negative */
	NULL,					/* nb_positive */
	NULL,					/* nb_absolute */
	int2_nonzero,		/* nb_nonzero */
	NULL,					/* nb_invert */
	NULL,					/* nb_lshift */
	NULL,					/* nb_rshift */
	NULL,					/* nb_and */
	NULL,					/* nb_xor */
	NULL,					/* nb_or */
	NULL,					/* nb_coerce */
	int2_int,			/* nb_int */
	int2_long,			/* nb_long */
	int2_float,			/* nb_float */
	int2_oct,			/* nb_oct */
	int2_hex,			/* nb_hex */

	NULL,
};

PyObj
int2_new(PyTypeObject *subtype, PyObj args, PyObj kw)
{
	long the_int;
	PyObj ob, intob, rob = NULL;

	if (!PyPgArg_ParseTypeSource(args, kw, &ob))
		return(NULL);

	intob = PyNumber_Int(ob);
	if (intob == NULL)
		return(NULL);

	the_int = PyInt_AsLong(intob);
	Py_DECREF(intob);
	if (the_int < -0x8000 || the_int > 0x7FFF)
	{
		PyErr_Format(PyExc_OverflowError,
			"number %ld overflows int2", the_int
		);
		return(NULL);
	}
	rob = PyPgObject_New(&PyPg_int2_Type, (int2) the_int);
	return(rob);
}

PyDoc_STRVAR(PyPg_int2_Type_Doc, "int2 interface type");

PyPgTypeObject PyPg_int2_Type = {{
	PyObject_HEAD_INIT(&PyPgType_Type)
	0,												/* ob_size */
	"int2",										/* tp_name */
	sizeof(struct PyPgObject),				/* tp_basicsize */
	0,												/* tp_itemsize */
	NULL,											/* tp_dealloc */
	NULL,											/* tp_print */
	NULL,											/* tp_getattr */
	NULL,											/* tp_setattr */
	NULL,											/* tp_compare */
	NULL,											/* tp_repr */
	&int2_as_number,							/* tp_as_number */
	NULL,											/* tp_as_sequence */
	NULL,											/* tp_as_mapping */
	NULL,											/* tp_hash */
	NULL,											/* tp_call */
	NULL,											/* tp_str */
	NULL,											/* tp_getattro */
	NULL,											/* tp_setattro */
	NULL,											/* tp_as_buffer */
	PyPg_TPFLAGS_DEFAULT,			   	/* tp_flags */
	PyPg_int2_Type_Doc,						/* tp_doc */
	NULL,											/* tp_traverse */
	NULL,											/* tp_clear */
	NULL,											/* tp_richcompare */
	0,												/* tp_weaklistoffset */
	NULL,											/* tp_iter */
	NULL,											/* tp_iternext */
	NULL,											/* tp_methods */
	NULL,											/* tp_members */
	NULL,											/* tp_getset */
	(PyTypeObject *) &PyPgObject_Type,	/* tp_base */
	NULL,											/* tp_dict */
	NULL,											/* tp_descr_get */
	NULL,											/* tp_descr_set */
	0,												/* tp_dictoffset */
	NULL,											/* tp_init */
	NULL,											/* tp_alloc */
	int2_new,						/* tp_new */
},
};
#endif /* INT2 */

#ifdef PGTYPE_INT4
static int
int4_nonzero(PyObj self)
{
	return(PyPgObject_FetchDatum(self) != 0 ? 1 : 0);
}

static PyObj
int4_int(PyObj self)
{
	PyObj rob;
	rob = PyInt_FromLong((long) DatumGetInt32(PyPgObject_FetchDatum(self)));
	return(rob);
}

static PyObj
int4_long(PyObj self)
{
	PyObj rob;
	rob = PyLong_FromLong((long) DatumGetInt32(PyPgObject_FetchDatum(self)));
	return(rob);
}

static PyObj
int4_float(PyObj self)
{
	PyObj rob;
	int4 src;
	src = DatumGetInt32(PyPgObject_FetchDatum(self));
	rob = PyFloat_FromDouble((double) src);
	return(rob);
}

static PyObj
int4_oct(PyObj self)
{
	PyObj ob, rob = NULL;
	ob = int4_int(self);
	if (ob)
	{
		rob = ob->ob_type->tp_as_number->nb_oct(ob);
		Py_DECREF(ob);
	}
	return(rob);
}

static PyObj
int4_hex(PyObj self)
{
	PyObj ob, rob = NULL;
	ob = int4_int(self);
	if (ob)
	{
		rob = ob->ob_type->tp_as_number->nb_hex(ob);
		Py_DECREF(ob);
	}
	return(rob);
}

static PyNumberMethods int4_as_number = {
	NULL,					/* nb_add */
	NULL,					/* nb_subtract */
	NULL,					/* nb_multiply */
	NULL,					/* nb_divide */
	NULL,					/* nb_remainder */
	NULL,					/* nb_divmod */
	NULL,					/* nb_power */
	NULL,					/* nb_negative */
	NULL,					/* nb_positive */
	NULL,					/* nb_absolute */
	int4_nonzero,		/* nb_nonzero */
	NULL,					/* nb_invert */
	NULL,					/* nb_lshift */
	NULL,					/* nb_rshift */
	NULL,					/* nb_and */
	NULL,					/* nb_xor */
	NULL,					/* nb_or */
	NULL,					/* nb_coerce */
	int4_int,			/* nb_int */
	int4_long,			/* nb_long */
	int4_float,			/* nb_float */
	int4_oct,			/* nb_oct */
	int4_hex,			/* nb_hex */

	NULL,
};

PyObj
int4_new(PyTypeObject *subtype, PyObj args, PyObj kw)
{
	long the_int;
	PyObj ob, intob, rob = NULL;

	if (!PyPgArg_ParseTypeSource(args, kw, &ob))
		return(NULL);

	intob = PyNumber_Int(ob);
	if (intob == NULL) return(NULL);

	the_int = PyInt_AsLong(intob);
	Py_DECREF(intob);
	if (the_int <= -0x7FFFFFFF || the_int > 0x7FFFFFFF)
	{
		PyErr_Format(PyExc_OverflowError,
			"number %ld overflows int4", the_int
		);
		return(NULL);
	}
	rob = PyPgObject_New(&PyPg_int4_Type, (int4) the_int);
	return(rob);
}

PyDoc_STRVAR(PyPg_int4_Type_Doc, "int4 interface type");

PyPgTypeObject PyPg_int4_Type = {{
	PyObject_HEAD_INIT(&PyPgType_Type)
	0,												/* ob_size */
	"int4",										/* tp_name */
	sizeof(struct PyPgObject),				/* tp_basicsize */
	0,												/* tp_itemsize */
	NULL,											/* tp_dealloc */
	NULL,											/* tp_print */
	NULL,											/* tp_getattr */
	NULL,											/* tp_setattr */
	NULL,											/* tp_compare */
	NULL,											/* tp_repr */
	&int4_as_number,							/* tp_as_number */
	NULL,											/* tp_as_sequence */
	NULL,											/* tp_as_mapping */
	NULL,											/* tp_hash */
	NULL,											/* tp_call */
	NULL,											/* tp_str */
	NULL,											/* tp_getattro */
	NULL,											/* tp_setattro */
	NULL,											/* tp_as_buffer */
	PyPg_TPFLAGS_DEFAULT,		   		/* tp_flags */
	PyPg_int4_Type_Doc,						/* tp_doc */
	NULL,											/* tp_traverse */
	NULL,											/* tp_clear */
	NULL,											/* tp_richcompare */
	0,												/* tp_weaklistoffset */
	NULL,											/* tp_iter */
	NULL,											/* tp_iternext */
	NULL,											/* tp_methods */
	NULL,											/* tp_members */
	NULL,											/* tp_getset */
	(PyTypeObject *) &PyPgObject_Type,	/* tp_base */
	NULL,											/* tp_dict */
	NULL,											/* tp_descr_get */
	NULL,											/* tp_descr_set */
	0,												/* tp_dictoffset */
	NULL,											/* tp_init */
	NULL,											/* tp_alloc */
	int4_new,									/* tp_new */
},
};
#endif /* INT4 */

#ifdef PGTYPE_INT8
static int
int8_nonzero(PyObj self)
{
	long long ll = DatumGetInt64(PyPgObject_FetchDatum(self));
	return(ll != 0 ? 1 : 0);
}

static PyObj
int8_int(PyObj self)
{
	long long ll;
	ll = DatumGetInt64(PyPgObject_FetchDatum(self));
	return(PyInt_FromLong((long) ll));
}

static PyObj
int8_long(PyObj self)
{
	long long ll;
	ll = DatumGetInt64(PyPgObject_FetchDatum(self));
	return(PyLong_FromLongLong(ll));
}

static PyObj
int8_float(PyObj self)
{
	long long ll;
	ll = DatumGetInt64(PyPgObject_FetchDatum(self));
	return(PyFloat_FromDouble((double) ll));
}

static PyObj
int8_oct(PyObj self)
{
	PyObj ob, rob = NULL;

	ob = int8_long(self);
	if (ob)
	{
		rob = ob->ob_type->tp_as_number->nb_oct(ob);
		Py_DECREF(ob);
	}
	return(rob);
}

static PyObj
int8_hex(PyObj self)
{
	PyObj ob, rob = NULL;

	ob = int8_long(self);
	if (ob)
	{
		rob = ob->ob_type->tp_as_number->nb_hex(ob);
		Py_DECREF(ob);
	}
	return(rob);
}

static PyNumberMethods int8_as_number = {
	NULL,					/* nb_add */
	NULL,					/* nb_subtract */
	NULL,					/* nb_multiply */
	NULL,					/* nb_divide */
	NULL,					/* nb_remainder */
	NULL,					/* nb_divmod */
	NULL,					/* nb_power */
	NULL,					/* nb_negative */
	NULL,					/* nb_positive */
	NULL,					/* nb_absolute */
	int8_nonzero,		/* nb_nonzero */
	NULL,					/* nb_invert */
	NULL,					/* nb_lshift */
	NULL,					/* nb_rshift */
	NULL,					/* nb_and */
	NULL,					/* nb_xor */
	NULL,					/* nb_or */
	NULL,					/* nb_coerce */
	int8_int,			/* nb_int */
	int8_long,			/* nb_long */
	int8_float,			/* nb_float */
	int8_oct,			/* nb_oct */
	int8_hex,			/* nb_hex */

	NULL,
};

PyObj
int8_new(PyTypeObject *subtype, PyObj args, PyObj kw)
{
	long long the_int;
	PyObj ob, intob, rob = NULL;

	if (!PyPgArg_ParseTypeSource(args, kw, &ob))
		return(NULL);

	intob = PyNumber_Long(ob);
	if (intob == NULL) return(NULL);

	the_int = PyLong_AsLongLong(intob);
	Py_DECREF(intob);
	if (the_int < -0x7FFFFFFFFFFFFFFELL || the_int > 0x7FFFFFFFFFFFFFFFLL)
	{
		PyErr_Format(PyExc_OverflowError,
			"number %lld overflows int8", the_int
		);
		return(NULL);
	}

	PgError_TRAP(
		rob = PyPgObject_New(&PyPg_int8_Type, Int64GetDatum(the_int))
	);
	return(rob);
}

PyDoc_STRVAR(PyPg_int8_Type_Doc, "int8 interface type");

PyPgTypeObject PyPg_int8_Type = {{
	PyObject_HEAD_INIT(&PyPgType_Type)
	0,												/* ob_size */
	"int8",										/* tp_name */
	sizeof(struct PyPgObject),				/* tp_basicsize */
	0,												/* tp_itemsize */
	NULL,											/* tp_dealloc */
	NULL,											/* tp_print */
	NULL,											/* tp_getattr */
	NULL,											/* tp_setattr */
	NULL,											/* tp_compare */
	NULL,											/* tp_repr */
	&int8_as_number,							/* tp_as_number */
	NULL,											/* tp_as_sequence */
	NULL,											/* tp_as_mapping */
	NULL,											/* tp_hash */
	NULL,											/* tp_call */
	NULL,											/* tp_str */
	NULL,											/* tp_getattro */
	NULL,											/* tp_setattro */
	NULL,											/* tp_as_buffer */
	PyPg_TPFLAGS_DEFAULT,			   	/* tp_flags */
	PyPg_int8_Type_Doc,						/* tp_doc */
	NULL,											/* tp_traverse */
	NULL,											/* tp_clear */
	NULL,											/* tp_richcompare */
	0,												/* tp_weaklistoffset */
	NULL,											/* tp_iter */
	NULL,											/* tp_iternext */
	NULL,											/* tp_methods */
	NULL,											/* tp_members */
	NULL,											/* tp_getset */
	(PyTypeObject *) &PyPgObject_Type,	/* tp_base */
	NULL,											/* tp_dict */
	NULL,											/* tp_descr_get */
	NULL,											/* tp_descr_set */
	0,												/* tp_dictoffset */
	NULL,											/* tp_init */
	NULL,											/* tp_alloc */
	int8_new,									/* tp_new */
},
};
#endif /* INT8 */

#ifdef PGTYPE_FLOAT4
static int
float4_nonzero(PyObj self)
{
	return(DatumGetFloat4(PyPgObject_FetchDatum(self)) != 0.0 ? 1 : 0);
}

static PyObj
float4_int(PyObj self)
{
	PyObj rob;
	rob = PyInt_FromLong((long) DatumGetFloat4(PyPgObject_FetchDatum(self)));
	return(rob);
}

static PyObj
float4_long(PyObj self)
{
	PyObj rob;
	rob = PyLong_FromLong((long) DatumGetFloat4(PyPgObject_FetchDatum(self)));
	return(rob);
}

static PyObj
float4_float(PyObj self)
{
	float4 src;
	src = DatumGetFloat4(PyPgObject_FetchDatum(self));
	return(PyFloat_FromDouble((double) src));
}

static PyNumberMethods float4_as_number = {
	NULL,					/* nb_add */
	NULL,					/* nb_subtract */
	NULL,					/* nb_multiply */
	NULL,					/* nb_divide */
	NULL,					/* nb_remainder */
	NULL,					/* nb_divmod */
	NULL,					/* nb_power */
	NULL,					/* nb_negative */
	NULL,					/* nb_positive */
	NULL,					/* nb_absolute */
	float4_nonzero,	/* nb_nonzero */
	NULL,					/* nb_invert */
	NULL,					/* nb_lshift */
	NULL,					/* nb_rshift */
	NULL,					/* nb_and */
	NULL,					/* nb_xor */
	NULL,					/* nb_or */
	NULL,					/* nb_coerce */
	float4_int,			/* nb_int */
	float4_long,		/* nb_long */
	float4_float,		/* nb_float */

	NULL,
};

PyObj
float4_new(PyTypeObject *subtype, PyObj args, PyObj kw)
{
	double the_float;
	PyObj ob, fob, rob = NULL;

	if (!PyPgArg_ParseTypeSource(args, kw, &ob))
		return(NULL);

	fob = PyNumber_Float(ob);
	if (fob == NULL) return(NULL);

	the_float = PyFloat_AsDouble(fob);
	Py_DECREF(fob);

	PgError_TRAP(
		rob = PyPgObject_New(&PyPg_float4_Type, Float4GetDatum((float) the_float))
	);
	return(rob);
}

PyDoc_STRVAR(PyPg_float4_Type_Doc, "float4 interface type");

PyPgTypeObject PyPg_float4_Type = {{
	PyObject_HEAD_INIT(&PyPgType_Type)
	0,												/* ob_size */
	"float4",									/* tp_name */
	sizeof(struct PyPgObject),				/* tp_basicsize */
	0,												/* tp_itemsize */
	NULL,											/* tp_dealloc */
	NULL,											/* tp_print */
	NULL,											/* tp_getattr */
	NULL,											/* tp_setattr */
	NULL,											/* tp_compare */
	NULL,											/* tp_repr */
	&float4_as_number,						/* tp_as_number */
	NULL,											/* tp_as_sequence */
	NULL,											/* tp_as_mapping */
	NULL,											/* tp_hash */
	NULL,											/* tp_call */
	NULL,											/* tp_str */
	NULL,											/* tp_getattro */
	NULL,											/* tp_setattro */
	NULL,											/* tp_as_buffer */
	PyPg_TPFLAGS_DEFAULT,			   	/* tp_flags */
	PyPg_float4_Type_Doc,					/* tp_doc */
	NULL,											/* tp_traverse */
	NULL,											/* tp_clear */
	NULL,											/* tp_richcompare */
	0,												/* tp_weaklistoffset */
	NULL,											/* tp_iter */
	NULL,											/* tp_iternext */
	NULL,											/* tp_methods */
	NULL,											/* tp_members */
	NULL,											/* tp_getset */
	(PyTypeObject *) &PyPgObject_Type,	/* tp_base */
	NULL,											/* tp_dict */
	NULL,											/* tp_descr_get */
	NULL,											/* tp_descr_set */
	0,												/* tp_dictoffset */
	NULL,											/* tp_init */
	NULL,											/* tp_alloc */
	float4_new,									/* tp_new */
},
};
#endif /* FLOAT4 */

#ifdef PGTYPE_FLOAT8
static int
float8_nonzero(PyObj self)
{
	double x = DatumGetFloat8(PyPgObject_FetchDatum(self));
	return(x != 0.0 ? 1 : 0);
}

static PyObj
float8_int(PyObj self)
{
	double x;
	x = DatumGetFloat8(PyPgObject_FetchDatum(self));
	return(PyInt_FromLong((long) x));
}

static PyObj
float8_long(PyObj self)
{
	double x;
	x = DatumGetFloat8(PyPgObject_FetchDatum(self));
	return(PyLong_FromLongLong((long long) x));
}

static PyObj
float8_float(PyObj self)
{
	double x;
	x = DatumGetFloat8(PyPgObject_FetchDatum(self));
	return(PyFloat_FromDouble(x));
}

static PyNumberMethods float8_as_number = {
	NULL,					/* nb_add */
	NULL,					/* nb_subtract */
	NULL,					/* nb_multiply */
	NULL,					/* nb_divide */
	NULL,					/* nb_remainder */
	NULL,					/* nb_divmod */
	NULL,					/* nb_power */
	NULL,					/* nb_negative */
	NULL,					/* nb_positive */
	NULL,					/* nb_absolute */
	float8_nonzero,	/* nb_nonzero */
	NULL,					/* nb_invert */
	NULL,					/* nb_lshift */
	NULL,					/* nb_rshift */
	NULL,					/* nb_and */
	NULL,					/* nb_xor */
	NULL,					/* nb_or */
	NULL,					/* nb_coerce */
	float8_int,			/* nb_int */
	float8_long,		/* nb_long */
	float8_float,		/* nb_float */
	NULL,
};

PyObj
float8_new(PyTypeObject *subtype, PyObj args, PyObj kw)
{
	double the_float;
	PyObj ob, fob, rob = NULL;

	if (!PyPgArg_ParseTypeSource(args, kw, &ob))
		return(NULL);

	fob = PyNumber_Float(ob);
	if (fob == NULL) return(NULL);

	the_float = PyFloat_AsDouble(fob);
	Py_DECREF(fob);

	PgError_TRAP(
		rob = PyPgObject_New(&PyPg_float8_Type, Float8GetDatum(the_float));
	);
	return(rob);
}

PyDoc_STRVAR(PyPg_float8_Type_Doc, "float8 interface type");

PyPgTypeObject PyPg_float8_Type = {{
	PyObject_HEAD_INIT(&PyPgType_Type)
	0,												/* ob_size */
	"float8",									/* tp_name */
	sizeof(struct PyPgObject),				/* tp_basicsize */
	0,												/* tp_itemsize */
	NULL,											/* tp_dealloc */
	NULL,											/* tp_print */
	NULL,											/* tp_getattr */
	NULL,											/* tp_setattr */
	NULL,											/* tp_compare */
	NULL,											/* tp_repr */
	&float8_as_number,						/* tp_as_number */
	NULL,											/* tp_as_sequence */
	NULL,											/* tp_as_mapping */
	NULL,											/* tp_hash */
	NULL,											/* tp_call */
	NULL,											/* tp_str */
	NULL,											/* tp_getattro */
	NULL,											/* tp_setattro */
	NULL,											/* tp_as_buffer */
	PyPg_TPFLAGS_DEFAULT,			   	/* tp_flags */
	PyPg_float8_Type_Doc,					/* tp_doc */
	NULL,											/* tp_traverse */
	NULL,											/* tp_clear */
	NULL,											/* tp_richcompare */
	0,												/* tp_weaklistoffset */
	NULL,											/* tp_iter */
	NULL,											/* tp_iternext */
	NULL,											/* tp_methods */
	NULL,											/* tp_members */
	NULL,											/* tp_getset */
	(PyTypeObject *) &PyPgObject_Type,	/* tp_base */
	NULL,											/* tp_dict */
	NULL,											/* tp_descr_get */
	NULL,											/* tp_descr_set */
	0,												/* tp_dictoffset */
	NULL,											/* tp_init */
	NULL,											/* tp_alloc */
	float8_new,									/* tp_new */
},
};
#endif /* FLOAT8 */

#ifdef PGTYPE_NUMERIC
PyDoc_STRVAR(PyPg_numeric_Type_Doc,
	"Time of day with time zone"
);

PyPgTypeObject PyPg_numeric_Type = {{
	PyObject_HEAD_INIT(&PyPgType_Type)
	0,												/* ob_size */
	"numeric",									/* tp_name */
	sizeof(struct PyPgObject),				/* tp_basicsize */
	0,												/* tp_itemsize */
	NULL,											/* tp_dealloc */
	NULL,											/* tp_print */
	NULL,											/* tp_getattr */
	NULL,											/* tp_setattr */
	NULL,											/* tp_compare */
	NULL,											/* tp_repr */
	NULL,											/* tp_as_number */
	NULL,											/* tp_as_sequence */
	NULL,											/* tp_as_mapping */
	NULL,											/* tp_hash */
	NULL,											/* tp_call */
	NULL,											/* tp_str */
	NULL,											/* tp_getattro */
	NULL,											/* tp_setattro */
	NULL,											/* tp_as_buffer */
	PyPg_TPFLAGS_DEFAULT,			   	/* tp_flags */
	PyPg_numeric_Type_Doc,					/* tp_doc */
	NULL,											/* tp_traverse */
	NULL,											/* tp_clear */
	NULL,											/* tp_richcompare */
	0,												/* tp_weaklistoffset */
	NULL,											/* tp_iter */
	NULL,											/* tp_iternext */
	NULL,											/* tp_methods */
	NULL,											/* tp_members */
	NULL,											/* tp_getset */
	(PyTypeObject *) &PyPgObject_Type,	/* tp_base */
	NULL,											/* tp_dict */
	NULL,											/* tp_descr_get */
	NULL,											/* tp_descr_set */
	0,												/* tp_dictoffset */
	NULL,											/* tp_init */
	NULL,											/* tp_alloc */
	PyPgArbitrary_new,						/* tp_new */
},
};
#endif /* NUMERIC */

#ifdef PGTYPE_CASH
PyDoc_STRVAR(PyPg_cash_Type_Doc,
	"Monetary quantity"
);

PyPgTypeObject PyPg_cash_Type = {{
	PyObject_HEAD_INIT(&PyPgType_Type)
	0,												/* ob_size */
	"cash",										/* tp_name */
	sizeof(struct PyPgObject),				/* tp_basicsize */
	0,												/* tp_itemsize */
	NULL,											/* tp_dealloc */
	NULL,											/* tp_print */
	NULL,											/* tp_getattr */
	NULL,											/* tp_setattr */
	NULL,											/* tp_compare */
	NULL,											/* tp_repr */
	NULL,											/* tp_as_number */
	NULL,											/* tp_as_sequence */
	NULL,											/* tp_as_mapping */
	NULL,											/* tp_hash */
	NULL,											/* tp_call */
	NULL,											/* tp_str */
	NULL,											/* tp_getattro */
	NULL,											/* tp_setattro */
	NULL,											/* tp_as_buffer */
	PyPg_TPFLAGS_DEFAULT,		   		/* tp_flags */
	PyPg_cash_Type_Doc,						/* tp_doc */
	NULL,											/* tp_traverse */
	NULL,											/* tp_clear */
	NULL,											/* tp_richcompare */
	0,												/* tp_weaklistoffset */
	NULL,											/* tp_iter */
	NULL,											/* tp_iternext */
	NULL,											/* tp_methods */
	NULL,											/* tp_members */
	NULL,											/* tp_getset */
	(PyTypeObject *) &PyPgObject_Type,	/* tp_base */
	NULL,											/* tp_dict */
	NULL,											/* tp_descr_get */
	NULL,											/* tp_descr_set */
	0,												/* tp_dictoffset */
	NULL,											/* tp_init */
	NULL,											/* tp_alloc */
	PyPgArbitrary_new,						/* tp_new */
},
};
#endif /* CASH */

/*
 * vim: ts=3:sw=3:noet:
 */
