/****************************************************************************
*                pvbmp.c
*
*  This module contains the code to read and write the BMP output file
*  format.
*
*  This module was written by Tim Rowley (tor@cs.brown.edu)
*
*  from Persistence of Vision Raytracer
*  Copyright 1996-2002 Persistence of Vision Team
*---------------------------------------------------------------------------
*  NOTICE: This source code file is provided so that users may experiment
*  with enhancements to POV-Ray and to port the software to platforms other
*  than those supported by the POV-Ray Team.  There are strict rules under
*  which you are permitted to use this file.  The rules are in the file
*  named POVLEGAL.DOC which should be distributed with this file. If
*  POVLEGAL.DOC is not available it may be found online at the following URL:
*
*    http://www.povray.org/povlegal.html.
*
* This program is based on the popular DKB raytracer version 2.12.
* DKBTrace was originally written by David K. Buck.
* DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
*
* $Id: //depot/povray/3.5/windows/pvbmp.c#18 $
* $Revision: #18 $
* $Change: 1814 $
* $DateTime: 2002/07/27 10:19:15 $
* $Author: chrisc $
* $Log$
*
*****************************************************************************/

#define POVWIN_FILE
#define _WIN32_IE COMMONCTRL_VERSION

/****************************************************************************
*
*  Explanation:
*
*    -
*
*  ---
*
*  Jan 1996 : Creation.
*
*****************************************************************************/

#include "frame.h"
#include "povproto.h"
#include "povray.h"
#include "pvbmp.h"
#include "file_pov.h"

/*****************************************************************************
* Local preprocessor defines
******************************************************************************/

#define BI_RGB  0
#define BI_RLE8 1
#define BI_RLE4 2
#define WIN_OS2_OLD 12
#define WIN_NEW     40
#define OS2_NEW     64
#define MAGIC1 0x15
#define MAGIC2 0x05
#define MAGIC3 0x75

/*****************************************************************************
* Local typedefs
******************************************************************************/

/*****************************************************************************
* Local variables
******************************************************************************/

/*****************************************************************************
* Static functions
******************************************************************************/

/*****************************************************************************
*
* FUNCTION
*
*   Safe_Getc
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   Tim Rowley
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   Jan 1996 : Creation.
*
******************************************************************************/

static unsigned char Safe_Getc (POV_ISTREAM& in)
{
  char ch = in.Read_Byte () ;
  if (!in)
    Error ("Error reading data from BMP image.") ;
  return (ch) ;
}

/*****************************************************************************
*
* FUNCTION
*
*   Read_BMP1
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   Tim Rowley
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   Jan 1996 : Creation.
*
******************************************************************************/

static void Read_BMP1(unsigned char **lines, POV_ISTREAM& in, unsigned width, unsigned height)
{
  int pwidth, bit, i, y;
  int c;
  
  pwidth = ((width+31)/32)*32; /* 4 byte boundary */
  
  for (y=height-1; y>=0; y--)
    for (i=bit=0; i<pwidth; i++, bit++)
    {
      if ((bit&7) == 0)
      {
        c = Safe_Getc (in);
        bit = 0;
      }
      if (i<width)
      {
        lines[y][i] = (c & 0x80) ? 1 : 0;
        c <<= 1;
      }
    }
}

/*****************************************************************************
*
* FUNCTION
*
*   Read_BMP4
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   Tim Rowley
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   Jan 1996 : Creation.
*
******************************************************************************/

static void Read_BMP4(unsigned char **lines, POV_ISTREAM& in, unsigned compression, 
                      unsigned width, unsigned height)
{
  int i, pwidth, nibble, x, y;
  int c, cc;
  
  if (compression == BI_RGB)
  {
    pwidth = ((width+7)/8)*8;
    for (y=height-1; y>=0; y--)
      for (i=nibble=0; i<pwidth; i++, nibble++)
      {
        if ((nibble&1)==0)
        {
          c = Safe_Getc(in);
          nibble = 0;
        }
        if (i<width)
        {
          lines[y][i] = (c&0xf0)>>4;
          c <<= 4;
        }
      }
  }
  else if (compression == BI_RLE4)
  {
    x = 0;
    y = height-1;
    while (1)
    {
      c = Safe_Getc (in);
      if (c)
      {
        cc = Safe_Getc (in);
        for (i=0; i<c; i++, x++)
          if ((y>=0) && (y<height) && (x>=0) && (x<width))
            lines[y][x] = (i&1) ? (cc &0x0f) : ((cc>>4)&0x0f);
      }
      else
      {
        c = Safe_Getc (in);
        if (c==0)
        {
          x=0;
          y--;
        }
        else if (c==1)
          return;
        else if (c==2)
        {
          x += Safe_Getc (in);
          y -= Safe_Getc (in);
        }
        else
        {
          for (i=0; i<c; i++, x++)
          {
            if ((i&1)==0)
              cc = Safe_Getc (in);
            if ((y>=0) && (y<height) && (x>=0) && (x<width))
              lines[y][x] = (i&1) ? (cc&0x0f) : ((cc>>4)&0x0f);
          }
          if (((c&3)==1) || ((c&3)==2))
            Safe_Getc (in);
        }
      }
    }
  }
  else
    Error ("Unknown compression scheme in BMP image.");
}

/*****************************************************************************
*
* FUNCTION
*
*   Read_BMP8
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   Tim Rowley
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   Jan 1996 : Creation.
*
******************************************************************************/

static void Read_BMP8(unsigned char **lines, POV_ISTREAM& in, unsigned compression, 
                      unsigned width, unsigned height)
{
  int i, pwidth, x, y;
  int c, cc;
  
  if (compression == BI_RGB)
  {
    pwidth = ((width+3)/4)*4;
    for (y=height-1; y>=0; y--)
      for (i=0; i<pwidth; i++)
      {
        c = Safe_Getc (in);
        if (i<width)
          lines[y][i] = c;
      }
  }
  else if (compression == BI_RLE8)
  {
    x = 0;
    y = height-1;
    while (1)
    {
      c = Safe_Getc (in);
      if (c)
      {
        cc = Safe_Getc (in);
        for (i=0; i<c; i++, x++)
          if ((y>=0) && (y<height) && (x>=0) && (x<width))
            lines[y][x] = cc;
      }
      else
      {
        c = Safe_Getc (in);
        if (c==0)
        {
          x = 0;
          y--;
        }
        else if (c==1)
          return;
        else if (c==2)
        {
          x += Safe_Getc (in);
          y -= Safe_Getc (in);
        }
        else
        {
          for (i=0; i<c; i++, x++)
            if ((y>=0) && (y<height) && (x>=0) && (x<width))
              lines[y][x] = Safe_Getc (in);
          if (c & 1)
            Safe_Getc (in); /* "absolute mode" runs are word-aligned */
        }
      }
    }
  }
  else
    Error ("Unknown compression scheme in BMP image.");
}

/*****************************************************************************
*
* FUNCTION
*
*   Read_BMP24
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   Tim Rowley
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   Jan 1996 : Creation.
*
******************************************************************************/

static void Read_BMP24(IMAGE8_LINE *lines, POV_ISTREAM& in, unsigned width, unsigned height)
{
  int pad, i, y;
  pad = (4-((width*3)%4)) &0x03;
  for (y=height-1; y>=0; y--)
  {
    for (i=0; i<width; i++)
    {
      lines[y].blue[i] = Safe_Getc (in);
      lines[y].green[i] = Safe_Getc (in);
      lines[y].red[i] = Safe_Getc (in);
    }
    if (pad)
      if (!in.seekg (pad, pov_io_base::seek_cur))
        Error ("Error reading data from BMP image.") ;
  }
}

/*****************************************************************************
*
* FUNCTION
*
*   BMP_Is_Continuable
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   Chris Cason
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   Sep 2001 : Creation.
*
******************************************************************************/

bool BMP_Is_Continuable (const char *filename, int width, int height)
{
  POV_ISTREAM *in ;

  if ((in = POV_NEW_ISTREAM (filename, POV_File_Image_System)) == NULL)
    return (false) ;

  if ((in->Read_Byte () != 'B') || (in->Read_Byte () !='M'))
  {
    delete in ;
    return (false) ;
  }

  in->ignore (12) ;
  if (in->Read_Long () != WIN_NEW)
  {
    delete in ;
    return (false) ;
  }

  if (width != -1 && height != -1)
  {
    if (width != in->Read_Long () || height != in->Read_Long ())
    {
      delete in ;
      return (false) ;
    }
  }
  else
    in->ignore (8) ;

  if (in->Read_Short () != 1 || in->Read_Short () != 24 || in->Read_Long () != BI_RGB)
  {
    delete in ;
    return (false) ;
  }

  delete in ;
  return (true) ;
}

/*****************************************************************************
*
* FUNCTION
*
*   Open_BMP_File
*
* INPUT
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   Tim Rowley
*   
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   Jan 1996 : Creation.
*   Mar 2000 : Update for new file IO classes [CJC]
*   Sep 2001 : Make into a class [CJC]
*
******************************************************************************/

BMP_Image::BMP_Image (char *file_name, int image_width, int image_height, int file_mode, int line /* = 0 */)
{
  unsigned dataLocation, depth, compression, planes, i;
  unsigned char *magic;

  mode = file_mode ;
  filename = file_name;
  width = image_width ;
  height = image_height ;
  line_number = line ;
  in_file = NULL ;
  out_file = NULL ;
  
  switch (mode) 
  {
    case READ_MODE:
      if ((in_file = POV_NEW_ISTREAM(filename, POV_File_Image_System)) == NULL)
      {
        Status_Info("\n");
        return ;
      }
      else
      {
        POV_ISTREAM& in = *in_file ;

        if ((in.Read_Byte () != 'B') || (in.Read_Byte () !='M'))
          Error ("Error reading magic number of BMP image.");

        in.ignore (8) ; // skip file size and reserved fields.
        dataLocation = in.Read_Long () ;
      
        if (in.Read_Long () != WIN_NEW)
          Error ("Wrong BMP image format.");

        width = in.Read_Long () ;
        height = in.Read_Long () ;
        planes = in.Read_Short () ;
        depth = in.Read_Short () ;
        compression = in.Read_Long () ;
        in.ignore (20) ; // skip image size in bytes, H&V pixels per meter, colors, needed colors

        if ((depth != 24) || (compression != BI_RGB) || (planes != 1))
          Error ("Wrong BMP image format.");

        if (image_width != width || image_height != height)
          Error ("BMP file dimensions do not match render resolution.");

        if (in.eof () || !in.seekg (dataLocation))
          return ;

        buffer_size = opts.File_Buffer_Size ;
      }
      break;

  case WRITE_MODE:
      if (opts.Options & TO_STDOUT)
      {
        // this can't work with BMP's ... we need to seek !
        Error ("BMP files cannot be written to stdout (cannot seek).");
      }
      else
        out_file = POV_NEW_OSTREAM (filename, POV_File_Image_System, false) ;
      if (out_file != NULL)
      {

        POV_OSTREAM& out = *out_file ;

        out << 'B' << 'M' ;
        out.Write_Long (14 + 40 + ((width * 24 + 31) / 32) * 4 * height) ;
        out.Write_Short (0) ;
        out.Write_Short (0) ;
        out.Write_Long (14 + 40) ;
        out.Write_Long (40) ;
        out.Write_Long (width) ;
        out.Write_Long (height) ;
        out.Write_Short (1) ;
        out.Write_Short (24) ;
        out.Write_Long (BI_RGB) ;
        out.Write_Long ((width * 24 + 31) / 32 * 4 * height) ;
        out.Write_Long (0) ;
        out.Write_Long (0) ;
        out.Write_Long (0) ;
        out.Write_Long (0) ;

        width = width;
        height = height;
        buffer_size = opts.File_Buffer_Size ;

        magic = (unsigned char *) POV_MALLOC (((width * 24 + 31) / 32) * 4 , "BMP magic") ;
        memset (magic, 0, ((width * 24 + 31) / 32) * 4) ;
        magic [0] = MAGIC1 ;
        magic [1] = MAGIC2 ;
        magic [2] = MAGIC3 ;
        for (i = 0 ; i < height ; i++)
          out.write ((char *) magic, (width * 24 + 31) / 32 * 4) ;
        POV_FREE (magic) ;
      }
      else
        return ;

      break;

  case APPEND_MODE:
      if (opts.Options & TO_STDOUT)
      {
        // this can't work with BMP's ... we need to seek !
        return ;
      }

      if ((out_file = POV_NEW_OSTREAM (filename, POV_File_Image_System, true)) == NULL)
        return ;

      break ;
    }

  valid = true;
}

/*****************************************************************************
*
* FUNCTION
*
*   Close_BMP_File
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   Tim Rowley
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   Jan 1996 : Creation.
*
******************************************************************************/

BMP_Image::~BMP_Image ()
{
  if(in_file != NULL)
    POV_DELETE (in_file, POV_ISTREAM);

  /* Close the output file (if open) */
  if (out_file != NULL)
  {
    out_file->flush () ;
    POV_DELETE (out_file, POV_OSTREAM);
  }

  in_file = NULL ;
  out_file = NULL ;
}

/*****************************************************************************
*
* FUNCTION
*
*   Write_BMP_Line
*
* INPUT
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   Tim Rowley
*   
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   Jan 1996 : Creation.
*
******************************************************************************/

void BMP_Image::Write_Line (COLOUR *line_data)
{
  int pad = (4 - ((width * 3) % 4)) & 0x03 ;
  POV_OSTREAM& out = *out_file ;

  if (!out.seekg (14 + 40 + (height - 1 - line_number) * (3 * width + pad)))
    Error ("Error seeking in BMP image.") ;

  for (int i = 0 ; i < width ; i++)
  {
    out << (unsigned char) floor (line_data [i] [pBLUE] * 255.0) ;
    out << (unsigned char) floor (line_data [i] [pGREEN] * 255.0) ;
    out << (unsigned char) floor (line_data [i] [pRED] * 255.0) ;
  }
  for (i = 0 ; i < pad; i++)
    out << (unsigned char) 0 ;

  if (!out)
    Error ("Error writing to BMP image.") ;

  line_number++;

  if (buffer_size == 0)
    out.flush () ;
}

      
/*****************************************************************************
*
* FUNCTION
*
*   Read_BMP_Line
*
* INPUT
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   Tim Rowley
*   
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   Jan 1996 : Creation.
*
******************************************************************************/

int BMP_Image::Read_Line (COLOUR *line_data)
{
  int pad, i, info;
  pad = (4 - ((width * 3) % 4)) & 0x03 ;

  POV_ISTREAM &in = *in_file ;
  if (!in.seekg (-(line_number + 1) * (3 * width + pad), pov_io_base::seek_end))
    return (-1) ;

  info = 0 ;
  for (i = 0 ; i < width ; i++)
  {
    char ch ;

    if (!in.Read_Byte (ch))
      return (-1) ;
    line_data [i] [pBLUE] = (DBL) ch / 255.0 ;
    if ((!i && (ch != MAGIC1)) || (i && ch))
      info = 1 ;

    if (!in.Read_Byte (ch))
      return (-1) ;
    line_data [i] [pGREEN] = (DBL) ch /255.0 ;
    if ((!i && (ch != MAGIC2)) || (i && ch))
      info = 1 ;

    if (!in.Read_Byte (ch))
      return (-1) ;
    line_data [i] [pRED] = (DBL) ch / 255.0 ;
    if ((!i && (ch != MAGIC3)) || (i && ch))
      info = 1 ;
  }

  if (info)
    line_number++ ;
  return (info) ;
}

/*****************************************************************************
*
* FUNCTION
*
*   Read_BMP_Image
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   Tim Rowley
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   Jan 1996 : Creation.
*
******************************************************************************/

void Read_BMP_Image(IMAGE *Image, char *name)
{
  int i;
  unsigned width, height, depth, colors, dataLocation, planes, compression;
  unsigned infoSize;
  IMAGE_COLOUR *cmap;
  IMAGE8_LINE *line_data;
  unsigned char *map_line;

  POV_ISTREAM *is ;
  if ((is = Locate_File (name, POV_File_Image_System, NULL, 0)) == NULL)
    Error ("Error opening BMP image.");
  POV_ISTREAM& in = *is ;

  if ((in.Read_Byte () != 'B') || (in.Read_Byte () !='M'))
    Error ("Error reading magic number of BMP image.");

  in.ignore (8) ; // skip file size and reserved fields.
  dataLocation = in.Read_Long () ;

  infoSize = in.Read_Long () ;
  if (infoSize != WIN_OS2_OLD)
  {
    width = in.Read_Long () ;
    height = in.Read_Long () ;
    planes = in.Read_Short () ;
    depth = in.Read_Short () ;
    compression = in.Read_Long () ;
    in.ignore (12) ; // skip image size in bytes, H&V pixels per meter
    colors = in.Read_Long () ;
    in.ignore (4) ; // skip needed colors
  }
  else  /* Old style */
  {
    width = in.Read_Short () ;
    height = in.Read_Short () ;
    planes = in.Read_Short () ;
    depth = in.Read_Short () ;
    compression = BI_RGB ;
    colors = 0 ;
  }

  if (in.eof ())
    Error ("Error reading data from BMP image.") ;

  /* sanity check */
  if (((depth!=1) && (depth!=4) && (depth!=8) && (depth!=24)) ||
      (planes!=1) || (compression>BI_RLE4) ||
      (((depth==1) || (depth==24)) && (compression!=BI_RGB)) ||
      ((depth==4) && (compression==BI_RLE8)) ||
      ((depth==8) && (compression==BI_RLE4)))
  {
    Error ("Invalid BMP image (depth=%d planes=%d compression=%d).",
	   depth, planes, compression);
  }

  Image->iwidth = width;
  Image->iheight = height;
  Image->width = (SNGL)width;   /* [LSK] */
  Image->height = (SNGL)height; /* [LSK] */
  Image->Colour_Map = NULL;

  /* Move to colormap */
  if (infoSize != WIN_OS2_OLD)
    in.ignore (infoSize - 40) ;
  
  /* Load colormap */
  if (depth!=24)
  {
    int length;
    if (colors)
      length = colors;
    else
      length = 1<<depth;

    cmap = (IMAGE_COLOUR *)POV_CALLOC((size_t)length, sizeof(IMAGE_COLOUR), "BMP color map");

    for (i=0; i<length; i++)
    {
      cmap[i].Blue = in.Read_Byte () ;
      cmap[i].Green = in.Read_Byte () ;
      cmap[i].Red = in.Read_Byte () ;
      if (infoSize != WIN_OS2_OLD)
        in.ignore (1) ;
    }

    if (in.eof ())
      Error("Error reading data from BMP image.");

    Image->Colour_Map_Size = length;
    Image->Colour_Map = cmap;
    Image->data.map_lines = 
      (unsigned char **)POV_MALLOC(height*sizeof(unsigned char **), "BMP image");
  }
  else
    Image->data.rgb8_lines = (IMAGE8_LINE *)POV_MALLOC(height*sizeof(IMAGE8_LINE), "BMP image");
  in.seekg (dataLocation) ;

  if (depth!=24)
  {
    for (i=height-1; i>=0; i--)
    {
      map_line = (unsigned char *)POV_MALLOC(width*sizeof(unsigned char), "BMP image line");
      Image->data.map_lines[i] = map_line;
    }
    switch (depth)
    {
    case 1:
      Read_BMP1(Image->data.map_lines, in, width, height);
      break;
    case 4:
      Read_BMP4(Image->data.map_lines, in, compression, width, height);
      break;
    case 8:
      Read_BMP8(Image->data.map_lines, in, compression, width, height);
      break;
    }
  }
  else
  {
    for (i=height-1; i>=0; i--)
    {
      line_data = &Image->data.rgb8_lines[i];
      line_data->red = (unsigned char *)POV_MALLOC(width*sizeof(unsigned char), "BMP image line");
      line_data->green = (unsigned char *)POV_MALLOC(width*sizeof(unsigned char), "BMP image line");
      line_data->blue = (unsigned char *)POV_MALLOC(width*sizeof(unsigned char), "BMP image line");
      // fix from Jim Kuzeja (thanks Jim !:)
      line_data->transm = NULL;
    }

    Read_BMP24 (Image->data.rgb8_lines, in, width, height) ;
  }

  /* Whew.... we're done */
	POV_DELETE (is, POV_ISTREAM);
}
