#!/bin/sh
set -e 

# sign_aocx - A script to sign the OpenCL aocx file

echo "The script assumes the PACsign and Intel Acceleration Stack environment is setup. If not run the command : <stack_installation_path>/init_env.sh"

usage()
{
   printf "\n******USAGE******: \n1) For creating signed aocx run command : \n./sign_aocx.sh [[[-H hsm_manager] [-i input_file ] [-r rootpublickey][-k cskkey] [-o output_file]]]| [-h]] \n2) For creating unsigned images run command : \n./sign_aocx.sh [[[-H hsm_manager] [-i file ] [-r NULL] [-k NULL] [-o output_file] \n*****************\n"
   return
}

error()
{ 
 	printf "Error : Valid arguments are required for using the script !! Please see the following usage instructions.\n "
	usage
	return
}  

#variables 
filename= 
aocx_filename=
input_pathtofile=
output_pathtofile=
root_public_key=
csk_pubic_key=
hsm_manager=
hsm_manager_options=
json_filename=
null=


if [ "$1" == "" ] ;
then
 	error         
	exit
else 

while [ "$1" != "" ]; do
    case $1 in

	-H | --hsm_manager )    shift
                                hsm_manager=$1
                                echo "hsm_manager=$1"
                                ;;

       -i | --input_file )      shift
                                input_pathtofile=$1
                                echo "aocx filename/path=$1"
                                ;;
	
	-r | --rootpublickey )  shift 
			        root_public_key=$1
				echo "root_public_key=$1"
                                ;;
        -k | --cskkey )         shift
				csk_public_key=$1
                                echo "csk_public_key=$1"
				;;
	-o | --output_file )    shift
                                output_pathtofile=$1
                                echo "output filename/path=$1"
                                ;;
        -h | --help )           shift 
				usage
				exit
                                ;;
	* )                    	usage
                               	exit
				;;
       	                     
    esac
    shift
done

fi 
if [[("$csk_public_key" == "NULL") || ("$root_public_key" == "NULL")]]
then
	null=1
	echo "null=1"
fi 


case "$hsm_manager" in
        "openssl_manager") 	hsm_manager_options="$hsm_manager"
                        	echo "openssl hsm_manager_options=$hsm_manager_options "
                        	;;

        "pkcs11_manager") 	echo "For using pkcs11_manager please give the .json filename with the path:"
				read json_filename
				hsm_manager_options="$hsm_manager -C $json_filename"
                        	echo "pkcs hsm_manager_options=$hsm_manager_options "
                        	;;
esac

#extract input file path and name

input_path=$(dirname "${input_pathtofile}")
echo "input path =$input_path"
aocx_filename="${input_pathtofile##*/}"
echo "input filename =$aocx_filename"

#extract output file path
output_path=$(dirname "${output_pathtofile}")
echo "output path =${output_path}"

output_filename="${output_pathtofile##*/}"
echo "output filename =$output_filename"

#compare input path to current dir , copy the aocx to current dir if not same 
cwd=$(pwd)
if [[("$input_path" != "$output_path")]]
then
	cp $input_pathtofile $output_path
	echo "Copied file from $input_pathtofile to $output_path directory. Performing required operations in output directory. " 
fi  

if [[("$output_path" != "$cwd") && ("$output_path" != ".")]]
then
        cd $output_path
        echo "Moving to output directory for processing files. " 
fi  

#rename the file to signed 
cp $aocx_filename $output_filename

# grabbing just the name of the aocx file 
out_filename=$( echo "$output_filename" | cut -f 1 -d '.' )

echo "Extracted the filename as $out_filename "

#Extract the bin file from the aocx
aocl binedit $out_filename\.aocx get .acl.fpga.bin temp_$out_filename\.fpga.bin
echo "1. Extracted the bin from the aocx "


#Extract the gzip compressed GBS file from the .bin 
aocl binedit temp_$out_filename\.fpga.bin get .acl.gbs.gz temp_$out_filename\.gbs.gz
echo "2. Extracted the gzip compressed GBS file from the .bin"

#uncompress it to get the GBS file
gunzip temp_$out_filename\.gbs.gz
echo "3. Uncompressed .gz it to get the GBS file"

echo "Initiating PACSign tool to sign the GBS. This process will take a couple of minutes..."
# Sign the GBS using the PACsign tool

if [[("$null" -eq 1)]]
then
echo "Creating unsigned aocx file by signing a NULL key "
PACSign PR -t UPDATE -H $hsm_manager_options -i temp_$out_filename\.gbs -o signed_$out_filename\.gbs 
else

echo "Creating signed aocx file by signing the provided keys "
PACSign PR -t UPDATE -H $hsm_manager_options -r $root_public_key -k $csk_public_key -i temp_$out_filename\.gbs -o signed_$out_filename\.gbs

fi

echo "4. Signed the GBS "

#Compress the gbs file 
gzip signed_$out_filename\.gbs -r
echo "5. Compressed the gbs file "

#Add the gzip file to fpga.bin
aocl binedit temp_$out_filename\.fpga.bin set .acl.gbs.gz signed_$out_filename\.gbs.gz 
echo "6. Added the signed gzip file to fpga.bin "

#Add the fpga.bin file back into aocx file. The aocx formed after this  setp is a signed file.
aocl binedit $out_filename\.aocx set .acl.fpga.bin temp_$out_filename\.fpga.bin 
echo "7. Added the fpga.bin file back into aocx file"


if [[("$output_path" != "$cwd") && ("$output_path" != ".")]]
then
        cd $cwd
        echo "Moving back to current directory after processing. " 
fi

echo "The signed file $out_filename.aocx has been generated. Use the command aocl program <device_name> <filename>.aocx to program it on the FPGA card"

