#!/bin/bash
# FPGA system check script for oneAPI

# realpath is used in other oneAPI components but it is not available in RHEL6
# Using 'readlink -e' instead
SCRIPTDIR=$(readlink -e $(dirname "${BASH_SOURCE[0]}"))

source $SCRIPTDIR/../../../../../common.sh $@

# Check if the user is on a supported OS
# If they are on RHEL7 we only work with a non-standard gcc so check for that case as well
DISTRO=""
DISTRO_STATUS=-1
RELEASE=""
RELEASE_STATUS=-1
GCC_VER=$(gcc -dumpversion)
GCC_VER_STATUS=$?

if type lsb_release >/dev/null 2>&1; then
  DISTRO=$(lsb_release -is)
  DISTRO_STATUS=$?
  RELEASE=$(lsb_release -rs)
  RELEASE_STATUS=$?
elif [ -f /etc/os-release ]; then
  . /etc/os-release
  if [ -z $ID ]; then
    DISTRO_STATUS=-1
  else
    DISTRO=$ID
    DISTRO_STATUS=0
  fi
  if [ -z $VERSION_ID ]; then
    RELEASE_STATUS=-1
  else
    RELEASE=$VERSION_ID
    RELEASE_STATUS=0
  fi
fi

DISTRO=${DISTRO,,}

if [ $DISTRO_STATUS != 0 ] || [ $RELEASE_STATUS != 0 ]; then
  echo "Warning: Unable to determine OS.  The FPGA backend may not work correctly"
elif [ $GCC_VER_STATUS != 0 ]; then
  echo "Warning: Unable to determine gcc version.  The FPGA backend may not work correctly"
else
  if [ "$DISTRO" == "redhatenterpriseserver" ] || [ "$DISTRO" == "rhel" ] || [ "$DISTRO" == "centos" ] && [[ $RELEASE =~ ^7 ]]; then
    if [ "$GCC_VER" != "7.4.0" ]; then
      echo "Warning: the gcc version found on your path is not 7.4.0.  The FPGA backend requires libstdc++ from gcc 7.4.0."
      echo "If you have installed gcc in a non-standard location you can point the compiler to that location with the --gcc-toolchain flag."
    else
      speak "Found a compatible OS and gcc version."
    fi
  elif [ "$DISTRO" == "redhatenterpriseserver" ] || [ "$DISTRO" == "redhatenterprise" ] || [ "$DISTRO" == "rhel" ] || [ "$DISTRO" == "centos" ] && [[ $RELEASE =~ ^8 ]]; then
    # If the user is on RHEL/CentOS 8 we assume they have the default gcc
    speak "Found a compatible OS."
  elif [ "$DISTRO" == "ubuntu" ] && [[ $RELEASE =~ ^(18|20).04 ]]; then
    # If the user is on Ubuntu we assume they have the default gcc
    speak "Found a compatible OS."
  elif [ "$DISTRO" == "suse" ] || [ "$DISTRO" == "opensuse" ] && [[ $RELEASE =~ ^15 ]]; then
    # If the user is on Ubuntu we assume they have the default gcc
    speak "Found a compatible OS."
  else
    echo "Warning: Your OS is not supported by the FPGA backend."
  fi
fi

# Check if user is in an FPGA environment
if [ "$INTELFPGAOCLSDKROOT" == "" ]; then
  echo "Warning: INTELFPGAOCLSDKROOT is not set."
  echo "         Please source oneAPI setvars.sh to set it."
  echo "         Continuing sys_check with INTELFPGAOCLSDKROOT set to $SCRIPTDIR."
  source ${SCRIPTDIR}/fpgavars.sh
elif [ "$INTELFPGAOCLSDKROOT" != "$SCRIPTDIR" ]; then
  echo "Warning: INTELFPGAOCLSDKROOT is set to $INTELFPGAOCLSDKROOT."
  echo "         oneAPI DPC++ Compiler FPGA backend's INTELFPGAOCLSDKROOT is $SCRIPTDIR."
  exit 1
else
  speak "INTELFPGAOCLSDKROOT is correctly set to $INTELFPGAOCLSDKROOT"
fi

if [ "$INTEL_FPGA_ROOT" == "" ]; then
  echo "Warning: INTEL_FPGA_ROOT is not set."
  echo "         Please ensure Intel(R) FPGA Add-on for oneAPI Base Toolkit is installed"
  echo "         if you want to perform FPGA bitstream generation."
else
  speak "Using Quartus in $INTEL_FPGA_ROOT"
fi

if [ ! -e /dev/intel-fpga-port.* ]; then
  echo "Warning: Open Programmable Acceleration Engine (OPAE) driver was not found."
  echo "         If Intel(R) Programmable Acceleration Card with Intel(R) Arria(R) 10 GX"
  echo "         or Intel(R) Stratix(R) 10 SX FPGA is installed in the system, please "
  echo "         install OPAE driver by running"
  echo "             aocl install <board package name or path>"
else
  speak "FPGA device driver is installed"

  FPGAINFO_PATH=$(which fpgainfo 2>&1)
  if [ $? -ne 0 ]; then
    echo "Warning: fpgainfo not found."
  else
    # Check firmware version of the installed PAC
    fpgainfo fme |while read -r line
    do
      if [ "$line" != "" ]; then
        if [[ "$line" =~ ^"Pr Interface Id" ]]; then
          EXPECTED_ID_PAC_A10="38d782e3-b612-5343-b934-2433e348ac4c"
          EXPECTED_ID_PAC_S10="9346116d-a52d-5ca8-b06a-a9a389ef7c8d"
          PR_IF_ID=$(echo $line |grep -o '[a-zA-Z0-9\-]*$')
          if [[ "${PR_IF_ID}" != "${EXPECTED_ID_PAC_A10}" && "${PR_IF_ID}" != "${EXPECTED_ID_PAC_S10}" ]]; then
            echo "Error: Installed Intel(R) Programmable Acceleration Card has"
            echo "       unsupported firmware installed."
            echo "       Please refer to Intel(R) Programmable Acceleration Card Getting Started"
            echo "       Guide to update it."
          else
            speak "Correct firmware installed on Intel(R) Programmable Acceleration Card"
            speak "PR Interface ID: $PR_IF_ID"
          fi
        fi
      fi
    done
  fi

  # Check if intel_a10gx_pac or intel_s10sx_pac is installed.
  BOARD_PACKAGE_INSTALLED_PAC_A10=0;
  BOARD_PACKAGE_INSTALLED_PAC_S10=0;
  board_package_lines=$(aoc -list-board-packages)
  while read -r line; do
    line="$(echo -e "$line" | sed -e 's/^[[:space:]]*//')"
    if [[ "$line" == "Board packages shipped with"* ]]; then
      break
    elif [[ "$line" == "Installed board packages:" ]]; then
      continue
    elif [[ "$line" == "No packages installed" ]]; then
      continue
    else
      if [[ "$line" =~ "intel_a10gx_pac" ]]; then
        BOARD_PACKAGE_INSTALLED_PAC_A10=1;
        continue
      fi
      if [[ "$line" =~ "intel_s10sx_pac" ]]; then
        BOARD_PACKAGE_INSTALLED_PAC_S10=1;
        continue
      fi
    fi
  done <<< "$board_package_lines"

  # intel_a10gx_pac or inte_s10sx_pac wasn't found in installed packages.
  if [[ ${BOARD_PACKAGE_INSTALLED_PAC_A10}+${BOARD_PACKAGE_INSTALLED_PAC_S10} -eq 0 ]]; then
    echo "Warning: FPGA board package for this version of oneAPI is not installed."
    echo "         If this host will be used for FPGA hardware run, please install it by"
    echo "         running"
    echo "             aocl install <board package name or path>"
  elif [[ ${BOARD_PACKAGE_INSTALLED_PAC_A10}+${BOARD_PACKAGE_INSTALLED_PAC_S10} -eq 1 ]]; then
    # intel_a10gx_pac OR inte_s10sx_pac has been installed
    FCD_SETUP=0
    FCD_FILE_PAC_A10=${ACL_BOARD_VENDOR_PATH}/intel_a10gx_pac.fcd
    if [[ -f "$FCD_FILE_PAC_A10" ]]; then
      FCD_LIBRARY=$(cat ${FCD_FILE_PAC_A10})
      if [[ "$FCD_LIBRARY" =~ "${INTELFPGAOCLSDKROOT}/board/intel_a10gx_pac" || "$FCD_LIBRARY" =~ "${INTEL_FPGA_ROOT}/board/intel_a10gx_pac" ]]; then
        FCD_SETUP=1
        speak "FPGA board package intel_a10gx_pac is installed"
      fi
    fi

    FCD_FILE_PAC_S10=${ACL_BOARD_VENDOR_PATH}/intel_s10sx_pac.fcd
    if [[ -f "$FCD_FILE_PAC_S10" ]]; then
      FCD_LIBRARY=$(cat ${FCD_FILE_PAC_S10})
      if [[ "$FCD_LIBRARY" =~ "${INTELFPGAOCLSDKROOT}/board/intel_s10sx_pac" || "$FCD_LIBRARY" =~ "${INTEL_FPGA_ROOT}/board/intel_s10sx_pac" ]]; then
        FCD_SETUP=1
        speak "FPGA board package intel_s10sx_pac is installed"
      fi
    fi

    if [[ $FCD_SETUP -eq 0 ]]; then
      echo "Warning: FPGA board package installation is corrupt."
      echo "         Please fix it by running"
      echo "             1. aocl uninstall <board package name or path>"
      echo "             2. aocl install <board package name or path>"
    else
      speak "FPGA FCD is installed correctly"
      # Run aocl diagnose to see if board enumerates
      AOCL_DIAGNOSE=$(aocl diagnose)
      if [ $? -eq 0 ]; then
        speak "${AOCL_DIAGNOSE}"
      else
        echo "${AOCL_DIAGNOSE}"
      fi
    fi

  else
    echo 'Warning: FPGA board package "inte_a10gx_pac" and "inte_s10sx_pac" should not be'
    echo "         simultaneously installed on the same machine."
    echo "         Please uninstall both by running"
    echo "             aocl uninstall <board package name or path>"
    echo "         then install one of them by running"
    echo "             aocl install <board package name or path>"
  fi
fi
