#!/bin/bash
# Test ISOs from ISO Constructor

# This script must run as user
if [ $EUID -eq 0 ]; then
    echo "Run $0 as $(logname)"
    sudo -i -u $(logname) "$0" $@
    exit
fi

DISTPATH=$1
if [ ! -z "$DISTPATH" ]; then
    ISO=$(ls "$DISTPATH"/*.iso | head -n 1)
fi
if [ -z "${ISO}" ]; then
    notify-send -u critical -a "ISO Constructor" -i iso-constructor "Cannot find the ISO - exiting"
    exit 1
fi

NAME="${ISO##*/}"
NAME="${NAME%.*}"
# Leave empty: no installation disk
HDQCOW=""

if [ $UID -eq 0 ]; then
  notify-send -u critical -a "ISO Constructor" -i iso-constructor "Run this script as $(logname), not as root - exiting"
  exit 2
fi

VINST=$(which virt-install)
if [ -z "$VINST" ]; then
    notify-send -u critical -a "ISO Constructor" -i iso-constructor "virt-install not installed - exiting"
    exit 3
fi

# Check if KVM is loaded
if [ -z "$(lsmod | grep kvm)" ]; then
    modprobe kvm_amd
    modprobe kvm_int
    if [ -z "$(lsmod | grep kvm)" ]; then
        notify-send -u critical -a "ISO Constructor" -i iso-constructor "KVM not loaded - exiting"
        exit 4
    fi
fi

# Get memory to use
MEMKB=$(grep MemTotal /proc/meminfo | awk '{print $2}')
MEMMB=$((${MEMKB}/1000))
if ((${MEMMB}<=4000)); then
    notify-send -u critical -a "ISO Constructor" -i iso-constructor "Not enough RAM (at least 4GB) - exiting"
    exit 5
fi
((${MEMMB}>=8000)) && MEM=4000
((${MEMMB}>=16000)) && MEM=8000
((${MEMMB}>=32000)) && MEM=16000

# Get nr of CPUs to use
CPU=1
TCPU=$(grep -c ^processor /proc/cpuinfo)
UCPU=$(($TCPU/2))
if ((${UCPU}<2)); then
    notify-send -u critical -a "ISO Constructor" -i iso-constructor "Just one CPU available - exiting"
    exit 6
fi
((${UCPU}>=4)) && CPU=2
((${UCPU}>=8)) && CPU=4
((${UCPU}>=16)) && CPU=8

# Create image
if [ -n "${HDQCOW}" ]; then
    if [ ! -e "${HDQCOW}" ]; then
        # Check if 25G space available for 20G hd image
        AVAIL=$(df -k --output=avail "/tmp" | tail -n 1)
        if [ ${AVAIL} -gt 26214400 ]; then
            HDQCOW="--disk ${HDQCOW},size=20,format=qcow2"
        else
            notify-send -u critical -a "ISO Constructor" -i iso-constructor "Missing hard drive" "${HDQCOW} cannot be created.\n${ISO} will boot, but cannot be installed."
            HDQCOW=""
        fi
    else
        HDQCOW="--disk ${HDQCOW}"
    fi
fi

ARGS=(
 --name ${NAME}
 --memory ${MEM}
 --memorybacking source.type=memfd,access.mode=shared
 --vcpus ${CPU}
 --cpu host-model
 --cdrom ${ISO}
 --boot cdrom,hd,menu=on,firmware=efi
 --osinfo debiantesting
 ${HDQCOW}
 --filesystem source=$HOME,target=hostshare
 --virt-type kvm
 --controller usb2
 --controller usb3
 --sound default
 --video qxl
 --channel spicevmc
 --rng /dev/urandom
 --destroy-on-exit
 --noreboot
 --check path_in_use=off
)

# Delete guest if it wasn't removed correctly the previous time
virsh destroy --domain ${NAME} 2>/dev/null
virsh undefine --nvram ${NAME} 2>/dev/null

echo
echo "$VINST ${ARGS[@]}"
echo
eval "$VINST ${ARGS[@]}"

# Delete the guest: this is only for testing ISOs
virsh destroy --domain ${NAME} 2>/dev/null
virsh undefine --nvram ${NAME} 2>/dev/null

exit 0
