// Stream buffer classes -*- C++ -*-

// Copyright (C) 1997-1999 Cygnus Solutions
//
// This file is part of the GNU ISO C++ Library.  This library is free
// software; you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option)
// any later version.

// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License along
// with this library; see the file COPYING.  If not, write to the Free
// Software Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307,
// USA.

// As a special exception, you may use this file as part of a free software
// library without restriction.  Specifically, if other files instantiate
// templates or use macros or inline functions from this file, or you compile
// this file and link it with other files to produce an executable, this
// file does not by itself cause the resulting executable to be covered by
// the GNU General Public License.  This exception does not however
// invalidate any other reasons why the executable file might be covered by
// the GNU General Public License.

//
// ISO C++ 14882: 27.5  Stream buffers
//

#ifndef _CPP_STREAMBUF
#define _CPP_STREAMBUF	1

#include <bits/c++config.h>
#include <bits/std_ios.h>

extern "C" {
#include <libio.h>
}

namespace std {

  // basic_streambuf
  template <class _CharT, class _Traits>
  class basic_streambuf 
  {
    // Types:
    typedef _CharT 			char_type;
    typedef typename _Traits::int_type 	int_type;
    typedef typename _Traits::pos_type 	pos_type;
    typedef typename _Traits::off_type 	off_type;
    typedef _Traits 			traits_type;

    friend class basic_istream<char_type, traits_type>;
    friend class basic_ostream<char_type, traits_type>;
    friend class istreambuf_iterator<char_type, traits_type>;
    friend class ostreambuf_iterator<char_type, traits_type>;

  protected:
    // This is based on _IO_FILE, just reordered to be more
    // consistent, and is intended to be the most minimal abstraction
    // that a buffer implementation can use. 
    // get == input == read
    // put == output == write
    char_type* 	_M_in_cur;	// Current read area. 
    char_type* 	_M_in_beg;  	// Start of get area. 
    char_type* 	_M_in_end;	// End of get area. 
    char_type* 	_M_out_cur;  	// Current put area. 
    char_type* 	_M_out_beg; 	// Start of put area. 
    char_type* 	_M_out_end;  	// End of put area. 
    locale 	_M_locale;
    bool 	_M_init;	// Initialization flag.

  public:
    virtual ~basic_streambuf () {}

    //  locales:
    locale   pubimbue(const locale &__loc)
      {
	locale __tmp(getloc());
	this->imbue (__loc);
	return __tmp;
      }

    locale   getloc() const
      {
	if (_M_init)
	  return _M_locale; 
	else 
	  return locale();
      } 

    // buffer and positioning:
    basic_streambuf<char_type, traits_type>* 
    pubsetbuf(char_type* __s, streamsize __n) 
      { return setbuf(__s, __n); }

    pos_type 
    pubseekoff(off_type __off, ios_base::seekdir __way, 
		ios_base::openmode __which = (ios_base::in | ios_base::out))
      { return seekoff(__off, __way, __which); }

    pos_type 
    pubseekpos(pos_type __sp,
		ios_base::openmode __which = (ios_base::in | ios_base::out))
      { return seekpos(__sp, __which); }

    int pubsync() { return sync(); }

    // Get and put areas:
    // Get area:
    streamsize in_avail() 
      { 
	if (_M_in_cur != _M_in_end)
	  return (egptr() - gptr());
	else
	  return showmanyc();
      }

    int_type snextc()
      {
	int_type __eof = traits_type::eof();
	return (sbumpc() == __eof ? __eof : sgetc()); 
      }

    int_type sbumpc()
      {
	if  (_M_in_cur != _M_in_end)
	  {
	    char_type __c = *gptr();
	    ++_M_in_cur;
	    return traits_type::to_int_type(__c);
	  }
	else 
	  return uflow();
      }

    int_type sgetc()
      {
	if  (_M_in_cur != _M_in_end)
	  return traits_type::to_int_type(*gptr());
	else 
	  return underflow();
      }

    streamsize sgetn(char_type* __s, streamsize __n)
      { return xsgetn(__s, __n); }

    // Putback:
    int_type sputbackc(char_type __c) 
      {
	int_type __retval;
	char_type* __pcur = gptr();
	bool __testpos = __pcur != _M_in_beg && __pcur <= _M_in_end; 
	if (!__testpos || !traits_type::eq(__c, __pcur[-1]))
	  __retval = pbackfail(traits_type::to_int_type(__c));
	else 
	  {
	    --_M_in_cur;
	    __retval = traits_type::to_int_type(*gptr());
	  }
	return __retval;
      }

    int_type sungetc()
      {
	char_type* __pcur = gptr();
	bool __testpos = __pcur != _M_in_beg && __pcur <= _M_in_end; 
	int_type __retval;

	if (__testpos)
	  {
	    --_M_in_cur;
	    __retval = traits_type::to_int_type(*gptr());
	  }
	else 
	  __retval = pbackfail();
	return __retval;
      }

    // Put area:
    int_type sputc(char_type __c)
      {
	if (_M_out_cur != _M_out_end)
	  {
	    *_M_out_cur = __c;
	    ++_M_out_cur; //*_M_out_cur++ = c
	    return traits_type::to_int_type(__c);
	  }
	else
	  return overflow(traits_type::to_int_type(__c));
      }

    streamsize sputn(const char_type* __s, streamsize __n)
      { return xsputn(__s, __n); }

  protected:
    basic_streambuf(): _M_in_cur(0), _M_in_beg(0), _M_in_end(0), 
    _M_out_cur(0), _M_out_beg(0), _M_out_end(0), 
    _M_locale(locale()), _M_init(false)
      {  }

    // Get area:
    char_type* eback() const
      { return _M_in_beg; }

    char_type* gptr()  const
      { 
	//return _IO_file_flags & _IO_IN_BACKUP ? _IO_save_base : _IO_in_cur; }
	return _M_in_cur; 
      }

    char_type* egptr() const
      { return _M_in_end; }

    void gbump(int __n)
      { _M_in_cur += __n; }

    void setg(char_type* __gbeg, char_type* __gnext, char_type* __gend)
      {
	_M_in_beg = __gbeg;
	_M_in_cur = __gnext;
	_M_in_end = __gend;
      }

    // Put area:
    char_type* pbase() const { return _M_out_beg; }

    char_type* pptr() const { return _M_out_cur; }

    char_type* epptr() const { return _M_out_end; }

    void pbump(int __n) { _M_out_cur += __n; }

    void setp(char_type* __pbeg, char_type* __pend)
      { 
	_M_out_beg = _M_out_cur = __pbeg; 
	_M_out_end = __pend; 
      }

    // Virtual functions:
    // Locales:
    virtual void imbue(const locale& __loc) 
      { 
	_M_init = true; 
	_M_locale = __loc;
      }

    // Buffer management and positioning:
    virtual basic_streambuf<char_type,_Traits>* 
    setbuf(char_type* __s, streamsize __n)
      {
	if (gptr() && gptr() != egptr())
	  return this;
      }

    virtual pos_type 
    seekoff(off_type __off, ios_base::seekdir __way,
	    ios_base::openmode __which = (ios_base::in | ios_base::out))
      { return pos_type(off_type(-1)); } 

    virtual pos_type 
    seekpos(pos_type __sp, 
	    ios_base::openmode __which = (ios_base::in | ios_base::out))
      { return pos_type(off_type(-1)); } 

    virtual int sync() { return 0; }

    // Get area:
    virtual int showmanyc() { return 0; }

    virtual streamsize xsgetn(char_type* __s, streamsize __n) 
      {
	streamsize __num_converted(0); 
	while (__num_converted < __n && _M_in_cur != _M_in_end)
	  {
	    *__s = *_M_in_cur;
	    ++__s;
	    ++_M_in_cur;
	    ++__num_converted;
	  }
	return __num_converted;
      }

    virtual int_type underflow()
      { return traits_type::eof(); }

    virtual int_type uflow() { underflow(); }

    // Putback:
    virtual int_type pbackfail(int_type __c = traits_type::eof()) 
      { return __c; }

    // Put area:
    virtual streamsize xsputn(const char_type* __s, streamsize __n)
      {
	unsigned int __num;
	for (__num = 0; __num < __n && _M_out_cur != _M_out_end;)
	  {
	    *_M_out_cur = *__s;
	    ++__s;
	    ++_M_out_cur;
	    ++__num;
	  }
	return __num;
      }

    virtual int_type overflow(int_type __c = traits_type::eof())
      { return __c; }

#ifdef _G_DEPRICATED
  public:
    void stossc() { _M_stossc(); }

    void _M_stossc()
      {
	 if (_M_in_cur < _M_in_end) ++_M_in_cur;
	 else uflow();
      }
#endif
  };

  typedef basic_streambuf<char>        	streambuf;
  typedef basic_streambuf<wchar_t> 	wstreambuf;
} // namespace std

#ifdef _G_NO_TEMPLATE_EXPORT
# define export
#ifndef _POSIX_SOURCE
#define _POSIX_SOURCE //why??
#endif
// #include <bits/std_cerrno.h>     // XXX delete these if unused.
// #include <bits/std_streambuf.h>
// #include <bits/std_string.h>
#include <bits/streambuf.tcc>
#endif

#endif	/* _CPP_STREAMBUF */


