/* 
Copyright (C) 1988 Free Software Foundation
    written by Doug Lea (dl@rocky.oswego.edu)

This file is part of GNU CC.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the GNU CC General Public
License for full details.

Everyone is granted permission to copy, modify and redistribute
GNU CC, but only under the conditions described in the
GNU CC General Public License.   A copy of this license is
supposed to have been given to you along with GNU CC so you
can know your rights and responsibilities.  It should be in a
file named COPYING.  Among other things, the copyright notice
and this notice must be preserved on all copies.  
*/

/* 
  BitSet class implementation
 */

#include <BitSet.h>
#include <std.h>
#include "libconfig.h"


// error handling

void default_BitSet_error_handler(char* msg)
{
  cerr << "Fatal BitSet error. " << msg << "\n";
  exit(1);
}

one_arg_error_handler_t BitSet_error_handler = default_BitSet_error_handler;

one_arg_error_handler_t set_BitSet_error_handler(one_arg_error_handler_t f)
{
  one_arg_error_handler_t old = BitSet_error_handler;
  BitSet_error_handler = f;
  return old;
}

void BitSet::error(char* msg)
{
  (*BitSet_error_handler)(msg);
}

//  globals & constants

_BitSetrep  _nil_BitSetrep = { 1, 0, 1, -1, {0} }; // nil BitSets point here

#define ONES     (~(0L))

#define MIN_BITSETREP_SIZE  2
#define MAX_BITSETREP_SIZE  (1 << (SHORTBITS - 1) - 1)


// allocate a new rep

static _BitSetrep*  new_BitSetrep(int l)
{
  int siz = l + 1;
  if (siz < MIN_BITSETREP_SIZE)
    siz = MIN_BITSETREP_SIZE;
  else if (siz >= MAX_BITSETREP_SIZE)
    (*BitSet_error_handler)("Requested length out of range");

  unsigned allocsiz = sizeof(_BitSetrep) + (siz - 1) * sizeof(long);

  _BitSetrep* z = (_BitSetrep *) (malloc(allocsiz));
  if (z == 0)
    (*BitSet_error_handler)("Out of memory.");

  z->len = l;
  z->virt = 0;
  z->sz  = siz;
  z->ref = 1;
  return z;
}

// expand a rep via realloc

static _BitSetrep*  expand_BitSetrep(_BitSetrep* p, int l)
{
  int siz = l + 1;    
  if (siz < MIN_BITSETREP_SIZE)   
    siz = MIN_BITSETREP_SIZE;
  else if (siz >= MAX_BITSETREP_SIZE)
    (*BitSet_error_handler)("Requested length out of range");

  unsigned allocsiz = sizeof(_BitSetrep) + (siz - 1) * sizeof(short);

#ifdef SHOULD_FREE_TO_REALLOC
  free((void*)p);
#endif

  _BitSetrep* z = (_BitSetrep *) (realloc((void*)p, allocsiz));
  if (z == 0)
    (*BitSet_error_handler)("Out of memory.");
  z->len = l;
  z->sz  = siz;
  z->ref = 1;
  return z;
}


// set up or expand a rep 

void BitSet::setlength(int newlen, int fill = 0)
{
  int oldlen = rep->len;

  if (rep->ref != 1)
  {
    if (rep->ref > 0) rep->ref--;
    _BitSetrep* newrep = new_BitSetrep(newlen);
    bcopy((void*)rep->s, (void*)newrep->s, rep->len * sizeof(long));
    newrep->virt = rep->virt;
    rep = newrep;
  }
  else if (newlen > rep->sz)
    rep = expand_BitSetrep(rep, newlen);
  else
    rep->len = newlen;

  if (fill)
  {
    if (rep->virt == 0)
    {
      if (newlen > oldlen)
        bzero(&(rep->s[oldlen]), (newlen - oldlen) * sizeof(long));
    }
    else
    {
      unsigned long* s = &(rep->s[oldlen]);
      unsigned long* top = &(rep->s[newlen]);
      while (s < top) *s++ = ONES;
    }
  }
}


// copy one rep to another; expand & dereference as necessary

void BitSet::copy(unsigned long* news, int newlen, int newvirt)
{
  if (rep->ref != 1)
  {
    if (rep->ref > 0) rep->ref--;
    rep = new_BitSetrep(newlen);
  }
  else if (newlen > rep->sz)
    rep = expand_BitSetrep(rep, newlen);

  bcopy((void*)news, (void*)rep->s, newlen * sizeof(long));
  rep->len = newlen;
  rep->virt = newvirt;
}

// remove unneeded top bits

void BitSet::trim()
{
  int l = rep->len;
  unsigned long* s = &(rep->s[l - 1]);

  if (rep->virt == 0)
    while (l > 1 && *s-- == 0) --l;
  else
    while (l > 1 && *s-- == ONES) --l;
  rep->len = l;
}

int operator == (BitSet& x, BitSet& y)
{
  return x.rep->len == y.rep->len && x.rep->virt == y.rep->virt &&
    bcmp((void*)x.rep->s, (void*)y.rep->s, 
         x.rep->len * sizeof(long)) == 0;
}


int operator <= (BitSet& x, BitSet& y)
{
  if (x.rep->virt > y.rep->virt)
    return 0;

  unsigned long* xs = x.rep->s;
  unsigned long* ys = y.rep->s;
  unsigned long* topx = &(xs[x.rep->len]);
  unsigned long* topy = &(ys[y.rep->len]);

  while (xs < topx && ys < topy)
  {
    unsigned long a = *xs++;
    unsigned long b = *ys++;
    if ((a | b) != b)
      return 0;
  }
  return x.rep->virt <= y.rep->virt;
}


int operator < (BitSet& x, BitSet& y)
{
  if (x.rep->virt > y.rep->virt)
    return 0;

  int xl = x.rep->len;
  int yl = y.rep->len;

  unsigned long* xs = x.rep->s;
  unsigned long* ys = y.rep->s;
  unsigned long* topx = &(xs[xl]);
  unsigned long* topy = &(ys[yl]);
  int one_diff = 0;
  while (xs < topx && ys < topy)
  {
    unsigned long a = *xs++;
    unsigned long b = *ys++;
    unsigned long c = a | b;
    if (c != b)
      return 0;
    else if (c != a)
      one_diff = 1;
  }
  if (xl == yl)
    return x.rep->virt < y.rep->virt || 
      (one_diff && x.rep->virt == y.rep->virt);
  else if (xl < yl)
    return !x.rep->virt;
  else
    return !y.rep->virt;
}

int BitSet::empty()
{
  if (rep->virt == 1)
    return 0;

  unsigned long* bots = rep->s;
  unsigned long* s = &(s[rep->len - 1]);
  while (s >= bots) if (*s-- != 0) return 0;
  return 1;
}


int BitSet::count(int b = 1)
{
  if (b == rep->virt)
    return -1;
  int l = 0;
  unsigned long* s = rep->s;
  unsigned long* tops = &(s[rep->len]);
  if (b == 1)
  {
    while (s < tops)
    {
      unsigned long a = *s++;
      for (int i = 0; i < B_SHIFT && a != 0; ++i)
      {
        if (a & 1)
          ++l;
        a >>= 1;
      }
    }
  }
  else
  {
    unsigned long maxbit = 1 << (B_SHIFT - 1);
    while (s < tops)
    {
      unsigned long a = *s++;
      for (int i = 0; i < B_SHIFT; ++i)
      {
        if ((a & maxbit) == 0)
          ++l;
        a <<= 1;
      }
    }
  }
  return l;
}

BitSet BitSet::operator ~ ()
{
  BitSet r;
  r.setlength(rep->len);
  r.rep->virt = !rep->virt;
  unsigned long* xs = rep->s;
  unsigned long* rs = r.rep->s;
  unsigned long* topr = &(rs[r.rep->len]);
  while (rs < topr) *rs++ = ~(*xs++);
  r.trim();
  return r;
}

BitSet& BitSet::complement()
{
  make_unique();
  rep->virt = !rep->virt;
  unsigned long* rs = rep->s;
  unsigned long* topr = &(rs[rep->len]);
  while (rs < topr)
  {
    unsigned long cmp = ~(*rs);
    *rs++ = cmp;
  }
  trim();
  return *this;
}


void setop(BitSet& x, BitSet& y, BitSet& r, char op)
{
  int xrsame = x.rep == r.rep;
  int yrsame = y.rep == r.rep;

  int rl;
  if (x.rep->len >= y.rep->len)
    rl = x.rep->len;
  else
    rl = y.rep->len;

  r.setlength(rl);
  unsigned long* rs = r.rep->s;
  unsigned long* topr = &(rs[rl]);

  _BitSetrep* a;
  _BitSetrep* b;

  if (x.rep->len <= y.rep->len)
  {
    a = (xrsame)? r.rep : x.rep;
    b = (yrsame)? r.rep : y.rep;
  }
  else
  {
    b = (xrsame)? r.rep : x.rep;
    a = (yrsame)? r.rep : y.rep;
    if (op == '-')              // reverse sense of difference
      op = 'D';
  }

  unsigned long* as = a->s;
  unsigned long* topa = &(as[a->len]);
  unsigned long* bs = b->s;
  unsigned long* topb = &(bs[b->len]);

  switch (op)
  {
  case '&':
    while (as < topa) *rs++ = *as++ & *bs++;
    r.rep->virt = a->virt & b->virt;
    if (a->virt)
      while (rs < topr) *rs++ = *bs++;
    else
      while (rs < topr) *rs++ = 0;
    break;
  case '|':
    while (as < topa) *rs++ = *as++ | *bs++;
    r.rep->virt = a->virt | b->virt;
    if (a->virt)
      while (rs < topr) *rs++ = ONES;
    else
      while (rs < topr) *rs++ = *bs++;
    break;
  case '^':
    while (as < topa) *rs++ = *as++ ^ *bs++;
    r.rep->virt = a->virt ^ b->virt;
    if (a->virt)
      while (rs < topr) *rs++ = ~(*bs++);
    else
      while (rs < topr) *rs++ = *bs++;
    break;
  case '-':
    while (as < topa) *rs++ = *as++ & ~(*bs++);
    r.rep->virt = a->virt & ~(b->virt);
    if (a->virt)
      while (rs < topr) *rs++ = ~(*bs++);
    else
      while (rs < topr) *rs++ = 0;
    break;
  case 'D':
    while (as < topa) *rs++ = ~(*as++) & (*bs++);
    r.rep->virt = ~(a->virt) & (b->virt);
    if (a->virt)
      while (rs < topr) *rs++ = 0;
    else
      while (rs < topr) *rs++ = *bs++;
    break;
  }
  r.trim();
}


void BitSet::set(int p)
{
  if (p < 0)
  {
    error("Illegal bit index");
    return;
  }

  int index = p / B_SHIFT;
  int pos   = p % B_SHIFT;

  if (index >= rep->len)
  {
    if (rep->virt)
      return;
    else
      setlength(index+1, 1);
  }
  else
    make_unique();

  rep->s[index] |= (1 << pos);
}

void BitSet::clear(int p)
{
  if (p < 0)
  {
    error("Illegal bit index");
    return;
  }

  int index = p / B_SHIFT;
  int pos   = p % B_SHIFT;

  if (index >= rep->len)
  {
    if (rep->virt == 0)
      return;
    else
      setlength(index+1, 1);
  }
  else
    make_unique();

  rep->s[index] &= ~(1 << pos);
}

void BitSet::invert(int p)
{
  if (p < 0)
  {
    error("Illegal bit index");
    return;
  }
  int index = p / B_SHIFT;
  int pos   = p % B_SHIFT;

  if (index >= rep->len)
    setlength(index+1, 1);
  else
    make_unique();

  rep->s[index] ^= (1 << pos);
}

int BitSet::test(int p)
{
  if (p < 0)
  {
    return 0;
  }

  int index = p / B_SHIFT;
  int pos   = p % B_SHIFT;

  if (index >= rep->len)
    return rep->virt;
  else
    return (rep->s[index] & (1 << pos)) != 0;
}

static inline unsigned long lmask(int p)
{
  return ((~(0L)) << p);
}

static inline unsigned long rmask(int p)
{
  int s = p + 1;
  if (s == B_SHIFT)
    return ~(0L);
  else
    return (1 << s) - 1;
}


void BitSet::set(int from, int to)
{
  if (from < 0 || from > to)
  {
    error("Illegal bit index");
    return;
  }

  int index1 = from / B_SHIFT;
  int pos1   = from % B_SHIFT;
  
  if (rep->virt && index1 >= rep->len)
    return;

  int index2 = to / B_SHIFT;
  int pos2   = to % B_SHIFT;

  if (index2 >= rep->len)
    setlength(index2+1, 1);
  else
    make_unique();

  unsigned long* s = &(rep->s[index1]);
  unsigned long m1 = lmask(pos1);
  unsigned long m2 = rmask(pos2);
  if (index2 == index1)
    *s |= m1 & m2;
  else
  {
    *s++ |= m1;
    unsigned long* top = &(rep->s[index2]);
    *top |= m2;
    while (s < top)
      *s++ = ONES;
  }
}

void BitSet::clear(int from, int to)
{
  if (from < 0 || from > to)
  {
    error("Illegal bit index");
    return;
  }

  int index1 = from / B_SHIFT;
  int pos1   = from % B_SHIFT;
  
  if (!rep->virt && index1 >= rep->len)
    return;

  int index2 = to / B_SHIFT;
  int pos2   = to % B_SHIFT;

  if (index2 >= rep->len)
    setlength(index2+1, 1);
  else
    make_unique();

  unsigned long* s = &(rep->s[index1]);
  unsigned long m1 = lmask(pos1);
  unsigned long m2 = rmask(pos2);
  if (index2 == index1)
    *s &= ~(m1 & m2);
  else
  {
    *s++ &= ~m1;
    unsigned long* top = &(rep->s[index2]);
    *top &= ~m2;
    while (s < top)
      *s++ = 0;
  }
}

void BitSet::invert(int from, int to)
{
  if (from < 0 || from > to)
  {
    error("Illegal bit index");
    return;
  }

  int index1 = from / B_SHIFT;
  int pos1   = from % B_SHIFT;
  int index2 = to / B_SHIFT;
  int pos2   = to % B_SHIFT;

  if (index2 >= rep->len)
    setlength(index2+1, 1);
  else
    make_unique();

  unsigned long* s = &(rep->s[index1]);
  unsigned long m1 = lmask(pos1);
  unsigned long m2 = rmask(pos2);
  if (index2 == index1)
    *s ^= m1 & m2;
  else
  {
    *s++ ^= m1;
    unsigned long* top = &(rep->s[index2]);
    *top ^= m2;
    while (s < top)
    {
      unsigned long cmp = ~(*s);
      *s++ = cmp;
    }
  }
}


int BitSet::test(int from, int to)
{
  if (from < 0 || from > to)
  {
    return 0;
  }

  int index1 = from / B_SHIFT;
  int pos1   = from % B_SHIFT;
  
  if (index1 >= rep->len)
    return rep->virt;

  int index2 = to / B_SHIFT;
  int pos2   = to % B_SHIFT;

  if (index2 >= rep->len)
  {
    if (rep->virt)
      return 1;
    else 
    {
      index2 = rep->len - 1;
      pos2 = B_SHIFT - 1;
    }
  }

  unsigned long* s = &(rep->s[index1]);
  unsigned long m1 = lmask(pos1);
  unsigned long m2 = rmask(pos2);

  if (index2 == index1)
    return (*s & m1 & m2) != 0;
  else
  {
    if (*s++ & m1)
      return 1;
    unsigned long* top = &(rep->s[index2]);
    if (*top & m2)
      return 1;
    while (s < top)
      if (*s++ != 0) 
        return 1;
    return 0;
  }
}

int BitSet::next(int p, int b = 1)
{
  ++p;
  int index = p / B_SHIFT;
  int pos   = p % B_SHIFT;

  int l = rep->len;
  
  if (index >= l)
  {
    if (rep->virt == b)
      return p;
    else
      return -1;
  }
  int j = index;
  unsigned long* s = rep->s;
  unsigned long a = s[j] >> pos;
  int i = pos;

  if (b == 1)
  {
    for (; i < B_SHIFT && a != 0; ++i)
    {
      if (a & 1)
        return j * B_SHIFT + i;
      a >>= 1;
    }
    for (++j; j < l; ++j)
    {
      a = s[j];
      for (i = 0; i < B_SHIFT && a != 0; ++i)
      {
        if (a & 1)
          return j * B_SHIFT + i;
        a >>= 1;
      }
    }
    if (rep->virt)
      return j * B_SHIFT;
    else
      return -1;
  }
  else
  {
    for (; i < B_SHIFT; ++i)
    {
      if ((a & 1) == 0)
        return j * B_SHIFT + i;
      a >>= 1;
    }
    for (++j; j < l; ++j)
    {
      a = s[j];
      if (a != ONES)
      {
        for (i = 0; i < B_SHIFT; ++i)
        {
          if ((a & 1) == 0)
            return j * B_SHIFT + i;
          a >>= 1;
        }
      }
    }
    if (!rep->virt)
      return j * B_SHIFT;
    else
      return -1;
  }
}

int BitSet::previous(int p, int b = 1)
{
  if (--p < 0)
    return -1;

  int index = p / B_SHIFT;
  int pos   = p % B_SHIFT;

  unsigned long* s = rep->s;
  int l = rep->len;

  if (index >= l)
  {
    if (rep->virt == b)
      return p;
    else
    {
      index = l - 1;
      pos = B_SHIFT - 1;
    }
  }

  int j = index;
  unsigned long a = s[j];

  int i = pos;
  unsigned long maxbit = 1 << pos;

  if (b == 1)
  {
    for (; i >= 0 && a != 0; --i)
    {
      if (a & maxbit)
        return j * B_SHIFT + i;
      a <<= 1;
    }
    maxbit = 1 << (B_SHIFT - 1);
    for (--j; j >= 0; --j)
    {
      a = s[j];
      for (i = B_SHIFT - 1; i >= 0 && a != 0; --i)
      {
        if (a & maxbit)
          return j * B_SHIFT + i;
        a <<= 1;
      }
    }
    return -1;
  }
  else
  {
    if (a != ONES)
    {
      for (; i >= 0; --i)
      {
        if ((a & maxbit) == 0)
          return j * B_SHIFT + i;
        a <<= 1;
      }
    }
    maxbit = 1 << (B_SHIFT - 1);
    for (--j; j >= 0; --j)
    {
      a = s[j];
      if (a != ONES)
      {
        for (i = B_SHIFT - 1; i >= 0; --i)
        {
          if ((a & maxbit) == 0)
            return j * B_SHIFT + i;
          a <<= 1;
        }
      }
    }
    return -1;
  }
}

int BitSet::last(int b = 1)
{
  if (b == rep->virt)
    return -1;
  else
    return previous((rep->len) * B_SHIFT, b);
}
