/* 
Copyright (C) 1988 Free Software Foundation
    written by Doug Lea (dl@rocky.oswego.edu)

This file is part of GNU CC.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the GNU CC General Public
License for full details.

Everyone is granted permission to copy, modify and redistribute
GNU CC, but only under the conditions described in the
GNU CC General Public License.   A copy of this license is
supposed to have been given to you along with GNU CC so you
can know your rights and responsibilities.  It should be in a
file named COPYING.  Among other things, the copyright notice
and this notice must be preserved on all copies.  
*/

/*
  Some of the following algorithms are very loosely based on those from 
  MIT C-Scheme bignum.c, which is
      Copyright (c) 1987 Massachusetts Institute of Technology

  with other guidance from Knuth, vol. 2

  Thanks to the creators of the algorithms.
*/

#include <Integer.h>
#include <Obstack.h>
#include <std.h>
#include <ctype.h>
#include <math.h>
#include "libconfig.h"

/*
  minimum and maximum sizes for an _Irep
*/

#define MIN_IREP_SIZE  (SHORT_PER_LONG + 1)
#define MAX_IREP_SIZE  I_MAXNUM



// arguments for setlength

#define NO_CLEAR       0
#define CLEAR_ALL      1
#define CLEAR_LAST     2
#define CLEAR_TRAILING 3

// error handling

void default_Integer_error_handler(char* msg)
{
  cerr << "Fatal Integer error. " << msg << "\n";
  exit(1);
}

one_arg_error_handler_t Integer_error_handler = default_Integer_error_handler;

one_arg_error_handler_t set_Integer_error_handler(one_arg_error_handler_t f)
{
  one_arg_error_handler_t old = Integer_error_handler;
  Integer_error_handler = f;
  return old;
}

void Integer::error(char* msg)
{
  (*Integer_error_handler)(msg);
}

// static globals

_Irep _nil_Irep = { 1, 0, 1, -1, { 0 } }; // uninitialized Integers point here
Integer _tmpInteger(0);                   // temporary operands kept here

static Integer _Integer_One(1);
static Integer _Integer_Zero(0);


// utilities to extract and transfer bits 

// get low bits

inline static unsigned short extract(unsigned long x)
{
  return x & I_MAXNUM;
}

// transfer high bits to low

inline static unsigned long down(unsigned long x)
{
  return (x >> I_SHIFT) & I_MAXNUM;
}

// transfer low bits to high

inline static unsigned long up(unsigned long x)
{
  return x << I_SHIFT;
}


// unpack a long into .s fields

static inline void cvt_long(unsigned long u, unsigned short* s)
{
  s[0] = extract(u);            // prevent looping when
  u = down(u);                  // SHORT_PER_LONG == 2
  s[1] = extract(u);  

  for (int i = 2; i < SHORT_PER_LONG; ++i)
  {
    u = down(u);
    s[i] = extract(u);  
  }
}

// compare two equal-length reps

inline static int docmp(unsigned short* x, unsigned short* y, int l)
{
  int diff = 0;
  const unsigned short* xs = &(x[l]);
  const unsigned short* ys = &(y[l]);
  while (l-- > 0 && (diff = (*--xs) - (*--ys)) == 0);
  return diff;
}

// figure out max length of result of +, -, etc.

inline static int calc_len(int len1, int len2, int pad)
{
  return (len1 >= len2)? len1 + pad : len2 + pad;
}

// ensure len & sgn are correct

void Integer::checklength()
{
  int l = rep->len;
  const unsigned short* p = &(rep->s[l]);
  while (l > 0 && *--p == 0) --l;
  if ((rep->len = l) == 0) rep->sgn = I_POSITIVE;
}

// allocate a new _Irep

static _Irep*  new_Irep(int l)
{
  int siz = l + 1;              // always pad by at least one
  if (siz < MIN_IREP_SIZE)     // and make sure can at least hold a long
    siz = MIN_IREP_SIZE;
  else if (siz >= MAX_IREP_SIZE)
    (*Integer_error_handler)("Requested length out of range");

  unsigned allocsiz = sizeof(_Irep) + (siz - 1) * sizeof(short);

  _Irep* z = (_Irep *) (malloc(allocsiz));
  if (z == 0)
    (*Integer_error_handler)("Out of memory.");

  z->sgn = I_POSITIVE;
  z->len = l;
  z->sz  = siz;
  z->ref = 1;
  return z;
}

// expand a rep via realloc

static _Irep*  expand_Irep(_Irep* p, int l)
{
  int siz = l + 1;    
  if (siz < MIN_IREP_SIZE)   
    siz = MIN_IREP_SIZE;
  else if (siz >= MAX_IREP_SIZE)
    (*Integer_error_handler)("Requested length out of range");

  unsigned allocsiz = sizeof(_Irep) + (siz - 1) * sizeof(short);

#ifdef SHOULD_FREE_TO_REALLOC
  free((void*)p);
#endif

  _Irep* z = (_Irep *) (realloc((void*)p, allocsiz));
  if (z == 0)
    (*Integer_error_handler)("Out of memory.");

  z->len = l;
  z->sz  = siz;
  z->ref = 1;
  return z;
}

// set up or expand a rep 

void Integer::setlength(int newlen, int clear = 0)
{
  int oldlen = rep->len;

  if (rep->ref != 1)
  {
    if (rep->ref > 0) rep->ref--;
    _Irep* newrep = new_Irep(newlen);
    bcopy((void*)(rep->s), (void*)(newrep->s), oldlen * sizeof(short));
    newrep->sgn = rep->sgn;
    rep = newrep;
  }
  else if (newlen > rep->sz)
    rep = expand_Irep(rep, newlen);
  else
    rep->len = newlen;

  switch(clear)
  {
  case NO_CLEAR:
    break;
  case CLEAR_ALL:
    bzero((void*)&(rep->s[oldlen]), newlen * sizeof(short));
    break;
  case CLEAR_LAST:
    if (newlen > oldlen) rep->s[newlen-1] = 0;
    break;
  case CLEAR_TRAILING:
    int l = newlen - oldlen;
    if (l > 0)
      bzero((void*)&(rep->s[oldlen]), l * sizeof(short));
    else if (l < 0)
      bzero((void*)&(rep->s[newlen]), -l * sizeof(short));
    break;
  default:
    break;
  }
}

// copy one rep to another; expand & dereference as necessary

void Integer::copy(const unsigned short* news, int newlen, int newsgn)
{
  if (rep->ref != 1)
  {
    if (rep->ref > 0) rep->ref--;
    rep = new_Irep(newlen);
  }
  else if (newlen > rep->sz)
    rep = expand_Irep(rep, newlen);

  bcopy((void*)news, (void*)(rep->s), newlen * sizeof(short));
  rep->len = newlen;
  rep->sgn = newsgn;

  checklength();
}


void Integer::copy(long x)
{
  unsigned short tmp[SHORT_PER_LONG];
  unsigned long u;
  int newsgn;
  if (newsgn = (x >= 0))
    u = x;
  else
    u = -x;
  
  int l = 0;
  while (u != 0)
  {
    tmp[l++] = extract(u);
    u = down(u);
  }
  copy(tmp, l, newsgn);
}

// copy a long into _tmpInteger 

void mk_tmpInteger(long x)
{
  unsigned long u;
  if (_tmpInteger.rep->sgn = (x >= 0))
    u = x;
  else
    u = -x;
  int l = 0;
  while (u != 0)
  {
    _tmpInteger.rep->s[l++] = extract(u);
    u = down(u);
  }
  _tmpInteger.rep->len = l;
}

// convert to a legal two's complement long if possible
// if too big, return most negative/positive value

long Integer::operator long()
{ 
  if (rep->len > SHORT_PER_LONG)
    return (rep->sgn == I_POSITIVE) ? MAXLONG : MINLONG;
  else
  {
    unsigned long a = rep->s[SHORT_PER_LONG - 1];
    if (a >= I_MINNUM)
      return (rep->sgn == I_POSITIVE) ? MAXLONG : MINLONG;
    else
    {
      for (int i = SHORT_PER_LONG - 2; i >= 0; --i)
        a = up(a) | rep->s[i];
      return (rep->sgn == I_POSITIVE)? a : -((long)a);
    }
  }
}

// test whether op long() will work.
// careful about asymmetry between MINLONG & MAXLONG

int Integer::fits_in_long()
{
  int l = rep->len;
  if (l < SHORT_PER_LONG)
    return 1;
  else if (l > SHORT_PER_LONG)
    return 0;
  else if (rep->s[SHORT_PER_LONG - 1] < I_MINNUM)
    return 1;
  else if (rep->sgn == I_NEGATIVE && rep->s[SHORT_PER_LONG - 1] == I_MINNUM)
  {
    for (int i = 0; i < SHORT_PER_LONG - 1; ++i)
      if (rep->s[i] != 0)
        return 0;
    return 1;
  }
  else
    return 0;
}

// convert to a double 

double Integer::operator double()
{ 
  double d = 0.0;
  double bound = HUGE / 2.0;
  for (int i = rep->len - 1; i >= 0; --i)
  {
    unsigned short a = I_RADIX >> 1;
    while (a != 0)
    {
      if (d >= bound)
        return (rep->sgn == I_NEGATIVE) ? -HUGE : HUGE;
      d *= 2.0;
      if (rep->s[i] & a)
        d += 1.0;
      a >>= 1;
    }
  }
  if (rep->sgn == I_NEGATIVE)
    return -d;
  else
    return d;
}

// see whether op double() will work-
// have to actually try it in order to find out
// since otherwise might trigger fp exception

int Integer::fits_in_double()   
{
  double d = 0.0;
  double bound = HUGE / 2.0;
  for (int i = rep->len - 1; i >= 0; --i)
  {
    unsigned short a = I_RADIX >> 1;
    while (a != 0)
    {
      if (d > bound || (d == bound && (i > 0 || (rep->s[i] & a))))
        return 0;
      d *= 2.0;
      if (rep->s[i] & a)
        d += 1.0;
      a >>= 1;
    }
  }
  return 1;
}

// comparison functions
  
int compare(Integer& x, Integer& y)
{
  int diff  = x.rep->sgn - y.rep->sgn;
  if (diff == 0)
  {
    diff = x.rep->len - y.rep->len;
    if (diff == 0)
      diff = docmp(x.rep->s, y.rep->s, x.rep->len);
    if (x.rep->sgn == I_NEGATIVE)
      diff = -diff;
  }
  return diff;
}


int ucompare(Integer& x, Integer& y)
{
  int diff = x.rep->len - y.rep->len;
  if (diff == 0)
    diff = docmp(x.rep->s, y.rep->s, x.rep->len);
  return diff;
}


int compare(Integer& x, long  y)
{
  int xl = x.rep->len;
  int xsgn = x.rep->sgn;
  if (y == 0)
  {
    if (xl == 0)
      return 0;
    else if (xsgn == I_NEGATIVE)
      return -1;
    else
      return 1;
  }
  else
  {
    int ysgn = y >= 0;
    unsigned long uy = (ysgn)? y : -y;
    int diff = xsgn - ysgn;
    if (diff == 0)
    {
      diff = xl - SHORT_PER_LONG;
      if (diff <= 0)
      {
        unsigned short tmp[SHORT_PER_LONG];
        int yl = 0;
        while (uy != 0)
        {
          tmp[yl++] = extract(uy);
          uy = down(uy);
        }
        diff = xl - yl;
        if (diff == 0)
          diff = docmp(x.rep->s, tmp, xl);
      }
    }
    if (xsgn == I_NEGATIVE)
      diff = -diff;
    return diff;
  }
}

int ucompare(Integer& x, long  y)
{
  int xl = x.rep->len;
  if (y == 0)
    return xl;
  else
  {
    unsigned long uy = (y >= 0)? y : -y;
    int diff = xl - SHORT_PER_LONG;
    if (diff <= 0)
    {
      unsigned short tmp[SHORT_PER_LONG];
      int yl = 0;
      while (uy != 0)
      {
        tmp[yl++] = extract(uy);
        uy = down(uy);
      }
      diff = xl - yl;
      if (diff == 0)
        diff = docmp(x.rep->s, tmp, xl);
    }
    return diff;
  }
}



// arithmetic functions

void add(Integer& x, int negatex, Integer& y, int negatey, Integer& r)
{
  int xl = x.rep->len;
  int yl = y.rep->len;

  int xsgn = (negatex && xl != 0) ? !x.rep->sgn : x.rep->sgn;
  int ysgn = (negatey && yl != 0) ? !y.rep->sgn : y.rep->sgn;

  int xrsame = x.rep == r.rep;
  int yrsame = y.rep == r.rep;

  if (yl == 0)
  {
    if (xrsame)
    {
      r.make_unique();
      r.rep->sgn = xsgn;
    }
    else
      r.copy(x.rep->s, xl, xsgn);
  }
  else if (xl == 0)
  {
    if (yrsame)
    {
      r.make_unique();
      r.rep->sgn == ysgn;
    }
    else
      r.copy(y.rep->s, yl, ysgn);
  }
  else if (xsgn == ysgn)
  {
    r.setlength(calc_len(xl, yl, 1), CLEAR_LAST);
    r.rep->sgn = xsgn;
    unsigned short* rs = r.rep->s;
    unsigned short* as;
    unsigned short* bs;
    unsigned short* topa;
    unsigned short* topb;
    if (xl >= yl)
    {
      as =  (xrsame)? r.rep->s : x.rep->s;
      topa = &(as[xl]);
      bs =  (yrsame)? r.rep->s : y.rep->s;
      topb = &(bs[yl]);
    }
    else
    {
      bs =  (xrsame)? r.rep->s : x.rep->s;
      topb = &(bs[xl]);
      as =  (yrsame)? r.rep->s : y.rep->s;
      topa = &(as[yl]);
    }
    unsigned long sum = 0;
    while (bs < topb)
    {
      sum += (unsigned long)(*as++) + (unsigned long)(*bs++);
      *rs++ = extract(sum);
      sum = down(sum);
    }
    while (sum != 0 && as < topa)
    {
      sum += (unsigned long)(*as++);
      *rs++ = extract(sum);
      sum = down(sum);
    }
    if (sum != 0)
      *rs = extract(sum);
    else if (rs != as)
      while (as < topa)
        *rs++ = *as++;
  }
  else
  {
    int comp = ucompare(x, y);
    if (comp == 0)
      r.copy(_Integer_Zero);
    else
    {
      r.setlength(calc_len(xl, yl, 0));
      unsigned short* rs = r.rep->s;
      unsigned short* as;
      unsigned short* bs;
      unsigned short* topa;
      unsigned short* topb;
      if (comp > 0)
      {
        as =  (xrsame)? r.rep->s : x.rep->s;
        topa = &(as[xl]);
        bs =  (yrsame)? r.rep->s : y.rep->s;
        topb = &(bs[yl]);
        r.rep->sgn = xsgn;
      }
      else
      {
        bs =  (xrsame)? r.rep->s : x.rep->s;
        topb = &(bs[xl]);
        as =  (yrsame)? r.rep->s : y.rep->s;
        topa = &(as[yl]);
        r.rep->sgn = ysgn;
      }
      unsigned long hi = 1;
      while (bs < topb)
      {
        hi += (unsigned long)(*as++) + I_MAXNUM - (unsigned long)(*bs++);
        *rs++ = extract(hi);
        hi = down(hi);
      }
      while (hi == 0 && as < topa)
      {
        hi = (unsigned long)(*as++) + I_MAXNUM;
        *rs++ = extract(hi);
        hi = down(hi);
      }
      if (rs != as)
        while (as < topa)
          *rs++ = *as++;
    }
  }
  r.checklength();
}

void add(Integer& x, int negatex, long y, Integer& r)
{
  int xl = x.rep->len;
  int xsgn = (negatex && xl != 0) ? !x.rep->sgn : x.rep->sgn;
  int xrsame = x.rep == r.rep;

  int ysgn = (y >= 0);
  unsigned long uy = (ysgn)? y : -y;

  if (y == 0)
  {
    if (xrsame)
    {
      r.make_unique();
      r.rep->sgn = xsgn;
    }
    else
      r.copy(x.rep->s, xl, xsgn);
  }
  else if (xl == 0)
    r.copy(y);
  else if (xsgn == ysgn)
  {
    r.setlength(calc_len(xl, SHORT_PER_LONG, 1), CLEAR_LAST);
    r.rep->sgn = xsgn;
    unsigned short* rs = r.rep->s;
    unsigned short* as =  (xrsame)? r.rep->s : x.rep->s;
    unsigned short* topa = &(as[xl]);
    unsigned long sum = 0;
    while (as < topa && uy != 0)
    {
      unsigned long u = extract(uy);
      uy = down(uy);
      sum += (unsigned long)(*as++) + u;
      *rs++ = extract(sum);
      sum = down(sum);
    }
    while (sum != 0 && as < topa)
    {
      sum += (unsigned long)(*as++);
      *rs++ = extract(sum);
      sum = down(sum);
    }
    if (sum != 0)
      *rs = extract(sum);
    else if (rs != as)
      while (as < topa)
        *rs++ = *as++;
  }
  else
  {
    unsigned short tmp[SHORT_PER_LONG];
    int yl = 0;
    while (uy != 0)
    {
      tmp[yl++] = extract(uy);
      uy = down(uy);
    }
    int comp = xl - yl;
    if (comp == 0)
      comp = docmp(x.rep->s, tmp, yl);
    if (comp == 0)
      r.copy(_Integer_Zero);
    else
    {
      r.setlength(calc_len(xl, yl, 0));
      unsigned short* rs = r.rep->s;
      unsigned short* as;
      unsigned short* bs;
      unsigned short* topa;
      unsigned short* topb;
      if (comp > 0)
      {
        as =  (xrsame)? r.rep->s : x.rep->s;
        topa = &(as[xl]);
        bs =  tmp;
        topb = &(bs[yl]);
        r.rep->sgn = xsgn;
      }
      else
      {
        bs =  (xrsame)? r.rep->s : x.rep->s;
        topb = &(bs[xl]);
        as =  tmp;
        topa = &(as[yl]);
        r.rep->sgn = ysgn;
      }
      unsigned long hi = 1;
      while (bs < topb)
      {
        hi += (unsigned long)(*as++) + I_MAXNUM - (unsigned long)(*bs++);
        *rs++ = extract(hi);
        hi = down(hi);
      }
      while (hi == 0 && as < topa)
      {
        hi = (unsigned long)(*as++) + I_MAXNUM;
        *rs++ = extract(hi);
        hi = down(hi);
      }
      if (rs != as)
        while (as < topa)
          *rs++ = *as++;
    }
  }
  r.checklength();
}


void multiply(Integer& x, Integer& y, Integer& r)
{
  int xl = x.rep->len;
  int yl = y.rep->len;
  int rl = xl + yl;
  int xrsame = x.rep == r.rep;
  int yrsame = y.rep == r.rep;
  int xysame = x.rep == y.rep;
  int rsgn = x.rep->sgn == y.rep->sgn;
  
  if (xl == 0 || yl == 0)
    r.copy(_Integer_Zero);
  else if (!xysame)
  {
    r.setlength(rl, CLEAR_TRAILING);
    r.rep->sgn = rsgn;

    unsigned short* rs = r.rep->s;
    unsigned short* topr = &(rs[rl]);

    // use best inner/outer loop params given constraints
    unsigned short* currentr;
    unsigned short* bota;
    unsigned short* as;
    unsigned short* botb;
    unsigned short* topb;
    if (xrsame)                 
    { 
      currentr = &(rs[xl-1]);
      bota = rs;
      as = currentr;
      botb = y.rep->s;
      topb = &(botb[yl]);
    }
    else if (yrsame)
    {
      currentr = &(rs[yl-1]);
      bota = rs;
      as = currentr;
      botb = x.rep->s;
      topb = &(botb[xl]);
    }
    else if (xl <= yl)
    {
      currentr = &(rs[xl-1]);
      bota = x.rep->s;
      as = &(bota[xl-1]);
      botb = y.rep->s;
      topb = &(botb[yl]);
    }
    else
    {
      currentr = &(rs[yl-1]);
      bota = y.rep->s;
      as = &(bota[yl-1]);
      botb = x.rep->s;
      topb = &(botb[xl]);
    }

    while (as >= bota)
    {
      unsigned long ai = (unsigned long)(*as--);
      unsigned short* rs = currentr--;
      *rs = 0;
      if (ai != 0)
      {
        unsigned long sum = 0;
        unsigned short* bs = botb;
        while (bs < topb)
        {
          sum += ai * (unsigned long)(*bs++) + (unsigned long)(*rs);
          *rs++ = extract(sum);
          sum = down(sum);
        }
        while (sum != 0 && rs < topr)
        {
          sum += (unsigned long)(*rs);
          *rs++ = extract(sum);
          sum = down(sum);
        }
      }
    }
    r.checklength();
  }
  else                          // x & y same; compute over diagonals
  {
    r.setlength(rl, CLEAR_LAST);
    r.rep->sgn = rsgn;

    unsigned short* botr = r.rep->s;
    unsigned short* topr = &(botr[rl]);
    unsigned short* rs =   &(botr[rl - 2]);

    unsigned short* bota = (xrsame)? botr : x.rep->s;
    unsigned short* loa =  &(bota[xl - 1]);
    unsigned short* hia =  loa;

    for (; rs >= botr; --rs)
    {
      unsigned short* h = hia;
      unsigned short* l = loa;
      unsigned long prod = (unsigned long)(*h) * (unsigned long)(*l);
      *rs = 0;

      for(;;)
      {
        unsigned short* rt = rs;
        unsigned long sum = prod + (unsigned long)(*rt);
        *rt++ = extract(sum);
        sum = down(sum);
        while (sum != 0 && rt < topr)
        {
          sum += (unsigned long)(*rt);
          *rt++ = extract(sum);
          sum = down(sum);
        }
        if (h > l)
        {
          rt = rs;
          sum = prod + (unsigned long)(*rt);
          *rt++ = extract(sum);
          sum = down(sum);
          while (sum != 0 && rt < topr)
          {
            sum += (unsigned long)(*rt);
            *rt++ = extract(sum);
            sum = down(sum);
          }
          if (--h >= ++l)
            prod = (unsigned long)(*h) * (unsigned long)(*l);
          else
            break;
        }
        else
          break;
      }
      if (loa > bota)
        --loa;
      else
        --hia;
    }
    r.checklength();
  }
}

void multiply(Integer& x, long y, Integer& r)
{
  int xl = x.rep->len;
    
  if (xl == 0 || y == 0)
    r.copy(_Integer_Zero);
  else
  {
    int ysgn = y >= 0;
    int rsgn = x.rep->sgn == ysgn;
    unsigned long uy = (ysgn)? y : -y;
    unsigned short tmp[SHORT_PER_LONG];
    int yl = 0;
    while (uy != 0)
    {
      tmp[yl++] = extract(uy);
      uy = down(uy);
    }

    int xrsame = x.rep == r.rep;
    int rl = xl + yl;
    r.setlength(rl, CLEAR_TRAILING);
    r.rep->sgn = rsgn;

    unsigned short* rs = r.rep->s;
    unsigned short* topr = &(rs[rl]);
    unsigned short* currentr;
    unsigned short* bota;
    unsigned short* as;
    unsigned short* botb;
    unsigned short* topb;

    if (xrsame)
    { 
      currentr = &(rs[xl-1]);
      bota = rs;
      as = currentr;
      botb = tmp;
      topb = &(botb[yl]);
    }
    else if (xl <= yl)
    {
      currentr = &(rs[xl-1]);
      bota = x.rep->s;
      as = &(bota[xl-1]);
      botb = tmp;
      topb = &(botb[yl]);
    }
    else
    {
      currentr = &(rs[yl-1]);
      bota = tmp;
      as = &(bota[yl-1]);
      botb = x.rep->s;
      topb = &(botb[xl]);
    }

    while (as >= bota)
    {
      unsigned long ai = (unsigned long)(*as--);
      unsigned short* rs = currentr--;
      *rs = 0;
      if (ai != 0)
      {
        unsigned long sum = 0;
        unsigned short* bs = botb;
        while (bs < topb)
        {
          sum += ai * (unsigned long)(*bs++) + (unsigned long)(*rs);
          *rs++ = extract(sum);
          sum = down(sum);
        }
        while (sum != 0 && rs < topr)
        {
          sum += (unsigned long)(*rs);
          *rs++ = extract(sum);
          sum = down(sum);
        }
      }
    }
    r.checklength();
  }
}

// Integer divide by single digit
// if code == 'r', only the remainder is allocated & kept
// if code == 'q', only the quotient it allocated & kept
// otherwise (default) both are

void divide(Integer& x, unsigned short y, Integer& q, int& r, char code = 0)
{
  int xl = x.rep->len;
  if (y == 0)
    x.error("attempted division by zero");
  else if (xl == 0)
  {
    if (code != 'r')
      q.copy(_Integer_Zero);
    if (code != 'q')
      r = 0;
  }
  else if (y == 1)
  {
    if (code != 'r')
      q = x;
    if (code != 'q')
      r = 0;
  }
  else if (code != 'r')
  {
    int xqsame = x.rep == q.rep;
    q.setlength(xl);
    q.rep->sgn = x.rep->sgn;
    unsigned short* botq = q.rep->s;
    unsigned short* qs = &(botq[xl - 1]);
    unsigned short* xs = (xqsame)? qs : &(x.rep->s[xl - 1]);
    unsigned long rem = 0;
    while (qs >= botq)
    {
      rem = up(rem) | *xs--;
      if (rem <= MAXLONG)       // avoid calling udiv if possible
      {
        long srem = rem;
        *qs-- = extract(srem / y);
        rem = srem % y;
      }
      else
      {
        unsigned long u = rem / y;
        *qs-- = extract(u);
        rem -= u * y;
      }
    }
    q.checklength();
    if (code != 'q')
    {
      r = extract(rem);
      if (q.rep->sgn == I_NEGATIVE)
        r = -r;
    }
  }
  else
  {
    unsigned short* botx = x.rep->s;
    unsigned short* xs = &(botx[xl - 1]);
    unsigned long rem = 0;
    while (xs >= botx)
    {
      rem = up(rem) | *xs--;
      if (rem <= MAXLONG)
      {
        long srem = rem;
        rem = srem % y;
      }
      else
      {
        unsigned long u = rem / y;
        rem -= u * y;
      }
    }
    r = extract(rem);
    if (x.rep->sgn == I_NEGATIVE)
      r = -r;
  }
}

void divide(Integer& x, Integer& y, Integer& quotient, Integer& remainder,
            char code = 0)
{
  int yl = y.rep->len;
  if (yl == 0)
    x.error("attempted division by zero");

  int comp = ucompare(x, y);
  int xsgn = x.rep->sgn;
  int ysgn = y.rep->sgn;

  int samesign = xsgn == ysgn;

  if (comp < 0)
  {
    if (code != 'r')
      quotient.copy(_Integer_Zero);
    if (code != 'q')
      remainder = x;
    return;
  }
  else if (comp == 0)
  {
    if (code != 'r')
      quotient.copy(_Integer_One.rep->s, SHORT_PER_LONG, samesign);
    if (code != 'q')
      remainder.copy(_Integer_Zero);
    return;
  }
  else if (yl == 1)
  {
    int rem;
    if (code != 'r')
    {
      divide(x, y.rep->s[0], quotient, rem);
      quotient.rep->sgn = samesign;
      if (code != 'q')
      {
        remainder.copy(rem);
        if (rem != 0)
          remainder.rep->sgn = xsgn;
      }
    }
    else
    {
      divide(x, y.rep->s[0], remainder, rem, 'r');
      remainder.copy(rem);
      if (rem != 0)
        remainder.rep->sgn = xsgn;
    }
    return;
  }


  Integer yy;
  unsigned short prescale = (I_RADIX / (1 + y.rep->s[yl - 1]));
  if (prescale != 1)
    multiply(y, ((long)prescale & I_MAXNUM), yy);
  else
    yy = y;
  unsigned short* ys = yy.rep->s;
  unsigned short* topy = &(ys[yl]);
  unsigned short d1 = ys[yl - 1];
  unsigned short d2 = ys[yl - 2];

  Integer r;               // conditionally using `r' minimizes copying
  int rl = x.rep->len;
  unsigned short* rs;

  if (code != 'q')
  {
    if (prescale != 1)
      multiply(x, ((long)prescale & I_MAXNUM), remainder);
    else
      remainder.copy(x);
    rs = remainder.rep->s;
  }
  else
  {
    if (prescale != 1)
      multiply(x, ((long)prescale & I_MAXNUM), r);
    else
      r.copy(x);
    rs = r.rep->s;
  }

  int ql = rl - yl + 1;
  unsigned short* qs;
  
  if (code != 'r')
  {
    quotient.setlength(ql, CLEAR_ALL);
    quotient.rep->sgn = samesign;
    qs = quotient.rep->s;
  }
  
  int l = ql - 1;
  int i = l + yl;
  
  for (; l >= 0; --l, --i)
  {
    unsigned short qhat;       // guess q
    if (d1 == rs[i])
      qhat = I_MAXNUM;
    else
    {
      unsigned long lr = up((unsigned long)rs[i]) | rs[i-1];
      qhat = lr / d1;
    }

    for(;;)     // adjust q, use docmp to avoid overflow problems
    {
      unsigned short ts[3];
      unsigned long prod = (unsigned long)d2 * (unsigned long)qhat;
      ts[0] = extract(prod);
      prod = down(prod) + (unsigned long)d1 * (unsigned long)qhat;
      ts[1] = extract(prod);
      ts[2] = extract(down(prod));
      if (docmp(ts, &(rs[i-2]), 3) > 0)
        --qhat;
      else
        break;
    };
    
    // multiply & subtract
    
    unsigned short* yt = ys;
    unsigned short* rt = &(rs[l]);
    unsigned long prod = 0;
    unsigned long hi = 1;
    while (yt < topy)
    {
      prod = (unsigned long)qhat * (unsigned long)(*yt++) + down(prod);
      hi += (unsigned long)(*rt) + I_MAXNUM - (unsigned long)(extract(prod));
      *rt++ = extract(hi);
      hi = down(hi);
    }
    hi += (unsigned long)(*rt) + I_MAXNUM - (unsigned long)(down(prod));
    *rt = extract(hi);
    hi = down(hi);
    
    // off-by-one, add back
    
    if (hi == 0)
    {
      --qhat;
      yt = ys;
      rt = &(rs[l]);
      hi = 0;
      while (yt < topy)
      {
        hi = (unsigned long)(*rt) + (unsigned long)(*yt++) + down(hi);
        *rt++ = extract(hi);
      }
      *rt = 0;
    }
    if (code != 'r')
      qs[l] = qhat;
  }

  if (code != 'q')
  {
    int junk;
    remainder.checklength();
    divide(remainder, prescale, remainder, junk);
  }
  if (code != 'r')
    quotient.checklength();
}

void divide(Integer& x, long y, Integer& quotient, Integer& remainder,
            char code = 0)
{
  mk_tmpInteger(y);
  divide(x, _tmpInteger, quotient, remainder, code);
}

void lshift(Integer& x, long y, Integer& r)
{
  int xl = x.rep->len;
  if (xl == 0 || y == 0)
  {
    r = x;
    return;
  }

  int xrsame = x.rep == r.rep;
  int rsgn = x.rep->sgn;

  long ay = (y < 0)? -y : y;
  int bw = ay / I_SHIFT;
  int sw = ay % I_SHIFT;

  if (y > 0)
  {
    int rl = bw + xl + 1;
    r.setlength(rl);
    r.rep->sgn = rsgn;
    unsigned short* botr = r.rep->s;
    unsigned short* botx = (xrsame)? botr : x.rep->s;
    unsigned short* rs = &(botr[rl - 1]);
    unsigned short* xs = &(botx[xl - 1]);
    unsigned long a = 0;
    while (xs >= botx)
    {
      a = up(a) | ((unsigned long)(*xs--) << sw);
      *rs-- = extract(down(a));
    }
    *rs-- = extract(a);
    while (rs >= botr)
      *rs-- = 0;
  }
  else
  {
    int rl = xl - bw;
    if (rl <= 0)
      r.copy(_Integer_Zero);
    else
    {
      r.setlength(rl);
      r.rep->sgn = rsgn;
      int rw = I_SHIFT - sw;
      unsigned short* rs = r.rep->s;
      unsigned short* topr = &(rs[rl]);
      unsigned short* botx = (xrsame)? rs : x.rep->s;
      unsigned short* xs =  &(botx[bw]);
      unsigned short* topx = &(botx[xl]);
      unsigned long a = (unsigned long)(*xs++) >> sw;
      while (xs < topx)
      {
        a |= (unsigned long)(*xs++) << rw;
        *rs++ = extract(a);
        a = down(a);
      }
      *rs++ = extract(a);
      while (rs < topr)
        *rs++ = 0;
    }
  }
  r.checklength();
}


void bitop(Integer& x, Integer& y, Integer& r, char op)
{
  int xl = x.rep->len;
  int yl = y.rep->len;
  int xrsame = x.rep == r.rep;
  int yrsame = y.rep == r.rep;
  r.setlength(calc_len(xl, yl, 0), CLEAR_TRAILING);
  r.rep->sgn = x.rep->sgn;
  unsigned short* rs = r.rep->s;
  unsigned short* topr = &(rs[r.rep->len]);
  unsigned short* as;
  unsigned short* bs;
  unsigned short* topb;
  if (xl >= yl)
  {
    as = (xrsame)? rs : x.rep->s;
    bs = (yrsame)? rs : y.rep->s;
    topb = &(bs[yl]);
  }
  else
  {
    bs = (xrsame)? rs : x.rep->s;
    topb = &(bs[xl]);
    as = (yrsame)? rs : y.rep->s;
  }

  switch (op)
  {
  case '&':
    while (bs < topb) *rs++ = *as++ & *bs++;
    while (rs < topr) *rs++ = 0;
    break;
  case '|':
    while (bs < topb) *rs++ = *as++ | *bs++;
    while (rs < topr) *rs++ = *as++;
    break;
  case '^':
    while (bs < topb) *rs++ = *as++ ^ *bs++;
    while (rs < topr) *rs++ = *as++;
    break;
  }
  r.checklength();
}

void bitop(Integer& x, long y, Integer& r, char op)
{
  mk_tmpInteger(y);
  bitop(x, _tmpInteger, r, op);
}

Integer  Integer::operator -()
{
  Integer x; 
  x.copy(rep->s, rep->len, !rep->sgn);
  return x; 
}

Integer  Integer::operator ~()
{
  Integer x;
  x.copy(rep->s, rep->len, rep->sgn);
  unsigned short* s = x.rep->s;
  unsigned short* top = &(s[x.rep->len]);
  while (s < top)
  {
    unsigned short cmp = ~(*s);
    *s++ = cmp;
  }
  x.checklength();
  return x;
}

void setbit(Integer& x, long b)
{
  if (b >= 0)
  {
    int bw = b / I_SHIFT;
    int sw = b % I_SHIFT;
    if (x.rep->len <= bw || x.rep == &_nil_Irep || x.rep->ref != 1) 
      x.setlength(calc_len(x.rep->len, bw+1, 0), CLEAR_TRAILING);
    x.rep->s[bw] |= (1 << sw);
    x.checklength();
  }
}

void clearbit(Integer& x, long b)
{
  if (b >= 0)
  {
    int bw = b / I_SHIFT;
    int sw = b % I_SHIFT;
    if (x.rep->len <= bw || x.rep == &_nil_Irep || x.rep->ref != 1) 
      x.setlength(calc_len(x.rep->len, bw+1, 0), CLEAR_TRAILING);
    x.rep->s[bw] &= ~(1 << sw);
    x.checklength();
  }
}

int testbit(Integer& x, long b)
{
  if (b >= 0)
  {
    int bw = b / I_SHIFT;
    int sw = b % I_SHIFT;
    return (bw < x.rep->len && (x.rep->s[bw] & (1 << sw)) != 0);
  }
  else
    return 0;
}


// A  version of knuth's algorithm B / ex. 4.5.3.34
// A better version that doesn't bother shifting all of `t' forthcoming

Integer gcd(Integer& x, Integer& y)
{
  int ul = x.rep->len;
  int vl = y.rep->len;

  Integer u;
  u.copy(x.rep->s, ul, I_POSITIVE);
  if (vl == 0)
    return u;

  Integer v;
  v.copy(y.rep->s, vl, I_POSITIVE);
  if (ul == 0)
    return v;

// find shift so that both not even

  long k = 0;
  int l = ul >? vl;
  int cont = 1;
  for (int i = 0;  i < l && cont; ++i)
  {
    unsigned long a =  (i < ul)? u.rep->s[i] : 0;
    unsigned long b =  (i < vl)? v.rep->s[i] : 0;
    for (int j = 0; j < I_SHIFT; ++j)
    {
      if ((a | b) & 1)
      {
        cont = 0;
        break;
      }
      else
      {
        ++k;
        a >>= 1;
        b >>= 1;
      }
    }
  }

  if (k != 0)
  {
    lshift(u, -k, u);
    lshift(v, -k, v);
  }

  Integer  t;
  if (odd(u))
    t.copy(v.rep->s, v.rep->len, !v.rep->sgn);
  else
    t.copy(u.rep->s, u.rep->len, u.rep->sgn);

  for(;;)
  {
    long s = 0;                 // shift t until odd
    cont = 1;
    int tl = t.rep->len;
    for (i = 0; i < tl && cont; ++i)
    {
      unsigned long a = t.rep->s[i];
      for (int j = 0; j < I_SHIFT; ++j)
      {
        if (a & 1)
        {
          cont = 0;
          break;
        }
        else
        {
          ++s;
          a >>= 1;
        }
      }
    }

    if (s != 0)
      lshift(t, -s, t);

    if (sign(t) > 0)
    {
      u.copy(t.rep->s, t.rep->len, t.rep->sgn);
      t -= v;
    }
    else
    {
      v.copy(t.rep->s, t.rep->len, !t.rep->sgn);
      t += u;
    }
    if (t.rep->len == 0)
      break;
  }
  if (k != 0)
    lshift(u, k, u);
  return u;
}

Integer lcm(Integer& x, Integer& y)
{
  Integer q;
  Integer g = gcd(x, y);
  if (ucompare(g, _Integer_One) != 0)
  {
    divide(x, g, q, q, 'q');
    multiply(q, y, q);
  }
  else
    multiply(x, y, q);
  return q;
}


long lg(Integer& x)
{
  int xl = x.rep->len;
  if (xl == 0)
    return 0;

  long l = (xl - 1) * I_SHIFT - 1;
  unsigned short a = x.rep->s[xl-1];

  while (a != 0)
  {
    a = a >> 1;
    ++l;
  }
  return l;
}
  
Integer pow(Integer& x, long y)
{
  int sgn;
  if (x.rep->sgn == I_POSITIVE || (!(y & 1)))
    sgn = I_POSITIVE;
  else
    sgn = I_NEGATIVE;

  int xl = x.rep->len;

  Integer r;

  if (xl == 0 || y < 0)
    r.copy(_Integer_Zero);
  else if (y == 0 || (xl == 1 && x.rep->s[0] == 1))
    r.copy(_Integer_One.rep->s, SHORT_PER_LONG, sgn);
  else
  {
    int maxsize = ((lg(x) + 1) * y) / I_SHIFT + 2;     // pre-allocate space
    r.setlength(maxsize);
    Integer b;
    b.setlength(maxsize);
    b.copy(x.rep->s, xl, I_POSITIVE);
    int firsttime = 1;
    for(;;)
    {
      if (y & 1)
      {
        if (firsttime)
        {
          firsttime = 0;
          r.copy(b);
        }
        else
          multiply(r, b, r);
      }
      if ((y >>= 1) == 0)
        break;
      else
        multiply(b, b, b);
    }
    r.rep->sgn = sgn;
  }
  return r;
}

// a simple-minded random number gen.

Integer rnd(Integer& x)
{
  extern int rand();

  if (sign(x) == 0)
    return x;

  Integer r;
  r.setlength(x.rep->len + 1);
  for (int i = 0; i < r.rep->len; ++i)
    r.rep->s[i] = extract((unsigned long)(rand()));
  r.rep->sgn = x.rep->sgn;
  Integer rem;
  divide(r, x, rem, rem, 'r');
  return rem;
}


Integer sqrt(Integer& x)
{
  Integer r;

  int s = sign(x);
  if (s < 0)
    x.error("Attempted square root of negative Integer");
  else if (s == 0)
    r = 0;
  else
  {
    r = x;
    r >>= lg(x) / 2;            // get close
    Integer q;
    for(;;)                     // then newton approx
    {
      divide(x, r, q, q, 'q');
      if (q >= r)
        break;
      else
      {
        r += q;
        r >>= 1;
      }
    }
  }
  return r;
}



