// This may look like C code, but it is really -*- C++ -*-
/* 
Copyright (C) 1988 Free Software Foundation
    written by Doug Lea (dl@rocky.oswego.edu)

This file is part of GNU CC.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the GNU CC General Public
License for full details.

Everyone is granted permission to copy, modify and redistribute
GNU CC, but only under the conditions described in the
GNU CC General Public License.   A copy of this license is
supposed to have been given to you along with GNU CC so you
can know your rights and responsibilities.  It should be in a
file named COPYING.  Among other things, the copyright notice
and this notice must be preserved on all copies.  
*/


#ifndef _String_h
#define _String_h 1

#include <stream.h>

struct _Srep                     // internal String representations
{
  unsigned short    len;         // string length 
  unsigned short    sz;          // allocated space
  short             ref;         // reference count;
  char              s[1];        // the string starts here 
                                 // (at least 1 char for trailing null)
                                 // allocated & expanded via non-public fcts
};


class String;
class SubString;


struct re_pattern_buffer;       // defined elsewhere
struct re_registers;

class Regex
{
  friend class       String;
  friend class       SubString;

  re_pattern_buffer* buf;
  re_registers*      reg;

  void               initialize(const char* t, int tlen, int fast, 
                                int bufsize, const char* transtable);

public:
                     Regex(const char* t, 
                           int fast = 0, 
                           int bufsize = 40, 
                           const char* transtable = 0);

                     Regex(String& x, 
                           int fast = 0, 
                           int bufsize = 40, 
                           const char* transtable = 0);

                     ~Regex();

  int                match(const char* s, int len, int pos = 0);
  int                search(const char* s, int len, 
                            int& matchlen, int startpos = 0);
};


class SubString
{
  friend class      String;

  String*           S;
  unsigned short    pos;
  unsigned short    len;

  void              assign(_Srep*, const char*, int = -1);
                    SubString(String* x, int p, int l);
                    SubString(SubString& x);

public:
                    ~SubString();

  void              operator =  (String&     y);
  void              operator =  (SubString&  y);
  void              operator =  (const char* t);
  void              operator =  (char        c);
  
  friend String     operator +  (SubString& x, String&     y);
  friend String     operator +  (SubString& x, SubString&  y);
  friend String     operator +  (SubString& x, const char* t);
  friend String     operator +  (SubString& x, char        c);

  friend int        operator == (SubString& x, String&     y);
  friend int        operator == (SubString& x, SubString&  y);
  friend int        operator == (SubString& x, const char* t);
  
  friend int        operator != (SubString& x, String&     y);
  friend int        operator != (SubString& x, SubString&  y);
  friend int        operator != (SubString& x, const char* t);
  
  friend int        operator <= (SubString& x, String&     y);
  friend int        operator <= (SubString& x, SubString&  y);
  friend int        operator <= (SubString& x, const char* t);
  
  friend int        operator <  (SubString& x, String&     y);
  friend int        operator <  (SubString& x, SubString&  y);
  friend int        operator <  (SubString& x, const char* t);
  
  friend int        operator >= (SubString& x, String&     y);
  friend int        operator >= (SubString& x, SubString&  y);
  friend int        operator >= (SubString& x, const char* t);
  
  friend int        operator >  (SubString& x, String&     y);
  friend int        operator >  (SubString& x, SubString&  y);
  friend int        operator >  (SubString& x, const char* t);

  int               contains(char        c);
  int               contains(String&     y);
  int               contains(SubString&  y);
  int               contains(const char* t);
  int               contains(Regex&       r);

  int               matches(Regex&  r);

// misc

  int               length();
  int               empty();
  const char*       operator char*();

  friend ostream&   operator<<(ostream& s, SubString& x);

// non-operator versions

  friend int        compare(SubString& x, String&     y);
  friend int        compare(SubString& x, SubString&  y);
  friend int        compare(SubString& x, const char* y);

  friend void       concat(SubString&  x, String&     y, String& result);
  friend void       concat(SubString&  x, SubString&  y, String& result);
  friend void       concat(SubString&  x, const char* y, String& result);
  friend void       concat(SubString&  x, char        y, String& result);
  friend void       concat(String&     x, SubString&  y, String& result);
};


class String
{
  friend class      SubString;

  _Srep*            rep;

  int               search(int, int, const char*, int = -1);
  int               search(int, int, char);
  int               match(int, int, int, const char*, int = -1);
  void              _gsub(const char*, int, const char* ,int);
  void              _gsub(Regex&, const char*, int);

public:

// constructors & assignment

                    String();
                    String(String&     x);
                    String(SubString&  x);
                    String(const char* t);
                    String(const char* t, int len);
                    String(char c);

                    ~String();

  String&           operator =  (String&     y);
  String&           operator =  (const char* y);
  String&           operator =  (char        c);
  String&           operator =  (SubString&  y);

// concatenation

  friend String     operator +  (String& x, String&     y);     
  friend String     operator +  (String& x, const char* t);
  friend String     operator +  (String& x, char        c);
  friend String     operator +  (String& x, SubString&  y);     

  String&           operator += (String&     y);
  String&           operator += (SubString&  y);
  String&           operator += (const char* t);
  String&           operator += (char        c);

// relational operators

  friend int        operator == (String&     x, String&     y);
  friend int        operator == (String&     x, const char* t);
  friend int        operator == (String&     x, SubString&  y);
  
  friend int        operator != (String&     x, String&     y);
  friend int        operator != (String&     x, const char* t);
  friend int        operator != (String&     x, SubString&  y);
  
  friend int        operator <= (String&     x, String&     y);
  friend int        operator <= (String&     x, const char* t);
  friend int        operator <= (String&     x, SubString&  y);
  
  friend int        operator <  (String&     x, String&     y);
  friend int        operator <  (String&     x, const char* t);
  friend int        operator <  (String&     x, SubString&  y);
  
  friend int        operator >= (String&     x, String&     y);
  friend int        operator >= (String&     x, const char* t);
  friend int        operator >= (String&     x, SubString&  y);
  
  friend int        operator >  (String&     x, String&     y);
  friend int        operator >  (String&     x, const char* t);
  friend int        operator >  (String&     x, SubString&  y);

  friend int        fcompare(String&   x, String&     y); // ignore case

// searching & matching

  int               index(char        c, int startpos = 0);      
  int               index(String&     y, int startpos = 0);      
  int               index(SubString&  y, int startpos = 0);      
  int               index(const char* t, int startpos = 0);  
  int               index(Regex&      r, int startpos = 0);       

  int               contains(char        c);
  int               contains(String&     y);
  int               contains(SubString&  y);
  int               contains(const char* t);
  int               contains(Regex&      r);

  int               contains(char        c, int pos);
  int               contains(String&     y, int pos);
  int               contains(SubString&  y, int pos);
  int               contains(const char* t, int pos);
  int               contains(Regex&      r, int pos);

  int               matches(char        c, int pos = 0);
  int               matches(String&     y, int pos = 0);
  int               matches(SubString&  y, int pos = 0);
  int               matches(const char* t, int pos = 0);
  int               matches(Regex&      r, int pos = 0);

// substring extraction

  SubString         at(int         pos, int len);
  SubString         at(String&     x, int startpos = 0); 
  SubString         at(SubString&  x, int startpos = 0); 
  SubString         at(const char* t, int startpos = 0);
  SubString         at(char        c, int startpos = 0);
  SubString         at(Regex&      r, int startpos = 0); 

  SubString         before(int          pos);
  SubString         before(String&      x, int startpos = 0);
  SubString         before(SubString&   x, int startpos = 0);
  SubString         before(const char*  t, int startpos = 0);
  SubString         before(char         c, int startpos = 0);
  SubString         before(Regex&       r, int startpos = 0);

  SubString         after(int         pos);
  SubString         after(String&     x, int startpos = 0);
  SubString         after(SubString&  x, int startpos = 0);
  SubString         after(const char* t, int startpos = 0);
  SubString         after(char        c, int startpos = 0);
  SubString         after(Regex&      r, int startpos = 0);

// modification

  void              del(int         pos, int len);
  void              del(String&     y, int startpos = 0);
  void              del(SubString&  y, int startpos = 0);
  void              del(const char* t, int startpos = 0);
  void              del(char        c, int startpos = 0);
  void              del(Regex&      r, int startpos = 0);

  void              gsub(String&     pat, String&     repl);
  void              gsub(SubString&  pat, String&     repl);
  void              gsub(const char* pat, String&     repl);
  void              gsub(const char* pat, const char* repl);
  void              gsub(Regex&      pat, String&     repl);

// friends & utilities

  friend int        decompose(String& x, String& l, String& m, String& r,
                              Regex& pat, int startpos = 0);

  friend int        split(String& x, String res[], int maxn, String& sep);
  friend int        split(String& x, String res[], int maxn, Regex&  sep);

  friend String     join(String src[], int n, String& sep);

  friend String     reverse(String& x);
  friend String     upcase(String& x);
  friend String     downcase(String& x);
  friend String     capitalize(String& x);

  friend String     replicate(char        c, int n);
  friend String     replicate(String&     y, int n);

  friend String     common_prefix(String& x, String& y, int startpos = 0);
  friend String     common_suffix(String& x, String& y, int startpos = -1);

// conversion

  char              operator [] (int i);
  const char*       operator char*();

// IO

  friend ostream&   operator<<(ostream& s, String& x);
  friend istream&   operator>>(istream& s, String& x);

// status

  int               length();
  int               empty();
  int               operator ! ();
  void              make_unique();
  void              setlength(int l);
  void              error(char* msg);

// non-operator versions

  friend int        compare(String&    x, String&     y);
  friend int        compare(String&    x, SubString&  y);
  friend int        compare(String&    x, const char* y);
  friend int        compare(SubString& x, String&     y);

  friend void       concat(String& x, String&     y, String& result);
  friend void       concat(String& x, SubString&  y, String& result);
  friend void       concat(String& x, const char* y, String& result);
  friend void       concat(String& x, char        y, String& result);

  void              copy(const char* y, int len = -1);
  void              copy(String& y);
  void              copy(SubString& y);
  void              copy(char y);
};



// some built in regular expressions

extern Regex RXwhite;          // = "[ \n\t]+"
extern Regex RXint;            // = "-?[0-9]+"
extern Regex RXdouble;         // = "-?\\(\\([0-9]+\\.[0-9]*\\)\\|
                               //    \\([0-9]+\\)\\|\\(\\.[0-9]+\\)\\)
                               //    \\([eE][---+]?[0-9]+\\)?"
extern Regex RXalpha;          // = "[A-Za-z]+"
extern Regex RXlowercase;      // = "[a-z]+"
extern Regex RXuppercase;      // = "[A-Z]+"
extern Regex RXalphanum;       // = "[0-9A-Za-z]+"
extern Regex RXidentifier;     // = "[A-Za-z_][A-Za-z0-9_]*"


extern void default_String_error_handler(char*);
extern one_arg_error_handler_t String_error_handler;

extern one_arg_error_handler_t 
        set_String_error_handler(one_arg_error_handler_t f);

//#ifdef __OPTIMIZE__


extern _Srep  _nil_Srep;

inline String::String()
{ 
  rep = &_nil_Srep;
}

inline String::String(String& x)
{ 
  rep = x.rep; if (rep->ref > 0) rep->ref++;
}

inline String::String(const char* t)
{
  rep = &_nil_Srep; copy(t);
}

inline String::String(const char* t, int tlen)
{
  rep = &_nil_Srep; copy(t, tlen);
}

inline String::String(SubString& y)
{
  rep = &_nil_Srep; copy(&(y.S->rep->s[y.pos]), y.len);
}

inline String::String(char c)
{
  rep = &_nil_Srep; char ch = c; copy(&ch, 1);
}

inline String::~String()
{ 
  if (rep->ref > 0 && --rep->ref == 0) delete rep;
}

inline String& String::operator =  (String& y)
{ 
  y.rep->ref++;
  if (rep->ref > 0 && --rep->ref == 0) delete rep;
  rep = y.rep;
  return *this; 
}

inline String& String::operator=(const char* t)
{
  copy(t); return *this;
}

inline String& String::operator=(SubString&  y)
{
  copy(&(y.S->rep->s[y.pos]), y.len); return *this;
}

inline String& String::operator=(char c)
{
  char ch = c;  copy(&ch, 1); return *this;
}


inline void String::copy(char c)
{
  char ch = c; copy(&ch, 1);
}

inline void String::copy(String& y)
{
  copy(y.rep->s, y.rep->len);
}

inline void String::copy(SubString& y)
{
  copy(&(y.S->rep->s[y.pos]), y.len);
}

inline SubString::SubString(SubString& x)
{ 
  S = x.S; pos = x.pos;   len = x.len; 
}

inline SubString::~SubString() {}


inline void SubString::operator = (const char* ys)
{
  assign(0, ys);
}

inline void SubString::operator = (char ch)
{
  assign(0, &ch, 1);
}

inline void SubString::operator = (String& y)
{
  assign(y.rep, y.rep->s, y.rep->len);
}

inline void SubString::operator = (SubString& y)
{
  assign(y.S->rep, &(y.S->rep->s[y.pos]), y.len);
}

inline String& String::operator+=(String& y)
{
  concat(*this, y, *this); return *this;
}

inline String& String::operator+=(SubString&  y)
{
  concat(*this, y, *this); return *this;
}

inline String& String::operator+=(const char* t)
{
  concat(*this, t, *this); return *this;
}

inline String& String::operator+=(char c)
{
  concat(*this, c, *this); return *this;
}

inline String operator + (String& x, String& y)
{
  String r; concat(x, y, r); return r;
}

inline String operator + (String& x, const char* t)
{
  String r; concat(x, t, r); return r;
}

inline String operator + (String& x, SubString& y)
{
  String r; concat(x, y, r); return r;
}

inline String operator + (String& x, char c)
{
  String r; concat(x, c, r); return r;
}

inline String operator + (SubString& x, String& y)
{
  String r; concat(x, y, r); return r;
}

inline String operator+(SubString& x, const char* y)
{
  String r; concat(x, y, r); return r;
}

inline String operator+(SubString& x, char c)
{
  String r; concat(x, c, r); return r;
}

inline String operator+(SubString& x, SubString& y)
{
  String r; concat(x, y, r); return r;
}
  
inline int String::length()
{ 
  return rep->len;
}

inline int String::empty()
{ 
  return rep->len == 0;
}

inline int String::operator ! ()
{ 
  return rep->len == 0;
}

inline char  String::operator [] (int i) 
{ 
  return rep->s[i];
}

inline int String::index(char c, int startpos = 0)
{
  return search(startpos, rep->len, c);
}

inline int String::index(const char* t, int startpos = 0)
{   
  return search(startpos, rep->len, t);
}

inline int String::index(String& y, int startpos = 0)
{   
  return search(startpos, rep->len, y.rep->s, y.rep->len);
}

inline int String::index(SubString& y, int startpos = 0)
{   
  return search(startpos, rep->len, &(y.S->rep->s[y.pos]), y.len);
}

inline int String::contains(char c)
{
  return search(0, rep->len, c) >= 0;
}

inline int SubString::contains(char c)
{
  return S->search(pos, pos+len, 0, c) >= 0;
}

inline int String::contains(const char* t)
{   
  return search(0, rep->len, t) >= 0;
}

inline int String::contains(String& y)
{   
  return search(0, rep->len, y.rep->s, y.rep->len) >= 0;
}

inline int String::contains(SubString& y)
{   
  return search(0, rep->len, &(y.S->rep->s[y.pos]), y.len) >= 0;
}

inline int SubString::contains(const char* t)
{   
  return S->search(pos, pos+len, t) >= 0;
}

inline int SubString::contains(String& y)
{   
  return S->search(pos, pos+len, y.rep->s, y.rep->len) >= 0;
}

inline int SubString::contains(SubString&  y)
{   
  return S->search(pos, pos+len, &(y.S->rep->s[y.pos]), y.len) >= 0;
}

inline int String::contains(char c, int p)
{
  char ch = c; return match(p, rep->len, 0, &ch, 1);
}

inline int String::matches(char c, int p = 0)
{
  char ch = c; return match(p, rep->len, 1, &ch, 1);
}

inline int String::contains(const char* t, int p)
{
  return match(p, rep->len, 0, t);
}

inline int String::matches(const char* t, int p = 0)
{
  return match(p, rep->len, 1, t);
}

inline int String::contains(String& y, int p)
{
  return match(p, rep->len, 0, y.rep->s, y.rep->len);
}

inline int String::matches(String& y, int p = 0)
{
  return match(p, rep->len, 1, y.rep->s, y.rep->len);
}

inline int String::contains(SubString& y, int p)
{
  return match(p, rep->len, 0, &(y.S->rep->s[y.pos]), y.len);
}

inline int String::matches(SubString& y, int p = 0)
{
  return match(p, rep->len, 1, &(y.S->rep->s[y.pos]), y.len);
}

inline int String::contains(Regex& r)
{
  int unused;
  return r.search(rep->s, rep->len, unused, 0);
}

inline int SubString::contains(Regex& r)
{
  int unused;
  return r.search(&(S->rep->s[pos]), len, unused, 0);
}

inline int String::contains(Regex& r, int p)
{
  return r.match(rep->s, rep->len, p);
}

inline int String::matches(Regex& r, int p = 0)
{
  int l = (p < 0)? -p : rep->len - p;
  return r.match(rep->s, rep->len, p) == l;
}

inline int SubString::matches(Regex& r)
{
  return r.match(&(S->rep->s[pos]), len, 0) == len;
}

inline const char* String::operator char*()
{ 
  return rep->s;
}

inline int String::index(Regex& r, int startpos = 0)
{
  int mlen;  return r.search(rep->s, rep->len, mlen, startpos);
}

inline const char* SubString::operator char*()
{ 
  return (char*)(String(*this)); 
}

inline  int SubString::length()
{ 
  return len;
}

inline  int SubString::empty()
{ 
  return len == 0;
}

inline  ostream& operator<<(ostream& s, String& x)
{ 
  s.put(x.rep->s); return s;
}

inline int operator==(String& x, String& y) 
{
  return x.rep->len == y.rep->len && compare(x, y) == 0; 
}

inline int operator!=(String& x, String& y)
{
  return x.rep->len != y.rep->len || compare(x, y) != 0; 
}

inline int operator>(String& x, String& y)
{
  return compare(x, y) > 0; 
}

inline int operator>=(String& x, String& y)
{
  return compare(x, y) >= 0; 
}

inline int operator<(String& x, String& y)
{
  return compare(x, y) < 0; 
}

inline int operator<=(String& x, String& y)
{
  return compare(x, y) <= 0; 
}

inline int operator==(String& x, SubString&  y) 
{
  return x.rep->len == y.len && compare(x, y) == 0; 
}

inline int operator!=(String& x, SubString&  y)
{
  return x.rep->len != y.len || compare(x, y) != 0; 
}

inline int operator>(String& x, SubString&  y)      
{
  return compare(x, y) > 0; 
}

inline int operator>=(String& x, SubString&  y)
{
  return compare(x, y) >= 0; 
}

inline int operator<(String& x, SubString&  y) 
{
  return compare(x, y) < 0; 
}

inline int operator<=(String& x, SubString&  y)
{
  return compare(x, y) <= 0; 
}

inline int operator==(String& x, const char* t) 
{
  return compare(x, t) == 0; 
}

inline int operator!=(String& x, const char* t) 
{
  return compare(x, t) != 0; 
}

inline int operator>(String& x, const char* t)  
{
  return compare(x, t) > 0; 
}

inline int operator>=(String& x, const char* t) 
{
  return compare(x, t) >= 0; 
}

inline int operator<(String& x, const char* t)  
{
  return compare(x, t) < 0; 
}

inline int operator<=(String& x, const char* t) 
{
  return compare(x, t) <= 0; 
}

inline int operator==(SubString& x, String& y) 
{
  return x.len == y.rep->len && compare(y, x) == 0; 
}

inline int operator!=(SubString& x, String& y)
{
  return x.len != y.rep->len || compare(y, x) != 0;
}

inline int operator>(SubString& x, String& y)      
{
  return compare(y, x) < 0;
}

inline int operator>=(SubString& x, String& y)     
{
  return compare(y, x) <= 0;
}

inline int operator<(SubString& x, String& y)      
{
  return compare(y, x) > 0;
}

inline int operator<=(SubString& x, String& y)     
{
  return compare(y, x) >= 0;
}

inline int operator==(SubString& x, SubString&  y) 
{
  return x.len == y.len && compare(x, y) == 0; 
}

inline int operator!=(SubString& x, SubString&  y)
{
  return x.len != y.len || compare(x, y) != 0;
}

inline int operator>(SubString& x, SubString&  y)      
{
  return compare(x, y) > 0;
}

inline int operator>=(SubString& x, SubString&  y)
{
  return compare(x, y) >= 0;
}

inline int operator<(SubString& x, SubString&  y) 
{
  return compare(x, y) < 0;
}

inline int operator<=(SubString& x, SubString&  y)
{
  return compare(x, y) <= 0;
}

inline int operator==(SubString& x, const char* t) 
{
  return compare(x, t) == 0; 
}

inline int operator!=(SubString& x, const char* t) 
{
  return compare(x, t) != 0;
}

inline int operator>(SubString& x, const char* t)  
{
  return compare(x, t) > 0; 
}

inline int operator>=(SubString& x, const char* t) 
{
  return compare(x, t) >= 0; 
}

inline int operator<(SubString& x, const char* t)  
{
  return compare(x, t) < 0; 
}

inline int operator<=(SubString& x, const char* t) 
{
  return compare(x, t) <= 0; 
}

inline void String::make_unique()
{
  if (rep->ref != 1) setlength(rep->len);
}

//#endif

#endif
