// This may look like C code, but it is really -*- C++ -*-
/* 
Copyright (C) 1988 Free Software Foundation
    written by Dirk Grunwald (grunwald@cs.uiuc.edu)

This file is part of GNU CC.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the GNU CC General Public
License for full details.

Everyone is granted permission to copy, modify and redistribute
GNU CC, but only under the conditions described in the
GNU CC General Public License.   A copy of this license is
supposed to have been given to you along with GNU CC so you
can know your rights and responsibilities.  It should be in a
file named COPYING.  Among other things, the copyright notice
and this notice must be preserved on all copies.  
*/
#include <stream.h>
#include <SampleStatistic.h>
#include <math.h>

// error handling

void default_SampleStatistic_error_handler(char* msg)
{
  cerr << "Fatal SampleStatistic error. " << msg << "\n";
  exit(1);
}

one_arg_error_handler_t SampleStatistic_error_handler = default_SampleStatistic_error_handler;

one_arg_error_handler_t set_SampleStatistic_error_handler(one_arg_error_handler_t f)
{
  one_arg_error_handler_t old = SampleStatistic_error_handler;
  SampleStatistic_error_handler = f;
  return old;
}

void SampleStatistic::error(char* msg)
{
  (*SampleStatistic_error_handler)(msg);
}

typedef struct tableFormat {
    double n;
    double parameter[3];
};

//
//	The following is Students T-distribution with parameter
//	alpha 	= 0.95  (for 90%)
//		= 0.975 (for 95%)
//		= 0.995 (for 99%)
//
static const int tableEntries = 35;

static const tableFormat StudentT[ tableEntries ] = {
{       1, { 6.314, 12.706, 63.657}},
{       2, { 2.920,  4.303,  9.925}},
{       3, { 2.353,  3.182,  5.841}},
{       4, { 2.132,  2.776,  4.604}},
{       5, { 2.015,  2.571,  4.032}},
{       6, { 1.943,  2.447,  3.707}},
{       7, { 1.895,  2.365,  3.499}},
{       8, { 1.860,  2.306,  3.355}},
{       9, { 1.833,  2.262,  3.250}},
{      10, { 1.812,  2.228,  3.169}},
{      11, { 1.796,  2.201,  3.106}},
{      12, { 1.782,  2.179,  3.055}},
{      13, { 1.771,  2.160,  3.012}},
{      14, { 1.761,  2.145,  2.977}},
{      15, { 1.753,  2.131,  2.947}},
{      16, { 1.746,  2.210,  2.921}},
{      17, { 1.740,  2.110,  2.898}},
{      18, { 1.734,  2.101,  2.878}},
{      19, { 1.729,  2.093,  2.861}},
{      20, { 1.725,  2.086,  2.845}},
{      21, { 1.721,  2.080,  2.831}},
{      22, { 1.717,  2.074,  2.819}},
{      23, { 1.714,  2.069,  2.807}},
{      24, { 1.711,  2.064,  2.797}},
{      25, { 1.708,  2.060,  2.787}},
{      26, { 1.706,  2.056,  2.779}},
{      27, { 1.703,  2.052,  2.771}},
{      28, { 1.701,  2.048,  2.763}},
{      29, { 1.699,  2.045,  2.756}},
{      30, { 1.697,  2.042,  2.750}},
{      40, { 1.684,  2.021,  2.704}},
{      50, { 1.676,  2.009,  2.678}},
{      75, { 1.665,  1.992,  2.643}},
{     100, { 1.660,  1.984,  2.626}},
{HUGE, { 1.645,  1.960,  2.576}}
};

void
SampleStatistic::reset()
{
    n = 0; x = x2 = 0.0;
    maxValue = -HUGE;
    minValue = HUGE;
}

void
SampleStatistic::operator+=(double value)
{
    n += 1;
    x += value;
    x2 += (value * value);
    if ( minValue > value) minValue = value;
    if ( maxValue < value) maxValue = value;
}

double
SampleStatistic::mean()
{
    if ( n > 0) {
	return (x / n);
    }
    else {
	return ( 0.0 );
    }
}

double
SampleStatistic::var()
{
    if ( n > 1) {
	return(( x2 - ((x * x) /  n)) / ( n - 1));
    }
    else {
	return ( 0.0 );
    }
}

double
SampleStatistic::stdDev()
{
    if ( n <= 0 || this -> var() <= 0) {
	return(0);
    } else {
	return( (double) sqrt( var() ) );
    }
}

double
SampleStatistic::confidence(int interval)
{
    int index;

    switch( interval ) {
    case 90 : 
	index = 0;
	break;
    case 95:
	index = 1;
	break;
    case 99:
	index = 2;
	break;
    default :
	index = -1;
	break;
    }

    if (index < 0 || index > 2)
      error("confidence: invalid index");

    if (n < 1) {
	return(0);
    }

    double dn = double(n);
    for (int i = 0; dn > StudentT[i+1].n; i++);

    double meanValue = mean();
    double sd = stdDev();

    if (i < 0 || i >= tableEntries)
      error("confidence: invalid index");

    double zAlpha = StudentT[i].parameter[index];

    return( (zAlpha * sd) / sqrt( dn ) );
}


