#include "defs.h"

#include <fcntl.h>

#include <imcbdef.h>
#include <kferesdef.h>
#include "bfd.h"
#include "inferior.h"
#include "environ.h"
#include "command.h"
#include "symfile.h"
#include "gdbcmd.h"
#include "objfiles.h"
#include "evax-solib.h"

#define MAX_PATH_SIZE 512

#define IMCB_ADDR(so) ((so) -> imcb.imcb$l_starting_address)
#define IMCB_NEXT(so) ((so) -> imcb.imcb$l_flink)
#define IMCB_NAME(so) ((so) -> imcb.imcb$t_log_image_name)
/* Test for first imcb entry; first entry is the exec-file. */
#define IGNORE_FIRST_IMCB_ENTRY(x) ((x).imcb$l_blink == imglstptr)

static IMCB first_imcb;

struct so_list {
  struct so_list *next;	              /* next structure in linked list */
  IMCB imcb;                          /* copy of imcb from inferior */
  IMCB *imcb_ptr;                     /* addr in inferior imcb was read from */
  CORE_ADDR imcbend;                  /* upper addr bound of mapped object */
  KFERES *kferes;
  KFERES_SECTION **kferes_sections;
  char so_name[MAX_PATH_SIZE];        /* shared object lib name (FIXME) */
  char symbols_loaded;                /* flag: symbols read in yet? */
  char from_tty;                      /* flag: print msgs? */
  struct objfile *objfile;            /* objfile for loaded lib */
  struct section_table *sections;
  struct section_table *sections_end;
  struct section_table *textsection;
  bfd *abfd;
};

static IMCB *imglstptr = 0;

static struct so_list *so_list_head = 0; /* List of known shared objects */

static struct so_list *find_solib PARAMS ((struct so_list *));

static IMCB *first_imcb_member PARAMS ((void));

static void info_sharedimage_command PARAMS ((char *, int));

static void sharedimage_command PARAMS ((char *, int));

static void solib_map_sections PARAMS ((struct so_list *));

static void special_symbol_handling PARAMS ((struct so_list *));

static int symbol_add_stub PARAMS ((char *));


/*

LOCAL FUNCTION

	find_solib -- step through list of shared objects

SYNOPSIS

	struct so_list *find_solib (struct so_list *so_list_ptr)

DESCRIPTION

	This module contains the routine which finds the names of any
	loaded "images" in the current process. The argument in must be
	NULL on the first call, and then the returned value must be passed
	in on subsequent calls. This provides the capability to "step" down
	the list of loaded objects. On the last object, a NULL value is
	returned.

 */

static struct so_list *
find_solib (so_list_ptr)
     struct so_list *so_list_ptr;	/* Last imcb or NULL for first one */
{
  struct so_list *so_list_next = NULL;
  IMCB *imcbptr = NULL;
  IMCB imcb;
  struct so_list *new_so;

  if (so_list_ptr == NULL)
    {
      /* We are setting up for a new scan through the loaded images. */
      if ((so_list_next = so_list_head) == NULL)
	{
	  /* Imglstptr may be null if we just attached to a running program. */
	  if (!imglstptr)
	    evax_solib_create_inferior_hook ();
	  imcbptr = first_imcb_member ();
	}
    }
  else
    {
      /* We have been called before, and are in the process of walking
	 the shared library list.  Advance to the next shared object. */
      if ((imcbptr = IMCB_NEXT (so_list_ptr)) == imglstptr)
	{
	  /* We have hit the end of the list, so check to see if any were
	     added. */
	  read_memory ((CORE_ADDR) so_list_ptr->imcb_ptr,
		       (char *) &imcb,
		       sizeof (IMCB));
	  imcbptr = imcb.imcb$l_flink;

	  IMCB_NEXT (so_list_ptr) = imcbptr;
	  if (imcbptr == imglstptr)
	    imcbptr = NULL;
	}
      so_list_next = so_list_ptr -> next;
    }
  if ((so_list_next == NULL) && (imcbptr != NULL))
    {
      /* Get next link map structure from inferior image and build a local
	 abbreviated load_map structure */
      new_so = (struct so_list *) xmalloc (sizeof (struct so_list));
      memset ((char *) new_so, 0, sizeof (struct so_list));
      new_so -> imcb_ptr = imcbptr;
      /* Add the new node as the next node in the list, or as the root
	 node if this is the first one. */
      if (so_list_ptr != NULL)
	{
	  so_list_ptr -> next = new_so;
	}
      else
	{
	  so_list_head = new_so;
	}      
      so_list_next = new_so;
      read_memory ((CORE_ADDR) imcbptr, (char *) &(new_so->imcb),
		   sizeof (IMCB));

      if (!IGNORE_FIRST_IMCB_ENTRY (new_so->imcb)
	  && new_so->imcb.imcb$v_discontiguous
	  && new_so->imcb.imcb$l_kferes_ptr)
	{
	  int k;
	  int count;
	  KFERES *new_k;
	  KFERES_SECTION *new_ks;
	  KFERES *kferes_ptr = new_so ->imcb.imcb$l_kferes_ptr;
	  KFERES_SECTION *kferes_section_ptr;

	  new_k = (KFERES *) xmalloc (sizeof (KFERES));
	  memset ((char *) new_k, 0, sizeof (KFERES));
	  read_memory ((CORE_ADDR) kferes_ptr, (char *)new_k, sizeof (KFERES));

	  new_so->kferes = new_k;
	  count = new_so->kferes->kferes$l_count
	    + new_so->kferes->kferes$l_data_count;

	  new_so->kferes_sections
	    = (KFERES_SECTION **) xmalloc (count * sizeof (KFERES_SECTION *));
	  
	  for (k = 0, kferes_section_ptr = (KFERES_SECTION *)(kferes_ptr + 1);
	       k < count;
	       k++, kferes_section_ptr++)
	    {
	      new_ks = (KFERES_SECTION *) xmalloc (sizeof (KFERES_SECTION));
	      memset ((char *) new_ks, 0, sizeof (KFERES_SECTION));
	      read_memory ((CORE_ADDR) kferes_section_ptr,
			   (char *)new_ks,
			   sizeof (KFERES_SECTION));

	      new_so->kferes_sections[k] = new_ks;
	      
	    }
	}
      /* The first entry in the imcb is for the
	 inferior executable, so we must ignore it.  For some versions
	 it has no name. */
      if (!IGNORE_FIRST_IMCB_ENTRY (new_so->imcb))
	{
	  if (new_so->imcb.imcb$t_log_image_name [0])
	    {
	      strncpy (new_so->so_name,
		       &new_so->imcb.imcb$t_log_image_name [1],
		       new_so->imcb.imcb$t_log_image_name [0]);
	      new_so -> so_name[new_so->imcb.imcb$t_log_image_name [0]] = '\0';
	    }
	  else	  if (new_so->so_name[0] == 0)
	    return 0;

	    {
	      strncpy (new_so->so_name,
		       &new_so->imcb.imcb$t_image_name [1],
		       new_so->imcb.imcb$t_image_name [0]);
	      new_so -> so_name[new_so->imcb.imcb$t_image_name [0]] = '\0';
	    }

#if 0
	  printf ("%s symbol vector: %llx - %llx\n",
		  new_so->so_name,
		  new_so->imcb.imcb$ps_symbol_vector_address,
		  new_so->imcb.imcb$r_fill_12.imcb$q_symbol_vector_address 
		  + new_so->imcb.imcb$l_symbol_vector_size - 1);
#endif

	  solib_map_sections (new_so);
	}      
    }
  return (so_list_next);
}

/*

LOCAL FUNCTION

	first_imcb_member -- locate first member in dynamic linker's map

SYNOPSIS

	static IMCB *first_imcb_member (void)

DESCRIPTION

	Read in a copy of the first member in the inferior's dynamic
	imcb from the inferior's dynamic linker structures, and return
	a pointer to the copy in our address space.
*/

static IMCB *
first_imcb_member ()
{
  read_memory
    ((CORE_ADDR) imglstptr, (char *)&first_imcb, sizeof (IMCB));

  return (first_imcb.imcb$l_flink);
}

/*

LOCAL FUNCTION

	info_sharedimage_command -- code for "info sharedimage"

SYNOPSIS

	static void info_sharedimage_command ()

DESCRIPTION

	Walk through the shareable image list and print information
	about each attached image.
*/

static void
info_sharedimage_command (ignore, from_tty)
     char *ignore;
     int from_tty;
{
  register struct so_list *so = NULL;
  int header_done = 0;
  
  if (!target_has_execution)
    {
      printf_unfiltered ("No execution.\n");
      return;
    }

  while ((so = find_solib (so)) != NULL)
    {
      if (so -> so_name[0])
	{
	  if (!header_done)
	    {
	      printf_unfiltered
		("   %-20s%-20s%-12s\n", "From", "To", "Symbols");
	      header_done++;
	    }
	  printf_unfiltered ("%s\n",  TO_HOST_FILE_SPEC (so -> so_name));
	  printf_unfiltered ("   %-20s",
		  local_hex_string_custom ((unsigned __int64) IMCB_ADDR (so),
					   "016ll"));
	  printf_unfiltered ("%-20s",
		  local_hex_string_custom ((unsigned __int64) so -> imcbend,
					   "016ll"));
	  printf_unfiltered
	    ("%-12s\n", so -> symbols_loaded ? "Read" : "Not read");
	}

      if (so->imcb.imcb$v_discontiguous)
	{
	  int k;
	  int count
	    = so->kferes->kferes$l_count + so->kferes->kferes$l_data_count;
	  
	  for (k = 0; k < count; k++)
	    {
	      if (k < so->kferes->kferes$l_count)
		printf_unfiltered ("   CODE: ");
	      else
		printf_unfiltered ("   DATA: ");
	      
	      printf_unfiltered
		("%-20s",
		 local_hex_string_custom
		 ((unsigned __int64) so->kferes_sections[k]->kferes$l_va,
		  "016ll"));
	      printf_unfiltered
		("%-20s\n",
		 local_hex_string_custom
		 ((unsigned __int64) so->kferes_sections[k]->kferes$l_va
		  + so->kferes_sections[k]->kferes$l_length - 1,
		  "016ll"));
	    }
	}
      printf_unfiltered ("\n");
    }
  if (so_list_head == NULL)
    {
      printf_unfiltered ("No shareable images loaded at this time.\n");	
    }
}

struct section_offsets *
evax_solib_offsets (objfile, addr, num_offsets)
     struct objfile *objfile;
     CORE_ADDR addr;
     int num_offsets;
{
  register struct so_list *so = NULL;
  int i;

  struct section_offsets *section_offsets = (struct section_offsets *)
    obstack_alloc (&objfile -> psymbol_obstack,
		   sizeof (struct section_offsets) +
		   sizeof (section_offsets->offsets) * (num_offsets-1));

  for (i = 0; i < num_offsets; i++)
    {
      ANOFFSET (section_offsets, i) = 0;
    }

  while ((so = find_solib (so)) != NULL)
    {
      if (so -> so_name[0]
	  && ((unsigned __int64) IMCB_ADDR (so) == (unsigned __int64) addr))
	{
	  if (so->imcb.imcb$v_discontiguous)
	    {
	      int k;
	      int count
		= so->kferes->kferes$l_count
		  + so->kferes->kferes$l_data_count;
	  
	      for (k = 0; k < count; k++)
		{
		  ANOFFSET (section_offsets, k)
		    = (unsigned __int64) so->kferes_sections[k]->kferes$l_va;
		}

	      if (count == 1)
		{
		  ANOFFSET (section_offsets, 1)
		    = (unsigned __int64) IMCB_ADDR (so);
		}
	    }
	  else
	    {
	      for (i = 0; i < num_offsets; i++)
		{
		  ANOFFSET (section_offsets, i)
		    = (unsigned __int64) IMCB_ADDR (so);
		}
	    }
	  return (section_offsets);
	}
    }
}

/*

LOCAL FUNCTION

	sharedimage_command -- handle command to explicitly add library

SYNOPSIS

	static void sharedimage_command (char *args, int from_tty)

DESCRIPTION

*/

static void
sharedimage_command (args, from_tty)
char *args;
int from_tty;
{
  dont_repeat ();
  evax_solib_add (args, from_tty, (struct target_ops *) 0);
}

/*

LOCAL FUNCTION

	solib_map_sections -- open bfd and build sections for shared lib

SYNOPSIS

	static void solib_map_sections (struct so_list *so)

DESCRIPTION

	Given a pointer to one of the shared objects in our list
	of mapped objects, use the recorded name to open a bfd
	descriptor for the object, build a section table, and then
	relocate all the section addresses by the base address at
	which the shared object was mapped.

FIXMES

	In most (all?) cases the shared object file name recorded in the
	dynamic linkage tables will be a fully qualified pathname.  For
	cases where it isn't, do we really mimic the systems search
	mechanism correctly in the below code (particularly the tilde
	expansion stuff?).
 */

static void
solib_map_sections (so)
     struct so_list *so;
{
  char *filename, *full_filename;
  char *scratch_pathname;
  int scratch_chan;
  struct cleanup *old_chain;
  bfd *abfd;
  
  filename = get_in_environ (inferior_environ, so -> so_name);

  if (filename == 0)
    filename = so -> so_name;
  
  full_filename = (char *)xmalloc (strlen (filename) + strlen (".EXE") + 1);
  strcpy (full_filename, filename);
  strcat (full_filename, ".EXE");

  old_chain = make_cleanup (free, full_filename);


  scratch_chan = openp ("/SYS$SHARE", 0, full_filename,
			O_RDONLY, 0, &scratch_pathname);

  if (scratch_chan < 0)
    {
      scratch_chan = openp ("/SYS$LOADABLE_IMAGES", 0, full_filename,
			    O_RDONLY, 0, &scratch_pathname);
    }

  if (scratch_chan < 0)
    {
      scratch_chan = openp ("/SYS$MESSAGE", 0, full_filename,
			    O_RDONLY, 0, &scratch_pathname);
    }

  if (scratch_chan < 0)
    {
      scratch_chan = openp ("/GNU/LIB", 0, full_filename,
			    O_RDONLY, 0, &scratch_pathname);
    }

  if (scratch_chan < 0 && getenv (filename))
    {
      scratch_chan = openp (0, 1, TO_NORMAL_FILE_SPEC (getenv (filename)),
			    O_RDONLY, 0, &scratch_pathname);
    }

  if (scratch_chan < 0)
    {
      scratch_chan = openp
	(TO_NORMAL_DIR_SPEC (get_in_environ (inferior_environ, "PATH")), 
	 1, full_filename, O_RDONLY, 0, &scratch_pathname);
    }

  if (scratch_chan < 0)
    {
      perror_with_name (full_filename);
    }

  abfd = bfd_fdopenr (scratch_pathname, 0, scratch_chan);
  if (!abfd)
    {
      close (scratch_chan);
      error ("Could not open `%s' as a shareable image file: %s",
	     TO_HOST_FILE_SPEC (scratch_pathname),
	     bfd_errmsg (bfd_get_error ()));
    }
/*  close (scratch_chan); */
  /* Leave bfd open, core_xfer_memory and "info files" need it.  */
  so -> abfd = abfd;
  abfd -> cacheable = true;

  /* copy full path name into so_name, so that later symbol_file_add can find
     it */
  if (strlen (scratch_pathname) >= MAX_PATH_SIZE)
    error ("Full path name length of shared library exceeds MAX_PATH_SIZE in so_list structure.");
  strcpy (so->so_name, scratch_pathname);

#if 0
  printf ("Mapping: %s\n", so->so_name);
#endif
  if (!bfd_check_format (abfd, bfd_object))
    {
      error ("\"%s\": not in executable format: %s.",
	     TO_HOST_FILE_SPEC (scratch_pathname),
	     bfd_errmsg (bfd_get_error ()));
    }

  if (build_section_table (abfd, &so -> sections, &so -> sections_end))
    {
      error ("Can't find the file sections in `%s': %s", 
	     bfd_get_filename (abfd), bfd_errmsg (bfd_get_error ()));
    }

  /* Relocate the section binding addresses as recorded in the shared
     object's file by the base address to which the object was actually
     mapped. */
  if (so->imcb.imcb$v_discontiguous)
    {
      /* Sections are discontiguous, usually this means the text section
	 is installed shareable and is in high memory and the rest of
	 the sections are elsewhere */

      int k;
      struct section_table *p;

      p = so->sections;
      /* Do the code sections (hopefully there's only one) */
      for (k=0; k<so->kferes->kferes$l_count; k++)
	{
	  KFERES_SECTION *ks;

	  ks = so->kferes_sections [k];
	  while (p < so->sections_end
		 && !(p->the_bfd_section->flags & SEC_CODE))
	    p++;

	  if (p < so->sections_end)
	    {
#if 0
	      printf ("text section %s (osize: %x, nsize: %x) moved to %llx\n",
		      p->the_bfd_section->name,
		      p->endaddr - p->addr,
		      ks->kferes$l_length,
		      ks->kferes$l_va);
#endif
	      p->addr = (CORE_ADDR) ks->kferes$l_va;
	      p->endaddr = p->addr + ks->kferes$l_length - 1;
	      so->textsection = p;

	      p++;
	    }
	  else
	    error ("Couldn't find text section in \"%s\".",
		   TO_HOST_FILE_SPEC (scratch_pathname));
	}

      /* Do the data sections */
      if (so->kferes->kferes$l_data_count > 0)
	{
	  /* Sometimes that data sections are listed in separate
	     KFERES_SECTION structures, but all the examples seen
	     have them actually at contiguous addresses. */

	  p = so->sections;
	  for (k = so->kferes->kferes$l_count;
	       k < (so->kferes->kferes$l_count
		    + so->kferes->kferes$l_data_count);
	       k++)
	    {
	      KFERES_SECTION *ks;
	      
	      ks = so->kferes_sections [k];
	      while (p < so->sections_end
		     && (p->the_bfd_section->flags & SEC_CODE))
		p++;
	      
	      if (p < so->sections_end)
		{
#if 0
		  printf ("data section %s (osize: %x, nsize: %x) moved to %llx\n",
			  p->the_bfd_section->name,
			  p->endaddr - p->addr,
			  ks->kferes$l_length,
			  ks->kferes$l_va);
#endif
		  p->addr = (CORE_ADDR) ks->kferes$l_va;
		  p->endaddr = p->addr + ks->kferes$l_length - 1;
		  so -> imcbend
		    = (CORE_ADDR) max (p -> endaddr, so -> imcbend);

		  p++;
		}
	      else
		error ("Couldn't find all data sections in \"%s\".",
		       TO_HOST_FILE_SPEC (scratch_pathname));
	    }
	}
      else
	{
	  /* And sometimes the data sections are lumped together */

	  for (p = so -> sections; p < so -> sections_end; p++)
	    {
	      if (!(p->the_bfd_section->flags & SEC_CODE))
		{
		  p -> addr += (CORE_ADDR) IMCB_ADDR (so);
		  p -> endaddr += (CORE_ADDR) IMCB_ADDR (so);

		  so -> imcbend
		    = (CORE_ADDR) max (p -> endaddr, so -> imcbend);

#if 0
		  printf ("data section %s moved to %llx\n",
			  p->the_bfd_section->name,
			  p->addr);
#endif
		}
	    }
	}
    }
  else
    {
      /* All sections are contiguous */

      struct section_table *p;

      for (p = so -> sections; p < so -> sections_end; p++)
	{
	  p -> addr += (CORE_ADDR) IMCB_ADDR (so);
	  p -> endaddr += (CORE_ADDR) IMCB_ADDR (so);
#if 0
	  printf ("section %s moved to %llx\n",
		  p->the_bfd_section->name,
		  p->addr);
#endif
	  so -> imcbend = (CORE_ADDR) max (p -> endaddr, so -> imcbend);
	  if (p->the_bfd_section->flags & SEC_CODE)
	    so -> textsection = p;
	}
    }

  do_cleanups (old_chain);
}

/*

LOCAL FUNCTION

	special_symbol_handling -- additional shared library symbol handling

SYNOPSIS

	void special_symbol_handling (struct so_list *so)

DESCRIPTION

	Once the symbols from a shared object have been loaded in the usual
	way, we are called to do any system specific symbol handling that 
	is needed.

*/

static void
special_symbol_handling (so)
     struct so_list *so;
{
}

/* A small stub to get us past the arg-passing pinhole of catch_errors.  */

static int
symbol_add_stub (arg)
     char *arg;
{
  register struct so_list *so = (struct so_list *) arg;	/* catch_errs bogon */
  
  so -> objfile =
    symbol_file_add
      (so->so_name, so->from_tty, (CORE_ADDR) IMCB_ADDR (so), 0, 0, 0);
  return (1);
}

extern void
print_section_info ();

int
evax_files_info ()
{
  struct so_list *so;
  bfd *abfd;

  for (so = so_list_head; so; so = so->next)
    {
      if (so->objfile)
	{
	  abfd = so->objfile->obfd;
	}
      else
	{
	  abfd = so->abfd;
	}
      if (abfd)
	{
	  struct target_ops t;

	  build_section_table (abfd, &t.to_sections, &t.to_sections_end);
	  printf_filtered ("Shareable image file:\n");
	  print_section_info (&t, abfd);

	  free (t.to_sections);
	}
    }

  return 0;
}

/*

GLOBAL FUNCTION

	evax_solib_add -- add a shared library file to the symtab
	                  and section list

SYNOPSIS

	void evax_solib_add (char *arg_string, int from_tty,
			struct target_ops *target)

DESCRIPTION

*/

void
evax_solib_add (arg_string, from_tty, target)
     char *arg_string;
     int from_tty;
     struct target_ops *target;
{	
  register struct so_list *so = NULL;   	/* link map state variable */

  /* Last shared library that we read.  */
  struct so_list *so_last = NULL;

  char *re_err;
  int count;
  int old;
  
  if ((re_err = re_comp (arg_string ? arg_string : ".")) != NULL)
    {
      error ("Invalid regexp: %s", re_err);
    }

  /* Add the shared library sections to the section table of the
     specified target, if any.  */
  if (target)
    {
      /* Count how many new section_table entries there are.  */
      so = NULL;
      count = 0;
      while ((so = find_solib (so)) != NULL)
	{
	  if (so -> so_name[0])
	    {
	      count += so -> sections_end - so -> sections;
	    }
	}
      
      if (count)
	{
	  /* Reallocate the target's section table including the new size.  */
	  if (target -> to_sections)
	    {
	      old = target -> to_sections_end - target -> to_sections;
	      target -> to_sections = (struct section_table *)
		xrealloc ((char *)target -> to_sections,
			 (sizeof (struct section_table)) * (count + old));
	    }
	  else
	    {
	      old = 0;
	      target -> to_sections = (struct section_table *)
		xmalloc ((sizeof (struct section_table)) * count);
	    }
	  target -> to_sections_end = target -> to_sections + (count + old);
	  
	  /* Add these section table entries to the target's table.  */
	  while ((so = find_solib (so)) != NULL)
	    {
	      if (so -> so_name[0])
		{
		  count = so -> sections_end - so -> sections;
		  memcpy ((char *) (target -> to_sections + old),
			  so -> sections, 
			  (sizeof (struct section_table)) * count);
		  old += count;
		}
	    }
	}
    }
  
  /* Now add the symbol files.  */
  while ((so = find_solib (so)) != NULL)
    {
      if (so -> so_name[0] && re_exec (so -> so_name))
	{
	  so -> from_tty = from_tty;
	  if (so -> symbols_loaded)
	    {
	      if (from_tty)
		{
		  printf_unfiltered ("Symbols already loaded for %s\n",
				     TO_HOST_FILE_SPEC (so -> so_name));
		}
	    }
	  else if (catch_errors
		   (symbol_add_stub, (char *) so,
		    "Error while reading shared library symbols:\n",
		    RETURN_MASK_ALL))
	    {
	      so_last = so;
	      so -> symbols_loaded = 1;
	    }
	}
    }

  /* Getting new symbols may change our opinion about what is
     frameless.  */
  if (so_last)
    reinit_frame_cache ();

  if (so_last)
    special_symbol_handling (so_last);
}

void
evax_solib_create_inferior_hook ()
{
  IMCB imcb;
  KFERES kferes;
  KFERES_SECTION kferes_section;
  char buff1 [256];
  char buff2 [256];
  struct so_list *so;

  /* Rereading the shared images causes a memory leak wrt symbol table info
     (objfile->obfd).  This needs to be rethought, maybe only reading in
     those images perturbed by run-time addition of message shared images.
     9/15/98 */

  if (so_list_head)
    return;

  /* Reinitialize imglstptr and reread shared images on restarts, otherwise
     the pointers get messed up due to the run-time addition of
     message shared images. */

  for (so = so_list_head; so; so = so->next)
    {
      if (so->abfd)
	bfd_close (so->abfd);
    }

  imglstptr = (IMCB *)vmschild_get_imglstptr ();
  so_list_head = 0;

#if 0
  target_read_memory ((CORE_ADDR) imglstptr, (char *)&imcb, sizeof (IMCB));
  for (; imcb.imcb$l_flink != imglstptr;)
    {
      target_read_memory
	((CORE_ADDR) imcb.imcb$l_flink, (char *)&imcb, sizeof (IMCB));
      strncpy (buff1,
	       &imcb.imcb$t_log_image_name [1],
	       imcb.imcb$t_log_image_name [0]);
      buff1 [imcb.imcb$t_log_image_name[0]] = 0;
      strncpy (buff2,
	       &imcb.imcb$t_image_name [1],
	       imcb.imcb$t_image_name [0]);
      buff2 [imcb.imcb$t_image_name[0]] = 0;
      
      printf ("%s %s: %llx to %llx\n",
	      buff1,
	      buff2,
	      imcb.imcb$l_starting_address,
	      imcb.imcb$l_end_address);
      
      if (imcb.imcb$v_discontiguous)
	{
	  int k;
	  int count;
	  KFERES_SECTION *kferes_section_ptr;
	  KFERES *kferes_ptr = imcb.imcb$l_kferes_ptr;
	  
	  target_read_memory ((CORE_ADDR) kferes_ptr,
			      (char *)&kferes,
			      sizeof (KFERES));
	  
	  count = kferes.kferes$l_count + kferes.kferes$l_data_count;
	  
	  for (k = 0, kferes_section_ptr = (KFERES_SECTION *)(kferes_ptr + 1);
	       k < count;
	       k++, kferes_section_ptr++)
	    {
	      if (k < kferes.kferes$l_count)
		printf ("   CODE: ");
	      else
		printf ("   DATA: ");
	      
	      target_read_memory ((CORE_ADDR) kferes_section_ptr,
				  (char *)&kferes_section,
				  sizeof (KFERES_SECTION));
	      
	      printf ("%llx to %llx\n",
		      kferes_section.kferes$l_va,
		      (unsigned long)kferes_section.kferes$l_va
		      + kferes_section.kferes$l_length - 1);
	    }
	}
    }
#endif

  if (auto_solib_add)
    evax_solib_add ((char *) 0, 0, (struct target_ops *) 0);
}

char *
evax_solib_address (address)
     CORE_ADDR address;
{
  register struct so_list *so = 0;   	/* link map state variable */
  
  while ((so = find_solib (so)) != NULL)
    {
      /* The main image name was ignored to make this field null, since
	 we probably don't want to have those functions show up in a
	 shared library.  Can this ever really happen? */
      if (so -> so_name[0])
	{
#if 0
	  printf ("checking if %x in %s: %x %x\n",
		  address,
		  so->so_name,
		  IMCB_ADDR (so),
		  so->imcbend);
#endif
	  if ((so->textsection)
	      && ((address >= so->textsection->addr) &&
		  (address <= so->textsection->endaddr)))
	    return (so->so_name);
	}
    }
  return (0);
}

void
_initialize_solib()
{
  add_com ("sharedimage", class_files, sharedimage_command,
	   "Load shareable image symbols for files matching REGEXP.");
  add_info ("sharedimage", info_sharedimage_command, 
	    "Status of loaded shareable images.");

  add_show_from_set
    (add_set_cmd ("auto-solib-add", class_support, var_zinteger,
		  (char *) &auto_solib_add,
		  "Set autoloading of shared library symbols.\n\
If nonzero, symbols from all shareable images will be loaded\n\
automatically when the inferior begins execution or when the loader\n\
informs gdb that a new image has been loaded.  Otherwise, symbols\n\
must be loaded manually, using `sharedimage'.",
		  &setlist),
     &showlist);

}
