
/*--------------------------------------------------------------------*/
/*--- PowerPC-specific functions for manipulating thread state.
                                                       ppc/state.c ---*/
/*--------------------------------------------------------------------*/

/*
   This file is part of Valgrind, an extensible
   emulator for monitoring program execution on Unixes.

   Copyright (C) 2000-2004 Julian Seward 
      jseward@acm.org
   Copyright (C) 2004 Paul Mackerras
      paulus@samba.org

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.

   The GNU General Public License is contained in the file COPYING.
*/

#include "core.h"
#include <string.h>
#include <errno.h>
#include <sys/ptrace.h>
#include <asm/ptrace.h>

#if 0
ULong VG_(total_synth_instrs);
ULong VG_(total_real_instrs);
#endif

void VG_(dump_thread_regs)(ThreadState *tst)
{
   int i;

   if (VG_(needs).shadow_regs) {
      for (i = 0; i < 32; ++i) {
	 if ((i & 3) == 0)
	    VG_(printf)("r%02d:", i);
	 VG_(printf)(" %08x:%08x", tst->arch.m_gpr[i], tst->arch.sh_gpr[i]);
	 if ((i & 3) == 3)
	    VG_(printf)("\n");
      }
      VG_(printf)(" lr: %08x:%08x  ctr: %08x:%08x\n",
		  tst->arch.m_lr, tst->arch.sh_lr,
		  tst->arch.m_ctr, tst->arch.sh_ctr);
      VG_(printf)(" cr: %08x:%08x  xer: %08x:%08x\n",
		  tst->arch.m_cr, tst->arch.sh_cr,
		  tst->arch.m_xer, tst->arch.sh_xer);
   } else {
      for (i = 0; i < 32; ++i) {
	 if ((i & 7) == 0)
	    VG_(printf)("r%02d:", i);
	 VG_(printf)(" %08x", tst->arch.m_gpr[i]);
	 if ((i & 7) == 7)
	    VG_(printf)("\n");
      }
      VG_(printf)(" lr: %08x  ctr: %08x  cr: %08x  xer: %08x\n",
		  tst->arch.m_lr, tst->arch.m_ctr,
		  tst->arch.m_cr, tst->arch.m_xer);
   }
   VG_(printf)("NIP: %08x\n", tst->arch.m_eip);
}

Bool VGA_(setup_pointercheck)(void)
{
   return False;
}

void VGA_(clear_thread)( arch_thread_t *arch )
{
}

void VGA_(init_thread)( ThreadId tid, Addr init_ip, Addr init_sp )
{
   ThreadState *tst = VG_(get_ThreadState)(tid);
   arch_thread_t *regs = &tst->arch;

   VG_(memset)(&regs->m_gpr, 0, 32 * sizeof(UInt));
   VG_(memset)(&regs->sh_gpr, 0, 32 * sizeof(UInt));
   regs->m_ctr = regs->sh_ctr = 0;
   regs->m_lr  = regs->sh_lr  = 0;
   regs->m_xer = regs->sh_xer = 0;
   regs->m_cr  = regs->sh_cr  = 0;

   regs->m_eip = init_ip;
   regs->m_gpr[1] = init_sp;

   /* Zero out the main thread's FP and altivec state. */
   VG_(memset)(&regs->m_fpr[0], 0, 33 * sizeof(double));

   if (VG_(hardware_capabilities) & VKI_HWCAP_ALTIVEC) {
      VG_(memset)(&regs->m_vr[0], 0, 33 * 16);
      regs->vr_live = 0;
   }
}

void VGA_(cleanup_thread) ( arch_thread_t *regs )
{
}

void VGA_(setup_child) ( arch_thread_t *regs, arch_thread_t *parent_regs )
{
   *regs = *parent_regs;
}

void VGA_(mark_from_registers)(ThreadId tid, void (*marker)(Addr))
{
   ThreadState *tst = VG_(get_ThreadState)(tid);
   arch_thread_t *regs = &tst->arch;
   int i;

   for (i = 0; i < 32; ++i)
      (*marker)(regs->m_gpr[i]);
   (*marker)(regs->m_ctr);	/* do we need to do this? */
   (*marker)(regs->m_lr);
   /* xer and cr can't contain addresses */
}

Int VGA_(ptrace_setregs_from_tst)(Int pid, arch_thread_t *tst)
{
   int r;

   for (r = 0; r < 32; ++r)
      if (ptrace(PTRACE_POKEUSER, pid, (PT_R0 + r) * 4, tst->m_gpr[r])) {
	 VG_(printf)("error setting gpr%d: %s\n", r, strerror(errno));
	 return -1;
      }
   if (ptrace(PTRACE_POKEUSER, pid, PT_NIP * 4, tst->m_eip) ||
       ptrace(PTRACE_POKEUSER, pid, PT_CCR * 4, tst->m_cr) ||
       ptrace(PTRACE_POKEUSER, pid, PT_LNK * 4, tst->m_lr) ||
       ptrace(PTRACE_POKEUSER, pid, PT_CTR * 4, tst->m_ctr) ||
       ptrace(PTRACE_POKEUSER, pid, PT_XER * 4, tst->m_xer))
      return -1;

   return 0;
}


UInt *VGA_(reg_addr_from_tst)(Int regno, arch_thread_t *tst)
{
   UInt *ret = 0;

   if (0 <= regno && regno <= 31)
      ret = &tst->m_gpr[regno];
   else
      switch (regno) {
      case R_CR:	ret = &tst->m_cr;	break;
      case R_LR:	ret = &tst->m_lr;	break;
      case R_CTR:	ret = &tst->m_ctr;	break;
      case R_XER:	ret = &tst->m_xer;	break;
      }

   return ret;
}

#define SET_PTHREQ_ARG(zztid, zzval)				\
   do { VG_(threads)[zztid].arch.m_gpr[3] = (zzval);		\
        VG_TRACK( post_reg_write_pthread_return, zztid, 3);	\
   } while (0)

#define SET_PTHREQ_RETADDR(zztid, zzval)			\
   do { VG_(threads)[zztid].arch.m_lr = (zzval);		\
        VG_TRACK( post_reg_write_pthread_return, zztid, R_LR);	\
   } while (0)

void VGA_(set_arg_and_bogus_ret)(ThreadId tid, UWord arg, Addr ret)
{
   /* Set R3 to the argument and LR to the bogus return address. */
   SET_PTHREQ_ARG(tid, arg);
   SET_PTHREQ_RETADDR(tid, ret);
}

void VGA_(thread_initial_stack)(ThreadId tid, UWord arg, Addr ret)
{
   UInt	    esp;

   esp = ARCH_STACK_PTR(VG_(threads)[tid].arch);
   /* make sure top of stack is 16-byte aligned */
   esp &= ~0xf;
   /* make an initial stack frame and initialize the link to 0 */
   esp -= 16;
   SET_PTHREQ_ESP(tid, esp);

   VG_TRACK(new_mem_stack, esp, 16);
   VG_TRACK(pre_mem_write, Vg_CorePThread, tid, "new thread: stack", esp, 4);
   *(UInt *)esp = 0;
   VG_TRACK(post_mem_write, esp, 4);

   /* Set r3 to the argument and lr to the bogus return address */
   SET_PTHREQ_ARG(tid, arg);
   SET_PTHREQ_RETADDR(tid, ret);
}
