/* Map and map widget code for the tcl/tk interface to Xconq.
   Copyright (C) 1998 Stanley T. Shebs.

Xconq is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.  See the file COPYING.  */

#include "conq.h"
#include "kpublic.h"
#include "tkconq.h"

extern Pixmap bordpics[NUMPOWERS];
extern Pixmap connpics[NUMPOWERS];
extern Pixmap windpics[5][NUMDIRS];
extern Pixmap antpic;

static void help_unit_type(Side *side, Map *map);
static void help_terrain_type(Side *side, Map *map);

create_map()
{
    int u;
    Map *map;
    Side *side2;

    DGprintf("Creating map\n");
    map = (Map *) xmalloc(sizeof(Map));

    map->mode = move_mode;
    map->autoselect = TRUE;
    map->move_on_click = TRUE;

    map->terrain_style = 1;

    map->prefixarg = -1;
    map->inptype = NONUTYPE;

    /* Newest map goes on the front of the list. */
    map->next = dside->ui->maps;
    dside->ui->maps = map;

    dside->ui->curmap = map;

    eval_tcl_cmd("create_map_window");

    for_all_sides(side2) {
	update_side_display(dside, side2, TRUE);
    }
    for_all_unit_types(u) {
	update_unit_type_list(dside, u);
    }
    update_turn_display(dside, TRUE);

    set_tool_cursor(dside, map);
    eval_tcl_cmd("update_mode move");
}

/* planned reduction in number of mag powers */
#undef NUMPOWERS
#define NUMPOWERS 7

zoom_in_out(Side *side, Map *map, int incr)
{
    int newpower;
    VP *vp = widget_vp(map);
    extern int lastrawx, lastrawy;
    char tclbuf[100];

    newpower = vp->power + incr;
    if (newpower < 0)
      newpower = 0;
    if (newpower > NUMPOWERS - 1)
      newpower = NUMPOWERS - 1;
    if (newpower != vp->power) {
	set_view_power(vp, newpower);
	center_on_focus(vp);
	redraw_map(map);
	update_mouseover(map, lastrawx, lastrawy);
	sprintf(tclbuf, "update_zoom %d %d",
		(newpower < (NUMPOWERS - 1)), (newpower > 0)); 
	Tcl_Eval(interp, tclbuf);
    }
}

/* Prompt for a type of a unit from player, maybe only allowing some types
   to be accepted.  Also allow specification of no unit type.  We do this
   by scanning the vector, building a string of chars and a vector of
   unit types, so as to be able to map back when done. */

int
ask_unit_type(side, map, prompt, possibles, handler)
Side *side;
Map *map;
char *prompt;
int *possibles;
void (*handler)();
{
    int u, numtypes = 0;
    char tclbuf[500];

    for_all_unit_types(u) {
	if (possibles == NULL || possibles[u]) {
	    map->uvec[numtypes] = u;
	    map->ustr[numtypes] = utype_name_n(u, 1)[0];
	    ++numtypes;
	    enable_in_unit_type_list(side, map, u, 1);
	} else {
	    enable_in_unit_type_list(side, map, u, -1);
	}
    }
    map->ustr[numtypes] = '\0';
    if (numtypes > 1) {
	sprintf(tclbuf, "ask_unit_type_mode {%s [%s]}",
		prompt, map->ustr);
	Tcl_Eval(interp, tclbuf);
	map->modalhandler = handler;
    }
    return numtypes;
}

/* Do something with the char or unit type that the player entered. */

int
grok_unit_type(side, map, typep)
Side *side;
Map *map;
int *typep;
{
    int i, u;
    char tclbuf[500];

    *typep = NONUTYPE;
    if (map->inptype != NONUTYPE) {
	*typep = map->inptype;
	/* Reset so doesn't affect subsequent unit type queries. */
	map->inptype = NONUTYPE;
    } else if (map->inpch != '\0') {
	if (map->inpch == '?') {
	    help_unit_type(side, map);
	    return FALSE;
	}
	i = iindex(map->inpch, map->ustr);
	if (i >= 0) {
	    *typep = map->uvec[i];
	} else {
	    notify(side, "Must type a unit type char from the list, or <esc>");
	    return FALSE;
	}
    } else {
	notify(side, "weird");
	return FALSE;
    }
    eval_tcl_cmd("ask_unit_type_done");
    /* Reset the appearance of the unit type list. */
    for_all_unit_types(u) {
	enable_in_unit_type_list(side, map, u, 0);
    }
    /* Make the unit type string be empty. */
    map->ustr[0] = '\0';
    return TRUE;
}

int
cancel_unit_type(side, map)
Side *side;
Map *map;
{
    int u;

    /* Reset the appearance of the unit type list. */
    for_all_unit_types(u) {
	enable_in_unit_type_list(side, map, u, 0);
    }
}

static void
help_unit_type(side, map)
Side *side;
Map *map;
{
    int i;
    char helpbuf[BUFSIZE];

    helpbuf[0] = '\0';
    for (i = 0; map->ustr[i] != '\0'; ++i) {
	/* Put out several types on each line. */
	if (i % 4 == 0) {
	    if (i > 0) {
		notify(side, "%s", helpbuf);
	    }
	    /* Indent each line a bit (also avoids notify's
	       auto-capitalization). */
	    strcpy(helpbuf, "  ");
	}
	tprintf(helpbuf, "%c %s, ", map->ustr[i], u_type_name(map->uvec[i]));
    }
    /* Add an extra helpful comment, then dump any leftovers. */
    tprintf(helpbuf, "? for this help info"); 
    notify(side, "%s", helpbuf);
}

enable_in_unit_type_list(side, map, u, flag)
Side *side;
Map *map;
int u, flag;
{
    char tclbuf[500];

    sprintf(tclbuf, "enable_unitlist %d %d", u, flag);
    Tcl_Eval(interp, tclbuf);
}

int
ask_terrain_type(side, map, prompt, possibles, handler)
Side *side;
Map *map;
char *prompt;
int *possibles;
void (*handler)(Side *side, Map *map, int cancelled);
{
    int numtypes = 0, t;

    for_all_terrain_types(t) {
	if (possibles == NULL || possibles[t]) {
	    map->tvec[numtypes] = t;
	    map->tstr[numtypes] =
	      (!empty_string(t_char(t)) ? t_char(t)[0] : (t - 'a'));
	    ++numtypes;
	}
    }
    map->tstr[numtypes] = '\0';
    if (numtypes > 1) {
	char tclbuf[500];

	sprintf(tclbuf, "ask_terrain_type_mode {%s [%s]}", prompt, map->tstr);
	Tcl_Eval(interp, tclbuf);
	map->modalhandler = handler;
    }
    return numtypes;
}

/* Do something with the char or terrain type that the player entered. */

int
grok_terrain_type(side, map, typep)
Side *side;
Map *map;
int *typep;
{
    int i;

    *typep = NONTTYPE;
    if (map->inpch == '?') {
	help_terrain_type(dside, map);
	return FALSE;
    }
    i = iindex(map->inpch, map->tstr);
    if (i >= 0) {
	*typep = map->tvec[i];
	eval_tcl_cmd("ask_terrain_type_done");
	return TRUE;
    } else {
	notify(dside, "Must type a terrain type char or <esc>");
	return FALSE;
    }
}

static void
help_terrain_type(side, map)
Side *side;
Map *map;
{
    int i;
    char helpbuf[BUFSIZE];

    for (i = 0; map->tstr[i] != '\0'; ++i) {
	/* Put out several types on each line. */
	if (i % 4 == 0) {
	    if (i > 0) {
		notify(side, "%s", helpbuf);
	    }
	    /* Indent each line a bit (also avoids confusion due to
	       notify's capitalization). */
	    strcpy(helpbuf, "  ");
	}
	tprintf(helpbuf, "%c %s, ", map->tstr[i], t_type_name(map->tvec[i]));
    }
    /* Add an extra helpful comment, then dump any leftovers. */
    tprintf(helpbuf, "? for this help info"); 
    notify(side, "%s", helpbuf);
}

/* User is asked to pick a position on map.  This will iterate until the
   space bar designates the final position. */

/* (should change the cursor temporarily) */

void
ask_position(side, map, prompt, handler)
Side *side;
Map *map;
char *prompt;
void (*handler)(Side *side, Map *map, int cancelled);
{
    char tclbuf[500];

    sprintf(tclbuf, "ask_position_mode {%s [click to set]}", prompt);
    Tcl_Eval(interp, tclbuf);
    map->answer[0] = '\0';
    map->modalhandler = handler;
}

int
grok_position(side, map, xp, yp, unitp)
Side *side;
Map *map;
int *xp, *yp;
Unit **unitp;
{
    if (in_area(map->inpx, map->inpy)) {
	*xp = map->inpx;  *yp = map->inpy;
	if (unitp != NULL)
	  *unitp = map->inpunit;
	eval_tcl_cmd("ask_position_done");
	return TRUE;
    } else {
	/* Make any possible usage attempts fail. */
	*xp = *yp = -1;
	if (unitp != NULL)
	  *unitp = NULL;
	return FALSE;
    }
}

/* Prompt for a yes/no answer with a settable default. */

void
ask_bool(side, map, question, dflt, handler)
Side *side;
Map *map;
char *question;
int dflt;
void (*handler)(Side *side, Map *map, int cancelled);
{
    char tclbuf[500];

    sprintf(tclbuf, "ask_bool_mode {%s [%s]}",
	    question, (dflt ? "yn" : "ny"));
    Tcl_Eval(interp, tclbuf);
    map->answer[0] = '\0';
    map->tmpint = dflt;
    map->modalhandler = handler;
}

/* Figure out what the answer actually is, keeping the default in mind. */

int
grok_bool(side, map)
Side *side;
Map *map;
{
    int dflt = map->tmpint;
    char ch = map->inpch;

    if (dflt ? (lowercase(ch) == 'n') : (lowercase(ch) == 'y'))
      dflt = !dflt;
    eval_tcl_cmd("ask_bool_done");
    return dflt;
}

/* Read a string from the prompt window.  Deletion is allowed, and a
   text cursor (an underscore) is displayed. */

void
ask_string(side, map, prompt, dflt, handler)
Side *side;
Map *map;
char *prompt, *dflt;
void (*handler)(Side *side, Map *map, int cancelled);
{
    char tclbuf[500];

    /* Default must be non-NULL. */
    if (dflt == NULL)
      dflt = "";
    sprintf(map->answer, "%s", dflt);
    sprintf(tclbuf, "ask_string_mode {%s} {%s}", prompt, map->answer);
    Tcl_Eval(interp, tclbuf);
    map->modalhandler = handler;
}

/* Dig a character from the input and add it into the string.
   Keep returning FALSE until we get something, then make a copy
   of the result string and return TRUE. */

int
grok_string(side, map, strp)
Side *side;
Map *map;
char **strp;
{
    char ch = map->inpch;
    int len;
    char tclbuf[500];

    if (ch == '\r' || ch == '\n') {
	*strp = copy_string(map->answer);
	eval_tcl_cmd("ask_string_done");
	return TRUE;
    } else {
	len = strlen(map->answer);
	if (ch == BACKSPACE_CHAR || ch == DELETE_CHAR) {
	    if (len > 0)
	      --len;
	} else {
	    map->answer[len++] = ch;
	}
	map->answer[len] = '\0';
	sprintf(tclbuf, "update_string_mode \"%s\"", map->answer);
	Tcl_Eval(interp, tclbuf);
	return FALSE;
    }
}

void
ask_side(side, map, prompt, dfltside, handler)
Side *side;
Map *map;
char *prompt;
Side *dfltside;
void (*handler)(Side *side, Map *map, int cancelled);
{
    char *dfltstr;
    char tclbuf[500];

    dfltstr = (dfltside == NULL ? "nobody" : side_name(dfltside));
    sprintf(map->answer, "%s", dfltstr);
    sprintf(tclbuf, "ask_side_mode {%s} {%s}", prompt, map->answer);
    Tcl_Eval(interp, tclbuf);
    map->modalhandler = handler;
}

int
grok_side(side, map, side2p)
Side *side;
Map *map;
Side **side2p;
{
    char ch = map->inpch;
    int len;
    Side *side3;
    char tclbuf[500];

    *side2p = NULL;
    if (ch == '\r' || ch == '\n') {
	if (empty_string(map->answer)
	    || strcmp(map->answer, "nobody") == 0) {
	    eval_tcl_cmd("ask_side_done");
	    return TRUE;
	}
	for_all_sides(side3) {
	    if (!empty_string(side3->name)
		&& strcmp(map->answer, side3->name) == 0) {
		*side2p = side3;
		eval_tcl_cmd("ask_side_done");
		return TRUE;
	    }
	    if (!empty_string(side3->noun)
		&& strcmp(map->answer, side3->noun) == 0) {
		*side2p = side3;
		eval_tcl_cmd("ask_side_done");
		return TRUE;
	    }
	    if (!empty_string(side3->adjective)
		&& strcmp(map->answer, side3->adjective) == 0) {
		*side2p = side3;
		eval_tcl_cmd("ask_side_done");
		return TRUE;
	    }
	}
	beep(side);
	return FALSE;
    } else {
	len = strlen(map->answer);
	if (ch == BACKSPACE_CHAR || ch == DELETE_CHAR) {
	    if (len > 0)
	      --len;
	} else {
	    map->answer[len++] = ch;
	}
	map->answer[len] = '\0';
	sprintf(tclbuf, "update_side_mode \"%s\"", map->answer);
	Tcl_Eval(interp, tclbuf);
	return FALSE;
    }
}

/* Map display widget. */

typedef struct {
    Tk_Window tkwin;		/* Window that embodies the MapW.  NULL
				 * means window has been deleted but
				 * widget record hasn't been cleaned up yet. */
    Display *display;		/* X's token for the window's display. */
    Tcl_Interp *interp;		/* Interpreter associated with widget. */
    Tcl_Command widgetCmd;	/* Token for MapW's widget command. */
    int border_width;		/* Width of 3-D border around whole widget. */
    Tk_3DBorder bg_border;	/* Used for drawing background. */
    int relief;			/* Indicates whether window as a whole is
				 * raised, sunken, or flat. */
    GC gc;
    GC copygc;			/* Graphics context for copying from
				 * off-screen pixmap onto screen. */
    int double_buffer;		/* Non-zero means double-buffer redisplay
				 * with pixmap;  zero means draw straight
				 * onto the display. */
    Drawable d;
    int update_pending;		/* Non-zero means a call to mapw_display
				 * has already been scheduled. */
    int rsx, rsy, rsw, rsh;     /* rect to update */

    int width, height;
    VP *vp;
    int power;
    int world;
    Map *map;

    int offsetx, offsety;
    int draw_polygons;
    int draw_lines;

    Tk_Font main_font;

    int blastsx, blastsy, blastsw, blastsh, blasttype;
} MapW;

static Tk_ConfigSpec config_specs[] = {
    {TK_CONFIG_BORDER, "-background", "background", "Background",
	"gray", Tk_Offset(MapW, bg_border), TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_BORDER, "-background", "background", "Background",
	"gray50", Tk_Offset(MapW, bg_border), TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_SYNONYM, "-bd", "borderWidth", (char *) NULL,
	(char *) NULL, 0, 0},
    {TK_CONFIG_SYNONYM, "-bg", "background", (char *) NULL,
	(char *) NULL, 0, 0},
    {TK_CONFIG_PIXELS, "-borderwidth", "borderWidth", "BorderWidth",
	"0", Tk_Offset(MapW, border_width), 0},
    {TK_CONFIG_INT, "-dbl", "doubleBuffer", "DoubleBuffer",
	"1", Tk_Offset(MapW, double_buffer), 0},
    {TK_CONFIG_PIXELS, "-height", "height", "Height",
	"0", Tk_Offset(MapW, height), 0},
    {TK_CONFIG_RELIEF, "-relief", "relief", "Relief",
	"sunken", Tk_Offset(MapW, relief), 0},
    {TK_CONFIG_INT, "-power", "power", "Power",
	"5", Tk_Offset(MapW, power), 0},
    {TK_CONFIG_PIXELS, "-width", "width", "Width",
	"0", Tk_Offset(MapW, width), 0},
    {TK_CONFIG_INT, "-world", "world", "World",
	"0", Tk_Offset(MapW, world), 0},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0}
};

static void mapw_cmd_deleted_proc(ClientData cldata);
static int mapw_configure(Tcl_Interp *interp, MapW *MapW,
			  int argc, char **argv, int flags);
static void mapw_destroy(char *memPtr);
static void mapw_event_proc(ClientData cldata, XEvent *eventPtr);
static int mapw_widget_cmd(ClientData cldata, Tcl_Interp *interp,
			   int argc, char **argv);
static void mapw_display(ClientData cldata);
static void draw_map_widget(MapW *mapw);

static void draw_row(MapW *mapw, int x0, int y0, int len);
static void draw_current(MapW *mapw);

static void draw_unit_image(MapW *mapw, int sx, int sy, int sw, int sh,
			    int u, int s2, int mod);
static void draw_side_emblem(MapW *mapw, int ex, int ey, int ew, int eh,
			     int s2);

Unit *x_find_unit_or_occ(MapW *mapw, Unit *unit,
			 int usx, int usy, int usw, int ush,
			 int sx, int sy);
Unit *x_find_unit_at(MapW *mapw, int x, int y, int sx, int sy);

static void xform(MapW *mapw, int x, int y, int *sxp, int *syp);
static void xform_fractional(MapW *mapw, int x, int y, int xf, int yf,
			     int *sxp, int *syp);

static void draw_feature_name(MapW *mapw, int f);
static void draw_hex_polygon(MapW *mapw, GC gc, int sx, int sy,
			     int power, int over, int dogrid);
static void draw_area_background(MapW *mapw);
static void meridian_line_callback(int x1, int y1, int x1f, int y1f,
				   int x2, int y2, int x2f, int y2f);
static void meridian_text_callback(int x1, int y1, int x1f, int y1f,
				   char *str);
static int cell_drawing_info(MapW *mapw, int x, int y, Pixmap *patp,
			     XColor **colorp, int *overp);
static void set_terrain_gc_for_image(MapW *mapw, GC gc, Image *timg);
static void draw_terrain_row(MapW *mapw, int x0, int y0, int len, int force);
static void draw_contours(MapW *mapw, int x0, int y0, int len);
static void draw_clouds_row(MapW *mapw, int x0, int y0, int len);
static void draw_temperature_row(MapW *mapw, int x0, int y0, int len);
static void draw_winds_row(MapW *mapw, int x0, int y0, int len);
static void draw_units(MapW *mapw, int x, int y);
static void draw_unit_and_occs(MapW *mapw, Unit *unit, int sx, int sy,
			       int sw, int sh, int drawoccs);
static void draw_unit_name(MapW *mapw, Unit *unit, int sx, int sy,
			   int sw, int sh);
static void draw_people(MapW *mapw, int x, int y);
static void draw_borders(MapW *mapw, int vx, int vy, int vw,
			 int y1, int y2, int b);
static void draw_connections(MapW *mapw, int vx, int vy, int vw,
			     int y1, int y2, int c);
static void draw_legend(MapW *mapw, int x, int y);
static void draw_country_border_line(MapW *mapw, int sx, int sy, int dir,
				     int con, int heavy);
static void draw_legend_text(MapW *mapw, int sx, int sy, int power,
			     char *str, XColor *color, int maskit);
static void draw_feature_boundary(MapW *mapw, int x, int y, int fid);
static void draw_meridians(MapW *mapw);

int
mapw_cmd(cldata, interp, argc, argv)
ClientData cldata;
Tcl_Interp *interp;
int argc;
char **argv;
{
    int x, y;
    Tk_Window main = (Tk_Window) cldata;
    MapW *mapw;
    Tk_Window tkwin;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		argv[0], " pathName ?options?\"", (char *) NULL);
	return TCL_ERROR;
    }

    tkwin = Tk_CreateWindowFromPath(interp, main, argv[1], (char *) NULL);
    if (tkwin == NULL)
      return TCL_ERROR;

    Tk_SetClass(tkwin, "MapW");

    /* Allocate and initialize the widget record.  */

    mapw = (MapW *) ckalloc(sizeof(MapW));
    mapw->tkwin = tkwin;
    mapw->display = Tk_Display(tkwin);
    mapw->interp = interp;
    mapw->widgetCmd =
      Tcl_CreateCommand(interp,
			Tk_PathName(mapw->tkwin), mapw_widget_cmd,
			(ClientData) mapw, mapw_cmd_deleted_proc);
    mapw->border_width = 0;
    mapw->bg_border = NULL;
    mapw->relief = TK_RELIEF_FLAT;
    mapw->gc = None;
    mapw->copygc = None;
    mapw->main_font = NULL;
    mapw->double_buffer = 1;
    mapw->update_pending = 0;

    mapw->offsetx = mapw->offsety = 0;
    mapw->draw_polygons = FALSE;
    mapw->draw_lines = FALSE;

    mapw->map = dside->ui->maps;

    mapw->rsw = mapw->rsh = -1;

    mapw->width = 0;
    mapw->height = 0;
    mapw->vp = new_vp();
    set_view_power(mapw->vp, 2);
    mapw->vp->draw_terrain = TRUE;
    mapw->vp->draw_grid = TRUE;
    mapw->vp->draw_units = TRUE;
    mapw->vp->latlong_interval = 600;

    Tk_CreateEventHandler(mapw->tkwin, ExposureMask|StructureNotifyMask,
			  mapw_event_proc, (ClientData) mapw);
    if (mapw_configure(interp, mapw, argc-2, argv+2, 0) != TCL_OK) {
	Tk_DestroyWindow(mapw->tkwin);
	return TCL_ERROR;
    }

    if (mapw->world)
      mapw->map->worldw = (char *) mapw;
    else
      mapw->map->widget = (char *) mapw;

    pick_a_focus(dside, &x, &y);
    set_view_focus(mapw->vp, x, y);

    mapw->blasttype = -1;

    interp->result = Tk_PathName(mapw->tkwin);
    return TCL_OK;
}

static int
mapw_widget_cmd(cldata, interp, argc, argv)
ClientData cldata;
Tcl_Interp *interp;
int argc;
char **argv;
{
    MapW *mapw = (MapW *) cldata;
    int result = TCL_OK;
    int update, sx, sy, nsx, nsy;
    size_t length;
    char c;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		argv[0], " option ?arg arg ...?\"", (char *) NULL);
	return TCL_ERROR;
    }
    Tcl_Preserve((ClientData) mapw);
    update = FALSE;
    sx = nsx = mapw->vp->sx;  sy = nsy = mapw->vp->sy;
    c = argv[1][0];
    length = strlen(argv[1]);
    if ((c == 'c') && (strncmp(argv[1], "cget", length) == 0)
	&& (length >= 2)) {
	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " cget option\"",
		    (char *) NULL);
	    goto error;
	}
	result = Tk_ConfigureValue(interp, mapw->tkwin, config_specs,
				   (char *) mapw, argv[2], 0);
	update = TRUE;
    } else if ((c == 'c') && (strncmp(argv[1], "configure", length) == 0)
	       && (length >= 2)) {
	if (argc == 2) {
	    result = Tk_ConfigureInfo(interp, mapw->tkwin, config_specs,
		    (char *) mapw, (char *) NULL, 0);
	} else if (argc == 3) {
	    result = Tk_ConfigureInfo(interp, mapw->tkwin, config_specs,
		    (char *) mapw, argv[2], 0);
	} else {
	    result = mapw_configure(interp, mapw, argc-2, argv+2,
		    TK_CONFIG_ARGV_ONLY);
	}
	update = TRUE;
    } else if ((c == 'x') && (strncmp(argv[1], "xview", length) == 0)) {
	int count, type;
	double fraction, fraction2;

	if (argc == 2) {
	    fraction = 0;
	    fraction2 = 1;
	    printf("map xview %g %g\n", fraction, fraction2);
	    sprintf(interp->result, "%g %g", fraction, fraction2);
	} else {
	    type = Tk_GetScrollInfo(interp, argc, argv, &fraction, &count);
	    switch (type) {
	      case TK_SCROLL_ERROR:
		goto error;
	      case TK_SCROLL_MOVETO:
		nsx = ((mapw->vp->sxmax - mapw->vp->sxmin) * fraction);
		break;
	      case TK_SCROLL_PAGES:
		nsx += (count * mapw->vp->pxw * 4) / 5;
		break;
	      case TK_SCROLL_UNITS:
		nsx += (count * 12);
		break;
	    }
	}
    } else if ((c == 'y') && (strncmp(argv[1], "yview", length) == 0)) {
	int count, type;
	double fraction, fraction2;

	if (argc == 2) {
	    fraction = 0;
	    fraction2 = 1;
	    printf("map yview %g %g\n", fraction, fraction2);
	    sprintf(interp->result, "%g %g", fraction, fraction2);
	} else {
	    type = Tk_GetScrollInfo(interp, argc, argv, &fraction, &count);
	    switch (type) {
	      case TK_SCROLL_ERROR:
		goto error;
	      case TK_SCROLL_MOVETO:
		nsy = ((mapw->vp->symax - mapw->vp->symin) * fraction);
		break;
	      case TK_SCROLL_PAGES:
		nsy += (count * mapw->vp->pxh * 4) / 5;
		break;
	      case TK_SCROLL_UNITS:
		nsy += (count * 12);
		break;
	    }
	}
    } else {
	Tcl_AppendResult(interp, "bad option \"", argv[1],
		"\": must be cget, configure, xview, or yview",
		(char *) NULL);
	goto error;
    }
    /* See if we're scrolling. */
    if (nsx != sx || nsy != sy) {
	int wid = Tk_Width(mapw->tkwin), hgt = Tk_Height(mapw->tkwin);
	Window win = Tk_WindowId(mapw->tkwin);
	GC gc = mapw->gc;

	XSetClipMask(mapw->display, gc, None);
	if (nsx != sx) {
	    if (nsx > sx) {
		XCopyArea(mapw->display, win, win, gc,
			  (nsx - sx), 0, wid, hgt, 0, 0);
		mapw->rsx = mapw->vp->pxw - (nsx - sx);  mapw->rsy = 0;
		mapw->rsw = nsx - sx;  mapw->rsh = mapw->vp->pxh;
	    } else {
		XCopyArea(mapw->display, win, win, gc,
			  0, 0, wid, hgt, (sx - nsx), 0);
		mapw->rsx = 0;  mapw->rsy = 0;
		mapw->rsw = (sx - nsx);  mapw->rsh = mapw->vp->pxh;
	    }
	} else { /* nsy != sy */
	    if (nsy > sy) {
		XCopyArea(mapw->display, win, win, gc,
			  0, (nsy - sy), wid, hgt, 0, 0);
		mapw->rsx = 0;  mapw->rsy = mapw->vp->pxh - (nsy - sy);
		mapw->rsw = mapw->vp->pxw;  mapw->rsh = nsy - sy;
	    } else {
		XCopyArea(mapw->display, win, win, gc,
			  0, 0, wid, hgt, 0, (sy - nsy));
		mapw->rsx = 0;  mapw->rsy = 0;
		mapw->rsw = mapw->vp->pxw;  mapw->rsh = sy - nsy;
	    }
	}
	set_view_position(mapw->vp, nsx, nsy);
	update = TRUE;
    }
    if (update && !mapw->update_pending) {
	Tcl_DoWhenIdle(mapw_display, (ClientData) mapw);
	mapw->update_pending = 1;
    }
    Tcl_Release((ClientData) mapw);
    return result;

error:
    Tcl_Release((ClientData) mapw);
    return TCL_ERROR;
}

static int
mapw_configure(interp, mapw, argc, argv, flags)
Tcl_Interp *interp;
MapW *mapw;
int argc;
char **argv;
int flags;
{
    int wid, hgt, pow;

    if (Tk_ConfigureWidget(interp, mapw->tkwin, config_specs,
			   argc, argv, (char *) mapw, flags) != TCL_OK)
      return TCL_ERROR;

    /* Set the background for the window and create a graphics context
       for use during redisplay.  */
    Tk_SetWindowBackground(mapw->tkwin,
			   Tk_3DBorderColor(mapw->bg_border)->pixel);
    if ((mapw->copygc == None) && 1/*(mapw->double_buffer)*/) {
	XGCValues gcValues;

	gcValues.function = GXcopy;
	gcValues.graphics_exposures = False;
	mapw->copygc = Tk_GetGC(mapw->tkwin,
				GCFunction|GCGraphicsExposures, &gcValues);
    }
    if (mapw->gc == None) {
	mapw->gc = Tk_GetGC(mapw->tkwin, None, NULL);
	XSetTSOrigin(mapw->display, mapw->gc, 0, 0);
    }
    if (mapw->main_font == NULL)
      mapw->main_font = Tk_GetFont(interp, mapw->tkwin, "fixed");

    /* Register the desired geometry for the window. */
    wid = mapw->width;  hgt = mapw->height;
    pow = mapw->power;
    if (wid > 0) {
	if (hgt > 0) {
	    /* do nothing */
	} else {
	    for (pow = 7; pow > 0; --pow) {
		if ((area.width * hws[pow] + 2 * mapw->border_width) <= wid)
		  break;
	    }
	    hgt = area.height * hcs[pow] + 2 * mapw->border_width;
	}
    } else {
	wid = 9000;
	hgt = 9000;
    }
    if (pow != mapw->vp->power) {
	set_view_power(mapw->vp, pow);
	center_on_focus(mapw->vp);
    }
    Tk_GeometryRequest(mapw->tkwin, wid, hgt);

    Tk_SetInternalBorder(mapw->tkwin, mapw->border_width);

    /* Set the map widget to be redisplayed when convenient. */
    if (!mapw->update_pending) {
	Tcl_DoWhenIdle(mapw_display, (ClientData) mapw);
	mapw->update_pending = 1;
    }
    return TCL_OK;
}

static void
mapw_event_proc(cldata, eventPtr)
ClientData cldata;
XEvent *eventPtr;
{
    MapW *mapw = (MapW *) cldata;

    if (eventPtr->type == Expose) {
	if (!mapw->update_pending) {
	    Tcl_DoWhenIdle(mapw_display, cldata);
	    mapw->update_pending = 1;
	}
    } else if (eventPtr->type == ConfigureNotify) {
	if (!mapw->update_pending) {
	    Tcl_DoWhenIdle(mapw_display, cldata);
	    mapw->update_pending = 1;
	}
    } else if (eventPtr->type == DestroyNotify) {
	if (mapw->tkwin != NULL) {
	    mapw->tkwin = NULL;
	    Tcl_DeleteCommand(mapw->interp,
			      Tcl_GetCommandName(mapw->interp,
						 mapw->widgetCmd));
	}
	if (mapw->update_pending) {
	    Tcl_CancelIdleCall(mapw_display, cldata);
	}
	Tcl_EventuallyFree(cldata, mapw_destroy);
    }
}

static void
mapw_cmd_deleted_proc(cldata)
ClientData cldata;
{
    MapW *mapw = (MapW *) cldata;
    Tk_Window tkwin = mapw->tkwin;

    if (tkwin != NULL) {
	mapw->tkwin = NULL;
	Tk_DestroyWindow(tkwin);
    }
}

static void
mapw_display(cldata)
ClientData cldata;
{
    Map *map;
    MapW *mapw = (MapW *) cldata;
    Tk_Window tkwin = mapw->tkwin;
    int winwidth = Tk_Width(tkwin), winheight = Tk_Height(tkwin);

    mapw->update_pending = 0;
    if (!Tk_IsMapped(tkwin))
      return;

    map = mapw->map;

    set_view_size(mapw->vp, winwidth, winheight);

    mapw->offsetx = mapw->offsety = 0;
    if (mapw->vp->totsw < mapw->vp->pxw)
      mapw->offsetx = (mapw->vp->pxw - mapw->vp->totsw) / 2;
    if (mapw->vp->totsh < mapw->vp->pxh)
      mapw->offsety = (mapw->vp->pxh - mapw->vp->totsh) / 2;

    if (mapw == (MapW *) map->widget)
      set_scrollbars(mapw);

    if (mapw->rsw >= 0 && mapw->rsh >= 0) {
	VP *saved_vp = mapw->vp, tmpvp;

#if 0 /* use this to see each rect update */
	XSetClipMask(mapw->display, mapw->gc, None);
	XSetForeground(mapw->display, mapw->gc, dside->ui->badcolor->pixel);
	XFillRectangle(mapw->display, Tk_WindowId(tkwin), mapw->gc,
		       mapw->rsx - 1, mapw->rsy - 1,
		       mapw->rsw + 2, mapw->rsh + 2);
#endif
	/* Redraw a restricted rect of the widget. */
	tmpvp = *(mapw->vp);
	if (mapw->rsw == 0)
	  mapw->rsw = 1;
	if (mapw->rsh == 0)
	  mapw->rsh = 1;
	set_view_size(&tmpvp, mapw->rsw, mapw->rsh);
	set_view_position(&tmpvp, tmpvp.sx + mapw->rsx, tmpvp.sy + mapw->rsy);
	mapw->vp = &tmpvp;
	mapw->d = Tk_GetPixmap(mapw->display, Tk_WindowId(tkwin),
			       mapw->rsw, mapw->rsh,
			       DefaultDepthOfScreen(Tk_Screen(tkwin)));
	Tk_Fill3DRectangle(tkwin, mapw->d, mapw->bg_border, 0, 0, mapw->rsw,
			   mapw->rsh, 0, mapw->relief);
	XSetTSOrigin(mapw->display, mapw->gc, - mapw->rsx, - mapw->rsy);
	draw_map_widget(mapw);
	XSetTSOrigin(mapw->display, mapw->gc, 0, 0);
	/* Copy to the screen and release the pixmap. */
	XCopyArea(mapw->display, mapw->d, Tk_WindowId(tkwin), mapw->copygc,
		  0, 0,
		  (unsigned) mapw->rsw, (unsigned) mapw->rsh,
		  mapw->rsx, mapw->rsy);
	Tk_FreePixmap(mapw->display, mapw->d);
	mapw->vp = saved_vp;
    } else {
	/* Do a full redraw of the widget. */
	mapw->d = Tk_WindowId(tkwin);
	/* Create a pixmap for double-buffering if preferred. */
	if (mapw->double_buffer) {
	    mapw->d = Tk_GetPixmap(mapw->display, Tk_WindowId(tkwin),
				   winwidth, winheight,
				   DefaultDepthOfScreen(Tk_Screen(tkwin)));
	}
	/* Redraw the widget's background and border. */
	Tk_Fill3DRectangle(tkwin, mapw->d, mapw->bg_border, 0, 0, winwidth,
			   winheight, mapw->border_width, mapw->relief);
	draw_map_widget(mapw);
	/* If double-buffered, copy to the screen and release the pixmap. */
	if (mapw->double_buffer) {
	    XCopyArea(mapw->display, mapw->d, Tk_WindowId(tkwin), mapw->copygc,
		      0, 0,
		      (unsigned) winwidth, (unsigned) winheight,
		      0, 0);
	    Tk_FreePixmap(mapw->display, mapw->d);
	}
    }
    mapw->rsw = mapw->rsh = -1;
}

set_scrollbars(mapw)
MapW *mapw;
{
    char tclbuf[500];
    float fsx, fsy, first, last;

    fsx = (float) mapw->vp->sx;
    first = (fsx - mapw->vp->sxmin) / mapw->vp->totsw;
    last = (fsx - mapw->vp->sxmin + mapw->vp->pxw) / mapw->vp->totsw;
    DGprintf("xscroll set %f %f\n", first, last);
    sprintf(tclbuf, "map_xscroll_set %f %f", first, last);
    Tcl_Eval(interp, tclbuf);
    fsy = (float) mapw->vp->sy;
    first = (fsy - mapw->vp->symin) / mapw->vp->totsh;
    last = (fsy - mapw->vp->symin + mapw->vp->pxh) / mapw->vp->totsh;
    DGprintf("yscroll set %f %f\n", first, last);
    sprintf(tclbuf, "map_yscroll_set %f %f", first, last);
    Tcl_Eval(interp, tclbuf);
}

static void
mapw_destroy(ptr)
char *ptr;
{
    MapW *mapw = (MapW *) ptr;

    Tk_FreeOptions(config_specs, (char *) mapw, mapw->display, 0);
    if (mapw->gc != None)
      Tk_FreeGC(mapw->display, mapw->gc);
    if (mapw->copygc != None)
      Tk_FreeGC(mapw->display, mapw->copygc);
    if (mapw->main_font != NULL)
      Tk_FreeFont(mapw->main_font);
    ckfree((char *) mapw);
}


/* Transform map coordinates into screen coordinates. */

static void
xform(MapW *mapw, int x, int y, int *sxp, int *syp)
{
    xform_cell(mapw->vp, x, y, sxp, syp);
    *sxp += mapw->border_width;  *syp += mapw->border_width;
    *sxp += mapw->offsetx;  *syp += mapw->offsety;
}

/* Transform, but also use position within cell. */

static void
xform_fractional(MapW *mapw, int x, int y, int xf, int yf, int *sxp, int *syp)
{
    xform_cell_fractional(mapw->vp, x, y, xf, yf, sxp, syp);
    *sxp += mapw->border_width;  *syp += mapw->border_width;
    *sxp += mapw->offsetx;  *syp += mapw->offsety;
}

void
x_xform_unit(MapW *mapw, Unit *unit, int *sxp, int *syp, int *swp, int *shp)
{
    xform_unit(mapw->vp, unit, sxp, syp, swp, shp);
}

void
x_xform_unit_self(MapW *mapw, Unit *unit, int *sxp, int *syp,
		  int *swp, int *shp)
{
    xform_unit_self(mapw->vp, unit, sxp, syp, swp, shp);
}

void
x_xform_occupant(MapW *mapw, Unit *transport, Unit *unit, int sx, int sy,
		 int sw, int sh, int *sxp, int *syp, int *swp, int *shp)
{
    xform_occupant(mapw->vp, transport, unit, sx, sy, sw, sh, sxp, syp,
		   swp, shp);
}

int
x_nearest_cell(MapW *mapw, int sx, int sy, int *xp, int *yp)
{
    sx -= mapw->border_width;  sy -= mapw->border_width;
    sx -= mapw->offsetx;  sy -= mapw->offsety;
    return nearest_cell(mapw->vp, sx, sy, xp, yp, NULL, NULL);
}

Unit *
x_find_unit_or_occ(MapW *mapw, Unit *unit, int usx, int usy, int usw, int ush,
		   int sx, int sy)
{
    int usx1, usy1, usw1, ush1;
    Unit *occ, *rslt;

    /* See if the point might be over an occupant. */
    if (unit->occupant != NULL) {
	for_all_occupants(unit, occ) {
	    x_xform_unit(mapw, occ, &usx1, &usy1, &usw1, &ush1);
	    rslt =
	      x_find_unit_or_occ(mapw, occ, usx1, usy1, usw1, ush1, sx, sy);
	    if (rslt) {
		return rslt;
	    }
	}
    }
    /* Otherwise see if it could be the unit itself.  This has the effect of
       "giving" the transport everything in its box that is not in an occ. */
    x_xform_unit(mapw, unit, &usx1, &usy1, &usw1, &ush1);
    if (between(usx1, sx, usx1 + usw1) && between(usy1, sy, usy1 + ush1)) {
	return unit;
    }
    return NULL;
}

Unit *
x_find_unit_at(mapw, x, y, sx, sy)
MapW *mapw;
int x, y, sx, sy;
{
    int usx, usy, usw, ush;
    Unit *unit, *rslt;
	
    for_all_stack(x, y, unit) {
	x_xform_unit(mapw, unit, &usx, &usy, &usw, &ush);
	rslt = x_find_unit_or_occ(mapw, unit, usx, usy, usw, ush, sx, sy);
	if (rslt)
	  return rslt;
    }
    return NULL;
}

int
x_nearest_unit(MapW *mapw, int sx, int sy, Unit **unitp)
{
    int x, y;

    if (!x_nearest_cell(mapw, sx, sy, &x, &y)) {
	*unitp = NULL;
    } else if (mapw->vp->uw >= 32) {
	*unitp = x_find_unit_at(mapw, x, y, sx, sy);
    } else {
	*unitp = unit_at(x, y);
    }
    DGprintf("Pixel %d,%d -> unit %s\n", sx, sy, unit_desig(*unitp));
    return TRUE;
}

/* Draw the background area for the map. */

static void
draw_area_background(mapw)
MapW *mapw;
{
    int sx, sy, sx2, sy2, sw, sh, aw, ah, i;
    int llx, lly, lrx, lry, rx, ry, urx, ury, ulx, uly, lx, ly;
    XPoint points[6];
    Display *dpy = mapw->display;
    GC gc = mapw->gc;
    XColor *color;

    /* If the world is unexplored, then it should have the unseen color. */
    color = (1 /* grid color matches unseen color */ ? dside->ui->grid_color : dside->ui->unseen_color);
    XSetClipMask(dpy, gc, None);
    XSetForeground(dpy, gc, color->pixel);
    XSetLineAttributes(dpy, gc, 1, LineSolid, CapButt, JoinMiter);

    if (area.xwrap) {
	/* Area is cylinder; draw a rectangle. */
	xform(mapw, 0, area.height - 1, &sx, &sy);
	xform(mapw, area.width - 1, 0, &sx2, &sy2);
	sw = sx2 - sx;
	sh = sy2 - sy;
	XFillRectangle(dpy, mapw->d, gc, sx, sy + mapw->vp->hh / 2, sw, sh);
	if (0 /* map bg matches widget bg */) {
	    XSetForeground(dpy, gc, dside->ui->whitecolor->pixel);
	    XDrawRectangle(dpy, mapw->d, gc, sx, sy + mapw->vp->hh / 2, sw, sh);
	}
    } else {
	/* Area is hexagon; draw a hexagon. */
	aw = area.width;  ah = area.height;
	xform(mapw, 0 + ah / 2, 0, &llx, &lly);
	points[0].x = llx;  points[0].y = lly;
	xform(mapw, aw - 1, 0, &lrx, &lry);
	points[1].x = lrx;  points[1].y = lry;
	xform(mapw, aw - 1, ah / 2, &rx, &ry);
	points[2].x = rx;   points[2].y = ry;
	xform(mapw, aw - 1 - ah / 2, ah - 1, &urx, &ury);
	points[3].x = urx;  points[3].y = ury;
	xform(mapw, 0, ah - 1, &ulx, &uly);
	points[4].x = ulx;  points[4].y = uly;
	xform(mapw, 0, ah / 2, &lx, &ly);
	points[5].x = lx;   points[5].y = ly;
	/* Offset so polygon edges run through middles of cells. */
	for (i = 0; i < 6; ++i) {
	    points[i].x += mapw->vp->hw / 2;  points[i].y += mapw->vp->hh / 2;
	}
	XFillPolygon(dpy, mapw->d, gc, points, 6, Convex, CoordModeOrigin);
	if (0 /* map bg matches widget bg */) {
	    XSetForeground(dpy, gc, dside->ui->whitecolor->pixel);
	    XDrawLines(dpy, mapw->d, gc, points, 7, CoordModeOrigin);
	}
    }
}

/* Draw the view proper. */

void
draw_map_widget(mapw)
MapW *mapw;
{
    int y1, y2, y, x1, vx, vy, vw, vh, len, t;

    draw_area_background(mapw);

    if (mapw->vp->vcx < 0 || mapw->vp->vcy < 0) {
    	run_warning("doing a nasty hack");
	mapw->vp->vcx = mapw->vp->vcy = 2;
    }
    /* Compute the width and height. */
    vw = min(area.width, mapw->vp->pxw / mapw->vp->hw + 2);
    vh = min(area.height, mapw->vp->pxh / mapw->vp->hch + 2);

    vy = ((mapw->vp->totsh - mapw->vp->sy) / mapw->vp->hch) - vh;
    /* Now adjust the bottom row so it doesn't go outside the area. */
    if (vy < 0)
      vy = 0;
    if (vy > area.height - vh)
      vy = area.height - vh;
    /* Compute the leftmost "column". */
    vx = mapw->vp->sx / mapw->vp->hw - vy / 2 - 1;
    DGprintf("Set map %x viewport to be %dx%d @ %d,%d (nom center %d,%d)\n",
	     mapw, vw, vh, vx, vy,
	     mapw->vp->vcx, mapw->vp->vcy);
    /* Compute top and bottom rows to be displayed. */
    /* Include rows that will only be partly drawn. */
    y1 = min(vy + vh, area.height - 1);
    y2 = vy;
    if (mapw->vp->draw_terrain) {
	for (y = y1; y >= y2; --y) {
	    if (!compute_x1_len(mapw, vx, vy, vw, y, &x1, &len))
	      continue;
	    draw_terrain_row(mapw, x1, y, len, FALSE);
	}
	if (any_aux_terrain_defined()) {
	    for_all_terrain_types(t) {
		if (t_is_border(t)
		    && aux_terrain_defined(t)
		    && bwid[mapw->vp->power] > 0) {
		    draw_borders(mapw, vx, vy, vw, y1, y2, t);
		} else if (t_is_connection(t)
			   && aux_terrain_defined(t)
			   && cwid[mapw->vp->power] > 0) {
		    draw_connections(mapw, vx, vy, vw, y1, y2, t);
		}
	    }
	}
	/* The relative ordering of these is quite important.  Note that
	   each should be prepared to run independently also, since the
	   other displays might have been turned off. */
	if (elevations_defined()
	    && mapw->vp->draw_elevations
	    && mapw->vp->angle == 90) {
	    for (y = y1; y >= y2; --y) {
		if (!compute_x1_len(mapw, vx, vy, vw, y, &x1, &len))
		  continue;
		draw_contours(mapw, x1, y, len);
	    }
	}
    }
    /* Now draw the lat-long grid if asked to do so. */
    if (mapw->vp->draw_latlong && mapw->vp->latlong_interval > 0)
      draw_meridians(mapw);
    for (y = y1; y >= y2; --y) {
	if (!compute_x1_len(mapw, vx, vy, vw, y, &x1, &len))
	  continue;
	draw_row(mapw, x1, y, len);
    }
    /* Occupants need to get redrawn on top of everything else if they're
       selected. */
    if (mapw->map->curunit && mapw->map->curunit->transport)
      draw_current(mapw);
    if (mapw->blasttype >= 0)
      draw_blast_image(mapw, mapw->blastsx - mapw->rsx, mapw->blastsy - mapw->rsy,
		       mapw->blastsw, mapw->blastsh, mapw->blasttype);
}

/* Given a row in the viewport, compute the starting cell and length
   of the row of cells to draw. */

int
compute_x1_len(MapW *mapw, int vx, int vy, int vw, int y, int *x1p, int *lenp)
{
    int adj, x1, x2;
    int halfheight = area.height / 2;

    /* Adjust the right and left bounds to fill the viewport as much
       as possible, without going too far (the drawing code will clip,
       but clipped drawing is still expensive). */
    /* could test by drawing viewport rect as lines... */
    adj = (y - vy) / 2;
    /* If the area doesn't wrap, then we might have to stop
	   drawing before we reach the edge of the viewport. */
	/* (is this really reliable?) */
    x1 = vx - (y - vy) / 2;
    x2 = x1 + vw + 2 /* bleah, shouldn't be necessary */;
    if (area.xwrap) {
    } else {
	/* Truncate x's to stay within the area. */
	x1 = max(0, min(x1, area.width-1));
	x2 = max(0, min(x2, area.width));
	/* If this row is entirely in the NE corner, don't draw
	   anything. */
	if (x1 + y > area.width + halfheight)
	  return FALSE;
	/* If this row is entirely in the SW corner, don't draw
	   anything. */
	if (x2 + y < halfheight)
	  return FALSE;
	/* If the row ends up in the NE corner, shorten it. */
	if (x2 + y > area.width + halfheight)
	  x2 = area.width + halfheight - y;
	/* If the row starts out in the SW corner, shorten it. */
	if (x1 + y < halfheight)
	  x1 = halfheight - y;
    }
    *x1p = x1;
    *lenp = x2 - x1;
    return (*lenp > 0);
}

/* Temporary stashes for the meridian drawing callbacks. */

static MapW *tmpmapw;

/* Draw latitude and longitude lines & labels. */

static void
draw_meridians(MapW *mapw)
{
    Display *dpy = mapw->display;

    XSetClipMask(dpy, mapw->gc, None);
    XSetLineAttributes(dpy, mapw->gc, 1, LineSolid, CapButt, JoinMiter); 
    XSetForeground(dpy, mapw->gc, dside->ui->meridian_color->pixel);
    XSetBackground(dpy, mapw->gc, dside->ui->bgcolor->pixel);
    tmpmapw = mapw;
    plot_meridians(mapw->vp, meridian_line_callback, meridian_text_callback);
}

static void
meridian_line_callback(x1, y1, x1f, y1f, x2, y2, x2f, y2f)
int x1, y1, x1f, y1f, x2, y2, x2f, y2f;
{
    int sx1, sy1, sx2, sy2;

    xform_fractional(tmpmapw, x1, y1, x1f, y1f, &sx1, &sy1);
    xform_fractional(tmpmapw, x2, y2, x2f, y2f, &sx2, &sy2);
    XDrawLine(tmpmapw->display, tmpmapw->d, tmpmapw->gc, sx1, sy1, sx2, sy2);
}

static void
meridian_text_callback(x1, y1, x1f, y1f, str)
int x1, y1, x1f, y1f;
char *str;
{
    int sx1, sy1;

    xform_fractional(tmpmapw, x1, y1, x1f, y1f, &sx1, &sy1);
    XSetClipMask(tmpmapw->display, tmpmapw->gc, None);
    XSetForeground(tmpmapw->display, tmpmapw->gc, dside->ui->fgcolor->pixel);
    Tk_DrawChars(tmpmapw->display, tmpmapw->d, tmpmapw->gc, tmpmapw->main_font,
		 str, strlen(str), sx1 + 1, sy1 + 1);
}

/* The basic map drawing routine does an entire row at a time, which yields
   order-of-magnitude speedups. */

static void
draw_row(MapW *mapw, int x0, int y0, int len)
{
    int x, i;

    if (clouds_defined() && mapw->vp->draw_clouds) {
	draw_clouds_row(mapw, x0, y0, len);
    }
    if (temperatures_defined() && mapw->vp->draw_temperature && mapw->vp->hw > 10) {
	draw_temperature_row(mapw, x0, y0, len);
    }
    if (winds_defined() && mapw->vp->draw_winds && mapw->vp->hw > 10) {
	draw_winds_row(mapw, x0, y0, len);
    }
    /* Skip the top and bottom rows if they are edge rows. */
    if (!between(1, y0, area.height - 2))
      return;
    /* Skip the rightmost and leftmost cells if on the edge. */
    if (!inside_area(x0 + len - 1, y0))
      --len;
    if (!inside_area(x0, y0)) {
	++x0;
	--len;
    }
    if (len <= 0)
      return;
    if (features_defined() && mapw->vp->draw_feature_boundaries) {
	for (x = x0; x < x0 + len; ++x) {
	    draw_feature_boundary(mapw, x, y0, raw_feature_at(x, y0));
	}
    }
    if (features_defined() && mapw->vp->draw_feature_names && dside->ui->legends) {
	for (i = 0; i < dside->ui->numfeatures; ++i) {
	    if (dside->ui->legends[i].oy == y0) {
		draw_feature_name(mapw, i);
	    }
	}
    }
    /* Draw sparse things on top of the basic row. */
    if (((people_sides_defined() && mapw->vp->draw_people)
	 || (control_sides_defined() && mapw->vp->draw_control))
	&& mapw->vp->hw >= 8) {
	for (x = x0; x < x0 + len; ++x) {
	    draw_people(mapw, x, y0);
	}
    }
#if 0
    /* Draw units. */
    /* (Do names first, so that unit images don't disappear behind names
       when densely packed) */
    if (mapw->vp->draw_names && mapw->vp->hh >= 8) {
	for (x = x0; x < x0 + len; ++x) {
	    draw_legend(mapw, x, y0);
	}
    }
#endif
    if (mapw->vp->draw_units /*&& mapw->vp->hw > 2*/) {
	for (x = x0; x < x0 + len; ++x) {
	    draw_units(mapw, x, y0);
	}
    }
}

char buffer[BUFSIZE];

static void
draw_feature_name(MapW *mapw, int f)
{
    Legend *legend = &dside->ui->legends[f];
    int x = legend->ox, y = legend->oy;
    int dist = ((legend->dx + 1) * mapw->vp->hw * 9) / 10;
    int i, b, l, lnew, sx0, sy0, sxc2, syc;
    char *pad, *name;
    Display *dpy = mapw->display;
    GC gc = mapw->gc;

    name = feature_desc(find_feature(f + 1), buffer);
    if (empty_string(name))
      return;

    xform(mapw, x, y, &sx0, &sy0);
    /* xform returns coordinates of the upper-left corner of the cell */
    sxc2 = 2 * sx0 + (legend->dx + 1) * mapw->vp->hw; /* twice center x */
    syc  = sy0 + mapw->vp->hh / 2;		  /* center y */
    if (sxc2 + 2 * dist < 0)
      return;

    XSetClipMask(dpy, gc, None);
    XSetFillStyle(dpy, gc, FillSolid);
    XSetForeground(dpy, gc, dside->ui->feature_color->pixel);
    Tk_DrawChars(dpy, mapw->d, gc, mapw->main_font, name, strlen(name),
		 sxc2 / 2, syc);
}

static int
cell_drawing_info(MapW *mapw, int x, int y, Pixmap *patp, XColor **colorp,
		  int *overp)
{
    int t;
    enum whattouse rslt;

    *patp = None;
    *colorp = dside->ui->whitecolor;
    *overp = 0;
    if (mapw->map->see_all || terrain_visible(dside, x, y)) {
	t = terrain_at(x, y);
	*patp = dside->ui->terrpics[mapw->vp->power][t];
	*colorp = dside->ui->cellcolor[t];
	if (*colorp == NULL)
	  *colorp = dside->ui->blackcolor;
	if (mapw->vp->draw_cover
	    && !mapw->map->see_all
	    && cover(dside, x, y) == 0)
	  *overp = -1;
	if (mapw->vp->draw_lighting && night_at(x, y))
	  *overp = -2;
	if (dside->designer && terrain_view(dside, x, y) == UNSEEN)
	  *overp = -2;
	rslt = (mapw->vp->power < 4 ? useblocks : usepictures);
	if (mapw->draw_polygons && mapw->vp->power >= 3)
	  rslt = usepolygons;
    } else {
	rslt = dontdraw;
    }
    return rslt;
}

static void
set_terrain_gc_for_image(MapW *mapw, GC gc, Image *timg)
{
    TkImage *tkimg;
    Display *dpy = mapw->display;

    if (timg != NULL) {
	tkimg = (TkImage *) timg->hook;
	if (tkimg != NULL) {
	    if (!dside->ui->monochrome) {
		if (mapw->map->terrain_style == 1 && mapw->vp->hw >= 4) {
		    if (tkimg->colr != None) {
			XSetFillStyle(dpy, gc, FillTiled);
			XSetTile(dpy, gc, tkimg->colr);
		    } else if (tkimg->mono != None) {
			XSetFillStyle(dpy, gc, FillOpaqueStippled);
			XSetStipple(dpy, gc, tkimg->mono);
		    } else {
			XSetFillStyle(dpy, gc, FillSolid);
		    }
		} else {
		    XSetFillStyle(dpy, gc, FillSolid);
		}
	    } else if (tkimg->mono != None) {
		XSetFillStyle(dpy, gc, FillOpaqueStippled);
		XSetStipple(dpy, gc, tkimg->mono);
	    } else {
		/* No pattern, no color - what to do? */
		XSetFillStyle(dpy, gc, FillSolid);
	    }
	} else {
	    XSetFillStyle(dpy, gc, FillSolid);
	}
    } else {
	XSetFillStyle(dpy, gc, FillSolid);
    }
}

/* This interfaces higher-level drawing decisions to the rendition of
   individual pieces of display.  The rendering technique chosen
   depends on what the init code has decided is appropriate given what
   it found during init and what magnification the display is at.

   This routine is performance-critical; any improvements will
   probably have a noticeable effect on the display.  But also note
   that X's main bottleneck is the network connection, so it's more
   useful to eliminate roundtrips to the server than anything else. */

/* (should cache best images, never have to look up in here) */

static void
draw_terrain_row(mapw, x0, y0, len, force)
MapW *mapw;
int x0, y0, len, force;
{
    int x0w, x1, x1w, x, xw, t, sx, sy, i = 0, j;
    int w = mapw->vp->hw, h = mapw->vp->hh, p = mapw->vp->power;
    int dogrid = mapw->vp->draw_grid;
    int over, segover;
    XColor *color, *segcolor;
    enum whattouse drawmethod, segdrawmethod;
    Pixmap pat, segpat;
    Image *timg;
    GC gc = mapw->gc;
    Display *dpy = mapw->display;
    enum grayshade shade;

    x0w = wrapx(x0);
    x1 = x0;
    x1w = wrapx(x1);
    segdrawmethod = cell_drawing_info(mapw, x0w, y0,
				      &segpat, &segcolor, &segover);
    for (x = x0; x < x0 + len + 1; ++x) {
	xw = wrapx(x);
	t = terrain_at(xw, y0);
	drawmethod = cell_drawing_info(mapw, xw, y0, &pat, &color, &over);
	/* Decide if the run is over and we need to dump some output. */
	if (x == x0 + len
	    || x == area.width
	    || drawmethod != segdrawmethod
	    || color != segcolor
	    || pat != segpat
	    || over != segover
	    || segdrawmethod == usepolygons
	    || force) {
	    /* Note: we might end up drawing something that matches
	       the background color, which wastes time, but apparently
	       the test "(segdrawmethod != dontdraw && segcolor !=
	       dside->ui->bgcolor)" is not completely sufficient.
	       (should figure this one out sometime) */
	    t = terrain_at(x1w, y0);
	    timg = best_image(dside->ui->timages[t], w, h);
	    xform(mapw, x1, y0, &sx, &sy);
	    XSetForeground(dpy, gc, segcolor->pixel);
	    switch (segdrawmethod) {
	      case dontdraw:
		/* Don't do anything. */
		break;
	      case useblocks:
		XSetClipMask(dpy, gc, None);
		set_terrain_gc_for_image(mapw, gc, timg);
		XFillRectangle(dpy, mapw->d, gc, sx, sy, i * w, h);
		if (segover < 0) {
		    shade = gray;
		    if (segover == -2)
		      shade = darkgray;
		    XSetFillStyle(dpy, gc, FillStippled);
		    XSetStipple(dpy, gc, dside->ui->grays[shade]);
		    XSetForeground(dpy, gc, dside->ui->blackcolor->pixel);
		    XFillRectangle(dpy, mapw->d, gc, sx, sy, i * w, h);
		}
		break;
	      case usepictures:
		if (dogrid) {
		    XSetClipMask(dpy, gc, dside->ui->bhexpics[p]);
		} else {
		    XSetClipMask(dpy, gc, dside->ui->hexpics[p]);
		}
		set_terrain_gc_for_image(mapw, gc, timg);
		for (j = 0; j < i; ++j) {
		    xform(mapw, x1 + j, y0, &sx, &sy);
		    XSetClipOrigin(dpy, gc, sx, sy);
		    XFillRectangle(dpy, mapw->d, gc, sx, sy, w, h);
		}
		if (segover < 0) {
		    shade = gray;
		    if (segover == -2)
		      shade = darkgray;
		    XSetFillStyle(dpy, gc, FillStippled);
		    XSetStipple(dpy, gc, dside->ui->grays[shade]);
		    XSetForeground(dpy, gc, dside->ui->blackcolor->pixel);
		    for (j = 0; j < i; ++j) {
			xform(mapw, x1 + j, y0, &sx, &sy);
			XSetClipOrigin(dpy, gc, sx, sy);
			XFillRectangle(dpy, mapw->d, gc, sx, sy, w, h);
		    }
		}
		break;
	      case usepolygons:
		XSetClipMask(dpy, gc, None);
		set_terrain_gc_for_image(mapw, gc, timg);
		draw_hex_polygon(mapw, gc, sx, sy, p, segover, dogrid);
	    }
	    XSetFillStyle(dpy, gc, FillSolid);
	    /* Reset everything for the next run. */
	    i = 0;
	    x1 = x;
	    x1w = wrapx(x1);
	    segdrawmethod = drawmethod;
	    segpat = pat;
	    segcolor = color;
	    segover = over;
	}
	++i;
    }
}

static void
draw_contours(MapW *mapw, int x0, int y0, int len)
{
    int xx, x, y;
    int i, sx, sy, numlines;
    LineSegment *lines;
    Display *dpy = mapw->display;
    GC gc = mapw->gc;

    if (mapw->vp->contour_interval < 1)
      return;
    XSetClipMask(dpy, gc, None);
    XSetLineAttributes(dpy, gc, 1, LineSolid, CapButt, JoinMiter); 
    XSetForeground(dpy, gc, dside->ui->contour_color->pixel);
    y = y0;
    for (xx = x0; xx < x0 + len; ++xx) {
	x = wrapx(xx);
	if (terrain_visible(dside, x, y0)) {
	    xform(mapw, x, y, &sx, &sy);
	    contour_lines_at(mapw->vp, x, y, sx, sy, &lines, &numlines);
	    for (i = 0; i < numlines; ++i) {
		XDrawLine(dpy, mapw->d, gc,
			  lines[i].sx1, lines[i].sy1,
			  lines[i].sx2, lines[i].sy2);
	    }
	}
    }
}

static void
draw_clouds_row(MapW *mapw, int x0, int y0, int len)
{
    int x, xw, sx, sy;
    Display *dpy = mapw->display;

    for (x = x0; x < x0 + len - 1; ++x) {
	xw = wrapx(x);
	if (mapw->map->see_all || terrain_visible(dside, xw, y0)) {
	    sprintf(spbuf, "%d", cloud_view(dside, xw, y0));
	    xform(mapw, x, y0, &sx, &sy);
	    XSetClipMask(dpy, mapw->gc, None);
	    XSetForeground(dpy, mapw->gc, dside->ui->fgcolor->pixel);
	    Tk_DrawChars(dpy, mapw->d, mapw->gc, mapw->main_font,
			 spbuf, strlen(spbuf), sx + 5, sy + mapw->vp->uh / 2);
	}
    }
}

static void
draw_temperature_row(MapW *mapw, int x0, int y0, int len)
{
    int x, xw, sx, sy;
    Display *dpy = mapw->display;

    for (x = x0; x < x0 + len - 1; ++x) {
	xw = wrapx(x);
	if (mapw->map->see_all || terrain_visible(dside, xw, y0)) {
	    sprintf(spbuf, "%d", temperature_view(dside, xw, y0));
	    xform(mapw, x, y0, &sx, &sy);
	    XSetClipMask(dpy, mapw->gc, None);
	    XSetForeground(dpy, mapw->gc, dside->ui->fgcolor->pixel);
	    Tk_DrawChars(dpy, mapw->d, mapw->gc, mapw->main_font,
			 spbuf, strlen(spbuf), sx + 5, sy + mapw->vp->uh / 2);
	}
    }
}

static void
draw_winds_row(MapW *mapw, int x0, int y0, int len)
{
    int x, xw, sx, sy;
    Display *dpy = mapw->display;
    GC gc = mapw->gc;

    for (x = x0; x < x0 + len - 1; ++x) {
	xw = wrapx(x);
	if (draw_winds_here(dside, xw, y0)) {
	    int rawwind = wind_view(dside, xw, y0);
	    int wdir = wind_dir(rawwind), wforce = wind_force(rawwind), swforce;
	    xform(mapw, x, y0, &sx, &sy);
	    
	    sx += (mapw->vp->hw - 16) / 2;  sy += (mapw->vp->hh - 16) / 2;
	    if (wforce < 0) {
		DGprintf("negative wind force %d, substituting 0", wforce);
		wforce = 0;
	    }
	    swforce =
	      ((wforce - minwindforce) * 5) / (maxwindforce - minwindforce);
	    if (swforce > 4)
	      swforce = 4;
	    if (swforce == 0)
	      wdir = 0;
	    XSetClipMask(dpy, gc, windpics[wforce][wdir]);
	    XSetClipOrigin(dpy, gc, sx, sy);
	    /* Draw a white arrow, offset slightly, for contrast on dark
	       backgrounds. */
	    XSetForeground(dpy, gc, dside->ui->whitecolor->pixel);
	    XFillRectangle(dpy, mapw->d, gc, sx + 1, sy + 1, 16, 16);
	    /* Draw the main black arrow. */
	    XSetForeground(dpy, gc, dside->ui->blackcolor->pixel);
	    XFillRectangle(dpy, mapw->d, gc, sx, sy, 16, 16);
	}
    }
}

/* Draw all the units in the given cell. */

static void
draw_units(MapW *mapw, int x, int y)
{
    int xw = wrapx(x), sx, sy, sw, sh, uview, u, s, osx, osy;
    int uw = mapw->vp->uw, uh = mapw->vp->uh, didone;
    Unit *unit;
    Display *dpy = mapw->display;
    GC gc = mapw->gc;

    if ((mapw->map->see_all || units_visible(dside, xw, y))
	&& unit_at(xw, y) != NULL) {
	if (uw <= 16) {
	    /* At smaller mags, we choose a single best unit to display. */
	    unit = NULL;
	    for_all_stack(x, y, unit) {
		if (unit->side == dside)
		  break;
	    }
	    if (unit == NULL) {
		for_all_stack(x, y, unit) {
		    if (mapw->map->see_all
			|| unit_actually_visible(dside, unit))
		      break;
		}
	    }
	    /* If no actual unit found, fall down to unit view display. */
	    if (unit != NULL) {
		xform(mapw, x, y, &sx, &sy);
		/* Adjust to unit part of cell. */
		sx += (mapw->vp->hw - uw) / 2;  sy += (mapw->vp->hh - uh) / 2;
		if (unit->occupant != NULL
		    && uw >= 8
		    && (side_controls_unit(dside, unit)
			|| mapw->map->see_all
			|| u_see_occupants(unit->type))) {
		    /* Draw a "grouping box", in white, but with no occs
		       actually drawn. */
		    XSetClipMask(dpy, gc, None);
		    XSetLineAttributes(dpy, gc, 0, LineSolid, CapButt,
				       JoinMiter);
		    XSetForeground(dpy, gc, dside->ui->whitecolor->pixel);
		    XFillRectangle(dpy, mapw->d, gc,
				   sx + 1, sy + 1, uw - 2, uh - 2);
		    /* Put a black border around it, for better
                           contrast. */
		    XSetForeground(dpy, gc, dside->ui->blackcolor->pixel);
		    XDrawRectangle(dpy, mapw->d, gc,
				   sx + 1, sy + 1, uw - 2, uh - 2);
		}
		if (unit == mapw->map->curunit)
		  draw_current(mapw);
		draw_unit_image(mapw, sx, sy, uw, uh, unit->type,
				side_number(unit->side), !completed(unit));
		/* Indicate that other units are stacked here also. */
		if (unit->nexthere != NULL && uw > 8) {
		    osx = sx + uw / 2 - 6;  osy = sy + uh - 2;
		    /* (should be able to do with one fill?) */
		    XSetClipMask(dpy, gc, None);
		    XSetClipOrigin(dpy, gc, osx, osy - 1);
		    XSetForeground(dpy, gc, dside->ui->whitecolor->pixel);
		    XFillRectangle(dpy, mapw->d, gc, osx, osy - 1, 12, 4);
		    XSetForeground(dpy, gc, dside->ui->blackcolor->pixel);
		    XSetClipMask(dpy, gc, dside->ui->dots);
		    XFillRectangle(dpy, mapw->d, gc, osx, osy - 1, 12, 4);
		}
		if (mapw->vp->draw_names)
		  draw_unit_name(mapw, unit, sx, sy, uw, uh);
		return;
	    }
	} else {
	    /* At 32x32 and up, we can display several units in the stack. */
	    didone = FALSE;
	    for_all_stack(x, y, unit) {
		if (unit_actually_visible(dside, unit)) {
		    x_xform_unit(mapw, unit, &sx, &sy, &sw, &sh);
		    draw_unit_and_occs(mapw, unit, sx, sy, sw, sh, TRUE);
		    didone = TRUE;
		}
	    }
	    if (didone)
	      return;
	}
    }
    /* Do this if we didn't get to look at any actual units. */
    if ((uview = unit_view(dside, xw, y)) != EMPTY) {
	u = vtype(uview);  s = vside(uview);
	xform(mapw, x, y, &sx, &sy);
	sx += (mapw->vp->hw - uw) / 2;  sy += (mapw->vp->hh - uh) / 2;
	draw_unit_image(mapw, sx, sy, uw, uh, u, s, 0);
    }
}

/* This is a recursive routine that can display occupants and
   suboccupants. */

static void
draw_unit_and_occs(MapW *mapw, Unit *unit, int sx, int sy, int sw, int sh,
		   int drawoccs)
{
    int u = unit->type, s = side_number(unit->side), sx2, sy2, sw2, sh2;
    Unit *occ;
    Display *dpy = mapw->display;
    GC gc = mapw->gc;

    /* If an occupant's side is the same as its transport's, then there's
       really no need to draw its side emblem, since the transport's emblem
       will also be visible. */
    if (unit->transport && unit->side == unit->transport->side)
      s = -1;
    if (unit->occupant == NULL
	|| sw <= 8
	|| !(mapw->map->see_all
	     || unit->side == dside
	     || u_see_occupants(unit->type)
	     || side_owns_occupant(dside, unit))) {
	if (unit == mapw->map->curunit)
	  draw_current(mapw);
	draw_unit_image(mapw, sx, sy, sw, sh, u, s, !completed(unit));
	if (mapw->vp->draw_names)
	  draw_unit_name(mapw, unit, sx, sy, sw, sh);
    } else {
	x_xform_occupant(mapw, unit, unit, sx, sy, sw, sh,
			 &sx2, &sy2, &sw2, &sh2);
	/* Draw a white box to indicate the grouping. */
	XSetClipMask(dpy, gc, None);
	XSetLineAttributes(dpy, gc, 0, LineSolid, CapButt, JoinMiter);
	XSetForeground(dpy, gc, dside->ui->whitecolor->pixel);
	XFillRectangle(dpy, mapw->d, gc, sx + 1, sy + 1, sw - 2, sh - 2);
	/* Put a black border around it, for better contrast. */
	XSetForeground(dpy, gc, dside->ui->blackcolor->pixel);
	XDrawRectangle(dpy, mapw->d, gc, sx + 1, sy + 1, sw - 2, sh - 2);
	if (unit == mapw->map->curunit)
	  draw_current(mapw);
	/* Draw the transport's image. */
	draw_unit_image(mapw, sx2, sy2, sw2, sh2, u, s,	!completed(unit));
	if (mapw->vp->draw_names)
	  draw_unit_name(mapw, unit, sx2, sy2, sw2, sh2); 
	/* Maybe draw all of its occupants, recursively. */
	if (drawoccs) {
	    for_all_occupants(unit, occ) {
		x_xform_occupant(mapw, unit, occ, sx, sy, sw, sh,
				 &sx2, &sy2, &sw2, &sh2);
		draw_unit_and_occs(mapw, occ, sx2, sy2, sw2, sh2, TRUE);
	    }
	}
    }
}

static void
draw_unit_name(MapW *mapw, Unit *unit, int sx, int sy, int sw, int sh)
{
    if (!empty_string(unit->name)) {
	draw_legend_text(mapw, sx + sw, sy + sh / 2 + 5,
			 mapw->vp->power, unit->name,
			 dside->ui->unit_name_color, TRUE);
    }
}


/* Indicate what kind of people are living in the given cell. */

static void
draw_people(MapW *mapw, int x, int y)
{
    int xw, xw1, pop, sx, sy, sw, sh, ex, ey, ew, eh, dir, x1, y1, pop1;
    int	con, con1, cbitmask1, cbitmask2, pbitmask1, pbitmask2, drawemblemhere;

    xw = wrapx(x);
    if (!terrain_visible(dside, xw, y))
      return;
    pop = (people_sides_defined() ? people_side_at(xw, y) : NOBODY);
    con = (control_sides_defined() ? control_side_at(xw, y) : NOCONTROL);
    cbitmask1 = cbitmask2 = pbitmask1 = pbitmask2 = 0;
    drawemblemhere = FALSE;
    /* Decide which edges are borders of the country. */
    for_all_directions(dir) {
	if (point_in_dir(x, y, dir, &x1, &y1)) {
	    xw1 = wrapx(x1);
	    if (inside_area(xw1, y1) && terrain_visible(dside, xw1, y1)) {
		pop1 = (people_sides_defined() ? people_side_at(xw1, y1) : NOBODY);
		con1 = (control_sides_defined() ? control_side_at(xw1, y1) : NOCONTROL);
		if (mapw->vp->draw_control && con != con1) {
		    cbitmask1 |= 1 << dir;
		    if (con != NOCONTROL && con1 != NOCONTROL) {
			cbitmask2 |= 1 << dir;
		    }
		    drawemblemhere = TRUE;
		} else if (mapw->vp->draw_people && pop != pop1) {
		    pbitmask1 |= 1 << dir;
		    if (pop != NOBODY && pop1 != NOBODY) {
			pbitmask2 |= 1 << dir;
		    }
		    drawemblemhere = TRUE;
		}
	    } else {
		/* Draw just people in the cells right at the edge of
                   the known world. */
		drawemblemhere = TRUE;
	    }
	}
    }
    if (drawemblemhere) {
	xform(mapw, x, y, &sx, &sy);
	for_all_directions(dir) {
	    int mask = 1 << dir;

	    if (cbitmask1 & mask) {
		draw_country_border_line(mapw, sx, sy, dir, TRUE,
					 (cbitmask2 & mask));
	    } else if (pbitmask1 & mask) {
		draw_country_border_line(mapw, sx, sy, dir, FALSE,
					 (pbitmask2 & mask));
	    }
	}
	/* Draw an emblem for the people in the cell. */
	if (mapw->vp->draw_people && pop != NOBODY) {
	    sw = mapw->vp->hw;  sh = mapw->vp->hh;
	    ew = min(sw, max(8, sw / 2));  eh = min(sh, max(8, sh / 2));
	    ex = sx + sw / 2 - ew / 2;  ey = sy + sh / 2 - eh / 2;
	    /* Maybe offset emblem so it will be visible underneath
	       control emblem. */
	    if (mapw->vp->draw_control && con != pop) {
		ex += ew / 4;  ey += eh / 4;
	    }
	    draw_side_emblem(mapw, ex, ey, ew, eh, pop);
	}
	if (mapw->vp->draw_control && con != NOCONTROL) {
	    sw = mapw->vp->hw;  sh = mapw->vp->hh;
	    ew = min(sw, max(8, sw / 2));  eh = min(sh, max(8, sh / 2));
	    ex = sx + sw / 2 - ew / 2;  ey = sy + sh / 2 - eh / 2;
	    if (mapw->vp->draw_people && con != pop) {
		ex -= ew / 4;  ey -= eh / 4;
	    }
	    draw_side_emblem(mapw, ex, ey, ew, eh, con);
	}
    }
}

static void
draw_borders(MapW *mapw, int vx, int vy, int vw, int y1, int y2, int b)
{
    int x, y, x1, len, dir, bitmask, sx, sy, x3, y3;
    int halfside;
    Image *timg;
    Display *dpy = mapw->display;
    GC gc = mapw->gc;
    XColor *color;
    int power = mapw->vp->power;
    int wid = bwid[power], wid2, sx1, sy1, sx2, sy2;
	
    wid2 = wid / 2;
    if (power == 6)
      halfside = 26;
    else if (power == 5)
      halfside = 13;
    else
      halfside = 7;
    XSetLineAttributes(dpy, gc, bwid[power], LineSolid, CapButt, JoinMiter); 
    color = dside->ui->cellcolor[b];
    if (color == NULL)
      color = dside->ui->blackcolor;
    XSetForeground(dpy, gc, color->pixel);
    XSetBackground(dpy, gc, dside->ui->whitecolor->pixel);
    timg = best_image(dside->ui->timages[b], 1, 1);
    set_terrain_gc_for_image(mapw, gc, timg);
    if (!mapw->draw_lines && between(4, mapw->vp->power, 6)) {
	XSetClipMask(dpy, gc, bordpics[power]);
    } else {
	XSetClipMask(dpy, gc, None);
    }
    for (y = y1; y >= y2; --y) {
	if (!compute_x1_len(mapw, vx, vy, vw, y, &x1, &len))
	  continue;
	for (x = x1; x < x1 + len; ++x) {
	    if (!mapw->draw_lines && between(4, power, 6)) {
		xform(mapw, x, y, &sx, &sy);
		/* Draw the junction at the top of the x,y hex. */
		bitmask = 0;
		if (point_in_dir(x, y, NORTHEAST, &x3, &y3)
		    && border_at(x3, y3, WEST, b)
		    && seen_border(dside, x3, y3, WEST))
		  bitmask |= 1;
		if (border_at(x, y, NORTHEAST, b)
		    && seen_border(dside, x, y, NORTHEAST))
		  bitmask |= 2;
		if (border_at(x, y, NORTHWEST, b)
		    && seen_border(dside, x, y, NORTHWEST))
		  bitmask |= 4;
		if (bitmask != 0) {
		    XSetClipOrigin(dpy, gc,
				   sx - (bitmask & 3) * mapw->vp->hw,
				   sy - halfside
				   - ((bitmask & 4) ? 3 : 1) * mapw->vp->hch);
		    XFillRectangle(dpy, mapw->d, gc,
				   sx, sy - halfside,
				   mapw->vp->hw, mapw->vp->hch);
		}
		/* Draw the junction at the top left corner of the x,y hex. */
		bitmask = 0;
		if (border_at(x, y, WEST, b)
		    && seen_border(dside, x, y, WEST))
		  bitmask |= 1;
		if (border_at(x, y, NORTHWEST, b)
		    && seen_border(dside, x, y, NORTHWEST))
		  bitmask |= 2;
		if (point_in_dir(x, y, WEST, &x3, &y3)
		    && border_at(x3, y3, NORTHEAST, b)
		    && seen_border(dside, x3, y3, NORTHEAST))
		  bitmask |= 4;
		if (bitmask != 0) {
		    XSetClipOrigin(dpy, gc,
				   sx - (bitmask & 3) * mapw->vp->hw - mapw->vp-> hw / 2,
				   sy - halfside
				   - ((bitmask & 4) ? 2 : 0) * mapw->vp->hch);
		    XFillRectangle(dpy, mapw->d, gc,
				   sx - mapw->vp->hw / 2, sy - halfside,
				   mapw->vp->hw, mapw->vp->hch);
		}
	    } else {
		if (!terrain_visible(dside, wrapx(x), y)
		    || !any_borders_at(x, y, b))
		  continue;
		bitmask = 0;
		for_all_directions(dir) {
		    if (border_at(x, y, dir, b)
			&& seen_border(dside, x, y, dir)) {
			bitmask |= 1 << dir;
		    }
		}
		if (bitmask != 0) {
		    xform(mapw, x, y, &sx, &sy);
		    for_all_directions(dir) {
			if (bitmask & (1 << dir)) {
			    sx1 = bsx[power][dir];  sy1 = bsy[power][dir];
			    sx2 = bsx[power][dir+1];  sy2 = bsy[power][dir+1];
			    XDrawLine(dpy, mapw->d, gc,
				      sx + sx1 - wid2, sy + sy1 - wid2,
				      sx + sx2 - wid2, sy + sy2 - wid2);
			}
		    }
		}
	    }
	}
    }
    XSetFillStyle(dpy, gc, FillSolid);
}

/* Draw all the connections of the given type that are visible. */

static void
draw_connections(MapW *mapw, int vx, int vy, int vw, int y1, int y2, int c)
{
    int x, y, x1, len, dir, bitmask, sx, sy;
    int power = mapw->vp->power;
    int wid = cwid[power], wid2, cx = hws[power] / 2, cy = hhs[power] / 2;
    Image *timg;
    Display *dpy = mapw->display;
    GC gc = mapw->gc;
    XColor *color;

    wid2 = wid / 2;
    XSetLineAttributes(dpy, gc, wid, LineSolid, CapButt, JoinMiter); 
    color = dside->ui->cellcolor[c];
    if (color < 0)
      color = dside->ui->blackcolor;
    XSetForeground(dpy, gc, color->pixel);
    XSetBackground(dpy, gc, dside->ui->whitecolor->pixel);
    timg = best_image(dside->ui->timages[c], wid, wid);
    set_terrain_gc_for_image(mapw, gc, timg);
    if (!mapw->draw_lines && (power == 4 || power == 5)) {
	XSetClipMask(dpy, gc, connpics[power]);
    } else {
	XSetClipMask(dpy, gc, None);
    }
    for (y = y1; y >= y2; --y) {
	if (!compute_x1_len(mapw, vx, vy, vw, y, &x1, &len))
	  continue;
	for (x = x1; x < x1 + len; ++x) {
	    if (!terrain_visible(dside, wrapx(x), y)
		|| !tt_drawable(c, terrain_at(wrapx(x), y)))
	      continue;
	    bitmask = 0;
	    for_all_directions(dir) {
		if (connection_at(x, y, dir, c))
		  bitmask |= (1 << dir);
	    }
	    if (bitmask != 0) {
		xform(mapw, x, y, &sx, &sy);
		if (!mapw->draw_lines && (power == 4 || power == 5)) {
		    XSetClipOrigin(dpy, gc,
				   sx - 2 - (bitmask & 7) * (mapw->vp->hw + 2),
				   sy - 2 - (bitmask >> 3) * (mapw->vp->hh + 2));
		    XFillRectangle(dpy, mapw->d, gc, sx, sy,
				   mapw->vp->hw, mapw->vp->hh);
		} else {
		    for_all_directions(dir) {
			if (bitmask & (1 << dir)) {
			    XDrawLine(dpy, mapw->d, gc,
				      sx + cx - wid2, sy + cy - wid2,
				      sx + cx + lsx[power][dir] - wid2,
				      sy + cy + lsy[power][dir] - wid2);
			}
		    }
		}
	    }
	}
    }
    XSetFillStyle(dpy, gc, FillSolid);
}

/* Draw any text that should be associated with this cell. */

/* (could precompute what the string will lap over and move or truncate str),
   should be deterministic for each mag, so redraw doesn't scramble */

/* do features, label at a cell with nothing else, and declared as the
   feature's "center" */

/* Black/white text should be consistent for each period, use mask only
   to fix difficulties. */

static void
draw_legend(MapW *mapw, int x, int y)
{
    int sx, sy, pixlen;
    char *featname;
    Feature *feature;

    if (!inside_area(x, y))
      return;
    if (!terrain_visible(dside, wrapx(x), y))
      return;
    /* feature object should specify legend's position */
    feature = feature_at(x, y);
    if (feature != NULL) {
	if (feature->size == 1) {
	    featname = feature_name_at(x, y);
	    if (featname != NULL) {
		pixlen = strlen(featname) * 8;
		xform(mapw, x, y, &sx, &sy);
		draw_legend_text(mapw,
				 sx + 1 + (pixlen > mapw->vp->hw ? 2 :
					   (mapw->vp->hw/2 - pixlen / 2)),
				 sy - 6 + mapw->vp->hh / 2,
				 mapw->vp->power, featname,
				 dside->ui->fgcolor, TRUE);
	    }
	}
    }
}

/* Cursor drawing also draws the unit in some other color if it's not the
   "top-level" unit in a cell. */

/* "color unders" here are not correct */

static void
draw_current(MapW *mapw)
{
    int sx, sy, sw, sh;
    int uw = mapw->vp->uw, uh = mapw->vp->uh;
    enum grayshade shade = black;
    Unit *unit = NULL;
    Display *dpy = mapw->display;
    GC gc = mapw->gc;
    XColor *color;

    unit = mapw->map->curunit;
    if (!in_play(unit))
      return;
    /* Compute the bounding box we're going to hilite. */
    if (mapw->vp->power >= 5) { /* not ideal test */
	x_xform_unit_self(mapw, unit, &sx, &sy, &sw, &sh);
    } else {
	xform(mapw, unit->x, unit->y, &sx, &sy);
	/* Adjust to unit part of cell. */
	sx += (mapw->vp->hw - uw) / 2;  sy += (mapw->vp->hh - uh) / 2;
	sw = uw;  sh = uh;
    }
    if (sw >= 16) {
	sx -= 1;  sy -= 1;
	sw += 2;  sh += 2;
    }
    /* Maybe redraw the unit that the cursor is showing. */
    if (unit->transport != NULL) {
	if (dside->ui->monochrome) {
	    draw_unit_image(mapw, sx, sy, sw, sh, unit->type, -1,
			    !completed(unit));
	} else {
	    /* Leave any underlying image alone, but (should) draw over it
	       in a different color. */
	    draw_unit_image(mapw, sx, sy, sw, sh, unit->type, -1,
			    !completed(unit));
	}
    }
    /* Draw the cursor icon proper. */
    XSetClipMask(dpy, gc, None);
    XSetLineAttributes(dpy, gc, 0, LineSolid, CapButt, JoinMiter); 
    if (mapw->map->autoselect) {
	XSetBackground(dpy, gc, dside->ui->blackcolor->pixel);
	XSetFillStyle(dpy, gc, FillOpaqueStippled);
	XSetStipple(dpy, gc, antpic);
	XSetTSOrigin(dpy, gc, mapw->map->anim_state % 8, 0);
    }
    XSetForeground(dpy, gc, dside->ui->whitecolor->pixel);
    if (sw >= 2 && sh >= 2) {
	XDrawRectangle(dpy, mapw->d, gc, sx, sy, sw - 1, sh - 1);
    } else {
	XFillRectangle(dpy, mapw->d, gc, sx, sy, sw, sh);
    }
    /* Reset the tile/stipple origin. */
    XSetTSOrigin(dpy, gc, 0, 0);
    if (sw >= 4 && sh >= 4) {
	XSetFillStyle(dpy, gc, FillSolid);
	/* Black is for units that can still act, dark gray for actors
	   that used all acp, gray if the unit can't do anything. */
	shade = gray;
	if (unit->act && unit->act->initacp > 0)
	  shade = ((unit->act->acp > 0) ? black : darkgray);
	XSetForeground(dpy, gc, dside->ui->blackcolor->pixel);
	if (shade != black) {
	    XSetFillStyle(dpy, gc, FillOpaqueStippled);
	    XSetStipple(dpy, gc, dside->ui->grays[shade]);
	    XSetForeground(dpy, gc, dside->ui->whitecolor->pixel);
	    XSetBackground(dpy, gc, dside->ui->blackcolor->pixel);
	}
	XDrawRectangle(dpy, mapw->d, gc, sx + 1, sy + 1, sw - 3, sh - 3);
	/* If the box is not too small, thicken the inner rectangle. */
	if (sw >= 8 && sh >= 8
	    && (unit ?
		!(unit->plan && (unit->plan->asleep || unit->plan->reserve))
		: TRUE)) {
#if 0 /* ? */
	    if (sw > 8 && sh > 8 && unit == NULL)
	      XSetForeground(dpy, gc, dside->ui->whitecolor->pixel);
#endif
	    XDrawRectangle(dpy, mapw->d, gc, sx + 2, sy + 2, sw - 5, sh - 5);
	}
    }
    XSetFillStyle(dpy, gc, FillSolid);
}

draw_blast(Map *map, Unit *unit, int blasttype)
{
    int sx, sy, sw, sh;
    MapW *mapw = (MapW *) map->widget;

    x_xform_unit(mapw, unit, &sx, &sy, &sw, &sh);
    mapw->blastsx = sx;  mapw->blastsy = sy;
    mapw->blastsw = sw;  mapw->blastsh = sh;
    mapw->blasttype = blasttype;
    update_at_unit(map, unit);
    Tcl_Sleep(400);
    mapw->blasttype = -1;
    update_at_unit(map, unit);
}

/*void*/
draw_blast_image(MapW *mapw, int sx, int sy, int sw, int sh, int blasttype)
{
    int sx2, sy2;
    Display *dpy = mapw->display;
    GC gc = mapw->gc;

    XSetClipMask(dpy, gc, None);
    if (blasttype == 3) {
	XSetForeground(dpy, gc, dside->ui->badcolor->pixel);
	XFillRectangle(dpy, mapw->d, gc, sx - 100, sy - 100, 200, 200);
    } else if (sw >= 16 && sh >= 16) {
	sx2 = sx + (sw - 16) / 2;  sy2 = sy + (sh - 16) / 2;
	/* Draw a red splat with a black offset shadow. */
	XSetClipMask(dpy, gc, dside->ui->hitpics[blasttype]);
	XSetClipOrigin(dpy, gc, sx2 + 1, sy2 + 1);
	XSetForeground(dpy, gc, dside->ui->blackcolor->pixel);
	XFillRectangle(dpy, mapw->d, gc, sx2 + 1, sy2 + 1, 16, 16);
	XSetClipOrigin(dpy, gc, sx2, sy2);
	XSetForeground(dpy, gc, dside->ui->badcolor->pixel);
	XFillRectangle(dpy, mapw->d, gc, sx2, sy2, 16, 16);
    } else if (sw >= 4 && sh >= 4) {
	/* Draw a red box with a black shadow. */
	XSetForeground(dpy, gc, dside->ui->blackcolor->pixel);
	XFillRectangle(dpy, mapw->d, gc, sx + 1, sy + 1, sw - 1, sh - 1);
	XSetForeground(dpy, gc, dside->ui->badcolor->pixel);
	XFillRectangle(dpy, mapw->d, gc, sx, sy, sw - 1, sh - 1);
    } else {
	/* Draw a red box only. */
	XSetForeground(dpy, gc, dside->ui->badcolor->pixel);
	XFillRectangle(dpy, mapw->d, gc, sx, sy, sw, sh);
    }
}

static void
draw_feature_boundary(MapW *mapw, int x, int y, int fid)
{
    int wid, p, wid2, dir, fid0, x1, y1, sx, sy;
    Display *dpy = mapw->display;
    GC gc = mapw->gc;
    Pixmap graylev;
    XColor *color;

    if (!terrain_visible(dside, wrapx(x), y))
      return;
    p = mapw->vp->power;
    wid = bwid[p];
    fid0 = raw_feature_at(x, y);
    if (fid0 == 0)
      return;
    xform(mapw, x, y, &sx, &sy);
    if (wid == 0)
      return;
    wid2 = wid / 2;

    XSetClipMask(dpy, gc, None);
    XSetLineAttributes(dpy, gc, bwid[p], LineSolid, CapButt, JoinMiter); 
    color = dside->ui->graycolor;
    /* for now: */
    if (dside->designer && fid0 == dside->ui->curfid)
      color = dside->ui->badcolor;
    XSetForeground(dpy, gc, color->pixel);
    XSetFillStyle(dpy, gc, FillSolid);
    if (dside->ui->monochrome) {
	graylev = dside->ui->grays[gray];
	if (dside->designer && fid0 == dside->ui->curfid)
	  graylev = dside->ui->grays[darkgray];
	XSetFillStyle(dpy, gc, FillStippled);
	XSetStipple(dpy, gc, graylev);
    }
    for_all_directions(dir) {
	if (point_in_dir(x, y, dir, &x1, &y1)) {
	    if (raw_feature_at(x1, y1) != fid0
		&& terrain_visible(dside, x1, y1)) {
		XDrawLine(dpy, mapw->d, gc,
			  sx + qx[p][dir], sy + qy[p][dir],
			  sx + qx[p][dir+1], sy + qy[p][dir+1]);
	    }
	}
    }
}

/* Do the grody work of drawing very large polygons accurately. */

static void
draw_hex_polygon(mapw, gc, sx, sy, power, over, dogrid)
MapW *mapw;
GC gc;
int sx, sy, power, over, dogrid;
{
    XPoint points[6];
    int hw = hws[power], hh = hhs[power], delt = (hhs[power] - hcs[power]);
    int ew = (dogrid ? 1 : 0);
    enum grayshade shade;
    Display *dpy = mapw->display;

    points[0].x = sx + hw / 2;         points[0].y = sy;
    points[1].x = sx + hw - ew;        points[1].y = sy + delt /*- ew*/;
    points[2].x = sx + hw - ew;        points[2].y = sy + (hh - delt - ew);
    points[3].x = sx + hw / 2;         points[3].y = sy + (hh - ew);
    points[4].x = sx;                  points[4].y = sy + (hh - delt - ew);
    points[5].x = sx;                  points[5].y = sy + delt;
    XFillPolygon(dpy, mapw->d, gc, points, 6, Convex, CoordModeOrigin);
    if (over < 0) {
	shade = gray;
	if (over == -2)
	  shade = darkgray;
	XSetClipMask(dpy, gc, None);
	XSetFillStyle(dpy, gc, FillStippled);
	XSetStipple(dpy, gc, dside->ui->grays[shade]);
	XSetForeground(dpy, gc, dside->ui->blackcolor->pixel);
	XFillPolygon(dpy, mapw->d, gc, points, 6, Convex, CoordModeOrigin);
    }
}

/* Map legends are stencils usually. */

static void
draw_legend_text(mapw, sx, sy, power, str, color, maskit)
MapW *mapw;
int sx, sy, power, maskit;
char *str;
XColor *color;
{
    Display *dpy = mapw->display;
    GC gc = mapw->gc;

    XSetClipMask(dpy, gc, None);
    XSetForeground(dpy, gc, color->pixel);
#if 0
    sy += (dside->ui->ulegendfonts[power][0])->max_bounds.ascent;
    XSetFont(dside->ui->dpy, dside->ui->ltextgc,
	     (dside->ui->ulegendfonts[power][0])->fid);
#endif
    if (maskit) {
#if 0
	XSetBackground(dside->ui->dpy, dside->ui->ltextgc, 
		       (color == dside->ui->bgcolor ? dside->ui->fgcolor :
			dside->ui->bgcolor));
	XDrawImageString(dside->ui->dpy, win, dside->ui->ltextgc,
			 sx, sy, str, strlen(str));
#endif
	XSetForeground(dpy, gc, dside->ui->whitecolor->pixel);
	Tk_DrawChars(dpy, mapw->d, gc, mapw->main_font, str, strlen(str),
		     sx-1, sy);
	Tk_DrawChars(dpy, mapw->d, gc, mapw->main_font, str, strlen(str),
		     sx+1, sy);
	Tk_DrawChars(dpy, mapw->d, gc, mapw->main_font, str, strlen(str),
		     sx, sy-1);
	Tk_DrawChars(dpy, mapw->d, gc, mapw->main_font, str, strlen(str),
		     sx, sy+1);
	XSetForeground(dpy, gc, dside->ui->blackcolor->pixel);
	Tk_DrawChars(dpy, mapw->d, gc, mapw->main_font, str, strlen(str),
		     sx, sy);
    } else {
	Tk_DrawChars(dpy, mapw->d, gc, mapw->main_font, str, strlen(str),
		     sx, sy);
    }
}

/* Put a unit image in the widget's window. */

void
draw_unit_image(mapw, sx, sy, sw, sh, u, s2, mod)
MapW *mapw;
int sx, sy, sw, sh, u, s2, mod;
{
    char buf[1], *ename;
    int sx2, sy2, ex, ey, ew, eh, desperate = FALSE;
    XColor *imagecolor = dside->ui->blackcolor;
    XColor *maskcolor = dside->ui->whitecolor;
    Image *uimg;
    TkImage *tkimg;
    Display *dpy = mapw->display;
    GC gc = mapw->gc;

    /* Filter out very small images. */
    if (sw <= 0)
      return;
    XSetClipMask(dpy, gc, None);
    XSetLineAttributes(dpy, gc, 0, LineSolid, CapButt, JoinMiter);
    /* Just draw a small box at 4x4 and below. */
    if (sw <= 4) {
	/* (should draw in side color if possible) */
	XSetForeground(dpy, gc, imagecolor->pixel);
	XFillRectangle(dpy, mapw->d, gc, sx, sy, sw, sh);
	return;
    }
    uimg = best_image(dside->ui->uimages[u], sw, sh);
    if (uimg != NULL) {
	/* Offset the image to draw in the middle of its area,
	   whether larger or smaller than the given area. */
	sx2 = sx + (sw - uimg->w) / 2;  sy2 = sy + (sh - uimg->h) / 2;
	/* Only change the size of the rectangle being drawn if it's
	   smaller than what was passed in. */
	if (uimg->w < sw) {
	    sx = sx2;
	    sw = uimg->w;
	}
	if (uimg->h < sh) {
	    sy = sy2;
	    sh = uimg->h;
	}
	/* Figure out what colors to use. */
	if (mapw->map->colorize_units && dside->ui->numcolors[s2] > 0)
	  imagecolor = dside->ui->colors[s2][0];
	if (mapw->map->colorize_units && dside->ui->numcolors[s2] > 1)
	  maskcolor = dside->ui->colors[s2][1];
	tkimg = (TkImage *) uimg->hook;
	if (tkimg != NULL) {
	    if (!dside->ui->monochrome
		&& mapw->map->unit_style == 0
		&& tkimg->colr != None) {
		if (tkimg->mask != None) {
		    /* set the clip mask */
		    XSetClipOrigin(dpy, gc, sx2, sy2);
		    XSetClipMask(dpy, gc, tkimg->mask);
		}
		/* Draw the color image. */
		XCopyArea(dpy, tkimg->colr, mapw->d, gc, 0, 0, sw, sh, sx, sy);
	    } else if (tkimg->mono != None || tkimg->mask != None) {
		/* Set the origin for any subsequent clipping. */
		XSetClipOrigin(dpy, gc, sx2, sy2);
		/* Set the color we're going to use for the mask; use
		   the imagecolor if we'll be using the mask as the
		   only image. */
		XSetForeground(dpy, gc,
			       (tkimg->mono == None ? imagecolor : maskcolor)->pixel);
		/* Set the clip mask to be explicit mask or unit's image. */
		if (tkimg->mask)
		  XSetClipMask(dpy, gc, tkimg->mask);
		else
		  XSetClipMask(dpy, gc, tkimg->mono);
		/* Draw the mask. */
		XFillRectangle(dpy, mapw->d, gc, sx, sy, sw, sh);
		/* Draw the image proper. */
		if (tkimg->mono != None) {
		    XSetForeground(dpy, gc, imagecolor->pixel);
		    XSetClipMask(dpy, gc, tkimg->mono);
		    XFillRectangle(dpy, mapw->d, gc, sx, sy, sw, sh);
		}
	    }
	} else {
	    desperate = TRUE;
	}
    } else {
	desperate = TRUE;
    }
    if (desperate) {
	/* If all else fails, draw a black box and maybe a white border. */
	XSetForeground(dpy, gc, dside->ui->blackcolor->pixel);
	XFillRectangle(dpy, mapw->d, gc, sx, sy, sw, sh);
	if (sw >= 16) {
	    XSetForeground(dpy, gc, dside->ui->whitecolor->pixel);
	    XDrawRectangle(dpy, mapw->d, gc, sx, sy, sw, sh);
	}
    }
    /* If modification was requested, add a 50% white stipple over the
       image. */
    if (mod != 0) {
	XSetFillStyle(dpy, gc, FillStippled);
	XSetStipple(dpy, gc, dside->ui->grays[gray]);
	XSetClipMask(dpy, gc, None);
	XSetForeground(dpy, gc, dside->ui->whitecolor->pixel);
	XFillRectangle(dpy, mapw->d, gc, sx, sy, sw, sh);
	XSetFillStyle(dpy, gc, FillSolid);
    }
    /* Draw a side emblem if one was requested. */
    if (between(0, s2, numsides)) {
	ename = (side_n(s2) ? side_n(s2)->emblemname : NULL);
	if (emblem_position(uimg, ename, NULL /* eimg */, sw, sh,
			    &ex, &ey, &ew, &eh)) {
	    draw_side_emblem(mapw, sx + ex, sy + ey, ew, eh, s2);
	}
    }
}

/* Draw an emblem identifying the given side.  If a side does not have a
   distinguishing emblem, fall back on some defaults. */

void
draw_side_emblem(mapw, ex, ey, ew, eh, s2)
MapW *mapw;
int ex, ey, ew, eh, s2;
{
    int ex2, ey2, ew2, eh2, offset;
    XColor *imagecolor, *maskcolor;
    Image *eimg;
    TkImage *tkimg;
    Display *dpy = mapw->display;
    GC gc = mapw->gc;

    /* Draw the emblem's mask, or else an enclosing box. */
    eimg = best_image(dside->ui->eimages[s2], ew, eh);
    if (eimg != NULL) {
	/* Offset the image to draw in the middle of its area,
	   whether larger or smaller than the given area. */
	ex2 = ex + (ew - eimg->w) / 2;  ey2 = ey + (eh - eimg->h) / 2;
	/* Only change the size of the rectangle being drawn if it's
	   smaller than what was passed in. */
	if (eimg->w < ew) {
	    ex = ex2;
	    ew = eimg->w;
	}
	if (eimg->h < eh) {
	    ey = ey2;
	    eh = eimg->h;
	}
	tkimg = (TkImage *) eimg->hook;
	/* Decide on the colors to use with the emblem. */
	if (dside->ui->numcolors[s2] > 0) {
	    imagecolor = dside->ui->colors[s2][0];
	} else {
	    imagecolor = dside->ui->blackcolor;
	}
	if (dside->ui->numcolors[s2] > 1) {
	    maskcolor = dside->ui->colors[s2][1];
	} else {
	    maskcolor = dside->ui->whitecolor;
	}
	/* Draw the mask. */
	XSetForeground(dpy, gc, maskcolor->pixel);
	XSetClipOrigin(dpy, gc, ex, ey);
	if (tkimg != NULL && tkimg->mask != None) {
	    XSetClipMask(dpy, gc, tkimg->mask);
	    XFillRectangle(dpy, mapw->d, gc, ex, ey, ew, eh);
	} else {
	    XSetClipMask(dpy, gc, None);
	    XFillRectangle(dpy, mapw->d, gc, ex - 1, ey, ew + 1, eh + 1);
	}
	/* Now draw the emblem proper. */
	/* (should fix so some color emblems can be used with mono displays) */
	if (!dside->ui->monochrome
	    && dside->ui->dflt_color_embl_images
	    && tkimg != NULL
	    && tkimg->colr != None) {
	    /* Draw the color image. */
	    XCopyArea(dpy, tkimg->colr, mapw->d, gc, 0, 0, ew, eh, ex, ey);
	} else {
	    XSetForeground(dpy, gc, imagecolor->pixel);
	    XSetClipMask(dpy, gc, (tkimg != NULL ? tkimg->mono : None));
	    XFillRectangle(dpy, mapw->d, gc, ex, ey, ew, eh);
	}
    }
}

static void
draw_country_border_line(mapw, sx, sy, dir, con, heavy)
MapW *mapw;
int sx, sy, dir, con, heavy;
{
    int pow = mapw->vp->power;
    int wid = bwid[pow];
    Display *dpy = mapw->display;
    GC gc = mapw->gc;

    if (wid == 0)
      return;
    wid = max(1, wid / 2);
    if (!heavy)
      wid = max(1, wid / 2);
    XSetClipMask(dpy, gc, None);
    XSetLineAttributes(dpy, gc, wid, LineSolid, CapButt, JoinMiter);
    if (con)
      XSetForeground(dpy, gc, dside->ui->frontline_color->pixel);
    else
      XSetForeground(dpy, gc, dside->ui->country_border_color->pixel);
    if (!heavy) {
	/* Lighten the color by stippling with white. */
	XSetBackground(dpy, gc, dside->ui->whitecolor->pixel);
	XSetFillStyle(dpy, gc, FillOpaqueStippled);
	XSetStipple(dpy, gc, dside->ui->grays[gray]);
    }
    XDrawLine(dpy, mapw->d, gc,
	      sx + bsx[pow][dir], sy + bsy[pow][dir],
	      sx + bsx[pow][dir+1], sy + bsy[pow][dir+1]);
    XSetFillStyle(dpy, gc, FillSolid);
}

/* Put the point x, y in the center of the mapw, or at least as close
   as possible. */

void
recenter(side, map, x, y)
Side *side;
Map *map;
int x, y;
{
    int oldsx, oldsy;
    VP *vp;

    if (map->widget == NULL)
      return;
    vp = widget_vp(map);
    oldsx = vp->sx;  oldsy = vp->sy;
    set_view_focus(vp, x, y);
    center_on_focus(vp);
    if (vp->sx != oldsx || vp->sy != oldsy) {
	redraw_map(map);
    }
}

/* Ensure that given location is visible on the front map.  We
   (should) also flush the input because any input relating to a
   different screen is probably worthless. */

void
put_on_screen(side, map, x, y)
Side *side;
Map *map;
int x, y;
{

    /* Ugly hack to prevent extra boxes being drawn during init - don't ask!*/
    if (x == 0 && y == 0)
      return;
    if (!in_middle(side, map, x, y))
      recenter(side, map, x, y);
}

/* Decide whether given location is not too close to edge of screen.
   We do this because it's a pain to move units when half the adjacent
   places aren't even visible.  This routine effectively places a
   lower limit of 5x5 for the map window. (I think) */

int
in_middle(side, map, x, y)
Side *side;
Map *map;
int x, y;
{
    int sx, sy, insetx1, insety1, insetx2, insety2;
    MapW *mapw = (MapW *) map->widget;

    if (mapw == NULL)
      return FALSE;

    xform(mapw, x, y, &sx, &sy);
    /* Adjust to be the center of the cell, more reasonable if large. */
    sx += mapw->vp->hw / 2;  sy += mapw->vp->hh / 2;
    insetx1 = min(mapw->vp->pxw / 4, 1 * mapw->vp->hw);
    insety1 = min(mapw->vp->pxh / 4, 1 * mapw->vp->hch);
    insetx2 = min(mapw->vp->pxw / 4, 2 * mapw->vp->hw);
    insety2 = min(mapw->vp->pxh / 4, 2 * mapw->vp->hch);
    if (sx < insetx2)
      return FALSE;
    if (sx > mapw->vp->pxw - insetx2)
      return FALSE;
    if (sy < (between(2, y, area.height-3) ? insety2 : insety1))
      return FALSE;
    if (sy > mapw->vp->pxh - (between(2, y, area.height-3) ? insety2 : insety1))
      return FALSE;
    return TRUE;
}

int mouse_is_down;

void
handle_mouse_down(Map *map, int sx, int sy, int button)
{
    int ax, ay;
    Unit *unit2;
    void (*fn)(Side *sidex, Map *mapx, int cancelledx);
    VP *vp = widget_vp(map);

    mouse_is_down = TRUE;
    if (map == NULL) {
	beep(dside);
	return;
    }
    if (!x_nearest_cell((MapW *) map->widget, sx, sy, &ax, &ay)) {
	beep(dside);
	return;
    }
    map->inpx = ax;  map->inpy = ay;
    x_nearest_unit((MapW *) map->widget, sx, sy, &(map->inpunit));
    /* Assume that last place clicked is a reasonable focus. */
    if (inside_area(ax, ay)) {
	set_view_focus(vp, ax, ay);
    }
#ifdef DESIGNERS
    if (dside->designer && dside->ui->curdesigntool != survey_mode) {
	handle_designer_mouse_down(dside, map, sx, sy);
	return;
    }
#endif /* DESIGNERS */
    if (map->modalhandler) {
	fn = map->modalhandler;
	map->modalhandler = NULL;
	move_look(map, sx, sy);
	(*fn)(dside, map, 0);
	return;
    }

    switch (map->mode) {
      case survey_mode:
	if (button == 1)
	  move_look(map, sx, sy);
	else if (button == 3) {
	    if (map->curunit && side_controls_unit(dside, map->curunit)) {
		move_the_selected_unit(map, map->curunit, sx, sy);
	    } else {
		beep(dside);
	    }
	}
	break;
      case move_mode:
	if (map->curunit && side_controls_unit(dside, map->curunit)) {
	    move_the_selected_unit(map, map->curunit, sx, sy);
	} else {
	    beep(dside);
	}
	break;
     default:
	/* error eventually */
	break;
    }
}

/*void*/
move_look(Map *map, int sx, int sy)
{
    int nx, ny;
    Unit *unit;
    MapW *mapw = (MapW *) map->widget;

    if (x_nearest_cell(mapw, sx, sy, &nx, &ny)) {
	if (inside_area(nx, ny)) {
	    x_nearest_unit(mapw, sx, sy, &unit);
	    if (unit != NULL && (!side_controls_unit(dside, unit)))
	      unit = NULL;
	    set_current_unit(map, unit);
	} else {
	    beep(dside);
	}
    }
}

/*void*/
move_the_selected_unit(Map *map, Unit *unit, int sx, int sy)
{
    int x, y;
    Unit *other = NULL;
    MapW *mapw = (MapW *) map->widget;

    x_nearest_cell(mapw, sx, sy, &x, &y);
    x_nearest_unit(mapw, sx, sy, &other);
    advance_into_cell(dside, unit, x, y, other);
}

void
handle_mouse_up(Map *map, int sx, int sy, int button)
{
    mouse_is_down = FALSE;
}

void
handle_world_mouse_down(Map *map, int sx, int sy, int button)
{
    int x, y;

    if (map == NULL) {
	beep(dside);
	return;
    }
    if (!x_nearest_cell((MapW *) map->worldw, sx, sy, &x, &y)) {
	beep(dside);
	return;
    }
    if (inside_area(x, y)) {
	put_on_screen(dside, map, x, y);
    }
}

void
handle_world_mouse_up(Map *map, int sx, int sy, int button)
{
}

VP *
widget_vp(Map *map)
{
    return ((MapW *) map->widget)->vp;
}

/* Force a refresh of the map widgets in a map. */

void
redraw_map(map)
Map *map;
{
    MapW *mapw;

    mapw = (MapW *) map->widget;
    if (mapw == NULL)
      return;
    if (!mapw->update_pending) {
	Tcl_DoWhenIdle(mapw_display, (ClientData) mapw);
	mapw->update_pending = 1;
    }
    mapw = (MapW *) map->worldw;
    if (mapw == NULL)
      return;
    if (!mapw->update_pending) {
	Tcl_DoWhenIdle(mapw_display, (ClientData) mapw);
	mapw->update_pending = 1;
    }
    eval_tcl_cmd("update idletasks");
}

set_tool_cursor(Side *side, Map *map)
{
    MapW *mapw = (MapW *) map->widget;
    Tk_Cursor cursor;

    if (mapw == NULL)
      return;
    cursor = side->ui->cursors[map->mode];
    if (cursor == None)
      cursor = side->ui->cursors[survey_mode];
    Tk_DefineCursor(mapw->tkwin, cursor);
}

int painttype;
int paintctrl;
int paintpeop;
int paintfid;
int painttview;
int paintuview;

handle_designer_mouse_down(Side *side, Map *map, int sx, int sy)
{
    int x, y, dir, oldt, oldctrl, oldpeop, oldfid;
    int oldtview, olduview;
    Unit *unit;

    nearest_boundary(widget_vp(map), sx, sy, &x, &y, &dir);
    switch (dside->ui->curdesigntool) {
      case cell_paint_mode:
	/* Choose to paint fg or bg type, depending on what's already there. */
	oldt = terrain_at(x, y);
	painttype = (dside->ui->curttype == oldt ? dside->ui->curbgttype : dside->ui->curttype);
	net_paint_cell(dside, x, y, dside->ui->curbrushradius, painttype);
	break;
      case unit_paint_mode:
	unit = net_designer_create_unit(dside, dside->ui->curutype,
					dside->ui->curusidenumber, x, y);
	break;
      case people_paint_mode:
	/* Paint people or clear, inverting from what is already here. */
	oldpeop = (people_sides_defined() ? people_side_at(x, y) : NOBODY);
	paintpeop = (dside->ui->curpeoplenumber == oldpeop ? NOBODY : dside->ui->curpeoplenumber);
	net_paint_people(dside, x, y, dside->ui->curbrushradius, paintpeop);
	break;
      case control_paint_mode:
	/* Paint control or clear it, inverting from what is already here. */
	oldctrl = (control_sides_defined() ? control_side_at(x, y) : NOCONTROL);
	paintctrl = (dside->ui->curcontrolnumber == oldctrl ? NOCONTROL : dside->ui->curcontrolnumber);
	net_paint_control(dside, x, y, dside->ui->curbrushradius, paintctrl);
	break;
      case feature_paint_mode:
	oldfid = (features_defined() ? raw_feature_at(x, y) : 0);
	paintfid = (dside->ui->curfid == oldfid ? 0 : dside->ui->curfid);
	net_paint_feature(dside, x, y, dside->ui->curbrushradius, paintfid);
	break;
      case view_paint_mode:
#if 0 /* define rest of vars */
	if (dside->terrview == NULL || dside->unitview == NULL)
	  break;
	oldtview = terrain_view(dside, x, y);
	painttview = (oldtview == UNSEEN ? buildtview(terrain_at(x, y)) : UNSEEN);
	olduview = unit_view(dside, x, y);
	paintuview = (curuview == olduview ? EMPTY : curuview);
	net_paint_view(dside, x, y, dside->ui->curbrushradius, painttview, paintuview);
	/* (should paint other view layers also) */
#endif
	break;
      default:
	break;
    }
}

paint_on_drag(Map *map, int sx, int sy)
{
    int x, y, brush;

    if (!mouse_is_down)
      return;
    if (x_nearest_cell((MapW *) map->widget, sx, sy, &x, &y)) {
	brush = dside->ui->curbrushradius;
	switch (dside->ui->curdesigntool) {
	  case cell_paint_mode:
	    net_paint_cell(dside, x, y, brush, painttype);
	    break;
	  case unit_paint_mode:
	    /* Drag-painting is almost never useful for unit placement. */
	    break;
	  case people_paint_mode:
	    net_paint_people(dside, x, y, brush, paintpeop);
	    break;
	  case control_paint_mode:
	    net_paint_control(dside, x, y, brush, paintctrl);
	    break;
	  case feature_paint_mode:
	    net_paint_feature(dside, x, y, brush, paintfid);
	    break;
	  default:
	    break;
	}
    }
}

update_at_cell(Map *map, int x, int y)
{
    int sx, sy;
    MapW *mapw;

    if (!in_area(x, y))
      return;
    mapw = (MapW *) map->widget;
    if (mapw->rsw >= 0) {
	printf("rsw is already %d\n", mapw->rsw);
	eval_tcl_cmd("update idletasks");
    }
    if (mapw->rsw >= 0) {
	printf("rsw is now %d\n", mapw->rsw);
    }
    xform(mapw, x, y, &sx, &sy);
    mapw->rsx = sx;  mapw->rsy = sy;
    mapw->rsw = mapw->vp->hw;  mapw->rsh = mapw->vp->hh;
    mapw = (MapW *) map->worldw;
    xform(mapw, x, y, &sx, &sy);
    mapw->rsx = sx;  mapw->rsy = sy;
    mapw->rsw = mapw->vp->hw;  mapw->rsh = mapw->vp->hh;
    redraw_map(map);
}

update_at_unit(Map *map, Unit *unit)
{
    int sx, sy;
    MapW *mapw;

    if (!in_area(unit->x, unit->y))
      return;
    mapw = (MapW *) map->widget;
    if (mapw->rsw >= 0) {
	printf("rsw is already %d\n", mapw->rsw);
	eval_tcl_cmd("update idletasks");
    }
    if (mapw->rsw >= 0) {
	printf("rsw is now %d\n", mapw->rsw);
    }
    xform(mapw, unit->x, unit->y, &sx, &sy);
    mapw->rsx = sx - 2;  mapw->rsy = sy - 2;
    mapw->rsw = mapw->vp->hw + 4;  mapw->rsh = mapw->vp->hh + 4;
    mapw = (MapW *) map->worldw;
    xform(mapw, unit->x, unit->y, &sx, &sy);
    mapw->rsx = sx - 2;  mapw->rsy = sy - 2;
    mapw->rsw = mapw->vp->hw + 4;  mapw->rsh = mapw->vp->hh + 4;
    redraw_map(map);
}

toggle_mapw_polygons(Map *map)
{
    MapW *mapw;

    mapw = (MapW *) map->widget;
    if (mapw)
      mapw->draw_polygons = !mapw->draw_polygons;
}

toggle_mapw_lines(Map *map)
{
    MapW *mapw;

    mapw = (MapW *) map->widget;
    if (mapw)
      mapw->draw_lines = !mapw->draw_lines;
}
