/***************************************************************************
 *   Copyright (C) 2005-2007 by Joachim Eibl                               *
 *   joachim.eibl at gmx.de                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA 02110-1301, USA.           *
 ***************************************************************************/

#include "smalldialogs.h"
#include "optiondialog.h"

#include <tqcombobox.h>
#include <tqcheckbox.h>
#include <tqlineedit.h>
#include <tqlabel.h>
#include <tqlayout.h>
#include <tqpushbutton.h>
#include <tqdragobject.h>
#include <tqregexp.h>
#include <tqtooltip.h>
#include <tqpopupmenu.h>
#include <tqcursor.h>


#include <tdefiledialog.h>
#include <tdelocale.h>

// OpenDialog **************************************************************

OpenDialog::OpenDialog(
   TQWidget* pParent, const TQString& n1, const TQString& n2, const TQString& n3,
   bool bMerge, const TQString& outputName, const char* slotConfigure, OptionDialog* pOptions )
: TQDialog( pParent, "OpenDialog", true /*modal*/ )
{
   m_pOptions = pOptions;

   TQVBoxLayout* v = new TQVBoxLayout( this, 5 );
   TQGridLayout* h = new TQGridLayout( v, 5, 4, 5 );
   h->setColStretch( 1, 10 );

   TQLabel* label  = new TQLabel( i18n("A (Base):"), this );

   m_pLineA = new TQComboBox( true, this );
   m_pLineA->insertStringList( m_pOptions->m_recentAFiles );
   m_pLineA->setEditText( KURL(n1).prettyURL() );
   m_pLineA->setMinimumSize( 200, m_pLineA->size().height() );
   TQPushButton * button = new TQPushButton( i18n("File..."), this );
   connect( button, TQ_SIGNAL(clicked()), this, TQ_SLOT( selectFileA() ) );
   TQPushButton * button2 = new TQPushButton( i18n("Dir..."), this );
   connect( button2, TQ_SIGNAL(clicked()), this, TQ_SLOT( selectDirA() ) );
   connect( m_pLineA, TQ_SIGNAL(textChanged(const TQString&)), this, TQ_SLOT(inputFilenameChanged() ) );

   h->addWidget( label,    0, 0 );
   h->addWidget( m_pLineA,  0, 1 );
   h->addWidget( button,   0, 2 );
   h->addWidget( button2,  0, 3 );

   label    = new TQLabel( "B:", this );
   m_pLineB  = new TQComboBox( true, this );
   m_pLineB->insertStringList( m_pOptions->m_recentBFiles );
   m_pLineB->setEditText( KURL(n2).prettyURL() );
   m_pLineB->setMinimumSize( 200, m_pLineB->size().height() );
   button   = new TQPushButton( i18n("File..."), this );
   connect( button, TQ_SIGNAL(clicked()), this, TQ_SLOT( selectFileB() ) );
   button2   = new TQPushButton( i18n("Dir..."), this );
   connect( button2, TQ_SIGNAL(clicked()), this, TQ_SLOT( selectDirB() ) );
   connect( m_pLineB, TQ_SIGNAL(textChanged(const TQString&)), this, TQ_SLOT(inputFilenameChanged() ) );

   h->addWidget( label,     1, 0 );
   h->addWidget( m_pLineB,   1, 1 );
   h->addWidget( button,    1, 2 );
   h->addWidget( button2,   1, 3 );

   label  = new TQLabel( i18n("C (Optional):"), this );
   m_pLineC= new TQComboBox( true, this );
   m_pLineC->insertStringList( m_pOptions->m_recentCFiles );
   m_pLineC->setEditText( KURL(n3).prettyURL() );
   m_pLineC->setMinimumSize( 200, m_pLineC->size().height() );
   button = new TQPushButton( i18n("File..."), this );
   connect( button, TQ_SIGNAL(clicked()), this, TQ_SLOT( selectFileC() ) );
   button2   = new TQPushButton( i18n("Dir..."), this );
   connect( button2, TQ_SIGNAL(clicked()), this, TQ_SLOT( selectDirC() ) );
   connect( m_pLineC, TQ_SIGNAL(textChanged(const TQString&)), this, TQ_SLOT(inputFilenameChanged() ) );

   h->addWidget( label,     2, 0 );
   h->addWidget( m_pLineC,   2, 1 );
   h->addWidget( button,    2, 2 );
   h->addWidget( button2,   2, 3 );

   m_pMerge = new TQCheckBox( i18n("Merge"), this );
   h->addWidget( m_pMerge, 3, 0 );

   TQHBoxLayout* hl = new TQHBoxLayout();
   h->addLayout( hl, 3, 1 );
   hl->addStretch(2);
   button = new TQPushButton(i18n("Swap/Copy Names ..."), this);
   //button->setToggleButton(false);
   hl->addWidget( button );

   TQPopupMenu* m = new TQPopupMenu(this);
   int id=0;
   m->insertItem( i18n("Swap %1<->%2").arg("A").arg("B"), id++ );
   m->insertItem( i18n("Swap %1<->%2").arg("B").arg("C"), id++ );
   m->insertItem( i18n("Swap %1<->%2").arg("C").arg("A"), id++ );
   m->insertItem( i18n("Copy %1->Output").arg("A"), id++ );
   m->insertItem( i18n("Copy %1->Output").arg("B"), id++ );
   m->insertItem( i18n("Copy %1->Output").arg("C"), id++  );
   m->insertItem( i18n("Swap %1<->Output").arg("A"), id++  );
   m->insertItem( i18n("Swap %1<->Output").arg("B"), id++  );
   m->insertItem( i18n("Swap %1<->Output").arg("C"), id++  );
   connect( m, TQ_SIGNAL(activated(int)), this, TQ_SLOT(slotSwapCopyNames(int)));
   button->setPopup(m);


   hl->addStretch(2);

   label  = new TQLabel( i18n("Output (optional):"), this );
   m_pLineOut = new TQComboBox( true, this );
   m_pLineOut->insertStringList( m_pOptions->m_recentOutputFiles );
   m_pLineOut->setEditText( KURL(outputName).prettyURL() );
   m_pLineOut->setMinimumSize( 200, m_pLineOut->size().height() );
   button = new TQPushButton( i18n("File..."), this );
   connect( button, TQ_SIGNAL(clicked()), this, TQ_SLOT( selectOutputName() ) );
   button2   = new TQPushButton( i18n("Dir..."), this );
   connect( button2, TQ_SIGNAL(clicked()), this, TQ_SLOT( selectOutputDir() ) );
   connect( m_pMerge, TQ_SIGNAL(stateChanged(int)), this, TQ_SLOT(internalSlot(int)) );
   connect( this, TQ_SIGNAL(internalSignal(bool)), m_pLineOut, TQ_SLOT(setEnabled(bool)) );
   connect( this, TQ_SIGNAL(internalSignal(bool)), button, TQ_SLOT(setEnabled(bool)) );
   connect( this, TQ_SIGNAL(internalSignal(bool)), button2, TQ_SLOT(setEnabled(bool)) );

   m_pMerge->setChecked( !bMerge );
   m_pMerge->setChecked( bMerge );
//   m_pLineOutput->setEnabled( bMerge );

//   button->setEnabled( bMerge );

   h->addWidget( label,          4, 0 );
   h->addWidget( m_pLineOut,      4, 1 );
   h->addWidget( button,         4, 2 );
   h->addWidget( button2,        4, 3 );

   h->addColSpacing( 1, 200 );

   TQHBoxLayout* l = new TQHBoxLayout( v, 5 );

   button = new TQPushButton( i18n("Configure..."), this );
   connect( button, TQ_SIGNAL(clicked()), pParent, slotConfigure );
   l->addWidget( button, 1 );

   l->addStretch(1);

   button = new TQPushButton( i18n("&OK"), this );
   button->setDefault( true );
   connect( button, TQ_SIGNAL(clicked()), this, TQ_SLOT( accept() ) );
   l->addWidget( button, 1 );

   button = new TQPushButton( i18n("&Cancel"), this );
   connect( button, TQ_SIGNAL(clicked()), this, TQ_SLOT( reject() ) );
   l->addWidget( button,1 );

   TQSize sh = sizeHint();
   setFixedHeight( sh.height() );
   m_bInputFileNameChanged = false;

#ifdef KREPLACEMENTS_H
   m_pLineA->lineEdit()->installEventFilter( this );
   m_pLineB->lineEdit()->installEventFilter( this );
   m_pLineC->lineEdit()->installEventFilter( this );
   m_pLineOut->lineEdit()->installEventFilter( this );
#endif
}

// Eventfilter: Only needed under Windows.
// Without this, files dropped in the line edit have URL-encoding.
// This eventfilter decodes the filenames as needed by KDiff3.
bool OpenDialog::eventFilter(TQObject* o, TQEvent* e)
{
   if (e->type()==TQEvent::Drop)
   {
      TQDropEvent* d = static_cast<TQDropEvent*>(e);

      if ( !TQUriDrag::canDecode( d ) ) {
         return false;
      }

      TQStringList lst;
      TQUriDrag::decodeLocalFiles( d, lst );

      if ( lst.count() > 0 )
      {
         static_cast<TQLineEdit*>(o)->setText( lst[0] );
         static_cast<TQLineEdit*>(o)->setFocus();
      }
       
      return true;
   }
   return false;
}


void OpenDialog::selectURL( TQComboBox* pLine, bool bDir, int i, bool bSave )
{
   TQString current = pLine->currentText();
   if (current.isEmpty() && i>3 ){  current = m_pLineC->currentText(); }
   if (current.isEmpty()        ){  current = m_pLineB->currentText(); }
   if (current.isEmpty()        ){  current = m_pLineA->currentText(); }
   KURL newURL = bDir ? KFileDialog::getExistingURL( current, this)
                      : bSave ? KFileDialog::getSaveURL( current, 0, this)
                              : KFileDialog::getOpenURL( current, 0, this);
   if ( !newURL.isEmpty() )
   {
      pLine->setEditText( newURL.url() );
   }
   // newURL won't be modified if nothing was selected.
}

void OpenDialog::selectFileA()     {  selectURL( m_pLineA,    false, 1, false );  }
void OpenDialog::selectFileB()     {  selectURL( m_pLineB,    false, 2, false );  }
void OpenDialog::selectFileC()     {  selectURL( m_pLineC,    false, 3, false );  }
void OpenDialog::selectOutputName(){  selectURL( m_pLineOut,  false, 4, true );  }
void OpenDialog::selectDirA()      {  selectURL( m_pLineA,    true,  1, false );  }
void OpenDialog::selectDirB()      {  selectURL( m_pLineB,    true,  2, false );  }
void OpenDialog::selectDirC()      {  selectURL( m_pLineC,    true,  3, false );  }
void OpenDialog::selectOutputDir() {  selectURL( m_pLineOut,  true,  4, true );  }

void OpenDialog::internalSlot(int i)
{
   emit internalSignal(i!=0);
}

// Clear the output-filename when any input-filename changed, 
// because users forgot to change the output and accidently overwrote it with
// wrong data during a merge.
void OpenDialog::inputFilenameChanged()
{
   if(!m_bInputFileNameChanged)
   {
      m_bInputFileNameChanged=true;
      m_pLineOut->clearEdit();
   }
}

static void fixCurrentText( TQComboBox* pCB )
{
   TQString s = pCB->currentText();

   int pos = s.find( '\n' );
   if ( pos>=0 )
      s=s.left(pos);
   pos = s.find( '\r' );
   if ( pos>=0 )
      s=s.left(pos);

   pCB->setCurrentText( s );
}

void OpenDialog::accept()
{
   unsigned int maxNofRecentFiles = 10;

   fixCurrentText( m_pLineA );
   TQString s = m_pLineA->currentText();
   s = KURL::fromPathOrURL(s).prettyURL();
   TQStringList* sl = &m_pOptions->m_recentAFiles;
   // If an item exist, remove it from the list and reinsert it at the beginning.
   sl->remove(s);
   if ( !s.isEmpty() ) sl->prepend( s );
   if (sl->count()>maxNofRecentFiles) sl->erase( sl->at(maxNofRecentFiles), sl->end() );

   fixCurrentText( m_pLineB );
   s = m_pLineB->currentText();
   s = KURL::fromPathOrURL(s).prettyURL();
   sl = &m_pOptions->m_recentBFiles;
   sl->remove(s);
   if ( !s.isEmpty() ) sl->prepend( s );
   if (sl->count()>maxNofRecentFiles) sl->erase( sl->at(maxNofRecentFiles), sl->end() );

   fixCurrentText( m_pLineC );
   s = m_pLineC->currentText();
   s = KURL::fromPathOrURL(s).prettyURL();
   sl = &m_pOptions->m_recentCFiles;
   sl->remove(s);
   if ( !s.isEmpty() ) sl->prepend( s );
   if (sl->count()>maxNofRecentFiles) sl->erase( sl->at(maxNofRecentFiles), sl->end() );

   fixCurrentText( m_pLineOut );
   s = m_pLineOut->currentText();
   s = KURL::fromPathOrURL(s).prettyURL();
   sl = &m_pOptions->m_recentOutputFiles;
   sl->remove(s);
   if ( !s.isEmpty() ) sl->prepend( s );
   if (sl->count()>maxNofRecentFiles) sl->erase( sl->at(maxNofRecentFiles), sl->end() );

   TQDialog::accept();
}

void OpenDialog::slotSwapCopyNames( int id ) // id selected in the popup menu
{
   TQComboBox* cb1=0;
   TQComboBox* cb2=0;
   switch(id)
   {
   case 0:  cb1=m_pLineA; cb2=m_pLineB; break;
   case 1:  cb1=m_pLineB; cb2=m_pLineC; break;
   case 2:  cb1=m_pLineC; cb2=m_pLineA; break;
   case 3:  cb1=m_pLineA; cb2=m_pLineOut; break;
   case 4:  cb1=m_pLineB; cb2=m_pLineOut; break;
   case 5:  cb1=m_pLineC; cb2=m_pLineOut; break;
   case 6:  cb1=m_pLineA; cb2=m_pLineOut; break;
   case 7:  cb1=m_pLineB; cb2=m_pLineOut; break;
   case 8:  cb1=m_pLineC; cb2=m_pLineOut; break;
   }
   if ( cb1 && cb2 )
   {
      TQString t1 = cb1->currentText();
      TQString t2 = cb2->currentText();
      cb2->setCurrentText(t1);
      if ( id<=2 || id>=6 )
      {
         cb1->setCurrentText( t2 );
      }
   }
}

// FindDialog *********************************************

FindDialog::FindDialog(TQWidget* pParent)
: TQDialog( pParent )
{
   TQGridLayout* layout = new TQGridLayout( this );
   layout->setMargin(5);
   layout->setSpacing(5);

   int line=0;
   layout->addMultiCellWidget( new TQLabel(i18n("Search text:"),this), line,line,0,1 );
   ++line;

   m_pSearchString = new TQLineEdit( this );
   layout->addMultiCellWidget( m_pSearchString, line,line,0,1 );
   ++line;

   m_pCaseSensitive = new TQCheckBox(i18n("Case sensitive"),this);
   layout->addWidget( m_pCaseSensitive, line, 1 );

   m_pSearchInA = new TQCheckBox(i18n("Search A"),this);
   layout->addWidget( m_pSearchInA, line, 0 );
   m_pSearchInA->setChecked( true );
   ++line;

   m_pSearchInB = new TQCheckBox(i18n("Search B"),this);
   layout->addWidget( m_pSearchInB, line, 0 );
   m_pSearchInB->setChecked( true );
   ++line;

   m_pSearchInC = new TQCheckBox(i18n("Search C"),this);
   layout->addWidget( m_pSearchInC, line, 0 );
   m_pSearchInC->setChecked( true );
   ++line;

   m_pSearchInOutput = new TQCheckBox(i18n("Search output"),this);
   layout->addWidget( m_pSearchInOutput, line, 0 );
   m_pSearchInOutput->setChecked( true );
   ++line;

   TQPushButton* pButton = new TQPushButton( i18n("&Search"), this );
   layout->addWidget( pButton, line, 0 );
   connect( pButton, TQ_SIGNAL(clicked()), this, TQ_SLOT(accept()));

   pButton = new TQPushButton( i18n("&Cancel"), this );
   layout->addWidget( pButton, line, 1 );
   connect( pButton, TQ_SIGNAL(clicked()), this, TQ_SLOT(reject()));

   hide();
}


RegExpTester::RegExpTester( TQWidget* pParent, const TQString& autoMergeRegExpToolTip,
   const TQString& historyStartRegExpToolTip, const TQString& historyEntryStartRegExpToolTip, const TQString& historySortKeyOrderToolTip )
: TQDialog( pParent)
{
   int line=0;
   setCaption(i18n("Regular Expression Tester"));
   TQGridLayout* pGrid = new TQGridLayout( this, 11, 2, 5, 5 );

   TQLabel* l = new TQLabel(i18n("Auto merge regular expression:"), this);
   pGrid->addWidget(l,line,0);
   TQToolTip::add( l, autoMergeRegExpToolTip );
   m_pAutoMergeRegExpEdit = new TQLineEdit(this);
   pGrid->addWidget(m_pAutoMergeRegExpEdit,line,1);
   connect( m_pAutoMergeRegExpEdit, TQ_SIGNAL(textChanged(const TQString&)), this, TQ_SLOT(slotRecalc()));
   ++line;

   l = new TQLabel(i18n("Example auto merge line:"), this);
   pGrid->addMultiCellWidget(l,line,line,0,1);
   TQToolTip::add( l, i18n("For auto merge test copy a line as used in your files.") );
   m_pAutoMergeExampleEdit = new TQLineEdit(this);
   pGrid->addWidget(m_pAutoMergeExampleEdit,line,1);
   connect( m_pAutoMergeExampleEdit, TQ_SIGNAL(textChanged(const TQString&)), this, TQ_SLOT(slotRecalc()));
   ++line;

   l = new TQLabel(i18n("Match result:"), this);
   pGrid->addWidget(l,line,0);
   m_pAutoMergeMatchResult = new TQLineEdit(this);
   m_pAutoMergeMatchResult->setReadOnly(true);
   pGrid->addWidget(m_pAutoMergeMatchResult,line,1);
   ++line;

   pGrid->addItem( new TQSpacerItem(100,20), line, 0 );
   pGrid->setRowStretch( line, 5);
   ++line;

   l = new TQLabel(i18n("History start regular expression:"), this);
   pGrid->addWidget(l,line,0);
   TQToolTip::add( l, historyStartRegExpToolTip );
   m_pHistoryStartRegExpEdit = new TQLineEdit(this);
   pGrid->addWidget(m_pHistoryStartRegExpEdit,line,1);
   connect( m_pHistoryStartRegExpEdit, TQ_SIGNAL(textChanged(const TQString&)), this, TQ_SLOT(slotRecalc()));
   ++line;

   l = new TQLabel(i18n("Example history start line (with leading comment):"), this);
   pGrid->addMultiCellWidget(l,line,line,0,1);
   ++line;
   TQToolTip::add( l, i18n("Copy a history start line as used in your files,\n"
                          "including the leading comment.") );
   m_pHistoryStartExampleEdit = new TQLineEdit(this);
   pGrid->addWidget(m_pHistoryStartExampleEdit,line,1);
   connect( m_pHistoryStartExampleEdit, TQ_SIGNAL(textChanged(const TQString&)), this, TQ_SLOT(slotRecalc()));
   ++line;

   l = new TQLabel(i18n("Match result:"), this);
   pGrid->addWidget(l,line,0);
   m_pHistoryStartMatchResult = new TQLineEdit(this);
   m_pHistoryStartMatchResult->setReadOnly(true);
   pGrid->addWidget(m_pHistoryStartMatchResult,line,1);
   ++line;

   pGrid->addItem( new TQSpacerItem(100,20), line, 0 );
   pGrid->setRowStretch( line, 5);
   ++line;

   l = new TQLabel(i18n("History entry start regular expression:"), this);
   pGrid->addWidget(l,line,0);
   TQToolTip::add( l, historyEntryStartRegExpToolTip );
   m_pHistoryEntryStartRegExpEdit = new TQLineEdit(this);
   pGrid->addWidget(m_pHistoryEntryStartRegExpEdit,line,1);
   connect( m_pHistoryEntryStartRegExpEdit, TQ_SIGNAL(textChanged(const TQString&)), this, TQ_SLOT(slotRecalc()));
   ++line;

   l = new TQLabel(i18n("History sort key order:"), this);
   pGrid->addWidget(l,line,0);
   TQToolTip::add( l, historySortKeyOrderToolTip );
   m_pHistorySortKeyOrderEdit = new TQLineEdit(this);
   pGrid->addWidget(m_pHistorySortKeyOrderEdit,line,1);
   connect( m_pHistorySortKeyOrderEdit, TQ_SIGNAL(textChanged(const TQString&)), this, TQ_SLOT(slotRecalc()));
   ++line;

   l = new TQLabel(i18n("Example history entry start line (without leading comment):"), this);
   pGrid->addMultiCellWidget(l,line,line,0,1);
   TQToolTip::add( l, i18n("Copy a history entry start line as used in your files,\n"
                          "but omit the leading comment.") );
   ++line;
   m_pHistoryEntryStartExampleEdit = new TQLineEdit(this);
   pGrid->addWidget(m_pHistoryEntryStartExampleEdit,line,1);
   connect( m_pHistoryEntryStartExampleEdit, TQ_SIGNAL(textChanged(const TQString&)), this, TQ_SLOT(slotRecalc()));
   ++line;

   l = new TQLabel(i18n("Match result:"), this);
   pGrid->addWidget(l,line,0);
   m_pHistoryEntryStartMatchResult = new TQLineEdit(this);
   m_pHistoryEntryStartMatchResult->setReadOnly(true);
   pGrid->addWidget(m_pHistoryEntryStartMatchResult,line,1);
   ++line;

   l = new TQLabel(i18n("Sort key result:"), this);
   pGrid->addWidget(l,line,0);
   m_pHistorySortKeyResult = new TQLineEdit(this);
   m_pHistorySortKeyResult->setReadOnly(true);
   pGrid->addWidget(m_pHistorySortKeyResult,line,1);
   ++line;

   TQPushButton* pButton = new TQPushButton(i18n("OK"), this);
   pGrid->addWidget(pButton,line,0);
   connect( pButton, TQ_SIGNAL(clicked()), this, TQ_SLOT(accept()));

   pButton = new TQPushButton(i18n("Cancel"), this);
   pGrid->addWidget(pButton,line,1);
   connect( pButton, TQ_SIGNAL(clicked()), this, TQ_SLOT(reject()));

   resize( 800, sizeHint().height() );
}

void RegExpTester::init( const TQString& autoMergeRegExp, const TQString& historyStartRegExp, const TQString& historyEntryStartRegExp, const TQString historySortKeyOrder )
{
   m_pAutoMergeRegExpEdit->setText( autoMergeRegExp );
   m_pHistoryStartRegExpEdit->setText( historyStartRegExp );
   m_pHistoryEntryStartRegExpEdit->setText( historyEntryStartRegExp );
   m_pHistorySortKeyOrderEdit->setText( historySortKeyOrder );
}

TQString RegExpTester::autoMergeRegExp()
{
   return m_pAutoMergeRegExpEdit->text();
}

TQString RegExpTester::historyStartRegExp()
{
   return m_pHistoryStartRegExpEdit->text();
}

TQString RegExpTester::historyEntryStartRegExp()
{
   return m_pHistoryEntryStartRegExpEdit->text();
}

TQString RegExpTester::historySortKeyOrder()
{
   return m_pHistorySortKeyOrderEdit->text();
}

void RegExpTester::slotRecalc()
{
   TQRegExp autoMergeRegExp = m_pAutoMergeRegExpEdit->text();
   if ( autoMergeRegExp.exactMatch( m_pAutoMergeExampleEdit->text() ) )
   {
      m_pAutoMergeMatchResult->setText( i18n("Match success.") );
   }
   else
   {
      m_pAutoMergeMatchResult->setText( i18n("Match failed.") );
   }

   TQRegExp historyStartRegExp = m_pHistoryStartRegExpEdit->text();
   if ( historyStartRegExp.exactMatch( m_pHistoryStartExampleEdit->text() ) )
   {
      m_pHistoryStartMatchResult->setText( i18n("Match success.") );
   }
   else
   {
      m_pHistoryStartMatchResult->setText( i18n("Match failed.") );
   }


   TQStringList parenthesesGroups;
   bool bSuccess = findParenthesesGroups( m_pHistoryEntryStartRegExpEdit->text(), parenthesesGroups );
   if ( ! bSuccess )
   {
      m_pHistoryEntryStartMatchResult->setText( i18n("Opening and closing parentheses don't match in regular expression.") );
      m_pHistorySortKeyResult->setText( "" );
      return;
   }
   TQRegExp historyEntryStartRegExp = m_pHistoryEntryStartRegExpEdit->text();
   TQString s = m_pHistoryEntryStartExampleEdit->text();

   if ( historyEntryStartRegExp.exactMatch( s ) )
   {
      m_pHistoryEntryStartMatchResult->setText( i18n("Match success.") );
      TQString key = calcHistorySortKey( m_pHistorySortKeyOrderEdit->text(),historyEntryStartRegExp,parenthesesGroups);
      m_pHistorySortKeyResult->setText(key);
   }
   else
   {
      m_pHistoryEntryStartMatchResult->setText( i18n("Match failed.") );
      m_pHistorySortKeyResult->setText( "" );
   }
}

#include "smalldialogs.moc"
