/*
  Copyright (C) 2006 Daniele Galdi <daniele.galdi@gmail.com>
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the Free
  Software Foundation; either version 2 of the License, or (at your option)
  any later version.
  
  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
  more details.
  
  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 51
  Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

#include "adblock.h"
#include "adblockdialogue.h"

#include <kdebug.h>
#include <tdepopupmenu.h>
#include <tdelocale.h>

#include <tqcursor.h>
#include <tqlabel.h>
#include <tqvbox.h>
#include <tqlineedit.h>
#include <tqcolor.h>
#include <tqfont.h>
#include <tqpainter.h>

AdBlockDlg::AdBlockDlg(TQWidget *parent, AdElementList &elements) :
    KDialogBase( parent, "Adblock dialogue", true, "Adblock - able Items", Ok|Cancel, Ok, true )
{
    TQVBox *page = makeVBoxMainWidget();
    m_label1 = new TQLabel( i18n("All blockable items in this page:"), page, "label1" );
    
    m_list = new TQListView(page);
    m_list->setAllColumnsShowFocus( true );

    m_list->addColumn( i18n("Source") );
    m_list->addColumn( i18n("Category") );
    m_list->addColumn( i18n("Node Name") );

    m_list->setColumnWidthMode(0, TQListView::Manual);    
    m_list->setColumnWidthMode(1, TQListView::Manual);    
    m_list->setColumnWidthMode(2, TQListView::Manual);    

    m_list->setColumnWidth(0, 600);
    m_list->setColumnWidth(1, 90);
    m_list->setColumnWidth(2, 90);

    AdElementList::iterator it;
    for ( it = elements.begin(); it != elements.end(); ++it )
    {
	AdElement &element = (*it);	

	TQString url = element.url();
	
	ListViewItem *item = new ListViewItem( m_list, url, element.category(), element.type() );
	item->setBlocked(element.isBlocked());	
    }
    
    m_label2 = new TQLabel( i18n("New filter (use * as a wildcard):"), page, "label2" );
    
    m_filter = new TQLineEdit( "", page, "lineedit" );    
    
    connect(this, TQ_SIGNAL( okClicked() ), this, TQ_SLOT( validateFilter() ));
    connect(m_list, TQ_SIGNAL( doubleClicked(TQListViewItem *, const TQPoint &, int) ), this, TQ_SLOT(updateFilter(TQListViewItem *)) );

    m_menu = new TDEPopupMenu(this);
    m_menu->insertItem(i18n("Filter this item"), this, TQ_SLOT(filterItem()));
    m_menu->insertItem(i18n("Filter all items at same path"), this, TQ_SLOT(filterPath()));

    connect(m_list, 
	    TQ_SIGNAL( contextMenuRequested( TQListViewItem *, const TQPoint& , int ) ), 
	    this, 
	    TQ_SLOT( showContextMenu(TQListViewItem *, const TQPoint &) ) );
}

void AdBlockDlg::updateFilter(TQListViewItem *selected)
{    
    ListViewItem *item = dynamic_cast<ListViewItem *>(selected);

    if (item->isBlocked()) 
    {
	m_filter->setText("");
	return;
    }

    m_filter->setText( item->text(0) );
}

void AdBlockDlg::validateFilter()
{
    const TQString text = m_filter->text().stripWhiteSpace();

    if (!text.isEmpty())
        emit notEmptyFilter(text);

    delayedDestruct();
}

void AdBlockDlg::showContextMenu(TQListViewItem *item, const TQPoint &point)
{
    if (!item) return;
    m_menu->popup(point);
}

void AdBlockDlg::filterItem()
{
    TQListViewItem* item = m_list->selectedItem();
    m_filter->setText( item->text(0) );
}

void AdBlockDlg::filterPath()
{
    TQListViewItem* item = m_list->selectedItem();
    TQString value = item->text(0);
    m_filter->setText( value.section( '/', 0, -2 ).append("/*") );
}

AdBlockDlg::~AdBlockDlg()
{
    delete m_label1;
    delete m_label2;
    delete m_filter;
    delete m_list;
}

// ----------------------------------------------------------------------------

void ListViewItem::paintCell(TQPainter *p, const TQColorGroup & cg, int column, int width, int align)
{
    p->save();
    TQColorGroup g( cg );
    
    if ( isBlocked() )
    {
	g.setColor(TQColorGroup::Text, red);
        TQFont font;
        font.setItalic(true);
        p->setFont(font);
    }
    
    TQListViewItem::paintCell(p, g, column, width, align);
    p->restore();
}

inline bool ListViewItem::isBlocked()
{
    return m_blocked;
}

inline void ListViewItem::setBlocked(bool blocked)
{
    m_blocked = blocked;
}

#include "adblockdialogue.moc"
