/*  This file is part of the KDE KMobile library
    Copyright (C) 2003 Helge Deller <deller@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License version 2 as published by the Free Software Foundation.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.

*/

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <unistd.h>
#include <time.h>
#include <errno.h>
#include <signal.h>
#include <fcntl.h>

#include <tqstring.h>
#include <tqstringlist.h>
#include <tqfile.h>

#include <klibloader.h>
#include <tdestandarddirs.h>
#include "kmobiledevice.h"
#include <tdemessagebox.h>
#include <tdeconfig.h>

#include <tdeio/global.h>
#include <kdebug.h>
#include <tdelocale.h>

#define KMOBILEDEVICE_DEBUG_AREA 5730
#define PRINT_DEBUG kdDebug(KMOBILEDEVICE_DEBUG_AREA) << "KMobileDevice: "

/**
 *  The base plugin class of all mobile device drivers.
 */

KMobileDevice::KMobileDevice(TQObject *obj, const char *name, const TQStringList &args)
	: KLibFactory(obj,name), m_config(0L), d(0L), m_fd(-1)
{
  setClassType(Unclassified);
  setCapabilities(hasNothing);
  m_deviceName = i18n("Unknown Device");
  m_deviceRevision = i18n("n/a");  /* not available */
  m_connectionName = i18n("Unknown Connection");

  // set the config file name
  m_configFileName = args[0];
  if (m_configFileName.isEmpty())
	m_config = new TDEConfig();
  else
	m_config = new TDEConfig(m_configFileName);
  PRINT_DEBUG << TQString("name of config file is %1\n").arg(m_configFileName);
}

KMobileDevice::~KMobileDevice()
{
  delete m_config;
}


bool KMobileDevice::connected()
{
  return m_connected;
}

// returns e.g. "Nokia mobile phone", "MP3 Player", "Handspring Organizer"
TQString KMobileDevice::deviceClassName() const
{
  return m_deviceClassName;
}

// returns real device name, e.g. "Nokia 6310" or "Rio MP3 Player"
TQString KMobileDevice::deviceName() const
{
  return m_deviceName;
}

TQString KMobileDevice::revision() const
{
  return m_deviceRevision;
}

bool KMobileDevice::isSlowDevice() const
{
  return false;
}

bool KMobileDevice::isReadOnly() const
{
  return false;
}

bool KMobileDevice::configDialog( TQWidget *parent )
{
  KMessageBox::information( parent,
		i18n("This device does not need any configuration."),
		deviceName() );
  return true;
}

void KMobileDevice::setClassType( enum ClassType ct )
{
  m_classType = ct;
  m_deviceClassName = defaultClassName(ct);
}

enum KMobileDevice::ClassType KMobileDevice::classType() const
{
  return m_classType;
}

TQString KMobileDevice::iconFileName() const
{
  return defaultIconFileName( classType() );
}

TQString KMobileDevice::defaultIconFileName( ClassType ct )
{
  TQString name;
  switch (ct) {
    case Phone:		name = "mobile_phone";		break;
    case Organizer:	name = "mobile_organizer";	break;
    case Camera:	name = "mobile_camera";		break;
    case MusicPlayer:	name = "mobile_mp3player";	break;
    case Unclassified:
    default:		name = KMOBILE_ICON_UNKNOWN;	break;
  }
  return name;
}

TQString KMobileDevice::defaultClassName( ClassType ct )
{
  TQString name;
  switch (ct) {
    case Phone:		name = i18n("Cellular Mobile Phone");	break;
    case Organizer:	name = i18n("Organizer");		break;
    case Camera:	name = i18n("Digital Camera");		break;
    case MusicPlayer:	name = i18n("Music/MP3 Player");	break;
    case Unclassified:
    default:		name = i18n("Unclassified Device");	break;
  }
  return name;
}

void KMobileDevice::setCapabilities( int caps )
{
  m_caps = caps;
}

int KMobileDevice::capabilities() const
{
  return m_caps;
}

const TQString KMobileDevice::nameForCap(int cap) const
{
  switch (cap) {
    case hasAddressBook: return i18n("Contacts");
    case hasCalendar:	 return i18n("Calendar");
    case hasNotes:	 return i18n("Notes");
    case hasFileStorage: return i18n("Files");
    default:		 return i18n("Unknown");
  }
}

// returns an error string for the given error code
TQString KMobileDevice::buildErrorString(TDEIO::Error err, const TQString &errorText) const
{
  return TDEIO::buildErrorString( err, errorText);
}

/*
 * Addressbook / Phonebook support
 */
int KMobileDevice::numAddresses()
{
  return 0;
}

int KMobileDevice::readAddress( int, TDEABC::Addressee & )
{
  return TDEIO::ERR_UNSUPPORTED_ACTION;
}

int KMobileDevice::storeAddress( int, const TDEABC::Addressee &, bool )
{
  return TDEIO::ERR_UNSUPPORTED_ACTION;
}

/*
 * Calendar support
 */
int KMobileDevice::numCalendarEntries()
{
  return 0;
}

int KMobileDevice::readCalendarEntry( int, KCal::Event & )
{
  return TDEIO::ERR_UNSUPPORTED_ACTION;
}

int KMobileDevice::storeCalendarEntry( int, const KCal::Event & )
{
  return TDEIO::ERR_UNSUPPORTED_ACTION;
}

/*
 * Notes support
 */
int KMobileDevice::numNotes()
{
  return 0;
}

int KMobileDevice::readNote( int, TQString & )
{
  return TDEIO::ERR_UNSUPPORTED_ACTION;
}

int KMobileDevice::storeNote( int, const TQString & )
{
  return TDEIO::ERR_UNSUPPORTED_ACTION;
}

/*
 * File storage support
 * @param fileName  path and name of a file in the mobile device, e.g. "/MYFILE.TXT", "/mp3/song1.mp3"
 */

static
void addAtom(TDEIO::UDSEntry& entry, unsigned int ID, long l, const TQString& s = TQString())
{
	TDEIO::UDSAtom atom;
	atom.m_uds = ID;
	atom.m_long = l;
	atom.m_str = s;
	entry.append(atom);
}

void KMobileDevice::createDirEntry(TDEIO::UDSEntry& entry, const TQString& name, const TQString& url, const TQString& mime) const
{
	entry.clear();
	addAtom(entry, TDEIO::UDS_NAME, 0, name);
	addAtom(entry, TDEIO::UDS_FILE_TYPE, S_IFDIR);
	addAtom(entry, TDEIO::UDS_ACCESS, 0500);
	addAtom(entry, TDEIO::UDS_MIME_TYPE, 0, mime);
	addAtom(entry, TDEIO::UDS_URL, 0, url);
	PRINT_DEBUG << TQString("createDirEntry: File: %1  MIME: %2  URL: %3\n").arg(name).arg(mime).arg(url);
//	addAtom(entry, TDEIO::UDS_SIZE, 0);
	addAtom(entry, TDEIO::UDS_GUESSED_MIME_TYPE, 0, mime);
}

void KMobileDevice::createFileEntry(TDEIO::UDSEntry& entry, const TQString& name, const TQString& url, const TQString& mime,
		const unsigned long size) const
{
	entry.clear();
	addAtom(entry, TDEIO::UDS_NAME, 0, name);
	addAtom(entry, TDEIO::UDS_FILE_TYPE, S_IFREG);
	addAtom(entry, TDEIO::UDS_URL, 0, url);
	addAtom(entry, TDEIO::UDS_ACCESS, 0400);
	addAtom(entry, TDEIO::UDS_MIME_TYPE, 0, mime);
	if (size) addAtom(entry, TDEIO::UDS_SIZE, size);
	addAtom(entry, TDEIO::UDS_GUESSED_MIME_TYPE, 0, mime);
	PRINT_DEBUG << TQString("createFileEntry: File: %1, Size: %2,  MIME: %3\n").arg(name).arg(size).arg(mime);
}


void KMobileDevice::listDir( const TQString & )
{
  emit error(TDEIO::ERR_CANNOT_ENTER_DIRECTORY,TQString());
}

void KMobileDevice::mkdir( const TQString &, int )
{
  emit error(TDEIO::ERR_COULD_NOT_MKDIR, TQString());
}

void KMobileDevice::rename( const TQString &, const TQString &, bool )
{
  emit error(TDEIO::ERR_UNSUPPORTED_ACTION, TQString());
}

void KMobileDevice::symlink( const TQString &, const TQString &, bool )
{
  emit error(TDEIO::ERR_UNSUPPORTED_ACTION, TQString());
}

void KMobileDevice::del( const TQString &, bool )
{
  emit error(TDEIO::ERR_UNSUPPORTED_ACTION, TQString());
}

void KMobileDevice::stat( const TQString & )
{
  emit error(TDEIO::ERR_UNSUPPORTED_ACTION, TQString());
}

void KMobileDevice::chmod( const TQString &, int )
{
  emit error(TDEIO::ERR_UNSUPPORTED_ACTION, TQString());
}

void KMobileDevice::get( const TQString & )
{
  emit error(TDEIO::ERR_UNSUPPORTED_ACTION, TQString());
}

void KMobileDevice::put( const TQString &, int, bool, bool )
{
  emit error(TDEIO::ERR_UNSUPPORTED_ACTION, TQString());
}

void KMobileDevice::mimetype( const TQString & )
{
  emit error(TDEIO::ERR_UNSUPPORTED_ACTION, TQString());
}

void KMobileDevice::special( const TQByteArray & )
{
  emit error(TDEIO::ERR_UNSUPPORTED_ACTION, TQString());
}




/*
 * device locking/unlocking functions
 */

#ifdef HAVE_BAUDBOY_H
// Header shipped with newer RedHat distributions.
// We have to use this here, because /var/lock is owned by root:lock
// and no one is in this group. Only the binary /usr/sbin/lockdev is
// owned by this group and has the setgid flag set. This binary is called
// in ttylock etc ...
# include <baudboy.h>
# include <cstdlib>
#endif

bool KMobileDevice::lockDevice(const TQString &device, TQString &err_reason)
{
#ifdef HAVE_BAUDBOY_H
  return ttylock(device.local8Bit()) == EXIT_SUCCESS;
#else
	if (m_fd != -1)
	  return true;

	TQCString dev = TQFile::encodeName(device.local8Bit());
	const char *fdev = dev.data();
	if ((m_fd = open(fdev, O_RDWR)) == -1)
	{
		PRINT_DEBUG << i18n("Unable to open device '%1'. "
												"Please check that you have sufficient permissions.").arg(fdev);
		return false;
	}
	if (flock(m_fd, LOCK_EX))
	{
	  // Locking failed
    PRINT_DEBUG << i18n("Unable to lock device '%1'.").arg(fdev);
    return false;
	}
  return true;
#endif
}

bool KMobileDevice::unlockDevice(const TQString &device)
{
#ifdef HAVE_BAUDBOY_H
  return ttyunlock(device.local8bit()) == EXIT_SUCCESS;
#else
  if (m_fd != -1) {
	if(flock(m_fd, LOCK_UN)) {
		return false;
	}
	close(m_fd);
  }
  m_fd = -1;
  return true;
#endif
}

#include "kmobiledevice.moc"
