/* This file is part of the KDE project
   Copyright (C) 2004 Alexander Dymo <adymo@kdevelop.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/
#include "kdevplugininfo.h"

#include <tqvariant.h>

#include <kservice.h>
#include <kdebug.h>

#include "kdevplugincontroller.h"

struct KDevPluginInfo::Private {
    TQString m_pluginName;
    TQString m_rawGenericName;
    
    TQString m_genericName;
    TQString m_description;
    TQString m_icon;
    
    TQString m_version;
    int m_licenseType;
    TQString m_copyrightStatement;
    TQString m_homePageAddress;
    TQString m_bugsEmailAddress;
    
    TQValueList<TDEAboutPerson> m_authors;
    TQValueList<TDEAboutPerson> m_credits;
    
    TDEAboutData *m_data;
};


KDevPluginInfo::KDevPluginInfo(const TQString &pluginName)
    :d(new Private())
{
    d->m_pluginName = pluginName;
    
    KService::Ptr offer = KService::serviceByDesktopName(pluginName);
    if (offer != 0)
    {
        d->m_genericName = offer->genericName();
        d->m_icon = offer->icon();
        d->m_description = offer->comment();
        
        d->m_rawGenericName = offer->untranslatedGenericName();
        
        d->m_version = offer->property("X-TDevelop-Plugin-Version").toString();
        d->m_homePageAddress = offer->property("X-TDevelop-Plugin-Homepage").toString();
        d->m_bugsEmailAddress = offer->property("X-TDevelop-Plugin-BugsEmailAddress").toString();
        d->m_copyrightStatement = offer->property("X-TDevelop-Plugin-Copyright").toString();
        
        TQString lic = offer->property("X-TDevelop-Plugin-License").toString();
        if (lic == "GPL")
            d->m_licenseType = TDEAboutData::License_GPL;
        else if (lic == "LGPL")
            d->m_licenseType = TDEAboutData::License_LGPL;
        else if (lic == "BSD")
            d->m_licenseType = TDEAboutData::License_BSD;
        else if (lic == "QPL")
            d->m_licenseType = TDEAboutData::License_QPL;
        else if (lic == "Artistic")
            d->m_licenseType = TDEAboutData::License_Artistic;
        else if (lic == "Custom")
            d->m_licenseType = TDEAboutData::License_Custom;
        else
            d->m_licenseType = TDEAboutData::License_Unknown;            

        d->m_data = new TDEAboutData(d->m_pluginName.ascii(), d->m_rawGenericName.ascii(), "1", 0, d->m_licenseType);
    }
    else
	kdDebug() << "Unable to load information for plugin: " << pluginName 
	    << ". Check if " << pluginName << ".desktop exists." << endl;
}


KDevPluginInfo::operator TDEAboutData *() const
{
    return d->m_data;
}

TQString KDevPluginInfo::pluginName() const
{
    return d->m_pluginName;
}

TQString KDevPluginInfo::genericName() const
{
    return d->m_genericName;
}

TQString KDevPluginInfo::icon() const
{
    return d->m_icon;
}

TQString KDevPluginInfo::description() const
{
    return d->m_description;
}

TQString KDevPluginInfo::version() const
{
    return d->m_version;
}

int KDevPluginInfo::licenseType() const
{
    return d->m_licenseType;
}

TQString KDevPluginInfo::license() const
{
    KDevPluginInfo &info = *const_cast<KDevPluginInfo*>(this);
//    return TDEAboutData(info).license();
    TDEAboutData *data = info;
    return data->license();
}

TQString KDevPluginInfo::copyrightStatement() const
{
    return d->m_copyrightStatement;
}

TQString KDevPluginInfo::homePageAddress() const
{
    return d->m_homePageAddress;
}

TQString KDevPluginInfo::bugsEmailAddress() const
{
    return d->m_bugsEmailAddress;
}

TQVariant KDevPluginInfo::property(const TQString &name) const
{
    TDETrader::OfferList offers = KDevPluginController::queryPlugins(TQString("Name='%1'").arg(d->m_pluginName));
    if (offers.count() == 1)
        return offers.first()->property(name);
    return TQVariant();
}

TQVariant KDevPluginInfo::operator [](const TQString &name) const
{
    return property(name);
}

TQStringList KDevPluginInfo::propertyNames( ) const
{
    TDETrader::OfferList offers = KDevPluginController::queryPlugins(TQString("Name='%1'").arg(d->m_pluginName));
    if (offers.count() == 1)
        return offers.first()->propertyNames();
    return TQStringList();
}

void KDevPluginInfo::addAuthor(const char *name, const char *task,
    const char *emailAddress, const char *webAddress)
{
    d->m_authors.append(TDEAboutPerson(name, task, emailAddress, webAddress));
}

void KDevPluginInfo::addCredit(const char *name, const char *task,
    const char *emailAddress, const char *webAddress)
{
    d->m_credits.append(TDEAboutPerson(name, task, emailAddress, webAddress));
}
