/* written by Guido Socher <guido.socher@linuxfocus.org>
 * based on code from lcdproc-0.3.3 by William W. Ferrell and Scott Scriven
 * copyright: GNU GENERAL PUBLIC LICENSE
 */
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <termios.h>
#include <fcntl.h>
#include <string.h>

/* global data */
static int fd;
static int opt_c=140;
static int opt_S=0;
static int opt_i=0;
static int opt_W=0;

void help()
{
        printf("mtxorb -- initialize matrix orbital lcd display\n\
\n\
USAGE: mtxorb [-hWSi] [-c contrast] device\n\
\n\
OPTIONS: -h this help\n\
         -c set the contrast [0-255], default=140\n\
         -i initially clear the display\n\
         -W set auto line wrap off, default is on\n\
         -S set auto scroll off, default is on\n\
 \n\
EXAMPLE: mtxorb /dev/lcd\n\
	    \n");
        exit(0);
}

/*
 * Changes screen contrast (0-255; 140 seems good)
 */
void MtxOrb_contrast(int contrast)
{
	char out[4];
	sprintf(out, "%cP%c", 254, contrast);
	write(fd, out, 3);
}

/*
 * Sets the backlight on or off 
 */
void MtxOrb_backlight(int on)
{
	char out[4];
	if (on) {
		sprintf(out, "%cB%c", 254, 0);
		write(fd, out, 3);
	} else {
		sprintf(out, "%cF", 254);
		write(fd, out, 2);
	}
}
/*
 * Toggle the built-in linewrapping feature
 */
static void MtxOrb_linewrap(int on)
{
	char out[4];
	if (on)
		sprintf(out, "%cC", 254);
	else
		sprintf(out, "%cD", 254);
	write(fd, out, 2);
}

/*
 * Clear the display
 */
static void MtxOrb_clear()
{
	char out[4];
	sprintf(out, "%cX", 0xfe);
	write(fd, out, 2);
}

/*
 * Toggle the built-in automatic scrolling feature
 */
static void MtxOrb_autoscroll(int on)
{
	char out[4];
	if (on)
		sprintf(out, "%cQ", 254);
	else
		sprintf(out, "%cR", 254);
	write(fd, out, 2);
}

/*
 * Opens com port and sets baud correctly...
 */
int MtxOrb_init(char *device)
{
	struct termios portset;

	/* Set up io port correctly, and open it... */
	fd = open(device, O_RDWR | O_NOCTTY | O_NDELAY);
	if (fd == -1) {
		fprintf(stderr, "ERROR in mtxorb: open for %s failed.\n",device);
		exit(1);
	}
	tcgetattr(fd, &portset);
	/* This is necessary in Linux, but does not exist in irix. */
#if defined(IRIX) || defined(sun)
#define NOCFMAKERAW
#endif
#ifndef NOCFMAKERAW
	cfmakeraw(&portset);
#endif
	cfsetospeed(&portset, B19200);
	tcsetattr(fd, TCSANOW, &portset);

	return fd;
}

int main(int argc, char *argv[])
{
        /* The following things are used for getopt: */ 
	int ch;
        extern char *optarg;
        extern int optind;
        extern int opterr;
                                        
        opterr = 0;
        while ((ch = getopt(argc, argv, "hWSic:")) != -1) {
                switch (ch) {
                case 'W':
                        opt_W=1;
			break;
                case 'S':
                        opt_S=1;
			break;
                case 'i':
                        opt_i=1;
			break;
                case 'c':
                        sscanf(optarg,"%d",&opt_c);
			break;
                case 'h':
                        help(); /*no break, help does not return */
                case '?':
                        fprintf(stderr, "ERROR: No such option. -h for help.\n");
                        exit(1);
                /*no default action for case */
                }
        }
	/* exactly one argument */
        if (optind != argc-1){
                help();
        }
	MtxOrb_init(argv[optind]);

	/* Set display-specific stuff.. */
	if (opt_W==0) MtxOrb_linewrap(1);
	if (opt_S==0) MtxOrb_autoscroll(1);
	MtxOrb_contrast(opt_c);
	MtxOrb_backlight(1);
	if (opt_i==1) MtxOrb_clear();
	/* clean up */
	close(fd);
	return(0);
}
/* vim: set sw=8 ts=8 si : */
