/* vim: set sw=8 ts=8 si et : */
/*************************************************************************
Title:    linuxfocus autonomous robot
Authors:   guido socher <guido[at]linuxfocus.org>, 
           katja socher <katja[at]linuxfocus.org>
Copyright: GPL
**************************************************************************/
#include <io.h>
#include <progmem.h>
#include <interrupt.h>
#include "uart.h"
#include <string-avr.h>
#include <stdlib.h>
#include <sig-avr.h>

/* the green led: */
#define LED2 PD2
/* the red led: */
#define LED1 PB0

/* defines if a pin is input or output,ddr= data direction register */
#define MOTORDDR DDRB  
/* the port controls the values (0 or 5V) of the output pins*/
#define MOTORPORT PORTB

#define MOTOR1PLUS PB2
#define MOTOR1MINUS PB3
#define MOTOR2PLUS PB4
#define MOTOR2MINUS PB5

/* to control the speed of the motors*/
#define MOTORENABLE PB1

/* defines if a pin is input or output,ddr= data direction register */
#define TSENSORDDR DDRD  
/* the input pin controls the values in the port register, PIN is needed
*  * instead of PORT when using it as input*/
#define TSENSORPIND PIND

#define TSENSORPORT PORTD

#define TSENSORFRONTRIGHT PD3
#define TSENSORFRONTLEFT PD4
#define TSENSORRIGHT PD5
#define TSENSORLEFT PD6
#define TSENSORBACK PD7
/* the bit positions for the input lines */
#define TSENSORPINFRONTRIGHT PIND3
#define TSENSORPINFRONTLEFT PIND4
#define TSENSORPINRIGHT PIND5
#define TSENSORPINLEFT PIND6
#define TSENSORPINBACK PIND7

/*  tolerance value when comparing light values, see formula below */
#define PHOTOTOLERANCE 20

static char uart_outbuf[UART_RX_BUFFER_SIZE+1];


static volatile unsigned char adcready;
static volatile unsigned char lowbyte;
static volatile unsigned char highbyte;

/* int = 16 bit */
static unsigned int photo[3];

/* prev direction, in which direction did we walk sofar */
static unsigned char pdir=0;
static unsigned char ppdir=0;
/* dont walk forever in cycles */
static unsigned char circlestop=0;


void delay_ms(unsigned int ms)
/* delay for a minimum of <ms> */
/* with a 4Mhz crystal, the resolution is 1 ms */
{
        unsigned int outer1, outer2;
        outer1 = 200; /* 4 MHZ */
        /* outer1 = 300;  6 MHZ */

        while (outer1) {
                outer2 = 1000;
                while (outer2) {
                        while ( ms ) ms--;
                        outer2--;
                }
                outer1--;
        }
}

/* compare 2 numbers with a tolerance */
unsigned char compare_with_tol(unsigned int value1,unsigned int value2)
{
        /* results of this function: 0 difference is within tolerance
        * 1 value1 is bigger
        * 2 value2 is bigger
        */
        unsigned int tmp;
        unsigned int mean;
        unsigned char whichcase;
        mean=(value1 + value2)/2;
        if (value1 >= value2){
                tmp=value1-value2;
                whichcase=1;
        }else{
                tmp=value2-value1;
                whichcase=2;
        }
        mean=mean / PHOTOTOLERANCE;
        uart_sendstr_P("m:");
        itoa(mean,uart_outbuf,10);
        uart_sendstr(uart_outbuf);
        uart_sendchar('\n');
        uart_sendstr_P("t:");
        itoa(tmp,uart_outbuf,10);
        uart_sendstr(uart_outbuf);
        uart_sendchar('\n');
        if (tmp < mean){
                return(0);
        }
        return(whichcase);
}

SIGNAL(SIG_ADC)
{
        /* this function will be called when the ADC conversion is ready*/
        lowbyte=inp(ADCL);
        highbyte=inp(ADCH);
        adcready=1;
}

void motoroff(void)
{
    /* motors off only on when button pressed (see below) */
    cbi(MOTORPORT,MOTOR1PLUS);
    cbi(MOTORPORT,MOTOR1MINUS);
    cbi(MOTORPORT,MOTOR2MINUS);
    cbi(MOTORPORT,MOTOR2PLUS);
}
void left(void)
{
        if (pdir != 1){
                /* change in direction */
                motoroff();
        }
        if (circlestop ==0){
                /* turn right weels only */
                cbi(MOTORPORT,MOTOR1PLUS);
                sbi(MOTORPORT,MOTOR1MINUS);
        }
        pdir=1;
}

void right(void)
{
        if (pdir != 2){
                /* change in direction */
                motoroff();
        }
        if (circlestop ==0){
                /* turn left weels only */
                cbi(MOTORPORT,MOTOR2PLUS);
                sbi(MOTORPORT,MOTOR2MINUS);
        }
        pdir=2;
}

/* turn like a tank on the spot: */
void left_onspot(void)
{
        if (circlestop ==0){
                /* turn right weels forward and left backward */
                cbi(MOTORPORT,MOTOR1PLUS);
                sbi(MOTORPORT,MOTOR1MINUS);

                sbi(MOTORPORT,MOTOR2PLUS);
                cbi(MOTORPORT,MOTOR2MINUS);
        }
        pdir=3;
}

/* turn like a tank on the spot: */
void right_onspot(void)
{
        if (circlestop ==0){
                /* turn left weels forward and right backward */
                cbi(MOTORPORT,MOTOR2PLUS);
                sbi(MOTORPORT,MOTOR2MINUS);

                sbi(MOTORPORT,MOTOR1PLUS);
                cbi(MOTORPORT,MOTOR1MINUS);
        }
        pdir=4;
}

void straight(void)
{
        cbi(MOTORPORT,MOTOR1PLUS);
        sbi(MOTORPORT,MOTOR1MINUS);
        cbi(MOTORPORT,MOTOR2PLUS);
        sbi(MOTORPORT,MOTOR2MINUS);
        pdir=5;
}

void cktouchsensor(void)
{
        char sensorhit=0;
        /*check if any of the touch sensors is pressed*/
        if (bit_is_clear(TSENSORPIND,TSENSORPINFRONTRIGHT)) {
                motoroff();
                /* turn motor on the right */
                cbi(MOTORPORT,MOTOR2PLUS);
                sbi(MOTORPORT,MOTOR2MINUS);
                sensorhit=1;
        }
        if (bit_is_clear(TSENSORPIND,TSENSORPINFRONTLEFT)) {
                /* turn motor on the left */
                motoroff();
                cbi(MOTORPORT,MOTOR1PLUS);
                sbi(MOTORPORT,MOTOR1MINUS);
                sensorhit=1;
        }
        if (bit_is_clear(TSENSORPIND,TSENSORPINRIGHT)) {
                motoroff();
                sbi(MOTORPORT,MOTOR2PLUS);
                cbi(MOTORPORT,MOTOR2MINUS);
                sensorhit=1;
        }
        if (bit_is_clear(TSENSORPIND,TSENSORPINLEFT)) {
                motoroff();
                sbi(MOTORPORT,MOTOR1PLUS);
                cbi(MOTORPORT,MOTOR1MINUS);
                sensorhit=1;
        }

        if (sensorhit==1){
                /* red led off: */
                sbi(PORTB,LED1);
                /* walk a bit: */
                delay_ms(400);
                motoroff();
        }
}

/* delay a bit but check the touch sensors */
void delay_ck(unsigned int ms)
{
        unsigned int outer1, outer2;
        outer1 = 2000; 
        while (outer1) {
                outer2=50;
                while(outer2){
                        while ( ms ) {
                                ms--;
                                cktouchsensor();
                        }
                        outer2--;
                }
                outer1--;
        }
}

void main(void)
{ 
        unsigned int average;
        unsigned char cmpres;
        unsigned char circledetect=0;

        /*enable led as output */
        sbi(DDRB,LED1);
        sbi(DDRD,LED2);
        
        /* led off */
        sbi(PORTB,LED1);
        sbi(PORTD,LED2);

        /*enable motor pins as output */
        sbi(MOTORDDR,MOTOR1PLUS);
        sbi(MOTORDDR,MOTOR1MINUS);
        sbi(MOTORDDR,MOTOR2PLUS);
        sbi(MOTORDDR,MOTOR2MINUS);
        sbi(MOTORDDR,MOTORENABLE);

        /* enable touch sensor as input line */
        cbi(TSENSORDDR,TSENSORFRONTRIGHT);
        cbi(TSENSORDDR,TSENSORFRONTLEFT);
        cbi(TSENSORDDR,TSENSORRIGHT);
        cbi(TSENSORDDR,TSENSORLEFT);

        /* pull up resistors on */
        sbi(TSENSORPORT,TSENSORPINFRONTRIGHT);
        sbi(TSENSORPORT,TSENSORPINFRONTLEFT);
        sbi(TSENSORPORT,TSENSORPINRIGHT);
        sbi(TSENSORPORT,TSENSORPINLEFT);
        sbi(TSENSORPORT,TSENSORPINBACK);

        /* enable to on (=full speed) */
        sbi(MOTORPORT,MOTORENABLE);

        motoroff();

        /* initialize rs232 */
        uart_init();

        /* enable analog to digital conversion in single run mode
        *  without noise canceler function. See datasheet of at90s4433 page 55 
        * We set ADPS2 and ADPS0 to have a clock division factor of 32.
        * This is needed to stay in the recommended range of 50-200kHz 
        * ADEN: Analog Digital Converter Enable
        * ADIE: ADC Interrupt Enable
        * ADIF: ADC Interrupt Flag
        * ADCSR: ADC Control and Status Register
        * ADPS2..ADPS0: ADC Prescaler Select Bits
        */
        outp((1<<ADEN)|(1<<ADIE)|(1<<ADIF)|(1<<ADPS2)|(1<<ADPS2),ADCSR);
        sei(); /* enable interrupts */
        
        while (1) {
                /*check the values of the photo resistors */
                
                outp(0,ADMUX); /* photoresistor 0*/
                adcready=0;
                /*  start conversion */
                sbi(ADCSR,ADSC);
                /* wait until conversion ready*/
                while(adcready==0);
                /* combine the lower and higher byte of 
                * the ADC result to one number */
                photo[0]=(highbyte<<8)|lowbyte;
                
                outp(1,ADMUX); /* photoresistor 1*/
                adcready=0;
                /*  start conversion */
                sbi(ADCSR,ADSC);
                /* wait until conversion ready*/
                while(adcready==0);
                /* combine the lower and higher byte of 
                * the ADC result to one number */
                photo[1]=(highbyte<<8)|lowbyte;
                
                outp(2,ADMUX); /* photoresistor 2*/
                adcready=0;
                /*  start conversion */
                sbi(ADCSR,ADSC);
                /* wait until conversion ready*/
                while(adcready==0);
                /* combine the lower and higher byte of 
                * the ADC result to one number */
                photo[2]=(highbyte<<8)|lowbyte; 


                uart_sendstr_P("0:");
                itoa(photo[0],uart_outbuf,10);
                uart_sendstr(uart_outbuf);
                uart_sendchar('\n');
                
                uart_sendstr_P("1:");
                itoa(photo[1],uart_outbuf,10);
                uart_sendstr(uart_outbuf);
                uart_sendchar('\n');
                
                uart_sendstr_P("2:");
                itoa(photo[2],uart_outbuf,10);
                uart_sendstr(uart_outbuf);
                uart_sendchar('\n');
                
                
                /* decide what to do */

                /* first check if we hit something */
                cktouchsensor();

                /* decide if there is more light in front or on the
                * back. We take the average of the 2 sensors in the front
                * and compare them with the back */
                average=(photo[0]+photo[1])/2;
                cmpres=compare_with_tol(photo[2],average);
                if (cmpres ==2 ){
                        uart_sendstr_P("front darker\n");
                        /* darker in front, turn on spot */
                        cmpres=compare_with_tol(photo[0],photo[1]);
                        if (cmpres ==2 ){
                                /* darker on left, turn right */
                                right_onspot();
                        }else{
                                left_onspot();
                        }
                        /* for onspot we let the robot turn a bit
                        * without reading new data */
                        delay_ck(5);
                }else if (cmpres ==1){
                        uart_sendstr_P("light ahead\n");
                        /* now just decide to go left or right */
                        cmpres=compare_with_tol(photo[0],photo[1]);
                        if (cmpres ==2 ){
                                /* darker on left, turn right */
                                right();
                        }else if (cmpres ==1){
                                left();
                        }else{
                                straight();
                        }
                }else{
                        uart_sendstr_P("hmm\n");
                        /* front and back is same light*/

                        /* check left/right */
                        cmpres=compare_with_tol(photo[0],photo[1]);
                        if (cmpres ==2 ){
                                /* darker on left, turn right */
                                right();
                        }else if (cmpres ==1){
                                left();
                        }else{
                                /* arrived under the sun, blink green led */
                                motoroff();
                                cbi(PORTD,LED2);
                                delay_ck(60);
                                ppdir=5; // avoid circle check
                        }
                }
                /* detect if we are walking in a circle direction 5 (straight)
                * must be ignored */
                if (pdir == ppdir && ppdir !=5){
                        circledetect++;
                        if (circledetect > 40){
                                uart_sendstr_P("circle\n");
                                motoroff();
                                /* indicate that we arrived (we give up running in circle */
                                cbi(PORTD,LED2);
                                delay_ck(60);
                                circlestop=1;
                                circledetect=100;
                        }else{
                                circlestop=0;
                        }
                }else{
                        circledetect=0;
                        circlestop=0;
                }
                ppdir=pdir;
                                

                /* wait a bit for the next decission */
                /* led on */
                cbi(PORTB,LED1);
                delay_ck(3);

                /* led off */
                sbi(PORTB,LED1);

                uart_sendchar('\n');
                sbi(PORTD,LED2);
        }

}
